# -*- coding: utf-8 -*-
#
"""
Axiomatic triples to be (possibly) added to the final graph.

**Requires**: `RDFLib`_, 4.0.0 and higher.

.. _RDFLib: https://github.com/RDFLib/rdflib

**License**: This software is available for use under the `W3C Software License`_.

.. _W3C Software License: http://www.w3.org/Consortium/Legal/2002/copyright-software-20021231

**Organization**: `World Wide Web Consortium`_

.. _World Wide Web Consortium: http://www.w3.org

**Author**: `Ivan Herman`_

.. _Ivan Herman: http://www.w3.org/People/Ivan/

"""

__author__ = "Ivan Herman"
__contact__ = "Ivan Herman, ivan@w3.org"
__license__ = "W3C® SOFTWARE NOTICE AND LICENSE, http://www.w3.org/Consortium/Legal/2002/copyright-software-20021231"

from rdflib.namespace import OWL, RDF, RDFS, XSD

# Simple RDF axiomatic triples (typing of subject, predicate, first, rest, etc)
_Simple_RDF_axiomatic_triples = [
    (RDF.type, RDF.type, RDF.Property),
    (RDF.subject, RDF.type, RDF.Property),
    (RDF.predicate, RDF.type, RDF.Property),
    (RDF.object, RDF.type, RDF.Property),
    (RDF.first, RDF.type, RDF.Property),
    (RDF.rest, RDF.type, RDF.Property),
    (RDF.value, RDF.type, RDF.Property),
    (RDF.nil, RDF.type, RDF.List),
]

# RDFS axiomatic triples (domain and range, as well as class setting for a number of RDFS symbols)
_RDFS_axiomatic_triples = [
    (RDF.type, RDFS.domain, RDFS.Resource),
    (RDFS.domain, RDFS.domain, RDF.Property),
    (RDFS.range, RDFS.domain, RDF.Property),
    (RDFS.subPropertyOf, RDFS.domain, RDF.Property),
    (RDFS.subClassOf, RDFS.domain, RDFS.Class),
    (RDF.subject, RDFS.domain, RDF.Statement),
    (RDF.predicate, RDFS.domain, RDF.Statement),
    (RDF.object, RDFS.domain, RDF.Statement),
    (RDFS.member, RDFS.domain, RDFS.Resource),
    (RDF.first, RDFS.domain, RDF.List),
    (RDF.rest, RDFS.domain, RDF.List),
    (RDFS.seeAlso, RDFS.domain, RDFS.Resource),
    (RDFS.isDefinedBy, RDFS.domain, RDFS.Resource),
    (RDFS.comment, RDFS.domain, RDFS.Resource),
    (RDFS.label, RDFS.domain, RDFS.Resource),
    (RDF.value, RDFS.domain, RDFS.Resource),
    (RDF.Property, RDF.type, RDFS.Class),
    (RDF.type, RDFS.range, RDFS.Class),
    (RDFS.domain, RDFS.range, RDFS.Class),
    (RDFS.range, RDFS.range, RDFS.Class),
    (RDFS.subPropertyOf, RDFS.range, RDF.Property),
    (RDFS.subClassOf, RDFS.range, RDFS.Class),
    (RDF.subject, RDFS.range, RDFS.Resource),
    (RDF.predicate, RDFS.range, RDFS.Resource),
    (RDF.object, RDFS.range, RDFS.Resource),
    (RDFS.member, RDFS.range, RDFS.Resource),
    (RDF.first, RDFS.range, RDFS.Resource),
    (RDF.rest, RDFS.range, RDF.List),
    (RDFS.seeAlso, RDFS.range, RDFS.Resource),
    (RDFS.isDefinedBy, RDFS.range, RDFS.Resource),
    (RDFS.comment, RDFS.range, RDFS.Literal),
    (RDFS.label, RDFS.range, RDFS.Literal),
    (RDF.value, RDFS.range, RDFS.Resource),
    (RDF.Alt, RDFS.subClassOf, RDFS.Container),
    (RDF.Bag, RDFS.subClassOf, RDFS.Container),
    (RDF.Seq, RDFS.subClassOf, RDFS.Container),
    (RDFS.ContainerMembershipProperty, RDFS.subClassOf, RDF.Property),
    (RDFS.isDefinedBy, RDFS.subPropertyOf, RDFS.seeAlso),
    (RDF.XMLLiteral, RDF.type, RDFS.Datatype),
    (RDF.XMLLiteral, RDFS.subClassOf, RDFS.Literal),
    (RDFS.Datatype, RDFS.subClassOf, RDFS.Class),
    # rdfs valid triples; these would be inferred by the RDFS expansion, but it may make things
    # a bit faster to add these upfront
    (RDFS.Resource, RDF.type, RDFS.Class),
    (RDFS.Class, RDF.type, RDFS.Class),
    (RDFS.Literal, RDF.type, RDFS.Class),
    (RDF.XMLLiteral, RDF.type, RDFS.Class),
    (RDFS.Datatype, RDF.type, RDFS.Class),
    (RDF.Seq, RDF.type, RDFS.Class),
    (RDF.Bag, RDF.type, RDFS.Class),
    (RDF.Alt, RDF.type, RDFS.Class),
    (RDFS.Container, RDF.type, RDFS.Class),
    (RDF.List, RDF.type, RDFS.Class),
    (RDFS.ContainerMembershipProperty, RDF.type, RDFS.Class),
    (RDF.Property, RDF.type, RDFS.Class),
    (RDF.Statement, RDF.type, RDFS.Class),
    (RDFS.domain, RDF.type, RDF.Property),
    (RDFS.range, RDF.type, RDF.Property),
    (RDFS.subPropertyOf, RDF.type, RDF.Property),
    (RDFS.subClassOf, RDF.type, RDF.Property),
    (RDFS.member, RDF.type, RDF.Property),
    (RDFS.seeAlso, RDF.type, RDF.Property),
    (RDFS.isDefinedBy, RDF.type, RDF.Property),
    (RDFS.comment, RDF.type, RDF.Property),
    (RDFS.label, RDF.type, RDF.Property),
]

# RDFS Axiomatic Triples all together
RDFS_Axiomatic_Triples = _Simple_RDF_axiomatic_triples + _RDFS_axiomatic_triples

# RDFS D-entailement triples, ie, possible subclassing of various datatypes
RDFS_D_Axiomatic_Triples_subclasses = [
    # See http://www.w3.org/TR/2004/REC-xmlschema-2-20041028/#built-in-datatypes
    (XSD.decimal, RDFS.subClassOf, RDFS.Literal),
    (XSD.integer, RDFS.subClassOf, XSD.decimal),
    (XSD.long, RDFS.subClassOf, XSD.integer),
    (XSD.int, RDFS.subClassOf, XSD.long),
    (XSD.short, RDFS.subClassOf, XSD.int),
    (XSD.byte, RDFS.subClassOf, XSD.short),
    (XSD.nonNegativeInteger, RDFS.subClassOf, XSD.integer),
    (XSD.positiveInteger, RDFS.subClassOf, XSD.nonNegativeInteger),
    (XSD.unsignedLong, RDFS.subClassOf, XSD.nonNegativeInteger),
    (XSD.unsignedInt, RDFS.subClassOf, XSD.unsignedLong),
    (XSD.unsignedShort, RDFS.subClassOf, XSD.unsignedInt),
    (XSD.unsignedByte, RDFS.subClassOf, XSD.unsignedShort),
    (XSD.nonPositiveInteger, RDFS.subClassOf, XSD.integer),
    (XSD.negativeInteger, RDFS.subClassOf, XSD.nonPositiveInteger),
    (XSD.normalizedString, RDFS.subClassOf, XSD.string),
    (XSD.token, RDFS.subClassOf, XSD.normalizedString),
    (XSD.language, RDFS.subClassOf, XSD.token),
    (XSD.Name, RDFS.subClassOf, XSD.token),
    (XSD.NMTOKEN, RDFS.subClassOf, XSD.token),
    (XSD.NCName, RDFS.subClassOf, XSD.Name),
    (XSD.dateTimeStamp, RDFS.subClassOf, XSD.dateTime),
]

#
RDFS_D_Axiomatic_Triples_types = [
    (XSD.integer, RDF.type, RDFS.Datatype),
    (XSD.decimal, RDF.type, RDFS.Datatype),
    (XSD.nonPositiveInteger, RDF.type, RDFS.Datatype),
    (XSD.nonPositiveInteger, RDF.type, RDFS.Datatype),
    (XSD.positiveInteger, RDF.type, RDFS.Datatype),
    (XSD.positiveInteger, RDF.type, RDFS.Datatype),
    (XSD.long, RDF.type, RDFS.Datatype),
    (XSD.int, RDF.type, RDFS.Datatype),
    (XSD.short, RDF.type, RDFS.Datatype),
    (XSD.byte, RDF.type, RDFS.Datatype),
    (XSD.unsignedLong, RDF.type, RDFS.Datatype),
    (XSD.unsignedInt, RDF.type, RDFS.Datatype),
    (XSD.unsignedShort, RDF.type, RDFS.Datatype),
    (XSD.unsignedByte, RDF.type, RDFS.Datatype),
    (XSD.float, RDF.type, RDFS.Datatype),
    (XSD.double, RDF.type, RDFS.Datatype),
    (XSD.string, RDF.type, RDFS.Datatype),
    (XSD.normalizedString, RDF.type, RDFS.Datatype),
    (XSD.token, RDF.type, RDFS.Datatype),
    (XSD.language, RDF.type, RDFS.Datatype),
    (XSD.Name, RDF.type, RDFS.Datatype),
    (XSD.NCName, RDF.type, RDFS.Datatype),
    (XSD.NMTOKEN, RDF.type, RDFS.Datatype),
    (XSD.boolean, RDF.type, RDFS.Datatype),
    (XSD.hexBinary, RDF.type, RDFS.Datatype),
    (XSD.base64Binary, RDF.type, RDFS.Datatype),
    (XSD.anyURI, RDF.type, RDFS.Datatype),
    (XSD.dateTimeStamp, RDF.type, RDFS.Datatype),
    (XSD.dateTime, RDF.type, RDFS.Datatype),
    (RDFS.Literal, RDF.type, RDFS.Datatype),
    (RDF.XMLLiteral, RDF.type, RDFS.Datatype),
]

RDFS_D_Axiomatic_Triples = (
    RDFS_D_Axiomatic_Triples_types + RDFS_D_Axiomatic_Triples_subclasses
)

# OWL Class axiomatic triples: definition of special classes
_OWL_axiomatic_triples_Classes = [
    (OWL.AllDifferent, RDF.type, RDFS.Class),
    (OWL.AllDifferent, RDFS.subClassOf, RDFS.Resource),
    (OWL.AllDisjointClasses, RDF.type, RDFS.Class),
    (OWL.AllDisjointClasses, RDFS.subClassOf, RDFS.Resource),
    (OWL.AllDisjointProperties, RDF.type, RDFS.Class),
    (OWL.AllDisjointProperties, RDFS.subClassOf, RDFS.Resource),
    (OWL.Annotation, RDF.type, RDFS.Class),
    (OWL.Annotation, RDFS.subClassOf, RDFS.Resource),
    (OWL.AnnotationProperty, RDF.type, RDFS.Class),
    (OWL.AnnotationProperty, RDFS.subClassOf, RDF.Property),
    (OWL.AsymmetricProperty, RDF.type, RDFS.Class),
    (OWL.AsymmetricProperty, RDFS.subClassOf, RDF.Property),
    (OWL.Class, RDF.type, RDFS.Class),
    (OWL.Class, OWL.equivalentClass, RDFS.Class),
    #    (OWL.DataRange, RDF.type, RDFS.Class),
    #    (OWL.DataRange, OWL.equivalentClass, RDFS.Datatype),
    (RDFS.Datatype, RDF.type, RDFS.Class),
    (OWL.DatatypeProperty, RDF.type, RDFS.Class),
    (OWL.DatatypeProperty, RDFS.subClassOf, RDF.Property),
    (OWL.DeprecatedClass, RDF.type, RDFS.Class),
    (OWL.DeprecatedClass, RDFS.subClassOf, RDFS.Class),
    (OWL.DeprecatedProperty, RDF.type, RDFS.Class),
    (OWL.DeprecatedProperty, RDFS.subClassOf, RDF.Property),
    (OWL.FunctionalProperty, RDF.type, RDFS.Class),
    (OWL.FunctionalProperty, RDFS.subClassOf, RDF.Property),
    (OWL.InverseFunctionalProperty, RDF.type, RDFS.Class),
    (OWL.InverseFunctionalProperty, RDFS.subClassOf, RDF.Property),
    (OWL.IrreflexiveProperty, RDF.type, RDFS.Class),
    (OWL.IrreflexiveProperty, RDFS.subClassOf, RDF.Property),
    (RDFS.Literal, RDF.type, RDFS.Datatype),
    #    (OWL.NamedIndividual, RDF.type, RDFS.Class),
    #    (OWL.NamedIndividual, OWL.equivalentClass, RDFS.Resource),
    (OWL.NegativePropertyAssertion, RDF.type, RDFS.Class),
    (OWL.NegativePropertyAssertion, RDFS.subClassOf, RDFS.Resource),
    (OWL.Nothing, RDF.type, RDFS.Class),
    (OWL.Nothing, RDFS.subClassOf, OWL.Thing),
    (OWL.ObjectProperty, RDF.type, RDFS.Class),
    (OWL.ObjectProperty, OWL.equivalentClass, RDF.Property),
    (OWL.Ontology, RDF.type, RDFS.Class),
    (OWL.Ontology, RDFS.subClassOf, RDFS.Resource),
    (OWL.OntologyProperty, RDF.type, RDFS.Class),
    (OWL.OntologyProperty, RDFS.subClassOf, RDF.Property),
    (RDF.Property, RDF.type, RDFS.Class),
    (OWL.ReflexiveProperty, RDF.type, RDFS.Class),
    (OWL.ReflexiveProperty, RDFS.subClassOf, RDF.Property),
    (OWL.Restriction, RDF.type, RDFS.Class),
    (OWL.Restriction, RDFS.subClassOf, RDFS.Class),
    (OWL.SymmetricProperty, RDF.type, RDFS.Class),
    (OWL.SymmetricProperty, RDFS.subClassOf, RDF.Property),
    (OWL.Thing, RDF.type, RDFS.Class),
    (OWL.Thing, RDFS.subClassOf, RDFS.Resource),
    (OWL.TransitiveProperty, RDF.type, RDFS.Class),
    (OWL.TransitiveProperty, RDFS.subClassOf, RDF.Property),
    # OWL valid triples; some of these would be inferred by the OWL RL expansion, but it may make things
    # a bit faster to add these upfront
    (OWL.AllDisjointProperties, RDF.type, OWL.Class),
    (OWL.AllDisjointClasses, RDF.type, OWL.Class),
    (OWL.AllDisjointProperties, RDF.type, OWL.Class),
    (OWL.Annotation, RDF.type, OWL.Class),
    (OWL.AsymmetricProperty, RDF.type, OWL.Class),
    (OWL.Axiom, RDF.type, OWL.Class),
    (OWL.DataRange, RDF.type, OWL.Class),
    (RDFS.Datatype, RDF.type, OWL.Class),
    (OWL.DatatypeProperty, RDF.type, OWL.Class),
    (OWL.DeprecatedClass, RDF.type, OWL.Class),
    (OWL.DeprecatedClass, RDFS.subClassOf, OWL.Class),
    (OWL.DeprecatedProperty, RDF.type, OWL.Class),
    (OWL.FunctionalProperty, RDF.type, OWL.Class),
    (OWL.InverseFunctionalProperty, RDF.type, OWL.Class),
    (OWL.IrreflexiveProperty, RDF.type, OWL.Class),
    (OWL.NamedIndividual, RDF.type, OWL.Class),
    (OWL.NegativePropertyAssertion, RDF.type, OWL.Class),
    (OWL.Nothing, RDF.type, OWL.Class),
    (OWL.ObjectProperty, RDF.type, OWL.Class),
    (OWL.Ontology, RDF.type, OWL.Class),
    (OWL.OntologyProperty, RDF.type, OWL.Class),
    (RDF.Property, RDF.type, OWL.Class),
    (OWL.ReflexiveProperty, RDF.type, OWL.Class),
    (OWL.Restriction, RDF.type, OWL.Class),
    (OWL.Restriction, RDFS.subClassOf, OWL.Class),
    #    (OWL.SelfRestriction, RDF.type, OWL.Class),
    (OWL.SymmetricProperty, RDF.type, OWL.Class),
    (OWL.Thing, RDF.type, OWL.Class),
    (OWL.TransitiveProperty, RDF.type, OWL.Class),
]

# OWL Property axiomatic triples: definition of domains and ranges
_OWL_axiomatic_triples_Properties = [
    (OWL.allValuesFrom, RDF.type, RDF.Property),
    (OWL.allValuesFrom, RDFS.domain, OWL.Restriction),
    (OWL.allValuesFrom, RDFS.range, RDFS.Class),
    (OWL.assertionProperty, RDF.type, RDF.Property),
    (OWL.assertionProperty, RDFS.domain, OWL.NegativePropertyAssertion),
    (OWL.assertionProperty, RDFS.range, RDF.Property),
    (OWL.backwardCompatibleWith, RDF.type, OWL.OntologyProperty),
    (OWL.backwardCompatibleWith, RDF.type, OWL.AnnotationProperty),
    (OWL.backwardCompatibleWith, RDFS.domain, OWL.Ontology),
    (OWL.backwardCompatibleWith, RDFS.range, OWL.Ontology),
    #    (OWL.bottomDataProperty, RDF.type, RDFS.DatatypeProperty),
    #    (OWL.bottomObjectProperty, RDF.type, OWL.ObjectProperty),
    #    (OWL.cardinality, RDF.type, RDF.Property),
    #    (OWL.cardinality, RDFS.domain, OWL.Restriction),
    #    (OWL.cardinality, RDFS.range, XSD.nonNegativeInteger),
    (RDFS.comment, RDF.type, OWL.AnnotationProperty),
    (RDFS.comment, RDFS.domain, RDFS.Resource),
    (RDFS.comment, RDFS.range, RDFS.Literal),
    (OWL.complementOf, RDF.type, RDF.Property),
    (OWL.complementOf, RDFS.domain, RDFS.Class),
    (OWL.complementOf, RDFS.range, RDFS.Class),
    #    (OWL.datatypeComplementOf, RDF.type, RDF.Property),
    #    (OWL.datatypeComplementOf, RDFS.domain, RDFS.Datatype),
    #    (OWL.datatypeComplementOf, RDFS.range, RDFS.Datatype),
    (OWL.deprecated, RDF.type, OWL.AnnotationProperty),
    (OWL.deprecated, RDFS.domain, RDFS.Resource),
    (OWL.deprecated, RDFS.range, RDFS.Resource),
    (OWL.differentFrom, RDF.type, RDF.Property),
    (OWL.differentFrom, RDFS.domain, RDFS.Resource),
    (OWL.differentFrom, RDFS.range, RDFS.Resource),
    #    (OWL.disjointUnionOf, RDF.type, RDF.Property),
    #    (OWL.disjointUnionOf, RDFS.domain, RDFS.Class),
    #    (OWL.disjointUnionOf, RDFS.range, RDF.List),
    (OWL.disjointWith, RDF.type, RDF.Property),
    (OWL.disjointWith, RDFS.domain, RDFS.Class),
    (OWL.disjointWith, RDFS.range, RDFS.Class),
    (OWL.distinctMembers, RDF.type, RDF.Property),
    (OWL.distinctMembers, RDFS.domain, OWL.AllDifferent),
    (OWL.distinctMembers, RDFS.range, RDF.List),
    (OWL.equivalentClass, RDF.type, RDF.Property),
    (OWL.equivalentClass, RDFS.domain, RDFS.Class),
    (OWL.equivalentClass, RDFS.range, RDFS.Class),
    (OWL.equivalentProperty, RDF.type, RDF.Property),
    (OWL.equivalentProperty, RDFS.domain, RDF.Property),
    (OWL.equivalentProperty, RDFS.range, RDF.Property),
    (OWL.hasKey, RDF.type, RDF.Property),
    (OWL.hasKey, RDFS.domain, RDFS.Class),
    (OWL.hasKey, RDFS.range, RDF.List),
    (OWL.hasValue, RDF.type, RDF.Property),
    (OWL.hasValue, RDFS.domain, OWL.Restriction),
    (OWL.hasValue, RDFS.range, RDFS.Resource),
    (OWL.imports, RDF.type, OWL.OntologyProperty),
    (OWL.imports, RDFS.domain, OWL.Ontology),
    (OWL.imports, RDFS.range, OWL.Ontology),
    (OWL.incompatibleWith, RDF.type, OWL.OntologyProperty),
    (OWL.incompatibleWith, RDF.type, OWL.AnnotationProperty),
    (OWL.incompatibleWith, RDFS.domain, OWL.Ontology),
    (OWL.incompatibleWith, RDFS.range, OWL.Ontology),
    (OWL.intersectionOf, RDF.type, RDF.Property),
    (OWL.intersectionOf, RDFS.domain, RDFS.Class),
    (OWL.intersectionOf, RDFS.range, RDF.List),
    (OWL.inverseOf, RDF.type, RDF.Property),
    (OWL.inverseOf, RDFS.domain, RDF.Property),
    (OWL.inverseOf, RDFS.range, RDF.Property),
    (RDFS.isDefinedBy, RDF.type, OWL.AnnotationProperty),
    (RDFS.isDefinedBy, RDFS.domain, RDFS.Resource),
    (RDFS.isDefinedBy, RDFS.range, RDFS.Resource),
    (RDFS.label, RDF.type, OWL.AnnotationProperty),
    (RDFS.label, RDFS.domain, RDFS.Resource),
    (RDFS.label, RDFS.range, RDFS.Literal),
    (OWL.maxCardinality, RDF.type, RDF.Property),
    (OWL.maxCardinality, RDFS.domain, OWL.Restriction),
    (OWL.maxCardinality, RDFS.range, XSD.nonNegativeInteger),
    (OWL.maxQualifiedCardinality, RDF.type, RDF.Property),
    (OWL.maxQualifiedCardinality, RDFS.domain, OWL.Restriction),
    (OWL.maxQualifiedCardinality, RDFS.range, XSD.nonNegativeInteger),
    (OWL.members, RDF.type, RDF.Property),
    (OWL.members, RDFS.domain, RDFS.Resource),
    (OWL.members, RDFS.range, RDF.List),
    #    (OWL.minCardinality, RDF.type, RDF.Property),
    #    (OWL.minCardinality, RDFS.domain, OWL.Restriction),
    #    (OWL.minCardinality, RDFS.range, XSD.nonNegativeInteger),
    #    (OWL.minQualifiedCardinality, RDF.type, RDF.Property),
    #    (OWL.minQualifiedCardinality, RDFS.domain, OWL.Restriction),
    #    (OWL.minQualifiedCardinality, RDFS.range, XSD.nonNegativeInteger),
    #    (OWL.annotatedTarget, RDF.type, RDF.Property),
    #    (OWL.annotatedTarget, RDFS.domain, RDFS.Resource),
    #    (OWL.annotatedTarget, RDFS.range, RDFS.Resource),
    (OWL.onClass, RDF.type, RDF.Property),
    (OWL.onClass, RDFS.domain, OWL.Restriction),
    (OWL.onClass, RDFS.range, RDFS.Class),
    #    (OWL.onDataRange, RDF.type, RDF.Property),
    #    (OWL.onDataRange, RDFS.domain, OWL.Restriction),
    #    (OWL.onDataRange, RDFS.range, RDFS.Datatype),
    (OWL.onDatatype, RDF.type, RDF.Property),
    (OWL.onDatatype, RDFS.domain, RDFS.Datatype),
    (OWL.onDatatype, RDFS.range, RDFS.Datatype),
    (OWL.oneOf, RDF.type, RDF.Property),
    (OWL.oneOf, RDFS.domain, RDFS.Class),
    (OWL.oneOf, RDFS.range, RDF.List),
    (OWL.onProperty, RDF.type, RDF.Property),
    (OWL.onProperty, RDFS.domain, OWL.Restriction),
    (OWL.onProperty, RDFS.range, RDF.Property),
    #    (OWL.onProperties, RDF.type, RDF.Property),
    #    (OWL.onProperties, RDFS.domain, OWL.Restriction),
    #    (OWL.onProperties, RDFS.range, RDF.List),
    #    (OWL.annotatedProperty, RDF.type, RDF.Property),
    #    (OWL.annotatedProperty, RDFS.domain, RDFS.Resource),
    #    (OWL.annotatedProperty, RDFS.range, RDF.Property),
    (OWL.priorVersion, RDF.type, OWL.OntologyProperty),
    (OWL.priorVersion, RDF.type, OWL.AnnotationProperty),
    (OWL.priorVersion, RDFS.domain, OWL.Ontology),
    (OWL.priorVersion, RDFS.range, OWL.Ontology),
    (OWL.propertyChainAxiom, RDF.type, RDF.Property),
    (OWL.propertyChainAxiom, RDFS.domain, RDF.Property),
    (OWL.propertyChainAxiom, RDFS.range, RDF.List),
    #    (OWL.propertyDisjointWith, RDF.type, RDF.Property),
    #    (OWL.propertyDisjointWith, RDFS.domain, RDF.Property),
    #    (OWL.propertyDisjointWith, RDFS.range, RDF.Property),
    #
    #    (OWL.qualifiedCardinality, RDF.type, RDF.Property),
    #    (OWL.qualifiedCardinality, RDFS.domain, OWL.Restriction),
    #    (OWL.qualifiedCardinality, RDFS.range, XSD.nonNegativeInteger),
    (OWL.sameAs, RDF.type, RDF.Property),
    (OWL.sameAs, RDFS.domain, RDFS.Resource),
    (OWL.sameAs, RDFS.range, RDFS.Resource),
    (RDFS.seeAlso, RDF.type, OWL.AnnotationProperty),
    (RDFS.seeAlso, RDFS.domain, RDFS.Resource),
    (RDFS.seeAlso, RDFS.range, RDFS.Resource),
    (OWL.someValuesFrom, RDF.type, RDF.Property),
    (OWL.someValuesFrom, RDFS.domain, OWL.Restriction),
    (OWL.someValuesFrom, RDFS.range, RDFS.Class),
    (OWL.sourceIndividual, RDF.type, RDF.Property),
    (OWL.sourceIndividual, RDFS.domain, OWL.NegativePropertyAssertion),
    (OWL.sourceIndividual, RDFS.range, RDFS.Resource),
    #
    #    (OWL.annotatedSource, RDF.type, RDF.Property),
    #    (OWL.annotatedSource, RDFS.domain, RDFS.Resource),
    #    (OWL.annotatedSource, RDFS.range, RDFS.Resource),
    #
    (OWL.targetIndividual, RDF.type, RDF.Property),
    (OWL.targetIndividual, RDFS.domain, OWL.NegativePropertyAssertion),
    (OWL.targetIndividual, RDFS.range, RDFS.Resource),
    (OWL.targetValue, RDF.type, RDF.Property),
    (OWL.targetValue, RDFS.domain, OWL.NegativePropertyAssertion),
    (OWL.targetValue, RDFS.range, RDFS.Literal),
    #    (OWL.topDataProperty, RDF.type, RDFS.DatatypeProperty),
    #    (OWL.topDataProperty, RDFS.domain, RDFS.Resource),
    #    (OWL.topDataProperty, RDFS.range, RDFS.Literal),
    #
    #    (OWL.topObjectProperty, RDF.type, OWL.ObjectProperty),
    #    (OWL.topObjectProperty, RDFS.domain, RDFS.Resource),
    #    (OWL.topObjectProperty, RDFS.range, RDFS.Resource),
    (OWL.unionOf, RDF.type, RDF.Property),
    (OWL.unionOf, RDFS.domain, RDFS.Class),
    (OWL.unionOf, RDFS.range, RDF.List),
    (OWL.versionInfo, RDF.type, OWL.AnnotationProperty),
    (OWL.versionInfo, RDFS.domain, RDFS.Resource),
    (OWL.versionInfo, RDFS.range, RDFS.Resource),
    (OWL.versionIRI, RDF.type, OWL.AnnotationProperty),
    (OWL.versionIRI, RDFS.domain, RDFS.Resource),
    (OWL.versionIRI, RDFS.range, RDFS.Resource),
    (OWL.withRestrictions, RDF.type, RDF.Property),
    (OWL.withRestrictions, RDFS.domain, RDFS.Datatype),
    (OWL.withRestrictions, RDFS.range, RDF.List),
    # some OWL valid triples; these would be inferred by the OWL RL expansion, but it may make things
    # a bit faster to add these upfront
    (OWL.allValuesFrom, RDFS.range, OWL.Class),
    (OWL.complementOf, RDFS.domain, OWL.Class),
    (OWL.complementOf, RDFS.range, OWL.Class),
    #    (OWL.datatypeComplementOf, domain, OWL.DataRange),
    #    (OWL.datatypeComplementOf, range, OWL.DataRange),
    (OWL.disjointUnionOf, RDFS.domain, OWL.Class),
    (OWL.disjointWith, RDFS.domain, OWL.Class),
    (OWL.disjointWith, RDFS.range, OWL.Class),
    (OWL.equivalentClass, RDFS.domain, OWL.Class),
    (OWL.equivalentClass, RDFS.range, OWL.Class),
    (OWL.hasKey, RDFS.domain, OWL.Class),
    (OWL.intersectionOf, RDFS.domain, OWL.Class),
    (OWL.onClass, RDFS.range, OWL.Class),
    #    (OWL.onDataRange, RDFS.range, OWL.DataRange),
    (OWL.onDatatype, RDFS.domain, OWL.DataRange),
    (OWL.onDatatype, RDFS.range, OWL.DataRange),
    (OWL.oneOf, RDFS.domain, OWL.Class),
    (OWL.someValuesFrom, RDFS.range, OWL.Class),
    (OWL.unionOf, RDFS.range, OWL.Class),
    #    (OWL.withRestrictions, RDFS.domain, OWL.DataRange)
]

# OWL RL axiomatic triples: combination of the RDFS triples plus the OWL specific ones
OWLRL_Axiomatic_Triples = (
    _OWL_axiomatic_triples_Classes + _OWL_axiomatic_triples_Properties
)

# Note that this is not used anywhere. But I encoded it once and I did not want to remove it...:-)
_OWL_axiomatic_triples_Facets = [
    # langPattern
    (XSD.length, RDF.type, RDF.Property),
    (XSD.maxExclusive, RDF.type, RDF.Property),
    (XSD.maxInclusive, RDF.type, RDF.Property),
    (XSD.maxLength, RDF.type, RDF.Property),
    (XSD.minExclusive, RDF.type, RDF.Property),
    (XSD.minInclusive, RDF.type, RDF.Property),
    (XSD.minLength, RDF.type, RDF.Property),
    (XSD.pattern, RDF.type, RDF.Property),
    (XSD.length, RDFS.domain, RDFS.Resource),
    (XSD.maxExclusive, RDFS.domain, RDFS.Resource),
    (XSD.maxInclusive, RDFS.domain, RDFS.Resource),
    (XSD.maxLength, RDFS.domain, RDFS.Resource),
    (XSD.minExclusive, RDFS.domain, RDFS.Resource),
    (XSD.minInclusive, RDFS.domain, RDFS.Resource),
    (XSD.minLength, RDFS.domain, RDFS.Resource),
    (XSD.pattern, RDFS.domain, RDFS.Resource),
    (XSD.length, RDFS.domain, RDFS.Resource),
    (XSD.maxExclusive, RDFS.range, RDFS.Literal),
    (XSD.maxInclusive, RDFS.range, RDFS.Literal),
    (XSD.maxLength, RDFS.range, RDFS.Literal),
    (XSD.minExclusive, RDFS.range, RDFS.Literal),
    (XSD.minInclusive, RDFS.range, RDFS.Literal),
    (XSD.minLength, RDFS.range, RDFS.Literal),
    (XSD.pattern, RDFS.range, RDFS.Literal),
]

# OWL D-entailment triples (additionally to the RDFS ones), ie, possible subclassing of various extra datatypes
_OWL_D_Axiomatic_Triples_types = [(RDF.PlainLiteral, RDF.type, RDFS.Datatype)]

#
OWL_D_Axiomatic_Triples_subclasses = [
    (XSD.string, RDFS.subClassOf, RDF.PlainLiteral),
    (XSD.normalizedString, RDFS.subClassOf, RDF.PlainLiteral),
    (XSD.token, RDFS.subClassOf, RDF.PlainLiteral),
    (XSD.Name, RDFS.subClassOf, RDF.PlainLiteral),
    (XSD.NCName, RDFS.subClassOf, RDF.PlainLiteral),
    (XSD.NMTOKEN, RDFS.subClassOf, RDF.PlainLiteral),
]

#
OWLRL_Datatypes_Disjointness = [
    (XSD.anyURI, OWL.disjointWith, XSD.base64Binary),
    (XSD.anyURI, OWL.disjointWith, XSD.boolean),
    (XSD.anyURI, OWL.disjointWith, XSD.dateTime),
    (XSD.anyURI, OWL.disjointWith, XSD.decimal),
    (XSD.anyURI, OWL.disjointWith, XSD.double),
    (XSD.anyURI, OWL.disjointWith, XSD.float),
    (XSD.anyURI, OWL.disjointWith, XSD.hexBinary),
    (XSD.anyURI, OWL.disjointWith, XSD.string),
    (XSD.anyURI, OWL.disjointWith, RDF.PlainLiteral),
    (XSD.anyURI, OWL.disjointWith, RDF.XMLLiteral),
    (XSD.base64Binary, OWL.disjointWith, XSD.boolean),
    (XSD.base64Binary, OWL.disjointWith, XSD.dateTime),
    (XSD.base64Binary, OWL.disjointWith, XSD.decimal),
    (XSD.base64Binary, OWL.disjointWith, XSD.double),
    (XSD.base64Binary, OWL.disjointWith, XSD.float),
    (XSD.base64Binary, OWL.disjointWith, XSD.hexBinary),
    (XSD.base64Binary, OWL.disjointWith, XSD.string),
    (XSD.base64Binary, OWL.disjointWith, RDF.PlainLiteral),
    (XSD.base64Binary, OWL.disjointWith, RDF.XMLLiteral),
    (XSD.boolean, OWL.disjointWith, XSD.dateTime),
    (XSD.boolean, OWL.disjointWith, XSD.decimal),
    (XSD.boolean, OWL.disjointWith, XSD.double),
    (XSD.boolean, OWL.disjointWith, XSD.float),
    (XSD.boolean, OWL.disjointWith, XSD.hexBinary),
    (XSD.boolean, OWL.disjointWith, XSD.string),
    (XSD.boolean, OWL.disjointWith, RDF.PlainLiteral),
    (XSD.boolean, OWL.disjointWith, RDF.XMLLiteral),
    (XSD.dateTime, OWL.disjointWith, XSD.decimal),
    (XSD.dateTime, OWL.disjointWith, XSD.double),
    (XSD.dateTime, OWL.disjointWith, XSD.float),
    (XSD.dateTime, OWL.disjointWith, XSD.hexBinary),
    (XSD.dateTime, OWL.disjointWith, XSD.string),
    (XSD.dateTime, OWL.disjointWith, RDF.PlainLiteral),
    (XSD.dateTime, OWL.disjointWith, RDF.XMLLiteral),
    (XSD.decimal, OWL.disjointWith, XSD.double),
    (XSD.decimal, OWL.disjointWith, XSD.float),
    (XSD.decimal, OWL.disjointWith, XSD.hexBinary),
    (XSD.decimal, OWL.disjointWith, XSD.string),
    (XSD.decimal, OWL.disjointWith, RDF.PlainLiteral),
    (XSD.decimal, OWL.disjointWith, RDF.XMLLiteral),
    (XSD.double, OWL.disjointWith, XSD.float),
    (XSD.double, OWL.disjointWith, XSD.hexBinary),
    (XSD.double, OWL.disjointWith, XSD.string),
    (XSD.double, OWL.disjointWith, RDF.PlainLiteral),
    (XSD.double, OWL.disjointWith, RDF.XMLLiteral),
    (XSD.float, OWL.disjointWith, XSD.hexBinary),
    (XSD.float, OWL.disjointWith, XSD.string),
    (XSD.float, OWL.disjointWith, RDF.PlainLiteral),
    (XSD.float, OWL.disjointWith, RDF.XMLLiteral),
    (XSD.hexBinary, OWL.disjointWith, XSD.string),
    (XSD.hexBinary, OWL.disjointWith, RDF.PlainLiteral),
    (XSD.hexBinary, OWL.disjointWith, RDF.XMLLiteral),
    (XSD.string, OWL.disjointWith, RDF.XMLLiteral),
]

# OWL RL D Axiomatic triples: combination of the RDFS ones, plus some extra statements on ranges and domains, plus
# some OWL specific datatypes
OWLRL_D_Axiomatic_Triples = (
    RDFS_D_Axiomatic_Triples
    + _OWL_D_Axiomatic_Triples_types
    + OWL_D_Axiomatic_Triples_subclasses
    + OWLRL_Datatypes_Disjointness
)
