///
/// This file is part of Rheolef.
///
/// Copyright (C) 2000-2009 Pierre Saramito <Pierre.Saramito@imag.fr>
///
/// Rheolef is free software; you can redistribute it and/or modify
/// it under the terms of the GNU General Public License as published by
/// the Free Software Foundation; either version 2 of the License, or
/// (at your option) any later version.
///
/// Rheolef is distributed in the hope that it will be useful,
/// but WITHOUT ANY WARRANTY; without even the implied warranty of
/// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
/// GNU General Public License for more details.
///
/// You should have received a copy of the GNU General Public License
/// along with Rheolef; if not, write to the Free Software
/// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
/// 
/// =========================================================================

/*Prog:msh2geo
NAME: @code{msh2geo} -- msh2geo - convert gmsh mesh in geo format
@cindex mesh
@toindex msh2geo
@pindex geo
@fiindex @file{.msh} gmsh mesh
@fiindex @file{.geo} mesh
@toindex @code{gmsh}
SYNOPSIS:
@example
  msh2geo [-zr|-rz] < @var{input}[.msh] > @var{output}.geo
@end example

DESCRIPTION:
Convert a gmsh @file{.msh} into @file{.geo} one.
The output goes to standart output.
See the @code{gmsh} documentation for a detailed description
of the @file{.mshcad} input file for @code{gmsh}.
EXAMPLE:
@example
  gmsh -2 toto.mshcad -o toto.msh
  msh2geo < toto.msh > toto.geo
  gmsh -2 -order 2 toto.mshcad -o toto2.msh
  msh2geo < toto2.msh > toto2.geo
@end example
COORDINATE SYSTEM OPTION:
Most of rheolef codes are coordinate-system independant.
The coordinate system is specified in the geometry file @file{.geo}.
@table @code
@cindex axisymmetric coordinate system
@itemx -zr
@itemx -rz
       the 2d mesh is axisymmetric: @code{zr} (resp. @code{rz}) stands when the symmetry is related
       to the first (resp. second) coordinate.
@end table
NOTES:
Pk triangle, when k>=5, may have internal nodes renumbered: from the
gmsh documentation:
  @example
  The nodes of a curved element are numbered in the following order:

    the element principal vertices;
    the internal nodes for each edge;
    the internal nodes for each face;
    the volume internal nodes. 

  The numbering for face and volume internal nodes is recursive,
  i.e., the numbering follows that of the nodes of an embedded face/volume.
  The higher order nodes are assumed to be equispaced on the element. 
  @end example

In rheolef, internal triangle nodes are numbered from left to right and then
from bottom to top. The numbering differ for triangle when k >= 5.
Thus, @code{msh2geo} fix the corresponding internal nodes numbering during the 
conversion.

Pk tetrahedrons and hexaedrons in gmsh and rheolef has not the same edge-node order
nd orientation.
E.g. for tetrahedrons, edges 13 and 23 should be swaped
and reoriented as 32 and 31.
Thus, @code{msh2geo} fix the corresponding internal nodes numbering.

TODO:
Fix for P3-tetra: swap edges orientations for 3,4,5
and swap faces 1 and 2. Check P4(T) for face orientation.
Perform face visualisation with gnuplot face fill.

See also hexa edges orient and faces numbers and orient.

Check that node are numbered by vertex-node, then edge-node, then face(tri,qua)-node and then volume(T,P,H)-node.
Otherwise, renumber all nodes.

Support for high order >= 6 element ? not documented in gmsh, but gmsh supports it at run
End:*/

#include "rheolef/point.h"
#include "rheolef/index_set.h"
#include "rheolef/reference_element.h"
#include "rheolef/rheostream.h"

using namespace std;
using namespace rheolef;

#include "msh2geo_defs.icc"
void msh2geo_node_renum (vector<size_t>& element, size_type variant, size_type order);

void
put_domain (
  ostream&                         out,
  size_type                        dim,
  size_type                        dom_dim,
  const map<size_t,list<size_t> >& domain_map,
        map<size_t, string>&       phys,
  const vector<char>&              element_name,
  const vector<size_t>&            element_order,
  const vector<vector<size_t> >&   element,
  const vector<size_t>&            old2new_inode)
{
  if (dim == dom_dim && domain_map.size() == 1) return;
  for (map<size_t,list<size_t> >::const_iterator
          first = domain_map.begin(),
          last  = domain_map.end(); first != last; first++) {
    size_type           dom_idx = (*first).first;
    const list<size_t>& dom     = (*first).second;
    string dom_name = phys[dom_idx];
    if (dom_name == "") dom_name = "unamed" + itos(dom_idx);
    out << "domain" << endl
        << dom_name << endl
        << "1 " << dom_dim << " " << dom.size() << endl;
    for (size_type variant = reference_element::first_variant_by_dimension(dom_dim);
                   variant < reference_element:: last_variant_by_dimension(dom_dim); variant++) {
      for (list<size_t>::const_iterator fe = dom.begin(), le = dom.end(); fe != le; fe++) {
        size_type i = *fe;
        if (reference_element::variant(element_name[i]) != variant) continue;
        out << element_name[i] << "\t";
        if (element_order[i] > 1) {
          out << "p" << element_order[i] << " ";
        }
        for (size_type j = 0; j < element[i].size(); j++) {
          out << old2new_inode[element[i][j]] << " ";
        }
        cout << endl;
      }
    }
    out << endl;
  }
}
void msh2geo (istream& in, ostream& out, std::string sys_coord_name = "cartesian")
{
  // ----------------------------------
  // 1. input gmsh
  // ----------------------------------
  // 1.0. preambule
  check_macro (scatch(in,"$MeshFormat"),
        "input stream does not contains a gmsh mesh file ($MeshFormat not found).");
  Float gmsh_fmt_version;
  size_type file_type, float_data_size;
  in >> gmsh_fmt_version >> file_type >> float_data_size;
  if (gmsh_fmt_version > 2.2) {
    warning_macro("gmsh format version " << gmsh_fmt_version << " founded ; expect version 2.2");
  }
  check_macro (file_type == 0, "unsupported gmsh non-ascii format");
  check_macro (scatch(in,"$EndMeshFormat"), "gmsh input error: $EndMeshFormat not found.");
  //
  // 1.1 optional domain names
  //
  bool full_match = true;
  bool partial_match = !full_match;
  check_macro (scatch(in,"$",partial_match), "gmsh input error: no more label found.");
  size_type nphys;
  map<size_t, string> phys;
  string label;
  in >> label;
  size_type n_names = 0;
  if (label == "PhysicalNames") {
    in  >> nphys;
    for (size_type i = 0; i < nphys; i++) {
      string name;
      size_type dom_dim, dom_idx;
      in  >> dom_dim >> dom_idx;
      // get name:
      char c; 
      in >> std::ws >>  c;
      if (c != '"') name.push_back(c);
      do {
        in.get(c);
        name.push_back(c);
      } while (c != '"' && c != '\n');
      // strip '"' in name
      size_type start = 0, end = name.length();
      if (name[start] == '"') start++;
      if (name[end-1] == '"') end--;
      name = name.substr(start,end-start);
      // rename spaces and tabs
      for (size_t i = 0; i < name.size(); i++) {
	if (name[i] == ' ' || name[i] == '\t') name[i] = '_';
      }
      phys[dom_idx] = name.substr(start,end-start);
    }
    check_macro (scatch(in,"$EndPhysicalNames"), "gmsh input error ($EndPhysicalNames not found).");
    check_macro (scatch(in,"$",partial_match), "gmsh input error: no more label found.");
    in >> label;
  }
  //
  // 1.2. nodes
  //
  check_macro (label == "Nodes", "$Nodes not found in gmsh file");
  size_type nnode;
  in  >> nnode;
  vector<point> node(nnode);
  Float infty = numeric_limits<Float>::max();
  point xmin ( infty,  infty,  infty);
  point xmax (-infty, -infty, -infty);
  for (size_type k = 0; k < nnode; k++) {
    size_type dummy;
    in  >> dummy >> node[k];
    for (size_type j = 0 ; j < 3; j++) {
      xmin[j] = min(node[k][j], xmin[j]);
      xmax[j] = max(node[k][j], xmax[j]);
    }
  }
  //
  // dimension is deduced from bounding box
  //
  size_type dim = 3;
  if (xmax[2] == xmin[2]) {
    dim = 2;
    if (xmax[1] == xmin[1]) {
      dim = 1;
      if (xmax[0] == xmin[0]) dim = 0;
    }
  }
  //
  // 1.3. elements
  //
  check_macro (scatch(in,"$Elements"), "$Elements not found in gmsh file");
  size_type nelement;
  in  >> nelement;
  vector<size_t>          element_gmshtype(nelement);
  vector<char>            element_name (nelement);
  vector<size_t>          element_order   (nelement);
  vector<vector<size_t> > element         (nelement);
  vector<size_t>          node_subgeo_variant (nnode, size_t(-1));
  map<size_t, list<size_t> >     point_domain_map, edge_domain_map, face_domain_map, volume_domain_map;
  size_type map_dim = 0;
  size_type order = 0;
  size_type size_by_dim [4] =  {0,0,0,0};
  size_type size_by_variant [reference_element::max_variant] = {0,0,0,0,0,0,0};
  for (size_type i = 0; i < nelement; i++) {
    size_type id, dummy, gmshtype;
    in  >> id >> gmshtype;
    size_type n_tag_gmsh;
    in >> n_tag_gmsh;
    size_type domain_idx = 0;
    for (size_type j = 0 ; j < n_tag_gmsh; j++) {
	// the first tag is the physical domain index
        // the second tag is the object index, defined for all elements
	// the third is zero (in all examples)
	size_type tag_dummy;
        in >> tag_dummy;
	if (j == 0) {
	  domain_idx = tag_dummy;
        }
    }
    check_macro (gmshtype < gmshtype_max,
	"element #" << id << ": unexpected gmsh type '" << gmshtype << "'");
    check_macro (gmsh_table[gmshtype].supported,
	"element #" << id << ": unsupported gmsh type '" << gmshtype << "'");

    element_name [i] = gmsh_table[gmshtype].variant;
    element_order[i] = gmsh_table[gmshtype].order;
    size_type nv           = gmsh_table[gmshtype].nv;
    size_type nn           = gmsh_table[gmshtype].nn_tot;
    size_type dim_i = element_dimension(element_name[i]);
    size_by_dim[dim_i]++;
    size_t variant = reference_element::variant(element_name[i]);
    size_by_variant [variant]++;
    map_dim = max(map_dim,dim_i);
    order = max(order,element_order[i]);
    element[i].resize(nn);
    for (size_type j = 0; j < nn; j++) {
      in >> element[i][j];
      element[i][j]--;
    }
    for (size_type subgeo_variant = 0; subgeo_variant <= variant; subgeo_variant++) { // set node dimension
      for (size_type loc_inod = reference_element::first_inod_by_variant(variant,element_order[i],subgeo_variant), 
                     loc_nnod = reference_element:: last_inod_by_variant(variant,element_order[i],subgeo_variant);
		     loc_inod < loc_nnod; loc_inod++) {
        node_subgeo_variant [element[i][loc_inod]] = subgeo_variant;
      }
    }
    msh2geo_node_renum (element[i], element_name[i], element_order[i]); 
    if (domain_idx != 0) {
        switch (dim_i) {
  	 case 0:   point_domain_map[domain_idx].push_back(i); break;
  	 case 1:    edge_domain_map[domain_idx].push_back(i); break;
  	 case 2:    face_domain_map[domain_idx].push_back(i); break;
  	 default: volume_domain_map[domain_idx].push_back(i); break;
        }
    }
  }
  // -------------------------------------------------
  // 2. renum nodes
  // -------------------------------------------------
  // permut: first vertex, then edge, faces and inernal nodes, by increasing subgeo_dim 
  vector<size_t> old2new_inode (nnode, size_t(-1));
  size_type inode = 0;
  for (size_type subgeo_variant = 0; subgeo_variant < reference_element::last_variant_by_dimension(map_dim); subgeo_variant++) {
    for (size_type k = 0; k < nnode; k++) {
      if (node_subgeo_variant [k] != subgeo_variant) continue;
      old2new_inode[k] = inode++;
    }
  }
  // 2.3. inv permut
  vector<size_t> new2old_inode (nnode, size_t(-1));
  for (size_type k = 0; k < nnode; k++) {
    new2old_inode[old2new_inode[k]] = k;
  }
  // ----------------------------------
  // 3. output geo
  // ---------------------------------- 
  size_type version = 4;
  static const char* geo_variant_name [reference_element::max_variant] = {
    "points",
    "edges",
    "triangles",
    "quadrangles",
    "tetrahedra",
    "prisms",
    "hexahedra"
  };
  out << setprecision(numeric_limits<Float>::digits10) 
      << "#!geo" << endl
      << endl
      << "mesh" << endl
      << version << endl
      << "header" << endl
      << " dimension\t" << dim << endl;
  if (sys_coord_name != "cartesian") {
      out << " coordinate_system " << sys_coord_name << endl;
  }
  if (order != 1) {
      out << " order\t\t" << order << endl;
  }
  out << " nodes\t\t" << nnode << endl;
  
  if (map_dim > 0) {
    for (size_type variant = reference_element::first_variant_by_dimension(map_dim);
                   variant < reference_element:: last_variant_by_dimension(map_dim); variant++) {
      if (size_by_variant[variant] > 0) {
        out << " " << geo_variant_name [variant] << "\t" << size_by_variant[variant] << endl;
      }
    }
  }
  out << "end header" << endl
      << endl;
  for (size_type k = 0; k < nnode; k++) {
    node[new2old_inode[k]].put (out, dim);
    cout << endl;
  }
  out << endl;
  for (size_type variant = reference_element::first_variant_by_dimension(map_dim);
                 variant < reference_element:: last_variant_by_dimension(map_dim); variant++) {
    for (size_type i = 0; i < nelement; i++) {
      if (reference_element::variant(element_name[i]) != variant) continue;
      if (element_name[i] != 'e' || element_order[i] > 1) {
        out << element_name[i] << "\t";
      }
      if (element_order[i] > 1) {
        out << "p" << element_order[i] << " ";
      }
      for (size_type iloc = 0, nloc = element[i].size(); iloc < nloc; iloc++) {
        out << old2new_inode[element[i][iloc]];
        if (iloc != nloc - 1) out << " ";
      }
      cout << endl;
    }
  }
  out << endl;
  // ----------------------------------
  // 4. output domains
  // ---------------------------------- 
  put_domain (out, map_dim, 0, point_domain_map,  phys, element_name, element_order, element, old2new_inode);
  put_domain (out, map_dim, 1, edge_domain_map,   phys, element_name, element_order, element, old2new_inode);
  put_domain (out, map_dim, 2, face_domain_map,   phys, element_name, element_order, element, old2new_inode);
  put_domain (out, map_dim, 3, volume_domain_map, phys, element_name, element_order, element, old2new_inode);
}
void usage()
{
  cerr << "msh2geo: usage:" << endl
       << "msh2geo [-rz|-zr] < in.msh | geo -upgrade - > out.geo" << endl;
  exit (1);
}
int main (int argc, char**argv) {
  // ----------------------------
  // scan the command line
  // ----------------------------
  std::string sys_coord_name = "cartesian";
  for (int i = 1; i < argc; i++) {
         if (strcmp (argv[i], "-rz") == 0)   sys_coord_name = "rz";
    else if (strcmp (argv[i], "-zr") == 0)   sys_coord_name = "zr";
    else {
            cerr << "field: unknown option `" << argv[i] << "'" << endl;
            usage();
    }
  }
  msh2geo (cin, cout, sys_coord_name);
}
