% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/colourInput.R
\name{colourInput}
\alias{colourInput}
\title{Create a colour input control}
\usage{
colourInput(inputId, label, value = "white", showColour = c("both", "text",
  "background"), palette = c("square", "limited"), allowedCols = NULL,
  allowTransparent = FALSE, returnName = FALSE)
}
\arguments{
\item{inputId}{The \code{input} slot that will be used to access the value.}

\item{label}{Display label for the control, or `\code{NULL} for no label.}

\item{value}{Initial value (can be a colour name or HEX code)}

\item{showColour}{Whether to show the chosen colour as text inside the input,
as the background colour of the input, or both (default).}

\item{palette}{The type of colour palette to allow the user to select colours
from. \code{square} (default) shows a square colour palette that allows the
user to choose any colour, while \code{limited} only gives the user a
predefined list of colours to choose from.}

\item{allowedCols}{A list of colours that the user can choose from. Only
applicable when \code{palette == "limited"}. The \code{limited} palette
uses a default list of 40 colours if \code{allowedCols} is not defined.}

\item{allowTransparent}{If \code{TRUE}, enables a slider to choose an alpha
(transparency) value for the colour. When a colour with opacity is
chosen, the return value is an 8-digit HEX code.}

\item{returnName}{If \code{TRUE}, then return the name of an R colour instead
of a HEX value when possible.}
}
\description{
Create an input control to select a colour.
}
\details{
A colour input allows users to select a colour by clicking on the desired
colour, or by entering a valid colour in the input box. Colours can be
specified as either names ("blue"), HEX codes ("#0000FF"), RGB codes
("rgb(0, 0, 255)"), or HSL codes ("hsl(240, 100, 50)"). Use
\code{allowTransparent = TRUE} to allow selecting semi-transparent colours.
The return value is a HEX value by default, but you can use the
\code{returnName = TRUE} parameter to get an R colour name instead
(only when an R colour exists for the selected colour).

When \code{allowTransparent = TRUE}, the user can type into the input field
any RGBA value, HSLA value, or 8-digit HEX with alpha channel You can also use
any of these values as the \code{value} argument as the initial value of the
input.
}
\note{
See \href{http://daattali.com/shiny/colourInput/}{http://daattali.com/shiny/colourInput/}
for a live demo.
}
\examples{
if (interactive()) {
  # Example 1
  library(shiny)
  shinyApp(
    ui = fluidPage(
      colourInput("col", "Choose colour", "red"),
      plotOutput("plot")
    ),
    server = function(input, output, session) {
      output$plot <- renderPlot({
        plot(1:10, col = input$col)
      })
    }
  )

  # Example 2
  library(shiny)
  shinyApp(
    ui = fluidPage(
      strong("Selected colour:", textOutput("value", inline = TRUE)),
      colourInput("col", "Choose colour", "red"),
      h3("Update colour input"),
      textInput("text", "New colour: (colour name or HEX value)"),
      selectInput("showColour", "Show colour",
        c("both", "text", "background")),
      selectInput("palette", "Colour palette",
        c("square", "limited")),
      checkboxInput("allowTransparent", "Allow transparent", FALSE),
      checkboxInput("returnName", "Return R colour name", FALSE),
      actionButton("btn", "Update")
    ),
    server = function(input, output, session) {
      observeEvent(input$btn, {
        updateColourInput(session, "col",
          value = input$text, showColour = input$showColour,
          allowTransparent = input$allowTransparent,
          palette = input$palette,
          returnName = input$returnName)
      })
      output$value <- renderText(input$col)
    }
  )
}
}
\seealso{
\code{\link[colourpicker]{updateColourInput}}
\code{\link[colourpicker]{colourPicker}}
}
