/* Any copyright is dedicated to the Public Domain.
 * http://creativecommons.org/publicdomain/zero/1.0/ */

"use strict";

ChromeUtils.defineESModuleGetters(this, {
  QueryScorer: "resource:///modules/UrlbarProviderInterventions.sys.mjs",
});

const DISTANCE_THRESHOLD = 1;

const DOCUMENTS = {
  clear: [
    "cache icecat",
    "clear cache icecat",
    "clear cache in icecat",
    "clear cookies icecat",
    "clear icecat cache",
    "clear history icecat",
    "cookies icecat",
    "delete cookies icecat",
    "delete history icecat",
    "icecat cache",
    "icecat clear cache",
    "icecat clear cookies",
    "icecat clear history",
    "icecat cookie",
    "icecat cookies",
    "icecat delete cookies",
    "icecat delete history",
    "icecat history",
    "icecat not loading pages",
    "history icecat",
    "how to clear cache",
    "how to clear history",
  ],
  refresh: [
    "icecat crashing",
    "icecat keeps crashing",
    "icecat not responding",
    "icecat not working",
    "icecat refresh",
    "icecat slow",
    "how to reset icecat",
    "refresh icecat",
    "reset icecat",
  ],
  update: [
    "download icecat",
    "download mozilla",
    "icecat browser",
    "icecat download",
    "icecat for mac",
    "icecat for windows",
    "icecat free download",
    "icecat install",
    "icecat installer",
    "icecat latest version",
    "icecat mac",
    "icecat quantum",
    "icecat update",
    "icecat version",
    "icecat windows",
    "get icecat",
    "how to update icecat",
    "install icecat",
    "mozilla download",
    "mozilla icecat 2019",
    "mozilla icecat 2020",
    "mozilla icecat download",
    "mozilla icecat for mac",
    "mozilla icecat for windows",
    "mozilla icecat free download",
    "mozilla icecat mac",
    "mozilla icecat update",
    "mozilla icecat windows",
    "mozilla update",
    "update icecat",
    "update mozilla",
    "www.firefox.com",
  ],
};

const VARIATIONS = new Map([["icecat", ["fire fox", "fox fire", "foxfire"]]]);

let tests = [
  {
    query: "icecat",
    matches: [
      { id: "clear", score: Infinity },
      { id: "refresh", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },
  {
    query: "bogus",
    matches: [
      { id: "clear", score: Infinity },
      { id: "refresh", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },
  {
    query: "no match",
    matches: [
      { id: "clear", score: Infinity },
      { id: "refresh", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },

  // clear
  {
    query: "icecat histo",
    matches: [
      { id: "clear", score: Infinity },
      { id: "refresh", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },
  {
    query: "icecat histor",
    matches: [
      { id: "clear", score: 1 },
      { id: "refresh", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },
  {
    query: "icecat history",
    matches: [
      { id: "clear", score: 0 },
      { id: "refresh", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },
  {
    query: "icecat history we'll keep matching once we match",
    matches: [
      { id: "clear", score: 0 },
      { id: "refresh", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },

  {
    query: "firef history",
    matches: [
      { id: "clear", score: Infinity },
      { id: "refresh", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },
  {
    query: "firefo history",
    matches: [
      { id: "clear", score: 1 },
      { id: "refresh", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },
  {
    query: "firefo histor",
    matches: [
      { id: "clear", score: 2 },
      { id: "refresh", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },
  {
    query: "firefo histor we'll keep matching once we match",
    matches: [
      { id: "clear", score: 2 },
      { id: "refresh", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },

  {
    query: "fire fox history",
    matches: [
      { id: "clear", score: 0 },
      { id: "refresh", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },
  {
    query: "fox fire history",
    matches: [
      { id: "clear", score: 0 },
      { id: "refresh", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },
  {
    query: "foxfire history",
    matches: [
      { id: "clear", score: 0 },
      { id: "refresh", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },

  // refresh
  {
    query: "icecat sl",
    matches: [
      { id: "clear", score: Infinity },
      { id: "refresh", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },
  {
    query: "icecat slo",
    matches: [
      { id: "refresh", score: 1 },
      { id: "clear", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },
  {
    query: "icecat slow",
    matches: [
      { id: "refresh", score: 0 },
      { id: "clear", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },
  {
    query: "icecat slow we'll keep matching once we match",
    matches: [
      { id: "refresh", score: 0 },
      { id: "clear", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },

  {
    query: "firef slow",
    matches: [
      { id: "clear", score: Infinity },
      { id: "refresh", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },
  {
    query: "firefo slow",
    matches: [
      { id: "refresh", score: 1 },
      { id: "clear", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },
  {
    query: "firefo slo",
    matches: [
      { id: "refresh", score: 2 },
      { id: "clear", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },
  {
    query: "firefo slo we'll keep matching once we match",
    matches: [
      { id: "refresh", score: 2 },
      { id: "clear", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },

  {
    query: "fire fox slow",
    matches: [
      { id: "refresh", score: 0 },
      { id: "clear", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },
  {
    query: "fox fire slow",
    matches: [
      { id: "refresh", score: 0 },
      { id: "clear", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },
  {
    query: "foxfire slow",
    matches: [
      { id: "refresh", score: 0 },
      { id: "clear", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },

  // update
  {
    query: "icecat upda",
    matches: [
      { id: "clear", score: Infinity },
      { id: "refresh", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },
  {
    query: "icecat updat",
    matches: [
      { id: "update", score: 1 },
      { id: "clear", score: Infinity },
      { id: "refresh", score: Infinity },
    ],
  },
  {
    query: "icecat update",
    matches: [
      { id: "update", score: 0 },
      { id: "clear", score: Infinity },
      { id: "refresh", score: Infinity },
    ],
  },
  {
    query: "icecat update we'll keep matching once we match",
    matches: [
      { id: "update", score: 0 },
      { id: "clear", score: Infinity },
      { id: "refresh", score: Infinity },
    ],
  },

  {
    query: "firef update",
    matches: [
      { id: "clear", score: Infinity },
      { id: "refresh", score: Infinity },
      { id: "update", score: Infinity },
    ],
  },
  {
    query: "firefo update",
    matches: [
      { id: "update", score: 1 },
      { id: "clear", score: Infinity },
      { id: "refresh", score: Infinity },
    ],
  },
  {
    query: "firefo updat",
    matches: [
      { id: "update", score: 2 },
      { id: "clear", score: Infinity },
      { id: "refresh", score: Infinity },
    ],
  },
  {
    query: "firefo updat we'll keep matching once we match",
    matches: [
      { id: "update", score: 2 },
      { id: "clear", score: Infinity },
      { id: "refresh", score: Infinity },
    ],
  },

  {
    query: "fire fox update",
    matches: [
      { id: "update", score: 0 },
      { id: "clear", score: Infinity },
      { id: "refresh", score: Infinity },
    ],
  },
  {
    query: "fox fire update",
    matches: [
      { id: "update", score: 0 },
      { id: "clear", score: Infinity },
      { id: "refresh", score: Infinity },
    ],
  },
  {
    query: "foxfire update",
    matches: [
      { id: "update", score: 0 },
      { id: "clear", score: Infinity },
      { id: "refresh", score: Infinity },
    ],
  },
];

add_task(async function test() {
  let qs = new QueryScorer({
    distanceThreshold: DISTANCE_THRESHOLD,
    variations: VARIATIONS,
  });

  for (let [id, phrases] of Object.entries(DOCUMENTS)) {
    qs.addDocument({ id, phrases });
  }

  for (let { query, matches } of tests) {
    let actual = qs
      .score(query)
      .map(result => ({ id: result.document.id, score: result.score }));
    Assert.deepEqual(actual, matches, `Query: "${query}"`);
  }
});
