/* Copyright (C) 2023 Wildfire Games.
 * This file is part of 0 A.D.
 *
 * 0 A.D. is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * 0 A.D. is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with 0 A.D.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef INCLUDED_MAPREADER
#define INCLUDED_MAPREADER

#include "MapIO.h"

#include "graphics/LightEnv.h"
#include "ps/CStr.h"
#include "ps/FileIo.h"
#include "scriptinterface/ScriptTypes.h"
#include "simulation2/system/Entity.h"

#include <memory>

class CTerrain;
class WaterManager;
class SkyManager;
class CLightEnv;
class CCinemaManager;
class CPostprocManager;
class CTriggerManager;
class CSimulation2;
class CSimContext;
class CTerrainTextureEntry;
class CGameView;
class CXMLReader;
class ScriptContext;
class ScriptInterface;

class CMapReader : public CMapIO
{
	friend class CXMLReader;

public:
	// constructor
	CMapReader();
	~CMapReader();

	// LoadMap: try to load the map from given file; reinitialise the scene to new data if successful
	void LoadMap(const VfsPath& pathname, const ScriptContext& cx, JS::HandleValue settings, CTerrain*, WaterManager*, SkyManager*, CLightEnv*, CGameView*,
		CCinemaManager*, CTriggerManager*, CPostprocManager* pPostproc, CSimulation2*, const CSimContext*,
	        int playerID, bool skipEntities);

	void LoadRandomMap(const CStrW& scriptFile, const ScriptContext& cx, JS::HandleValue settings, CTerrain*, WaterManager*, SkyManager*, CLightEnv*, CGameView*, CCinemaManager*, CTriggerManager*, CPostprocManager* pPostproc_, CSimulation2*, int playerID);

private:
	// Load script settings for use by scripts
	int LoadScriptSettings();

	// load player settings only
	int LoadPlayerSettings();

	// load map settings only
	int LoadMapSettings();

	// UnpackTerrain: unpack the terrain from the input stream
	int UnpackTerrain();
	// UnpackCinema: unpack the cinematic tracks from the input stream
	int UnpackCinema();

	// ApplyData: take all the input data, and rebuild the scene from it
	int ApplyData();
	int ApplyTerrainData();

	// read some misc data from the XML file
	int ReadXML();

	// read entity data from the XML file
	int ReadXMLEntities();

	// Copy random map settings over to sim
	int LoadRMSettings();

	// Generate random map
	int StartMapGeneration(const CStrW& scriptFile);
	int PollMapGeneration();

	// Parse script data into terrain
	int ParseTerrain();

	// Parse script data into entities
	int ParseEntities();

	// Parse script data into environment
	int ParseEnvironment();

	// Parse script data into camera
	int ParseCamera();


	// size of map
	ssize_t m_PatchesPerSide{0};
	// heightmap for map
	std::vector<u16> m_Heightmap;
	// list of terrain textures used by map
	std::vector<CTerrainTextureEntry*> m_TerrainTextures;
	// tile descriptions for each tile
	std::vector<STileDesc> m_Tiles;
	// lightenv stored in file
	CLightEnv m_LightEnv;
	// startup script
	CStrW m_Script;

	// random map data
	JS::PersistentRootedValue m_ScriptSettings;
	JS::PersistentRootedValue m_MapData;

	struct GeneratorState;
	std::unique_ptr<GeneratorState> m_GeneratorState;

	CFileUnpacker unpacker;
	CTerrain* pTerrain;
	WaterManager* pWaterMan;
	SkyManager* pSkyMan;
	CPostprocManager* pPostproc;
	CLightEnv* pLightEnv;
	CGameView* pGameView;
	CCinemaManager* pCinema;
	CTriggerManager* pTrigMan;
	CSimulation2* pSimulation2;
	const CSimContext* pSimContext;
	int m_PlayerID;
	bool m_SkipEntities;
	VfsPath filename_xml;
	bool only_xml;
	u32 file_format_version;
	entity_id_t m_StartingCameraTarget;
	CVector3D m_StartingCamera;

	// UnpackTerrain generator state
	// It's important to initialize it to 0 - resets generator state
	size_t cur_terrain_tex{0};
	size_t num_terrain_tex;

	CXMLReader* xml_reader{nullptr};
};

/**
 * A restricted map reader that returns various summary information
 * for use by scripts (particularly the GUI).
 */
class CMapSummaryReader
{
public:
	/**
	 * Try to load a map file.
	 * @param pathname Path to .pmp or .xml file
	 */
	PSRETURN LoadMap(const VfsPath& pathname);

	/**
	 * Returns a value of the form:
	 * @code
	 * {
	 *   "settings": { ... contents of the map's <ScriptSettings> ... }
	 * }
	 * @endcode
	 */
	void GetMapSettings(const ScriptInterface& scriptInterface, JS::MutableHandleValue);

private:
	CStr m_ScriptSettings;
};

#endif
