/****************************************************************************
**
*W  precord.c                   GAP source                   Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions for plain records.
**
**  A plain record  with <n>  components is stored  as  a bag  with 2*<n>+2
**  entries.  The odd entries are the record  names of the components and the
**  even entries are the corresponding values.
**  The first pointer in the bag is reserved for the pointer to the type
**  object if the record becomes a component object. The second slot
**  in the bag is the number <n> of components as a C int.
**  Normally, the rnams are sorted such that we can do a binary lookup.
**  However, we allow for lazy sorting. This is done as follows: New
**  components are always appended (record bags grow by a factor of 5/4
**  if needed as plists do), already sorted rnams are stored by storing
**  their negative value to indicate sortedness. The new entries will have
**  positive rnams and can thus be distinguished. Every read access will 
**  clean up the mess by sorting the new part and then merging the two
**  sorted areas. After that, all rnams are negative indicating sortedness.
**
*/
#include        <stdlib.h>              /* for qsort */
#include        "system.h"              /* system dependent part           */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */
#include        "calls.h"               /* generic call mechanism          */
#include        "opers.h"               /* generic operations              */

#include        "ariths.h"              /* basic arithmetic                */
#include        "records.h"             /* generic records                 */
#include        "lists.h"               /* generic lists                   */

#include        "bool.h"                /* booleans                        */

#include        "precord.h"             /* plain records                   */

#include        "plist.h"               /* plain lists                     */
#include        "string.h"              /* strings                         */

#include        "saveload.h"            /* saving and loading              */


/****************************************************************************
**

*F * * * * * * * * * standard functions for plain records * * * * * * * * * *
*/


/****************************************************************************
**

*F  TypePRec( <rec> ) . . . . . . . . . . . . . . . .  kind of a plain record
**
**  'TypePRec' returns the kind of the plain record <rec>.
**
**  'TypePRec' is the function in 'TypeObjFuncs' for plain records.
*/
libGAP_Obj libGAP_TYPE_PREC_MUTABLE;

libGAP_Obj libGAP_TypePRecMut (
    libGAP_Obj                 prec )
{
    return libGAP_TYPE_PREC_MUTABLE;
}


libGAP_Obj libGAP_TYPE_PREC_IMMUTABLE;

libGAP_Obj libGAP_TypePRecImm (
    libGAP_Obj                 prec )
{
    return libGAP_TYPE_PREC_IMMUTABLE;
}


/****************************************************************************
**
*F  IsMutablePRecYes( <rec> ) . . . . . . mutability test for mutable records
*F  IsMutablePRecNo( <rec> )  . . . . . mutability test for immutable records
**
**  'IsMutablePRecYes' simply returns 1.  'IsMutablePRecNo' simply returns 0.
**  Note that we can decide from the type number whether  a record is mutable
**  or immutable.
**
**  'IsMutablePRecYes'  is the function   in 'IsMutableObjFuncs'  for mutable
**  records.   'IsMutablePRecNo' is  the function  in 'IsMutableObjFuncs' for
**  immutable records.
*/
libGAP_Int libGAP_IsMutablePRecYes (
    libGAP_Obj                 rec )
{
    return 1;
}

libGAP_Int libGAP_IsMutablePRecNo (
    libGAP_Obj                 rec )
{
    return 0;
}


/****************************************************************************
**
*F  IsCopyablePRecYes( <rec> )  . . . . . . . .  copyability test for records
**
**  'IsCopyablePRec' simply returns 1.  Note that all records are copyable.
**
**  'IsCopyablePRec' is the function in 'IsCopyableObjFuncs' for records.
*/
libGAP_Int libGAP_IsCopyablePRecYes (
    libGAP_Obj                 rec )
{
    return 1;
}

/****************************************************************************
**
*F  NEW_PREC( <len> ) . . . . . . . . . . . . . . . . make a new plain record
**
**  'NEW_PREC' returns a new plain record with room for <len> components,
**  already set to length len.
*/
libGAP_Obj libGAP_NEW_PREC(libGAP_UInt len)
{
    libGAP_Obj o = libGAP_NewBag( libGAP_T_PREC, (len) * 2*sizeof(libGAP_Obj) + 2*sizeof(libGAP_Obj) ); 
    libGAP_SET_LEN_PREC(o,0);
    return o;
}

/****************************************************************************
**
*F  GrowPRec( <rec>, <len> ) . . . .  make sure <rec> can hold <len> components
**
**  Returns 0 if nothing changed and 1 if enlarged.
*/
libGAP_Int             libGAP_GrowPRec (
    libGAP_Obj                 rec,
    libGAP_UInt                need )
{
    libGAP_UInt                newsize, want, good;     

    /* check if big enough */
    want = (2*need+2)*sizeof(libGAP_Obj);
    if (libGAP_SIZE_OBJ(rec) >= want) return 0L;

    
    /* find out how large the bag should become at least                   */
    good = ((5 * libGAP_LEN_PREC(rec) + 3)/4 + 1) * 2 * sizeof(libGAP_Obj) + 2; 

    /* but maybe we need more                                              */
    newsize = (want < good) ? good : want;

    /* resize the plain list                                               */
    libGAP_ResizeBag( rec, newsize );

    return 1L;
}

/****************************************************************************
**
*F  CopyPRec( <rec> ) . . . . . . . . . . . . . . . . . . copy a plain record
*F  CleanPRec( <rec> )  . . . . . . . . . . . . . . . clean up a plain record
**
**  'CopyPRec' returns a structural (deep) copy  of the record <rec>, i.e., a
**  recursive copy that preserves the structure.
**
**  If <rec>  has not yet  been  copied, it makes a   copy, leaves a  forward
**  pointer to the copy in  the first entry   of the record, where the  first
**  record name usually resides,  and copies all the  entries.  If the record
**  has alread been copied, it returns the value of the forwarding pointer.
**
**  'CopyPRec' is the function in 'TabCopyObj' for records.
**
**  'CleanPRec' removes the  mark and the forwarding  pointer from the record
**  <rec>.
**
**  'CleanPRec' is the function in 'TabCleanObj' for records.
*/
libGAP_Obj libGAP_CopyPRec (
    libGAP_Obj                 rec,
    libGAP_Int                 mut )
{
    libGAP_Obj                 copy;           /* copy, result                    */
    libGAP_Obj                 tmp;            /* temporary variable              */
    libGAP_UInt                i;              /* loop variable                   */

    /* don't change immutable objects                                      */
    if ( ! libGAP_IS_MUTABLE_OBJ(rec) ) {
        return rec;
    }

    /* if an empty record has not yet been copied                          */
    if ( libGAP_LEN_PREC(rec) == 0 ) {

        /* make a copy                                                     */
        if ( mut ) {
            copy = libGAP_NewBag( libGAP_TNUM_OBJ(rec), libGAP_SIZE_OBJ(rec) );
        }
        else {
            copy = libGAP_NewBag( libGAP_IMMUTABLE_TNUM(libGAP_TNUM_OBJ(rec)), libGAP_SIZE_OBJ(rec) );
        }

        /* leave a forwarding pointer                                      */
        libGAP_ResizeBag( rec, libGAP_SIZE_OBJ(rec) + sizeof(libGAP_Obj) );
        libGAP_SET_RNAM_PREC( rec, 1, (libGAP_UInt)copy );
        libGAP_CHANGED_BAG( rec );

        /* now it is copied                                                */
        libGAP_RetypeBag( rec, libGAP_TNUM_OBJ(rec) + libGAP_COPYING );
    }

    /* if the record has not yet been copied                               */
    else {

        /* make a copy                                                     */
        if ( mut ) {
            copy = libGAP_NewBag( libGAP_TNUM_OBJ(rec), libGAP_SIZE_OBJ(rec) );
        }
        else {
            copy = libGAP_NewBag( libGAP_IMMUTABLE_TNUM(libGAP_TNUM_OBJ(rec)), libGAP_SIZE_OBJ(rec) );
        }
        libGAP_SET_RNAM_PREC( copy, 1, libGAP_GET_RNAM_PREC( rec, 1 ) );

        /* leave a forwarding pointer                                      */
        libGAP_SET_RNAM_PREC( rec, 1, (libGAP_UInt)copy );
        libGAP_CHANGED_BAG( rec );

        /* now it is copied                                                */
        libGAP_RetypeBag( rec, libGAP_TNUM_OBJ(rec) + libGAP_COPYING );

        /* copy the subvalues                                              */
        tmp = libGAP_COPY_OBJ( libGAP_GET_ELM_PREC( rec, 1 ), mut );
        libGAP_SET_LEN_PREC( copy, libGAP_LEN_PREC(rec) );
        libGAP_SET_ELM_PREC( copy, 1, tmp );
        libGAP_CHANGED_BAG( copy );
        for ( i = 2; i <= libGAP_LEN_PREC(copy); i++ ) {
            libGAP_SET_RNAM_PREC( copy, i, libGAP_GET_RNAM_PREC( rec, i ) );
            tmp = libGAP_COPY_OBJ( libGAP_GET_ELM_PREC( rec, i ), mut );
            libGAP_SET_ELM_PREC( copy, i, tmp );
            libGAP_CHANGED_BAG( copy );
        }

    }

    /* return the copy                                                     */
    return copy;
}

libGAP_Obj libGAP_CopyPRecCopy (
    libGAP_Obj                 rec,
    libGAP_Int                 mut )
{
    return (libGAP_Obj)libGAP_GET_RNAM_PREC( rec, 1 );
}

void libGAP_CleanPRec (
    libGAP_Obj                 rec )
{
}

void libGAP_CleanPRecCopy (
    libGAP_Obj                 rec )
{
    libGAP_UInt                i;              /* loop variable                   */

    /* empty record                                                        */
    if ( libGAP_LEN_PREC(rec) == 0 ) {

        /* remove the forwarding pointer                                   */
        libGAP_ResizeBag( rec, libGAP_SIZE_OBJ(rec) - sizeof(libGAP_Obj) );

        /* now it is cleaned                                               */
        libGAP_RetypeBag( rec, libGAP_TNUM_OBJ(rec) - libGAP_COPYING );
    }

    /* nonempty record                                                     */
    else {

        /* remove the forwarding pointer                                   */
        libGAP_SET_RNAM_PREC( rec, 1, libGAP_GET_RNAM_PREC( libGAP_GET_RNAM_PREC( rec, 1 ), 1 ) );

        /* now it is cleaned                                               */
        libGAP_RetypeBag( rec, libGAP_TNUM_OBJ(rec) - libGAP_COPYING );

        /* clean the subvalues                                             */
        libGAP_CLEAN_OBJ( libGAP_GET_ELM_PREC( rec, 1 ) );
        for ( i = 2; i <= libGAP_LEN_PREC(rec); i++ ) {
            libGAP_CLEAN_OBJ( libGAP_GET_ELM_PREC( rec, i ) );
        }
    }
}

/****************************************************************************
**
*F  MakeImmutablePRec( <rec> )
*/

void libGAP_MakeImmutablePRec( libGAP_Obj rec)
{
  libGAP_UInt len;
  libGAP_UInt i;
  len = libGAP_LEN_PREC( rec );
  for ( i = 1; i <= len; i++ )
    libGAP_MakeImmutable(libGAP_GET_ELM_PREC(rec,i));
  libGAP_RetypeBag(rec, libGAP_IMMUTABLE_TNUM(libGAP_TNUM_OBJ(rec)));
}


/****************************************************************************
 * FindPRec( <rec>, <rnam>, <pos>, <cleanup> )  
 *   . . . . . . . . . . . . . . . . . find a component name by binary search
 *
 * Searches rnam in rec, sets pos to the position where it is found (return
 * value 1) or where it should be inserted if it is not found (return val 0).
 * If cleanup is nonzero, a dirty record is automatically cleaned up.
 * If cleanup is 0, this does not happen.
 */

libGAP_UInt libGAP_FindPRec( libGAP_Obj rec, libGAP_UInt rnam, libGAP_UInt *pos, int cleanup )
{
    /* This only assumes that the rnam values in the record are sorted! */
    libGAP_UInt i;
    libGAP_Int rnam2;
    libGAP_UInt low = 1;
    libGAP_UInt high;

    high = libGAP_LEN_PREC(rec);
    if (high > 0 && (libGAP_Int) (libGAP_GET_RNAM_PREC(rec,high)) > 0) {  
        /* DIRTY! Not everything sorted! */
        if (cleanup) {
            libGAP_SortPRecRNam(rec,0);
            /* Note that this does not change the length and it cannot 
             * trigger a garbage collection if cleanup is 1! 
             * We do not want record accesses to trigger garbage
             * collections! */
        } else {
            /* We are not allowed to cleanup, so we live with it, we
             * first try to find rnam in the mess at the end, then
             * fall back to binary search: */
            i = high;
            while (i >= 1) {
                rnam2 = (libGAP_Int) (libGAP_GET_RNAM_PREC(rec,i));
                if (rnam == rnam2) {
                    *pos = i;
                    return 1;
                }
                if (rnam2 < 0) { /* reached the sorted area! */
                    high = i;  /* will be incremented by 1 */
                    break;
                }
                i--;
            }
            if (i == 0) return 0;
            /* Here, high is the number of the sorted entries. */
        }
    }
    high++;
    while (low < high) {
        i = (low + high) >> 1;   /* we always have low <= i < high */
        rnam2 = -(libGAP_Int)(libGAP_GET_RNAM_PREC( rec, i ));
        if (rnam2 < rnam) low = i+1;
        else if (rnam2 > rnam) high = i;
        else {
            /* found! */
            *pos = i;
            return 1;
        }
    }
    /* Now low == high and we did not find it. */
    *pos = low;
    return 0;
}

/****************************************************************************
**
*F  IsbPRec( <rec>, <rnam> )  . . . . test for an element from a plain record
**
**  'IsbPRec' returns 1 if the record <rec> has a component with  the  record
**  name <rnam>, and 0 otherwise.
*/
libGAP_Int libGAP_IsbPRec (
    libGAP_Obj                 rec,
    libGAP_UInt                rnam )
{
    libGAP_UInt                i;              /* loop variable                   */

    return libGAP_FindPRec(rec,rnam,&i,1);
}


/****************************************************************************
**
*F  ElmPRec( <rec>, <rnam> )  . . . . . select an element from a plain record
**
**  'ElmPRec' returns the element, i.e., the value of the component, with the
**  record name <rnam> in  the plain record <rec>.   An error is signalled if
**  <rec> has no component with record name <rnam>.
*/
libGAP_Obj libGAP_ElmPRec (
    libGAP_Obj                 rec,
    libGAP_UInt                rnam )
{
    libGAP_UInt                i;              /* loop variable                   */

    if (libGAP_FindPRec(rec,rnam,&i,1))
        return libGAP_GET_ELM_PREC( rec, i );
    else {
        libGAP_ErrorReturnVoid(
            "Record: '<rec>.%s' must have an assigned value",
            (libGAP_Int)libGAP_NAME_RNAM(rnam), 0L,
            "you can 'return;' after assigning a value" );
        return libGAP_ELM_REC( rec, rnam );
    }
}


/****************************************************************************
**
*F  UnbPRec( <rec>, <rnam> )  . unbind a record component from a plain record
**
**  'UnbPRec'  removes the record component  with the record name <rnam> from
**  the record <rec>.
*/
void libGAP_UnbPRec (
    libGAP_Obj                 rec,
    libGAP_UInt                rnam )
{
    libGAP_UInt                len;            /* length of <rec>                 */
    libGAP_UInt                i;              /* loop variable                   */

    if (libGAP_FindPRec( rec, rnam, &i, 1 )) {
        /* otherwise move everything forward                               */
        len = libGAP_LEN_PREC( rec );
        for ( ; i < len; i++ ) {
            libGAP_SET_RNAM_PREC( rec, i, libGAP_GET_RNAM_PREC( rec, i+1 ) );
            libGAP_SET_ELM_PREC(  rec, i, libGAP_GET_ELM_PREC(  rec, i+1 ) );
        }
        libGAP_SET_RNAM_PREC( rec, len, 0 );
        libGAP_SET_ELM_PREC( rec, len, 0L );


        /* resize the record                                               */
        libGAP_SET_LEN_PREC(rec,libGAP_LEN_PREC(rec)-1);

    } else
        /* do nothing if no such component exists                          */
        return;
}

void            libGAP_UnbPRecImm (
    libGAP_Obj                 rec,
    libGAP_UInt                rnam )
{
    libGAP_ErrorReturnVoid(
        "Record Unbind: <rec> must be a mutable record",
        0L, 0L,
        "you can 'return;' and ignore the unbind" );
}


/****************************************************************************
**
*F  AssPRec( <rec>, <rnam>, <val> ) . . . . . . . .  assign to a plain record
**
**  'AssPRec' assigns the value <val> to the record component with the record
**  name <rnam> in the plain record <rec>.
*/
void libGAP_AssPRec (
    libGAP_Obj                 rec,
    libGAP_UInt                rnam,
    libGAP_Obj                 val )
{
    libGAP_UInt                len;            /* length of <rec>                 */
    libGAP_UInt                i;              /* loop variable                   */

    /* get the length of the record                                        */
    len = libGAP_LEN_PREC( rec );

    if (len % 1000 == 0) {   /* A hack to occasionally do some cleanup! */
        libGAP_SortPRecRNam(rec,0);
    }

    if (!libGAP_FindPRec( rec, rnam, &i, 0 )) {
        /* No cleanup allowed here to allow for multiple assignments! */
        /* extend the record if no such component exists                   */
        len++;
        libGAP_GrowPRec(rec, len);
        i = len;
        libGAP_SET_RNAM_PREC( rec, i, rnam );
        libGAP_SET_LEN_PREC(rec,len);
    }
    /* assign the value to the component                                   */
    libGAP_SET_ELM_PREC( rec, i, val );
    libGAP_CHANGED_BAG( rec );
}

void            libGAP_AssPRecImm (
    libGAP_Obj                 rec,
    libGAP_UInt                rnam,
    libGAP_Obj                 val )
{
    libGAP_ErrorReturnVoid(
        "Records Assignment: <rec> must be a mutable record",
        0L, 0L,
        "you can 'return;' and ignore the assignment" );
}

/****************************************************************************
**
*F  PrintPRec( <rec> )  . . . . . . . . . . . . . . . . . . .  print a record
**
**  'PrintRec' prints the plain record <rec>.
*/
extern libGAP_Obj libGAP_PrintObjOper;

void libGAP_PrintPRec (
    libGAP_Obj                 rec )
{
    libGAP_DoOperation1Args( libGAP_PrintObjOper, rec );
}



/****************************************************************************
**
*F  SortPRecRNam(<rec>, <inplace>) . . . . . . . sort the Rnams of the record
**
**  This is needed after the components of a record have been assigned
**  in not necessarily sorted order in the kernel. It is automatically
**  called on the first read access if necessary. See the top of "precord.c"
**  for a comment on lazy sorting.
**  If inplace is 1 then a slightly slower algorithm is used of
**  which we know that it does not produce garbage collections.
**  If inplace is 0 a garbage collection may be triggered.
**
*/
static int libGAP_PrecComparer(const void *a, const void *b)
{
    const libGAP_UInt *aa = (const libGAP_UInt *) a;
    const libGAP_UInt *bb = (const libGAP_UInt *) b;
    if (*aa < *bb) return -1;
    else if (*aa == *bb) return 0;
    else return 1;
}

void libGAP_SortPRecRNam ( 
    libGAP_Obj                 rec, int inplace )
{
    libGAP_UInt len = libGAP_LEN_PREC(rec);
    libGAP_UInt i,j,k,save;
    int issorted = 1;
    libGAP_Obj space;
    libGAP_Obj tmp;

    /* Nothing has to be done if it is already sorted: */
    if ( len == 0 || (libGAP_Int) (libGAP_GET_RNAM_PREC(rec,len)) < 0) return;

    /* First find the "unsorted part" and check whether it is sorted! */
    for (i = len-1;i >= 1 && (libGAP_Int)(libGAP_GET_RNAM_PREC(rec,i)) > 0;i--) {
        if (issorted && libGAP_GET_RNAM_PREC(rec,i) > libGAP_GET_RNAM_PREC(rec,i+1)) {
            issorted = 0;
        }
    }
    i++;
    /* Now i points to the last positive (and thus still unsorted)
     * component and issorted is 1 iff positions i to len are sorted! */
    if (!issorted && len - i + 1 > 1) {
        qsort( libGAP_ADDR_OBJ(rec)+2*i, len-i+1, 2*sizeof(libGAP_Obj), &libGAP_PrecComparer );
    }
    /* Now i points to the first positive component and the rest is
     * sorted. */
    save = i;
    if (save == 1 ||
        -(libGAP_Int)(libGAP_GET_RNAM_PREC(rec,save-1)) < libGAP_GET_RNAM_PREC(rec,save)) {
        /* Otherwise, nothing has to be done since it is already
         * sorted, we only have to negate the RNams! */
        for (j = save;j <= len;j++)
            libGAP_SET_RNAM_PREC(rec,j,-(libGAP_Int)(libGAP_GET_RNAM_PREC(rec,j)));
        return;
    }
    /* Next we perform a merge sort on the two presorted areas. */
    /* For optimal performance, we need some space to mess around: */
    if (!inplace) {
        space = libGAP_NEW_PREC(len);
        j = 1;
        k = 1;
        while (j < save && i <= len) {
            if (-(libGAP_Int)(libGAP_GET_RNAM_PREC(rec,j)) < libGAP_GET_RNAM_PREC(rec,i)) {
                libGAP_SET_RNAM_PREC(space,k,libGAP_GET_RNAM_PREC(rec,j));
                libGAP_SET_ELM_PREC(space,k,libGAP_GET_ELM_PREC(rec,j));
                j++; k++;
            } else {
                libGAP_SET_RNAM_PREC(space,k,-(libGAP_Int)(libGAP_GET_RNAM_PREC(rec,i)));
                libGAP_SET_ELM_PREC(space,k,libGAP_GET_ELM_PREC(rec,i));
                i++; k++;
            }
        }
        /* Copy the rest of the part still missing: */
        while (j < save) {
            libGAP_SET_RNAM_PREC(space,k,libGAP_GET_RNAM_PREC(rec,j));
            libGAP_SET_ELM_PREC(space,k,libGAP_GET_ELM_PREC(rec,j));
            j++; k++;
        }
        while (i <= len) {
            libGAP_SET_RNAM_PREC(space,k,-(libGAP_Int)(libGAP_GET_RNAM_PREC(rec,i)));
            libGAP_SET_ELM_PREC(space,k,libGAP_GET_ELM_PREC(rec,i));
            i++; k++;
        }
        /* Finally, copy everything back to where it came from: */
        memcpy(libGAP_ADDR_OBJ(rec)+2,libGAP_ADDR_OBJ(space)+2,sizeof(libGAP_Obj)*2*len);
    } else {   /* We have to work in place to avoid a garbage collection. */
        /* i == save is the cut point */
        j = 1;
        for (j = 1; j < save; j++) {
            if (-(libGAP_Int)(libGAP_GET_RNAM_PREC(rec,j)) > libGAP_GET_RNAM_PREC(rec,i)) {
                /* we have to move something to position j! */
                tmp = (libGAP_Obj) (-(libGAP_Int)(libGAP_GET_RNAM_PREC(rec,j)));
                libGAP_SET_RNAM_PREC(rec,j,-(libGAP_Int)(libGAP_GET_RNAM_PREC(rec,i)));
                libGAP_SET_RNAM_PREC(rec,i,(libGAP_UInt) tmp);
                tmp = libGAP_GET_ELM_PREC(rec,j);
                libGAP_SET_ELM_PREC(rec,j,libGAP_GET_ELM_PREC(rec,i));
                libGAP_SET_ELM_PREC(rec,i,tmp);
                /* Now we have to "bubble pos i up" until it is in the
                 * right position: */
                for (k = i;k < len;k++) {
                    if (libGAP_GET_RNAM_PREC(rec,k) > libGAP_GET_RNAM_PREC(rec,k+1)) {
                        tmp = (libGAP_Obj) libGAP_GET_RNAM_PREC(rec,k);
                        libGAP_SET_RNAM_PREC(rec,k,libGAP_GET_RNAM_PREC(rec,k+1));
                        libGAP_SET_RNAM_PREC(rec,k+1,(libGAP_UInt) tmp);
                        tmp = libGAP_GET_ELM_PREC(rec,k);
                        libGAP_SET_ELM_PREC(rec,k,libGAP_GET_ELM_PREC(rec,k+1));
                        libGAP_SET_ELM_PREC(rec,k+1,tmp);
                    } else break;
                }
            }
        }
        /* Finally, we have to negate everything in the end: */
        for (j = save;j <= len;j++)
            libGAP_SET_RNAM_PREC(rec,j,-(libGAP_Int)(libGAP_GET_RNAM_PREC(rec,j)));
    }
}

#if 0
void libGAP_SortPRec ( libGAP_Obj rec )
{
    libGAP_UInt                rnam;           /* name of component               */
    libGAP_Obj                 val;            /* value of component              */
    libGAP_UInt                h;              /* gap width in shellsort          */
    libGAP_UInt                i,  k;          /* loop variables                  */

    /* sort the right record with a shellsort                              */
    h = 1;  while ( 9*h + 4 < libGAP_LEN_PREC(rec) )  h = 3*h + 1;
    while ( 0 < h ) {
        for ( i = h+1; i <= libGAP_LEN_PREC(rec); i++ ) {
            rnam = libGAP_GET_RNAM_PREC( rec, i );
            val  = libGAP_GET_ELM_PREC(  rec, i );
            k = i;
            while ( h < k
                 && strcmp( libGAP_NAME_RNAM(rnam),
                              libGAP_NAME_RNAM( libGAP_GET_RNAM_PREC(rec,k-h) ) ) < 0 ) {
                libGAP_SET_RNAM_PREC( rec, k, libGAP_GET_RNAM_PREC( rec, k-h ) );
                libGAP_SET_ELM_PREC(  rec, k, libGAP_GET_ELM_PREC(  rec, k-h ) );
                k -= h;
            }
            libGAP_SET_RNAM_PREC( rec, k, rnam );
            libGAP_SET_ELM_PREC(  rec, k, val  );
        }
        h = h / 3;
    }

}
#endif

/****************************************************************************
**

*F * * * * * * * * * * * default functions for records  * * * * * * * * * * *
*/


/****************************************************************************
**

*F  MethodPRec( <rec>, <rnam> ) . . . . . . . . . . get a method for a record
**
**  'MethodPRec' returns  the function in the  component with the record name
**  <rnam> in the  record in the component  with the name 'operations' (which
**  really should be  called 'methods') of  the record <rec>  if this exists.
**  Otherwise it return 0.
*/
libGAP_UInt libGAP_OperationsRNam;                    /* 'operations' record name        */
libGAP_UInt libGAP_COMPONENTSRNam;                    /* COMPONENTS record name          */

libGAP_Obj libGAP_MethodPRec (
    libGAP_Obj                 rec,
    libGAP_UInt                rnam )
{
    libGAP_Obj                 method;         /* method, result                  */
    libGAP_Obj                 opers;          /* operations record               */
    libGAP_UInt                i;              /* loop variable                   */

    /* is <rec> a record?                                                  */
    if ( ! libGAP_IS_PREC_REP(rec) )
        return 0;

    /* try to get the operations record                                    */
    if (!libGAP_FindPRec(rec,libGAP_OperationsRNam,&i,1)) {
        return 0;
    }
    opers = libGAP_GET_ELM_PREC( rec, i );

    /* check for an Operations Record object */
    if ( libGAP_TNUM_OBJ(opers) == libGAP_T_COMOBJ)
      {
        /* Make use of the fact the Component objects look like Precs */
        if ( !libGAP_FindPRec(opers,libGAP_COMPONENTSRNam,&i,1) ) {
          return 0;
        }
        opers = libGAP_GET_ELM_PREC( opers, i );
      }


    if ( ! libGAP_IS_PREC_REP(opers) ) {
        return 0;
    }

    /* try to get the method                                               */
    if ( !libGAP_FindPRec(opers,rnam,&i,1) ) {
        return 0;
    }
    method = libGAP_GET_ELM_PREC( opers, i );

    /* return the method                                                   */
    return method;
}

void libGAP_PrintPathPRec (
    libGAP_Obj                 rec,
    libGAP_Int                 indx )
{
    libGAP_Pr( ".%I", (libGAP_Int)libGAP_NAME_RNAM( labs((libGAP_Int)(libGAP_GET_RNAM_PREC(rec,indx))) ), 0L );
}

/****************************************************************************
**
*F  FuncREC_NAMES( <self>, <rec> )  . . . . . . . .  record names of a record
**
**  'FuncREC_NAMES' implements a method for the operations 'RecNames'.
**
**  'RecNames( <rec> )'
**
**  'RecNames'  returns a list containing the  names of the components of the
**  record <rec> as strings.
*/
libGAP_Obj libGAP_InnerRecNames( libGAP_Obj rec )
{
    libGAP_Obj                 list;           /* list of record names, result    */
    libGAP_UInt                rnam;           /* one name of record              */
    libGAP_Obj                 string;         /* one name as string              */
    libGAP_UInt                i, len;      

    libGAP_SortPRecRNam(rec,0);   /* Make sure rnams are sorted and thus negative */

    /* allocate the list                                                   */
    list = libGAP_NEW_PLIST( libGAP_T_PLIST, libGAP_LEN_PREC(rec) );
    libGAP_SET_LEN_PLIST( list, libGAP_LEN_PREC(rec) );

    /* loop over the components                                            */
    for ( i = 1; i <= libGAP_LEN_PREC(rec); i++ ) {
        rnam = -(libGAP_Int)(libGAP_GET_RNAM_PREC( rec, i ));
        len = strlen(libGAP_NAME_RNAM(rnam));
        /* could have been moved by garbage collection */
        libGAP_C_NEW_STRING( string, len, libGAP_NAME_RNAM(rnam) );
        libGAP_SET_ELM_PLIST( list, i, string );
        libGAP_CHANGED_BAG( list );
    }

    /* return the list                                                     */
    return list;
}

libGAP_Obj libGAP_FuncREC_NAMES (
    libGAP_Obj                 self,
    libGAP_Obj                 rec )
{
    /* check the argument                                                  */
    while ( ! libGAP_IS_PREC_REP(rec) ) {
        rec = libGAP_ErrorReturnObj(
            "RecNames: <rec> must be a record (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(rec), 0L,
            "you can replace <rec> via 'return <rec>;'" );
    }

    return libGAP_InnerRecNames( rec );
}


/****************************************************************************
**
*F  FuncREC_NAMES_COMOBJ( <self>, <rec> ) . . . record names of a record object
*/
/* same as FuncREC_NAMES except for different argument check  */
libGAP_Obj libGAP_FuncREC_NAMES_COMOBJ (
    libGAP_Obj                 self,
    libGAP_Obj                 rec )
{
    /* check the argument                                                  */
    while ( libGAP_TNUM_OBJ(rec) != libGAP_T_COMOBJ ) {
        rec = libGAP_ErrorReturnObj(
            "RecNames: <rec> must be a component object (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(rec), 0L,
            "you can replace <rec> via 'return <rec>;'" );
    }
    return libGAP_InnerRecNames( rec );
}


/****************************************************************************
**
*F  FuncSUM_PREC( <self>, <left>, <right> ) . . . . . . .  sum of two records
**
**  'SumRec' returns the  sum  of the two   operands <left> and <right>.   At
**  least one of the operands must be a plain record.
**
**  If  at least one of the  operands is a  record and has a '.operations.\+'
**  method, than this  is called and its  result  is returned.
*/
libGAP_UInt libGAP_SumRNam;                           /* '+' record name                 */

libGAP_Obj libGAP_FuncSUM_PREC (
    libGAP_Obj                 self,
    libGAP_Obj                 left,
    libGAP_Obj                 right )
{
    libGAP_Obj                 method;         /* method                          */

    /* try to find an applicable method                                    */
    if ( ! (method = libGAP_MethodPRec( right, libGAP_SumRNam ))
      && ! (method = libGAP_MethodPRec( left,  libGAP_SumRNam )) )
    {
        return libGAP_TRY_NEXT_METHOD;
    }

    /* call that function                                                  */
    return libGAP_CALL_2ARGS( method, left, right );
}


/****************************************************************************
**
*F  FuncDIFF_PREC( <self>, <left>, <right> )  . . . difference of two records
**
**  'DiffRec' returns the difference of the two  operands <left> and <right>.
**  At least one of the operands must be a plain record.
**
**  If  at least one  of the operands is  a record and has a '.operations.\-'
**  method, then this  is called  and  its result is returned.
*/
libGAP_UInt libGAP_DiffRNam;                          /* '-' record name                 */

libGAP_Obj libGAP_FuncDIFF_PREC (
    libGAP_Obj                 self,
    libGAP_Obj                 left,
    libGAP_Obj                 right )
{
    libGAP_Obj                 method;           /* operation                       */

    /* try to find an applicable method                                    */
    if ( ! (method = libGAP_MethodPRec( right, libGAP_DiffRNam ))
      && ! (method = libGAP_MethodPRec( left,  libGAP_DiffRNam )) )
    {
        return libGAP_TRY_NEXT_METHOD;
    }

    /* call that function                                                  */
    return libGAP_CALL_2ARGS( method, left, right );
}


/****************************************************************************
**
*F  FuncPROD_PREC( <self>, <left>, <right> )  . . . .  product of two records
**
**  'ProdRec' returns the product of the two operands <left> and <right>.  At
**  least one of the operands must be a plain record.
**
**  If  at least one  of the operands is  a record and has a '.operations.\*'
**  method, then this  is called  and  its result is returned.
*/
libGAP_UInt libGAP_ProdRNam;                          /* '*' record name                 */

libGAP_Obj libGAP_FuncPROD_PREC (
    libGAP_Obj                 self,
    libGAP_Obj                 left,
    libGAP_Obj                 right )
{
    libGAP_Obj                 method;         /* method                          */

    /* try to find an applicable method                                    */
    if ( ! (method = libGAP_MethodPRec( right, libGAP_ProdRNam ))
      && ! (method = libGAP_MethodPRec( left,  libGAP_ProdRNam )) )
    {
        return libGAP_TRY_NEXT_METHOD;
    }

    /* call that function                                                  */
    return libGAP_CALL_2ARGS( method, left, right );
}


/****************************************************************************
**
*F  FuncQUO_PREC( <self>, <left>, <right> ) . . . . . quotient of two records
**
**  'QuoRec' returns the quotient of the two operands <left> and <right>.  At
**  least one of the operands must be a plain record.
**
**  If  at least one  of the operands is  a record and has a '.operations.\/'
**  method, then this  is called  and  its result is returned.
*/
libGAP_UInt libGAP_QuoRNam;                           /* '/' record name                 */

libGAP_Obj libGAP_FuncQUO_PREC (
    libGAP_Obj                 self,
    libGAP_Obj                 left,
    libGAP_Obj                 right )
{
    libGAP_Obj                 method;         /* method                          */

    /* try to find an applicable method                                    */
    if ( ! (method = libGAP_MethodPRec( right, libGAP_QuoRNam ))
      && ! (method = libGAP_MethodPRec( left,  libGAP_QuoRNam )) )
    {
        return libGAP_TRY_NEXT_METHOD;
    }

    /* call that function                                                  */
    return libGAP_CALL_2ARGS( method, left, right );
}


/****************************************************************************
**
*F  FuncLQUO_PREC( <self>, <left>, <right> )  .  left quotient of two records
**
**  'LQuoPRec' returns the   left quotient  of  the two   operands <left> and
**  <right>.  At least one of the operands must be a plain record.
**
**  If  at   least   one   of  the  operands     is a  record   and    has  a
**  '.operations.LeftQuotient' method, then this is  called and its result is
**  returned.
*/
libGAP_UInt libGAP_LQuoRNam;                          /* 'LeftQuotient' record name      */

libGAP_Obj libGAP_FuncLQUO_PREC (
    libGAP_Obj                 self,
    libGAP_Obj                 left,
    libGAP_Obj                 right )
{
    libGAP_Obj                 method;         /* method                          */

    /* try to find an applicable method                                    */
    if ( ! (method = libGAP_MethodPRec( right, libGAP_LQuoRNam ))
      && ! (method = libGAP_MethodPRec( left,  libGAP_LQuoRNam )) )
    {
        return libGAP_TRY_NEXT_METHOD;
    }

    /* call that function                                                  */
    return libGAP_CALL_2ARGS( method, left, right );
}


/****************************************************************************
**
*F  FuncPOW_PREC( <self>, <left>, <right> ) . . . . . .  power of two records
**
**  'PowPRec' returns the power of  the two operands  <left> and <right>.  At
**  least one of the operands must be a plain record.
**
**  If  at least one  of the operands is  a record and has a '.operations.\^'
**  method, then this  is called  and  its result is returned.
*/
libGAP_UInt libGAP_PowRNam;                           /* '^' record name                 */

libGAP_Obj libGAP_FuncPOW_PREC (
    libGAP_Obj                 self,
    libGAP_Obj                 left,
    libGAP_Obj                 right )
{
    libGAP_Obj                 method;         /* method                          */

    /* try to find an applicable method                                    */
    if ( ! (method = libGAP_MethodPRec( right, libGAP_PowRNam ))
      && ! (method = libGAP_MethodPRec( left,  libGAP_PowRNam )) )
    {
        return libGAP_TRY_NEXT_METHOD;
    }

    /* call that function                                                  */
    return libGAP_CALL_2ARGS( method, left, right );
}


/****************************************************************************
**
*F  FuncCOMM_PREC( <self>, <left>, <right> )  . .   commutator of two records
**
**  'CommPRec' returns the commutator of the two operands <left> and <right>.
**  At least one of the operands must be a plain record.
**
**  If at least one of the operands is  a record and has a '.operations.Comm'
**  method, then this  is called and its  result  is returned.
*/
libGAP_UInt libGAP_CommRNam;                          /* 'Comm' record name              */

libGAP_Obj libGAP_FuncCOMM_PREC (
    libGAP_Obj                 self,
    libGAP_Obj                 left,
    libGAP_Obj                 right )
{
    libGAP_Obj                 method;         /* method                          */

    /* try to find an applicable method                                    */
    if ( ! (method = libGAP_MethodPRec( right, libGAP_CommRNam ))
      && ! (method = libGAP_MethodPRec( left,  libGAP_CommRNam )) )
    {
        return libGAP_TRY_NEXT_METHOD;
    }

    /* call that function                                                  */
    return libGAP_CALL_2ARGS( method, left, right );
}


/****************************************************************************
**
*F  FuncMOD_PREC( <self>, <left>, <right> ) . . . .  remainder of two records
**
**  'ModPRec' returns the   remainder  the operands  <left>   by  the operand
**  <right>.  At least one of the operands must be a plain record.
**
**  If at least one of the operands is a  record and has a '.operations.\mod'
**  method, then this  is  called and its  result is  returned.
*/
libGAP_UInt libGAP_ModRNam;                /* 'mod' record name               */

libGAP_Obj libGAP_FuncMOD_PREC (
    libGAP_Obj                 self,
    libGAP_Obj                 left,
    libGAP_Obj                 right )
{
    libGAP_Obj                 method;         /* method                          */

    /* try to find an applicable method                                    */
    if ( ! (method = libGAP_MethodPRec( right, libGAP_ModRNam ))
      && ! (method = libGAP_MethodPRec( left,  libGAP_ModRNam )) )
    {
        return libGAP_TRY_NEXT_METHOD;
    }

    /* call that function                                                  */
    return libGAP_CALL_2ARGS( method, left, right );
}


/****************************************************************************
**
*F  FuncIN_PREC( <self>, <left>, <right> )  . . . membership test for records
**
**  'InRec' returns 'True' if the operand <left>  is an element of the record
**  <right>.  <right> must be a plain record.
**
**  If <right> has  a '.operations.\in' method, than this  is called  and its
**  result is returned.
*/
libGAP_UInt libGAP_InRNam;                            /* 'in' record name                */

libGAP_Obj libGAP_FuncIN_PREC (
    libGAP_Obj                 self,
    libGAP_Obj                 left,
    libGAP_Obj                 right )
{
    libGAP_Obj                 method;         /* method                          */

    /* try to find an applicable method                                    */
    if ( ! (method = libGAP_MethodPRec( right, libGAP_InRNam )) ) {
        return libGAP_TRY_NEXT_METHOD;
    }

    /* call that function                                                  */
    return libGAP_CALL_2ARGS( method, left, right );
}


/****************************************************************************
**
*F  FuncEQ_PREC_DEFAULT( <self>, <left>, <right> )  comparison of two records
**
**  'EqRec' returns '1L'  if the two  operands <left> and <right> are equal
**  and '0L' otherwise.  At least one operand must be a plain record.
*/
libGAP_Obj libGAP_FuncEQ_PREC_DEFAULT (
    libGAP_Obj                 self,
    libGAP_Obj                 left,
    libGAP_Obj                 right )
{
    libGAP_UInt                i;              /* loop variable                   */

    /* quick first checks                                                  */
    if ( ! libGAP_IS_PREC_REP(left) )
        return libGAP_False;
    if ( ! libGAP_IS_PREC_REP(right) )
        return libGAP_False;
    if ( libGAP_LEN_PREC(left) != libGAP_LEN_PREC(right) )
        return libGAP_False;

    /* Records are now always sorted by their RNam */

    /* compare componentwise                                               */
    for ( i = 1; i <= libGAP_LEN_PREC(right); i++ ) {

        /* compare the names                                               */
        if ( libGAP_GET_RNAM_PREC(left,i) != libGAP_GET_RNAM_PREC(right,i) ) {
            return libGAP_False;
        }

        /* compare the values                                              */
        if ( ! libGAP_EQ(libGAP_GET_ELM_PREC(left,i),libGAP_GET_ELM_PREC(right,i)) ) {
            return libGAP_False;
        }
    }

    /* the records are equal                                               */
    return libGAP_True;
}


/****************************************************************************
**
*F  FuncEQ_PREC( <self>, <left>, <right> )  . . . . comparison of two records
**
**  'EqRec' returns '1L'  if the two  operands <left> and <right> are equal
**  and '0L' otherwise.  At least one operand must be a plain record.
**
**  If at  least one of the  operands is a  record and has a '.operations.\='
**  method,  than this is called  and its result  is returned.  Otherwise the
**  records are compared componentwise.
*/
libGAP_UInt libGAP_EqRNam;                            /* '=' record name                 */

libGAP_Obj libGAP_FuncEQ_PREC (
    libGAP_Obj                 self,
    libGAP_Obj                 left,
    libGAP_Obj                 right )
{
    libGAP_Obj                 method;         /* method                          */

    /* try to find an applicable method                                    */
    if ( ! (method = libGAP_MethodPRec( right, libGAP_EqRNam ))
      && ! (method = libGAP_MethodPRec( left,  libGAP_EqRNam )) )
    {
        return libGAP_FuncEQ_PREC_DEFAULT( self, left, right );
    }

    /* call that function                                                  */
    return libGAP_CALL_2ARGS( method, left, right );
}


/****************************************************************************
**
*F  FuncLT_PREC_DEFAULT( <self>, <left>, <right> )  comparison of two records
**
**  'LtRec' returns '1L'  if the operand  <left> is  less than the  operand
**  <right>, and '0L'  otherwise.  At least  one operand  must be a  plain
**  record.
*/
libGAP_Obj libGAP_FuncLT_PREC_DEFAULT (
    libGAP_Obj                 self,
    libGAP_Obj                 left,
    libGAP_Obj                 right )
{
    libGAP_UInt                i;              /* loop variable                   */

    /* quick first checks                                                  */
    if ( ! libGAP_IS_PREC_REP(left) || ! libGAP_IS_PREC_REP(right) ) {
        if ( libGAP_TNUM_OBJ(left ) < libGAP_TNUM_OBJ(right) )  return libGAP_True;
        if ( libGAP_TNUM_OBJ(left ) > libGAP_TNUM_OBJ(right) )  return libGAP_False;
    }

    /* Records are now always sorted by their RNam */

    /* compare componentwise                                               */
    for ( i = 1; i <= libGAP_LEN_PREC(right); i++ ) {

        /* if the left is a proper prefix of the right one                 */
        if ( libGAP_LEN_PREC(left) < i )  return libGAP_True;

        /* compare the names                                               */
        /* The sense of this comparison is determined by the rule that
           unbound entries compare less than bound ones                    */
        if ( libGAP_GET_RNAM_PREC(left,i) != libGAP_GET_RNAM_PREC(right,i) ) {
            if ( strcmp( libGAP_NAME_RNAM( labs((libGAP_Int)(libGAP_GET_RNAM_PREC(left,i))) ),
                   libGAP_NAME_RNAM( labs((libGAP_Int)(libGAP_GET_RNAM_PREC(right,i))) ) ) > 0 ) {
                return libGAP_True;
            }
            else {
                return libGAP_False;
            }
        }

        /* compare the values                                              */
        if ( ! libGAP_EQ(libGAP_GET_ELM_PREC(left,i),libGAP_GET_ELM_PREC(right,i)) ) {
            return libGAP_LT( libGAP_GET_ELM_PREC(left,i), libGAP_GET_ELM_PREC(right,i) ) ?
                   libGAP_True : libGAP_False;
        }

    }

    /* the records are equal or the right is a prefix of the left          */
    return libGAP_False;
}


/****************************************************************************
**
*F  FuncLT_PREC( <self>, <left>, <right> )   . . .  comparison of two records
**
**  'LtRec' returns '1L'  if the operand  <left> is  less than the  operand
**  <right>, and '0L'  otherwise.  At least  one operand  must be a  plain
**  record.
**
**  If at least  one of the operands is  a record and  has a '.operations.\<'
**  method, than this  is called and its  result is  returned.  Otherwise the
**  records are compared componentwise.
*/
libGAP_UInt libGAP_LtRNam;                            /* '<' record name                 */

libGAP_Obj libGAP_FuncLT_PREC (
    libGAP_Obj                 self,
    libGAP_Obj                 left,
    libGAP_Obj                 right )
{
    libGAP_Obj                 method;         /* method                          */

    /* try to find an applicable method                                    */
    if ( ! (method = libGAP_MethodPRec( right, libGAP_LtRNam ))
      && ! (method = libGAP_MethodPRec( left,  libGAP_LtRNam )) )
    {
        return libGAP_FuncLT_PREC_DEFAULT( self, left, right );
    }

    /* call that function                                                  */
    return libGAP_CALL_2ARGS( method, left, right );
}


/****************************************************************************
**
*F  SavePRec( <prec> )
**
*/

void libGAP_SavePRec( libGAP_Obj prec )
{
  libGAP_UInt len,i;
  len = libGAP_LEN_PREC(prec);
  libGAP_SaveUInt(len);
  for (i = 1; i <= len; i++)
    {
      libGAP_SaveUInt(libGAP_GET_RNAM_PREC(prec, i));
      libGAP_SaveSubObj(libGAP_GET_ELM_PREC(prec, i));
    }
  return;
}

/****************************************************************************
**
*F  LoadPRec( <prec> )
**
*/

void libGAP_LoadPRec( libGAP_Obj prec )
{
  libGAP_UInt len,i;
  len = libGAP_LoadUInt();
  libGAP_SET_LEN_PREC(prec,len);
  for (i = 1; i <= len; i++)
    {
      libGAP_SET_RNAM_PREC(prec, i, libGAP_LoadUInt());
      libGAP_SET_ELM_PREC(prec, i, libGAP_LoadSubObj());
    }
  return;
}

/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/

/****************************************************************************
**

*V  BagNames  . . . . . . . . . . . . . . . . . . . . . . . list of bag names
*/
static libGAP_StructBagNames libGAP_BagNames[] = {
  { libGAP_T_PREC,                     "record (plain)"            },
  { libGAP_T_PREC +libGAP_IMMUTABLE,          "record (plain,imm)"        },
  { libGAP_T_PREC            +libGAP_COPYING, "record (plain,copied)"     },
  { libGAP_T_PREC +libGAP_IMMUTABLE +libGAP_COPYING, "record (plain,imm,copied)" },
  { -1,                         ""                          }
};


/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "REC_NAMES", 1, "rec",
      libGAP_FuncREC_NAMES, "src/precord.c:REC_NAMES" },

    { "REC_NAMES_COMOBJ", 1, "rec obj",
      libGAP_FuncREC_NAMES_COMOBJ, "src/precord.c:REC_NAMES_COMOBJ" },

    { "SUM_PREC", 2, "left, right",
      libGAP_FuncSUM_PREC, "src/precord.c:SUM_PREC" },

    { "DIFF_PREC", 2, "left, right",
      libGAP_FuncDIFF_PREC, "src/precord.c:DIFF_PREC" },

    { "PROD_PREC", 2, "left, right",
      libGAP_FuncPROD_PREC, "src/precord.c:PROD_PREC" },

    { "QUO_PREC", 2, "left, right",
      libGAP_FuncQUO_PREC, "src/precord.c:QUO_PREC" },

    { "LQUO_PREC", 2, "left, right",
      libGAP_FuncLQUO_PREC, "src/precord.c:LQUO_PREC" },

    { "POW_PREC", 2, "left, right",
      libGAP_FuncPOW_PREC, "src/precord.c:POW_PREC" },

    { "MOD_PREC", 2, "left, right",
      libGAP_FuncMOD_PREC, "src/precord.c:MOD_PREC" },

    { "COMM_PREC", 2, "left, right",
      libGAP_FuncCOMM_PREC, "src/precord.c:COMM_PREC" },

    { "IN_PREC", 2, "left, right",
      libGAP_FuncIN_PREC, "src/precord.c:IN_PREC" },

    { "EQ_PREC", 2, "left, right",
      libGAP_FuncEQ_PREC, "src/precord.c:EQ_PREC" },

    { "EQ_PREC_DEFAULT",2,"left, right",
      libGAP_FuncEQ_PREC_DEFAULT, "src/precord.c:EQ_PREC_DEFAULT" },

    { "LT_PREC", 2, "left, right",
      libGAP_FuncLT_PREC, "src/precord.c:LT_PREC" },

    { "LT_PREC_DEFAULT",2,"left, right",
      libGAP_FuncLT_PREC_DEFAULT, "src/precord.c:LT_PREC_DEFAULT" },

    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* check dependencies                                                  */
    libGAP_RequireModule( libGAP_module, "records", 503600000UL );

    /* GASMAN marking functions and GASMAN names                           */
    libGAP_InitBagNamesFromTable( libGAP_BagNames );

    libGAP_InitMarkFuncBags( libGAP_T_PREC                     , libGAP_MarkAllSubBags );
    libGAP_InitMarkFuncBags( libGAP_T_PREC +libGAP_IMMUTABLE          , libGAP_MarkAllSubBags );
    libGAP_InitMarkFuncBags( libGAP_T_PREC            +libGAP_COPYING , libGAP_MarkAllSubBags );
    libGAP_InitMarkFuncBags( libGAP_T_PREC +libGAP_IMMUTABLE +libGAP_COPYING , libGAP_MarkAllSubBags );

    /* init filters and functions                                          */
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* Install saving functions                                            */
    libGAP_SaveObjFuncs[ libGAP_T_PREC            ] = libGAP_SavePRec;
    libGAP_SaveObjFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ] = libGAP_SavePRec;
    libGAP_LoadObjFuncs[ libGAP_T_PREC            ] = libGAP_LoadPRec;
    libGAP_LoadObjFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ] = libGAP_LoadPRec;

    /* install into record function tables                                 */
    libGAP_ElmRecFuncs[ libGAP_T_PREC            ] = libGAP_ElmPRec;
    libGAP_ElmRecFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ] = libGAP_ElmPRec;
    libGAP_IsbRecFuncs[ libGAP_T_PREC            ] = libGAP_IsbPRec;
    libGAP_IsbRecFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ] = libGAP_IsbPRec;
    libGAP_AssRecFuncs[ libGAP_T_PREC            ] = libGAP_AssPRec;
    libGAP_AssRecFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ] = libGAP_AssPRecImm;
    libGAP_UnbRecFuncs[ libGAP_T_PREC            ] = libGAP_UnbPRec;
    libGAP_UnbRecFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ] = libGAP_UnbPRecImm;

    /* install mutability test                                             */
    libGAP_IsMutableObjFuncs[  libGAP_T_PREC            ] = libGAP_IsMutablePRecYes;
    libGAP_IsMutableObjFuncs[  libGAP_T_PREC +libGAP_IMMUTABLE ] = libGAP_IsMutablePRecNo;
    libGAP_IsCopyableObjFuncs[ libGAP_T_PREC            ] = libGAP_IsCopyablePRecYes;
    libGAP_IsCopyableObjFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ] = libGAP_IsCopyablePRecYes;

    /* install into copy function tables                                  */
    libGAP_CopyObjFuncs [ libGAP_T_PREC                     ] = libGAP_CopyPRec;
    libGAP_CopyObjFuncs [ libGAP_T_PREC +libGAP_IMMUTABLE          ] = libGAP_CopyPRec;
    libGAP_CopyObjFuncs [ libGAP_T_PREC            +libGAP_COPYING ] = libGAP_CopyPRecCopy;
    libGAP_CopyObjFuncs [ libGAP_T_PREC +libGAP_IMMUTABLE +libGAP_COPYING ] = libGAP_CopyPRecCopy;
    libGAP_CleanObjFuncs[ libGAP_T_PREC                     ] = libGAP_CleanPRec;
    libGAP_CleanObjFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE          ] = libGAP_CleanPRec;
    libGAP_CleanObjFuncs[ libGAP_T_PREC            +libGAP_COPYING ] = libGAP_CleanPRecCopy;
    libGAP_CleanObjFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE +libGAP_COPYING ] = libGAP_CleanPRecCopy;

    /* install printer                                                     */
    libGAP_PrintObjFuncs[  libGAP_T_PREC            ] = libGAP_PrintPRec;
    libGAP_PrintObjFuncs[  libGAP_T_PREC +libGAP_IMMUTABLE ] = libGAP_PrintPRec;
    libGAP_PrintPathFuncs[ libGAP_T_PREC            ] = libGAP_PrintPathPRec;
    libGAP_PrintPathFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ] = libGAP_PrintPathPRec;

    /* install the kind function                                           */
    libGAP_ImportGVarFromLibrary( "TYPE_PREC_MUTABLE",   &libGAP_TYPE_PREC_MUTABLE   );
    libGAP_ImportGVarFromLibrary( "TYPE_PREC_IMMUTABLE", &libGAP_TYPE_PREC_IMMUTABLE );

    libGAP_TypeObjFuncs[ libGAP_T_PREC            ] = libGAP_TypePRecMut;
    libGAP_TypeObjFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ] = libGAP_TypePRecImm;

    libGAP_MakeImmutableObjFuncs[ libGAP_T_PREC   ] = libGAP_MakeImmutablePRec;

    /* return success                                                      */
    return 0;
}


/****************************************************************************
<**
*F  PostRestore( <module> ) . . . . . . . . . . . . . after restore workspace
*/
static libGAP_Int libGAP_PostRestore (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* get the appropriate record record name                              */
    libGAP_OperationsRNam = libGAP_RNamName( "operations"   );
    libGAP_COMPONENTSRNam = libGAP_RNamName( "COMPONENTS"   );
    libGAP_EqRNam         = libGAP_RNamName( "="            );
    libGAP_LtRNam         = libGAP_RNamName( "<"            );
    libGAP_InRNam         = libGAP_RNamName( "in"           );
    libGAP_SumRNam        = libGAP_RNamName( "+"            );
    libGAP_DiffRNam       = libGAP_RNamName( "-"            );
    libGAP_ProdRNam       = libGAP_RNamName( "*"            );
    libGAP_QuoRNam        = libGAP_RNamName( "/"            );
    libGAP_LQuoRNam       = libGAP_RNamName( "LeftQuotient" );
    libGAP_PowRNam        = libGAP_RNamName( "^"            );
    libGAP_CommRNam       = libGAP_RNamName( "Comm"         );
    libGAP_ModRNam        = libGAP_RNamName( "mod"          );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return libGAP_PostRestore( libGAP_module );
}


/****************************************************************************
**
*F  InitInfoPRecord() . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "precord",                          /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    libGAP_PostRestore                         /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoPRecord ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}


/****************************************************************************
**

*E  precord.c . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/



