/****************************************************************************
**
*W  permutat.c                  GAP source                   Martin Schönert
**                                                           & Alice Niemeyer
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions for permutations (small and large).
**
**  Mathematically a permutation is a bijective mapping  of a finite set onto
**  itself.  In \GAP\ this subset must always be of the form [ 1, 2, .., N ],
**  where N is at most $2^16$.
**
**  Internally a permutation  is viewed as a mapping  of [ 0,  1,  .., N-1 ],
**  because in C indexing of  arrays is done with the origin  0 instead of 1.
**  A permutation is represented by a bag of type 'T_PERM' of the form
**
**      +-------+-------+-------+-------+- - - -+-------+-------+
**      | image | image | image | image |       | image | image |
**      | of  0 | of  1 | of  2 | of  3 |       | of N-2| of N-1|
**      +-------+-------+-------+-------+- - - -+-------+-------+
**
**  The entries of the bag are of type  'UInt2'  (defined in 'system.h' as an
**  at least 16 bit   wide unsigned integer  type).   The first entry is  the
**  image of 0, the second is the image of 1, and so  on.  Thus, the entry at
**  C index <i> is the image of <i>, if we view the permutation as mapping of
**  [ 0, 1, 2, .., N-1 ] as described above.
**
**  Permutations are never  shortened.  For  example, if  the product  of two
**  permutations of degree 100 is the identity, it  is nevertheless stored as
**  array of length 100, in  which the <i>-th  entry is of course simply <i>.
**  Testing whether a product has trailing  fixpoints would be pretty costly,
**  and permutations of equal degree can be handled by the functions faster.
**
*N  13-Jan-91 martin should add 'CyclesPerm', 'CycleLengthsPerm'
*/
#include        "system.h"              /* system dependent part           */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */

#include        "calls.h"               /* generic call mechanism          */
#include        "opers.h"               /* generic operations              */

#include        "ariths.h"              /* basic arithmetic                */

#include        "bool.h"                /* booleans                        */

#include        "integer.h"             /* integers                        */

#include        "permutat.h"            /* permutations                    */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "plist.h"               /* plain lists                     */
#include        "range.h"               /* ranges                          */
#include        "string.h"              /* strings                         */

#include        "saveload.h"            /* saving and loading              */


/****************************************************************************
**

*F  NEW_PERM2(<deg>)  . . . . . . . . . . . .  make a new (small) permutation
*F  DEG_PERM2(<perm>) . . . . . . . . . . . . . degree of (small) permutation
*F  ADDR_PERM2(<perm>)  . . . . . . . absolute address of (small) permutation
*F  NEW_PERM4(<deg>)  . . . . . . . . . . . .  make a new (large) permutation
*F  DEG_PERM4(<perm>) . . . . . . . . . . . . . degree of (large) permutation
*F  ADDR_PERM4(<perm>)  . . . . . . . absolute address of (large) permutation
**
**  'NEW_PERM2', 'DEG_PERM2',  'ADDR_PERM2',   'NEW_PERM4', 'DEG_PERM4',  and
**  'ADDR_PERM4'  are defined in  the declaration  part   of this package  as
**  follows
**
#define NEW_PERM2(deg)          NewBag( T_PERM2, (deg) * sizeof(UInt2))
#define DEG_PERM2(perm)         (SIZE_OBJ(perm) / sizeof(UInt2))
#define ADDR_PERM2(perm)        ((UInt2*)ADDR_OBJ(perm))
#define NEW_PERM4(deg)          NewBag( T_PERM4, (deg) * sizeof(UInt4))
#define DEG_PERM4(perm)         (SIZE_OBJ(perm) / sizeof(UInt4))
#define ADDR_PERM4(perm)        ((UInt4*)ADDR_OBJ(perm))
*/


/****************************************************************************
**
*F  IMAGE(<i>,<pt>,<dg>)  . . . . . .  image of <i> under <pt> of degree <dg>
**
**  'IMAGE'  returns the  image of the   point <i> under  the permutation  of
**  degree <dg> pointed to  by <pt>.   If the  point  <i> is greater  than or
**  equal to <dg> the image is <i> itself.
**
**  'IMAGE' is  implemented as a macro so  do not use  it with arguments that
**  have side effects.
*/
#define libGAP_IMAGE(i,pt,dg)  (((i) < (dg)) ? (pt)[(i)] : (i))

/****************************************************************************
**
*F  IMAGETWO(<i>,<pt>,<dg>)  . . . . . .  image of <i> under <pt> of degree <dg>
**
**  as IMAGE but for a 2-byte permutation: If we map, we have to cast in UInt2.
*/
#define libGAP_IMAGETWO(i,pt,dg)  (((i) < (dg)) ? (pt)[(libGAP_UInt2)(i)] : (i))


/****************************************************************************
**

*V  IdentityPerm  . . . . . . . . . . . . . . . . . . .  identity permutation
**
**  'IdentityPerm' is an identity permutation.
*/
libGAP_Obj             libGAP_IdentityPerm;


/****************************************************************************
**
*V  TmpPerm . . . . . . . handle of the buffer bag of the permutation package
**
**  'TmpPerm' is the handle  of a bag of type  'T_PERM4', which is created at
**  initialization time  of this package.  Functions  in this package can use
**  this bag for  whatever  purpose they want.  They   have to make sure   of
**  course that it is large enough.
**  The buffer is *not* guaranteed to have any particular value, routines
**  that require a zero-initialization need to do this at the start.
*/
libGAP_Obj                     libGAP_TmpPerm;


/****************************************************************************
**
*F  TypePerm( <perm> )  . . . . . . . . . . . . . . . . kind of a permutation
**
**  'TypePerm' returns the kind of permutations.
**
**  'TypePerm' is the function in 'TypeObjFuncs' for permutations.
*/
libGAP_Obj             libGAP_TYPE_PERM2;

libGAP_Obj             libGAP_TYPE_PERM4;

libGAP_Obj             libGAP_TypePerm2 (
    libGAP_Obj                 perm )
{
    return libGAP_TYPE_PERM2;
}

libGAP_Obj             libGAP_TypePerm4 (
    libGAP_Obj                 perm )
{
    return libGAP_TYPE_PERM4;
}


/****************************************************************************
**
*F  PrintPerm( <perm> ) . . . . . . . . . . . . . . . . . print a permutation
**
**  'PrintPerm' prints the permutation <perm> in the usual cycle notation. It
**  uses the degree to print all points with same width, which  looks  nicer.
**  Linebreaks are prefered most after cycles and  next  most  after  commas.
**
**  It does not remember which points have already  been  printed.  To  avoid
**  printing a cycle twice each is printed with the smallest  element  first.
**  This may in the worst case, for (1,2,..,n), take n^2/2 steps, but is fast
**  enough to keep a terminal at 9600 baud busy for all but the extrem cases.
*/
void            libGAP_PrintPermP (
    libGAP_Obj                 perm )
{
    libGAP_UInt                degPerm;        /* degree of the permutation       */
    libGAP_UInt2 *             ptPerm;         /* pointer to the permutation      */
    libGAP_UInt                p,  q;          /* loop variables                  */
    libGAP_UInt                isId;           /* permutation is the identity?    */
    const char *        fmt1;           /* common formats to print points  */
    const char *        fmt2;           /* common formats to print points  */

    /* set up the formats used, so all points are printed with equal width */
    degPerm = libGAP_DEG_PERM2(perm);
    if      ( degPerm <    10 ) { fmt1 = "%>(%>%1d%<"; fmt2 = ",%>%1d%<"; }
    else if ( degPerm <   100 ) { fmt1 = "%>(%>%2d%<"; fmt2 = ",%>%2d%<"; }
    else if ( degPerm <  1000 ) { fmt1 = "%>(%>%3d%<"; fmt2 = ",%>%3d%<"; }
    else if ( degPerm < 10000 ) { fmt1 = "%>(%>%4d%<"; fmt2 = ",%>%4d%<"; }
    else                        { fmt1 = "%>(%>%5d%<"; fmt2 = ",%>%5d%<"; }

    /* run through all points                                              */
    isId = 1;
    ptPerm = libGAP_ADDR_PERM2(perm);
    for ( p = 0; p < degPerm; p++ ) {

        /* find the smallest element in this cycle                         */
        q = ptPerm[p];
        while ( p < q )  q = ptPerm[q];

        /* if the smallest is the one we started with lets print the cycle */
        if ( p == q && ptPerm[p] != p ) {
            isId = 0;
            libGAP_Pr(fmt1,(libGAP_Int)(p+1),0L);
            for ( q = libGAP_ADDR_PERM2(perm)[p]; q != p; q = libGAP_ADDR_PERM2(perm)[q] ) {
                libGAP_Pr(fmt2,(libGAP_Int)(q+1),0L);
            }
            libGAP_Pr("%<)",0L,0L);
            /* restore pointer, in case Pr caused a garbage collection */
            ptPerm = libGAP_ADDR_PERM2(perm);  
        }

    }

    /* special case for the identity                                       */
    if ( isId )  libGAP_Pr("()",0L,0L);
}

void            libGAP_PrintPermQ (
    libGAP_Obj                 perm )
{
    libGAP_UInt                degPerm;        /* degree of the permutation       */
    libGAP_UInt4 *             ptPerm;         /* pointer to the permutation      */
    libGAP_UInt                p,  q;          /* loop variables                  */
    libGAP_UInt                isId;           /* permutation is the identity?    */
    const char *        fmt1;           /* common formats to print points  */
    const char *        fmt2;           /* common formats to print points  */

    /* set up the formats used, so all points are printed with equal width */
    degPerm = libGAP_DEG_PERM4(perm);
    if      ( degPerm <    10 ) { fmt1 = "%>(%>%1d%<"; fmt2 = ",%>%1d%<"; }
    else if ( degPerm <   100 ) { fmt1 = "%>(%>%2d%<"; fmt2 = ",%>%2d%<"; }
    else if ( degPerm <  1000 ) { fmt1 = "%>(%>%3d%<"; fmt2 = ",%>%3d%<"; }
    else if ( degPerm < 10000 ) { fmt1 = "%>(%>%4d%<"; fmt2 = ",%>%4d%<"; }
    else                        { fmt1 = "%>(%>%5d%<"; fmt2 = ",%>%5d%<"; }

    /* run through all points                                              */
    isId = 1;
    ptPerm = libGAP_ADDR_PERM4(perm);
    for ( p = 0; p < degPerm; p++ ) {

        /* find the smallest element in this cycle                         */
        q = ptPerm[p];
        while ( p < q )  q = ptPerm[q];

        /* if the smallest is the one we started with lets print the cycle */
        if ( p == q && ptPerm[p] != p ) {
            isId = 0;
            libGAP_Pr(fmt1,(libGAP_Int)(p+1),0L);
            for ( q = libGAP_ADDR_PERM4(perm)[p]; q != p; q = libGAP_ADDR_PERM4(perm)[q] )
                libGAP_Pr(fmt2,(libGAP_Int)(q+1),0L);
            libGAP_Pr("%<)",0L,0L);
            /* restore pointer, in case Pr caused a garbage collection */
            ptPerm = libGAP_ADDR_PERM4(perm);
        }

    }

    /* special case for the identity                                       */
    if ( isId )  libGAP_Pr("()",0L,0L);
}


/****************************************************************************
**
*F  EqPerm( <opL>, <opR> )  . . . . . . .  test if two permutations are equal
**
**  'EqPerm' returns 'true' if the two permutations <opL> and <opR> are equal
**  and 'false' otherwise.
**
**  Two permutations may be equal, even if the two sequences do not have  the
**  same length, if  the  larger  permutation  fixes  the  exceeding  points.
*/
libGAP_Int             libGAP_EqPerm22 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt2 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt2 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt                p;              /* loop variable                   */

    /* get the degrees                                                     */
    degL = libGAP_DEG_PERM2(opL);
    degR = libGAP_DEG_PERM2(opR);

    /* set up the pointers                                                 */
    ptL = libGAP_ADDR_PERM2(opL);
    ptR = libGAP_ADDR_PERM2(opR);

    /* search for a difference and return False if you find one          */
    if ( degL <= degR ) {
        for ( p = 0; p < degL; p++ )
            if ( *(ptL++) != *(ptR++) )
                return 0L;
        for ( p = degL; p < degR; p++ )
            if (        p != *(ptR++) )
                return 0L;
    }
    else {
        for ( p = 0; p < degR; p++ )
            if ( *(ptL++) != *(ptR++) )
                return 0L;
        for ( p = degR; p < degL; p++ )
            if ( *(ptL++) !=        p )
                return 0L;
    }

    /* otherwise they must be equal                                        */
    return 1L;
}

libGAP_Int             libGAP_EqPerm24 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt2 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt4 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt                p;              /* loop variable                   */

    /* get the degrees                                                     */
    degL = libGAP_DEG_PERM2(opL);
    degR = libGAP_DEG_PERM4(opR);

    /* set up the pointers                                                 */
    ptL = libGAP_ADDR_PERM2(opL);
    ptR = libGAP_ADDR_PERM4(opR);

    /* search for a difference and return False if you find one          */
    if ( degL <= degR ) {
        for ( p = 0; p < degL; p++ )
            if ( *(ptL++) != *(ptR++) )
                return 0L;
        for ( p = degL; p < degR; p++ )
            if (        p != *(ptR++) )
                return 0L;
    }
    else {
        for ( p = 0; p < degR; p++ )
            if ( *(ptL++) != *(ptR++) )
                return 0L;
        for ( p = degR; p < degL; p++ )
            if ( *(ptL++) !=        p )
                return 0L;
    }

    /* otherwise they must be equal                                        */
    return 1L;
}

libGAP_Int             libGAP_EqPerm42 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt4 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt2 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt                p;              /* loop variable                   */

    /* get the degrees                                                     */
    degL = libGAP_DEG_PERM4(opL);
    degR = libGAP_DEG_PERM2(opR);

    /* set up the pointers                                                 */
    ptL = libGAP_ADDR_PERM4(opL);
    ptR = libGAP_ADDR_PERM2(opR);

    /* search for a difference and return False if you find one          */
    if ( degL <= degR ) {
        for ( p = 0; p < degL; p++ )
            if ( *(ptL++) != *(ptR++) )
                return 0L;
        for ( p = degL; p < degR; p++ )
            if (        p != *(ptR++) )
                return 0L;
    }
    else {
        for ( p = 0; p < degR; p++ )
            if ( *(ptL++) != *(ptR++) )
                return 0L;
        for ( p = degR; p < degL; p++ )
            if ( *(ptL++) !=        p )
                return 0L;
    }

    /* otherwise they must be equal                                        */
    return 1L;
}

libGAP_Int             libGAP_EqPerm44 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt4 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt4 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt                p;              /* loop variable                   */

    /* get the degrees                                                     */
    degL = libGAP_DEG_PERM4(opL);
    degR = libGAP_DEG_PERM4(opR);

    /* set up the pointers                                                 */
    ptL = libGAP_ADDR_PERM4(opL);
    ptR = libGAP_ADDR_PERM4(opR);

    /* search for a difference and return False if you find one          */
    if ( degL <= degR ) {
        for ( p = 0; p < degL; p++ )
            if ( *(ptL++) != *(ptR++) )
                return 0L;
        for ( p = degL; p < degR; p++ )
            if (        p != *(ptR++) )
                return 0L;
    }
    else {
        for ( p = 0; p < degR; p++ )
            if ( *(ptL++) != *(ptR++) )
                return 0L;
        for ( p = degR; p < degL; p++ )
            if ( *(ptL++) !=        p )
                return 0L;
    }

    /* otherwise they must be equal                                        */
    return 1L;
}


/****************************************************************************
**
*F  LtPerm( <opL>, <opR> )  . test if one permutation is smaller than another
**
**  'LtPerm' returns  'true' if the permutation <opL>  is strictly  less than
**  the permutation  <opR>.  Permutations are  ordered lexicographically with
**  respect to the images of 1,2,.., etc.
*/
libGAP_Int             libGAP_LtPerm22 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt2 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt2 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt                p;              /* loop variable                   */

    /* get the degrees of the permutations                                 */
    degL = libGAP_DEG_PERM2(opL);
    degR = libGAP_DEG_PERM2(opR);

    /* set up the pointers                                                 */
    ptL = libGAP_ADDR_PERM2(opL);
    ptR = libGAP_ADDR_PERM2(opR);

    /* search for a difference and return if you find one                  */
    if ( degL <= degR ) {
        for ( p = 0; p < degL; p++ )
            if ( *(ptL++) != *(ptR++) ) {
                if ( *(--ptL) < *(--ptR) )  return 1L ;
                else                        return 0L;
	    }
        for ( p = degL; p < degR; p++ )
            if (        p != *(ptR++) ) {
                if (        p < *(--ptR) )  return 1L ;
                else                        return 0L;
	    }
    }
    else {
        for ( p = 0; p < degR; p++ )
            if ( *(ptL++) != *(ptR++) ) {
                if ( *(--ptL) < *(--ptR) )  return 1L ;
                else                        return 0L;
	    }
        for ( p = degR; p < degL; p++ )
            if ( *(ptL++) != p ) {
                if ( *(--ptL) <        p )  return 1L ;
                else                        return 0L;
	    }
    }

    /* otherwise they must be equal                                        */
    return 0L;
}

libGAP_Int             libGAP_LtPerm24 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt2 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt4 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt                p;              /* loop variable                   */

    /* get the degrees of the permutations                                 */
    degL = libGAP_DEG_PERM2(opL);
    degR = libGAP_DEG_PERM4(opR);

    /* set up the pointers                                                 */
    ptL = libGAP_ADDR_PERM2(opL);
    ptR = libGAP_ADDR_PERM4(opR);

    /* search for a difference and return if you find one                  */
    if ( degL <= degR ) {
        for ( p = 0; p < degL; p++ )
            if ( *(ptL++) != *(ptR++) ) {
                if ( *(--ptL) < *(--ptR) )  return 1L ;
                else                        return 0L;
	    }
        for ( p = degL; p < degR; p++ )
            if (        p != *(ptR++) ) {
                if (        p < *(--ptR) )  return 1L ;
                else                        return 0L;
	    }
    }
    else {
        for ( p = 0; p < degR; p++ )
            if ( *(ptL++) != *(ptR++) ) {
                if ( *(--ptL) < *(--ptR) )  return 1L ;
                else                        return 0L;
	    }
        for ( p = degR; p < degL; p++ )
            if ( *(ptL++) != p ) {
                if ( *(--ptL) <        p )  return 1L ;
                else                        return 0L;
	    }
    }

    /* otherwise they must be equal                                        */
    return 0L;
}

libGAP_Int             libGAP_LtPerm42 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt4 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt2 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt                p;              /* loop variable                   */

    /* get the degrees of the permutations                                 */
    degL = libGAP_DEG_PERM4(opL);
    degR = libGAP_DEG_PERM2(opR);

    /* set up the pointers                                                 */
    ptL = libGAP_ADDR_PERM4(opL);
    ptR = libGAP_ADDR_PERM2(opR);

    /* search for a difference and return if you find one                  */
    if ( degL <= degR ) {
        for ( p = 0; p < degL; p++ )
            if ( *(ptL++) != *(ptR++) ) {
                if ( *(--ptL) < *(--ptR) )  return 1L ;
                else                        return 0L;
	    }
        for ( p = degL; p < degR; p++ )
            if (        p != *(ptR++) ) {
                if (        p < *(--ptR) )  return 1L ;
                else                        return 0L;
	    }
    }
    else {
        for ( p = 0; p < degR; p++ )
            if ( *(ptL++) != *(ptR++) ) {
                if ( *(--ptL) < *(--ptR) )  return 1L ;
                else                        return 0L;
	    }
        for ( p = degR; p < degL; p++ )
            if ( *(ptL++) != p ) {
                if ( *(--ptL) <        p )  return 1L ;
                else                        return 0L;
	    }
    }

    /* otherwise they must be equal                                        */
    return 0L;
}

libGAP_Int             libGAP_LtPerm44 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt4 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt4 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt                p;              /* loop variable                   */

    /* get the degrees of the permutations                                 */
    degL = libGAP_DEG_PERM4(opL);
    degR = libGAP_DEG_PERM4(opR);

    /* set up the pointers                                                 */
    ptL = libGAP_ADDR_PERM4(opL);
    ptR = libGAP_ADDR_PERM4(opR);

    /* search for a difference and return if you find one                  */
    if ( degL <= degR ) {
        for ( p = 0; p < degL; p++ )
            if ( *(ptL++) != *(ptR++) ) {
                if ( *(--ptL) < *(--ptR) )  return 1L ;
                else                        return 0L;
	    }
        for ( p = degL; p < degR; p++ )
            if (        p != *(ptR++) ) {
                if (        p < *(--ptR) )  return 1L ;
                else                        return 0L;
	    }
    }
    else {
        for ( p = 0; p < degR; p++ )
            if ( *(ptL++) != *(ptR++) ) {
                if ( *(--ptL) < *(--ptR) )  return 1L ;
                else                        return 0L;
	    }
        for ( p = degR; p < degL; p++ )
            if ( *(ptL++) != p ) {
                if ( *(--ptL) <        p )  return 1L ;
                else                        return 0L;
	    }
    }

    /* otherwise they must be equal                                        */
    return 0L;
}


/****************************************************************************
**
*F  ProdPerm( <opL>, <opR> )  . . . . . . . . . . . . product of permutations
**
**  'ProdPerm' returns the product of the two permutations <opL> and <opR>.
**
**  This is a little bit tuned but should be sufficiently easy to understand.
*/
libGAP_Obj             libGAP_ProdPerm22 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 prd;            /* handle of the product (result)  */
    libGAP_UInt                degP;           /* degree of the product           */
    libGAP_UInt2 *             ptP;            /* pointer to the product          */
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt2 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt2 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt                p;              /* loop variable                   */

    /* compute the size of the result and allocate a bag                   */
    degL = libGAP_DEG_PERM2(opL);
    degR = libGAP_DEG_PERM2(opR);
    degP = degL < degR ? degR : degL;
    prd  = libGAP_NEW_PERM2( degP );

    /* set up the pointers                                                 */
    ptL = libGAP_ADDR_PERM2(opL);
    ptR = libGAP_ADDR_PERM2(opR);
    ptP = libGAP_ADDR_PERM2(prd);

    /* if the left (inner) permutation has smaller degree, it is very easy */
    if ( degL <= degR ) {
        for ( p = 0; p < degL; p++ )
            *(ptP++) = ptR[ *(ptL++) ];
        for ( p = degL; p < degR; p++ )
            *(ptP++) = ptR[ p ];
    }

    /* otherwise we have to use the macro 'IMAGE'                          */
    else {
        for ( p = 0; p < degL; p++ )
            *(ptP++) = libGAP_IMAGE( ptL[ p ], ptR, degR );
    }

    /* return the result                                                   */
    return prd;
}


libGAP_Obj             libGAP_ProdPerm24 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 prd;            /* handle of the product (result)  */
    libGAP_UInt                degP;           /* degree of the product           */
    libGAP_UInt4 *             ptP;            /* pointer to the product          */
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt2 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt4 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt                p;              /* loop variable                   */

    /* compute the size of the result and allocate a bag                   */
    degL = libGAP_DEG_PERM2(opL);
    degR = libGAP_DEG_PERM4(opR);
    degP = degL < degR ? degR : degL;
    prd  = libGAP_NEW_PERM4( degP );

    /* set up the pointers                                                 */
    ptL = libGAP_ADDR_PERM2(opL);
    ptR = libGAP_ADDR_PERM4(opR);
    ptP = libGAP_ADDR_PERM4(prd);

    /* if the left (inner) permutation has smaller degree, it is very easy */
    if ( degL <= degR ) {
        for ( p = 0; p < degL; p++ )
            *(ptP++) = ptR[ *(ptL++) ];
        for ( p = degL; p < degR; p++ )
            *(ptP++) = ptR[ p ];
    }

    /* otherwise we have to use the macro 'IMAGE'                          */
    else {
        for ( p = 0; p < degL; p++ )
            *(ptP++) = libGAP_IMAGE( ptL[ p ], ptR, degR );
    }

    /* return the result                                                   */
    return prd;
}


libGAP_Obj             libGAP_ProdPerm42 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 prd;            /* handle of the product (result)  */
    libGAP_UInt                degP;           /* degree of the product           */
    libGAP_UInt4 *             ptP;            /* pointer to the product          */
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt4 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt2 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt                p;              /* loop variable                   */

    /* compute the size of the result and allocate a bag                   */
    degL = libGAP_DEG_PERM4(opL);
    degR = libGAP_DEG_PERM2(opR);
    degP = degL < degR ? degR : degL;
    prd  = libGAP_NEW_PERM4( degP );

    /* set up the pointers                                                 */
    ptL = libGAP_ADDR_PERM4(opL);
    ptR = libGAP_ADDR_PERM2(opR);
    ptP = libGAP_ADDR_PERM4(prd);

    /* if the left (inner) permutation has smaller degree, it is very easy */
    if ( degL <= degR ) {
        for ( p = 0; p < degL; p++ )
            *(ptP++) = ptR[ *(ptL++) ];
        for ( p = degL; p < degR; p++ )
            *(ptP++) = ptR[ p ];
    }

    /* otherwise we have to use the macro 'IMAGE'                          */
    else {
        for ( p = 0; p < degL; p++ )
            *(ptP++) = libGAP_IMAGE( ptL[ p ], ptR, degR );
    }

    /* return the result                                                   */
    return prd;
}


libGAP_Obj             libGAP_ProdPerm44 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 prd;            /* handle of the product (result)  */
    libGAP_UInt                degP;           /* degree of the product           */
    libGAP_UInt4 *             ptP;            /* pointer to the product          */
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt4 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt4 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt                p;              /* loop variable                   */

    /* compute the size of the result and allocate a bag                   */
    degL = libGAP_DEG_PERM4(opL);
    degR = libGAP_DEG_PERM4(opR);
    degP = degL < degR ? degR : degL;
    prd  = libGAP_NEW_PERM4( degP );

    /* set up the pointers                                                 */
    ptL = libGAP_ADDR_PERM4(opL);
    ptR = libGAP_ADDR_PERM4(opR);
    ptP = libGAP_ADDR_PERM4(prd);

    /* if the left (inner) permutation has smaller degree, it is very easy */
    if ( degL <= degR ) {
        for ( p = 0; p < degL; p++ )
            *(ptP++) = ptR[ *(ptL++) ];
        for ( p = degL; p < degR; p++ )
            *(ptP++) = ptR[ p ];
    }

    /* otherwise we have to use the macro 'IMAGE'                          */
    else {
        for ( p = 0; p < degL; p++ )
            *(ptP++) = libGAP_IMAGE( ptL[ p ], ptR, degR );
    }

    /* return the result                                                   */
    return prd;
}

libGAP_Obj libGAP_ProdPerm44Cooperman(
    libGAP_Obj                 opL,
    libGAP_Obj                 opR,
    libGAP_UInt                logBucketSize)
{
    libGAP_Obj                 prd;            /* handle of the product (result)  */
    libGAP_UInt                degP;           /* degree of the product           */
    libGAP_UInt4 *             ptP;            /* pointer to the product          */
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt4 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt4 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt                p;              /* loop variable                   */
    libGAP_UInt4 *             ptB;
    libGAP_UInt4 **             ptBP;
    libGAP_UInt                nBuckets;
    libGAP_Obj                 bucketPointers;
    libGAP_UInt4               im;
    libGAP_UInt                bucketSize;

    /* compute the size of the result and allocate a bag                   */
    degL = libGAP_DEG_PERM4(opL);
    degR = libGAP_DEG_PERM4(opR);
    degP = degL < degR ? degR : degL;
    prd  = libGAP_NEW_PERM4( degP );
    bucketSize = 1 << logBucketSize;
    nBuckets = (degP+(bucketSize-1))>>logBucketSize;
    if (libGAP_SIZE_BAG(libGAP_TmpPerm) < sizeof(libGAP_Obj)+4*degP)
      libGAP_ResizeBag(libGAP_TmpPerm, sizeof(libGAP_Obj)+4*degP);
    bucketPointers = libGAP_NewBag(libGAP_T_DATOBJ, sizeof(libGAP_Obj)+sizeof(libGAP_UInt4 *)*nBuckets);
    
    

    /* set up the pointers no GC now                                        */
    ptL = libGAP_ADDR_PERM4(opL);
    ptR = libGAP_ADDR_PERM4(opR);
    ptP = libGAP_ADDR_PERM4(prd);
    ptB = libGAP_ADDR_PERM4(libGAP_TmpPerm);
    ptBP = (libGAP_UInt4 **)(libGAP_ADDR_OBJ(bucketPointers)+1);
    for (p = 0; p < nBuckets; p++)
      ptBP[p] = ptB+ (p << logBucketSize);
    
    /* Pass 1 */
    for (p = 0; p < degL; p++)
      {
	im = ptL[p];
	*(ptBP[im>>logBucketSize]++) = im;
      }
    for (; p < degP; p++)
	*(ptBP[p>>logBucketSize]++) = p;
    
    /* Pass 2 */

    if (degR < degP)
      for (p = 0; p < degP; p++)
	{
	  im = ptB[p];
	  if (im < degR)
	    ptB[p] = ptR[im];	
	}
    else
      for (p = 0; p < degP; p++)
	{
	  im = ptB[p];
	  ptB[p] = ptR[im];	
	}
      
    /* Pass 3 */
    for (p = 0; p < nBuckets; p++)
      ptBP[p] = ptB+ (p << logBucketSize);
    for (p = 0; p < degL; p++)
      {
	im = ptL[p];
	ptP[p] = *(ptBP[im>>logBucketSize]++);
      }
    for (;p < degP; p++)
      {
	ptP[p] = *(ptBP[p>>logBucketSize]++);
      }

    /* return the result                                                   */
    return prd;
}

libGAP_Obj libGAP_FuncMUL_PERMS_COOPERMAN( libGAP_Obj self, libGAP_Obj permL, libGAP_Obj permR, libGAP_Obj logbucketSize) {
  return libGAP_ProdPerm44Cooperman(permL, permR, libGAP_INT_INTOBJ(logbucketSize));
}


/****************************************************************************
**
*F  QuoPerm( <opL>, <opR> ) . . . . . . . . . . . .  quotient of permutations
**
**  'QuoPerm' returns the quotient of the permutations <opL> and <opR>, i.e.,
**  the product '<opL>\*<opR>\^-1'.
**
**  Unfortunatly this can not be done in <degree> steps, we need 2 * <degree>
**  steps.
*/
libGAP_Obj             libGAP_QuoPerm22 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 quo;            /* handle of the quotient (result) */
    libGAP_UInt                degQ;           /* degree of the quotient          */
    libGAP_UInt2 *             ptQ;            /* pointer to the quotient         */
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt2 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt2 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt2 *             ptI;            /* pointer to the inverse          */
    libGAP_UInt                p;              /* loop variable                   */

    /* compute the size of the result and allocate a bag                   */
    degL = libGAP_DEG_PERM2(opL);
    degR = libGAP_DEG_PERM2(opR);
    degQ = degL < degR ? degR : degL;
    quo  = libGAP_NEW_PERM2( degQ );

    /* make sure that the buffer bag is large enough to hold the inverse   */
    if ( libGAP_SIZE_OBJ(libGAP_TmpPerm) < libGAP_SIZE_OBJ(opR) ) {
        libGAP_ResizeBag( libGAP_TmpPerm, libGAP_SIZE_OBJ(opR) );
    }

    /* invert the right permutation into the buffer bag                    */
    ptI = libGAP_ADDR_PERM2(libGAP_TmpPerm);
    ptR = libGAP_ADDR_PERM2(opR);
    for ( p = 0; p < degR; p++ )
        ptI[ *ptR++ ] = p;

    /* multiply the left permutation with the inverse                      */
    ptL = libGAP_ADDR_PERM2(opL);
    ptI = libGAP_ADDR_PERM2(libGAP_TmpPerm);
    ptQ = libGAP_ADDR_PERM2(quo);
    if ( degL <= degR ) {
        for ( p = 0; p < degL; p++ )
            *(ptQ++) = ptI[ *(ptL++) ];
        for ( p = degL; p < degR; p++ )
            *(ptQ++) = ptI[ p ];
    }
    else {
        for ( p = 0; p < degL; p++ )

            *(ptQ++) = libGAP_IMAGE( ptL[ p ], ptI, degR );
    }

    /* return the result                                                   */
    return quo;
}

libGAP_Obj             libGAP_QuoPerm24 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 quo;            /* handle of the quotient (result) */
    libGAP_UInt                degQ;           /* degree of the quotient          */
    libGAP_UInt4 *             ptQ;            /* pointer to the quotient         */
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt2 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt4 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt4 *             ptI;            /* pointer to the inverse          */
    libGAP_UInt                p;              /* loop variable                   */

    /* compute the size of the result and allocate a bag                   */
    degL = libGAP_DEG_PERM2(opL);
    degR = libGAP_DEG_PERM4(opR);
    degQ = degL < degR ? degR : degL;
    quo  = libGAP_NEW_PERM4( degQ );

    /* make sure that the buffer bag is large enough to hold the inverse   */
    if ( libGAP_SIZE_OBJ(libGAP_TmpPerm) < libGAP_SIZE_OBJ(opR) ) {
        libGAP_ResizeBag( libGAP_TmpPerm, libGAP_SIZE_OBJ(opR) );
    }

    /* invert the right permutation into the buffer bag                    */
    ptI = libGAP_ADDR_PERM4(libGAP_TmpPerm);
    ptR = libGAP_ADDR_PERM4(opR);
    for ( p = 0; p < degR; p++ )
        ptI[ *ptR++ ] = p;

    /* multiply the left permutation with the inverse                      */
    ptL = libGAP_ADDR_PERM2(opL);
    ptI = libGAP_ADDR_PERM4(libGAP_TmpPerm);
    ptQ = libGAP_ADDR_PERM4(quo);
    if ( degL <= degR ) {
        for ( p = 0; p < degL; p++ )
            *(ptQ++) = ptI[ *(ptL++) ];
        for ( p = degL; p < degR; p++ )
            *(ptQ++) = ptI[ p ];
    }
    else {
        for ( p = 0; p < degL; p++ )
            *(ptQ++) = libGAP_IMAGE( ptL[ p ], ptI, degR );
    }

    /* return the result                                                   */
    return quo;
}

libGAP_Obj             libGAP_QuoPerm42 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 quo;            /* handle of the quotient (result) */
    libGAP_UInt                degQ;           /* degree of the quotient          */
    libGAP_UInt4 *             ptQ;            /* pointer to the quotient         */
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt4 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt2 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt2 *             ptI;            /* pointer to the inverse          */
    libGAP_UInt                p;              /* loop variable                   */

    /* compute the size of the result and allocate a bag                   */
    degL = libGAP_DEG_PERM4(opL);
    degR = libGAP_DEG_PERM2(opR);
    degQ = degL < degR ? degR : degL;
    quo  = libGAP_NEW_PERM4( degQ );

    /* make sure that the buffer bag is large enough to hold the inverse   */
    if ( libGAP_SIZE_OBJ(libGAP_TmpPerm) < libGAP_SIZE_OBJ(opR) ) {
        libGAP_ResizeBag( libGAP_TmpPerm, libGAP_SIZE_OBJ(opR) );
    }

    /* invert the right permutation into the buffer bag                    */
    ptI = libGAP_ADDR_PERM2(libGAP_TmpPerm);
    ptR = libGAP_ADDR_PERM2(opR);
    for ( p = 0; p < degR; p++ )
        ptI[ *ptR++ ] = p;

    /* multiply the left permutation with the inverse                      */
    ptL = libGAP_ADDR_PERM4(opL);
    ptI = libGAP_ADDR_PERM2(libGAP_TmpPerm);
    ptQ = libGAP_ADDR_PERM4(quo);
    if ( degL <= degR ) {
        for ( p = 0; p < degL; p++ )
            *(ptQ++) = ptI[ *(ptL++) ];
        for ( p = degL; p < degR; p++ )
            *(ptQ++) = ptI[ p ];
    }
    else {
        for ( p = 0; p < degL; p++ )
            *(ptQ++) = libGAP_IMAGE( ptL[ p ], ptI, degR );
    }

    /* return the result                                                   */
    return quo;
}

libGAP_Obj             libGAP_QuoPerm44 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 quo;            /* handle of the quotient (result) */
    libGAP_UInt                degQ;           /* degree of the quotient          */
    libGAP_UInt4 *             ptQ;            /* pointer to the quotient         */
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt4 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt4 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt4 *             ptI;            /* pointer to the inverse          */
    libGAP_UInt                p;              /* loop variable                   */

    /* compute the size of the result and allocate a bag                   */
    degL = libGAP_DEG_PERM4(opL);
    degR = libGAP_DEG_PERM4(opR);
    degQ = degL < degR ? degR : degL;
    quo  = libGAP_NEW_PERM4( degQ );

    /* make sure that the buffer bag is large enough to hold the inverse   */
    if ( libGAP_SIZE_OBJ(libGAP_TmpPerm) < libGAP_SIZE_OBJ(opR) ) {
        libGAP_ResizeBag( libGAP_TmpPerm, libGAP_SIZE_OBJ(opR) );
    }

    /* invert the right permutation into the buffer bag                    */
    ptI = libGAP_ADDR_PERM4(libGAP_TmpPerm);
    ptR = libGAP_ADDR_PERM4(opR);
    for ( p = 0; p < degR; p++ )
        ptI[ *ptR++ ] = p;

    /* multiply the left permutation with the inverse                      */
    ptL = libGAP_ADDR_PERM4(opL);
    ptI = libGAP_ADDR_PERM4(libGAP_TmpPerm);
    ptQ = libGAP_ADDR_PERM4(quo);
    if ( degL <= degR ) {
        for ( p = 0; p < degL; p++ )
            *(ptQ++) = ptI[ *(ptL++) ];
        for ( p = degL; p < degR; p++ )
            *(ptQ++) = ptI[ p ];
    }
    else {
        for ( p = 0; p < degL; p++ )
            *(ptQ++) = libGAP_IMAGE( ptL[ p ], ptI, degR );
    }

    /* return the result                                                   */
    return quo;
}


/****************************************************************************
**
*F  LQuoPerm( <opL>, <opR> )  . . . . . . . . . left quotient of permutations
**
**  'LQuoPerm' returns the  left quotient of  the  two permutations <opL> and
**  <opR>, i.e., the value of '<opL>\^-1*<opR>', which sometimes comes handy.
**
**  This can be done as fast as a single multiplication or inversion.
*/
libGAP_Obj             libGAP_LQuoPerm22 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 mod;            /* handle of the quotient (result) */
    libGAP_UInt                degM;           /* degree of the quotient          */
    libGAP_UInt2 *             ptM;            /* pointer to the quotient         */
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt2 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt2 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt                p;              /* loop variable                   */

    /* compute the size of the result and allocate a bag                   */
    degL = libGAP_DEG_PERM2(opL);
    degR = libGAP_DEG_PERM2(opR);
    degM = degL < degR ? degR : degL;
    mod = libGAP_NEW_PERM2( degM );

    /* set up the pointers                                                 */
    ptL = libGAP_ADDR_PERM2(opL);
    ptR = libGAP_ADDR_PERM2(opR);
    ptM = libGAP_ADDR_PERM2(mod);

    /* its one thing if the left (inner) permutation is smaller            */
    if ( degL <= degR ) {
        for ( p = 0; p < degL; p++ )
            ptM[ *(ptL++) ] = *(ptR++);
        for ( p = degL; p < degR; p++ )
            ptM[ p ] = *(ptR++);
    }

    /* and another if the right (outer) permutation is smaller             */
    else {
        for ( p = 0; p < degR; p++ )
            ptM[ *(ptL++) ] = *(ptR++);
        for ( p = degR; p < degL; p++ )
            ptM[ *(ptL++) ] = p;
    }

    /* return the result                                                   */
    return mod;
}

libGAP_Obj             libGAP_LQuoPerm24 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 mod;            /* handle of the quotient (result) */
    libGAP_UInt                degM;           /* degree of the quotient          */
    libGAP_UInt4 *             ptM;            /* pointer to the quotient         */
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt2 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt4 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt                p;              /* loop variable                   */

    /* compute the size of the result and allocate a bag                   */
    degL = libGAP_DEG_PERM2(opL);
    degR = libGAP_DEG_PERM4(opR);
    degM = degL < degR ? degR : degL;
    mod = libGAP_NEW_PERM4( degM );

    /* set up the pointers                                                 */
    ptL = libGAP_ADDR_PERM2(opL);
    ptR = libGAP_ADDR_PERM4(opR);
    ptM = libGAP_ADDR_PERM4(mod);

    /* its one thing if the left (inner) permutation is smaller            */
    if ( degL <= degR ) {
        for ( p = 0; p < degL; p++ )
            ptM[ *(ptL++) ] = *(ptR++);
        for ( p = degL; p < degR; p++ )
            ptM[ p ] = *(ptR++);
    }

    /* and another if the right (outer) permutation is smaller             */
    else {
        for ( p = 0; p < degR; p++ )
            ptM[ *(ptL++) ] = *(ptR++);
        for ( p = degR; p < degL; p++ )
            ptM[ *(ptL++) ] = p;
    }

    /* return the result                                                   */
    return mod;
}

libGAP_Obj             libGAP_LQuoPerm42 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 mod;            /* handle of the quotient (result) */
    libGAP_UInt                degM;           /* degree of the quotient          */
    libGAP_UInt4 *             ptM;            /* pointer to the quotient         */
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt4 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt2 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt                p;              /* loop variable                   */

    /* compute the size of the result and allocate a bag                   */
    degL = libGAP_DEG_PERM4(opL);
    degR = libGAP_DEG_PERM2(opR);
    degM = degL < degR ? degR : degL;
    mod = libGAP_NEW_PERM4( degM );

    /* set up the pointers                                                 */
    ptL = libGAP_ADDR_PERM4(opL);
    ptR = libGAP_ADDR_PERM2(opR);
    ptM = libGAP_ADDR_PERM4(mod);

    /* its one thing if the left (inner) permutation is smaller            */
    if ( degL <= degR ) {
        for ( p = 0; p < degL; p++ )
            ptM[ *(ptL++) ] = *(ptR++);
        for ( p = degL; p < degR; p++ )
            ptM[ p ] = *(ptR++);
    }

    /* and another if the right (outer) permutation is smaller             */
    else {
        for ( p = 0; p < degR; p++ )
            ptM[ *(ptL++) ] = *(ptR++);
        for ( p = degR; p < degL; p++ )
            ptM[ *(ptL++) ] = p;
    }

    /* return the result                                                   */
    return mod;
}

libGAP_Obj             libGAP_LQuoPerm44 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 mod;            /* handle of the quotient (result) */
    libGAP_UInt                degM;           /* degree of the quotient          */
    libGAP_UInt4 *             ptM;            /* pointer to the quotient         */
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt4 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt4 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt                p;              /* loop variable                   */

    /* compute the size of the result and allocate a bag                   */
    degL = libGAP_DEG_PERM4(opL);
    degR = libGAP_DEG_PERM4(opR);
    degM = degL < degR ? degR : degL;
    mod = libGAP_NEW_PERM4( degM );

    /* set up the pointers                                                 */
    ptL = libGAP_ADDR_PERM4(opL);
    ptR = libGAP_ADDR_PERM4(opR);
    ptM = libGAP_ADDR_PERM4(mod);

    /* its one thing if the left (inner) permutation is smaller            */
    if ( degL <= degR ) {
        for ( p = 0; p < degL; p++ )
            ptM[ *(ptL++) ] = *(ptR++);
        for ( p = degL; p < degR; p++ )
            ptM[ p ] = *(ptR++);
    }

    /* and another if the right (outer) permutation is smaller             */
    else {
        for ( p = 0; p < degR; p++ )
            ptM[ *(ptL++) ] = *(ptR++);
        for ( p = degR; p < degL; p++ )
            ptM[ *(ptL++) ] = p;
    }

    /* return the result                                                   */
    return mod;
}


/****************************************************************************
**
*F  PowPermInt( <opL>, <opR> )  . . . . . . .  integer power of a permutation
**
**  'PowPermInt' returns the <opR>-th power  of the permutation <opL>.  <opR>
**  must be a small integer.
**
**  This repeatedly applies the permutation <opR> to all points  which  seems
**  to be faster than binary powering, and does not need  temporary  storage.
*/

libGAP_Obj libGAP_InvPerm4Cooperman ( libGAP_Obj perm, libGAP_UInt logBucketSize )
{
  libGAP_UInt deg = libGAP_DEG_PERM4(perm);
  libGAP_UInt bucketSize = 1 << logBucketSize;
  libGAP_UInt nBuckets;
  libGAP_Obj bucketPointers;
  libGAP_Obj inv;
  libGAP_UInt4* ptP;
  libGAP_UInt4* ptI;
  libGAP_UInt4** ptBP;
  libGAP_UInt4* ptB;
  libGAP_UInt4 p;

  if (libGAP_SIZE_BAG(libGAP_TmpPerm) < sizeof(libGAP_Obj)+4*2*deg)
    libGAP_ResizeBag(libGAP_TmpPerm, sizeof(libGAP_Obj)+4*2*deg);
  nBuckets = (deg+(bucketSize-1))>>logBucketSize;
  bucketPointers = libGAP_NewBag(libGAP_T_DATOBJ, sizeof(libGAP_Obj)+sizeof(libGAP_UInt4 *)*nBuckets);
  inv = libGAP_NEW_PERM4(deg);
  
  ptP = libGAP_ADDR_PERM4(perm);
  ptI = libGAP_ADDR_PERM4(inv);
  ptB = libGAP_ADDR_PERM4(libGAP_TmpPerm);
  ptBP = (libGAP_UInt4 **)(libGAP_ADDR_OBJ(bucketPointers)+1);
  for (p = 0; p < nBuckets; p++)
    ptBP[p] = ptB+ 2*(p << logBucketSize);
  
  for (p = 0;p<deg; p++)
    {
      libGAP_UInt4 im = ptP[p];
      libGAP_UInt b = im >>logBucketSize;
      libGAP_UInt4 * pt = ptBP[b];
      *(pt++) = p;
      *(pt++) = im;
      ptBP[b] = pt;
    }
  for (p = 0;p<deg; p++)
    {
      libGAP_UInt4 im =*(ptB++);
      ptI[*(ptB++)] = im;
    }
  return inv;

}

libGAP_Obj libGAP_FuncINV_PERM_COOPERMAN( libGAP_Obj self, libGAP_Obj perm, libGAP_Obj logBucketSize)
{
  return libGAP_InvPerm4Cooperman(perm, libGAP_INT_INTOBJ(logBucketSize));
}

libGAP_Obj libGAP_FuncINV_PERM_SIMPLE( libGAP_Obj self, libGAP_Obj perm)
{
  libGAP_UInt deg = libGAP_DEG_PERM4(perm);
  libGAP_Obj inv = libGAP_NEW_PERM4(deg);
  libGAP_UInt4 *ptP = libGAP_ADDR_PERM4(perm);
  libGAP_UInt4 *ptI = libGAP_ADDR_PERM4(inv);
  libGAP_UInt p;
  for (p = 0; p < deg; p++)
    ptI[ptP[p]] = p;
  return inv;
}

libGAP_Obj libGAP_LQuoPerm4Cooperman ( libGAP_Obj perm1, libGAP_Obj perm2, libGAP_UInt logBucketSize )
{
  libGAP_UInt deg1 = libGAP_DEG_PERM4(perm1);
  libGAP_UInt deg2 = libGAP_DEG_PERM4(perm2);
  libGAP_UInt degQ = (deg1 > deg2) ? deg1 : deg2;
  libGAP_UInt degmin = deg1+deg2 - degQ;
  libGAP_UInt bucketSize = 1 << logBucketSize;
  libGAP_UInt nBuckets;
  libGAP_Obj bucketPointers;
  libGAP_Obj quo;
  libGAP_UInt4* ptP1;
  libGAP_UInt4* ptP2;
  libGAP_UInt4* ptQ;
  libGAP_UInt4** ptBP;
  libGAP_UInt4* ptB;
  libGAP_UInt4 p;

  if (libGAP_SIZE_BAG(libGAP_TmpPerm) < sizeof(libGAP_Obj)+4*2*degQ)
    libGAP_ResizeBag(libGAP_TmpPerm, sizeof(libGAP_Obj)+4*2*degQ);
  nBuckets = (degQ+(bucketSize-1))>>logBucketSize;
  bucketPointers = libGAP_NewBag(libGAP_T_DATOBJ, sizeof(libGAP_Obj)+sizeof(libGAP_UInt4 *)*nBuckets);
  quo = libGAP_NEW_PERM4(degQ);
  
  ptP1 = libGAP_ADDR_PERM4(perm1);
  ptP2 = libGAP_ADDR_PERM4(perm2);
  ptQ = libGAP_ADDR_PERM4(quo);
  ptB = libGAP_ADDR_PERM4(libGAP_TmpPerm);
  ptBP = (libGAP_UInt4 **)(libGAP_ADDR_OBJ(bucketPointers)+1);
  for (p = 0; p < nBuckets; p++)
    ptBP[p] = ptB+ 2*(p << logBucketSize);
  
  for (p = 0;p<degmin; p++)
    {
      libGAP_UInt4 im = ptP2[p];
      libGAP_UInt b = im >>logBucketSize;
      libGAP_UInt4 *pt = ptBP[b];
      *(pt++) = ptP1[p];
      *(pt++) = im;
      ptBP[b] = pt;
    }
  for (; p < deg1; p++)
    {
      libGAP_UInt4 im = p;
      libGAP_UInt b = im >>logBucketSize;
      *(ptBP[b]++) = ptP1[p];
      *(ptBP[b]++) = im;

    }
  for (; p < deg2; p++)
    {
      libGAP_UInt4 im = ptP2[p];
      libGAP_UInt b = im >>logBucketSize;
      *(ptBP[b]++) = p;
      *(ptBP[b]++) = im;

    }
  for (p = 0;p<degQ; p++)
    {
      libGAP_UInt4 im =*(ptB++);
      ptQ[*(ptB++)] = im;
    }
  return quo;

}

libGAP_Obj libGAP_FuncLQUO_PERMS_COOPERMAN( libGAP_Obj self, libGAP_Obj perm1, libGAP_Obj perm2, libGAP_Obj logBucketSize)
{
  return libGAP_LQuoPerm4Cooperman(perm1, perm2, libGAP_INT_INTOBJ(logBucketSize));
}

libGAP_Obj             libGAP_PowPerm2Int (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 pow;            /* handle of the power (result)    */
    libGAP_UInt2 *             ptP;            /* pointer to the power            */
    libGAP_UInt2 *             ptL;            /* pointer to the permutation      */
    libGAP_UInt2 *             ptKnown;        /* pointer to temporary bag        */
    libGAP_UInt                deg;            /* degree of the permutation       */
    libGAP_Int                 exp,  e;        /* exponent (right operand)        */
    libGAP_UInt                len;            /* length of cycle (result)        */
    libGAP_UInt                p,  q,  r;      /* loop variables                  */

    
    /* handle zeroth and first powers separately */
    if ( opR == libGAP_INTOBJ_INT(0)) 
      return libGAP_IdentityPerm;
    if ( opR == libGAP_INTOBJ_INT(1))
      return opL;

    /* get the operands and allocate a result bag                          */
    deg = libGAP_DEG_PERM2(opL);
    pow = libGAP_NEW_PERM2( deg );

    /* compute the power by repeated mapping for small positive exponents  */
    if ( libGAP_TNUM_OBJ(opR) == libGAP_T_INT
      && 2 <= libGAP_INT_INTOBJ(opR) && libGAP_INT_INTOBJ(opR) < 8 ) {

        /* get pointer to the permutation and the power                    */
        exp = libGAP_INT_INTOBJ(opR);
        ptL = libGAP_ADDR_PERM2(opL);
        ptP = libGAP_ADDR_PERM2(pow);

        /* loop over the points of the permutation                         */
        for ( p = 0; p < deg; p++ ) {
            q = p;
            for ( e = 0; e < exp; e++ )
                q = ptL[q];
            ptP[p] = q;
        }

    }

    /* compute the power by raising the cycles individually for large exps */
    else if ( libGAP_TNUM_OBJ(opR) == libGAP_T_INT && 8 <= libGAP_INT_INTOBJ(opR) ) {

        /* make sure that the buffer bag is large enough                   */
        if ( libGAP_SIZE_OBJ(libGAP_TmpPerm) < libGAP_SIZE_OBJ(opL) ) {
            libGAP_ResizeBag( libGAP_TmpPerm, libGAP_SIZE_OBJ(opL) );
        }
        ptKnown = libGAP_ADDR_PERM2(libGAP_TmpPerm);

        /* clear the buffer bag                                            */
        for ( p = 0; p < libGAP_DEG_PERM2(opL); p++ )
            ptKnown[p] = 0;

        /* get pointer to the permutation and the power                    */
        exp = libGAP_INT_INTOBJ(opR);
        ptL = libGAP_ADDR_PERM2(opL);
        ptP = libGAP_ADDR_PERM2(pow);

        /* loop over all cycles                                            */
        for ( p = 0; p < deg; p++ ) {

            /* if we haven't looked at this cycle so far                   */
            if ( ptKnown[p] == 0 ) {

                /* find the length of this cycle                           */
                len = 1;
                for ( q = ptL[p]; q != p; q = ptL[q] ) {
                    len++;  ptKnown[q] = 1;
                }

                /* raise this cycle to the power <exp> mod <len>           */
                r = p;
                for ( e = 0; e < exp % len; e++ )
                    r = ptL[r];
                ptP[p] = r;
                r = ptL[r];
                for ( q = ptL[p]; q != p; q = ptL[q] ) {
                    ptP[q] = r;
                    r = ptL[r];
                }

            }

        }


    }

    /* compute the power by raising the cycles individually for large exps */
    else if ( libGAP_TNUM_OBJ(opR) == libGAP_T_INTPOS ) {

        /* make sure that the buffer bag is large enough                   */
        if ( libGAP_SIZE_OBJ(libGAP_TmpPerm) < libGAP_SIZE_OBJ(opL) ) {
            libGAP_ResizeBag( libGAP_TmpPerm, libGAP_SIZE_OBJ(opL) );
        }
        ptKnown = libGAP_ADDR_PERM2(libGAP_TmpPerm);

        /* clear the buffer bag                                            */
        for ( p = 0; p < libGAP_DEG_PERM2(opL); p++ )
            ptKnown[p] = 0;

        /* get pointer to the permutation and the power                    */
        ptL = libGAP_ADDR_PERM2(opL);
        ptP = libGAP_ADDR_PERM2(pow);

        /* loop over all cycles                                            */
        for ( p = 0; p < deg; p++ ) {

            /* if we haven't looked at this cycle so far                   */
            if ( ptKnown[p] == 0 ) {

                /* find the length of this cycle                           */
                len = 1;
                for ( q = ptL[p]; q != p; q = ptL[q] ) {
                    len++;  ptKnown[q] = 1;
                }

                /* raise this cycle to the power <exp> mod <len>           */
                r = p;
                exp = libGAP_INT_INTOBJ( libGAP_ModInt( opR, libGAP_INTOBJ_INT(len) ) );
                for ( e = 0; e < exp; e++ )
                    r = ptL[r];
                ptP[p] = r;
                r = ptL[r];
                for ( q = ptL[p]; q != p; q = ptL[q] ) {
                    ptP[q] = r;
                    r = ptL[r];
                }

            }

        }

    }

    /* special case for inverting permutations                             */
    else if ( libGAP_TNUM_OBJ(opR) == libGAP_T_INT && libGAP_INT_INTOBJ(opR) == -1 ) {

        /* get pointer to the permutation and the power                    */
        ptL = libGAP_ADDR_PERM2(opL);
        ptP = libGAP_ADDR_PERM2(pow);

        /* invert the permutation                                          */
        for ( p = 0; p < deg; p++ )
            ptP[ *(ptL++) ] = p;

    }

    /* compute the power by repeated mapping for small negative exponents  */
    else if ( libGAP_TNUM_OBJ(opR) == libGAP_T_INT
          && -8 < libGAP_INT_INTOBJ(opR) && libGAP_INT_INTOBJ(opR) < 0 ) {

        /* get pointer to the permutation and the power                    */
        exp = -libGAP_INT_INTOBJ(opR);
        ptL = libGAP_ADDR_PERM2(opL);
        ptP = libGAP_ADDR_PERM2(pow);

        /* loop over the points                                            */
        for ( p = 0; p < deg; p++ ) {
            q = p;
            for ( e = 0; e < exp; e++ )
                q = ptL[q];
            ptP[q] = p;
        }

    }

    /* compute the power by raising the cycles individually for large exps */
    else if ( libGAP_TNUM_OBJ(opR) == libGAP_T_INT && libGAP_INT_INTOBJ(opR) <= -8 ) {

        /* make sure that the buffer bag is large enough                   */
        if ( libGAP_SIZE_OBJ(libGAP_TmpPerm) < libGAP_SIZE_OBJ(opL) ) {
            libGAP_ResizeBag( libGAP_TmpPerm, libGAP_SIZE_OBJ(opL) );
        }
        ptKnown = libGAP_ADDR_PERM2(libGAP_TmpPerm);

        /* clear the buffer bag                                            */
        for ( p = 0; p < libGAP_DEG_PERM2(opL); p++ )
            ptKnown[p] = 0;

        /* get pointer to the permutation and the power                    */
        exp = -libGAP_INT_INTOBJ(opR);
        ptL = libGAP_ADDR_PERM2(opL);
        ptP = libGAP_ADDR_PERM2(pow);

        /* loop over all cycles                                            */
        for ( p = 0; p < deg; p++ ) {

            /* if we haven't looked at this cycle so far                   */
            if ( ptKnown[p] == 0 ) {

                /* find the length of this cycle                           */
                len = 1;
                for ( q = ptL[p]; q != p; q = ptL[q] ) {
                    len++;  ptKnown[q] = 1;
                }

                /* raise this cycle to the power <exp> mod <len>           */
                r = p;
                for ( e = 0; e < exp % len; e++ )
                    r = ptL[r];
                ptP[r] = p;
                r = ptL[r];
                for ( q = ptL[p]; q != p; q = ptL[q] ) {
                    ptP[r] = q;
                    r = ptL[r];
                }

            }

        }

    }

    /* compute the power by raising the cycles individually for large exps */
    else if ( libGAP_TNUM_OBJ(opR) == libGAP_T_INTNEG ) {

        /* make sure that the buffer bag is large enough                   */
        if ( libGAP_SIZE_OBJ(libGAP_TmpPerm) < libGAP_SIZE_OBJ(opL) ) {
            libGAP_ResizeBag( libGAP_TmpPerm, libGAP_SIZE_OBJ(opL) );
        }
        ptKnown = libGAP_ADDR_PERM2(libGAP_TmpPerm);

        /* clear the buffer bag                                            */
        for ( p = 0; p < libGAP_DEG_PERM2(opL); p++ )
            ptKnown[p] = 0;

        /* get pointer to the permutation and the power                    */
        opR = libGAP_ProdInt( libGAP_INTOBJ_INT(-1), opR );
        ptL = libGAP_ADDR_PERM2(opL);
        ptP = libGAP_ADDR_PERM2(pow);

        /* loop over all cycles                                            */
        for ( p = 0; p < deg; p++ ) {

            /* if we haven't looked at this cycle so far                   */
            if ( ptKnown[p] == 0 ) {

                /* find the length of this cycle                           */
                len = 1;
                for ( q = ptL[p]; q != p; q = ptL[q] ) {
                    len++;  ptKnown[q] = 1;
                }

                /* raise this cycle to the power <exp> mod <len>           */
                r = p;
                exp = libGAP_INT_INTOBJ( libGAP_ModInt( opR, libGAP_INTOBJ_INT(len) ) );
                for ( e = 0; e < exp % len; e++ )
                    r = ptL[r];
                ptP[r] = p;
                r = ptL[r];
                for ( q = ptL[p]; q != p; q = ptL[q] ) {
                    ptP[r] = q;
                    r = ptL[r];
                }

            }

        }

    }

    /* return the result                                                   */
    return pow;
}

libGAP_Obj             libGAP_PowPerm4Int (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 pow;            /* handle of the power (result)    */
    libGAP_UInt4 *             ptP;            /* pointer to the power            */
    libGAP_UInt4 *             ptL;            /* pointer to the permutation      */
    libGAP_UInt4 *             ptKnown;        /* pointer to temporary bag        */
    libGAP_UInt                deg;            /* degree of the permutation       */
    libGAP_Int                 exp,  e;        /* exponent (right operand)        */
    libGAP_UInt                len;            /* length of cycle (result)        */
    libGAP_UInt                p,  q,  r;      /* loop variables                  */

    /* handle zeroth and first powers separately */
    if ( opR == libGAP_INTOBJ_INT(0)) 
      return libGAP_IdentityPerm;
    if ( opR == libGAP_INTOBJ_INT(1))
      return opL;

    /* get the operands and allocate a result bag                          */
    deg = libGAP_DEG_PERM4(opL);
    pow = libGAP_NEW_PERM4( deg );

    /* compute the power by repeated mapping for small positive exponents  */
    if ( libGAP_TNUM_OBJ(opR) == libGAP_T_INT
      && 0 <= libGAP_INT_INTOBJ(opR) && libGAP_INT_INTOBJ(opR) < 8 ) {

        /* get pointer to the permutation and the power                    */
        exp = libGAP_INT_INTOBJ(opR);
        ptL = libGAP_ADDR_PERM4(opL);
        ptP = libGAP_ADDR_PERM4(pow);

        /* loop over the points of the permutation                         */
        for ( p = 0; p < deg; p++ ) {
            q = p;
            for ( e = 0; e < exp; e++ )
                q = ptL[q];
            ptP[p] = q;
        }

    }

    /* compute the power by raising the cycles individually for large exps */
    else if ( libGAP_TNUM_OBJ(opR) == libGAP_T_INT && 8 <= libGAP_INT_INTOBJ(opR) ) {

        /* make sure that the buffer bag is large enough                   */
        if ( libGAP_SIZE_OBJ(libGAP_TmpPerm) < libGAP_SIZE_OBJ(opL) ) {
            libGAP_ResizeBag( libGAP_TmpPerm, libGAP_SIZE_OBJ(opL) );
        }
        ptKnown = libGAP_ADDR_PERM4(libGAP_TmpPerm);

        /* clear the buffer bag                                            */
        for ( p = 0; p < libGAP_DEG_PERM4(opL); p++ )
            ptKnown[p] = 0;

        /* get pointer to the permutation and the power                    */
        exp = libGAP_INT_INTOBJ(opR);
        ptL = libGAP_ADDR_PERM4(opL);
        ptP = libGAP_ADDR_PERM4(pow);

        /* loop over all cycles                                            */
        for ( p = 0; p < deg; p++ ) {

            /* if we haven't looked at this cycle so far                   */
            if ( ptKnown[p] == 0 ) {

                /* find the length of this cycle                           */
                len = 1;
                for ( q = ptL[p]; q != p; q = ptL[q] ) {
                    len++;  ptKnown[q] = 1;
                }

                /* raise this cycle to the power <exp> mod <len>           */
                r = p;
                for ( e = 0; e < exp % len; e++ )
                    r = ptL[r];
                ptP[p] = r;
                r = ptL[r];
                for ( q = ptL[p]; q != p; q = ptL[q] ) {
                    ptP[q] = r;
                    r = ptL[r];
                }

            }

        }

    }

    /* compute the power by raising the cycles individually for large exps */
    else if ( libGAP_TNUM_OBJ(opR) == libGAP_T_INTPOS ) {

        /* make sure that the buffer bag is large enough                   */
        if ( libGAP_SIZE_OBJ(libGAP_TmpPerm) < libGAP_SIZE_OBJ(opL) ) {
            libGAP_ResizeBag( libGAP_TmpPerm, libGAP_SIZE_OBJ(opL) );
        }
        ptKnown = libGAP_ADDR_PERM4(libGAP_TmpPerm);

        /* clear the buffer bag                                            */
        for ( p = 0; p < libGAP_DEG_PERM4(opL); p++ )
            ptKnown[p] = 0;

        /* get pointer to the permutation and the power                    */
        ptL = libGAP_ADDR_PERM4(opL);
        ptP = libGAP_ADDR_PERM4(pow);

        /* loop over all cycles                                            */
        for ( p = 0; p < deg; p++ ) {

            /* if we haven't looked at this cycle so far                   */
            if ( ptKnown[p] == 0 ) {

                /* find the length of this cycle                           */
                len = 1;
                for ( q = ptL[p]; q != p; q = ptL[q] ) {
                    len++;  ptKnown[q] = 1;
                }

                /* raise this cycle to the power <exp> mod <len>           */
                r = p;
                exp = libGAP_INT_INTOBJ( libGAP_ModInt( opR, libGAP_INTOBJ_INT(len) ) );
                for ( e = 0; e < exp; e++ )
                    r = ptL[r];
                ptP[p] = r;
                r = ptL[r];
                for ( q = ptL[p]; q != p; q = ptL[q] ) {
                    ptP[q] = r;
                    r = ptL[r];
                }

            }

        }

    }

    /* special case for inverting permutations                             */
    else if ( libGAP_TNUM_OBJ(opR) == libGAP_T_INT && libGAP_INT_INTOBJ(opR) == -1 ) {

        /* get pointer to the permutation and the power                    */
        ptL = libGAP_ADDR_PERM4(opL);
        ptP = libGAP_ADDR_PERM4(pow);

        /* invert the permutation                                          */
        for ( p = 0; p < deg; p++ )
            ptP[ *(ptL++) ] = p;

    }

    /* compute the power by repeated mapping for small negative exponents  */
    else if ( libGAP_TNUM_OBJ(opR) == libGAP_T_INT
           && -8 < libGAP_INT_INTOBJ(opR) && libGAP_INT_INTOBJ(opR) < 0 ) {

        /* get pointer to the permutation and the power                    */
        exp = -libGAP_INT_INTOBJ(opR);
        ptL = libGAP_ADDR_PERM4(opL);
        ptP = libGAP_ADDR_PERM4(pow);

        /* loop over the points                                            */
        for ( p = 0; p < deg; p++ ) {
            q = p;
            for ( e = 0; e < exp; e++ )
                q = ptL[q];
            ptP[q] = p;
        }

    }

    /* compute the power by raising the cycles individually for large exps */
    else if ( libGAP_TNUM_OBJ(opR) == libGAP_T_INT && libGAP_INT_INTOBJ(opR) <= -8 ) {

        /* make sure that the buffer bag is large enough                   */
        if ( libGAP_SIZE_OBJ(libGAP_TmpPerm) < libGAP_SIZE_OBJ(opL) ) {
            libGAP_ResizeBag( libGAP_TmpPerm, libGAP_SIZE_OBJ(opL) );
        }
        ptKnown = libGAP_ADDR_PERM4(libGAP_TmpPerm);

        /* clear the buffer bag                                            */
        for ( p = 0; p < libGAP_DEG_PERM4(opL); p++ )
            ptKnown[p] = 0;

        /* get pointer to the permutation and the power                    */
        exp = -libGAP_INT_INTOBJ(opR);
        ptL = libGAP_ADDR_PERM4(opL);
        ptP = libGAP_ADDR_PERM4(pow);

        /* loop over all cycles                                            */
        for ( p = 0; p < deg; p++ ) {

            /* if we haven't looked at this cycle so far                   */
            if ( ptKnown[p] == 0 ) {

                /* find the length of this cycle                           */
                len = 1;
                for ( q = ptL[p]; q != p; q = ptL[q] ) {
                    len++;  ptKnown[q] = 1;
                }

                /* raise this cycle to the power <exp> mod <len>           */
                r = p;
                for ( e = 0; e < exp % len; e++ )
                    r = ptL[r];
                ptP[r] = p;
                r = ptL[r];
                for ( q = ptL[p]; q != p; q = ptL[q] ) {
                    ptP[r] = q;
                    r = ptL[r];
                }

            }

        }

    }

    /* compute the power by raising the cycles individually for large exps */
    else if ( libGAP_TNUM_OBJ(opR) == libGAP_T_INTNEG ) {

        /* make sure that the buffer bag is large enough                   */
        if ( libGAP_SIZE_OBJ(libGAP_TmpPerm) < libGAP_SIZE_OBJ(opL) ) {
            libGAP_ResizeBag( libGAP_TmpPerm, libGAP_SIZE_OBJ(opL) );
        }
        ptKnown = libGAP_ADDR_PERM4(libGAP_TmpPerm);

        /* clear the buffer bag                                            */
        for ( p = 0; p < libGAP_DEG_PERM4(opL); p++ )
            ptKnown[p] = 0;

        /* get pointer to the permutation and the power                    */
        opR = libGAP_ProdInt( libGAP_INTOBJ_INT(-1), opR );
        ptL = libGAP_ADDR_PERM4(opL);
        ptP = libGAP_ADDR_PERM4(pow);

        /* loop over all cycles                                            */
        for ( p = 0; p < deg; p++ ) {

            /* if we haven't looked at this cycle so far                   */
            if ( ptKnown[p] == 0 ) {

                /* find the length of this cycle                           */
                len = 1;
                for ( q = ptL[p]; q != p; q = ptL[q] ) {
                    len++;  ptKnown[q] = 1;
                }

                /* raise this cycle to the power <exp> mod <len>           */
                r = p;
                exp = libGAP_INT_INTOBJ( libGAP_ModInt( opR, libGAP_INTOBJ_INT(len) ) );
                for ( e = 0; e < exp % len; e++ )
                    r = ptL[r];
                ptP[r] = p;
                r = ptL[r];
                for ( q = ptL[p]; q != p; q = ptL[q] ) {
                    ptP[r] = q;
                    r = ptL[r];
                }

            }

        }

    }

    /* return the result                                                   */
    return pow;
}


/****************************************************************************
**
*F  InvPerm( <perm> ) . . . . . . . . . . . . . . .  inverse of a permutation
*/
libGAP_Obj libGAP_InvPerm (
    libGAP_Obj             perm )
{
    return libGAP_POW( perm, libGAP_INTOBJ_INT(-1) );
}


/****************************************************************************
**
*F  PowIntPerm( <opL>, <opR> )  . . . image of an integer under a permutation
**
**  'PowIntPerm' returns the  image of the positive  integer  <opL> under the
**  permutation <opR>.  If <opL>  is larger than the  degree of <opR> it is a
**  fixpoint of the permutation and thus simply returned.
*/
libGAP_Obj             libGAP_PowIntPerm2 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Int                 img;            /* image (result)                  */

    /* large positive integers (> 2^28-1) are fixed by any permutation     */
    if ( libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS )
        return opL;

    /* permutations do not act on negative integers                        */
    img = libGAP_INT_INTOBJ( opL );
    if ( img <= 0 ) {
        opL = libGAP_ErrorReturnObj(
            "Perm. Operations: <point> must be a positive integer (not %d)",
            (libGAP_Int)img, 0L,
            "you can replace <point> via 'return <point>;'" );
        return libGAP_POW( opL, opR );
    }

    /* compute the image                                                   */
    if ( img <= libGAP_DEG_PERM2(opR) ) {
        img = (libGAP_ADDR_PERM2(opR))[img-1] + 1;
    }

    /* return it                                                           */
    return libGAP_INTOBJ_INT(img);
}

libGAP_Obj             libGAP_PowIntPerm4 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Int                 img;            /* image (result)                  */

    /* large positive integers (> 2^28-1) are fixed by any permutation     */
    if ( libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS )
        return opL;

    /* permutations do not act on negative integers                        */
    img = libGAP_INT_INTOBJ( opL );
    if ( img <= 0 ) {
        opL = libGAP_ErrorReturnObj(
            "Perm. Operations: <point> must be a positive integer (not %d)",
            (libGAP_Int)img, 0L,
            "you can replace <point> via 'return <point>;'" );
        return libGAP_POW( opL, opR );
    }

    /* compute the image                                                   */
    if ( img <= libGAP_DEG_PERM4(opR) ) {
        img = (libGAP_ADDR_PERM4(opR))[img-1] + 1;
    }

    /* return it                                                           */
    return libGAP_INTOBJ_INT(img);
}


/****************************************************************************
**
*F  QuoIntPerm( <opL>, <opR> )  .  preimage of an integer under a permutation
**
**  'QuoIntPerm' returns the preimage of the preimage integer <opL> under the
**  permutation <opR>.  If <opL> is larger than  the degree of  <opR> is is a
**  fixpoint, and thus simply returned.
**
**  There are basically two ways to find the preimage.  One is to run through
**  <opR>  and  look  for <opL>.  The index where it's found is the preimage.
**  The other is to  find  the image of  <opL> under <opR>, the image of that
**  point and so on, until we come  back to  <opL>.  The  last point  is  the
**  preimage of <opL>.  This is faster because the cycles are  usually short.
*/
libGAP_Obj             libGAP_QuoIntPerm2 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Int                 pre;            /* preimage (result)               */
    libGAP_Int                 img;            /* image (left operand)            */
    libGAP_UInt2 *             ptR;            /* pointer to the permutation      */

    /* large positive integers (> 2^28-1) are fixed by any permutation     */
    if ( libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS )
        return opL;

    /* permutations do not act on negative integers                        */
    img = libGAP_INT_INTOBJ(opL);
    if ( img <= 0 ) {
        opL = libGAP_ErrorReturnObj(
            "Perm. Operations: <point> must be a positive integer (not %d)",
            (libGAP_Int)img, 0L,
            "you can replace <point> via 'return <point>;'" );
        return libGAP_QUO( opL, opR );
    }

    /* compute the preimage                                                */
    pre = img;
    ptR = libGAP_ADDR_PERM2(opR);
    if ( img <= libGAP_DEG_PERM2(opR) ) {
        while ( ptR[ pre-1 ] != img-1 )
            pre = ptR[ pre-1 ] + 1;
    }

    /* return it                                                           */
    return libGAP_INTOBJ_INT(pre);
}

libGAP_Obj             libGAP_QuoIntPerm4 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Int                 pre;            /* preimage (result)               */
    libGAP_Int                 img;            /* image (left operand)            */
    libGAP_UInt4 *             ptR;            /* pointer to the permutation      */

    /* large positive integers (> 2^28-1) are fixed by any permutation     */
    if ( libGAP_TNUM_OBJ(opL) == libGAP_T_INTPOS )
        return opL;

    /* permutations do not act on negative integers                        */
    img = libGAP_INT_INTOBJ(opL);
    if ( img <= 0 ) {
        opL = libGAP_ErrorReturnObj(
            "Perm. Operations: <point> must be a positive integer (not %d)",
            (libGAP_Int)img, 0L,
            "you can replace <point> via 'return <point>;'" );
        return libGAP_QUO( opL, opR );
    }

    /* compute the preimage                                                */
    pre = img;
    ptR = libGAP_ADDR_PERM4(opR);
    if ( img <= libGAP_DEG_PERM4(opR) ) {
        while ( ptR[ pre-1 ] != img-1 )
            pre = ptR[ pre-1 ] + 1;
    }

    /* return it                                                           */
    return libGAP_INTOBJ_INT(pre);
}


/****************************************************************************
**
*F  PowPerm( <opL>, <opR> ) . . . . . . . . . . . conjugation of permutations
**
**  'PowPerm' returns the   conjugation  of the  two permutations  <opL>  and
**  <opR>, that s  defined as the  following product '<opR>\^-1 \*\ <opL> \*\
**  <opR>'.
*/
libGAP_Obj             libGAP_PowPerm22 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 cnj;            /* handle of the conjugation (res) */
    libGAP_UInt                degC;           /* degree of the conjugation       */
    libGAP_UInt2 *             ptC;            /* pointer to the conjugation      */
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt2 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt2 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt                p;              /* loop variable                   */

    /* compute the size of the result and allocate a bag                   */
    degL = libGAP_DEG_PERM2(opL);
    degR = libGAP_DEG_PERM2(opR);
    degC = degL < degR ? degR : degL;
    cnj = libGAP_NEW_PERM2( degC );

    /* set up the pointers                                                 */
    ptL = libGAP_ADDR_PERM2(opL);
    ptR = libGAP_ADDR_PERM2(opR);
    ptC = libGAP_ADDR_PERM2(cnj);

    /* its faster if the both permutations have the same size              */
    if ( degL == degR ) {
        for ( p = 0; p < degC; p++ )
            ptC[ ptR[p] ] = ptR[ ptL[p] ];
    }

    /* otherwise we have to use the macro 'IMAGE' three times              */
    else {
        for ( p = 0; p < degC; p++ )
            ptC[ libGAP_IMAGE(p,ptR,degR) ] = libGAP_IMAGE( libGAP_IMAGE(p,ptL,degL), ptR, degR );
    }

    /* return the result                                                   */
    return cnj;
}

libGAP_Obj             libGAP_PowPerm24 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 cnj;            /* handle of the conjugation (res) */
    libGAP_UInt                degC;           /* degree of the conjugation       */
    libGAP_UInt4 *             ptC;            /* pointer to the conjugation      */
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt2 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt4 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt                p;              /* loop variable                   */

    /* compute the size of the result and allocate a bag                   */
    degL = libGAP_DEG_PERM2(opL);
    degR = libGAP_DEG_PERM4(opR);
    degC = degL < degR ? degR : degL;
    cnj = libGAP_NEW_PERM4( degC );

    /* set up the pointers                                                 */
    ptL = libGAP_ADDR_PERM2(opL);
    ptR = libGAP_ADDR_PERM4(opR);
    ptC = libGAP_ADDR_PERM4(cnj);

    /* its faster if the both permutations have the same size              */
    if ( degL == degR ) {
        for ( p = 0; p < degC; p++ )
            ptC[ ptR[p] ] = ptR[ ptL[p] ];
    }

    /* otherwise we have to use the macro 'IMAGE' three times              */
    else {
        for ( p = 0; p < degC; p++ )
            ptC[ libGAP_IMAGE(p,ptR,degR) ] = libGAP_IMAGE( libGAP_IMAGE(p,ptL,degL), ptR, degR );
    }

    /* return the result                                                   */
    return cnj;
}

libGAP_Obj             libGAP_PowPerm42 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 cnj;            /* handle of the conjugation (res) */
    libGAP_UInt                degC;           /* degree of the conjugation       */
    libGAP_UInt4 *             ptC;            /* pointer to the conjugation      */
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt4 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt2 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt                p;              /* loop variable                   */

    /* compute the size of the result and allocate a bag                   */
    degL = libGAP_DEG_PERM4(opL);
    degR = libGAP_DEG_PERM2(opR);
    degC = degL < degR ? degR : degL;
    cnj = libGAP_NEW_PERM4( degC );

    /* set up the pointers                                                 */
    ptL = libGAP_ADDR_PERM4(opL);
    ptR = libGAP_ADDR_PERM2(opR);
    ptC = libGAP_ADDR_PERM4(cnj);

    /* its faster if the both permutations have the same size              */
    if ( degL == degR ) {
        for ( p = 0; p < degC; p++ )
            ptC[ ptR[p] ] = ptR[ ptL[p] ];
    }

    /* otherwise we have to use the macro 'IMAGE' three times              */
    else {
        for ( p = 0; p < degC; p++ )
            ptC[ libGAP_IMAGE(p,ptR,degR) ] = libGAP_IMAGE( libGAP_IMAGE(p,ptL,degL), ptR, degR );
    }

    /* return the result                                                   */
    return cnj;
}

libGAP_Obj             libGAP_PowPerm44 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 cnj;            /* handle of the conjugation (res) */
    libGAP_UInt                degC;           /* degree of the conjugation       */
    libGAP_UInt4 *             ptC;            /* pointer to the conjugation      */
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt4 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt4 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt                p;              /* loop variable                   */

    /* compute the size of the result and allocate a bag                   */
    degL = libGAP_DEG_PERM4(opL);
    degR = libGAP_DEG_PERM4(opR);
    degC = degL < degR ? degR : degL;
    cnj = libGAP_NEW_PERM4( degC );

    /* set up the pointers                                                 */
    ptL = libGAP_ADDR_PERM4(opL);
    ptR = libGAP_ADDR_PERM4(opR);
    ptC = libGAP_ADDR_PERM4(cnj);

    /* its faster if the both permutations have the same size              */
    if ( degL == degR ) {
        for ( p = 0; p < degC; p++ )
            ptC[ ptR[p] ] = ptR[ ptL[p] ];
    }

    /* otherwise we have to use the macro 'IMAGE' three times              */
    else {
        for ( p = 0; p < degC; p++ )
            ptC[ libGAP_IMAGE(p,ptR,degR) ] = libGAP_IMAGE( libGAP_IMAGE(p,ptL,degL), ptR, degR );
    }

    /* return the result                                                   */
    return cnj;
}


/****************************************************************************
**
*F  CommPerm( <opL>, <opR> )  . . . . . . . .  commutator of two permutations
**
**  'CommPerm' returns the  commutator  of  the  two permutations  <opL>  and
**  <opR>, that is defined as '<hd>\^-1 \*\ <opR>\^-1 \*\ <opL> \*\ <opR>'.
*/
libGAP_Obj             libGAP_CommPerm22 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 com;            /* handle of the commutator  (res) */
    libGAP_UInt                degC;           /* degree of the commutator        */
    libGAP_UInt2 *             ptC;            /* pointer to the commutator       */
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt2 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt2 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt                p;              /* loop variable                   */

    /* compute the size of the result and allocate a bag                   */
    degL = libGAP_DEG_PERM2(opL);
    degR = libGAP_DEG_PERM2(opR);
    degC = degL < degR ? degR : degL;
    com = libGAP_NEW_PERM2( degC );

    /* set up the pointers                                                 */
    ptL = libGAP_ADDR_PERM2(opL);
    ptR = libGAP_ADDR_PERM2(opR);
    ptC = libGAP_ADDR_PERM2(com);

    /* its faster if the both permutations have the same size              */
    if ( degL == degR ) {
        for ( p = 0; p < degC; p++ )
            ptC[ ptL[ ptR[ p ] ] ] = ptR[ ptL[ p ] ];
    }

    /* otherwise we have to use the macro 'IMAGE' four times               */
    else {
        for ( p = 0; p < degC; p++ )
            ptC[ libGAP_IMAGE( libGAP_IMAGE(p,ptR,degR), ptL, degL ) ]
               = libGAP_IMAGE( libGAP_IMAGE(p,ptL,degL), ptR, degR );
    }

    /* return the result                                                   */
    return com;
}

libGAP_Obj             libGAP_CommPerm24 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 com;            /* handle of the commutator  (res) */
    libGAP_UInt                degC;           /* degree of the commutator        */
    libGAP_UInt4 *             ptC;            /* pointer to the commutator       */
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt2 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt4 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt                p;              /* loop variable                   */

    /* compute the size of the result and allocate a bag                   */
    degL = libGAP_DEG_PERM2(opL);
    degR = libGAP_DEG_PERM4(opR);
    degC = degL < degR ? degR : degL;
    com = libGAP_NEW_PERM4( degC );

    /* set up the pointers                                                 */
    ptL = libGAP_ADDR_PERM2(opL);
    ptR = libGAP_ADDR_PERM4(opR);
    ptC = libGAP_ADDR_PERM4(com);

    /* its faster if the both permutations have the same size              */
    if ( degL == degR ) {
        for ( p = 0; p < degC; p++ )
            ptC[ ptL[ ptR[ p ] ] ] = ptR[ ptL[ p ] ];
    }

    /* otherwise we have to use the macro 'IMAGE' four times               */
    else {
        for ( p = 0; p < degC; p++ )
            ptC[ libGAP_IMAGE( libGAP_IMAGE(p,ptR,degR), ptL, degL ) ]
               = libGAP_IMAGE( libGAP_IMAGE(p,ptL,degL), ptR, degR );
    }

    /* return the result                                                   */
    return com;
}

libGAP_Obj             libGAP_CommPerm42 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 com;            /* handle of the commutator  (res) */
    libGAP_UInt                degC;           /* degree of the commutator        */
    libGAP_UInt4 *             ptC;            /* pointer to the commutator       */
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt4 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt2 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt                p;              /* loop variable                   */

    /* compute the size of the result and allocate a bag                   */
    degL = libGAP_DEG_PERM4(opL);
    degR = libGAP_DEG_PERM2(opR);
    degC = degL < degR ? degR : degL;
    com = libGAP_NEW_PERM4( degC );

    /* set up the pointers                                                 */
    ptL = libGAP_ADDR_PERM4(opL);
    ptR = libGAP_ADDR_PERM2(opR);
    ptC = libGAP_ADDR_PERM4(com);

    /* its faster if the both permutations have the same size              */
    if ( degL == degR ) {
        for ( p = 0; p < degC; p++ )
            ptC[ ptL[ ptR[ p ] ] ] = ptR[ ptL[ p ] ];
    }

    /* otherwise we have to use the macro 'IMAGE' four times               */
    else {
        for ( p = 0; p < degC; p++ )
            ptC[ libGAP_IMAGE( libGAP_IMAGE(p,ptR,degR), ptL, degL ) ]
               = libGAP_IMAGE( libGAP_IMAGE(p,ptL,degL), ptR, degR );
    }

    /* return the result                                                   */
    return com;
}

libGAP_Obj             libGAP_CommPerm44 (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 com;            /* handle of the commutator  (res) */
    libGAP_UInt                degC;           /* degree of the commutator        */
    libGAP_UInt4 *             ptC;            /* pointer to the commutator       */
    libGAP_UInt                degL;           /* degree of the left operand      */
    libGAP_UInt4 *             ptL;            /* pointer to the left operand     */
    libGAP_UInt                degR;           /* degree of the right operand     */
    libGAP_UInt4 *             ptR;            /* pointer to the right operand    */
    libGAP_UInt                p;              /* loop variable                   */

    /* compute the size of the result and allocate a bag                   */
    degL = libGAP_DEG_PERM4(opL);
    degR = libGAP_DEG_PERM4(opR);
    degC = degL < degR ? degR : degL;
    com = libGAP_NEW_PERM4( degC );

    /* set up the pointers                                                 */
    ptL = libGAP_ADDR_PERM4(opL);
    ptR = libGAP_ADDR_PERM4(opR);
    ptC = libGAP_ADDR_PERM4(com);

    /* its faster if the both permutations have the same size              */
    if ( degL == degR ) {
        for ( p = 0; p < degC; p++ )
            ptC[ ptL[ ptR[ p ] ] ] = ptR[ ptL[ p ] ];
    }

    /* otherwise we have to use the macro 'IMAGE' four times               */
    else {
        for ( p = 0; p < degC; p++ )
            ptC[ libGAP_IMAGE( libGAP_IMAGE(p,ptR,degR), ptL, degL ) ]
               = libGAP_IMAGE( libGAP_IMAGE(p,ptL,degL), ptR, degR );
    }

    /* return the result                                                   */
    return com;
}


/****************************************************************************
**
*F  OnePerm( <perm> )
*/
libGAP_Obj libGAP_OnePerm (
    libGAP_Obj                 op )
{
    return libGAP_IdentityPerm;
}



/****************************************************************************
**
*F  IsPermHandler( <self>, <val> )  . . . .  test if a value is a permutation
**
**  'FuncIsPerm' implements the internal function 'IsPerm'.
**
**  'IsPerm( <val> )'
**
**  'IsPerm' returns 'true' if the value <val> is a permutation and  'false'
**  otherwise.
*/
libGAP_Obj libGAP_IsPermFilt;

libGAP_Obj libGAP_IsPermHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 val )
{
    /* return 'true' if <val> is a permutation and 'false' otherwise       */
    switch ( libGAP_TNUM_OBJ(val) ) {

        case libGAP_T_PERM2:
        case libGAP_T_PERM4:
            return libGAP_True;

        case libGAP_T_COMOBJ:
        case libGAP_T_POSOBJ:
        case libGAP_T_DATOBJ:
            return libGAP_DoFilter( self, val );

        default:
            return libGAP_False;
    }
}


/****************************************************************************
**
*F  FuncPermList( <self>, <list> )  . . . . . convert a list to a permutation
**
**  'FuncPermList' implements the internal function 'PermList'
**
**  'PermList( <list> )'
**
**  Converts the list <list> into a  permutation,  which  is  then  returned.
**
**  'FunPermList' simply copies the list pointwise into  a  permutation  bag.
**  It also does some checks to make sure that the  list  is  a  permutation.
*/
libGAP_Obj             libGAP_FuncPermList (
    libGAP_Obj                 self,
    libGAP_Obj                 list )
{
    libGAP_Obj                 perm;           /* handle of the permutation       */
    libGAP_UInt2 *             ptPerm2;        /* pointer to the permutation      */
    libGAP_UInt4 *             ptPerm4;        /* pointer to the permutation      */
    libGAP_UInt                degPerm;        /* degree of the permutation       */
    libGAP_Obj *               ptList;         /* pointer to the list             */
    libGAP_UInt2 *             ptTmp2;         /* pointer to the buffer bag       */
    libGAP_UInt4 *             ptTmp4;         /* pointer to the buffer bag       */
    libGAP_Int                 i,  k;          /* loop variables                  */

    /* check the arguments                                                 */
    while ( ! libGAP_IS_SMALL_LIST( list ) ) {
        list = libGAP_ErrorReturnObj(
            "PermList: <list> must be a list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
            "you can replace <list> via 'return <list>;'" );
    }
    libGAP_PLAIN_LIST( list );

    /* handle small permutations                                           */
    if ( libGAP_LEN_LIST( list ) <= 65536 ) {

        degPerm = libGAP_LEN_LIST( list );

        /* make sure that the global buffer bag is large enough for checkin*/
        if ( libGAP_SIZE_OBJ(libGAP_TmpPerm) < degPerm * sizeof(libGAP_UInt2) ) {
            libGAP_ResizeBag( libGAP_TmpPerm, degPerm * sizeof(libGAP_UInt2) );
        }

        /* allocate the bag for the permutation and get pointer            */
        perm    = libGAP_NEW_PERM2( degPerm );
        ptPerm2 = libGAP_ADDR_PERM2(perm);
        ptList  = libGAP_ADDR_OBJ(list);
        ptTmp2  = libGAP_ADDR_PERM2(libGAP_TmpPerm);

        /* make the buffer bag clean                                       */
        for ( i = 1; i <= degPerm; i++ )
            ptTmp2[i-1] = 0;

        /* run through all entries of the list                             */
        for ( i = 1; i <= degPerm; i++ ) {

            /* get the <i>th entry of the list                             */
            if ( ptList[i] == 0 ) {
                /* list = ErrorReturnObj(
                    "PermList: <list>[%d] must have an assigned value",
                    (Int)i, 0L,
                    "you can replace <list> via 'return <list>;'" );
                return FuncPermList( 0, list ); */
		return libGAP_Fail;
            }
            if ( libGAP_TNUM_OBJ(ptList[i]) != libGAP_T_INT ) {
                /* list = ErrorReturnObj(
                    "PermList: <list>[%d] must be a integer",
                    (Int)i, 0L,
                    "you can replace <list> via 'return <list>;'" );
                return FuncPermList( 0, list ); */
		return libGAP_Fail;
            }
            k = libGAP_INT_INTOBJ(ptList[i]);
            if ( k <= 0 || degPerm < k ) {
                /* list = ErrorReturnObj(
                    "PermList: <list>[%d] must lie in [1..%d]",
                    (Int)i, (Int)degPerm,
                    "you can replace <list> via 'return <list>;'" );
                return FuncPermList( 0, list ); */
		return libGAP_Fail;
            }

            /* make sure we haven't seen this entry yet                     */
            if ( ptTmp2[k-1] != 0 ) {
		/* list = ErrorReturnObj(
                    "PermList: the point %d must occur only once",
                    (Int)k, 0L,
                    "you can replace <list> via 'return <list>;'" );
                return FuncPermList( 0, list ); */
		return libGAP_Fail;
            }
            ptTmp2[k-1] = 1;

            /* and finally copy it into the permutation                    */
            ptPerm2[i-1] = k-1;
        }

    }

    /* handle large permutations                                           */
    else {

        degPerm = libGAP_LEN_LIST( list );

        /* make sure that the global buffer bag is large enough for checkin*/
        if ( libGAP_SIZE_OBJ(libGAP_TmpPerm) < degPerm * sizeof(libGAP_UInt4) ) {
            libGAP_ResizeBag( libGAP_TmpPerm, degPerm * sizeof(libGAP_UInt4) );
        }

        /* allocate the bag for the permutation and get pointer            */
        perm    = libGAP_NEW_PERM4( degPerm );
        ptPerm4 = libGAP_ADDR_PERM4(perm);
        ptList  = libGAP_ADDR_OBJ( list);
        ptTmp4  = libGAP_ADDR_PERM4(libGAP_TmpPerm);

        /* make the buffer bag clean                                       */
        for ( i = 1; i <= degPerm; i++ )
            ptTmp4[i-1] = 0;

        /* run through all entries of the list                             */
        for ( i = 1; i <= degPerm; i++ ) {

            /* get the <i>th entry of the list                             */
            if ( ptList[i] == 0 ) {
                /* list = ErrorReturnObj(
                    "PermList: <list>[%d] must have an assigned value",
                    (Int)i, 0L,
                    "you can replace <list> via 'return <list>;'" );
                return FuncPermList( 0, list ); */
		return libGAP_Fail;
            }
            if ( libGAP_TNUM_OBJ(ptList[i]) != libGAP_T_INT ) {
                /* list = ErrorReturnObj(
                    "PermList: <list>[%d] must be a integer",
                    (Int)i, 0L,
                    "you can replace <list> via 'return <list>;'" );
                return FuncPermList( 0, list ); */
		return libGAP_Fail;
            }
            k = libGAP_INT_INTOBJ(ptList[i]);
            if ( k <= 0 || degPerm < k ) {
                /* list = ErrorReturnObj(
                    "PermList: <list>[%d] must lie in [1..%d]",
                    (Int)i, (Int)degPerm,
                    "you can replace <list> via 'return <list>;'" );
                return FuncPermList( 0, list ); */
		return libGAP_Fail;
            }

            /* make sure we haven't seen this entry yet                     */
            if ( ptTmp4[k-1] != 0 ) {
                /* list = ErrorReturnObj(
                    "PermList: the point %d must occur only once",
                    (Int)k, 0L,
                    "you can replace <list> via 'return <list>;'" );
                return FuncPermList( 0, list ); */
		return libGAP_Fail;
            }
            ptTmp4[k-1] = 1;

            /* and finally copy it into the permutation                    */
            ptPerm4[i-1] = k-1;
        }

    }

    /* return the permutation                                              */
    return perm;
}


/****************************************************************************
**
*F  FuncLARGEST_MOVED_POINT_PERM( <self>, <perm> ) largest point moved by a perm
**
**  'FuncLARGEST_MOVED_POINT_PERM' implements the internal function
**  'LargestMovedPointPerm'.
**
**  'LargestMovedPointPerm( <perm> )'
**
**  'LargestMovedPointPerm' returns  the  largest  positive  integer that  is
**  moved by the permutation <perm>.
**
**  This is easy, except that permutations may  contain  trailing  fixpoints.
*/
libGAP_Obj             libGAP_FuncLARGEST_MOVED_POINT_PERM (
    libGAP_Obj                 self,
    libGAP_Obj                 perm )
{
    libGAP_UInt                sup;            /* support (result)                */
    libGAP_UInt2 *             ptPerm2;        /* pointer to the permutation      */
    libGAP_UInt4 *             ptPerm4;        /* pointer to the permutation      */

    /* check the argument                                                  */
    while ( libGAP_TNUM_OBJ(perm) != libGAP_T_PERM2 && libGAP_TNUM_OBJ(perm) != libGAP_T_PERM4 ) {
        perm = libGAP_ErrorReturnObj(
            "LargestMovedPointPerm: <perm> must be a permutation (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(perm), 0L,
            "you can replace <perm> via 'return <perm>;'" );
    }

    /* handle small permutations                                           */
    if ( libGAP_TNUM_OBJ(perm) == libGAP_T_PERM2 ) {

        /* find the largest moved point                                    */
        ptPerm2 = libGAP_ADDR_PERM2(perm);
        for ( sup = libGAP_DEG_PERM2(perm); 1 <= sup; sup-- ) {
            if ( ptPerm2[sup-1] != sup-1 )
                break;
        }

    }

    /* handle large permutations                                           */
    else {

        /* find the largest moved point                                    */
        ptPerm4 = libGAP_ADDR_PERM4(perm);
        for ( sup = libGAP_DEG_PERM4(perm); 1 <= sup; sup-- ) {
            if ( ptPerm4[sup-1] != sup-1 )
                break;
        }

    }

    /* ahulpke: 19-nov-99: Removed special treatment of identity */

    /* return it                                                           */
    return libGAP_INTOBJ_INT( sup );
}


/****************************************************************************
**
*F  FuncCycleLengthPermInt( <self>, <perm>, <point> ) . . . . . . . . . . . .
*F  . . . . . . . . . . . . . . . . . . length of a cycle under a permutation
**
**  'FuncCycleLengthInt' implements the internal function
**  'CycleLengthPermInt'
**
**  'CycleLengthPermInt( <perm>, <point> )'
**
**  'CycleLengthPermInt' returns the length of the cycle  of  <point>,  which
**  must be a positive integer, under the permutation <perm>.
**
**  Note that the order of the arguments to this function has been  reversed.
*/
libGAP_Obj             libGAP_FuncCycleLengthPermInt (
    libGAP_Obj                 self,
    libGAP_Obj                 perm,
    libGAP_Obj                 point )
{
    libGAP_UInt2 *             ptPerm2;        /* pointer to the permutation      */
    libGAP_UInt4 *             ptPerm4;        /* pointer to the permutation      */
    libGAP_UInt                deg;            /* degree of the permutation       */
    libGAP_UInt                pnt;            /* value of the point              */
    libGAP_UInt                len;            /* length of cycle (result)        */
    libGAP_UInt                p;              /* loop variable                   */

    /* evaluate and check the arguments                                    */
    while ( libGAP_TNUM_OBJ(perm) != libGAP_T_PERM2 && libGAP_TNUM_OBJ(perm) != libGAP_T_PERM4 ) {
        perm = libGAP_ErrorReturnObj(
            "CycleLengthPermInt: <perm> must be a permutation (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(perm), 0L,
            "you can replace <perm> via 'return <perm>;'" );
    }
    while ( libGAP_TNUM_OBJ(point) != libGAP_T_INT || libGAP_INT_INTOBJ(point) <= 0 ) {
        point = libGAP_ErrorReturnObj(
         "CycleLengthPermInt: <point> must be a positive integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(point), 0L,
            "you can replace <point> via 'return <point>;'" );
    }

    /* handle small permutations                                           */
    if ( libGAP_TNUM_OBJ(perm) == libGAP_T_PERM2 ) {

        /* get pointer to the permutation, the degree, and the point       */
        ptPerm2 = libGAP_ADDR_PERM2(perm);
        deg = libGAP_DEG_PERM2(perm);
        pnt = libGAP_INT_INTOBJ(point)-1;

        /* now compute the length by looping over the cycle                */
        len = 1;
        if ( pnt < deg ) {
            for ( p = ptPerm2[pnt]; p != pnt; p = ptPerm2[p] )
                len++;
        }

    }

    /* handle large permutations                                           */
    else {

        /* get pointer to the permutation, the degree, and the point       */
        ptPerm4 = libGAP_ADDR_PERM4(perm);
        deg = libGAP_DEG_PERM4(perm);
        pnt = libGAP_INT_INTOBJ(point)-1;

        /* now compute the length by looping over the cycle                */
        len = 1;
        if ( pnt < deg ) {
            for ( p = ptPerm4[pnt]; p != pnt; p = ptPerm4[p] )
                len++;
        }

    }

    /* return the length                                                   */
    return libGAP_INTOBJ_INT(len);
}


/****************************************************************************
**
*F  FuncCyclePermInt( <self>, <perm>, <point> ) . . .  cycle of a permutation
*
**  'FuncCyclePermInt' implements the internal function 'CyclePermInt'.
**
**  'CyclePermInt( <perm>, <point> )'
**
**  'CyclePermInt' returns the cycle of <point>, which  must  be  a  positive
**  integer, under the permutation <perm> as a list.
*/
libGAP_Obj             libGAP_FuncCyclePermInt (
    libGAP_Obj                 self,
    libGAP_Obj                 perm,
    libGAP_Obj                 point )
{
    libGAP_Obj                 list;           /* handle of the list (result)     */
    libGAP_Obj *               ptList;         /* pointer to the list             */
    libGAP_UInt2 *             ptPerm2;        /* pointer to the permutation      */
    libGAP_UInt4 *             ptPerm4;        /* pointer to the permutation      */
    libGAP_UInt                deg;            /* degree of the permutation       */
    libGAP_UInt                pnt;            /* value of the point              */
    libGAP_UInt                len;            /* length of the cycle             */
    libGAP_UInt                p;              /* loop variable                   */

    /* evaluate and check the arguments                                    */
    while ( libGAP_TNUM_OBJ(perm) != libGAP_T_PERM2 && libGAP_TNUM_OBJ(perm) != libGAP_T_PERM4 ) {
        perm = libGAP_ErrorReturnObj(
            "CyclePermInt: <perm> must be a permutation (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(perm), 0L,
            "you can replace <perm> via 'return <perm>;'" );
    }
    while ( libGAP_TNUM_OBJ(point) != libGAP_T_INT || libGAP_INT_INTOBJ(point) <= 0 ) {
        point = libGAP_ErrorReturnObj(
            "CyclePermInt: <point> must be a positive integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(point), 0L,
            "you can replace <point> via 'return <point>;'" );
    }

    /* handle small permutations                                           */
    if ( libGAP_TNUM_OBJ(perm) == libGAP_T_PERM2 ) {

        /* get pointer to the permutation, the degree, and the point       */
        ptPerm2 = libGAP_ADDR_PERM2(perm);
        deg = libGAP_DEG_PERM2(perm);
        pnt = libGAP_INT_INTOBJ(point)-1;

        /* now compute the length by looping over the cycle                */
        len = 1;
        if ( pnt < deg ) {
            for ( p = ptPerm2[pnt]; p != pnt; p = ptPerm2[p] )
                len++;
        }

        /* allocate the list                                               */
        list = libGAP_NEW_PLIST( libGAP_T_PLIST, len );
        libGAP_SET_LEN_PLIST( list, len );
        ptList = libGAP_ADDR_OBJ(list);
        ptPerm2 = libGAP_ADDR_PERM2(perm);

        /* copy the points into the list                                   */
        len = 1;
        ptList[len++] = libGAP_INTOBJ_INT( pnt+1 );
        if ( pnt < deg ) {
            for ( p = ptPerm2[pnt]; p != pnt; p = ptPerm2[p] )
                ptList[len++] = libGAP_INTOBJ_INT( p+1 );
        }

    }

    /* handle large permutations                                           */
    else {

        /* get pointer to the permutation, the degree, and the point       */
        ptPerm4 = libGAP_ADDR_PERM4(perm);
        deg = libGAP_DEG_PERM4(perm);
        pnt = libGAP_INT_INTOBJ(point)-1;

        /* now compute the length by looping over the cycle                */
        len = 1;
        if ( pnt < deg ) {
            for ( p = ptPerm4[pnt]; p != pnt; p = ptPerm4[p] )
                len++;
        }

        /* allocate the list                                               */
        list = libGAP_NEW_PLIST( libGAP_T_PLIST, len );
        libGAP_SET_LEN_PLIST( list, len );
        ptList = libGAP_ADDR_OBJ(list);
        ptPerm4 = libGAP_ADDR_PERM4(perm);

        /* copy the points into the list                                   */
        len = 1;
        ptList[len++] = libGAP_INTOBJ_INT( pnt+1 );
        if ( pnt < deg ) {
            for ( p = ptPerm4[pnt]; p != pnt; p = ptPerm4[p] )
                ptList[len++] = libGAP_INTOBJ_INT( p+1 );
        }

    }

    /* return the list                                                     */
    return list;
}

/****************************************************************************
**
*F  FuncCycleStructurePerm( <self>, <perm> ) . . .  cycle structure of perm
*
**  'FuncCycleStructurePerm' implements the internal function
**  `CycleStructPerm'.
**
**  `CycleStructPerm( <perm> )'
**
**  'CycleStructPerm' returns a list of the form as described under
**  `CycleStructure'.
**  integer, under the permutation <perm> as a list.
*/
libGAP_Obj             libGAP_FuncCycleStructurePerm (
    libGAP_Obj                 self,
    libGAP_Obj                 perm )
{
    libGAP_Obj                 list;           /* handle of the list (result)     */
    libGAP_Obj *               ptList;         /* pointer to the list             */
    libGAP_UInt2 *             ptPerm2;        /* pointer to the permutation      */
    libGAP_UInt2 * 		scratch2;
    libGAP_UInt2 *		offset2;
    libGAP_UInt4 *             ptPerm4;        /* pointer to the permutation      */
    libGAP_UInt4 * 		scratch4;
    libGAP_UInt4 *		offset4;
    libGAP_UInt                deg;            /* degree of the permutation       */
    libGAP_UInt                pnt;            /* value of the point              */
    libGAP_UInt                len;            /* length of the cycle             */
    libGAP_UInt                p;              /* loop variable                   */
    libGAP_UInt 		max;		/* maximal cycle length            */
    libGAP_UInt		cnt;
    libGAP_UInt		ende;
    libGAP_UInt		bytes;
    libGAP_UInt1 *		clr;

    /* evaluate and check the arguments                                    */
    while ( libGAP_TNUM_OBJ(perm) != libGAP_T_PERM2 && libGAP_TNUM_OBJ(perm) != libGAP_T_PERM4 ) {
        perm = libGAP_ErrorReturnObj(
            "CycleStructPerm: <perm> must be a permutation (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(perm), 0L,
            "you can replace <perm> via 'return <perm>;'" );
    }

    /* make sure that the buffer bag is large enough                       */
    if ( libGAP_SIZE_OBJ(libGAP_TmpPerm) < libGAP_SIZE_OBJ(perm)+8 ) {
        libGAP_ResizeBag( libGAP_TmpPerm, libGAP_SIZE_OBJ(perm)+8 );
    }

    /* handle small permutations                                           */
    if ( libGAP_TNUM_OBJ(perm) == libGAP_T_PERM2 ) {

        /* get pointer to the permutation and the degree       */

        /* find the largest moved point                                    */
        ptPerm2 = libGAP_ADDR_PERM2(perm);
        for ( deg = libGAP_DEG_PERM2(perm); 1 <= deg; deg-- ) {
            if ( ptPerm2[deg-1] != deg-1 )
                break;
        }
	if (deg==0) {
	  /* special treatment of identity */
	  list = libGAP_NEW_PLIST( libGAP_T_PLIST, 0 );
	  libGAP_SET_LEN_PLIST( list, 0 );
	  return list;
	}

        scratch2=libGAP_ADDR_PERM2(libGAP_TmpPerm);

	/* the first deg bytes of TmpPerm hold a bit list of points done
	 * so far. The remaining bytes will form the lengths of nontrivial 
	 * cycles (as 2 byte numbers). As every nontrivial cycle requires at
	 * least 2 points, this is guaranteed to fit. */
        bytes=((deg/2)+1)*2; /* ensure 2-byte align */
        offset2=(libGAP_UInt2*)((libGAP_UInt)scratch2+(bytes));
	clr=(libGAP_UInt1*)scratch2;
	/* clear out the bits */
	for (cnt=0;cnt<bytes;cnt++) {
	  clr[cnt]=(libGAP_UInt1)0;
	}

	cnt=0;
	clr=(libGAP_UInt1*)scratch2;
	max=0;
	for (pnt=0;pnt<deg;pnt++) {
	  if ( clr[pnt] ==0 ) {
	    len=1;
	    clr[pnt]=1;
	    for ( p = ptPerm2[pnt]; p != pnt; p = ptPerm2[p] ) {
	      clr[p]=1;
	      len++;
	    }
	    /*Pr("pnt:%d, len=%d\n",pnt,len);*/
	    if (len>1) {
	      offset2[cnt]=(libGAP_UInt2)len;
	      cnt++;
	      if (len>max) { max=len;}
	    }
	  }
	}
       
	ende=cnt;

	/*Pr("max=%d cnt=%d\n",max,cnt);*/
	/* create the list */
	list=libGAP_NEW_PLIST(libGAP_T_PLIST,max-1);
	libGAP_SET_LEN_PLIST(list,max-1);
        ptList = libGAP_ADDR_OBJ(list);
	for (pnt=1;pnt<=max-1;pnt++) { ptList[pnt]=0; } /* clean out */

	for (cnt=0; cnt<ende;cnt++) {
	  pnt=(libGAP_UInt)offset2[cnt];
	  /*Pr("> cnt=%d pnt=%d\n",cnt,pnt);*/
	  pnt--;
	  ptList[pnt]=(libGAP_Obj)((libGAP_UInt)ptList[pnt]+1);

	} 

    } 

    /* handle large permutations                                           */
    else {

        /* get pointer to the permutation and the degree       */

        /* find the largest moved point                                    */
        ptPerm4 = libGAP_ADDR_PERM4(perm);
        for ( deg = libGAP_DEG_PERM4(perm); 1 <= deg; deg-- ) {
            if ( ptPerm4[deg-1] != deg-1 )
                break;
        }
	if (deg==0) {
	  /* special treatment of identity */
	  list = libGAP_NEW_PLIST( libGAP_T_PLIST, 0 );
	  libGAP_SET_LEN_PLIST( list, 0 );
	  return list;
	}

	/* the first deg bytes of TmpPerm hold a bit list of points done
	 * so far. The remaining bytes will form the lengths of nontrivial 
	 * cycles (as 4 byte numbers). As every nontrivial cycle requires at
	 * least 2 points, this is guaranteed to fit. */
        scratch4=libGAP_ADDR_PERM4(libGAP_TmpPerm);
        bytes=((deg/4)+1)*4; /* ensure 4-byte align */
        offset4=(libGAP_UInt4*)((libGAP_UInt)scratch4+(bytes));
	clr=(libGAP_UInt1*)scratch4;
	/* clear out the bits */
	for (cnt=0;cnt<bytes;cnt++) {
	  clr[cnt]=(libGAP_UInt1)0;
	}

	cnt=0;
	clr=(libGAP_UInt1*)scratch4;
	max=0;
	for (pnt=0;pnt<deg;pnt++) {
	  if ( clr[pnt] ==0 ) {
	    len=1;
	    clr[pnt]=1;
	    for ( p = ptPerm4[pnt]; p != pnt; p = ptPerm4[p] ) {
	      clr[p]=1;
	      len++;
	    }
	    /*Pr("pnt:%d, len=%d\n",pnt,len);*/
	    if (len>1) {
	      offset4[cnt]=(libGAP_UInt4)len;
	      cnt++;
	      if (len>max) { max=len;}
	    }
	  }
	}
       
	ende=cnt;

	/*Pr("max=%d cnt=%d\n",max,cnt);*/
	/* create the list */
	list=libGAP_NEW_PLIST(libGAP_T_PLIST,max-1);
	libGAP_SET_LEN_PLIST(list,max-1);
        ptList = libGAP_ADDR_OBJ(list);
	for (pnt=1;pnt<=max;pnt++) { ptList[pnt]=0; } /* clean out */

	for (cnt=0; cnt<ende;cnt++) {
	  pnt=(libGAP_UInt)offset4[cnt];
	  /*Pr("> cnt=%d pnt=%d\n",cnt,pnt);*/
	  pnt--;
	  ptList[pnt]=(libGAP_Obj)((libGAP_UInt)ptList[pnt]+1);

	} 

    }

    for (pnt=1; pnt<max;pnt++) {
      if (ptList[pnt]!=0) { ptList[pnt]=libGAP_INTOBJ_INT(ptList[pnt]);}
    } 

    /* return the list                                                     */
    return list;
}

/****************************************************************************
**
*F  FuncOrderPerm( <self>, <perm> ) . . . . . . . . .  order of a permutation
**
**  'FuncOrderPerm' implements the internal function 'OrderPerm'.
**
**  'OrderPerm( <perm> )'
**
**  'OrderPerm' returns the  order  of  the  permutation  <perm>,  i.e.,  the
**  smallest positive integer <n> such that '<perm>\^<n>' is the identity.
**
**  Since the largest element in S(65536) has oder greater than  10^382  this
**  computation may easily overflow.  So we have to use  arbitrary precision.
*/
libGAP_Obj             libGAP_FuncOrderPerm (
    libGAP_Obj                 self,
    libGAP_Obj                 perm )
{
    libGAP_UInt2 *             ptPerm2;        /* pointer to the permutation      */
    libGAP_UInt4 *             ptPerm4;        /* pointer to the permutation      */
    libGAP_Obj                 ord;            /* order (result), may be huge     */
    libGAP_UInt2 *             ptKnown2;       /* pointer to temporary bag        */
    libGAP_UInt4 *             ptKnown4;       /* pointer to temporary bag        */
    libGAP_UInt                len;            /* length of one cycle             */
    libGAP_UInt                gcd, s, t;      /* gcd( len, ord ), temporaries    */
    libGAP_UInt                p, q;           /* loop variables                  */

    /* check arguments and extract permutation                             */
    while ( libGAP_TNUM_OBJ(perm) != libGAP_T_PERM2 && libGAP_TNUM_OBJ(perm) != libGAP_T_PERM4 ) {
        perm = libGAP_ErrorReturnObj(
            "OrderPerm: <perm> must be a permutation (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(perm), 0L,
            "you can replace <perm> via 'return <perm>;'" );
    }

    /* make sure that the buffer bag is large enough                       */
    if ( libGAP_SIZE_OBJ(libGAP_TmpPerm) < libGAP_SIZE_OBJ(perm) ) {
        libGAP_ResizeBag( libGAP_TmpPerm, libGAP_SIZE_OBJ(perm) );
    }

    /* handle small permutations                                           */
    if ( libGAP_TNUM_OBJ(perm) == libGAP_T_PERM2 ) {

        /* get the pointer to the bags                                     */
        ptPerm2  = libGAP_ADDR_PERM2(perm);
        ptKnown2 = libGAP_ADDR_PERM2(libGAP_TmpPerm);

        /* clear the buffer bag                                            */
        for ( p = 0; p < libGAP_DEG_PERM2(perm); p++ )
            ptKnown2[p] = 0;

        /* start with order 1                                              */
        ord = libGAP_INTOBJ_INT(1);

        /* loop over all cycles                                            */
        for ( p = 0; p < libGAP_DEG_PERM2(perm); p++ ) {

            /* if we haven't looked at this cycle so far                   */
            if ( ptKnown2[p] == 0 && ptPerm2[p] != p ) {

                /* find the length of this cycle                           */
                len = 1;
                for ( q = ptPerm2[p]; q != p; q = ptPerm2[q] ) {
                    len++;  ptKnown2[q] = 1;
                }

                /* compute the gcd with the previously order ord           */
                /* Note that since len is single precision, ord % len is to*/
                gcd = len;  s = libGAP_INT_INTOBJ( libGAP_ModInt( ord, libGAP_INTOBJ_INT(len) ) );
                while ( s != 0 ) {
                    t = s;  s = gcd % s;  gcd = t;
                }
                ord = libGAP_ProdInt( ord, libGAP_INTOBJ_INT( len / gcd ) );

            }

        }

    }

    /* handle larger permutations                                          */
    else {

        /* get the pointer to the bags                                     */
        ptPerm4  = libGAP_ADDR_PERM4(perm);
        ptKnown4 = libGAP_ADDR_PERM4(libGAP_TmpPerm);

        /* clear the buffer bag                                            */
        for ( p = 0; p < libGAP_DEG_PERM4(perm); p++ )
            ptKnown4[p] = 0;

        /* start with order 1                                              */
        ord = libGAP_INTOBJ_INT(1);

        /* loop over all cycles                                            */
        for ( p = 0; p < libGAP_DEG_PERM4(perm); p++ ) {

            /* if we haven't looked at this cycle so far                   */
            if ( ptKnown4[p] == 0 && ptPerm4[p] != p ) {

                /* find the length of this cycle                           */
                len = 1;
                for ( q = ptPerm4[p]; q != p; q = ptPerm4[q] ) {
                    len++;  ptKnown4[q] = 1;
                }

                /* compute the gcd with the previously order ord           */
                /* Note that since len is single precision, ord % len is to*/
                gcd = len;  s = libGAP_INT_INTOBJ( libGAP_ModInt( ord, libGAP_INTOBJ_INT(len) ) );
                while ( s != 0 ) {
                    t = s;  s = gcd % s;  gcd = t;
                }
                ord = libGAP_ProdInt( ord, libGAP_INTOBJ_INT( len / gcd ) );

            }

        }

    }

    /* return the order                                                    */
    return ord;
}


/****************************************************************************
**
*F  FuncSignPerm( <self>, <perm> )  . . . . . . . . . . sign of a permutation
**
**  'FuncSignPerm' implements the internal function 'SignPerm'.
**
**  'SignPerm( <perm> )'
**
**  'SignPerm' returns the sign of the permutation <perm>.  The sign is +1 if
**  <perm> is the product of an *even* number of transpositions,  and  -1  if
**  <perm> is the product of an *odd*  number  of  transpositions.  The  sign
**  is a homomorphism from the symmetric group onto the multiplicative  group
**  $\{ +1, -1 \}$, the kernel of which is the alternating group.
*/
libGAP_Obj             libGAP_FuncSignPerm (
    libGAP_Obj                 self,
    libGAP_Obj                 perm )
{
    libGAP_UInt2 *             ptPerm2;        /* pointer to the permutation      */
    libGAP_UInt4 *             ptPerm4;        /* pointer to the permutation      */
    libGAP_Int                 sign;           /* sign (result)                   */
    libGAP_UInt2 *             ptKnown2;       /* pointer to temporary bag        */
    libGAP_UInt4 *             ptKnown4;       /* pointer to temporary bag        */
    libGAP_UInt                len;            /* length of one cycle             */
    libGAP_UInt                p,  q;          /* loop variables                  */

    /* check arguments and extract permutation                             */
    while ( libGAP_TNUM_OBJ(perm) != libGAP_T_PERM2 && libGAP_TNUM_OBJ(perm) != libGAP_T_PERM4 ) {
        perm = libGAP_ErrorReturnObj(
            "SignPerm: <perm> must be a permutation (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(perm), 0L,
            "you can replace <perm> via 'return <perm>;'" );
    }

    /* make sure that the buffer bag is large enough                       */
    if ( libGAP_SIZE_OBJ(libGAP_TmpPerm) < libGAP_SIZE_OBJ(perm) ) {
        libGAP_ResizeBag( libGAP_TmpPerm, libGAP_SIZE_OBJ(perm) );
    }

    /* handle small permutations                                           */
    if ( libGAP_TNUM_OBJ(perm) == libGAP_T_PERM2 ) {

        /* get the pointer to the bags                                     */
        ptPerm2  = libGAP_ADDR_PERM2(perm);
        ptKnown2 = libGAP_ADDR_PERM2(libGAP_TmpPerm);

        /* clear the buffer bag                                            */
        for ( p = 0; p < libGAP_DEG_PERM2(perm); p++ )
            ptKnown2[p] = 0;

        /* start with sign  1                                              */
        sign = 1;

        /* loop over all cycles                                            */
        for ( p = 0; p < libGAP_DEG_PERM2(perm); p++ ) {

            /* if we haven't looked at this cycle so far                   */
            if ( ptKnown2[p] == 0 && ptPerm2[p] != p ) {

                /* find the length of this cycle                           */
                len = 1;
                for ( q = ptPerm2[p]; q != p; q = ptPerm2[q] ) {
                    len++;  ptKnown2[q] = 1;
                }

                /* if the length is even invert the sign                   */
                if ( len % 2 == 0 )
                    sign = -sign;

            }

        }

    }

    /* handle large permutations                                           */
    else {

        /* get the pointer to the bags                                     */
        ptPerm4  = libGAP_ADDR_PERM4(perm);
        ptKnown4 = libGAP_ADDR_PERM4(libGAP_TmpPerm);

        /* clear the buffer bag                                            */
        for ( p = 0; p < libGAP_DEG_PERM4(perm); p++ )
            ptKnown4[p] = 0;

        /* start with sign  1                                              */
        sign = 1;

        /* loop over all cycles                                            */
        for ( p = 0; p < libGAP_DEG_PERM4(perm); p++ ) {

            /* if we haven't looked at this cycle so far                   */
            if ( ptKnown4[p] == 0 && ptPerm4[p] != p ) {

                /* find the length of this cycle                           */
                len = 1;
                for ( q = ptPerm4[p]; q != p; q = ptPerm4[q] ) {
                    len++;  ptKnown4[q] = 1;
                }

                /* if the length is even invert the sign                   */
                if ( len % 2 == 0 )
                    sign = -sign;

            }

        }

    }

    /* return the sign                                                     */
    return libGAP_INTOBJ_INT( sign );
}


/****************************************************************************
**
*F  FuncSmallestGeneratorPerm( <self>, <perm> ) . . . . . . . . . . . . . . .
*F  . . . . . . . smallest generator of cyclic group generated by permutation
**
**  'FuncSmallestGeneratorPerm' implements the internal function
**  'SmallestGeneratorPerm'.
**
**  'SmallestGeneratorPerm( <perm> )'
**
**  'SmallestGeneratorPerm' returns the   smallest generator  of  the  cyclic
**  group generated by the  permutation  <perm>.  That  is   the result is  a
**  permutation that generates the same  cyclic group as  <perm> and is  with
**  respect  to the lexicographical order  defined  by '\<' the smallest such
**  permutation.
*/
libGAP_Obj             libGAP_FuncSmallestGeneratorPerm (
    libGAP_Obj                 self,
    libGAP_Obj                 perm )
{
    libGAP_Obj                 small;          /* handle of the smallest gen      */
    libGAP_UInt2 *             ptSmall2;       /* pointer to the smallest gen     */
    libGAP_UInt4 *             ptSmall4;       /* pointer to the smallest gen     */
    libGAP_UInt2 *             ptPerm2;        /* pointer to the permutation      */
    libGAP_UInt4 *             ptPerm4;        /* pointer to the permutation      */
    libGAP_UInt2 *             ptKnown2;       /* pointer to temporary bag        */
    libGAP_UInt4 *             ptKnown4;       /* pointer to temporary bag        */
    libGAP_Obj                 ord;            /* order, may be huge              */
    libGAP_Obj                 pow;            /* power, may also be huge         */
    libGAP_UInt                len;            /* length of one cycle             */
    libGAP_UInt                gcd,  s,  t;    /* gcd( len, ord ), temporaries    */
    libGAP_UInt                min;            /* minimal element in a cycle      */
    libGAP_UInt                p,  q;          /* loop variables                  */
    libGAP_UInt                l, n, x, gcd2;  /* loop variable                   */

    /* check arguments and extract permutation                             */
    while ( libGAP_TNUM_OBJ(perm) != libGAP_T_PERM2 && libGAP_TNUM_OBJ(perm) != libGAP_T_PERM4 ) {
        perm = libGAP_ErrorReturnObj(
            "SmallestGeneratorPerm: <perm> must be a permutation (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(perm), 0L,
            "you can replace <perm> via 'return <perm>;'" );
    }

    /* make sure that the buffer bag is large enough                       */
    if ( libGAP_SIZE_OBJ(libGAP_TmpPerm) < libGAP_SIZE_OBJ(perm) ) {
        libGAP_ResizeBag( libGAP_TmpPerm, libGAP_SIZE_OBJ(perm) );
    }

    /* handle small permutations                                           */
    if ( libGAP_TNUM_OBJ(perm) == libGAP_T_PERM2 ) {

        /* allocate the result bag                                         */
        small = libGAP_NEW_PERM2( libGAP_DEG_PERM2(perm) );

        /* get the pointer to the bags                                     */
        ptPerm2   = libGAP_ADDR_PERM2(perm);
        ptKnown2  = libGAP_ADDR_PERM2(libGAP_TmpPerm);
        ptSmall2  = libGAP_ADDR_PERM2(small);

        /* clear the buffer bag                                            */
        for ( p = 0; p < libGAP_DEG_PERM2(perm); p++ )
            ptKnown2[p] = 0;

        /* we only know that we must raise <perm> to a power = 0 mod 1     */
        ord = libGAP_INTOBJ_INT(1);  pow = libGAP_INTOBJ_INT(0);

        /* loop over all cycles                                            */
        for ( p = 0; p < libGAP_DEG_PERM2(perm); p++ ) {

            /* if we haven't looked at this cycle so far                   */
            if ( ptKnown2[p] == 0 ) {

                /* find the length of this cycle                           */
                len = 1;
                for ( q = ptPerm2[p]; q != p; q = ptPerm2[q] ) {
                    len++;  ptKnown2[q] = 1;
                }

                /* compute the gcd with the previously order ord           */
                /* Note that since len is single precision, ord % len is to*/
                gcd = len;  s = libGAP_INT_INTOBJ( libGAP_ModInt( ord, libGAP_INTOBJ_INT(len) ) );
                while ( s != 0 ) {
                    t = s;  s = gcd % s;  gcd = t;
                }

                /* we must raise the cycle into a power = pow mod gcd      */
                x = libGAP_INT_INTOBJ( libGAP_ModInt( pow, libGAP_INTOBJ_INT( gcd ) ) );

                /* find the smallest element in the cycle at such a positio*/
                min = libGAP_DEG_PERM2(perm)-1;
                n = 0;
                for ( q = p, l = 0; l < len; l++ ) {
                    gcd2 = len;  s = l;
                    while ( s != 0 ) { t = s; s = gcd2 % s; gcd2 = t; }
                    if ( l % gcd == x && gcd2 == 1 && q <= min ) {
                        min = q;
                        n = l;
                    }
                    q = ptPerm2[q];
                }

                /* raise the cycle to that power and put it in the result  */
                ptSmall2[p] = min;
                for ( q = ptPerm2[p]; q != p; q = ptPerm2[q] ) {
                    min = ptPerm2[min];  ptSmall2[q] = min;
                }

                /* compute the new order and the new power                 */
                while ( libGAP_INT_INTOBJ( libGAP_ModInt( pow, libGAP_INTOBJ_INT(len) ) ) != n )
                    pow = libGAP_SumInt( pow, ord );
                ord = libGAP_ProdInt( ord, libGAP_INTOBJ_INT( len / gcd ) );

            }

        }

    }

    /* handle large permutations                                           */
    else {

        /* allocate the result bag                                         */
        small = libGAP_NEW_PERM4( libGAP_DEG_PERM4(perm) );

        /* get the pointer to the bags                                     */
        ptPerm4   = libGAP_ADDR_PERM4(perm);
        ptKnown4  = libGAP_ADDR_PERM4(libGAP_TmpPerm);
        ptSmall4  = libGAP_ADDR_PERM4(small);

        /* clear the buffer bag                                            */
        for ( p = 0; p < libGAP_DEG_PERM4(perm); p++ )
            ptKnown4[p] = 0;

        /* we only know that we must raise <perm> to a power = 0 mod 1     */
        ord = libGAP_INTOBJ_INT(1);  pow = libGAP_INTOBJ_INT(0);

        /* loop over all cycles                                            */
        for ( p = 0; p < libGAP_DEG_PERM4(perm); p++ ) {

            /* if we haven't looked at this cycle so far                   */
            if ( ptKnown4[p] == 0 ) {

                /* find the length of this cycle                           */
                len = 1;
                for ( q = ptPerm4[p]; q != p; q = ptPerm4[q] ) {
                    len++;  ptKnown4[q] = 1;
                }

                /* compute the gcd with the previously order ord           */
                /* Note that since len is single precision, ord % len is to*/
                gcd = len;  s = libGAP_INT_INTOBJ( libGAP_ModInt( ord, libGAP_INTOBJ_INT(len) ) );
                while ( s != 0 ) {
                    t = s;  s = gcd % s;  gcd = t;
                }

                /* we must raise the cycle into a power = pow mod gcd      */
                x = libGAP_INT_INTOBJ( libGAP_ModInt( pow, libGAP_INTOBJ_INT( gcd ) ) );

                /* find the smallest element in the cycle at such a positio*/
                min = libGAP_DEG_PERM4(perm)-1;
                n = 0;
                for ( q = p, l = 0; l < len; l++ ) {
                    gcd2 = len;  s = l;
                    while ( s != 0 ) { t = s; s = gcd2 % s; gcd2 = t; }
                    if ( l % gcd == x && gcd2 == 1 && q <= min ) {
                        min = q;
                        n = l;
                    }
                    q = ptPerm4[q];
                }

                /* raise the cycle to that power and put it in the result  */
                ptSmall4[p] = min;
                for ( q = ptPerm4[p]; q != p; q = ptPerm4[q] ) {
                    min = ptPerm4[min];  ptSmall4[q] = min;
                }

                /* compute the new order and the new power                 */
                while ( libGAP_INT_INTOBJ( libGAP_ModInt( pow, libGAP_INTOBJ_INT(len) ) ) != n )
                    pow = libGAP_SumInt( pow, ord );
                ord = libGAP_ProdInt( ord, libGAP_INTOBJ_INT( len / gcd ) );

            }

        }

    }

    /* return the smallest generator                                       */
    return small;
}

/****************************************************************************
**
*F  FuncRESTRICTED_PERM( <self>, <perm>, <dom>, <test> ) . . RestrictedPerm
**
**  'FuncRESTRICTED_PERM' implements the internal function
**  'RESTRICTED_PERM'.
**
**  'RESTRICTED_PERM( <perm>, <dom>, <test> )'
**
**  'RESTRICTED_PERM' returns the restriction of <perm> to <dom>. If <test>
**  is set to `true' is is verified that <dom> is the union of cycles of
**  <perm>.
*/
libGAP_Obj             libGAP_FuncRESTRICTED_PERM (
    libGAP_Obj                 self,
    libGAP_Obj                 perm,
    libGAP_Obj                 dom,
    libGAP_Obj 		test )
{
    libGAP_Obj rest;
    libGAP_UInt2 *            ptRest2;
    libGAP_UInt2 *            ptPerm2;
    libGAP_UInt4 *            ptRest4;
    libGAP_UInt4 *            ptPerm4;
    libGAP_Obj *              ptDom;
    libGAP_Int i,inc,len,p,deg;

    /* check arguments and extract permutation                             */
    while ( libGAP_TNUM_OBJ(perm) != libGAP_T_PERM2 && libGAP_TNUM_OBJ(perm) != libGAP_T_PERM4 ) {
        perm = libGAP_ErrorReturnObj(
            "RestrictedPerm: <perm> must be a permutation (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(perm), 0L,
            "you can replace <perm> via 'return <perm>;'" );
    }

    /* make sure that the buffer bag is large enough */
    if ( libGAP_SIZE_OBJ(libGAP_TmpPerm) < libGAP_SIZE_OBJ(perm) ) {
        libGAP_ResizeBag( libGAP_TmpPerm, libGAP_SIZE_OBJ(perm) );
    }

    /* handle small permutations                                           */
    if ( libGAP_TNUM_OBJ(perm) == libGAP_T_PERM2 ) {

      /* allocate the result bag                                         */
      deg = libGAP_DEG_PERM2(perm);
      rest = libGAP_NEW_PERM2(deg);

      /* get the pointer to the bags                                     */
      ptPerm2  = libGAP_ADDR_PERM2(perm);
      ptRest2  = libGAP_ADDR_PERM2(rest);

      /* create identity everywhere */
      for ( p = 0; p < deg; p++ ) {
	  ptRest2[p]=(libGAP_UInt2)p;
      }

      if ( ! libGAP_IS_RANGE(dom) ) {
	if ( ! libGAP_IS_PLIST( dom ) ) {
	  return libGAP_Fail;
	}
	/* domain is list */
	ptPerm2  = libGAP_ADDR_PERM2(perm);
	ptRest2  = libGAP_ADDR_PERM2(rest);
	ptDom  = libGAP_ADDR_OBJ(dom);
	len = libGAP_LEN_LIST(dom);
	for (i=1;i<=len;i++) {
	  p=libGAP_INT_INTOBJ(ptDom[i]);
	  if ( libGAP_IS_INTOBJ(ptDom[i]) && p>0 ) {
	    if (p<=deg) {
	      p-=1;
	      ptRest2[p]=ptPerm2[p];
	    }
	  }
	  else{
	    return libGAP_Fail;
	  }
	}
      }
      else {
	len = libGAP_GET_LEN_RANGE(dom);
	p = libGAP_GET_LOW_RANGE(dom);
	inc = libGAP_GET_INC_RANGE(dom);
	while (p<1) {
	  p+=inc;
	  len=-1;
	}
	i=p+(inc*len)-1;
	while (i>deg) {
	  i-=inc;
	}
	p-=1;
	i-=1;
	while (p<=i) {
	  ptRest2[p]=ptPerm2[p];
	  p+=inc;
	}
      }

      if (test==libGAP_True) {

	ptPerm2  = libGAP_ADDR_PERM2(libGAP_TmpPerm);

	/* cleanout */
	for (p=0; p<deg; p++ ) {
	  ptPerm2[p]=0;
	}

        /* check whether the result is a permutation */
	for (p=0;p<deg;p++) {
	  inc=ptRest2[p];
	  if (ptPerm2[inc]==1) return libGAP_Fail; /* point was known */
	  else ptPerm2[inc]=1; /* now point is known */
	}

      }

    }
    else {
      /* allocate the result bag                                         */
      deg = libGAP_DEG_PERM4(perm);
      rest = libGAP_NEW_PERM4(deg);

      /* get the pointer to the bags                                     */
      ptPerm4  = libGAP_ADDR_PERM4(perm);
      ptRest4  = libGAP_ADDR_PERM4(rest);

      /* create identity everywhere */
      for ( p = 0; p < deg; p++ ) {
	  ptRest4[p]=(libGAP_UInt4)p;
      }

      if ( ! libGAP_IS_RANGE(dom) ) {
	if ( ! libGAP_IS_PLIST( dom ) ) {
	  return libGAP_Fail;
	}
	/* domain is list */
	ptPerm4  = libGAP_ADDR_PERM4(perm);
	ptRest4  = libGAP_ADDR_PERM4(rest);
	ptDom  = libGAP_ADDR_OBJ(dom);
	len = libGAP_LEN_LIST(dom);
	for (i=1;i<=len;i++) {
	  p=libGAP_INT_INTOBJ(ptDom[i]);
	  if ( libGAP_IS_INTOBJ(ptDom[i]) && p>0 ) {
	    if (p<=deg) {
	      p-=1;
	      ptRest4[p]=ptPerm4[p];
	    }
	  }
	  else{
	    return libGAP_Fail;
	  }
	}
      }
      else {
	len = libGAP_GET_LEN_RANGE(dom);
	p = libGAP_GET_LOW_RANGE(dom);
	inc = libGAP_GET_INC_RANGE(dom);
	while (p<1) {
	  p+=inc;
	  len=-1;
	}
	i=p+(inc*len)-1;
	while (i>deg) {
	  i-=inc;
	}
	p-=1;
	i-=1;
	while (p<=i) {
	  ptRest4[p]=ptPerm4[p];
	  p+=inc;
	}
      }

      if (test==libGAP_True) {

	ptPerm4  = libGAP_ADDR_PERM4(libGAP_TmpPerm);

	/* cleanout */
	for ( p = 0; p < deg; p++ ) {
	  ptPerm4[p]=0;
	}

        /* check whether the result is a permutation */
	for (p=0;p<deg;p++) {
	  inc=ptRest4[p];
	  if (ptPerm4[inc]==1) return libGAP_Fail; /* point was known */
	  else ptPerm4[inc]=1; /* now point is known */
	}

      }
    }

    /* return the restriction */
    return rest;
}

/****************************************************************************
**
*F  FuncSHIFTED_PERM( <self>, <perm>, <shift>)
**
**  'FuncSHIFTED_PERM' implements the internal function 'SHIFTED_PERM'.
**
**  'SHIFTED_PERM( <perm>, <shift> )'
**
**  It returns a new permutation that is obtained by shifting the domain by
**  <shift>. Points that become negative are ignored. The resulting
**  permutation has a storage degree changed by `shift'.
*/
libGAP_Obj             libGAP_FuncSHIFTED_PERM (
    libGAP_Obj                 self,
    libGAP_Obj                 perm,
    libGAP_Obj                 shiftval )
{
  libGAP_Obj new;
  libGAP_UInt2 *            ptTo2;
  libGAP_UInt2 *            ptPerm2;
  libGAP_UInt4 *            ptTo4;
  libGAP_UInt4 *            ptPerm4;
  libGAP_Int shift,odeg,deg,from,to,i,j,a;

  shift=libGAP_INT_INTOBJ(shiftval);
  if ( libGAP_TNUM_OBJ(perm) == libGAP_T_PERM2 ) 
    odeg=libGAP_DEG_PERM2(perm);
  else
    odeg=libGAP_DEG_PERM4(perm);
  deg=odeg+shift; 
  if (deg<0) deg=0; /* everything shift away */

  if (deg>65536) {
    to=4;
    new=libGAP_NEW_PERM4(deg);
    ptTo2=libGAP_ADDR_PERM2(new); /* please compiler */
    ptTo4=libGAP_ADDR_PERM4(new);
    if (shift>0) { /* start with `shift' identity points */
      for (i=0;i<shift;i++) 
        ptTo4[i]=i;
    }
  }
  else {
    to=2;
    new=libGAP_NEW_PERM2(deg);
    ptTo2=libGAP_ADDR_PERM2(new);
    ptTo4=libGAP_ADDR_PERM4(new); /* please compiler */
    if (shift>0) { /* start with `shift' identity points */
      for (i=0;i<shift;i++) ptTo2[i]=i;
    }
  }

  if ( libGAP_TNUM_OBJ(perm) == libGAP_T_PERM2 ) from=2;
  else from=4;

  ptPerm2=libGAP_ADDR_PERM2(perm);
  ptPerm4=libGAP_ADDR_PERM4(perm);

  if (shift<0) {
    i=-shift; /* from index */
    j=0; /* to index */
  }
  else {
    i=0; /* from index */
    j=shift; /* to index */
  }

  if (from==2) {
    if (to==2) {
      while (j<deg) {
	a=(libGAP_Int)ptPerm2[i]+shift;
	if (a<0) return libGAP_Fail;
	ptTo2[j]=(libGAP_UInt2)a;
	i++;
	j++;
      }
    }
    else {
      while (j<deg) {
	a=(libGAP_Int)ptPerm2[i]+shift;
	if (a<0) return libGAP_Fail;
	ptTo4[j]=(libGAP_UInt4)a;
	i++;
	j++;
      }
    }
  }
  else {
    if (to==2) {
      while (j<deg) {
	a=(libGAP_Int)ptPerm4[i]+shift;
	if (a<0) return libGAP_Fail;
	ptTo2[j]=(libGAP_UInt2)a;
	i++;
	j++;
      }
    }
    else {
      while (j<deg) {
	a=(libGAP_Int)ptPerm4[i]+shift;
	if (a<0) return libGAP_Fail;
	ptTo4[j]=(libGAP_UInt4)a;
	i++;
	j++;
      }
    }
  }

  return new;
}



/****************************************************************************
**
*F  FuncTRIM_PERM( <self>, <perm>, <n> ) . . . . . . . . . trim a permutation
**
**  'TRIM_PERM' trims a permutation to the first <n> points. This can be
##  useful to save memory
*/
libGAP_Obj             libGAP_FuncTRIM_PERM (
    libGAP_Obj			self,
    libGAP_Obj                 perm,
    libGAP_Obj                 n )
{
    libGAP_UInt	deg,rdeg,i;
    libGAP_UInt4*	addr;

    /* check arguments and extract permutation */
    while ( libGAP_TNUM_OBJ(perm) != libGAP_T_PERM2 && libGAP_TNUM_OBJ(perm) != libGAP_T_PERM4 ) {
        perm = libGAP_ErrorReturnObj(
            "TRIM_PERM: <perm> must be a permutation (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(perm), 0L,
            "you can replace <perm> via 'return <perm>;'" );
    }

    deg = libGAP_INT_INTOBJ(n);
    /*T a test might be useful here */

    if ( libGAP_TNUM_OBJ(perm) == libGAP_T_PERM2 ) {
      rdeg = deg < libGAP_DEG_PERM2(perm) ? deg : libGAP_DEG_PERM2(perm);
      libGAP_ResizeBag( perm, sizeof(libGAP_UInt2)*rdeg);
    }
    else {
      rdeg = deg < libGAP_DEG_PERM4(perm) ? deg : libGAP_DEG_PERM4(perm);
      if (rdeg > 65536UL ) {
	libGAP_ResizeBag( perm, sizeof(libGAP_UInt4)*rdeg);
      }
      else {
	/* Convert to 2Byte rep: move the points up */
        addr=libGAP_ADDR_PERM4(perm);
	for (i=0;i<=rdeg;i++) {
	  ((libGAP_UInt2*)addr)[i]=(libGAP_UInt2)addr[i];
	}
	libGAP_RetypeBag( perm, libGAP_T_PERM2 );
	libGAP_ResizeBag( perm, sizeof(libGAP_UInt2)*rdeg);
      }
    }

  return (libGAP_Obj)0;
}

/****************************************************************************
**
*F  FunSPLIT_PARTITION( <Ppoints>, <Qnum>,<j>,<g>,<l>)
**  <l> is a list [<a>,<b>,<max>] -- needed because of large parameter number
**
**  This function is used in the partition backtrack to split a partition.
**  The points <i> in the list Ppoints between a (start) and b (end) such
**  that Qnum[i^g]=j will be moved at the end.
**  At most <max> points will be moved.
**  The function returns the start point of the new partition (or -1 if too
**  many are moved).
**  Ppoints and Qnum must be plain lists of small integers.
*/
libGAP_Obj libGAP_FunSPLIT_PARTITION(
    libGAP_Obj self,
    libGAP_Obj Ppoints,
    libGAP_Obj Qnum,
    libGAP_Obj jval,
    libGAP_Obj libGAP_g,
    libGAP_Obj lst)
{
  libGAP_Int a;
  libGAP_Int b;
  libGAP_Int cnt;
  libGAP_Int max;
  libGAP_Int blim;
  libGAP_UInt deg;
  libGAP_UInt2 * gpt; /* perm pointer for 2 and 4 bytes */
  libGAP_UInt4 * gpf;
  libGAP_Obj tmp;


  a=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(lst,1))-1;
  b=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(lst,2))+1;
  max=libGAP_INT_INTOBJ(libGAP_ELM_PLIST(lst,3));
  cnt=0;
  blim=b-max-1;

  if (libGAP_TNUM_OBJ(libGAP_g)==libGAP_T_PERM2) {
    deg=libGAP_DEG_PERM2(libGAP_g);
    gpt=libGAP_ADDR_PERM2(libGAP_g);
    while ( (a<b)) {
      do {
	b--;
	if (b<blim) {
	  /* too many points got moved out */
	  return libGAP_INTOBJ_INT(-1);
	}
      } while (libGAP_ELM_PLIST(Qnum,
	      libGAP_IMAGETWO(libGAP_INT_INTOBJ(libGAP_ELM_PLIST(Ppoints,b))-1,gpt,deg)+1)==jval);
      do {
	a++;
      } while ((a<b)
	      &&(!(libGAP_ELM_PLIST(Qnum,
	      libGAP_IMAGETWO(libGAP_INT_INTOBJ(libGAP_ELM_PLIST(Ppoints,a))-1,gpt,deg)+1)==jval)));
      /* swap */
      if (a<b) {
	tmp=libGAP_ELM_PLIST(Ppoints,a);
	libGAP_SET_ELM_PLIST(Ppoints,a,libGAP_ELM_PLIST(Ppoints,b));
	libGAP_SET_ELM_PLIST(Ppoints,b,tmp);
	cnt++;
      }
    }
  }
  else {
    deg=libGAP_DEG_PERM4(libGAP_g);
    gpf=libGAP_ADDR_PERM4(libGAP_g);
    while ( (a<b)) {
      do {
	b--;
	if (b<blim) {
	  /* too many points got moved out */
	  return libGAP_INTOBJ_INT(-1);
	}
      } while (libGAP_ELM_PLIST(Qnum,
		  libGAP_IMAGE(libGAP_INT_INTOBJ(libGAP_ELM_PLIST(Ppoints,b))-1,gpf,deg)+1)==jval);
      do {
	a++;
      } while ((a<b)
	      &&(!(libGAP_ELM_PLIST(Qnum,
		  libGAP_IMAGE(libGAP_INT_INTOBJ(libGAP_ELM_PLIST(Ppoints,a))-1,gpf,deg)+1)==jval)));
      /* swap */
      if (a<b) {
	tmp=libGAP_ELM_PLIST(Ppoints,a);
	libGAP_SET_ELM_PLIST(Ppoints,a,libGAP_ELM_PLIST(Ppoints,b));
	libGAP_SET_ELM_PLIST(Ppoints,b,tmp);
	cnt++;
      }
    }
  }
  /* list is not necc. sorted wrt. \< (any longer) */
  libGAP_RESET_FILT_LIST(Ppoints, libGAP_FN_IS_SSORT);
  libGAP_RESET_FILT_LIST(Ppoints, libGAP_FN_IS_NSORT);

  return libGAP_INTOBJ_INT(b+1);
}

/*****************************************************************************
**
*F  FuncDistancePerms( <perm1>, <perm2> )
**
**  'DistancePerms' returns the number of points moved by <perm1>/<perm2>
**
*/

libGAP_Obj libGAP_FuncDistancePerms( libGAP_Obj self, libGAP_Obj p1, libGAP_Obj p2) 
{
  libGAP_UInt dist = 0;
  if (libGAP_TNUM_OBJ(p1) == libGAP_T_PERM2 && libGAP_TNUM_OBJ(p2) == libGAP_T_PERM2) {
    libGAP_UInt2 *pt1 = libGAP_ADDR_PERM2(p1);
    libGAP_UInt2 *pt2 = libGAP_ADDR_PERM2(p2);
    libGAP_UInt l1 = libGAP_DEG_PERM2(p1);
    libGAP_UInt l2 = libGAP_DEG_PERM2(p2);
    libGAP_UInt lmin = (l1 < l2) ? l1 : l2;
    libGAP_UInt i;
    for (i = 0; i < lmin; i++)
      if (pt1[i] != pt2[i])
	dist++;
    for (; i < l1; i++)
      if (pt1[i] != i)
	dist++;
    for (; i < l2; i++)
      if (pt2[i] != i)
	dist++;
  } else {
    if (libGAP_TNUM_OBJ(p1) == libGAP_T_PERM2 && libGAP_TNUM_OBJ(p2) == libGAP_T_PERM4) {
      libGAP_Obj temp = p1;
      p1 = p2;
      p2 = temp;
    }
    if (libGAP_TNUM_OBJ(p1) == libGAP_T_PERM4 && libGAP_TNUM_OBJ(p2) == libGAP_T_PERM2) {
      libGAP_UInt4 *pt1 = libGAP_ADDR_PERM4(p1);
      libGAP_UInt2 *pt2 = libGAP_ADDR_PERM2(p2);
      libGAP_UInt l1 = libGAP_DEG_PERM4(p1);
      libGAP_UInt l2 = libGAP_DEG_PERM2(p2);
      libGAP_UInt lmin = (l1 < l2) ? l1 : l2;
      libGAP_UInt i;
      for (i = 0; i < lmin; i++)
	if (pt1[i] != pt2[i])
	  dist++;
      for (; i < l1; i++)
	if (pt1[i] != i)
	  dist++;
      for (; i < l2; i++)
	if (pt2[i] != i)
	  dist++;
    } else {
      libGAP_UInt4 *pt1 = libGAP_ADDR_PERM4(p1);
      libGAP_UInt4 *pt2 = libGAP_ADDR_PERM4(p2);
      libGAP_UInt l1 = libGAP_DEG_PERM4(p1);
      libGAP_UInt l2 = libGAP_DEG_PERM4(p2);
      libGAP_UInt lmin = (l1 < l2) ? l1 : l2;
      libGAP_UInt i;
      for (i = 0; i < lmin; i++)
	if (pt1[i] != pt2[i])
	  dist++;
      for (; i < l1; i++)
	if (pt1[i] != i)
	  dist++;
      for (; i < l2; i++)
	if (pt2[i] != i)
	  dist++;
    }
  }
  return libGAP_INTOBJ_INT(dist);
}

/****************************************************************************
**
*F  Funsmallestimgtupleperm( <tup>, <perm> )
**
**  `SmallestImgTuplePerm' returns the smallest image of the  tuple  <tup>
**  under  the permutation <perm>.
*/
libGAP_Obj             libGAP_FunSmallestImgTuplePerm (
    libGAP_Obj			self,
    libGAP_Obj                 tup,
    libGAP_Obj                 perm )
{
    libGAP_UInt                res;            /* handle of the image, result     */
    libGAP_Obj *               ptTup;          /* pointer to the tuple            */
    libGAP_UInt2 *             ptPrm2;         /* pointer to the permutation      */
    libGAP_UInt4 *             ptPrm4;         /* pointer to the permutation      */
    libGAP_UInt                tmp;            /* temporary handle                */
    libGAP_UInt                lmp;            /* largest moved point             */
    libGAP_UInt                i, k;           /* loop variables                  */

    res = 2UL << 30; /* ``infty''. 
                        We have no permutations on over 2^28 points        */
    /* handle small permutations                                           */
    if ( libGAP_TNUM_OBJ(perm) == libGAP_T_PERM2 ) {

        /* get the pointer                                                 */
        ptTup = libGAP_ADDR_OBJ(tup) + libGAP_LEN_LIST(tup);
        ptPrm2 = libGAP_ADDR_PERM2(perm);
        lmp = libGAP_DEG_PERM2(perm);

        /* loop over the entries of the tuple                              */
        for ( i = libGAP_LEN_LIST(tup); 1 <= i; i--, ptTup-- ) {
	  k = libGAP_INT_INTOBJ( *ptTup );
	  if ( k <= lmp )
	      tmp = ptPrm2[k-1] + 1;
	  else
	      tmp = k ;
	  if (tmp<res) res = tmp;
        }

    }

    /* handle large permutations                                           */
    else {

        /* get the pointer                                                 */
        ptTup = libGAP_ADDR_OBJ(tup) + libGAP_LEN_LIST(tup);
        ptPrm4 = libGAP_ADDR_PERM4(perm);
        lmp = libGAP_DEG_PERM4(perm);

        /* loop over the entries of the tuple                              */
        for ( i = libGAP_LEN_LIST(tup); 1 <= i; i--, ptTup-- ) {
	  k = libGAP_INT_INTOBJ( *ptTup );
	  if ( k <= lmp )
	      tmp = ptPrm4[k-1] + 1;
	  else
	      tmp = k;
	  if (tmp<res) res = tmp;
        }

    }

    /* return the result                                                   */
    return libGAP_INTOBJ_INT(res);
}

/****************************************************************************
**
*F  OnTuplesPerm( <tup>, <perm> )  . . . .  operations on tuples of points
**
**  'OnTuplesPerm'  returns  the  image  of  the  tuple  <tup>   under  the
**  permutation <perm>.  It is called from 'FunOnTuples'.
*/
libGAP_Obj             libGAP_OnTuplesPerm (
    libGAP_Obj                 tup,
    libGAP_Obj                 perm )
{
    libGAP_Obj                 res;            /* handle of the image, result     */
    libGAP_Obj *               ptRes;          /* pointer to the result           */
    libGAP_Obj *               ptTup;          /* pointer to the tuple            */
    libGAP_UInt2 *             ptPrm2;         /* pointer to the permutation      */
    libGAP_UInt4 *             ptPrm4;         /* pointer to the permutation      */
    libGAP_Obj                 tmp;            /* temporary handle                */
    libGAP_UInt                lmp;            /* largest moved point             */
    libGAP_UInt                i, k;           /* loop variables                  */

    /* make a bag for the result and initialize pointers                   */
    res = libGAP_NEW_PLIST( libGAP_IS_MUTABLE_PLIST(tup) ? libGAP_T_PLIST : libGAP_T_PLIST + libGAP_IMMUTABLE,
		     libGAP_LEN_LIST(tup) );
    libGAP_ADDR_OBJ(res)[0] = libGAP_ADDR_OBJ(tup)[0];

    /* handle small permutations                                           */
    if ( libGAP_TNUM_OBJ(perm) == libGAP_T_PERM2 ) {

        /* get the pointer                                                 */
        ptTup = libGAP_ADDR_OBJ(tup) + libGAP_LEN_LIST(tup);
        ptRes = libGAP_ADDR_OBJ(res) + libGAP_LEN_LIST(tup);
        ptPrm2 = libGAP_ADDR_PERM2(perm);
        lmp = libGAP_DEG_PERM2(perm);

        /* loop over the entries of the tuple                              */
        for ( i = libGAP_LEN_LIST(tup); 1 <= i; i--, ptTup--, ptRes-- ) {
            if (libGAP_IS_INTOBJ(*ptTup)) {
                k = libGAP_INT_INTOBJ( *ptTup );
                if ( k <= 0 || k > lmp )
                    tmp = *ptTup;
                else
                    tmp = libGAP_INTOBJ_INT( ptPrm2[k-1] + 1 );
                *ptRes = tmp;
            }
            else {
                if (*ptTup == NULL) {
                  libGAP_ErrorQuit("OnTuples for perm: list must not contain holes",
                            0L, 0L);
                }
                tmp = libGAP_POW( *ptTup, perm );
                ptTup = libGAP_ADDR_OBJ(tup) + i;
                ptRes = libGAP_ADDR_OBJ(res) + i;
                ptPrm2 = libGAP_ADDR_PERM2(perm);
                *ptRes = tmp;
                libGAP_CHANGED_BAG( res );
            }
        }

    }

    /* handle large permutations                                           */
    else {

        /* get the pointer                                                 */
        ptTup = libGAP_ADDR_OBJ(tup) + libGAP_LEN_LIST(tup);
        ptRes = libGAP_ADDR_OBJ(res) + libGAP_LEN_LIST(tup);
        ptPrm4 = libGAP_ADDR_PERM4(perm);
        lmp = libGAP_DEG_PERM4(perm);

        /* loop over the entries of the tuple                              */
        for ( i = libGAP_LEN_LIST(tup); 1 <= i; i--, ptTup--, ptRes-- ) {
            if (libGAP_IS_INTOBJ(*ptTup)) {
                k = libGAP_INT_INTOBJ( *ptTup );
                if ( k <= 0 || k > lmp )
                    tmp = *ptTup;
                else
                    tmp = libGAP_INTOBJ_INT( ptPrm4[k-1] + 1 );
                *ptRes = tmp;
            }
            else {
                if (*ptTup == NULL) {
                  libGAP_ErrorQuit("OnTuples for perm: list must not contain holes",
                            0L, 0L);
                }
                tmp = libGAP_POW( *ptTup, perm );
                ptTup = libGAP_ADDR_OBJ(tup) + i;
                ptRes = libGAP_ADDR_OBJ(res) + i;
                ptPrm4 = libGAP_ADDR_PERM4(perm);
                *ptRes = tmp;
                libGAP_CHANGED_BAG( res );
            }
        }

    }

    /* return the result                                                   */
    return res;
}


/****************************************************************************
**
*F  OnSetsPerm( <set>, <perm> ) . . . . . . . .  operations on sets of points
**
**  'OnSetsPerm' returns the  image of the  tuple <set> under the permutation
**  <perm>.  It is called from 'FunOnSets'.
*/
libGAP_Obj             libGAP_OnSetsPerm (
    libGAP_Obj                 set,
    libGAP_Obj                 perm )
{
    libGAP_Obj                 res;            /* handle of the image, result     */
    libGAP_Obj *               ptRes;          /* pointer to the result           */
    libGAP_Obj *               ptTup;          /* pointer to the tuple            */
    libGAP_UInt2 *             ptPrm2;         /* pointer to the permutation      */
    libGAP_UInt4 *             ptPrm4;         /* pointer to the permutation      */
    libGAP_Obj                 tmp;            /* temporary handle                */
    libGAP_UInt                lmp;            /* largest moved point             */
    libGAP_UInt                isint;          /* <set> only holds integers       */
    libGAP_UInt                len;            /* logical length of the list      */
    libGAP_UInt                h;              /* gap width in the shellsort      */
    libGAP_UInt                i, k;           /* loop variables                  */

    /* make a bag for the result and initialize pointers                   */
    res = libGAP_NEW_PLIST(  libGAP_IS_MUTABLE_PLIST(set) ? libGAP_T_PLIST : libGAP_T_PLIST + libGAP_IMMUTABLE , libGAP_LEN_LIST(set) );
    libGAP_ADDR_OBJ(res)[0] = libGAP_ADDR_OBJ(set)[0];

    /* handle small permutations                                           */
    if ( libGAP_TNUM_OBJ(perm) == libGAP_T_PERM2 ) {

        /* get the pointer                                                 */
        ptTup = libGAP_ADDR_OBJ(set) + libGAP_LEN_LIST(set);
        ptRes = libGAP_ADDR_OBJ(res) + libGAP_LEN_LIST(set);
        ptPrm2 = libGAP_ADDR_PERM2(perm);
        lmp = libGAP_DEG_PERM2(perm);

        /* loop over the entries of the tuple                              */
        isint = 1;
        for ( i = libGAP_LEN_LIST(set); 1 <= i; i--, ptTup--, ptRes-- ) {
            if ( libGAP_TNUM_OBJ( *ptTup ) == libGAP_T_INT && 0 < libGAP_INT_INTOBJ( *ptTup ) ) {
                k = libGAP_INT_INTOBJ( *ptTup );
                if ( k <= lmp )
                    tmp = libGAP_INTOBJ_INT( ptPrm2[k-1] + 1 );
                else
                    tmp = libGAP_INTOBJ_INT( k );
                *ptRes = tmp;
            }
            else {
                isint = 0;
                tmp = libGAP_POW( *ptTup, perm );
                ptTup = libGAP_ADDR_OBJ(set) + i;
                ptRes = libGAP_ADDR_OBJ(res) + i;
                ptPrm2 = libGAP_ADDR_PERM2(perm);
                *ptRes = tmp;
                libGAP_CHANGED_BAG( res );
            }
        }

    }

    /* handle large permutations                                           */
    else {

        /* get the pointer                                                 */
        ptTup = libGAP_ADDR_OBJ(set) + libGAP_LEN_LIST(set);
        ptRes = libGAP_ADDR_OBJ(res) + libGAP_LEN_LIST(set);
        ptPrm4 = libGAP_ADDR_PERM4(perm);
        lmp = libGAP_DEG_PERM4(perm);

        /* loop over the entries of the tuple                              */
        isint = 1;
        for ( i = libGAP_LEN_LIST(set); 1 <= i; i--, ptTup--, ptRes-- ) {
            if ( libGAP_TNUM_OBJ( *ptTup ) == libGAP_T_INT && 0 < libGAP_INT_INTOBJ( *ptTup ) ) {
                k = libGAP_INT_INTOBJ( *ptTup );
                if ( k <= lmp )
                    tmp = libGAP_INTOBJ_INT( ptPrm4[k-1] + 1 );
                else
                    tmp = libGAP_INTOBJ_INT( k );
                *ptRes = tmp;
            }
            else {
                isint = 0;
                tmp = libGAP_POW( *ptTup, perm );
                ptTup = libGAP_ADDR_OBJ(set) + i;
                ptRes = libGAP_ADDR_OBJ(res) + i;
                ptPrm4 = libGAP_ADDR_PERM4(perm);
                *ptRes = tmp;
                libGAP_CHANGED_BAG( res );
            }
        }

    }

    /* special case if the result only holds integers                      */
    if ( isint ) {

        /* sort the set with a shellsort                                   */
        len = libGAP_LEN_LIST(res);
        h = 1;  while ( 9*h + 4 < len )  h = 3*h + 1;
        while ( 0 < h ) {
            for ( i = h+1; i <= len; i++ ) {
                tmp = libGAP_ADDR_OBJ(res)[i];  k = i;
                while ( h < k && ((libGAP_Int)tmp < (libGAP_Int)(libGAP_ADDR_OBJ(res)[k-h])) ) {
                    libGAP_ADDR_OBJ(res)[k] = libGAP_ADDR_OBJ(res)[k-h];
                    k -= h;
                }
                libGAP_ADDR_OBJ(res)[k] = tmp;
            }
            h = h / 3;
        }
        libGAP_RetypeBag( res, libGAP_IS_MUTABLE_PLIST(set) ? libGAP_T_PLIST_CYC_SSORT : libGAP_T_PLIST_CYC_SSORT + libGAP_IMMUTABLE );
    }

    /* general case                                                        */
    else {

        /* sort the set with a shellsort                                   */
        len = libGAP_LEN_LIST(res);
        h = 1;  while ( 9*h + 4 < len )  h = 3*h + 1;
        while ( 0 < h ) {
            for ( i = h+1; i <= len; i++ ) {
                tmp = libGAP_ADDR_OBJ(res)[i];  k = i;
                while ( h < k && libGAP_LT( tmp, libGAP_ADDR_OBJ(res)[k-h] ) ) {
                    libGAP_ADDR_OBJ(res)[k] = libGAP_ADDR_OBJ(res)[k-h];
                    k -= h;
                }
                libGAP_ADDR_OBJ(res)[k] = tmp;
            }
            h = h / 3;
        }

        /* remove duplicates, shrink bag if possible                       */
        if ( 0 < len ) {
            tmp = libGAP_ADDR_OBJ(res)[1];  k = 1;
            for ( i = 2; i <= len; i++ ) {
                if ( ! libGAP_EQ( tmp, libGAP_ADDR_OBJ(res)[i] ) ) {
                    k++;
                    tmp = libGAP_ADDR_OBJ(res)[i];
                    libGAP_ADDR_OBJ(res)[k] = tmp;
                }
            }
            if ( k < len ) {
                libGAP_ResizeBag( res, (k+1)*sizeof(libGAP_Obj) );
                libGAP_ADDR_OBJ(res)[0] = libGAP_INTOBJ_INT(k);
            }
        }

    }

    /* return the result                                                   */
    return res;
}

/****************************************************************************
**
*F  SavePerm2( <perm2> )
**
*/

void libGAP_SavePerm2( libGAP_Obj perm)
{
  libGAP_UInt i;
  libGAP_UInt2 *ptr;
  libGAP_UInt len;
  len = libGAP_DEG_PERM2(perm);
  ptr = libGAP_ADDR_PERM2(perm);
  for (i = 0; i < len; i++)
    libGAP_SaveUInt2( *ptr++);
}

/****************************************************************************
**
*F  SavePerm4( <perm4> )
**
*/

void libGAP_SavePerm4( libGAP_Obj perm)
{
  libGAP_UInt i;
  libGAP_UInt4 *ptr;
  libGAP_UInt len;
  len = libGAP_DEG_PERM4(perm);
  ptr = libGAP_ADDR_PERM4(perm);
  for (i = 0; i < len; i++)
    libGAP_SaveUInt4( *ptr++);
}

/****************************************************************************
**
*F  LoadPerm2( <perm2> )
**
*/

void libGAP_LoadPerm2( libGAP_Obj perm)
{
  libGAP_UInt i;
  libGAP_UInt2 *ptr;
  libGAP_UInt len;
  len = libGAP_DEG_PERM2(perm);
  ptr = libGAP_ADDR_PERM2(perm);
  for (i = 0; i < len; i++)
    *ptr++ = libGAP_LoadUInt2();
}

/****************************************************************************
**
*F  LoadPerm4( <perm4> )
**
*/

void libGAP_LoadPerm4( libGAP_Obj perm)
{
  libGAP_UInt i;
  libGAP_UInt4 *ptr;
  libGAP_UInt len;
  len = libGAP_DEG_PERM4(perm);
  ptr = libGAP_ADDR_PERM4(perm);
  for (i = 0; i < len; i++)
    *ptr++ = libGAP_LoadUInt4( );
}


/****************************************************************************
**
*F  Array2Perm( <array> ) . . . . . . . . . convert array of cycles into perm
*/
libGAP_Obj libGAP_Array2Perm (
    libGAP_Obj                 array )
{
    libGAP_Obj                 perm;           /* permutation, result             */
    libGAP_UInt4 *             ptr4;           /* pointer into perm               */
    libGAP_UInt2 *             ptr2;           /* pointer into perm               */
    libGAP_Obj                 val;            /* one entry as value              */
    libGAP_UInt                c, p, l;        /* entries in permutation          */
    libGAP_UInt                m;              /* maximal entry in permutation    */
    libGAP_Obj                 cycle;          /* one cycle of permutation        */
    libGAP_UInt                i, j, k;        /* loop variable                   */

    /* special case for identity permutation                               */
    if ( libGAP_LEN_LIST(array) == 0 ) {
        return libGAP_IdentityPerm;
    }

    /* allocate the new permutation                                        */
    m = 0;
    perm = libGAP_NEW_PERM4( 0 );

    /* loop over the cycles                                                */
    for ( i = 1; i <= libGAP_LEN_LIST(array); i++ ) {
        cycle = libGAP_ELM_LIST( array, i );
        while ( ! libGAP_IS_SMALL_LIST(cycle) ) {
            cycle = libGAP_ErrorReturnObj(
                "Array2Perm: <cycle> must be a small list (not a %s)",
                (libGAP_Int)libGAP_TNAM_OBJ(cycle), 0L,
                "you can replace <cycle> via 'return <cycle>;'" );
        }

        /* loop over the entries of the cycle                              */
        c = p = l = 0;
        for ( j = libGAP_LEN_LIST(cycle); 1 <= j; j-- ) {

            /* get and check current entry for the cycle                   */
            val = libGAP_ELM_LIST( cycle, j );
            while ( ! libGAP_IS_INTOBJ(val) || libGAP_INT_INTOBJ(val) <= 0 ) {
                val = libGAP_ErrorReturnObj(
              "Permutation: <expr> must be a positive integer (not to a %s)",
                    (libGAP_Int)libGAP_TNAM_OBJ(val), 0L,
                    "you can replace <expr> via 'return <expr>;'" );
            }
            c = libGAP_INT_INTOBJ(val);

            /* if necessary resize the permutation                         */
            if ( libGAP_SIZE_OBJ(perm)/sizeof(libGAP_UInt4) < c ) {
                libGAP_ResizeBag( perm, (c + 1023) / 1024 * 1024 * sizeof(libGAP_UInt4) );
                ptr4 = libGAP_ADDR_PERM4( perm );
                for ( k = m+1; k <= libGAP_SIZE_OBJ(perm)/sizeof(libGAP_UInt4); k++ ) {
                    ptr4[k-1] = k-1;
                }
            }
            if ( m < c ) {
                m = c;
            }

            /* check that the cycles are disjoint                          */
            ptr4 = libGAP_ADDR_PERM4( perm );
            if ( (p != 0 && p == c) || (ptr4[c-1] != c-1) ) {
                return libGAP_ErrorReturnObj(
                    "Permutation: cycles must be disjoint",
                    0L, 0L,
                    "you can replace the permutation <perm> via 'return <perm>;'" );
            }

            /* enter the previous entry at current location                */
            ptr4 = libGAP_ADDR_PERM4( perm );
            if ( p != 0 ) { ptr4[c-1] = p-1; }
            else          { l = c;          }

            /* remember current entry for next round                       */
            p = c;
        }

        /* enter first (last popped) entry at last (first popped) location */
        ptr4 = libGAP_ADDR_PERM4( perm );
        ptr4[l-1] = p-1;

    }

    /* if possible represent the permutation with short entries            */
    if ( m <= 65536UL ) {
        ptr2 = libGAP_ADDR_PERM2( perm );
        ptr4 = libGAP_ADDR_PERM4( perm );
        for ( k = 1; k <= m; k++ ) {
            ptr2[k-1] = ptr4[k-1];
        };
        libGAP_RetypeBag( perm, libGAP_T_PERM2 );
        libGAP_ResizeBag( perm, m * sizeof(libGAP_UInt2) );
    }

    /* otherwise just shorten the permutation                              */
    else {
        libGAP_ResizeBag( perm, m * sizeof(libGAP_UInt4) );
    }

    /* return the permutation                                              */
    return perm;
}



/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/

/****************************************************************************
**

*V  GVarFilts . . . . . . . . . . . . . . . . . . . list of filters to export
*/
static libGAP_StructGVarFilt libGAP_GVarFilts [] = {

    { "IS_PERM", "obj", &libGAP_IsPermFilt,
      libGAP_IsPermHandler, "src/permutat.c:IS_PERM" },

    { 0 }

};


/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "PermList", 1, "list",
      libGAP_FuncPermList, "src/permutat.c:PermList" },

    { "LARGEST_MOVED_POINT_PERM", 1, "perm",
      libGAP_FuncLARGEST_MOVED_POINT_PERM, "src/permutat.c:LARGEST_MOVED_POINT_PERM" },

    { "CYCLE_LENGTH_PERM_INT", 2, "perm, point",
      libGAP_FuncCycleLengthPermInt, "src/permutat.c:CycleLengthPermInt" },

    { "CYCLE_PERM_INT", 2, "perm, point",
      libGAP_FuncCyclePermInt, "src/permutat.c:CyclePermInt" },

    { "CYCLE_STRUCT_PERM", 1, "perm",
      libGAP_FuncCycleStructurePerm, "src/permutat.c:CycleStructPerm" },

    { "ORDER_PERM", 1, "perm",
      libGAP_FuncOrderPerm, "src/permutat.c:OrderPerm" },

    { "SIGN_PERM", 1, "perm",
      libGAP_FuncSignPerm, "src/permutat.c:SignPerm" },

    { "SMALLEST_GENERATOR_PERM", 1, "perm",
      libGAP_FuncSmallestGeneratorPerm, "src/permutat.c:SmallestGeneratorPerm" },

    { "RESTRICTED_PERM", 3, "perm,domain,test",
      libGAP_FuncRESTRICTED_PERM, "src/permutat.c:RESTRICTED_PERM" },

    { "SHIFTED_PERM", 2, "perm,shift",
      libGAP_FuncSHIFTED_PERM, "src/permutat.c:SHIFTED_PERM" },

    { "TRIM_PERM", 2, "perm, degree",
      libGAP_FuncTRIM_PERM, "src/permutat.c:TRIM_PERM" },

    { "SPLIT_PARTITION", 5, "Ppoints, Qn,j,g,a,b,max",
      libGAP_FunSPLIT_PARTITION, "src/permutat.c:SPLIT_PARTITION" },

    { "SMALLEST_IMG_TUP_PERM", 2, "tuple, perm",
      libGAP_FunSmallestImgTuplePerm, "src/permutat.c:SMALLEST_IMG_TUP_PERM" },

    { "MUL_PERMS_COOPERMAN", 3, "perm, perm, logBucketsize",
      libGAP_FuncMUL_PERMS_COOPERMAN, "src/permutat.c:MUL_PERMS_COOPERMAN" },
    
    { "INV_PERM_COOPERMAN", 2, "perm, logBucketsize",
      libGAP_FuncINV_PERM_COOPERMAN, "src/permutat.c:INV_PERM_COOPERMAN" },

    /*    { "INV_PERM_SIMPLE", 1, "perm",
	  FuncINV_PERM_SIMPLE, "src/permutat.cINV_PERM_SIMPLE" }, */
    
    { "LQUO_PERMS_COOPERMAN", 3, "perm1, perm2, logBucketsize",
      libGAP_FuncLQUO_PERMS_COOPERMAN, "src/permutat.c:LQUO_PERMS_COOPERMAN" },
    
    { "DISTANCE_PERMS", 2, "perm1, perm2",
      libGAP_FuncDistancePerms, "src/permutat.c:DISTANCE_PERMS" },
    


    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* install the marking function                                        */
    libGAP_InfoBags[           libGAP_T_PERM2         ].name = "permutation (small)";
    libGAP_InitMarkFuncBags(   libGAP_T_PERM2         , libGAP_MarkNoSubBags );
    libGAP_InfoBags[           libGAP_T_PERM4         ].name = "permutation (large)";
    libGAP_InitMarkFuncBags(   libGAP_T_PERM4         , libGAP_MarkNoSubBags );

    /* install the kind function                                           */
    libGAP_ImportGVarFromLibrary( "TYPE_PERM2", &libGAP_TYPE_PERM2 );
    libGAP_ImportGVarFromLibrary( "TYPE_PERM4", &libGAP_TYPE_PERM4 );

    libGAP_TypeObjFuncs[ libGAP_T_PERM2 ] = libGAP_TypePerm2;
    libGAP_TypeObjFuncs[ libGAP_T_PERM4 ] = libGAP_TypePerm4;

    /* init filters and functions                                          */
    libGAP_InitHdlrFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* make the buffer bag                                                 */
    libGAP_InitGlobalBag( &libGAP_TmpPerm, "src/permutat.c:TmpPerm" );

    /* make the identity permutation                                       */
    libGAP_InitGlobalBag( &libGAP_IdentityPerm, "src/permutat.c:IdentityPerm" );

    /* install the saving functions */
    libGAP_SaveObjFuncs[ libGAP_T_PERM2 ] = libGAP_SavePerm2;
    libGAP_SaveObjFuncs[ libGAP_T_PERM4 ] = libGAP_SavePerm4;
    libGAP_LoadObjFuncs[ libGAP_T_PERM2 ] = libGAP_LoadPerm2;
    libGAP_LoadObjFuncs[ libGAP_T_PERM4 ] = libGAP_LoadPerm4;

    /* install the printing functions                                      */
    libGAP_PrintObjFuncs[ libGAP_T_PERM2   ] = libGAP_PrintPermP;
    libGAP_PrintObjFuncs[ libGAP_T_PERM4   ] = libGAP_PrintPermQ;

    /* install the comparison methods                                      */
    libGAP_EqFuncs  [ libGAP_T_PERM2  ][ libGAP_T_PERM2  ] = libGAP_EqPerm22;
    libGAP_EqFuncs  [ libGAP_T_PERM2  ][ libGAP_T_PERM4  ] = libGAP_EqPerm24;
    libGAP_EqFuncs  [ libGAP_T_PERM4  ][ libGAP_T_PERM2  ] = libGAP_EqPerm42;
    libGAP_EqFuncs  [ libGAP_T_PERM4  ][ libGAP_T_PERM4  ] = libGAP_EqPerm44;
    libGAP_LtFuncs  [ libGAP_T_PERM2  ][ libGAP_T_PERM2  ] = libGAP_LtPerm22;
    libGAP_LtFuncs  [ libGAP_T_PERM2  ][ libGAP_T_PERM4  ] = libGAP_LtPerm24;
    libGAP_LtFuncs  [ libGAP_T_PERM4  ][ libGAP_T_PERM2  ] = libGAP_LtPerm42;
    libGAP_LtFuncs  [ libGAP_T_PERM4  ][ libGAP_T_PERM4  ] = libGAP_LtPerm44;

    /* install the binary operations                                       */
    libGAP_ProdFuncs[ libGAP_T_PERM2  ][ libGAP_T_PERM2  ] = libGAP_ProdPerm22;
    libGAP_ProdFuncs[ libGAP_T_PERM2  ][ libGAP_T_PERM4  ] = libGAP_ProdPerm24;
    libGAP_ProdFuncs[ libGAP_T_PERM4  ][ libGAP_T_PERM2  ] = libGAP_ProdPerm42;
    libGAP_ProdFuncs[ libGAP_T_PERM4  ][ libGAP_T_PERM4  ] = libGAP_ProdPerm44;
    libGAP_QuoFuncs [ libGAP_T_PERM2  ][ libGAP_T_PERM2  ] = libGAP_QuoPerm22;
    libGAP_QuoFuncs [ libGAP_T_PERM2  ][ libGAP_T_PERM4  ] = libGAP_QuoPerm24;
    libGAP_QuoFuncs [ libGAP_T_PERM4  ][ libGAP_T_PERM2  ] = libGAP_QuoPerm42;
    libGAP_QuoFuncs [ libGAP_T_PERM4  ][ libGAP_T_PERM4  ] = libGAP_QuoPerm44;
    libGAP_LQuoFuncs[ libGAP_T_PERM2  ][ libGAP_T_PERM2  ] = libGAP_LQuoPerm22;
    libGAP_LQuoFuncs[ libGAP_T_PERM2  ][ libGAP_T_PERM4  ] = libGAP_LQuoPerm24;
    libGAP_LQuoFuncs[ libGAP_T_PERM4  ][ libGAP_T_PERM2  ] = libGAP_LQuoPerm42;
    libGAP_LQuoFuncs[ libGAP_T_PERM4  ][ libGAP_T_PERM4  ] = libGAP_LQuoPerm44;
    libGAP_PowFuncs [ libGAP_T_PERM2  ][ libGAP_T_INT    ] = libGAP_PowPerm2Int;
    libGAP_PowFuncs [ libGAP_T_PERM2  ][ libGAP_T_INTPOS ] = libGAP_PowPerm2Int;
    libGAP_PowFuncs [ libGAP_T_PERM2  ][ libGAP_T_INTNEG ] = libGAP_PowPerm2Int;
    libGAP_PowFuncs [ libGAP_T_PERM4  ][ libGAP_T_INT    ] = libGAP_PowPerm4Int;
    libGAP_PowFuncs [ libGAP_T_PERM4  ][ libGAP_T_INTPOS ] = libGAP_PowPerm4Int;
    libGAP_PowFuncs [ libGAP_T_PERM4  ][ libGAP_T_INTNEG ] = libGAP_PowPerm4Int;
    libGAP_PowFuncs [ libGAP_T_INT    ][ libGAP_T_PERM2  ] = libGAP_PowIntPerm2;
    libGAP_PowFuncs [ libGAP_T_INTPOS ][ libGAP_T_PERM2  ] = libGAP_PowIntPerm2;
    libGAP_PowFuncs [ libGAP_T_INT    ][ libGAP_T_PERM4  ] = libGAP_PowIntPerm4;
    libGAP_PowFuncs [ libGAP_T_INTPOS ][ libGAP_T_PERM4  ] = libGAP_PowIntPerm4;
    libGAP_QuoFuncs [ libGAP_T_INT    ][ libGAP_T_PERM2  ] = libGAP_QuoIntPerm2;
    libGAP_QuoFuncs [ libGAP_T_INTPOS ][ libGAP_T_PERM2  ] = libGAP_QuoIntPerm2;
    libGAP_QuoFuncs [ libGAP_T_INT    ][ libGAP_T_PERM4  ] = libGAP_QuoIntPerm4;
    libGAP_QuoFuncs [ libGAP_T_INTPOS ][ libGAP_T_PERM4  ] = libGAP_QuoIntPerm4;
    libGAP_PowFuncs [ libGAP_T_PERM2  ][ libGAP_T_PERM2  ] = libGAP_PowPerm22;
    libGAP_PowFuncs [ libGAP_T_PERM2  ][ libGAP_T_PERM4  ] = libGAP_PowPerm24;
    libGAP_PowFuncs [ libGAP_T_PERM4  ][ libGAP_T_PERM2  ] = libGAP_PowPerm42;
    libGAP_PowFuncs [ libGAP_T_PERM4  ][ libGAP_T_PERM4  ] = libGAP_PowPerm44;
    libGAP_CommFuncs[ libGAP_T_PERM2  ][ libGAP_T_PERM2  ] = libGAP_CommPerm22;
    libGAP_CommFuncs[ libGAP_T_PERM2  ][ libGAP_T_PERM4  ] = libGAP_CommPerm24;
    libGAP_CommFuncs[ libGAP_T_PERM4  ][ libGAP_T_PERM2  ] = libGAP_CommPerm42;
    libGAP_CommFuncs[ libGAP_T_PERM4  ][ libGAP_T_PERM4  ] = libGAP_CommPerm44;

    /* install the 'ONE' function for permutations                         */
    libGAP_OneFuncs[ libGAP_T_PERM2 ] = libGAP_OnePerm;
    libGAP_OneFuncs[ libGAP_T_PERM4 ] = libGAP_OnePerm;
    libGAP_OneMutFuncs[ libGAP_T_PERM2 ] = libGAP_OnePerm;
    libGAP_OneMutFuncs[ libGAP_T_PERM4 ] = libGAP_OnePerm;

    /* install the 'INV' function for permutations                         */
    libGAP_InvFuncs[ libGAP_T_PERM2 ] = libGAP_InvPerm;
    libGAP_InvFuncs[ libGAP_T_PERM4 ] = libGAP_InvPerm;
    libGAP_InvMutFuncs[ libGAP_T_PERM2 ] = libGAP_InvPerm;
    libGAP_InvMutFuncs[ libGAP_T_PERM4 ] = libGAP_InvPerm;

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitGVarFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* make the buffer bag                                                 */
    libGAP_TmpPerm = libGAP_NEW_PERM4( 1000 );

    /* make the identity permutation                                       */
    libGAP_IdentityPerm = libGAP_NEW_PERM2(0);

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoPermutat()  . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "permutat",                         /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoPermutat ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}


/****************************************************************************
**

*E  permutat.c  . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
  
