/****************************************************************************
**
*W  lists.c                     GAP source                   Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions of the generic list package.
**
**  This package provides a uniform   interface to the functions that  access
**  lists and their elements  for the other packages  in the GAP kernel.  For
**  example, 'ExecFor' can loop over the elements  in a list using the macros
**  'LEN_LIST' and 'ELM_LIST' independently of the type of the list.
**
**  This package uses plain lists (of type 'T_PLIST') and  assumes that it is
**  possible to put values of any type into plain  lists.  It uses the macros
**  'LEN_PLIST', 'SET_LEN_PLIST',   'ELM_PLIST', and 'SET_ELM_PLIST' exported
**  by the plain list package to access and modify plain lists.
*/
#include        "system.h"              /* Ints, UInts                     */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */

#include        "calls.h"               /* generic call mechanism          */
#include        "opers.h"               /* generic operations              */
#include        "ariths.h"              /* basic arithmetic                */

#include        "records.h"             /* generic records                 */

#include        "lists.h"               /* generic lists                   */

#include        "bool.h"                /* booleans                        */

#include        "precord.h"             /* plain records                   */

#include        "plist.h"               /* plain lists                     */
#include        "range.h"               /* ranges                          */
#include        "string.h"              /* strings                         */
#include        "integer.h"             /* integers                        */


/****************************************************************************
**
*F  IS_LIST(<obj>)  . . . . . . . . . . . . . . . . . . . is an object a list
*V  IsListFuncs[<type>] . . . . . . . . . . . . . . . . . table for list test
**
**  'IS_LIST' only calls the function pointed  to  by  'IsListFuncs[<type>]',
**  passing <obj> as argument.
**
**  'IS_LIST' is defined in the declaration part of this package as follows
**
#define IS_LIST(obj)    (*IsListFuncs[ TNUM_OBJ( (obj) ) ])( obj )
*/
libGAP_Int             (*libGAP_IsListFuncs [libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj obj );

libGAP_Obj             libGAP_IsListFilt;

libGAP_Obj             libGAP_FuncIS_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return (libGAP_IS_LIST( obj ) ? libGAP_True : libGAP_False);
}

libGAP_Int             libGAP_IsListNot (
    libGAP_Obj                 obj )
{
    return 0L;
}

libGAP_Int             libGAP_IsListYes (
    libGAP_Obj                 obj )
{
    return 1L;
}

libGAP_Int             libGAP_IsListObject (
    libGAP_Obj                 obj )
{
    return (libGAP_DoFilter( libGAP_IsListFilt, obj ) == libGAP_True);
}

/****************************************************************************
**
*F  IS_SMALL_LIST(<obj>)  . . . . . . . . . . . . . . . . . . . is an object a list
*V  IsListFuncs[<type>] . . . . . . . . . . . . . . . . . table for list test
**
**  'IS_SMALL_LIST' only calls the function pointed  to  by  'IsListFuncs[<type>]',
**  passing <obj> as argument.
**
**  'IS_SMALL_LIST' is defined in the declaration part of this package as follows
**
**  This is, in some sense, a workaround for the not yet implemented features
**  below (see LENGTH).
** 
#define IS_SMALL_LIST(obj)    (*IsSmallListFuncs[ TNUM_OBJ( (obj) ) ])( obj )
*/
libGAP_Int             (*libGAP_IsSmallListFuncs [libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj obj );

libGAP_Obj             libGAP_IsSmallListFilt;
libGAP_Obj             libGAP_HasIsSmallListFilt;
libGAP_Obj             libGAP_LengthAttr;
libGAP_Obj             libGAP_SetIsSmallList;

libGAP_Int             libGAP_IsSmallListNot (
    libGAP_Obj                 obj )
{
    return 0L;
}

libGAP_Int             libGAP_IsSmallListYes (
    libGAP_Obj                 obj )
{
    return 1L;
}

libGAP_Int             libGAP_IsSmallListObject (
    libGAP_Obj                 obj )
{
  libGAP_Obj len;
  if (libGAP_DoFilter(libGAP_IsListFilt, obj) != libGAP_True)
    return 0;
  if (libGAP_DoFilter(libGAP_HasIsSmallListFilt, obj) == libGAP_True)
    return libGAP_DoFilter(libGAP_IsSmallListFilt, obj) == libGAP_True;
  if (libGAP_DoTestAttribute(libGAP_LengthAttr, obj) == libGAP_True)
    {
      len = libGAP_DoAttribute(libGAP_LengthAttr, obj);
      if (libGAP_IS_INTOBJ(len))
        {
          libGAP_CALL_2ARGS(libGAP_SetIsSmallList, obj, libGAP_True);
          return 1;
        }
      else
        {
          libGAP_CALL_2ARGS(libGAP_SetIsSmallList, obj, libGAP_False);
          return 0;
        }
    }
  return 0;
}



/****************************************************************************
**
*F  FuncLENGTH( <self>, <list> ) . . . . . . . . . . .  'Length' interface
**
**  There are  the ``relatively''  easy  changes to  'LEN_LIST' to  allow  it
**  return GAP  objects instead of small C  integers, but then the kernel has
**  to be very careful not to assume that the length is small and most of the
**  code has to duplicated,  namely  one large  and  one small version.    So
**  instead the following solution has been taken:
**
**  - internal lists  have always a  small length,  that means that it is not
**    possible to have plain list of length larger than 2^28  (or maybe 2^32)
**    on 32-bit machines, 'LEN_LIST' can only be applied to internal objects,
**    'LENGTH' is the GAP interface for all kind of objects
**
**  - on  the  other hand we want ranges to have  large start and end points,
**    therefore  ranges  are no  longer  *internal*  objects,  they  are  now
**    external objects (NOT YET IMPLEMENTED)
**
**  - the for/list assignment has to be carefull to catch the special case of
**    a range constructor with small integer bounds
**
**  - the list access/assigment is a binary operation (NOT YET IMPLEMENTED)
**
**  - the conversion/test functions are split into three different functions
**    (NOT YET IMPLEMENTED)
**
**  - 'ResetFilterObj' and 'SetFilterObj'  are implemented using a table  for
**    internal types (NOT YET IMPLEMENTED)
*/

libGAP_Obj libGAP_FuncLENGTH (
    libGAP_Obj             self,
    libGAP_Obj             list )
{
    /* internal list types                                                 */
    if ( libGAP_FIRST_LIST_TNUM<=libGAP_TNUM_OBJ(list) && libGAP_TNUM_OBJ(list)<=libGAP_LAST_LIST_TNUM) {
        return libGAP_ObjInt_Int( libGAP_LEN_LIST(list) );
    }

    /* external types                                                      */
    else {
        return libGAP_DoAttribute( libGAP_LengthAttr, list );
    }
}


/****************************************************************************
**
*F  LEN_LIST(<list>)  . . . . . . . . . . . . . . . . . . .  length of a list
*V  LenListFuncs[<type>]  . . . . . . . . . . . . . table of length functions
*F  LenListError(<list>)  . . . . . . . . . . . . . . . error length function
**
**  'LEN_LIST' only calls  the function pointed to by 'LenListFuncs[<type>]',
**  passing  <list> as argument.  If <type>  is not the type  of a list, then
**  'LenListFuncs[<type>]'  points to  'LenListError', which  just signals an
**  error.
**
**  'LEN_LIST' is defined in the declaration part of this package as follows
**
#define LEN_LIST(list)  ((*LenListFuncs[ TNUM_OBJ((list)) ])( (list) ))
**
**  At the  moment  this also handles external    types but this   is a hack,
**  because external  lists can have large  length or even  be infinite.  See
**  'FuncLENGTH'.
*/
libGAP_Int (*libGAP_LenListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list );

libGAP_Obj libGAP_FuncLEN_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 list )
{
    /* special case for plain lists (avoid conversion back and forth)      */
    if ( libGAP_IS_PLIST(list) ) {
        return libGAP_INTOBJ_INT( libGAP_LEN_PLIST( list ) );
    }

    /* generic case (will signal an error if <list> is not a list)         */
    else {
        return libGAP_FuncLENGTH( libGAP_LengthAttr, list );
    }
}


libGAP_Int libGAP_LenListError (
    libGAP_Obj                 list )
{
    list = libGAP_ErrorReturnObj(
        "Length: <list> must be a list (not a %s)",
        (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
        "you can replace <list> via 'return <list>;'" );
    return libGAP_LEN_LIST( list );
}


libGAP_Int libGAP_LenListObject (
    libGAP_Obj                 obj )
{
    libGAP_Obj                 len;

    len = libGAP_FuncLENGTH( libGAP_LengthAttr, obj );
    while ( libGAP_TNUM_OBJ(len) != libGAP_T_INT || libGAP_INT_INTOBJ(len) < 0 ) {
        len = libGAP_ErrorReturnObj(
            "Length: method must return a nonnegative value (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(len), 0L,
            "you can replace value <length> via 'return <length>;'" );
    }
    return libGAP_INT_INTOBJ( len );
}

/****************************************************************************
**
*F  LENGTH(<list>)  . . . . . . . . . . . . . . . . . . .  length of a list
*V  LengthFuncs[<type>]  . . . . . . . . . . . . . table of length functions
**
**  'LENGTH' returns the logical length of the list <list>  as a GAP object
**  An error is signalled if <list> is not a list.
**
**  Note that  'LENGTH' is a  macro, so do  not call it with arguments that
**  have sideeffects.
**
**  A package  implementing a list type <type>  must  provide such a function
**  and install it in 'LengthFuncs[<type>]'.

#define LENGTH(list)  ((*LengthFuncs[ TNUM_OBJ(list) ])( list )) 
*/

libGAP_Obj             (*libGAP_LengthFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list );

libGAP_Obj libGAP_LengthError (
    libGAP_Obj                 list )
{
    list = libGAP_ErrorReturnObj(
        "Length: <list> must be a list (not a %s)",
        (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
        "you can replace <list> via 'return <list>;'" );
    return libGAP_LENGTH( list );
}


libGAP_Obj libGAP_LengthObject (
    libGAP_Obj                 obj )
{
    return libGAP_FuncLENGTH( libGAP_LengthAttr, obj );
}

libGAP_Obj libGAP_LengthInternal (
    libGAP_Obj                 obj )
{
    return libGAP_INTOBJ_INT(libGAP_LEN_LIST(obj));
}




/****************************************************************************
**
*F  ISB_LIST(<list>,<pos>)  . . . . . . . . . .  test for element from a list
*F  ISBV_LIST(<list>,<pos>) . . . . . . . . . .  test for element from a list
*V  IsbListFuncs[<type>]  . . . . . . . . . . . . . . table of test functions
*V  IsbvListFuncs[<type>] . . . . . . . . . . . . . . table of test functions
**
**  'ISB_LIST' only calls the function pointed to by  'IsbListFuncs[<type>]',
**  passing <list> and <pos> as arguments.  If <type> is not the  type  of  a
**  list, then 'IsbListFuncs[<type>]' points to 'IsbListError', which signals
**  the error.
**
**  'ISB_LIST' and 'ISBV_LIST'  are defined in  the declaration  part of this
**  package as follows
**
#define ISB_LIST(list,pos) \
                        ((*IsbListFuncs[TNUM_OBJ(list)])(list,pos))

#define ISBV_LIST(list,pos) \
                        ((*IsbvListFuncs[TNUM_OBJ(list)])(list,pos))
*/
libGAP_Int             (*libGAP_IsbListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list, libGAP_Int pos );

libGAP_Int             (*libGAP_IsbvListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list, libGAP_Int pos );

libGAP_Obj             libGAP_IsbListOper;

libGAP_Obj             libGAP_IsbListHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 pos )
{
    return (libGAP_ISB_LIST( list, libGAP_INT_INTOBJ(pos) ) ? libGAP_True : libGAP_False);
}

libGAP_Int             libGAP_IsbListError (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    list = libGAP_ErrorReturnObj(
        "IsBound: <list> must be a list (not a %s)",
        (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
        "you can replace <list> via 'return <list>;'" );
    return libGAP_ISB_LIST( list, pos );
}

libGAP_Int             libGAP_IsbListObject (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    return (libGAP_DoOperation2Args( libGAP_IsbListOper, list, libGAP_INTOBJ_INT(pos) ) == libGAP_True);
}

/****************************************************************************
**
*F  ISBB_LIST(<list>,<pos>,<obj>)  . . . . . isbound for an element to a list
*V  IsbbListFuncs[<type>]  . . . . . . . . .  . table of isbound functions
*F  IsbbListError(<list>,<pos>,<obj>)  . . . . . . . error isbound function
**
**  'ISBB_LIST' only calls the  function pointed to by 'IsbbListFuncs[<type>]',
**  passing <list>, <pos>, and <obj> as arguments.  If <type> is not the type
**  of  a list, then 'IsbbListFuncs[<type>]'  points to 'IsbbListError',  which
**  just signals an error.
**
**  'ISBB_LIST' is defined in the declaration part of this package as follows.
**
#define ISBB_LIST(list,pos,obj) \
                        ((*IsbbListFuncs[TNUM_OBJ(list)])(list,pos,obj))
*/
libGAP_Int            (*libGAP_IsbbListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list, libGAP_Obj pos);

libGAP_Obj             libGAP_FuncISBB_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 pos)
{
    return libGAP_ISBB_LIST( list, pos ) ? libGAP_True: libGAP_False;
}

libGAP_Int            libGAP_IsbbListError (
    libGAP_Obj                 list,
    libGAP_Obj                 pos )
{
    list = libGAP_ErrorReturnObj(
        "Isbound: <list> must be a list (not a %s)",
        (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
        "you can replace <list> via 'return <list>;'" );
    return libGAP_ISBB_LIST( list, pos );
}

libGAP_Int            libGAP_IsbbListInternal (
    libGAP_Obj                 list,
    libGAP_Obj                 pos)
{
  return 0;
}


/****************************************************************************
**
*F  IsbbListObject( <list>, <pos>, <obj> ) . . . . . . . assign to list object
*/

libGAP_Int libGAP_IsbbListObject (
    libGAP_Obj                 list,
    libGAP_Obj                 pos )
{
    return libGAP_DoOperation2Args( libGAP_IsbListOper, list, pos ) == libGAP_True ? 1 : 0;
}


/****************************************************************************
**

*F * * * * * * * * * * * * list access functions  * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*V  Elm0ListFuncs[ <type> ] . . . . . . . . . .  table of selection functions
**
**  'ELM0_LIST' returns the element at the position <pos> in the list <list>,
**  or 0 if <list>  has no assigned  object at position  <pos>.  An  error is
**  signalled if <list>  is  not a list.  It   is the responsibility   of the
**  caller to ensure that <pos> is a positive integer.
*/
libGAP_Obj (*libGAP_Elm0ListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list, libGAP_Int pos );


/****************************************************************************
**
*V  Elm0vListFuncs[ <type> ]  . . . . . . . . .  table of selection functions
**
**  'ELMV0_LIST' does the same as 'ELM0_LIST', but the caller also guarantees
**  that <list> is a list and that <pos> is less than  or equal to the length
**  of <list>.
*/
libGAP_Obj (*libGAP_Elm0vListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list, libGAP_Int pos );


/****************************************************************************
**
*F  Elm0ListError( <list>, <pos> )  . . . . . . . . . . . . . . error message
*/
libGAP_Obj libGAP_Elm0ListError (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    list = libGAP_ErrorReturnObj(
        "List Element: <list> must be a list (not a %s)",
        (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
        "you can replace <list> via 'return <list>;'" );
    return libGAP_ELM0_LIST( list, pos );
}


/****************************************************************************
**
*F  Elm0ListObject( <list>, <pos> ) . . . . . . select an element from a list
**
**  `Elm0ListObject'  is    the  `ELM0_LIST'  and  `ELMV0_LIST' function  for
**  objects.  The  function returns the element at  the position <pos> of the
**  list object <list>, or 0 if <list>  has no assigned object  at <pos>.  It
**  is the responsibility  of the caller to  ensure that <pos> is a  positive
**  integer.
**
**  Note that the method   returns `Fail' if there  is  no entry  at position
**  <pos>, in this case `Elm0ListObject' must  check if the position is bound
**  and `Fail'  means that there realy is  the object `Fail' at this position
**  or if it is unbound in which case 0 is returned.
*/
libGAP_Obj libGAP_Elm0ListOper;

libGAP_Obj libGAP_Elm0ListObject (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    libGAP_Obj                 elm;

    elm = libGAP_DoOperation2Args( libGAP_Elm0ListOper, list, libGAP_INTOBJ_INT(pos) );

    if ( elm == libGAP_Fail ) {
        if ( libGAP_DoOperation2Args(libGAP_IsbListOper,list,libGAP_INTOBJ_INT(pos)) == libGAP_True )
            return libGAP_Fail;
        else
            return 0;
    } else {
        return elm;
    }
}


/****************************************************************************
**
*F  FuncELM0_LIST( <self>, <list>, <pos> )  . . . . . . operation `ELM0_LIST'
*/
libGAP_Obj libGAP_FuncELM0_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 pos )
{
    libGAP_Obj                 elm;
    elm = libGAP_ELM0_LIST( list, libGAP_INT_INTOBJ(pos) );
    if ( elm == 0 ) {
        return libGAP_Fail;
    }
    else {
        return elm;
    }
}

/****************************************************************************
**
*V  ElmListFuncs[<type>]  . . . . . . . . . . .  table of selection functions
**
**  'ELM_LIST' returns the element at the position  <pos> in the list <list>.
**  An  error is signalled if  <list> is not a list,  if <pos> is larger than
**  the length of <list>, or if <list>  has no assigned  object at <pos>.  It
**  is the responsibility  of the caller to  ensure that <pos>  is a positive
**  integer.
**
**  'ELM_LIST' only calls the functions  pointed to by 'ElmListFuncs[<type>]'
**  passing <list> and <pos>  as arguments.  If  <type> is not  the type of a
**  list, then 'ElmListFuncs[<type>]' points to 'ElmListError', which signals
**  the error.
*/
libGAP_Obj (*libGAP_ElmListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list, libGAP_Int pos );

/****************************************************************************
**
*V  ElmbListFuncs[<type>]  . . . . . . . . . . .  table of selection functions
**
**  'ELMB_LIST' returns the element at the position  <pos> in the list <list>.
**  An  error is signalled if  <list> is not a list,  if <pos> is larger than
**  the length of <list>, or if <list>  has no assigned  object at <pos>.  It
**  is the responsibility  of the caller to  ensure that <pos>  is a positive
**  integer.
**
**  'ELMB_LIST' only calls the functions  pointed to by 'ElmbListFuncs[<type>]'
**  passing <list> and <pos>  as arguments.  If  <type> is not  the type of a
**  list, then 'ElmbListFuncs[<type>]' points to 'ElmbListError', which signals
**  the error.
*/
libGAP_Obj (*libGAP_ElmbListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list, libGAP_Obj pos );


/****************************************************************************
**
*V  ElmvListFuncs[<type>] . . . . . . . . . . .  table of selection functions
**
**  'ELMV_LIST' does  the same as 'ELM_LIST', but  the caller also guarantees
**  that <list> is a list and that <pos> is less  than or equal to the length
**  of <list>.
*/
libGAP_Obj (*libGAP_ElmvListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list, libGAP_Int pos );


/****************************************************************************
**
*V  ElmwListFuncs[<type>] . . . . . . . . . . .  table of selection functions
**
**  'ELMW_LIST' does the same as 'ELMV_LIST', but  the caller also guarantees
**  that <list> has an assigned object at the position <pos>.
*/
libGAP_Obj (*libGAP_ElmwListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list, libGAP_Int pos );


/****************************************************************************
**
*F  ElmListError( <list>, <pos> ) . . . . . . . . . . . . . . . error message
*/
libGAP_Obj libGAP_ElmListError (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    list = libGAP_ErrorReturnObj(
        "List Element: <list> must be a list (not a %s)",
        (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
        "you can replace <list> via 'return <list>;'" );
    return libGAP_ELM_LIST( list, pos );
}


/****************************************************************************
**
*F  ElmListObject( <list>, <pos>  . . . . . . . select an element from a list
**
**  `ElmListObject' is the `ELM_LIST',  `ELMV_LIST', and `ELMW_LIST' function
**  for objects.   'ElmListObjects' selects the  element at position <pos> of
**  list  object <list>.   It is the  responsibility  of the caller to ensure
**  that <pos> is a positive integer.  The methods have to signal an error if
**  <pos> is larger than the length of <list> or if the entry is not bound.
*/
libGAP_Obj libGAP_ElmListOper;

libGAP_Obj libGAP_ElmListObject (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    libGAP_Obj                 elm;

    elm = libGAP_DoOperation2Args( libGAP_ElmListOper, list, libGAP_INTOBJ_INT(pos) );
    while ( elm == 0 ) {
        elm = libGAP_ErrorReturnObj(
            "List access method must return a value", 0L, 0L,
            "you can supply a value <val> via 'return <val>;'" );
    }
    return elm;
}

#if 0
/****************************************************************************
**
*F  ElmbListError( <list>, <pos> ) . . . . . . . . . . . . . . . error message
*/
libGAP_Obj libGAP_ElmbListError (
    libGAP_Obj                 list,
    libGAP_Obj                 pos )
{
    list = libGAP_ErrorReturnObj(
        "List Element: <list> must be a list (not a %s)",
        (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
        "you can replace <list> via 'return <list>;'" );
    return libGAP_ELMB_LIST( list, pos );
}

/****************************************************************************
**
*F  ElmbListInternal( <list>, <pos> ) . . . . . . . . . . . . . error message
*/
libGAP_Obj libGAP_ElmbListInternal (
    libGAP_Obj                 list,
    libGAP_Obj                 pos )
{
  do {
    pos = libGAP_ErrorReturnObj(
        "List Element: an internal list cannot have an element in such a position",
        0L, 0L,
        "you can supply a new position <pos> via 'return <pos>;'" );
  } while (!libGAP_IS_INTOBJ(pos) || libGAP_INT_INTOBJ(pos) < 0);
  return libGAP_ELM_LIST( list, libGAP_INT_INTOBJ(pos) );
}


/****************************************************************************
**
*F  ElmbListObject( <list>, <pos>  . . . . . . . select an element from a list
**
**  `ElmbListObject' is the `ELMB_LIST',  function
**  for objects.   'ElmbListObjects' selects the  element at position <pos> of
**  list  object <list>.   It is the  responsibility  of the caller to ensure
**  that <pos> is a positive integer.  The methods have to signal an error if
**  <pos> is larger than the length of <list> or if the entry is not bound.
*/

libGAP_Obj libGAP_ElmbListObject (
    libGAP_Obj                 list,
    libGAP_Obj                 pos )
{
    libGAP_Obj                 elm;

    elm = libGAP_DoOperation2Args( libGAP_ElmListOper, list, pos );
    while ( elm == 0 ) {
        elm = libGAP_ErrorReturnObj(
            "List access method must return a value", 0L, 0L,
            "you can supply a value <val> via 'return <val>;'" );
    }
    return elm;
}

#endif

libGAP_Obj libGAP_ELMB_LIST(libGAP_Obj list, libGAP_Obj pos)     {
   libGAP_Obj                 elm;

    elm = libGAP_DoOperation2Args( libGAP_ElmListOper, list, pos );
    while ( elm == 0 ) {
        elm = libGAP_ErrorReturnObj(
            "List access method must return a value", 0L, 0L,
            "you can supply a value <val> via 'return <val>;'" );
    }
    return elm;
}


/****************************************************************************
**
*F  FuncELM_LIST( <self>, <list>, <pos> ) . . . . . . .  operation `ELM_LIST'
*/
libGAP_Obj libGAP_FuncELM_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 pos )
{
  if (libGAP_IS_INTOBJ(pos))
    return libGAP_ELM_LIST( list, libGAP_INT_INTOBJ(pos) );
  else
    return libGAP_ELMB_LIST(list, pos );
}


/****************************************************************************
**

*V  ElmsListFuncs[<type>] . . . . . . . . . . .  table of selection functions
**
**  'ELMS_LIST' returns a  new list containing the  elements at the positions
**  given in the list  <poss> from the <list>.  It  is the responsibility  of
**  the caller  to ensure that  <poss>  is dense and  contains only  positive
**  integers.  An error  is signalled if an element  of <poss> is larger than
**  the length of <list>.
**
**  'ELMS_LIST'    only    calls    the     function   pointed     to      by
**  'ElmsListFuncs[<type>]',  passing  <list> and  <poss>   as arguments.  If
**  <type> is not the type of  a list, then 'ElmsListFuncs[<type>]' points to
**  'ElmsListError', which just signals an error.
*/
libGAP_Obj (*libGAP_ElmsListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list, libGAP_Obj poss );


/****************************************************************************
**
*F  ElmsListError(<list>,<poss>)  . . . . . . . . .  error selection function
*/
libGAP_Obj libGAP_ElmsListError (
    libGAP_Obj                 list,
    libGAP_Obj                 poss )
{
    list = libGAP_ErrorReturnObj(
        "List Elements: <list> must be a list (not a %s)",
        (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
        "you can replace <list> via 'return <list>;'" );
    return libGAP_ELMS_LIST( list, poss );
}


/****************************************************************************
**
*F  ElmsListObject( <list>, <pos> ) . . . . . . . select elements from a list
**
**  `ElmsListObject' is the `ELMS_LIST' function for objects.
*/
libGAP_Obj libGAP_ElmsListOper;

libGAP_Obj libGAP_ElmsListObject (
    libGAP_Obj                 list,
    libGAP_Obj                 poss )
{
    libGAP_Obj                 elm;

    elm = libGAP_DoOperation2Args( libGAP_ElmsListOper, list, poss );
    while ( elm == 0 ) {
        elm = libGAP_ErrorReturnObj(
            "List multi-access method must return a value", 0L, 0L,
            "you can supply a value <val> via 'return <val>;'");
    }
    return elm;
}


/****************************************************************************
**
*F  FuncELMS_LIST( <self>, <list>, <poss> ) . . . . . . `ELMS_LIST' operation
*/
libGAP_Obj libGAP_FuncELMS_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 poss )
{
    return libGAP_ELMS_LIST( list, poss );
}


/****************************************************************************
**
*F  ElmsListDefault( <list>, <poss> ) . . .  default function for `ELMS_LIST'
**
**  Create a new plain list as result. <list> must be small.
*/
libGAP_Obj libGAP_ElmsListDefault (
    libGAP_Obj                 list,
    libGAP_Obj                 poss )
{
    libGAP_Obj                 elms;           /* selected sublist, result        */
    libGAP_Int                 lenList;        /* length of <list>                */
    libGAP_Obj                 elm;            /* one element from <list>         */
    libGAP_Int                 lenPoss;        /* length of <positions>           */
    libGAP_Int                 pos;            /* <position> as integer           */
    libGAP_Int                 inc;            /* increment in a range            */
    libGAP_Int                 i;              /* loop variable                   */
    libGAP_Obj                 p;

    /* general code                                                        */
    if ( ! libGAP_IS_RANGE(poss) ) {

        /* get the length of <list>                                        */
        lenList = libGAP_LEN_LIST( list );

        /* get the length of <positions>                                   */
        /* OK because all positions lists are small                        */
        lenPoss = libGAP_LEN_LIST( poss );

        /* make the result list                                            */
        elms = libGAP_NEW_PLIST( libGAP_T_PLIST, lenPoss );
        libGAP_SET_LEN_PLIST( elms, lenPoss );

        /* loop over the entries of <positions> and select                 */
        for ( i = 1; i <= lenPoss; i++ ) {

            /* get <position>                                              */
          p = libGAP_ELMW_LIST( poss, i);
          while (!libGAP_IS_INTOBJ(p))
            {
              p = libGAP_ErrorReturnObj("List Elements: position is too large for this type of list",
                                 0L, 0L, 
                                 "you can supply a new position <pos> via 'return <pos>;'" );
            }
            pos = libGAP_INT_INTOBJ( p );

            /* select the element                                          */
            elm = libGAP_ELM0_LIST( list, pos );
            if ( elm == 0 ) {
                libGAP_ErrorReturnVoid(
                    "List Elements: <list>[%d] must have an assigned value",
                    (libGAP_Int)pos, 0L,
                    "you can 'return;' after assigning a value" );
                return libGAP_ELMS_LIST( list, poss );
            }

            /* assign the element into <elms>                              */
            libGAP_SET_ELM_PLIST( elms, i, elm );

            /* notify Gasman                                               */
            libGAP_CHANGED_BAG( elms );

        }

    }

    /* special code for ranges                                             */
    else {

        /* get the length of <list>                                        */
        lenList = libGAP_LEN_LIST( list );

        /* get the length of <positions>, the first elements, and the inc. */
        lenPoss = libGAP_GET_LEN_RANGE( poss );
        pos = libGAP_GET_LOW_RANGE( poss );
        inc = libGAP_GET_INC_RANGE( poss );

        /* check that no <position> is larger than 'LEN_LIST(<list>)'      */
        if ( lenList < pos ) {
            libGAP_ErrorReturnVoid(
                "List Elements: <list>[%d] must have an assigned value",
                (libGAP_Int)pos, 0L,
                "you can 'return;' after assigning a value" );
            return libGAP_ELMS_LIST( list, poss );
        }
        if ( lenList < pos + (lenPoss-1) * inc ) {
            libGAP_ErrorReturnVoid(
                "List Elements: <list>[%d] must have an assigned value",
                (libGAP_Int)pos + (lenPoss-1) * inc, 0L,
                "you can 'return;' after assigning a value" );
            return libGAP_ELMS_LIST( list, poss );
        }

        /* make the result list                                            */
        elms = libGAP_NEW_PLIST( libGAP_T_PLIST, lenPoss );
        libGAP_SET_LEN_PLIST( elms, lenPoss );

        /* loop over the entries of <positions> and select                 */
        for ( i = 1; i <= lenPoss; i++, pos += inc ) {

            /* select the element                                          */
            elm = libGAP_ELMV0_LIST( list, pos );
            if ( elm == 0 ) {
                libGAP_ErrorReturnVoid(
                    "List Elements: <list>[%d] must have an assigned value",
                    (libGAP_Int)pos, 0L,
                    "you can 'return;' after assigning a value" );
                return libGAP_ELMS_LIST( list, poss );
            }

            /* assign the element to <elms>                                */
            libGAP_SET_ELM_PLIST( elms, i, elm );

            /* notify Gasman                                               */
            libGAP_CHANGED_BAG( elms );

        }

    }

    /* return the result                                                   */
    return elms;
}


/****************************************************************************
**
*F  FuncELMS_LIST_DEFAULT( <self>, <list>, <poss> ) . . . . `ElmsListDefault'
*/
libGAP_Obj libGAP_FuncELMS_LIST_DEFAULT (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 poss )
{
    return libGAP_ElmsListDefault( list, poss );
}


/****************************************************************************
**
*F  ElmsListCheck( <list>, <poss> ) . . . . . . . . . . . . . . . . ELMS_LIST
**
**  `ElmsListCheck' checks that <poss> is  a possitions lists before  calling
**  `ELMS_LIST'.
*/
libGAP_Obj libGAP_ElmsListCheck (
    libGAP_Obj                 list,
    libGAP_Obj                 poss )
{
    if ( ! libGAP_IS_POSS_LIST(poss) ) {
        libGAP_ErrorQuit(
      "List Elements: <positions> must be a dense list of positive integers",
            0L, 0L );
    }
    return libGAP_ELMS_LIST( list, poss );
}


/****************************************************************************
**
*F  ElmsListLevelCheck( <lists>, <poss>, <level> )  . . . . . . ElmsListLevel
**
**  `ElmsListLevelCheck'   checks that  <poss> is  a  possitions lists before
**  calling `ElmsListLevel'.
*/
void libGAP_ElmsListLevelCheck (
    libGAP_Obj                 lists,
    libGAP_Obj                 poss,
    libGAP_Int                 level )
{
    if ( ! libGAP_IS_POSS_LIST(poss) ) {
        libGAP_ErrorQuit(
      "List Elements: <positions> must be a dense list of positive integers",
            0L, 0L );
    }
    libGAP_ElmsListLevel( lists, poss, level );
}


/****************************************************************************
**

*F  UNB_LIST(<list>,<pos>)  . . . . . . . . . . .  unbind element from a list
*V  UnbListFuncs[<type>]  . . . . . . . . . . . . . table of unbind functions
*F  UnbListError(<list>,<pos>)  . . . . . . . . . . . . error unbind function
**
#define UNB_LIST(list,pos) \
                        ((*UnbListFuncs[TNUM_OBJ(list)])(list,pos))
*/
void             (*libGAP_UnbListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list, libGAP_Int pos );

libGAP_Obj             libGAP_UnbListOper;

libGAP_Obj             libGAP_FuncUNB_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 pos )
{
    libGAP_UNB_LIST( list, libGAP_INT_INTOBJ(pos) );
    return 0;
}

void            libGAP_UnbListError (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    list = libGAP_ErrorReturnObj(
        "Unbind: <list> must be a list (not a %s)",
        (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
        "you can replace <list> via 'return <list>;'" );
    libGAP_UNB_LIST( list, pos );
}

void            libGAP_UnbListDefault (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    libGAP_PLAIN_LIST( list );
    libGAP_UNB_LIST( list, pos );
}

void            libGAP_UnbListObject (
    libGAP_Obj                 list,
    libGAP_Int                 pos )
{
    libGAP_DoOperation2Args( libGAP_UnbListOper, list, libGAP_INTOBJ_INT(pos) );
}

/****************************************************************************
**
*F  UNBB_LIST(<list>,<pos>,<obj>)  . . . . . . . . unbind an element to a list
*V  UnbbListFuncs[<type>]  . . . . . . . . . . .  table of unbinding functions
*F  UnbbListError(<list>,<pos>,<obj>)  . . . . . . . .error unbinding function
**
**  'UNBB_LIST' only calls the  function pointed to by 'UnbbListFuncs[<type>]',
**  passing <list>, <pos>, and <obj> as arguments.  If <type> is not the type
**  of  a list, then 'UnbbListFuncs[<type>]'  points to 'UnbbListError',  which
**  just signals an error.
**
**  'UNBB_LIST' is defined in the declaration part of this package as follows.
**
#define UNBB_LIST(list,pos,obj) \
                        ((*UnbbListFuncs[TNUM_OBJ(list)])(list,pos,obj))
*/
void            (*libGAP_UnbbListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list, libGAP_Obj pos );

libGAP_Obj             libGAP_FuncUNBB_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 pos )
{
    libGAP_UNBB_LIST( list, pos );
    return 0;
}

void            libGAP_UnbbListError (
    libGAP_Obj                 list,
    libGAP_Obj                 pos )
{
    list = libGAP_ErrorReturnObj(
        "List Unbindment: <list> must be a list (not a %s)",
        (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
        "you can replace <list> via 'return <list>;'" );
    libGAP_UNBB_LIST( list, pos );
}

void            libGAP_UnbbListInternal (
    libGAP_Obj                 list,
    libGAP_Obj                 pos)
{
  /* large positions are already unbound */
  return;
}


/****************************************************************************
**
*F  UnbbListObject( <list>, <pos>, <obj> ) . . . . . . . unbind  list object
*/

void libGAP_UnbbListObject (
    libGAP_Obj                 list,
    libGAP_Obj                 pos )
{
    libGAP_DoOperation2Args( libGAP_UnbListOper, list, pos );
}

/****************************************************************************
**
*F  ASS_LIST(<list>,<pos>,<obj>)  . . . . . . . . assign an element to a list
*V  AssListFuncs[<type>]  . . . . . . . . . . . table of assignment functions
*F  AssListError(<list>,<pos>,<obj>)  . . . . . . . error assignment function
**
**  'ASS_LIST' only calls the  function pointed to by 'AssListFuncs[<type>]',
**  passing <list>, <pos>, and <obj> as arguments.  If <type> is not the type
**  of  a list, then 'AssListFuncs[<type>]'  points to 'AssListError',  which
**  just signals an error.
**
**  'ASS_LIST' is defined in the declaration part of this package as follows.
**
#define ASS_LIST(list,pos,obj) \
                        ((*AssListFuncs[TNUM_OBJ(list)])(list,pos,obj))
*/
void            (*libGAP_AssListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list, libGAP_Int pos, libGAP_Obj obj );

libGAP_Obj             libGAP_FuncASS_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 pos,
    libGAP_Obj                 obj )
{
    libGAP_ASS_LIST( list, libGAP_INT_INTOBJ(pos), obj );
    return 0;
}

void            libGAP_AssListError (
    libGAP_Obj                 list,
    libGAP_Int                 pos,
    libGAP_Obj                 obj )
{
    list = libGAP_ErrorReturnObj(
        "List Assignment: <list> must be a list (not a %s)",
        (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
        "you can replace <list> via 'return <list>;'" );
    libGAP_ASS_LIST( list, pos, obj );
}

void            libGAP_AssListDefault (
    libGAP_Obj                 list,
    libGAP_Int                 pos,
    libGAP_Obj                 obj )
{
    libGAP_PLAIN_LIST( list );
    libGAP_ASS_LIST( list, pos, obj );
}


/****************************************************************************
**
*F  AssListObject( <list>, <pos>, <obj> ) . . . . . . . assign to list object
*/
libGAP_Obj libGAP_AssListOper;

void libGAP_AssListObject (
    libGAP_Obj                 list,
    libGAP_Int                 pos,
    libGAP_Obj                 obj )
{
    libGAP_DoOperation3Args( libGAP_AssListOper, list, libGAP_INTOBJ_INT(pos), obj );
}
/****************************************************************************
**
*F  ASSB_LIST(<list>,<pos>,<obj>)  . . . . . . . . assign an element to a list
*V  AssbListFuncs[<type>]  . . . . . . . . . . . table of assignment functions
*F  AssbListError(<list>,<pos>,<obj>)  . . . . . . . error assignment function
**
**  'ASSB_LIST' only calls the  function pointed to by 'AssbListFuncs[<type>]',
**  passing <list>, <pos>, and <obj> as arguments.  If <type> is not the type
**  of  a list, then 'AssbListFuncs[<type>]'  points to 'AssbListError',  which
**  just signals an error.
**
**  'ASSB_LIST' is defined in the declaration part of this package as follows.
**
#define ASSB_LIST(list,pos,obj) \
                        ((*AssbListFuncs[TNUM_OBJ(list)])(list,pos,obj))
*/
void            (*libGAP_AssbListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list, libGAP_Obj pos, libGAP_Obj obj );

libGAP_Obj             libGAP_FuncASSB_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 pos,
    libGAP_Obj                 obj )
{
    libGAP_ASSB_LIST( list, pos, obj );
    return 0;
}

void            libGAP_AssbListError (
    libGAP_Obj                 list,
    libGAP_Obj                 pos,
    libGAP_Obj                 obj )
{
    list = libGAP_ErrorReturnObj(
        "List Assignment: <list> must be a list (not a %s)",
        (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
        "you can replace <list> via 'return <list>;'" );
    libGAP_ASSB_LIST( list, pos, obj );
}

void            libGAP_AssbListInternal (
    libGAP_Obj                 list,
    libGAP_Obj                 pos,
    libGAP_Obj                 obj )
{
  do {
    pos = libGAP_ErrorReturnObj( "List assignment: you cannot assign to such a large position in an internal list",
                          0, 0, 
                          "you can supply a new position <pos> via 'return <pos>;'" );
  } while (!libGAP_IS_INTOBJ(pos) || libGAP_INT_INTOBJ(pos) <= 0);
  libGAP_ASS_LIST(list, libGAP_INT_INTOBJ(pos), obj);
}


/****************************************************************************
**
*F  AssbListObject( <list>, <pos>, <obj> ) . . . . . . . assign to list object
*/

void libGAP_AssbListObject (
    libGAP_Obj                 list,
    libGAP_Obj                 pos,
    libGAP_Obj                 obj )
{
    libGAP_DoOperation3Args( libGAP_AssListOper, list, pos, obj );
}


/****************************************************************************
**
*F  ASSS_LIST(<list>,<poss>,<objs>) . . . . assign several elements to a list
*V  AsssListFuncs[<type>] . . . . . . . . . . .  table of assignment function
*F  AsssListError(<list>,<poss>,<objs>) . . . . . . error assignment function
**
**  'ASSS_LIST'    only      calls      the   function pointed      to     by
**  'AsssListFuncs[<type>]', passing <list>, <poss>, and <objs> as arguments.
**  If <type> is not the type of  a list, then 'AsssListFuncs[<type>]' points
**  to 'AsssListError', which just signals an error.
**
**  'ASSS_LIST'  is  defined in the  declaration  part  of  this  package  as
**  follows
**
#define ASSS_LIST(list,poss,objs) \
                        ((*AsssListFuncs[TNUM_OBJ(list)])(list,poss,objs))
*/
void            (*libGAP_AsssListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list, libGAP_Obj poss, libGAP_Obj objs );

libGAP_Obj             libGAP_AsssListOper;

libGAP_Obj             libGAP_FuncASSS_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 poss,
    libGAP_Obj                 objs )
{
    libGAP_ASSS_LIST( list, poss, objs );
    return 0;
}

void            libGAP_AsssListError (
    libGAP_Obj                 list,
    libGAP_Obj                 poss,
    libGAP_Obj                 objs )
{
    list = libGAP_ErrorReturnObj(
        "List Assignments: <list> must be a list (not a %s)",
        (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
        "you can replace <list> via 'return <list>;'" );
    libGAP_ASSS_LIST( list, poss, objs );
}

void            libGAP_AsssListDefault (
    libGAP_Obj                 list,
    libGAP_Obj                 poss,
    libGAP_Obj                 objs )
{
    libGAP_Int                 lenPoss;        /* length of <positions>           */
    libGAP_Obj                 p;              /* <position> */
    libGAP_Int                 pos;            /* <position> as integer           */
    libGAP_Int                 inc;            /* increment in a range            */
    libGAP_Obj                 obj;            /* one element from <objs>         */
    libGAP_Int                 i;              /* loop variable                   */

    /* general code                                                        */
    if ( ! libGAP_IS_RANGE(poss) ) {

        /* get the length of <positions>                                   */
        lenPoss = libGAP_LEN_LIST( poss );

        /* loop over the entries of <positions> and select                 */
        for ( i = 1; i <= lenPoss; i++ ) {

            /* get <position>                                              */
          p  = libGAP_ELMW_LIST( poss, i );
          
          /* select the element                                          */
          obj = libGAP_ELMW_LIST( objs, i );
          if (libGAP_IS_INTOBJ(p) )
            {
              /* assign the element into <elms>                              */
              libGAP_ASS_LIST( list, libGAP_INT_INTOBJ(p), obj );
            }
          else
            libGAP_ASSB_LIST(list, p, obj);

        }

    }

    /* special code for ranges                                             */
    else {

        /* get the length of <positions>                                   */
        lenPoss = libGAP_GET_LEN_RANGE( poss );
        pos = libGAP_GET_LOW_RANGE( poss );
        inc = libGAP_GET_INC_RANGE( poss );

        /* loop over the entries of <positions> and select                 */
        for ( i = 1; i <= lenPoss; i++, pos += inc ) {

            /* select the element                                          */
            obj = libGAP_ELMW_LIST( objs, i );

            /* assign the element to <elms>                                */
            libGAP_ASS_LIST( list, pos, obj );

        }

    }

}

void            libGAP_AsssListObject (
    libGAP_Obj                 list,
    libGAP_Obj                 poss,
    libGAP_Obj                 objs )
{
    libGAP_DoOperation3Args( libGAP_AsssListOper, list, poss, objs );
}

libGAP_Obj libGAP_FuncASSS_LIST_DEFAULT (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 poss,
    libGAP_Obj                 objs )
{
    libGAP_AsssListDefault( list, poss, objs );
    return 0;
}


/****************************************************************************
**
*F  IS_DENSE_LIST(<list>) . . . . . . . . . . . . . . .  test for dense lists
*V  IsDenseListFuncs[<type>]  . . . . . . table for dense list test functions
*F  IsDenseListNot(<list>)  . . . . .  dense list test function for non lists
**
**  'IS_DENSE_LIST'  only     calls   the      function   pointed    to    by
**  'IsDenseListFuncs[<type>]', passing <list> as argument.  If <type> is not
**  the   type  of  a    list,  then  'IsDenseListFuncs[<type>]'  points   to
**  'IsDenseListNot', which just returns 0.
**
**  'IS_DENSE_LIST'  is defined in the declaration  part  of this  package as
**  follows
**
#define IS_DENSE_LIST(list) \
                        ((*IsDenseListFuncs[TNUM_OBJ(list)])(list))
*/
libGAP_Int             (*libGAP_IsDenseListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list );

libGAP_Obj             libGAP_IsDenseListFilt;

libGAP_Obj             libGAP_FuncIS_DENSE_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return (libGAP_IS_DENSE_LIST( obj ) ? libGAP_True : libGAP_False);
}

libGAP_Int             libGAP_IsDenseListNot (
    libGAP_Obj                 list )
{
    return 0L;
}

libGAP_Int             libGAP_IsDenseListYes (
    libGAP_Obj                 list )
{
    return 1L;
}

libGAP_Int             libGAP_IsDenseListDefault (
    libGAP_Obj                 list )
{
    libGAP_Int                 lenList;        /* length of <list>                */
    libGAP_Int                 i;              /* loop variable                   */

    /* get the length of the list                                          */
    lenList = libGAP_LEN_LIST( list );

    /* special case for the empty list                                     */
    if ( lenList == 0 ) {
        return 1L;
    }

    /* loop over the entries of the list                                   */
    for ( i = 1; i <= lenList; i++ ) {
        if ( ! libGAP_ISB_LIST( list, i ) ) {
            return 0L;
        }
    }

    /* the list is dense                                                   */
    return 1L;
}

libGAP_Int             libGAP_IsDenseListObject (
    libGAP_Obj                 obj )
{
    return (libGAP_DoFilter( libGAP_IsDenseListFilt, obj ) == libGAP_True);
}


/****************************************************************************
**
*F  IS_HOMOG_LIST(<list>) . . . . . . . . . . . .  test for homogeneous lists
*V  IsHomogListFuncs[<type>]  . . . table for homogeneous list test functions
*F  IsHomogListNot(<list>)  . .  homogeneous list test function for non lists
**
**  'IS_HOMOG_LIST' only calls the function pointed to by
**  'IsHomogListFuncs[<type>]', passing <list> as argument.  If <type> is not
**  the type of a list, then 'IsHomogListFuncs[<type>]' points to
**  'IsHomogListNot', which just returns 0.
**
**  'IS_HOMOG_LIST' is defined in the declaration part  of  this  package  as
**  follows
**
#define IS_HOMOG_LIST(list) \
                        ((*IsHomogListFuncs[TNUM_OBJ(list)])(list))
*/
libGAP_Int             (*libGAP_IsHomogListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list );

libGAP_Obj             libGAP_IsHomogListFilt;

libGAP_Obj             libGAP_FuncIS_HOMOG_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return (libGAP_IS_HOMOG_LIST( obj ) ? libGAP_True : libGAP_False);
}

libGAP_Int             libGAP_IsHomogListNot (
    libGAP_Obj                 list )
{
    return 0L;
}

libGAP_Int             libGAP_IsHomogListYes (
    libGAP_Obj                 list )
{
    return 1L;
}

libGAP_Int             libGAP_IsHomogListDefault (
    libGAP_Obj                 list )
{
    libGAP_Int                 lenList;        /* length of <list>                */
    libGAP_Obj                 elm;            /* one element of <list>           */
    libGAP_Obj                 fam;            /* family of elements of <list>    */
    libGAP_Int                 i;              /* loop variable                   */

    /* get the length of the list                                          */
    lenList = libGAP_LEN_LIST( list );

    /* special case for the empty list                                     */
    if ( lenList == 0 ) {
        return 0L;
    }

    /* get the family                                                      */
    elm = libGAP_ELMV0_LIST( list, 1 );
    if ( elm == 0 ) {
        return 0L;
    }
    fam = libGAP_FAMILY_TYPE( libGAP_TYPE_OBJ( elm ) );

    /* loop over the entries of the list                                   */
    for ( i = 2; i <= lenList; i++ ) {
        elm = libGAP_ELMV0_LIST( list, i );
        if ( elm == 0 || fam != libGAP_FAMILY_TYPE( libGAP_TYPE_OBJ( elm ) ) ) {
            return 0L;
        }
    }

    /* the list is homogeneous                                             */
    return 1L;
}

libGAP_Int             libGAP_IsHomogListObject (
    libGAP_Obj                 obj )
{
    return (libGAP_DoFilter( libGAP_IsHomogListFilt, obj ) == libGAP_True);
}


/****************************************************************************
**
*F  IS_TABLE_LIST(<list>) . . . . . . . . . . . . . . .  test for table lists
*V  IsTableListFuncs[<type>]  . . . . . . table for table list test functions
*F  IsTableListNot(<list>)  . . . . .  table list test function for non lists
**
**  'IS_TABLE_LIST' only calls the function pointed to by
**  'IsTableListFuncs[<type>]', passing <list> as argument.  If <type> is not
**  the type of a list, then 'IsTableListFuncs[<type>]' points to
**  'IsTableListNot', which just returns 0.
**
**  'IS_TABLE_LIST' is defined in the declaration part  of  this  package  as
**  follows
**
#define IS_TABLE_LIST(list) \
                        ((*IsTableListFuncs[TNUM_OBJ(list)])(list))
*/
libGAP_Int             (*libGAP_IsTableListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list );

libGAP_Obj             libGAP_IsTableListFilt;

libGAP_Obj             libGAP_Func_IS_TABLE_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return (libGAP_IS_TABLE_LIST( obj ) ? libGAP_True : libGAP_False);
}

libGAP_Int             libGAP_IsTableListNot (
    libGAP_Obj                 list )
{
    return 0L;
}

libGAP_Int             libGAP_IsTableListYes (
    libGAP_Obj                 list )
{
    return 1L;
}

libGAP_Int             libGAP_IsTableListDefault (
    libGAP_Obj                 list )
{
    libGAP_Int                 lenList;        /* length of <list>                */
    libGAP_Obj                 elm;            /* one element of <list>           */
    libGAP_Obj                 fam;            /* family of elements of <list>    */
/*  Int                 len;            / length of elements              */
    libGAP_Int                 i;              /* loop variable                   */

    /* get the length of the list                                          */
    lenList = libGAP_LEN_LIST( list );

    /* special case for the empty list                                     */
    if ( lenList == 0 ) {
        return 0L;
    }

    /* get the family                                                      */
    elm = libGAP_ELMV0_LIST( list, 1 );
    if ( elm == 0 ) {
        return 0L;
    }
    if ( ! libGAP_IS_HOMOG_LIST( elm ) ) {
        return 0L;
    }
    fam = libGAP_FAMILY_TYPE( libGAP_TYPE_OBJ( elm ) );
    /*     len = LEN_LIST( elm ); */

    /* loop over the entries of the list                                   */
    for ( i = 2; i <= lenList; i++ ) {
        elm = libGAP_ELMV0_LIST( list, i );
        if ( elm == 0 || fam != libGAP_FAMILY_TYPE( libGAP_TYPE_OBJ( elm ) ) ) {
            return 0L;
        }
        /*        if ( ! IS_LIST( elm ) || LEN_LIST( elm ) != len ) {
            return 0L;
            } */
    }

    /* the list is equal length                                            */
    return 1L;
}

libGAP_Int             libGAP_IsTableListObject (
    libGAP_Obj                 obj )
{
    return (libGAP_DoFilter( libGAP_IsTableListFilt, obj ) == libGAP_True);
}


/****************************************************************************
**
*F  IS_SSORT_LIST( <list> ) . . . . . . . . .  test for strictly sorted lists
*V  IsSSortListFuncs[<type>]  .  table of strictly sorted list test functions
*F  IsSSortListNot( <list> ) strictly sorted list test function for non lists
**
**  'IS_SSORT_LIST' only calls the function pointed to by
**  'IsSSortListFuncs[<type>]', passing <list> as argument.
**  If <type> is not the type of a list, then 'IsSSortListFuncs[<type>]'
**  points to 'IsSSortListNot', which just returns 0.
**
**  'IS_SSORTED_LIST'  is defined in the  declaration part of this package as
**  follows
**
#define IS_SSORTED_LIST(list) \
                        ((*IsSSortListFuncs[TNUM_OBJ(list)])(list))
*/
libGAP_Int (*libGAP_IsSSortListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list );

libGAP_Obj libGAP_IsSSortListProp;

libGAP_Obj libGAP_FuncIS_SSORT_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return (libGAP_IS_SSORT_LIST( obj ) ? libGAP_True : libGAP_False);
}

libGAP_Int libGAP_IsSSortListNot (
    libGAP_Obj                 list )
{
    return 0L;
}

libGAP_Int libGAP_IsSSortListYes (
    libGAP_Obj                 list )
{
    return 1L;
}

libGAP_Int libGAP_IsSSortListDefault (
    libGAP_Obj                 list )
{
    libGAP_Int                 lenList;
    libGAP_Obj                 elm1;
    libGAP_Obj                 elm2;
    libGAP_Int                 i;

    /* get the length of the list                                          */
    lenList = libGAP_LEN_LIST( list );

    /* special case for the empty list                                     */
    if ( lenList == 0 ) {
        return 2L;
    }

    /* a list must be homogeneous to be strictly sorted                    */
    if ( ! libGAP_IS_HOMOG_LIST(list) ) {
        return 0L;
    }

    /* get the first element                                               */
    elm1 = libGAP_ELMW_LIST( list, 1 );

    /* compare each element with its precursor                             */
    for ( i = 2; i <= lenList; i++ ) {
        elm2 = libGAP_ELMW_LIST( list, i );
        if ( ! libGAP_LT( elm1, elm2 ) ) {
            return 0L;
        }
        elm1 = elm2;
    }

    /* the list is strictly sorted                                         */
    return 2L;
}

libGAP_Int             libGAP_IsSSortListObject (
    libGAP_Obj                 obj )
{
    return (libGAP_DoProperty( libGAP_IsSSortListProp, obj ) == libGAP_True);
}

libGAP_Obj libGAP_FuncIS_SSORT_LIST_DEFAULT (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return (libGAP_IsSSortListDefault( obj ) ? libGAP_True : libGAP_False);
}


/****************************************************************************
**
*F  IsNSortListProp( <list> ) . . . . . . . . . . . . list which are unsorted
**
*/
libGAP_Obj libGAP_IsNSortListProp;

libGAP_Obj libGAP_FuncIS_NSORT_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    libGAP_ErrorQuit( "not ready yet", 0L, 0L );
    return (libGAP_Obj)0L;
}


/****************************************************************************
**
*F  IS_POSS_LIST(<list>)  . . . . . . . . . . . . .  test for positions lists
*V  IsPossListFuncs[<type>] . . . . . . table of positions list test function
*F  IsPossListNot(<list>) . . . .  positions list test function for non lists
**
**  'IS_POSS_LIST'     only   calls    the     function  pointed      to   by
**  'IsPossListFuncs[<type>]', passing <list> as  argument.  If <type> is not
**  the   type    of a   list,    then  'IsPossListFuncs[<type>]'   points to
**  'NotIsPossList', which just returns 0.
**
**  'IS_POSS_LIST' is  defined  in the  declaration  part of this  package as
**  follows
**
#define IS_POSS_LIST(list) \
                        ((*IsPossListFuncs[TNUM_OBJ(list)])(list))
*/
libGAP_Int             (*libGAP_IsPossListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list );

libGAP_Obj             libGAP_IsPossListProp;

libGAP_Obj             libGAP_FuncIS_POSS_LIST (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return (libGAP_IS_POSS_LIST(obj) ? libGAP_True : libGAP_False);
}

libGAP_Int             libGAP_IsPossListNot (
    libGAP_Obj                 list )
{
    return 0L;
}

libGAP_Int             libGAP_IsPossListYes (
    libGAP_Obj                 list )
{
    return 1L;
}

libGAP_Int             libGAP_IsPossListDefault (
    libGAP_Obj                 list )
{
    libGAP_Int                 lenList;        /* length of <list>                */
    libGAP_Obj                 elm;            /* one element of <list>           */
    libGAP_Int                 i;              /* loop variable                   */

    /* get the length of the variable                                      */
    lenList = libGAP_LEN_LIST( list );

    /* loop over the entries of the list                                   */
    for ( i = 1; i <= lenList; i++ ) {
        elm = libGAP_ELMV0_LIST( list, i );

        /* if it's a hole then its not a poss list */
        if ( elm == 0)
          return 0L;

        /* if it's a small integer and non-positive then
           it's not a poss list */
        if ( libGAP_IS_INTOBJ(elm)) {
          if (libGAP_INT_INTOBJ(elm) <= 0)
            return  0L;
        }
        /* or if it's not a small integer or a positive large integer then it's
           not a poss list */
        else if (libGAP_TNUM_OBJ(elm) != libGAP_T_INTPOS)
          return 0L;
    }

    /* the list is a positions list                                        */
    return 1L;
}

libGAP_Int             libGAP_IsPossListObject (
    libGAP_Obj                 obj )
{
    return (libGAP_DoProperty( libGAP_IsPossListProp, obj ) == libGAP_True);
}

libGAP_Obj libGAP_FuncIS_POSS_LIST_DEFAULT (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return (libGAP_IsPossListDefault( obj ) ? libGAP_True : libGAP_False);
}


/****************************************************************************
**
*F  POS_LIST(<list>,<obj>,<start>)  . . . . . . . . find an element in a list
*V  PosListFuncs[<type>]  . . . . . . . . . . .  table of searching functions
*F  PosListError(<list>,<obj>,<start>)  . . . . . .  error searching function
**
**  'POS_LIST' only calls  the function pointed to by 'PosListFuncs[<type>]',
**  passing <list>, <obj>,  and <start> as arguments.  If  <type>  is not the
**  type  of  a list, then  'PosListFuncs[<type>]'  points to 'PosListError',
**  which just signals an error.
**
**  'POS_LIST' is defined in the declaration part of this package as follows
**
#define POS_LIST(list,obj,start) \
                        ((*PosListFuncs[TNUM_OBJ(list)])(list,obj,start))
*/
libGAP_Obj             (*libGAP_PosListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list, libGAP_Obj obj, libGAP_Obj start );

libGAP_Obj             libGAP_PosListOper;

libGAP_Obj             libGAP_PosListHandler2 (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 obj )
{
    return libGAP_POS_LIST( list, obj, libGAP_INTOBJ_INT(0) );
}

libGAP_Obj             libGAP_PosListHandler3 (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 obj,
    libGAP_Obj                 start )
{
    while ( libGAP_TNUM_OBJ(start) != libGAP_T_INTPOS &&
            (! libGAP_IS_INTOBJ(start) || libGAP_INT_INTOBJ(start) < 0) ) {
        start = libGAP_ErrorReturnObj(
            "Position: <start> must be a nonnegative integer (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(start), 0L,
            "you can replace <start> via 'return <start>;'" );
    }
    return libGAP_POS_LIST( list, obj, start );
}

libGAP_Obj             libGAP_PosListError (
    libGAP_Obj                 list,
    libGAP_Obj                 obj,
    libGAP_Obj                 start )
{
    list = libGAP_ErrorReturnObj(
        "Position: <list> must be a list (not a %s)",
        (libGAP_Int)libGAP_TNAM_OBJ(list), 0L,
        "you can replace <list> via 'return <list>;'" );
    return libGAP_POS_LIST( list, obj, start );
}

libGAP_Obj             libGAP_PosListDefault (
    libGAP_Obj                 list,
    libGAP_Obj                 obj,
    libGAP_Obj                 start )
{
    libGAP_Int                 lenList;
    libGAP_Obj                 elm;
    libGAP_Int                 i;

    /* if the starting position is too big to be a small int
       then there can't be anything to find */
    if (!libGAP_IS_INTOBJ(start))
      return libGAP_Fail;

    /* get the length of the list                                          */
    lenList = libGAP_LEN_LIST( list );

    /* loop over all bound entries of the list, and compare against <obj>  */
    for ( i = libGAP_INT_INTOBJ(start)+1; i <= lenList; i++ ) {
        elm = libGAP_ELMV0_LIST( list, i );
        if ( elm != 0 && libGAP_EQ( elm, obj ) ) {
            break;
        }
    }

    /* return the position if found, and 0 otherwise                       */
    if ( i <= lenList ) {
      return libGAP_INTOBJ_INT(i);
    }
    else {
      return libGAP_Fail;
    }
}

libGAP_Obj             libGAP_PosListObject (
    libGAP_Obj                 list,
    libGAP_Obj                 obj,
    libGAP_Obj                 start )
{
    return libGAP_DoOperation3Args( libGAP_PosListOper, list, obj, start );
}

libGAP_Obj libGAP_FuncPOS_LIST_DEFAULT (
    libGAP_Obj                 self,
    libGAP_Obj                 list,
    libGAP_Obj                 obj,
    libGAP_Obj                 start )
{
    return libGAP_PosListDefault( list, obj, start ) ;
}


/****************************************************************************
**
*F  ElmListLevel(<lists>,<pos>,<level>) . . . . . . . . . . . . . . . . . . .
*F  . . . . . . . . . . . . .  select an element of several lists in parallel
**
**  'ElmListLevel' either  selects an element  from all  lists in parallel if
**  <level> is 1, or recurses if <level> is greater than 1.
*/
void            libGAP_ElmListLevel (
    libGAP_Obj                 lists,
    libGAP_Obj                 pos,
    libGAP_Int                 level )
{
    libGAP_Int                 len;            /* length of <lists>               */
    libGAP_Obj                 list;           /* one list from <lists>           */
    libGAP_Obj                 elm;            /* selected element from <list>    */
    libGAP_Int                 i;              /* loop variable                   */

    /* if <level> is one, perform the replacements                         */
    if ( level == 1 ) {

        /* loop over the elements of <lists> (which must be a plain list)  */
        len = libGAP_LEN_PLIST( lists );
        for ( i = 1; i <= len; i++ ) {

            /* get the list                                                */
            list = libGAP_ELM_PLIST( lists, i );

            /* select the element                                          */
            if (libGAP_IS_INTOBJ(pos))
              elm = libGAP_ELM_LIST( list, libGAP_INT_INTOBJ(pos) );
            else
              elm = libGAP_ELMB_LIST(list, pos);

            /* replace the list with the element                           */
            libGAP_SET_ELM_PLIST( lists, i, elm );

            /* notify Gasman                                               */
            libGAP_CHANGED_BAG( lists );

        }
        libGAP_RetypeBag(lists, libGAP_T_PLIST_DENSE);

    }

    /* otherwise recurse                                                   */
    else {

        /* loop over the elements of <lists> (which must be a plain list)  */
        len = libGAP_LEN_PLIST( lists );
        for ( i = 1; i <= len; i++ ) {

            /* get the list                                                */
            list = libGAP_ELM_PLIST( lists, i );

            /* recurse                                                     */
            libGAP_ElmListLevel( list, pos, level-1 );

        }

    }

}


/****************************************************************************
**
*F  ElmsListLevel(<lists>,<poss>,<level>) . . . . . . . . . . . . . . . . . .
*F  . . . . . . . . . .  select several elements of several lists in parallel
**
**  'ElmsListLevel' either selects  elements  from all lists in  parallel  if
**  <level> is 1, or recurses if <level> is greater than 1.
*/
void            libGAP_ElmsListLevel (
    libGAP_Obj                 lists,
    libGAP_Obj                 poss,
    libGAP_Int                 level )
{
    libGAP_Int                 len;            /* length of <lists>               */
    libGAP_Obj                 list;           /* one list from <lists>           */
    libGAP_Obj                 elm;            /* selected elements from <list>   */
    libGAP_Int                 i;              /* loop variable                   */

    /* Workaround for issue #312: Accessing a two-level sublist
       of a compressed FFE vector could lead to crashes because
       FuncELMS_VEC8BIT and FuncELMS_GF2VEC may return lists which are
       not plists. This boils down to a conflict between the documented
       behavior and requirements of ElmsListLevel and ElmsListFuncs.
       Resolving this properly requires some more discussion. But until
       then, this change at least prevents hard crashes. */
    if (!libGAP_IS_PLIST(lists)) {
        libGAP_ErrorMayQuit(
            "List Elements: <lists> must be a list (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(lists), 0L );
    }

    /* if <level> is one, perform the replacements                         */
    if ( level == 1 ) {

        /* loop over the elements of <lists> (which must be a plain list)  */
        len = libGAP_LEN_PLIST( lists );
        for ( i = 1; i <= len; i++ ) {

            /* get the list                                                */
            list = libGAP_ELM_PLIST( lists, i );

            /* select the elements                                         */
            elm = libGAP_ELMS_LIST( list, poss );

            /* replace the list with the elements                          */
            libGAP_SET_ELM_PLIST( lists, i, elm );

            /* notify Gasman                                               */
            libGAP_CHANGED_BAG( lists );

        }

    }

    /* otherwise recurse                                                   */
    else {

        /* loop over the elements of <lists> (which must be a plain list)  */
        len = libGAP_LEN_PLIST( lists );
        for ( i = 1; i <= len; i++ ) {

            /* get the list                                                */
            list = libGAP_ELM_PLIST( lists, i );

            /* recurse                                                     */
            libGAP_ElmsListLevel( list, poss, level-1 );

        }

    }

}


/****************************************************************************
**
*F  AssListLevel(<lists>,<pos>,<objs>,<level>)  . . . . . . . . . . . . . . .
*F  . . . . . . . . . . . . .  assign an element to several lists in parallel
**
**  'AssListLevel'  either assigns an  element  to all  lists in parallel  if
**  <level> is 1, or recurses if <level> is greater than 1.
*/
void            libGAP_AssListLevel (
    libGAP_Obj                 lists,
    libGAP_Obj                 pos,
    libGAP_Obj                 objs,
    libGAP_Int                 level )
{
    libGAP_Int                 len;            /* length of <lists> and <objs>    */
    libGAP_Obj                 list;           /* one list of <lists>             */
    libGAP_Obj                 obj;            /* one value from <objs>           */
    libGAP_Int                 i;              /* loop variable                   */

    /* check <objs>                                                        */
    while ( ! libGAP_IS_DENSE_LIST(objs) || libGAP_LEN_LIST(lists) != libGAP_LEN_LIST(objs) ) {
        if ( ! libGAP_IS_DENSE_LIST(objs) ) {
            objs = libGAP_ErrorReturnObj(
                "List Assignment: <objs> must be a dense list (not a %s)",
                (libGAP_Int)libGAP_TNAM_OBJ(objs), 0L,
                "you can replace <objs> via 'return <objs>;'" );
        }
        if ( libGAP_LEN_LIST(lists) != libGAP_LEN_LIST(objs) ) {
            objs = libGAP_ErrorReturnObj(
         "List Assignment: <objs> must have the same length as <lists> (%d)",
                libGAP_LEN_LIST(lists), 0L,
                "you can replace <objs> via 'return <objs>;'" );
        }
    }

    /* if <level> is one, perform the assignments                          */
    if ( level == 1 ) {

        /* loop over the elements of <lists> (which must be a plain list)  */
        len = libGAP_LEN_PLIST( lists );
        for ( i = 1; i <= len; i++ ) {

            /* get the list                                                */
            list = libGAP_ELM_PLIST( lists, i );

            /* select the element to assign                                */
            obj = libGAP_ELMW_LIST( objs, i );

            /* assign the element                                          */
            if (libGAP_IS_INTOBJ(pos))
              libGAP_ASS_LIST( list, libGAP_INT_INTOBJ(pos), obj );
            else
              libGAP_ASSB_LIST(list, pos, obj);

        }

    }

    /* otherwise recurse                                                   */
    else {

        /* loop over the elements of <lists> (which must be a plain list)  */
        len = libGAP_LEN_PLIST( lists );
        for ( i = 1; i <= len; i++ ) {

            /* get the list                                                */
            list = libGAP_ELM_PLIST( lists, i );

            /* get the values                                              */
            obj = libGAP_ELMW_LIST( objs, i );

            /* recurse                                                     */
            libGAP_AssListLevel( list, pos, obj, level-1 );

        }

    }

}


/****************************************************************************
**
*F  AsssListLevel(<lists>,<poss>,<objs>,<level>)  . . . . . . . . . . . . . .
*F  . . . . . . . . . .  assign several elements to several lists in parallel
**
**  'AsssListLevel'  either  assigns elements  to   all lists in parallel  if
**  <level> is 1, or recurses if <level> is greater than 1.
*/
void            libGAP_AsssListLevel (
    libGAP_Obj                 lists,
    libGAP_Obj                 poss,
    libGAP_Obj                 objs,
    libGAP_Int                 lev )
{
    libGAP_Int                 len;            /* length of <lists> and <objs>    */
    libGAP_Obj                 list;           /* one list of <lists>             */
    libGAP_Obj                 obj;            /* one value from <objs>           */
    libGAP_Int                 i;              /* loop variable                   */

    /* check <objs>                                                        */
    while ( ! libGAP_IS_DENSE_LIST(objs) || libGAP_LEN_LIST(lists) != libGAP_LEN_LIST(objs) ) {
        if ( ! libGAP_IS_DENSE_LIST(objs) ) {
            objs = libGAP_ErrorReturnObj(
                "List Assignment: <objs> must be a dense list (not a %s)",
                (libGAP_Int)libGAP_TNAM_OBJ(objs), 0L,
                "you can replace <objs> via 'return <objs>;'" );
        }
        if ( libGAP_LEN_LIST(lists) != libGAP_LEN_LIST(objs) ) {
            objs = libGAP_ErrorReturnObj(
         "List Assignment: <objs> must have the same length as <lists> (%d)",
                libGAP_LEN_LIST(lists), 0L,
                "you can replace <objs> via 'return <objs>;'" );
        }
    }

    /* if <lev> is one, loop over all the lists and assign the value       */
    if ( lev == 1 ) {

        /* loop over the list entries (which must be lists too)            */
        len = libGAP_LEN_PLIST( lists );
        for ( i = 1; i <= len; i++ ) {

            /* get the list                                                */
            list = libGAP_ELM_PLIST( lists, i );

            /* select the elements to assign                               */
            obj = libGAP_ELMW_LIST( objs, i );
            while ( ! libGAP_IS_DENSE_LIST( obj )
                 || libGAP_LEN_LIST( poss ) != libGAP_LEN_LIST( obj ) ) {
                if ( ! libGAP_IS_DENSE_LIST( obj ) ) {
                    obj = libGAP_ErrorReturnObj(
                  "List Assignments: <objs> must be a dense list (not a %s)",
                        (libGAP_Int)libGAP_TNAM_OBJ(obj), 0L,
                        "you can replace <objs> via 'return <objs>;'" );
                }
                if ( libGAP_LEN_LIST( poss ) != libGAP_LEN_LIST( obj ) ) {
                    obj = libGAP_ErrorReturnObj(
     "List Assigments: <objs> must have the same length as <positions> (%d)",
                        libGAP_LEN_LIST( poss ), 0L,
                        "you can replace <objs> via 'return <objs>;'" );
                }
            }

            /* assign the elements                                         */
            libGAP_ASSS_LIST( list, poss, obj );

        }

    }

    /* otherwise recurse                                                   */
    else {

        /* loop over the list entries (which must be lists too)            */
        len = libGAP_LEN_PLIST( lists );
        for ( i = 1; i <= len; i++ ) {

            /* get the list                                                */
            list = libGAP_ELM_PLIST( lists, i );

            /* get the values                                              */
            obj = libGAP_ELMW_LIST( objs, i );

            /* recurse                                                     */
            libGAP_AsssListLevel( list, poss, obj, lev-1 );

        }

    }

}


/****************************************************************************
**
*F  PLAIN_LIST(<list>)  . . . . . . . . . . .  convert a list to a plain list
*V  PlainListFuncs[<type>]  . . . . . . . . . . table of conversion functions
*F  PlainListError(<list>)  . . . . . . . . . . . . error conversion function
**
**  'PLAIN_LIST'    only    calls       the    function  pointed    to     by
**  'PlainListFuncs[<type>]', passing <list>  as argument.  If  <type> is not
**  the     type of   a    list,  then    'PlainListFuncs[<type>]'  points to
**  'PlainListError', which just signals an error.
**
**  'PLAIN_LIST'  is defined in  the  declaration  part  of  this  package as
**  follows
**
#define PLAIN_LIST(list) \
                        ((*PlainListFuncs[TNUM_OBJ(list)])(list))
*/
void            (*libGAP_PlainListFuncs[libGAP_LAST_REAL_TNUM+1]) ( libGAP_Obj list );

void            libGAP_PlainListError (
    libGAP_Obj                 list )
{
    libGAP_ErrorQuit(
        "Panic: cannot convert <list> (is a %s) to a plain list",
        (libGAP_Int)libGAP_TNAM_OBJ(list), 0L );
}


/****************************************************************************
**
*F  TYPES_LIST_FAM(<fam>) . . . . . . .  list of kinds of lists over a family
*/
libGAP_UInt            libGAP_TYPES_LIST_FAM_RNam;

libGAP_Obj             libGAP_TYPES_LIST_FAM (
    libGAP_Obj                 fam )
{
    return libGAP_ElmPRec( fam, libGAP_TYPES_LIST_FAM_RNam );
}


/****************************************************************************
**
*F  IsMutableListYes(<list>)  . . . . . . . mutability test for mutable lists
*F  IsMutableListNo(<list>) . . . . . . . mutability test for immutable lists
**
**  'IsMutableListYes' simply returns 1.  'IsMutableListNo' simply returns 0.
**  Note that we can decide from the type number whether a list is mutable or
**  immutable.
**
**  'IsMutableListYes' is  the function  in 'IsMutableObjFuncs'   for mutable
**  lists.   'IsMutableListNo'  is  the function  in 'IsMutableObjFuncs'  for
**  immutable lists.
*/
libGAP_Int             libGAP_IsMutableListNo (
    libGAP_Obj                 list )
{
    return 0L;
}

libGAP_Int             libGAP_IsMutableListYes (
    libGAP_Obj                 list )
{
    return 1L;
}


/****************************************************************************
**
*F  IsCopyableListYes(<list>) . . . . . . . . . .  copyability test for lists
**
**  'IsCopyableListYes' simply returns 1.  Note that all lists are copyable.
**
**  'IsCopyableListYes' is the function in 'IsCopyableObjFuncs' for lists.
*/
libGAP_Int             libGAP_IsCopyableListYes (
    libGAP_Obj                     list )
{
    return 1;
}


/****************************************************************************
**
*F  PrintListDefault(<list>)  . . . . . . . . . . . . . . . . .  print a list
*F  PrintPathList(<list>,<indx>)  . . . . . . . . . . . . . print a list path
**
**  'PrintList' simply prints the list.
*/
void            libGAP_PrintListDefault (
    libGAP_Obj                 list )
{
    libGAP_Obj                 elm;

    if ( 0 < libGAP_LEN_LIST(list) && libGAP_IsStringConv(list) ) {
        libGAP_PrintString(list);
        return;
    }

    libGAP_Pr("%2>[ %2>",0L,0L);
    for ( libGAP_PrintObjIndex=1; libGAP_PrintObjIndex<=libGAP_LEN_LIST(list); libGAP_PrintObjIndex++ ) {
        elm = libGAP_ELMV0_LIST( list, libGAP_PrintObjIndex );
        if ( elm != 0 ) {
            if ( 1 < libGAP_PrintObjIndex )  libGAP_Pr( "%<,%< %2>", 0L, 0L );
            libGAP_PrintObj( elm );
        }
        else {
            if ( 1 < libGAP_PrintObjIndex )  libGAP_Pr( "%2<,%2>", 0L, 0L );
        }
    }
    libGAP_Pr(" %4<]",0L,0L);
}

void            libGAP_PrintPathList (
    libGAP_Obj                 list,
    libGAP_Int                 indx )
{
    libGAP_Pr( "[%d]", indx, 0L );
}


/****************************************************************************
**

*V  SetFiltListTNums[ <tnum> ][ <fnum> ]  . . . . . new tnum after filter set
**
**  If a list  with type number <tnum>  gains  the filter  with filter number
**  <fnum>, then the new type number is stored in:
**
**  `SetFiltListTNums[<tnum>][<fnum>]'
**
**  The macro  `SET_FILT_LIST' is  used  to  set  the filter  for a  list  by
**  changing its type number.
*/
libGAP_UInt libGAP_SetFiltListTNums [ libGAP_LAST_REAL_TNUM ] [ libGAP_LAST_FN + 1 ];


/****************************************************************************
**
*V  ResetFiltListTNums[ <tnum> ][ <fnum> ]  . . . new tnum after filter reset
**
**  If a list  with type number <tnum>  loses  the filter  with filter number
**  <fnum>, then the new type number is stored in:
**
**  `ResetFiltListTNums[<tnum>][<fnum>]'
**
**  The macro `RESET_FILT_LIST' is used  to  set  the filter  for a  list  by
**  changing its type number.
*/
libGAP_UInt libGAP_ResetFiltListTNums [ libGAP_LAST_REAL_TNUM ] [ libGAP_LAST_FN  + 1];


/****************************************************************************
**
*V  HasFiltListTNums[ <tnum> ][ <fnum> ]  . . . . . . . . . . . .  has filter
*/
libGAP_Int libGAP_HasFiltListTNums [ libGAP_LAST_REAL_TNUM ] [ libGAP_LAST_FN + 1 ];


/****************************************************************************
**
*V  ClearFiltsTNums[ <tnum> ] . clear all list filters except `FN_IS_MUTABLE'
**
**  The type  number without any  known properties  of a  list of type number
**  <tnum> is stored in:
**
**  `ClearPropsTNums[<tnum>]'
**
**  The macro `CLEAR_PROPS_LIST' is used to clear all properties of a list.
*/
libGAP_UInt libGAP_ClearFiltsTNums [ libGAP_LAST_REAL_TNUM ];


/****************************************************************************
**
*F  FuncSET_FILTER_LIST( <self>, <list>, <filter> ) . . . . . . .  set filter
*/
libGAP_Obj libGAP_FuncSET_FILTER_LIST (
    libGAP_Obj             self,
    libGAP_Obj             list,
    libGAP_Obj             filter )
{
    libGAP_Int             new;
    libGAP_Obj             flags;

    /* this could be done by a table lookup                                */
    flags = libGAP_FLAGS_FILT(filter);
    if (libGAP_FuncIS_SUBSET_FLAGS(0,flags,libGAP_FLAGS_FILT(libGAP_IsSSortListProp))==libGAP_True) {
        new = libGAP_SetFiltListTNums[libGAP_TNUM_OBJ(list)][libGAP_FN_IS_DENSE];
        if ( new < 0 )  goto error;
        new = libGAP_SetFiltListTNums[libGAP_TNUM_OBJ(list)][libGAP_FN_IS_HOMOG];
        if ( new < 0 )  goto error;
        new = libGAP_SetFiltListTNums[libGAP_TNUM_OBJ(list)][libGAP_FN_IS_SSORT];
        if ( new > 0 )  libGAP_RetypeBag( list, new );  else goto error;
    }
    if (libGAP_FuncIS_SUBSET_FLAGS(0,flags,libGAP_FLAGS_FILT(libGAP_IsNSortListProp))==libGAP_True) {
        new = libGAP_SetFiltListTNums[libGAP_TNUM_OBJ(list)][libGAP_FN_IS_NSORT];
        if ( new > 0 )  libGAP_RetypeBag( list, new );  else goto error;
    }
    return 0;

    /* setting of filter failed                                            */
error:
    libGAP_ErrorReturnVoid( "filter not possible for %s",
                     (libGAP_Int)libGAP_TNAM_OBJ(list), 0,
                     "you can 'return;'" );
    return 0;
}


/****************************************************************************
**
*F  FuncRESET_FILTER_LIST( <self>, <list>, <filter> ) . . . . .  reset filter
*/
libGAP_Obj libGAP_FuncRESET_FILTER_LIST (
    libGAP_Obj             self,
    libGAP_Obj             list,
    libGAP_Obj             filter )
{
    libGAP_Int             fn;
    libGAP_Int             new;

    /* this could be done by a table lookup                                */
    if ( filter == libGAP_IsSSortListProp ) {
        fn = libGAP_FN_IS_SSORT;
    }
    else if ( filter == libGAP_IsNSortListProp ) {
        fn = libGAP_FN_IS_NSORT;
    }
    else {
        return 0;
    }

    /* try to set the filter                                               */
    new = libGAP_ResetFiltListTNums[libGAP_TNUM_OBJ(list)][fn];
    if ( new > 0 ) {
        libGAP_RetypeBag( list, new );
    }
    else if ( new < 0 ) {
        libGAP_ErrorReturnVoid( "filter not possible for %s",
                         (libGAP_Int)libGAP_TNAM_OBJ(list), 0,
                         "you can 'return;'" );
    }
    return 0;
}


/****************************************************************************
**

*F * * * * * * * * * * * functions with checking  * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*F  AsssListCheck( <list>, <poss>, <rhss> ) . . . . . . . . . . . . ASSS_LIST
*/
void libGAP_AsssListCheck (
    libGAP_Obj                 list,
    libGAP_Obj                 poss,
    libGAP_Obj                 rhss )
{
    if ( ! libGAP_IS_POSS_LIST(poss) ) {
        libGAP_ErrorQuit(
    "List Assignment: <positions> must be a dense list of positive integers",
            0L, 0L );
    }
    if ( ! libGAP_IS_DENSE_LIST(rhss) ) {
        libGAP_ErrorQuit(
            "List Assignment: <rhss> must be a dense list",
            0L, 0L );
    }
    if ( libGAP_LEN_LIST( poss ) != libGAP_LEN_LIST( rhss ) ) {
        libGAP_ErrorQuit(
     "List Assignment: <rhss> must have the same length as <positions> (%d)",
            (libGAP_Int)libGAP_LEN_LIST(poss), 0L );
    }
    libGAP_ASSS_LIST( list, poss, rhss );
}


/****************************************************************************
**
*F  AsssPosObjCheck( <list>, <poss>, <rhss> ) . . . . . . . . . . . ASSS_LIST
*/
void libGAP_AsssPosObjCheck (
    libGAP_Obj                 list,
    libGAP_Obj                 poss,
    libGAP_Obj                 rhss )
{
    if ( ! libGAP_IS_POSS_LIST(poss) ) {
        libGAP_ErrorQuit(
    "List Assignment: <positions> must be a dense list of positive integers",
            0L, 0L );
    }
    if ( ! libGAP_IS_DENSE_LIST(rhss) ) {
        libGAP_ErrorQuit(
            "List Assignment: <rhss> must be a dense list",
            0L, 0L );
    }
    if ( libGAP_LEN_LIST( poss ) != libGAP_LEN_LIST( rhss ) ) {
        libGAP_ErrorQuit(
     "List Assignment: <rhss> must have the same length as <positions> (%d)",
            (libGAP_Int)libGAP_LEN_LIST(poss), 0L );
    }
    if ( libGAP_TNUM_OBJ(list) == libGAP_T_POSOBJ ) {
        libGAP_ErrorQuit( "sorry: <posobj>!{<poss>} not yet implemented", 0L, 0L );
    }
    else {
        libGAP_ASSS_LIST( list, poss, rhss );
    }
}


/****************************************************************************
**
*F  AsssListLevelCheck( <lists>, <poss>, <rhss>, <level> )  . . AsssListLevel
*/
void libGAP_AsssListLevelCheck (
    libGAP_Obj                 lists,
    libGAP_Obj                 poss,
    libGAP_Obj                 rhss,
    libGAP_Int                 level )
{
    if ( ! libGAP_IS_POSS_LIST(poss) ) {
        libGAP_ErrorQuit(
    "List Assignment: <positions> must be a dense list of positive integers",
            0L, 0L );
    }
    libGAP_AsssListLevel( lists, poss, rhss, level );
}


/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/


/****************************************************************************
**

*V  GVarFilts . . . . . . . . . . . . . . . . . . . list of filters to export
*/
static libGAP_StructGVarFilt libGAP_GVarFilts [] = {

    { "IS_LIST", "obj", &libGAP_IsListFilt,
      libGAP_FuncIS_LIST, "src/lists.c:IS_LIST" },

    { "IS_DENSE_LIST", "obj", &libGAP_IsDenseListFilt,
      libGAP_FuncIS_DENSE_LIST, "src/lists.c:IS_DENSE_LIST" },

    { "IS_HOMOG_LIST", "obj", &libGAP_IsHomogListFilt,
      libGAP_FuncIS_HOMOG_LIST, "src/lists.c:IS_HOMOG_LIST" },

    { "IS_TABLE_LIST", "obj", &libGAP_IsTableListFilt,
      libGAP_Func_IS_TABLE_LIST, "src/lists.c:IS_TABLE_LIST" },

    { 0 }

};


/****************************************************************************
**
*V  GVarAttrs . . . . . . . . . . . . . . . . .  list of attributes to export
*/
static libGAP_StructGVarAttr libGAP_GVarAttrs [] = {

    { "LENGTH", "list", &libGAP_LengthAttr,
      libGAP_FuncLENGTH, "src/lists.c:LENGTH" },

    { 0 }

};


/****************************************************************************
**
*V  GVarProps . . . . . . . . . . . . . . . . .  list of properties to export
*/
static libGAP_StructGVarProp libGAP_GVarProps [] = {

    { "IS_SSORT_LIST", "obj", &libGAP_IsSSortListProp,
      libGAP_FuncIS_SSORT_LIST, "src/lists.c:IS_SSORT_LIST" },

    { "IS_NSORT_LIST", "obj", &libGAP_IsNSortListProp,
      libGAP_FuncIS_NSORT_LIST, "src/lists.c:IS_NSORT_LIST" },

    { "IS_POSS_LIST", "obj", &libGAP_IsPossListProp,
      libGAP_FuncIS_POSS_LIST, "src/lists.c:IS_POSS_LIST" },


    { 0 }

};


/****************************************************************************
**
*V  GVarOpers . . . . . . . . . . . . . . . . .  list of operations to export
*/
static libGAP_StructGVarOper libGAP_GVarOpers [] = {

    { "POS_LIST", -1, "list, obj", &libGAP_PosListOper,
      libGAP_DoOperation0Args, "src/lists.c:POS_LIST" },

    { "ISB_LIST", 2, "list, pos", &libGAP_IsbListOper,
      libGAP_IsbListHandler, "src/lists.c:ISB_LIST" },

    { "ELM0_LIST", 2, "list, pos", &libGAP_Elm0ListOper,
      libGAP_FuncELM0_LIST, "src/lists.c:ELM0_LIST" },

    { "ELM_LIST", 2, "list, pos", &libGAP_ElmListOper,
      libGAP_FuncELM_LIST, "src/lists.c:ELM_LIST" },

    { "ELMS_LIST", 2, "list, poss", &libGAP_ElmsListOper,
      libGAP_FuncELMS_LIST, "src/lists.c:ELMS_LIST" },

    { "UNB_LIST", 2, "list, pos", &libGAP_UnbListOper, 
      libGAP_FuncUNB_LIST, "src/lists.c:UNB_LIST" },

    { "ASS_LIST", 3, "list, pos, obj", &libGAP_AssListOper, 
      libGAP_FuncASS_LIST, "src/lists.c:ASS_LIST" },

    { "ASSS_LIST", 3, "list, poss, objs", &libGAP_AsssListOper, 
      libGAP_FuncASSS_LIST, "src/lists.c:ASSS_LIST" },

    { 0 }

};


/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "LEN_LIST", 1, "list", 
      libGAP_FuncLEN_LIST, "src/lists.c:LEN_LIST" },

    { "ELMS_LIST_DEFAULT", 2, "list, poss",
      libGAP_FuncELMS_LIST_DEFAULT, "src/list.c:ELMS_LIST_DEFAULT" },

    { "ASSS_LIST_DEFAULT", 3, "list, poss, objs",
      libGAP_FuncASSS_LIST_DEFAULT, "src/list.c:ASSS_LIST_DEFAULT" },
    
    { "IS_SSORT_LIST_DEFAULT", 1, "list",
      libGAP_FuncIS_SSORT_LIST_DEFAULT, "src/list.c:IS_SSORT_LIST_DEFAULT" },

    { "IS_POSS_LIST_DEFAULT", 1, "list",
      libGAP_FuncIS_POSS_LIST_DEFAULT, "src/list.c:IS_POSS_LIST_DEFAULT" },

    { "POS_LIST_DEFAULT", 3, "list, obj, start",
      libGAP_FuncPOS_LIST_DEFAULT, "src/list.c:POS_LIST_DEFAULT" },

    { "SET_FILTER_LIST", 2, "list, filter",
      libGAP_FuncSET_FILTER_LIST, "src/lists.c:SET_FILTER_LIST" },

    { "RESET_FILTER_LIST", 2, "list, filter",
      libGAP_FuncRESET_FILTER_LIST, "src/lists.c:RESET_FILTER_LIST" },

    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_UInt                type;           /* loop variable                   */
    libGAP_Int                 i;              /* loop variable                   */

    /* make and install the 'POS_LIST' operation                           */
    libGAP_InitHandlerFunc( libGAP_PosListHandler2, "src/lists.c:PosListHandler2" );
    libGAP_InitHandlerFunc( libGAP_PosListHandler3, "src/lists.c:PosListHandler3" );

    /* init filters and functions                                          */
    libGAP_InitHdlrFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitHdlrAttrsFromTable( libGAP_GVarAttrs );
    libGAP_InitHdlrPropsFromTable( libGAP_GVarProps );
    libGAP_InitHdlrOpersFromTable( libGAP_GVarOpers );
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* import small list machinery from the library */
    libGAP_ImportFuncFromLibrary("IsSmallList", &libGAP_IsSmallListFilt);
    libGAP_ImportFuncFromLibrary("HasIsSmallList", &libGAP_HasIsSmallListFilt);
    libGAP_ImportFuncFromLibrary("SetIsSmallList", &libGAP_SetIsSmallList);

    /* make and install the 'IS_LIST' filter                               */
    for ( type = libGAP_FIRST_REAL_TNUM; type <= libGAP_LAST_REAL_TNUM; type++ ) {
        libGAP_IsListFuncs[ type ] = libGAP_IsListNot;
    }
    for ( type = libGAP_FIRST_LIST_TNUM; type <= libGAP_LAST_LIST_TNUM; type++ ) {
        libGAP_IsListFuncs[ type ] = libGAP_IsListYes;
    }
    for ( type = libGAP_FIRST_EXTERNAL_TNUM; type <= libGAP_LAST_EXTERNAL_TNUM; type++ ) {
        libGAP_IsListFuncs[ type ] = libGAP_IsListObject;
    }

    /* make and install the 'IS_SMALL_LIST' filter                   */
    /* non-lists are not small lists */
    for ( type = libGAP_FIRST_REAL_TNUM; type <= libGAP_LAST_REAL_TNUM; type++ ) {
        libGAP_IsSmallListFuncs[ type ] = libGAP_IsSmallListNot;
    }
    /* internal lists ARE small lists */
    for ( type = libGAP_FIRST_LIST_TNUM; type <= libGAP_LAST_LIST_TNUM; type++ ) {
        libGAP_IsSmallListFuncs[ type ] = libGAP_IsSmallListYes;
    }
    /* external lists need to be asked */
    for ( type = libGAP_FIRST_EXTERNAL_TNUM; type <= libGAP_LAST_EXTERNAL_TNUM; type++ ) {
        libGAP_IsSmallListFuncs[ type ] = libGAP_IsSmallListObject;
    }


    /* make and install the 'LEN_LIST' function                            */
    for ( type = libGAP_FIRST_REAL_TNUM; type <= libGAP_LAST_REAL_TNUM; type++ ) {
        libGAP_LenListFuncs[ type ] = libGAP_LenListError;
    }
    for ( type = libGAP_FIRST_EXTERNAL_TNUM; type <= libGAP_LAST_EXTERNAL_TNUM; type++ ) {
        libGAP_LenListFuncs[ type ] = libGAP_LenListObject;
    }
    libGAP_LenListFuncs[ libGAP_T_SINGULAR ] = libGAP_LenListObject;

    /* make and install the 'LENGTH' function                            */
    for ( type = libGAP_FIRST_REAL_TNUM; type <= libGAP_LAST_REAL_TNUM; type++ ) {
        libGAP_LengthFuncs[ type ] = libGAP_LengthError;
    }
    for ( type = libGAP_FIRST_EXTERNAL_TNUM; type <= libGAP_LAST_EXTERNAL_TNUM; type++ ) {
        libGAP_LengthFuncs[ type ] = libGAP_LengthObject;
    }
    for ( type = libGAP_FIRST_LIST_TNUM; type <= libGAP_LAST_LIST_TNUM; type++ ) {
        libGAP_LengthFuncs[ type ] = libGAP_LengthInternal;
    }
    libGAP_LengthFuncs[ libGAP_T_SINGULAR ] = libGAP_LengthObject;


    /* make and install the 'ISB_LIST' operation                           */
    for ( type = libGAP_FIRST_REAL_TNUM; type <= libGAP_LAST_REAL_TNUM; type++ ) {
        libGAP_IsbListFuncs[  type ] = libGAP_IsbListError;
        libGAP_IsbvListFuncs[ type ] = libGAP_IsbListError;
    }
    for ( type = libGAP_FIRST_EXTERNAL_TNUM; type <= libGAP_LAST_EXTERNAL_TNUM; type++ ) {
        libGAP_IsbListFuncs[  type ] = libGAP_IsbListObject;
        libGAP_IsbvListFuncs[ type ] = libGAP_IsbListObject;
    }
    libGAP_IsbListFuncs[ libGAP_T_SINGULAR ] = libGAP_IsbListObject;
    libGAP_IsbvListFuncs[ libGAP_T_SINGULAR ] = libGAP_IsbListObject;

    /* make and install the 'ISBB_LIST' operation                           */
    for ( type = libGAP_FIRST_REAL_TNUM; type <= libGAP_LAST_REAL_TNUM; type++ ) {
        libGAP_IsbbListFuncs[  type ] = libGAP_IsbbListError;
    }

    for (type = libGAP_FIRST_LIST_TNUM; type <= libGAP_LAST_LIST_TNUM; type++ ) {
      libGAP_IsbbListFuncs[ type ] = libGAP_IsbbListInternal;
    }
    
    for ( type = libGAP_FIRST_EXTERNAL_TNUM; type <= libGAP_LAST_EXTERNAL_TNUM; type++ ) {
        libGAP_IsbbListFuncs[  type ] = libGAP_IsbbListObject;
    }
    libGAP_IsbbListFuncs[ libGAP_T_SINGULAR ] = libGAP_IsbbListObject;

    /* make and install the 'ELM0_LIST' operation                          */
    for ( type = libGAP_FIRST_REAL_TNUM; type <= libGAP_LAST_REAL_TNUM; type++ ) {
        libGAP_Elm0ListFuncs[  type ] = libGAP_Elm0ListError;
        libGAP_Elm0vListFuncs[ type ] = libGAP_Elm0ListError;
    }
    for ( type = libGAP_FIRST_EXTERNAL_TNUM; type <= libGAP_LAST_EXTERNAL_TNUM; type++ ) {
        libGAP_Elm0ListFuncs[  type ] = libGAP_Elm0ListObject;
        libGAP_Elm0vListFuncs[ type ] = libGAP_Elm0ListObject;
    }
    libGAP_Elm0ListFuncs[ libGAP_T_SINGULAR ] = libGAP_Elm0ListObject;
    libGAP_Elm0vListFuncs[ libGAP_T_SINGULAR ] = libGAP_Elm0ListObject;


    /* make and install the 'ELM_LIST' operation                           */
    for ( type = libGAP_FIRST_REAL_TNUM; type <= libGAP_LAST_REAL_TNUM; type++ ) {
        libGAP_ElmListFuncs[  type ] = libGAP_ElmListError;
        libGAP_ElmvListFuncs[ type ] = libGAP_ElmListError;
        libGAP_ElmwListFuncs[ type ] = libGAP_ElmListError;
    }
    for ( type = libGAP_FIRST_EXTERNAL_TNUM; type <= libGAP_LAST_EXTERNAL_TNUM; type++ ) {
        libGAP_ElmListFuncs[  type ] = libGAP_ElmListObject;
        libGAP_ElmvListFuncs[ type ] = libGAP_ElmListObject;
        libGAP_ElmwListFuncs[ type ] = libGAP_ElmListObject;
    }
    libGAP_ElmListFuncs[  libGAP_T_SINGULAR ] = libGAP_ElmListObject;
    libGAP_ElmvListFuncs[ libGAP_T_SINGULAR ] = libGAP_ElmListObject;
    libGAP_ElmwListFuncs[ libGAP_T_SINGULAR ] = libGAP_ElmListObject;

    /* make and install the 'ELMB_LIST' operation                           
    for ( type = FIRST_REAL_TNUM; type <= LAST_REAL_TNUM; type++ ) {
        ElmbListFuncs[  type ] = ElmbListError;
    }

    for (type = FIRST_LIST_TNUM; type <= LAST_LIST_TNUM; type++ ) {
      ElmbListFuncs[ type ] = ElmbListInternal;
    }
    
    for ( type = FIRST_EXTERNAL_TNUM; type <= LAST_EXTERNAL_TNUM; type++ ) {
        ElmbListFuncs[  type ] = ElmbListObject;
    }

    */

    /* make and install the 'ELMS_LIST' operation                          */
    for ( type = libGAP_FIRST_REAL_TNUM; type <= libGAP_LAST_REAL_TNUM; type++ ) {
        libGAP_ElmsListFuncs[ type ] = libGAP_ElmsListError;
    }
    for ( type = libGAP_FIRST_LIST_TNUM; type <= libGAP_LAST_LIST_TNUM; type++ ) {
        libGAP_ElmsListFuncs[ type ] = libGAP_ElmsListDefault;
    }
    for ( type = libGAP_FIRST_EXTERNAL_TNUM; type <= libGAP_LAST_EXTERNAL_TNUM; type++ ) {
        libGAP_ElmsListFuncs[ type ] = libGAP_ElmsListObject;
    }
    libGAP_ElmsListFuncs[ libGAP_T_SINGULAR ] = libGAP_ElmsListObject;


    /* make and install the 'UNB_LIST' operation                           */
    for ( type = libGAP_FIRST_REAL_TNUM; type <= libGAP_LAST_REAL_TNUM; type++ ) {
        libGAP_UnbListFuncs[ type ] = libGAP_UnbListError;
    }
    for ( type = libGAP_FIRST_LIST_TNUM; type <= libGAP_LAST_LIST_TNUM; type++ ) {
        libGAP_UnbListFuncs[ type ] = libGAP_UnbListDefault;
    }
    for ( type = libGAP_FIRST_EXTERNAL_TNUM; type <= libGAP_LAST_EXTERNAL_TNUM; type++ ) {
        libGAP_UnbListFuncs[ type ] = libGAP_UnbListObject;
    }
    libGAP_UnbListFuncs[ libGAP_T_SINGULAR ] = libGAP_UnbListObject;

    /* make and install the 'UNBB_LIST' operation                           */
    for ( type = libGAP_FIRST_REAL_TNUM; type <= libGAP_LAST_REAL_TNUM; type++ ) {
        libGAP_UnbbListFuncs[  type ] = libGAP_UnbbListError;
    }

    for (type = libGAP_FIRST_LIST_TNUM; type <= libGAP_LAST_LIST_TNUM; type++ ) {
      libGAP_UnbbListFuncs[ type ] = libGAP_UnbbListInternal;
    }
    
    for ( type = libGAP_FIRST_EXTERNAL_TNUM; type <= libGAP_LAST_EXTERNAL_TNUM; type++ ) {
        libGAP_UnbbListFuncs[  type ] = libGAP_UnbbListObject;
    }
    libGAP_UnbbListFuncs[ libGAP_T_SINGULAR ] = libGAP_UnbbListObject;

    /* make and install the 'ASS_LIST' operation                           */
    for ( type = libGAP_FIRST_REAL_TNUM; type <= libGAP_LAST_REAL_TNUM; type++ ) {
        libGAP_AssListFuncs[ type ] = libGAP_AssListError;
    }
    for ( type = libGAP_FIRST_LIST_TNUM; type <= libGAP_LAST_LIST_TNUM; type++ ) {
        libGAP_AssListFuncs[ type ] = libGAP_AssListDefault;
    }
    for ( type = libGAP_FIRST_EXTERNAL_TNUM; type <= libGAP_LAST_EXTERNAL_TNUM; type++ ) {
        libGAP_AssListFuncs[ type ] = libGAP_AssListObject;
    }
    libGAP_AssListFuncs[ libGAP_T_SINGULAR ] = libGAP_AssListObject;


    /* make and install the 'ASSB_LIST' operation                           */
    for ( type = libGAP_FIRST_REAL_TNUM; type <= libGAP_LAST_REAL_TNUM; type++ ) {
        libGAP_AssbListFuncs[  type ] = libGAP_AssbListError;
    }

    for (type = libGAP_FIRST_LIST_TNUM; type <= libGAP_LAST_LIST_TNUM; type++ ) {
      libGAP_AssbListFuncs[ type ] = libGAP_AssbListInternal;
    }
    
    for ( type = libGAP_FIRST_EXTERNAL_TNUM; type <= libGAP_LAST_EXTERNAL_TNUM; type++ ) {
        libGAP_AssbListFuncs[  type ] = libGAP_AssbListObject;
    }
    libGAP_AssbListFuncs[ libGAP_T_SINGULAR ] = libGAP_AssbListObject;

    /* make and install the 'ASSS_LIST' operation                          */
    for ( type = libGAP_FIRST_REAL_TNUM; type <= libGAP_LAST_REAL_TNUM; type++ ) {
        libGAP_AsssListFuncs[ type ] = libGAP_AsssListError;
    }
    for ( type = libGAP_FIRST_LIST_TNUM; type <= libGAP_LAST_LIST_TNUM; type++ ) {
        libGAP_AsssListFuncs[ type ] = libGAP_AsssListDefault;
    }
    for ( type = libGAP_FIRST_EXTERNAL_TNUM; type <= libGAP_LAST_EXTERNAL_TNUM; type++ ) {
        libGAP_AsssListFuncs[ type ] = libGAP_AsssListObject;
    }
    libGAP_AsssListFuncs[ libGAP_T_SINGULAR ] = libGAP_AsssListObject;
    

    /* make and install the 'IS_DENSE_LIST' filter                         */
    for ( type = libGAP_FIRST_REAL_TNUM; type <= libGAP_LAST_REAL_TNUM; type++ ) {
        libGAP_IsDenseListFuncs[ type ] = libGAP_IsDenseListNot;
    }
    for ( type = libGAP_FIRST_LIST_TNUM; type <= libGAP_LAST_LIST_TNUM; type++ ) {
        libGAP_IsDenseListFuncs[ type ] = libGAP_IsDenseListDefault;
    }
    for ( type = libGAP_FIRST_EXTERNAL_TNUM; type <= libGAP_LAST_EXTERNAL_TNUM; type++ ) {
        libGAP_IsDenseListFuncs[ type ] = libGAP_IsDenseListObject;
    }
    libGAP_IsDenseListFuncs[ libGAP_T_SINGULAR ] = libGAP_IsDenseListObject;


    /* make and install the 'IS_HOMOG_LIST' filter                         */
    for ( type = libGAP_FIRST_REAL_TNUM; type <= libGAP_LAST_REAL_TNUM; type++ ) {
        libGAP_IsHomogListFuncs[ type ] = libGAP_IsHomogListNot;
    }
    for ( type = libGAP_FIRST_LIST_TNUM; type <= libGAP_LAST_LIST_TNUM; type++ ) {
        libGAP_IsHomogListFuncs[ type ] = libGAP_IsHomogListDefault;
    }
    for ( type = libGAP_FIRST_EXTERNAL_TNUM; type <= libGAP_LAST_EXTERNAL_TNUM; type++ ) {
        libGAP_IsHomogListFuncs[ type ] = libGAP_IsHomogListObject;
    }
    libGAP_IsHomogListFuncs[ libGAP_T_SINGULAR ] = libGAP_IsHomogListObject;


    /* make and install the 'IS_TABLE_LIST' filter                         */
    for ( type = libGAP_FIRST_REAL_TNUM; type <= libGAP_LAST_REAL_TNUM; type++ ) {
        libGAP_IsTableListFuncs[ type ] = libGAP_IsTableListNot;
    }
    for ( type = libGAP_FIRST_LIST_TNUM; type <= libGAP_LAST_LIST_TNUM; type++ ) {
        libGAP_IsTableListFuncs[ type ] = libGAP_IsTableListDefault;
    }
    for ( type = libGAP_FIRST_EXTERNAL_TNUM; type <= libGAP_LAST_EXTERNAL_TNUM; type++ ) {
        libGAP_IsTableListFuncs[ type ] = libGAP_IsTableListObject;
    }
    libGAP_IsTableListFuncs[ libGAP_T_SINGULAR ] = libGAP_IsTableListObject;


    /* make and install the 'IS_SSORT_LIST' property                       */
    for ( type = libGAP_FIRST_REAL_TNUM; type <= libGAP_LAST_REAL_TNUM; type++ ) {
        libGAP_IsSSortListFuncs[ type ] = libGAP_IsSSortListNot;
    }
    for ( type = libGAP_FIRST_LIST_TNUM; type <= libGAP_LAST_LIST_TNUM; type++ ) {
        libGAP_IsSSortListFuncs[ type ] = libGAP_IsSSortListDefault;
    }
    for ( type = libGAP_FIRST_EXTERNAL_TNUM; type <= libGAP_LAST_EXTERNAL_TNUM; type++ ) {
        libGAP_IsSSortListFuncs[ type ] = libGAP_IsSSortListObject;
    }
    libGAP_IsSSortListFuncs[ libGAP_T_SINGULAR ] = libGAP_IsSSortListObject;


    /* make and install the 'IS_POSS_LIST' property                        */
    for ( type = libGAP_FIRST_REAL_TNUM; type <= libGAP_LAST_REAL_TNUM; type++ ) {
        libGAP_IsPossListFuncs[ type ] = libGAP_IsPossListNot;
    }
    for ( type = libGAP_FIRST_LIST_TNUM; type <= libGAP_LAST_LIST_TNUM; type++ ) {
        libGAP_IsPossListFuncs[ type ] = libGAP_IsPossListDefault;
    }
    for ( type = libGAP_FIRST_EXTERNAL_TNUM; type <= libGAP_LAST_EXTERNAL_TNUM; type++ ) {
        libGAP_IsPossListFuncs[ type ] = libGAP_IsPossListObject;
    }
    libGAP_IsPossListFuncs[ libGAP_T_SINGULAR ] = libGAP_IsPossListObject;


    /* make and install the 'POS_LIST' operation                           */
    for ( type = libGAP_FIRST_REAL_TNUM; type <= libGAP_LAST_REAL_TNUM; type++ ) {
        libGAP_PosListFuncs[ type ] = libGAP_PosListError;
    }
    for ( type = libGAP_FIRST_LIST_TNUM; type <= libGAP_LAST_LIST_TNUM; type++ ) {
        libGAP_PosListFuncs[ type ] = libGAP_PosListDefault;
    }
    for ( type = libGAP_FIRST_EXTERNAL_TNUM; type <= libGAP_LAST_EXTERNAL_TNUM; type++ ) {
        libGAP_PosListFuncs[ type ] = libGAP_PosListObject;
    }
    libGAP_PosListFuncs[ libGAP_T_SINGULAR ] = libGAP_PosListObject;


    /* install the error functions into the other tables                   */
    for ( type = libGAP_FIRST_REAL_TNUM; type <= libGAP_LAST_REAL_TNUM; type++ ) {
        libGAP_PlainListFuncs [ type ] = libGAP_PlainListError;
    }


    /* install the generic mutability test function                        */
    for ( type = libGAP_FIRST_LIST_TNUM; type <= libGAP_LAST_LIST_TNUM; type += 2 ) {
        libGAP_IsMutableObjFuncs[  type           ] = libGAP_IsMutableListYes;
        libGAP_IsMutableObjFuncs[  type+libGAP_IMMUTABLE ] = libGAP_IsMutableListNo;
        libGAP_IsCopyableObjFuncs[ type           ] = libGAP_IsCopyableListYes;
        libGAP_IsCopyableObjFuncs[ type+libGAP_IMMUTABLE ] = libGAP_IsCopyableListYes;
    }

    /* install the default printers                                        */
    for ( type = libGAP_FIRST_LIST_TNUM; type <= libGAP_LAST_LIST_TNUM; type++ ) {
        libGAP_PrintObjFuncs [ type ] = libGAP_PrintListDefault;
        libGAP_PrintPathFuncs[ type ] = libGAP_PrintPathList;
    }


    /* initialise filter table                                             */
    for ( type = libGAP_FIRST_LIST_TNUM;  type <= libGAP_LAST_LIST_TNUM;  type +=2 ) {
        libGAP_ClearFiltsTNums   [ type            ] = 0;
        libGAP_ClearFiltsTNums   [ type +libGAP_IMMUTABLE ] = 0;
        for ( i = 0;  i <= libGAP_LAST_FN;  i++ ) {
            libGAP_SetFiltListTNums  [ type            ][i] = 0;
            libGAP_SetFiltListTNums  [ type +libGAP_IMMUTABLE ][i] = 0;
            libGAP_ResetFiltListTNums[ type            ][i] = 0;
            libGAP_ResetFiltListTNums[ type +libGAP_IMMUTABLE ][i] = 0;
            libGAP_HasFiltListTNums  [ type            ][i] = -1;
            libGAP_HasFiltListTNums  [ type +libGAP_IMMUTABLE ][i] = -1;
        }
    }

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  PostRestore( <module> ) . . . . . . . . . . . . . after restore workspace
*/
static libGAP_Int libGAP_PostRestore (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* whats that?                                                         */
    libGAP_TYPES_LIST_FAM_RNam = libGAP_RNamName( "TYPES_LIST_FAM" );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitGVarFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitGVarAttrsFromTable( libGAP_GVarAttrs );
    libGAP_InitGVarPropsFromTable( libGAP_GVarProps );
    libGAP_InitGVarOpersFromTable( libGAP_GVarOpers );
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* make and install the 'POS_LIST' operation                           */
    libGAP_HDLR_FUNC( libGAP_PosListOper, 2 ) = libGAP_PosListHandler2;
    libGAP_HDLR_FUNC( libGAP_PosListOper, 3 ) = libGAP_PosListHandler3;

    /* return success                                                      */
    return libGAP_PostRestore( libGAP_module );
}


/****************************************************************************
**
*F  CheckInit( <module> ) . . . . . . . . . . . . . . .  check initialisation
*/
static libGAP_Int libGAP_CheckInit (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_Int         i;              /* loop variable                           */
    libGAP_Int         j;              /* loop variable                           */
    libGAP_Int         success = 1;

    libGAP_Int         fnums[] = { libGAP_FN_IS_MUTABLE, libGAP_FN_IS_EMPTY, libGAP_FN_IS_DENSE,
                            libGAP_FN_IS_NDENSE, libGAP_FN_IS_HOMOG, libGAP_FN_IS_NHOMOG,
                            libGAP_FN_IS_TABLE, libGAP_FN_IS_SSORT, libGAP_FN_IS_NSORT };
    const libGAP_Char *fnams[] = { "mutable", "empty", "dense", "ndense",
                            "homog", "nhomog", "table", "ssort",
                            "nsort" };


    /* fix unknown list types                                              */
    for ( i = libGAP_FIRST_LIST_TNUM;  i <= libGAP_LAST_LIST_TNUM;  i +=2 ) {
        if ( libGAP_InfoBags[i].name == 0 ) {
            libGAP_InfoBags[i].name = "unknown list type";
        }
        if ( libGAP_InfoBags[i+libGAP_IMMUTABLE].name == 0 ) {
            libGAP_InfoBags[i+libGAP_IMMUTABLE].name = "unknown immutable list type";
        }
    }

    /* check that all relevant `ClearFiltListTNums' are installed          */
    for ( i = libGAP_FIRST_LIST_TNUM;  i <= libGAP_LAST_LIST_TNUM;  i++ ) {
        if ( libGAP_ClearFiltsTNums[i] == 0 ) {
            libGAP_Pr( "#W  ClearFiltsListTNums [%s] missing\n",
                    (libGAP_Int)(libGAP_InfoBags[i].name), 0L );
            success = 0;
        }
    }


    /* check that all relevant `HasFiltListTNums' are installed            */
    for ( i = libGAP_FIRST_LIST_TNUM;  i <= libGAP_LAST_LIST_TNUM;  i++ ) {
        for ( j = 0;  j < sizeof(fnums)/sizeof(fnums[1]);  j++ ) {
            if ( libGAP_HasFiltListTNums[i][fnums[j]] == -1 ) {
                libGAP_Pr( "#W  HasFiltListTNums [%s] [%s] missing\n",
                    (libGAP_Int)(libGAP_InfoBags[i].name), (libGAP_Int)fnams[j] );
                success = 0;
                libGAP_HasFiltListTNums[i][fnums[j]] = 0;
            }
        }
    }


    /* check that all relevant `SetFiltListTNums' are installed            */
    for ( i = libGAP_FIRST_LIST_TNUM;  i <= libGAP_LAST_LIST_TNUM;  i++ ) {
        for ( j = 0;  j < sizeof(fnums)/sizeof(fnums[1]);  j++ ) {
            if ( libGAP_SetFiltListTNums[i][fnums[j]] == 0 ) {
                libGAP_Pr( "#W  SetFiltListTNums [%s] [%s] missing\n",
                    (libGAP_Int)(libGAP_InfoBags[i].name), (libGAP_Int)fnams[j] );
                success = 0;
            }
        }
    }


    /* check that all relevant `ResetFiltListTNums' are installed          */
    for ( i = libGAP_FIRST_LIST_TNUM;  i <= libGAP_LAST_LIST_TNUM;  i++ ) {
        for ( j = 0;  j < sizeof(fnums)/sizeof(fnums[1]);  j++ ) {
            if ( libGAP_ResetFiltListTNums[i][fnums[j]] == 0 ) {
                libGAP_Pr( "#W  ResetFiltListTNums [%s] [%s] missing\n",
                    (libGAP_Int)(libGAP_InfoBags[i].name), (libGAP_Int)fnams[j] );
                success = 0;
            }
        }
    }

    /* if a tnum has a filter, reset must change the tnum                  */
    for ( i = libGAP_FIRST_LIST_TNUM;  i <= libGAP_LAST_LIST_TNUM;  i++ ) {
        for ( j = 0;  j < sizeof(fnums)/sizeof(fnums[1]);  j++ ) {
            if ( libGAP_HasFiltListTNums[i][fnums[j]] ) {
                libGAP_Int     new;
                new = libGAP_ResetFiltListTNums[i][fnums[j]];
                if ( new == i ) {
                    continue;   /* filter coded into the representation    */

                }
                else if ( new != -1 && libGAP_HasFiltListTNums[new][fnums[j]] ) {
                    libGAP_Pr(
                     "#W  ResetFiltListTNums [%s] [%s] failed to reset\n",
                     (libGAP_Int)(libGAP_InfoBags[i].name), (libGAP_Int)fnams[j] );
                    success = 0;
                }
            }
        }
    }

    /* if a tnum has a filter, set must not change the tnum                */
    for ( i = libGAP_FIRST_LIST_TNUM;  i <= libGAP_LAST_LIST_TNUM;  i++ ) {
        for ( j = 0;  j < sizeof(fnums)/sizeof(fnums[1]);  j++ ) {
            if ( libGAP_HasFiltListTNums[i][fnums[j]] ) {
                libGAP_Int     new;
                new = libGAP_SetFiltListTNums[i][fnums[j]];
                if ( new != -1 && new != i ) {
                    libGAP_Pr(
                     "#W  SetFiltListTNums [%s] [%s] must not change\n",
                     (libGAP_Int)(libGAP_InfoBags[i].name), (libGAP_Int)fnams[j] );
                    success = 0;
                }
            }
        }
    }

    /* check implications                                                  */
    for ( i = libGAP_FIRST_LIST_TNUM;  i <= libGAP_LAST_LIST_TNUM;  i++ ) {

        if ( libGAP_HasFiltListTNums[i][libGAP_FN_IS_EMPTY] ) {
            if ( ! libGAP_HasFiltListTNums[i][libGAP_FN_IS_DENSE] ) {
                libGAP_Pr(
                 "#W  HasFiltListTNums [%s] [ empty -> dense ] missing\n",
                 (libGAP_Int)(libGAP_InfoBags[i].name), 0L );   
                success = 0;
            }
            if ( libGAP_HasFiltListTNums[i][libGAP_FN_IS_NDENSE] ) {
                libGAP_Pr(
                 "#W  HasFiltListTNums [%s] [ empty + ndense ] illegal\n",
                 (libGAP_Int)(libGAP_InfoBags[i].name), 0L );   
                success = 0;
            }
            if ( ! libGAP_HasFiltListTNums[i][libGAP_FN_IS_HOMOG] ) {
                libGAP_Pr(
                 "#W  HasFiltListTNums [%s] [ empty -> homog ] missing\n",
                 (libGAP_Int)(libGAP_InfoBags[i].name), 0L );
                success = 0;
            }
            if ( libGAP_HasFiltListTNums[i][libGAP_FN_IS_NHOMOG] ) {
                libGAP_Pr(
                 "#W  HasFiltListTNums [%s] [ empty + nhomog ] illegal\n",
                 (libGAP_Int)(libGAP_InfoBags[i].name), 0L );   
                success = 0;
            }
            if ( ! libGAP_HasFiltListTNums[i][libGAP_FN_IS_SSORT] ) {
                libGAP_Pr(
                 "#W  HasFiltListTNums [%s] [ empty -> ssort ] missing\n",
                 (libGAP_Int)(libGAP_InfoBags[i].name), 0L );   
                success = 0;
            }
            if ( libGAP_HasFiltListTNums[i][libGAP_FN_IS_NSORT] ) {
                libGAP_Pr(
                 "#W  HasFiltListTNums [%s] [ empty + nsort ] illegal\n",
                 (libGAP_Int)(libGAP_InfoBags[i].name), 0L );   
                success = 0;
            }
            if ( libGAP_HasFiltListTNums[i][libGAP_FN_IS_TABLE] ) {
                libGAP_Pr(
                 "#W  HasFiltListTNums [%s] [ empty + table ] illegal\n",
                 (libGAP_Int)(libGAP_InfoBags[i].name), 0L );   
                success = 0;
            }
        }

        if ( libGAP_HasFiltListTNums[i][libGAP_FN_IS_DENSE] ) {
            if ( libGAP_HasFiltListTNums[i][libGAP_FN_IS_NDENSE] ) {
                libGAP_Pr(
                 "#W  HasFiltListTNums [%s] [ dense + ndense ] illegal\n",
                 (libGAP_Int)(libGAP_InfoBags[i].name), 0L );   
                success = 0;
            }
        }

        if ( libGAP_HasFiltListTNums[i][libGAP_FN_IS_NDENSE] ) {
            if ( libGAP_HasFiltListTNums[i][libGAP_FN_IS_HOMOG] ) {
                libGAP_Pr(
                 "#W  HasFiltListTNums [%s] [ ndense + homog ] illegal\n",
                 (libGAP_Int)(libGAP_InfoBags[i].name), 0L );   
                success = 0;
            }
            if ( libGAP_HasFiltListTNums[i][libGAP_FN_IS_TABLE] ) {
                libGAP_Pr(
                 "#W  HasFiltListTNums [%s] [ ndense + table ] illegal\n",
                 (libGAP_Int)(libGAP_InfoBags[i].name), 0L );   
                success = 0;
            }
        }

        if ( libGAP_HasFiltListTNums[i][libGAP_FN_IS_HOMOG] ) {
            if ( libGAP_HasFiltListTNums[i][libGAP_FN_IS_NHOMOG] ) {
                libGAP_Pr(
                 "#W  HasFiltListTNums [%s] [ homog + nhomog ] illegal\n",
                 (libGAP_Int)(libGAP_InfoBags[i].name), 0L );   
                success = 0;
            }
            if ( ! libGAP_HasFiltListTNums[i][libGAP_FN_IS_DENSE] ) {
                libGAP_Pr(
                 "#W  HasFiltListTNums [%s] [ homog -> dense ] missing\n",
                 (libGAP_Int)(libGAP_InfoBags[i].name), 0L );   
                success = 0;
            }
            if ( libGAP_HasFiltListTNums[i][libGAP_FN_IS_NDENSE] ) {
                libGAP_Pr(
                 "#W  HasFiltListTNums [%s] [ homog + ndense ] illegal\n",
                 (libGAP_Int)(libGAP_InfoBags[i].name), 0L );   
                success = 0;
            }
        }

        if ( libGAP_HasFiltListTNums[i][libGAP_FN_IS_NHOMOG] ) {
            if ( libGAP_HasFiltListTNums[i][libGAP_FN_IS_TABLE] ) {
                libGAP_Pr(
                 "#W  HasFiltListTNums [%s] [ nhomog + table ] illegal\n",
                 (libGAP_Int)(libGAP_InfoBags[i].name), 0L );   
                success = 0;
            }
        }

        if ( libGAP_HasFiltListTNums[i][libGAP_FN_IS_TABLE] ) {
            if ( ! libGAP_HasFiltListTNums[i][libGAP_FN_IS_HOMOG] ) {
                libGAP_Pr(
                 "#W  HasFiltListTNums [%s] [ table -> homog ] missing\n",
                 (libGAP_Int)(libGAP_InfoBags[i].name), 0L );   
                success = 0;
            }
            if ( ! libGAP_HasFiltListTNums[i][libGAP_FN_IS_DENSE] ) {
                libGAP_Pr(
                 "#W  HasFiltListTNums [%s] [ table -> dense ] missing\n",
                 (libGAP_Int)(libGAP_InfoBags[i].name), 0L );   
                success = 0;
            }
        }

        if ( libGAP_HasFiltListTNums[i][libGAP_FN_IS_SSORT] ) {
            if ( libGAP_HasFiltListTNums[i][libGAP_FN_IS_NSORT] ) {
                libGAP_Pr(
                 "#W  HasFiltListTNums [%s] [ ssort + nsort ] illegal\n",
                 (libGAP_Int)(libGAP_InfoBags[i].name), 0L );   
                success = 0;
            }
        }           
    }

    /* return success                                                      */
    return ! success;
}


/****************************************************************************
**
*F  InitInfoLists() . . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "lists",                            /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    libGAP_CheckInit,                          /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    libGAP_PostRestore                         /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoLists ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}


/****************************************************************************
**

*E  lists.c . . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/



