/****************************************************************************
**
*W  ariths.c                    GAP source                       Frank Celler
*W                                                         & Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions of the  arithmetic  operations  package.
*/
#include        "system.h"              /* system dependent part           */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */

#include        "gvars.h"               /* global variables                */
#include        "calls.h"               /* generic call mechanism          */
#include        "opers.h"               /* generic operations              */

#include        "ariths.h"              /* basic arithmetic                */

#include        "bool.h"                /* booleans                        */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "string.h"              /* strings                         */


/****************************************************************************
**

*F * * * * * * * * * * *  unary arithmetic operations * * * * * * * * * * * *
*/

/****************************************************************************
**
*V  ZeroFuncs[ <type> ] . . . . . . . . . . . . . . . . table of zero methods
*/
libGAP_ArithMethod1 libGAP_ZeroFuncs [libGAP_LAST_VIRTUAL_TNUM+1];


/****************************************************************************
**
*F  ZeroObject( <obj> ) . . . . . . . . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_ZEROOp;

libGAP_Obj libGAP_ZeroObject (
    libGAP_Obj                 obj )

{
  libGAP_Obj val;
  val = libGAP_DoOperation1Args( libGAP_ZEROOp, obj );
  while (val == 0)
    val = libGAP_ErrorReturnObj("ZEROOp: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  VerboseZeroObject( <obj> )  . . . . . . . . . . . .  call verbose methsel
*/
libGAP_Obj libGAP_VerboseZeroObject (
    libGAP_Obj                 obj )

{
  libGAP_Obj val;
  val = libGAP_DoVerboseOperation1Args( libGAP_ZEROOp, obj );
  while (val == 0)
    val = libGAP_ErrorReturnObj("ZEROOp: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  InstallZeroObject( <verb> ) . . . . . . . . . . . .  install zero methods
*/
void libGAP_InstallZeroObject ( libGAP_Int verb )
{
    libGAP_UInt                t1;             /* type of left  operand           */
    libGAP_ArithMethod1        func;           /* zero function                   */

    func = ( verb ? libGAP_VerboseZeroObject : libGAP_ZeroObject );
    for ( t1 = libGAP_FIRST_EXTERNAL_TNUM; t1 <= libGAP_LAST_EXTERNAL_TNUM; t1++ ) {
        libGAP_ZeroFuncs[t1] = func;
    }
    libGAP_ZeroFuncs[ libGAP_T_PREC            ] = func;
    libGAP_ZeroFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ] = func;
}


/****************************************************************************
**
*F  FuncZERO( <self>, <obj> ) . . . . . . . . . . . . . . . . . . call 'ZERO'
*/
libGAP_Obj libGAP_FuncZERO (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return libGAP_ZERO(obj);
}

/****************************************************************************
**
*V  ZeroMutFuncs[ <type> ] . . . . . . . . . . . . . . . . table of zero methods
*/
libGAP_ArithMethod1 libGAP_ZeroMutFuncs [libGAP_LAST_VIRTUAL_TNUM+1];


/****************************************************************************
**
*F  ZeroMutObject( <obj> ) . . . . . . . . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_ZeroOp;

libGAP_Obj libGAP_ZeroMutObject (
    libGAP_Obj                 obj )

{
  libGAP_Obj val;
  val = libGAP_DoOperation1Args( libGAP_ZeroOp, obj );
  while (val == 0)
    val = libGAP_ErrorReturnObj("ZeroOp: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  VerboseZeroMutObject( <obj> )  . . . . . . . . . . . .  call verbose methsel
*/
libGAP_Obj libGAP_VerboseZeroMutObject (
    libGAP_Obj                 obj )

{
  libGAP_Obj val;
  val = libGAP_DoVerboseOperation1Args( libGAP_ZeroOp, obj );
  while (val == 0)
    val = libGAP_ErrorReturnObj("ZeroOp: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  InstallZeroMutObject( <verb> ) . . . . . . . . . . . .  install zero methods
*/
void libGAP_InstallZeroMutObject ( libGAP_Int verb )
{
    libGAP_UInt                t1;             /* type of left  operand           */
    libGAP_ArithMethod1        func;           /* zero function                   */

    func = ( verb ? libGAP_VerboseZeroMutObject : libGAP_ZeroMutObject );
    for ( t1 = libGAP_FIRST_EXTERNAL_TNUM; t1 <= libGAP_LAST_EXTERNAL_TNUM; t1++ ) {
        libGAP_ZeroMutFuncs[t1] = func;
    }
    libGAP_ZeroMutFuncs[ libGAP_T_PREC            ] = func;
    libGAP_ZeroMutFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ] = func;
}


/****************************************************************************
**
*F  FuncZERO_MUT( <self>, <obj> ) . . . . . . . . . . . . . . call 'ZERO_MUT'
*/
libGAP_Obj libGAP_FuncZERO_MUT (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return libGAP_ZERO_MUT(obj);
}


/****************************************************************************
**
*V  AInvFuncs[ <type> ] . . . . . . . . . . table of additive inverse methods
*V  AInvMutFuncs[ <type> ] . .  . . . . . . table of additive inverse methods
**                                          which return mutable results
*/
libGAP_ArithMethod1 libGAP_AInvFuncs [libGAP_LAST_VIRTUAL_TNUM+1];
libGAP_ArithMethod1 libGAP_AInvMutFuncs[ libGAP_LAST_VIRTUAL_TNUM + 1];


/****************************************************************************
**
*F  AInvObj( <obj> )  . . . . . . . . . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_AInvOp;

libGAP_Obj libGAP_AInvObject (
    libGAP_Obj                 obj )
{
  libGAP_Obj val;
  val = libGAP_DoOperation1Args( libGAP_AInvOp, obj );
  while (val == 0)
    val = libGAP_ErrorReturnObj("AInvOp: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  VerboseAInvObject( <obj> )  . . . . . . . . . . . .  call verbose methsel
*/
libGAP_Obj libGAP_VerboseAInvObject (
    libGAP_Obj                 obj )
{
  libGAP_Obj val;
  val = libGAP_DoVerboseOperation1Args( libGAP_AInvOp, obj );
  while (val == 0)
    val = libGAP_ErrorReturnObj("AInvOp: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  InstallAinvObject( <verb> ) . . . . . .  install additive inverse methods
*/
void libGAP_InstallAinvObject ( libGAP_Int verb )
{
    libGAP_UInt                t1;             /* type of left  operand           */
    libGAP_ArithMethod1        func;           /* ainv function                   */

    func = ( verb ? libGAP_VerboseAInvObject : libGAP_AInvObject );
    for ( t1 = libGAP_FIRST_EXTERNAL_TNUM; t1 <= libGAP_LAST_EXTERNAL_TNUM; t1++ ) {
        libGAP_AInvFuncs[t1] = func;
    }
    libGAP_AInvFuncs[ libGAP_T_PREC            ] = func;
    libGAP_AInvFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ] = func;
}


/****************************************************************************
**
*F  FuncAINV( <self>, <obj> ) . . . . . . . . . . . . . . . . . . call 'AINV'
*/
libGAP_Obj libGAP_FuncAINV (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return libGAP_AINV(obj);
}

/****************************************************************************
**
*F  AInvMutObject( <obj> )  . .. . . . . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_AdditiveInverseOp;

libGAP_Obj libGAP_AInvMutObject (
    libGAP_Obj                 obj )
{
  libGAP_Obj val;
  val = libGAP_DoOperation1Args( libGAP_AdditiveInverseOp, obj );
  while (val == 0)
    val = libGAP_ErrorReturnObj("AdditiveInverseOp: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  VerboseAInvMutObject( <obj> )  . . . . . . . . . . . .  call verbose methsel
*/
libGAP_Obj libGAP_VerboseAInvMutObject (
    libGAP_Obj                 obj )
{
  libGAP_Obj val;
  val = libGAP_DoVerboseOperation1Args( libGAP_AdditiveInverseOp, obj );
  while (val == 0)
    val = libGAP_ErrorReturnObj("AdditiveInverseOp: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  InstallAinvMutObject( <verb> ) . . . . . .  install additive inverse methods
*/
void libGAP_InstallAinvMutObject ( libGAP_Int verb )
{
    libGAP_UInt                t1;             /* type of left  operand           */
    libGAP_ArithMethod1        func;           /* ainv function                   */

    func = ( verb ? libGAP_VerboseAInvMutObject : libGAP_AInvMutObject );
    for ( t1 = libGAP_FIRST_EXTERNAL_TNUM; t1 <= libGAP_LAST_EXTERNAL_TNUM; t1++ ) {
        libGAP_AInvMutFuncs[t1] = func;
    }
    libGAP_AInvMutFuncs[ libGAP_T_PREC            ] = func;
    libGAP_AInvMutFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ] = func;
}


/****************************************************************************
**
*F  FuncAINV_MUT( <self>, <obj> ) . . . . . . . . . . . . . . . . . . call 'AINV'
*/
libGAP_Obj libGAP_FuncAINV_MUT (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return libGAP_AINV_MUT(obj);
}


/****************************************************************************
**

*V  OneFuncs[ <type> ]  . . . . . . . . . . . . . . . .  table of one methods
*/
libGAP_ArithMethod1 libGAP_OneFuncs [libGAP_LAST_VIRTUAL_TNUM+1];


/****************************************************************************
**
*F  OneObject( <obj> )  . . . . . . . . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_OneOp;

libGAP_Obj libGAP_OneObject (
    libGAP_Obj                 obj )
{
  libGAP_Obj val;
  val = libGAP_DoOperation1Args( libGAP_OneOp, obj );
  while (val == 0)
    val = libGAP_ErrorReturnObj("OneOp: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  VerboseOneObject( <obj> ) . . . . . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_VerboseOneObject (
    libGAP_Obj                 obj )
{
  libGAP_Obj val;
  val = libGAP_DoVerboseOperation1Args( libGAP_OneOp, obj );
  while (val == 0)
    val = libGAP_ErrorReturnObj("OneOp: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  InstallOneObject( <verb> )  . . . . . . . . . . . . . install one methods
*/
void libGAP_InstallOneObject ( libGAP_Int verb )
{
    libGAP_UInt                t1;             /* type of left  operand           */
    libGAP_ArithMethod1        func;           /* one function                    */

    func = ( verb ? libGAP_VerboseOneObject : libGAP_OneObject );
    for ( t1 = libGAP_FIRST_EXTERNAL_TNUM; t1 <= libGAP_LAST_EXTERNAL_TNUM; t1++ ) {
        libGAP_OneFuncs[t1] = func;
    }
    libGAP_OneFuncs[ libGAP_T_PREC            ] = func;
    libGAP_OneFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ] = func;
}


/****************************************************************************
**
*F  FuncONE( <self>, <obj> ) . . . . . . . . . . . . . . . . .  call 'ONE'
*/
libGAP_Obj libGAP_FuncONE (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return libGAP_ONE(obj);
}

/****************************************************************************
**

*V  OneMutFuncs[ <type> ]  . . . . .table of mutability retaining one methods
*/
libGAP_ArithMethod1 libGAP_OneMutFuncs [libGAP_LAST_VIRTUAL_TNUM+1];


/****************************************************************************
**
*F  OneMutObject( <obj> )  . . . . . . . . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_OneMutOp;

libGAP_Obj libGAP_OneMutObject (
    libGAP_Obj                 obj )
{
  libGAP_Obj val;
  val = libGAP_DoOperation1Args( libGAP_OneMutOp, obj );
  while (val == 0)
    val = libGAP_ErrorReturnObj("ONEOp: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  VerboseOneMutObject( <obj> ) . . .  . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_VerboseOneMutObject (
    libGAP_Obj                 obj )
{
  libGAP_Obj val;
  val = libGAP_DoVerboseOperation1Args( libGAP_OneMutOp, obj );
  while (val == 0)
    val = libGAP_ErrorReturnObj("ONEOp: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  InstallOneMutObject( <verb> )  . . . . . . . . . . . . . install one methods
*/
void libGAP_InstallOneMutObject ( libGAP_Int verb )
{
    libGAP_UInt                t1;             /* type of left  operand           */
    libGAP_ArithMethod1        func;           /* one function                    */

    func = ( verb ? libGAP_VerboseOneMutObject : libGAP_OneMutObject );
    for ( t1 = libGAP_FIRST_EXTERNAL_TNUM; t1 <= libGAP_LAST_EXTERNAL_TNUM; t1++ ) {
        libGAP_OneMutFuncs[t1] = func;
    }
    libGAP_OneMutFuncs[ libGAP_T_PREC            ] = func;
    libGAP_OneMutFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ] = func;
}


/****************************************************************************
**
*F  FuncONE_MUT( <self>, <obj> ) . . . . . . . . . . . . . . . .call 'ONE_MUT'
*/
libGAP_Obj libGAP_FuncONE_MUT (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return libGAP_ONE_MUT(obj);
}


/****************************************************************************
**

*V  InvFuncs[ <type> ]  . . . . . . . . . . . . .  table of inverse functions
*/
libGAP_ArithMethod1 libGAP_InvFuncs [libGAP_LAST_VIRTUAL_TNUM+1];

    
/****************************************************************************
**
*F  InvObject( <obj> )  . . . . . . . . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_InvOp;

libGAP_Obj libGAP_InvObject (
    libGAP_Obj                 obj )
{
  libGAP_Obj val;
  val = libGAP_DoOperation1Args( libGAP_InvOp, obj );
  while (val == 0)
    val = libGAP_ErrorReturnObj("InvOp: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  VerboseInvObject( <obj> ) . . . . . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_VerboseInvObject (
    libGAP_Obj                 obj )
{
  libGAP_Obj val;
  val = libGAP_DoVerboseOperation1Args( libGAP_InvOp, obj );
  while (val == 0)
    val = libGAP_ErrorReturnObj("InvOp: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  InstallInvObject( <verb> )  . . . . . . . . . . . install inverse methods
*/
void libGAP_InstallInvObject ( libGAP_Int verb )
{
    libGAP_UInt                t1;             /* type of left  operand           */
    libGAP_ArithMethod1        func;           /* inv function                    */

    func = ( verb ? libGAP_VerboseInvObject : libGAP_InvObject );
    for ( t1 = libGAP_FIRST_EXTERNAL_TNUM; t1 <= libGAP_LAST_EXTERNAL_TNUM; t1++ ) {
        libGAP_InvFuncs[t1] = func;
    }
    libGAP_InvFuncs[ libGAP_T_PREC            ] = func;
    libGAP_InvFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ] = func;
}


/****************************************************************************
**
*F  FuncINV( <self>, <obj> )  . . . . . . . . . . . . . . . . . .  call 'INV'
*/
libGAP_Obj libGAP_FuncINV (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return libGAP_INV( obj );
}


/****************************************************************************
**
*V  InvMutFuncs[ <type> ]  . table of mutability-preserving inverse functions
*/
libGAP_ArithMethod1 libGAP_InvMutFuncs [libGAP_LAST_VIRTUAL_TNUM+1];

    
/****************************************************************************
**
*F  InvMutObject( <obj> )  . . . . . . . . . . . . . . .. . . . .  call methsel
*/
libGAP_Obj libGAP_InvMutOp;

libGAP_Obj libGAP_InvMutObject (
    libGAP_Obj                 obj )
{
  libGAP_Obj val;
  val = libGAP_DoOperation1Args( libGAP_InvMutOp, obj );
  while (val == 0)
    val = libGAP_ErrorReturnObj("INVOp: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  VerboseInvMutObject( <obj> ) . . .  . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_VerboseInvMutObject (
    libGAP_Obj                 obj )
{
  libGAP_Obj val;
  val = libGAP_DoVerboseOperation1Args( libGAP_InvMutOp, obj );
  while (val == 0)
    val = libGAP_ErrorReturnObj("INVOp: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  InstallInvMutObject( <verb> ) install mutability preserving inverse methods
*/
void libGAP_InstallInvMutObject ( libGAP_Int verb )
{
    libGAP_UInt                t1;             /* type of left  operand           */
    libGAP_ArithMethod1        func;           /* inv function                    */

    func = ( verb ? libGAP_VerboseInvMutObject : libGAP_InvMutObject );
    for ( t1 = libGAP_FIRST_EXTERNAL_TNUM; t1 <= libGAP_LAST_EXTERNAL_TNUM; t1++ ) {
        libGAP_InvMutFuncs[t1] = func;
    }
    libGAP_InvMutFuncs[ libGAP_T_PREC            ] = func;
    libGAP_InvMutFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ] = func;
}


/****************************************************************************
**
*F  FuncINV_MUT( <self>, <obj> )  . . .  . . . . . . . . . .  call 'INV_MUT'
*/
libGAP_Obj libGAP_FuncINV_MUT (
    libGAP_Obj                 self,
    libGAP_Obj                 obj )
{
    return libGAP_INV_MUT( obj );
}


/****************************************************************************
**

*F * * * * * * * * * * * * * comparison operations  * * * * * * * * * * * * *
*/

/****************************************************************************
**

*V  EqFuncs[ <typeL> ][ <typeR> ] . . . . . . . . table of comparison methods
*/
libGAP_CompaMethod libGAP_EqFuncs [libGAP_LAST_VIRTUAL_TNUM+1][libGAP_LAST_VIRTUAL_TNUM+1];


/****************************************************************************
**
*F  EqNot( <opL>, <opR> ) . . . . . . . . . . . . . . . . . . . . . not equal
*/
libGAP_Int libGAP_EqNot (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    return 0L;
}


/****************************************************************************
**
*F  EqObject( <opL>, <opR> )  . . . . . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_EqOper;

libGAP_Int libGAP_EqObject (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    return (libGAP_DoOperation2Args( libGAP_EqOper, opL, opR ) == libGAP_True);
}


/****************************************************************************
**
*F  VerboseEqObject( <opL>, <opR> ) . . . . . . . . . . . . . .  call methsel
*/
libGAP_Int libGAP_VerboseEqObject (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    return (libGAP_DoVerboseOperation2Args( libGAP_EqOper, opL, opR ) == libGAP_True);
}


/****************************************************************************
**
*F  InstallEqObject( <verb> ) . . . . . . . . . .  install comparison methods
*/
void libGAP_InstallEqObject ( libGAP_Int verb )
{
    libGAP_UInt                t1;             /* type of left  operand           */
    libGAP_UInt                t2;             /* type of right operand           */
    libGAP_CompaMethod         func;           /* equal function                  */

    func = ( verb ? libGAP_VerboseEqObject : libGAP_EqObject );
    for ( t1 = libGAP_FIRST_EXTERNAL_TNUM;  t1 <= libGAP_LAST_EXTERNAL_TNUM;  t1++ ) {
        for ( t2 = libGAP_FIRST_REAL_TNUM;  t2 <= libGAP_LAST_VIRTUAL_TNUM;  t2++ ) {
            libGAP_EqFuncs[t1][t2] = func;
            libGAP_EqFuncs[t2][t1] = func;
        }
    }
    for ( t2 = libGAP_FIRST_REAL_TNUM;  t2 <= libGAP_LAST_VIRTUAL_TNUM;  t2++ ) {

        libGAP_EqFuncs[ t2 ][ libGAP_T_PREC            ] = func;
        libGAP_EqFuncs[ t2 ][ libGAP_T_PREC +libGAP_IMMUTABLE ] = func;
        libGAP_EqFuncs[ libGAP_T_PREC            ][ t2 ] = func;
        libGAP_EqFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ][ t2 ] = func;
    }
}


/****************************************************************************
**
*F  FuncEQ( <self>, <opL>, <opR> )  . . . . . . . . . . . . . . . . call 'EQ'
*/
libGAP_Obj libGAP_FuncEQ (
    libGAP_Obj                 self,
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
  /* if both operands are T_MACFLOAT, we use the comparison method in all cases,
     even if the objects are identical. In this manner, we can have 0./0. != 0./0. as
     the IEEE754 standard requires.
    if (TNUM_OBJ(opL) == T_MACFLOAT && TNUM_OBJ(opR) == T_MACFLOAT)
      return (*EqFuncs[T_MACFLOAT][T_MACFLOAT])(opL,opR) ? True : False;
  */

    return (libGAP_EQ( opL, opR ) ? libGAP_True : libGAP_False);
}


/****************************************************************************
**

*V  LtFuncs[ <typeL> ][ <typeR> ] . . . . . . . . table of comparison methods
*/
libGAP_CompaMethod libGAP_LtFuncs [libGAP_LAST_VIRTUAL_TNUM+1][libGAP_LAST_VIRTUAL_TNUM+1];


/****************************************************************************
**
*F  LtObject( <opL>, <opR> )  . . . . . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_LtOper;

libGAP_Int libGAP_LtObject (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    return (libGAP_DoOperation2Args( libGAP_LtOper, opL, opR ) == libGAP_True);
}


/****************************************************************************
**
*F  VerboseLtObject( <opL>, <opR> ) . . . . . . . . . . . . . .  call methsel
*/
libGAP_Int libGAP_VerboseLtObject (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    return (libGAP_DoVerboseOperation2Args( libGAP_LtOper, opL, opR ) == libGAP_True);
}


/****************************************************************************
**
*F  InstallLtObject( <verb> ) . . . . . . . . . . . install less than methods
*/
void libGAP_InstallLtObject ( libGAP_Int verb )
{
    libGAP_UInt                t1;             /* type of left  operand           */
    libGAP_UInt                t2;             /* type of right operand           */
    libGAP_CompaMethod         func;           /* less than function              */

    func = ( verb ? libGAP_VerboseLtObject : libGAP_LtObject );
    for ( t1 = libGAP_FIRST_EXTERNAL_TNUM;  t1 <= libGAP_LAST_EXTERNAL_TNUM;  t1++ ) {
        for ( t2 = libGAP_FIRST_REAL_TNUM;  t2 <= libGAP_LAST_VIRTUAL_TNUM;  t2++ ) {
            libGAP_LtFuncs[t1][t2] = func;
            libGAP_LtFuncs[t2][t1] = func;
        }
    }
    for ( t2 = libGAP_FIRST_REAL_TNUM;  t2 <= libGAP_LAST_VIRTUAL_TNUM;  t2++ ) {

        libGAP_LtFuncs[ t2 ][ libGAP_T_PREC            ] = func;
        libGAP_LtFuncs[ t2 ][ libGAP_T_PREC +libGAP_IMMUTABLE ] = func;
        libGAP_LtFuncs[ libGAP_T_PREC            ][ t2 ] = func;
        libGAP_LtFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ][ t2 ] = func;
    }
}


/****************************************************************************
**
*F  FuncLT( <self>, <opL>, <opR> )  . . . . . . . . . . . . . . . . call 'LT'
*/
libGAP_Obj libGAP_FuncLT (
    libGAP_Obj                 self,
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    return (libGAP_LT( opL, opR ) ? libGAP_True : libGAP_False);
}


/****************************************************************************
**

*V  InFuncs[ <typeL> ][ <typeR> ] . . . . . . . . table of membership methods
*/
libGAP_CompaMethod libGAP_InFuncs [libGAP_LAST_VIRTUAL_TNUM+1][libGAP_LAST_VIRTUAL_TNUM+1];


/****************************************************************************
**
*F  InUndefined( <self>, <opL>, <opR> ) . . . . . . . . . . . . . cannot 'in'
*/
libGAP_Int libGAP_InUndefined (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    return (libGAP_ErrorReturnObj(
        "operations: IN of %s and %s is not defined",
        (libGAP_Int)libGAP_TNAM_OBJ(opL),
        (libGAP_Int)libGAP_TNAM_OBJ(opR),
        "you can 'return <boolean>;' to give a value for the result" ) == libGAP_True);
}


/****************************************************************************
**
*F  InObject( <opL>, <opR> )  . . . . . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_InOper;

libGAP_Int libGAP_InObject (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    return (libGAP_DoOperation2Args( libGAP_InOper, opL, opR ) == libGAP_True);
}


/****************************************************************************
**
*F  VerboseInObject( <opL>, <opR> ) . . . . . . . . . . . . . .  call methsel
*/
libGAP_Int libGAP_VerboseInObject (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    return (libGAP_DoVerboseOperation2Args( libGAP_InOper, opL, opR ) == libGAP_True);
}


/****************************************************************************
**
*F  InstallInObject( <verb> ) . . . . . . . . . . . . . .  install in methods
*/
void libGAP_InstallInObject ( libGAP_Int verb )
{
    libGAP_UInt                t1;             /* type of left  operand           */
    libGAP_UInt                t2;             /* type of right operand           */
    libGAP_CompaMethod         func;           /* in function                     */

    func = ( verb ? libGAP_VerboseInObject : libGAP_InObject );
    for ( t1 = libGAP_FIRST_EXTERNAL_TNUM;  t1 <= libGAP_LAST_EXTERNAL_TNUM;  t1++ ) {
        for ( t2 = libGAP_FIRST_REAL_TNUM;  t2 <= libGAP_LAST_VIRTUAL_TNUM;  t2++ ) {
            libGAP_InFuncs[t1][t2] = func;
            libGAP_InFuncs[t2][t1] = func;
        }
    }
    for ( t2 = libGAP_FIRST_REAL_TNUM;  t2 <= libGAP_LAST_VIRTUAL_TNUM;  t2++ ) {

        libGAP_InFuncs[ t2 ][ libGAP_T_PREC            ] = func;
        libGAP_InFuncs[ t2 ][ libGAP_T_PREC +libGAP_IMMUTABLE ] = func;
    }
}


/****************************************************************************
**
*F  FuncIN( <self>, <opL>, <opR> )  . . . . . . . . . . . . . . . . call 'IN'
*/
libGAP_Obj libGAP_FuncIN (
    libGAP_Obj                 self,
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    return (libGAP_IN( opL, opR ) ? libGAP_True : libGAP_False);
}


/****************************************************************************
**

*F * * * * * * * * * * * binary arithmetic operations * * * * * * * * * * * *
*/

/****************************************************************************
**

*V  SumFuncs[ <typeL> ][ <typeR> ]  . . . . . . . . . .  table of sum methods
*/
libGAP_ArithMethod2    libGAP_SumFuncs [libGAP_LAST_VIRTUAL_TNUM+1][libGAP_LAST_VIRTUAL_TNUM+1];


/****************************************************************************
**
*F  SumObject( <opL>, <opR> ) . . . . . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_SumOper;

libGAP_Obj libGAP_SumObject (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
  libGAP_Obj val;
  val = libGAP_DoOperation2Args( libGAP_SumOper, opL, opR );
  while (val == 0)
    val = libGAP_ErrorReturnObj("SUM: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  VerboseSumObject( <opL>, <opR> )  . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_VerboseSumObject (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
  libGAP_Obj val;
  val = libGAP_DoVerboseOperation2Args( libGAP_SumOper, opL, opR );
  while (val == 0)
    val = libGAP_ErrorReturnObj("SUM: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  InstallSumObject( <verb> )  . . . . . . . . . . . . . install sum methods
*/
void libGAP_InstallSumObject ( libGAP_Int verb )
{
    libGAP_UInt                t1;             /* type of left  operand           */
    libGAP_UInt                t2;             /* type of right operand           */
    libGAP_ArithMethod2        func;           /* sum function                    */

    func = ( verb ? libGAP_VerboseSumObject : libGAP_SumObject );
    for ( t1 = libGAP_FIRST_EXTERNAL_TNUM; t1 <= libGAP_LAST_EXTERNAL_TNUM; t1++ ) {
        for ( t2 = libGAP_FIRST_REAL_TNUM; t2 <= libGAP_LAST_VIRTUAL_TNUM; t2++ ) {
            libGAP_SumFuncs[t1][t2] = func;
            libGAP_SumFuncs[t2][t1] = func;
        }
    }
    for ( t2 = libGAP_FIRST_REAL_TNUM;  t2 <= libGAP_LAST_VIRTUAL_TNUM; t2++ ) {

        libGAP_SumFuncs[ t2 ][ libGAP_T_PREC            ] = func;
        libGAP_SumFuncs[ t2 ][ libGAP_T_PREC +libGAP_IMMUTABLE ] = func;
        libGAP_SumFuncs[ libGAP_T_PREC            ][ t2 ] = func;
        libGAP_SumFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ][ t2 ] = func;
    }
}


/****************************************************************************
**
*F  FuncSUM( <self>, <opL>, <opR> ) . . . . . . . . . . . . . . .  call 'SUM'
*/
libGAP_Obj libGAP_FuncSUM (
    libGAP_Obj                 self,
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    return libGAP_SUM( opL, opR );
}


/****************************************************************************
**

*V  DiffFuncs[ <typeL> ][ <typeR> ] . . . . . . . table of difference methods
*/
libGAP_ArithMethod2 libGAP_DiffFuncs [libGAP_LAST_VIRTUAL_TNUM+1][libGAP_LAST_VIRTUAL_TNUM+1];


/****************************************************************************
**
*F  DiffDefault( <opL>, <opR> ) . . . . . . . . . . . . call 'SUM' and 'AINV'
*/
libGAP_Obj libGAP_DiffDefault (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 tmp;

    tmp = libGAP_AINV( opR );
    return libGAP_SUM( opL, tmp );
}


/****************************************************************************
**
*F  DiffObject( <opL>, <opR> )  . . . . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_DiffOper;

libGAP_Obj libGAP_DiffObject (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
  libGAP_Obj val;
  val = libGAP_DoOperation2Args( libGAP_DiffOper, opL, opR );
  while (val == 0)
    val = libGAP_ErrorReturnObj("DIFF: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  VerboseDiffObject( <opL>, <opR> ) . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_VerboseDiffObject (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
  libGAP_Obj val;
  val = libGAP_DoVerboseOperation2Args( libGAP_DiffOper, opL, opR );
  while (val == 0)
    val = libGAP_ErrorReturnObj("DIFF: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  InstallDiffObject( <verb> ) . . . . . . . . .  install difference methods
*/
void libGAP_InstallDiffObject ( libGAP_Int verb )
{
    libGAP_UInt                t1;             /* type of left  operand           */
    libGAP_UInt                t2;             /* type of right operand           */
    libGAP_ArithMethod2        func;           /* difference function             */

    func = ( verb ? libGAP_VerboseDiffObject : libGAP_DiffObject );
    for ( t1 = libGAP_FIRST_EXTERNAL_TNUM;  t1 <= libGAP_LAST_EXTERNAL_TNUM;  t1++ ) {
        for ( t2 = libGAP_FIRST_REAL_TNUM;  t2 <= libGAP_LAST_VIRTUAL_TNUM;  t2++ ) {
            libGAP_DiffFuncs[t1][t2] = func;
            libGAP_DiffFuncs[t2][t1] = func;
        }
    }
    for ( t2 = libGAP_FIRST_REAL_TNUM; t2 <= libGAP_LAST_VIRTUAL_TNUM; t2++ ) {

        libGAP_DiffFuncs[ t2 ][ libGAP_T_PREC            ] = func;
        libGAP_DiffFuncs[ t2 ][ libGAP_T_PREC +libGAP_IMMUTABLE ] = func;
        libGAP_DiffFuncs[ libGAP_T_PREC            ][ t2 ] = func;
        libGAP_DiffFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ][ t2 ] = func;
    }
}


/****************************************************************************
**
*F  FuncDIFF_DEFAULT( <self>, <opL>, <opR> )  . . . . . .  call 'DiffDefault'
*/
libGAP_Obj libGAP_DiffDefaultFunc;

libGAP_Obj libGAP_FuncDIFF_DEFAULT (
    libGAP_Obj                 self,
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    return libGAP_DiffDefault( opL, opR );
}


/****************************************************************************
**
*F  FuncDIFF( <self>, <opL>, <opR> )  . . . . . . . . . . . . . . call 'DIFF'
*/
libGAP_Obj libGAP_FuncDIFF (
    libGAP_Obj                 self,
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    return libGAP_DIFF( opL, opR );
}


/****************************************************************************
**

*V  ProdFuncs[ <typeL> ][ <typeR> ] . . . . . . . .  table of product methods
*/
libGAP_ArithMethod2    libGAP_ProdFuncs [libGAP_LAST_VIRTUAL_TNUM+1][libGAP_LAST_VIRTUAL_TNUM+1];


/****************************************************************************
**
*F  ProdObject( <opL>, <opR> )  . . . . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_ProdOper;

libGAP_Obj libGAP_ProdObject (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
  libGAP_Obj val;
  val = libGAP_DoOperation2Args( libGAP_ProdOper, opL, opR );
  while (val == 0)
    val = libGAP_ErrorReturnObj("PROD: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  VerboseProdObject( <opL>, <opR> ) . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_VerboseProdObject (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
  libGAP_Obj val;
  val = libGAP_DoVerboseOperation2Args( libGAP_ProdOper, opL, opR );
  while (val == 0)
    val = libGAP_ErrorReturnObj("PROD: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  InstallProdObject( <verb> ) . . . . . . . . . . . install product methods
*/
void libGAP_InstallProdObject ( libGAP_Int verb )
{
    libGAP_UInt                t1;             /* type of left  operand           */
    libGAP_UInt                t2;             /* type of right operand           */
    libGAP_ArithMethod2        func;           /* product function                */

    func = ( verb ? libGAP_VerboseProdObject : libGAP_ProdObject );
    for ( t1 = libGAP_FIRST_EXTERNAL_TNUM;  t1 <= libGAP_LAST_EXTERNAL_TNUM;  t1++ ) {
        for ( t2 = libGAP_FIRST_REAL_TNUM;  t2 <= libGAP_LAST_VIRTUAL_TNUM;  t2++ ) {
            libGAP_ProdFuncs[t1][t2] = func;
            libGAP_ProdFuncs[t2][t1] = func;
        }
    }
    for ( t2 = libGAP_FIRST_REAL_TNUM; t2 <= libGAP_LAST_VIRTUAL_TNUM; t2++ ) {

        libGAP_ProdFuncs[ t2 ][ libGAP_T_PREC            ] = func;
        libGAP_ProdFuncs[ t2 ][ libGAP_T_PREC +libGAP_IMMUTABLE ] = func;
        libGAP_ProdFuncs[ libGAP_T_PREC            ][ t2 ] = func;
        libGAP_ProdFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ][ t2 ] = func;
    }
}


/****************************************************************************
**
*F  FuncPROD( <self>, <opL>, <opR> )  . . . . . . . . . . . . . . call 'PROD'
*/
libGAP_Obj libGAP_FuncPROD (
    libGAP_Obj                 self,
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    return libGAP_PROD( opL, opR );
}


/****************************************************************************
**

*V  QuoFuncs[ <typeL> ][ <typeR> ]  . . . . . . . . table of quotient methods
*/
libGAP_ArithMethod2 libGAP_QuoFuncs [libGAP_LAST_VIRTUAL_TNUM+1][libGAP_LAST_VIRTUAL_TNUM+1];


/****************************************************************************
**
*F  QuoDefault( <opL>, <opR> )  . . . . . . . . . . . . call 'INV' and 'PROD'
*/
libGAP_Obj libGAP_QuoDefault (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 tmp;
    tmp = libGAP_INV_MUT( opR );
    return libGAP_PROD( opL, tmp );
}


/****************************************************************************
**
*F  QuoObject( <opL>, <opR> ) . . . . . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_QuoOper;

libGAP_Obj libGAP_QuoObject (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
  libGAP_Obj val;
  val = libGAP_DoOperation2Args( libGAP_QuoOper, opL, opR );
  while (val == 0)
    val = libGAP_ErrorReturnObj("QUO: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  VerboseQuoObject( <opL>, <opR> )  . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_VerboseQuoObject (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
  libGAP_Obj val;
  val = libGAP_DoVerboseOperation2Args( libGAP_QuoOper, opL, opR );
  while (val == 0)
    val = libGAP_ErrorReturnObj("QUO: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  InstallQuoObject( <verb> )  . . . . . . . . . .  install quotient methods
*/
void libGAP_InstallQuoObject ( libGAP_Int verb )
{
    libGAP_UInt                t1;             /* type of left  operand           */
    libGAP_UInt                t2;             /* type of right operand           */
    libGAP_ArithMethod2        func;           /* quotient function               */

    func = ( verb ? libGAP_VerboseQuoObject : libGAP_QuoObject );
    for ( t1 = libGAP_FIRST_EXTERNAL_TNUM;  t1 <= libGAP_LAST_EXTERNAL_TNUM;  t1++ ) {
        for ( t2 = libGAP_FIRST_REAL_TNUM;  t2 <= libGAP_LAST_VIRTUAL_TNUM;  t2++ ) {
            libGAP_QuoFuncs[t1][t2] = func;
            libGAP_QuoFuncs[t2][t1] = func;
        }
    }
    for ( t2 = libGAP_FIRST_REAL_TNUM;  t2 <= libGAP_LAST_VIRTUAL_TNUM;  t2++ ) {

        libGAP_QuoFuncs[ t2 ][ libGAP_T_PREC            ] = func;
        libGAP_QuoFuncs[ t2 ][ libGAP_T_PREC +libGAP_IMMUTABLE ] = func;
        libGAP_QuoFuncs[ libGAP_T_PREC            ][ t2 ] = func;
        libGAP_QuoFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ][ t2 ] = func;
    }
}


/****************************************************************************
**
*F  FuncQUO_DEFAULT( <self>, <opL>, <opR> ) . . . . . . . . call 'QuoDefault'
*/
libGAP_Obj libGAP_QuoDefaultFunc;

libGAP_Obj libGAP_FuncQUO_DEFAULT (
    libGAP_Obj                 self,
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    return libGAP_QuoDefault( opL, opR );
}


/****************************************************************************
**
*F  FuncQUO( <self>, <opL>, <opR> ) . . . . . . . . . . . . . . .  call 'QUO'
*/
libGAP_Obj libGAP_FuncQUO (
    libGAP_Obj                 self,
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    return libGAP_QUO( opL, opR );
}


/****************************************************************************
**

*V  LQuoFuncs[ <typeL> ][ <typeR> ] . . . . .  table of left quotient methods
*/
libGAP_ArithMethod2 libGAP_LQuoFuncs [libGAP_LAST_VIRTUAL_TNUM+1][libGAP_LAST_VIRTUAL_TNUM+1];


/****************************************************************************
**
*F  LQuoDefault( <opL>, <opR> ) . . . . . . . . . . . . call 'INV' and 'PROD'
*/
libGAP_Obj libGAP_LQuoDefault (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 tmp;
    tmp = libGAP_INV_MUT( opL );
    return libGAP_PROD( tmp, opR );
}


/****************************************************************************
**
*F  LQuoObject( <opL>, <opR> )  . . . . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_LQuoOper;

libGAP_Obj libGAP_LQuoObject (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
  libGAP_Obj val;
  val = libGAP_DoOperation2Args( libGAP_LQuoOper, opL, opR );
  while (val == 0)
    val = libGAP_ErrorReturnObj("LeftQuotient: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  VerboseLQuoObject( <opL>, <opR> ) . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_VerboseLQuoObject (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
  libGAP_Obj val;
  val = libGAP_DoOperation2Args( libGAP_LQuoOper, opL, opR );
  while (val == 0)
    val = libGAP_ErrorReturnObj("LeftQuotient: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  InstallLQuoObject( <verb> ) . . . . . . . . install left quotient methods
*/
void libGAP_InstallLQuoObject ( libGAP_Int verb )
{
    libGAP_UInt                t1;             /* type of left  operand           */
    libGAP_UInt                t2;             /* type of right operand           */
    libGAP_ArithMethod2        func;           /* left quotient function          */

    func = ( verb ? libGAP_VerboseLQuoObject : libGAP_LQuoObject );
    for ( t1 = libGAP_FIRST_EXTERNAL_TNUM; t1 <= libGAP_LAST_EXTERNAL_TNUM; t1++ ) {
        for ( t2 = libGAP_FIRST_REAL_TNUM; t2 <= libGAP_LAST_VIRTUAL_TNUM; t2++ ) {
            libGAP_LQuoFuncs[t1][t2] = func;
            libGAP_LQuoFuncs[t2][t1] = func;
        }
    }
    for ( t2 = libGAP_FIRST_REAL_TNUM; t2 <= libGAP_LAST_VIRTUAL_TNUM; t2++ ) {
        libGAP_LQuoFuncs[ t2 ][ libGAP_T_PREC            ] = func;
        libGAP_LQuoFuncs[ t2 ][ libGAP_T_PREC +libGAP_IMMUTABLE ] = func;
        libGAP_LQuoFuncs[ libGAP_T_PREC            ][ t2 ] = func;
        libGAP_LQuoFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ][ t2 ] = func;
    }
}


/****************************************************************************
**
*F  FuncLQUO_DEFAULT( <self>, <opL>, <opR> )  . . . . . .  call 'LQuoDefault'
*/
libGAP_Obj libGAP_LQuoDefaultFunc;

libGAP_Obj libGAP_FuncLQUO_DEFAULT (
    libGAP_Obj                 self,
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    return libGAP_LQuoDefault( opL, opR );
}


/****************************************************************************
**
*F  FuncLQUO( <self>, <opL>, <opR> )  . . . . . . . . . . . . . . call 'LQUO'
*/
libGAP_Obj libGAP_FuncLQUO (
    libGAP_Obj                 self,
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    return libGAP_LQUO( opL, opR );
}


/****************************************************************************
**

*V  PowFuncs[ <typeL> ][ <typeR> ]  . . . . . . . . .  table of power methods
*/
libGAP_ArithMethod2 libGAP_PowFuncs [libGAP_LAST_VIRTUAL_TNUM+1][libGAP_LAST_VIRTUAL_TNUM+1];


/****************************************************************************
**
*F  PowDefault( <opL>, <opR> )  . . . . . . . . . . .  call 'LQUO' and 'PROD'
*/
libGAP_Obj libGAP_PowDefault (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 tmp;
    tmp = libGAP_LQUO( opR, opL );
    return libGAP_PROD( tmp, opR );
}


/****************************************************************************
**
*F  PowObject( <opL>, <opR> ) . . . . . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_PowOper;

libGAP_Obj libGAP_PowObject (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
  libGAP_Obj val;
  val = libGAP_DoOperation2Args( libGAP_PowOper, opL, opR );
  while (val == 0)
    val = libGAP_ErrorReturnObj("POW: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  VerbosePowObject( <opL>, <opR> )  . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_VerbosePowObject (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
   
  libGAP_Obj val;
  val = libGAP_DoVerboseOperation2Args( libGAP_PowOper, opL, opR );
  while (val == 0)
    val = libGAP_ErrorReturnObj("POW: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  InstallPowObject( <verb> )  . . . . . . . . . . install the power methods
*/
void libGAP_InstallPowObject ( libGAP_Int verb )
{
    libGAP_UInt                t1;             /* type of left  operand           */
    libGAP_UInt                t2;             /* type of right operand           */
    libGAP_ArithMethod2        func;           /* power function                  */

    func = ( verb ? libGAP_VerbosePowObject : libGAP_PowObject );
    for ( t1 = libGAP_FIRST_EXTERNAL_TNUM;  t1 <= libGAP_LAST_EXTERNAL_TNUM;  t1++ ) {
        for ( t2 = libGAP_FIRST_REAL_TNUM;  t2 <= libGAP_LAST_VIRTUAL_TNUM;  t2++ ) {
            libGAP_PowFuncs[t1][t2] = func;
            libGAP_PowFuncs[t2][t1] = func;
        }
    }
    for ( t2 = libGAP_FIRST_REAL_TNUM; t2 <= libGAP_LAST_VIRTUAL_TNUM; t2++ ) {

        libGAP_PowFuncs[ t2 ][ libGAP_T_PREC            ] = func;
        libGAP_PowFuncs[ t2 ][ libGAP_T_PREC +libGAP_IMMUTABLE ] = func;
        libGAP_PowFuncs[ libGAP_T_PREC            ][ t2 ] = func;
        libGAP_PowFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ][ t2 ] = func;
    }
}


/****************************************************************************
**
*F  FuncPOW_DEFAULT( <self>, <opL>, <opR> ) . . . . . . . . call 'PowDefault'
*/
libGAP_Obj libGAP_PowDefaultFunc;

libGAP_Obj libGAP_FuncPOW_DEFAULT (
    libGAP_Obj                 self,
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    return libGAP_PowDefault( opL, opR );
}


/****************************************************************************
**
*F  FuncPOW( <self>, <opL>, <opR> ) . . . . . . . . . . . . . . .  call 'POW'
*/
libGAP_Obj libGAP_FuncPOW (
    libGAP_Obj                 self,
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    return libGAP_POW( opL, opR );
}


/****************************************************************************
**

*V  CommFuncs[ <typeL> ][ <typeR> ] . . . . . . . table of commutator methods
*/
libGAP_ArithMethod2 libGAP_CommFuncs [libGAP_LAST_VIRTUAL_TNUM+1][libGAP_LAST_VIRTUAL_TNUM+1];


/****************************************************************************
**
*F  CommDefault( <opL>, <opR> ) . . . . . . . . . . .  call 'LQUO' and 'PROD'
*/
libGAP_Obj libGAP_CommDefault (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 tmp1;
    libGAP_Obj                 tmp2;
    tmp1 = libGAP_PROD( opR, opL );
    tmp2 = libGAP_PROD( opL, opR );
    return libGAP_LQUO( tmp1, tmp2 );
}


/****************************************************************************
**
*F  CommObject( <opL>, <opR> )  . . . . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_CommOper;

libGAP_Obj libGAP_CommObject (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
  libGAP_Obj val;
  val = libGAP_DoOperation2Args( libGAP_CommOper, opL, opR );
  while (val == 0)
    val = libGAP_ErrorReturnObj("Comm: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  VerboseCommObject( <opL>, <opR> ) . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_VerboseCommObject (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
  libGAP_Obj val;
  val = libGAP_DoVerboseOperation2Args( libGAP_CommOper, opL, opR );
  while (val == 0)
    val = libGAP_ErrorReturnObj("Comm: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  InstallCommObject( <verb> ) . . . . . . . . .  install commutator methods
*/
void libGAP_InstallCommObject ( libGAP_Int verb )
{
    libGAP_UInt                t1;             /* type of left  operand           */
    libGAP_UInt                t2;             /* type of right operand           */
    libGAP_ArithMethod2        func;           /* commutator function             */

    func = ( verb ? libGAP_VerboseCommObject : libGAP_CommObject );
    for ( t1 = libGAP_FIRST_EXTERNAL_TNUM;  t1 <= libGAP_LAST_EXTERNAL_TNUM;  t1++ ) {
        for ( t2 = libGAP_FIRST_REAL_TNUM;  t2 <= libGAP_LAST_VIRTUAL_TNUM;  t2++ ) {
            libGAP_CommFuncs[t1][t2] = func;
            libGAP_CommFuncs[t2][t1] = func;
        }
    }
    for ( t2 = libGAP_FIRST_REAL_TNUM;  t2 <= libGAP_LAST_VIRTUAL_TNUM;  t2++ ) {

        libGAP_CommFuncs[ t2 ][ libGAP_T_PREC            ] = func;
        libGAP_CommFuncs[ t2 ][ libGAP_T_PREC +libGAP_IMMUTABLE ] = func;
        libGAP_CommFuncs[ libGAP_T_PREC            ][ t2 ] = func;
        libGAP_CommFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ][ t2 ] = func;
    }
}


/****************************************************************************
**
*F  FuncCOMM_DEFAULT( <self>, <opL>, <opR> )  . . . . . .  call 'CommDefault'
*/
libGAP_Obj libGAP_CommDefaultFunc;

libGAP_Obj libGAP_FuncCOMM_DEFAULT (
    libGAP_Obj                 self,
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    return libGAP_CommDefault( opL, opR );
}


/****************************************************************************
**
*F  FuncCOMM( <self>, <opL>, <opR> )  . . . . . . . . . . . . . . call 'COMM'
*/
libGAP_Obj libGAP_FuncCOMM (
    libGAP_Obj                 self,
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    return libGAP_COMM( opL, opR );
}


/****************************************************************************
**

*V  ModFuncs[ <typeL> ][ <typeR> ]  . . . . . . .  table of remainder methods
*/
libGAP_ArithMethod2 libGAP_ModFuncs [libGAP_LAST_VIRTUAL_TNUM+1][libGAP_LAST_VIRTUAL_TNUM+1];



/****************************************************************************
**
*F  ModObject( <opL>, <opR> ) . . . . . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_ModOper;

libGAP_Obj libGAP_ModObject (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
  libGAP_Obj val;
  val = libGAP_DoOperation2Args( libGAP_ModOper, opL, opR );
  while (val == 0)
    val = libGAP_ErrorReturnObj("mod: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  VerboseModObject( <opL>, <opR> )  . . . . . . . . . . . . .  call methsel
*/
libGAP_Obj libGAP_VerboseModObject (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
  libGAP_Obj val;
  val = libGAP_DoVerboseOperation2Args( libGAP_ModOper, opL, opR );
  while (val == 0)
    val = libGAP_ErrorReturnObj("mod: method should have returned a value", 0L, 0L,
                         "you can supply one by 'return <value>;'");
  return val;
}


/****************************************************************************
**
*F  InstallModObject( <verb> )  . . . . . . . . . . . install the mod methods
*/
void libGAP_InstallModObject ( libGAP_Int verb )
{
    libGAP_UInt                t1;             /* type of left  operand           */
    libGAP_UInt                t2;             /* type of right operand           */
    libGAP_ArithMethod2        func;           /* mod function                    */

    func = ( verb ? libGAP_VerboseModObject : libGAP_ModObject );
    for ( t1 = libGAP_FIRST_EXTERNAL_TNUM;  t1 <= libGAP_LAST_EXTERNAL_TNUM;  t1++ ) {
        for ( t2 = libGAP_FIRST_REAL_TNUM;  t2 <= libGAP_LAST_VIRTUAL_TNUM;  t2++ ) {
            libGAP_ModFuncs[t1][t2] = func;
            libGAP_ModFuncs[t2][t1] = func;
        }
    }
    for ( t2 = libGAP_FIRST_REAL_TNUM; t2 <= libGAP_LAST_VIRTUAL_TNUM; t2++ ) {

        libGAP_ModFuncs[ t2 ][ libGAP_T_PREC            ] = func;
        libGAP_ModFuncs[ t2 ][ libGAP_T_PREC +libGAP_IMMUTABLE ] = func;
        libGAP_ModFuncs[ libGAP_T_PREC            ][ t2 ] = func;
        libGAP_ModFuncs[ libGAP_T_PREC +libGAP_IMMUTABLE ][ t2 ] = func;
    }
}


/****************************************************************************
**
*F  FuncMOD( <self>, <opL>, <opR> ) . . . . . . . . . . . . . . .  call 'MOD'
*/
libGAP_Obj libGAP_FuncMOD (
    libGAP_Obj                 self,
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    return libGAP_MOD( opL, opR );
}


/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/

/****************************************************************************
**

*V  GVarAttrs . . . . . . . . . . . . . . . . .  list of attributes to export
*/
static libGAP_StructGVarAttr libGAP_GVarAttrs [] = {


    { 0 }

};


/****************************************************************************
**
*V  GVarOpers . . . . . . . . . . . . . . . . .  list of operations to export
*/
static libGAP_StructGVarOper libGAP_GVarOpers [] = {

    { "EQ", 2, "opL, opR", &libGAP_EqOper,
      libGAP_FuncEQ, "src/ariths.c:EQ" },

    { "LT", 2, "opL, opR", &libGAP_LtOper,
      libGAP_FuncLT, "src/ariths.c:LT" },

    { "IN", 2, "opL, opR", &libGAP_InOper,
      libGAP_FuncIN, "src/ariths.c:IN" },

    { "SUM", 2, "opL, opR", &libGAP_SumOper,
      libGAP_FuncSUM, "src/ariths.c:SUM" },

    { "DIFF", 2, "opL, opR", &libGAP_DiffOper,
      libGAP_FuncDIFF, "src/ariths.c:DIFF" },

    { "PROD", 2, "opL, opR", &libGAP_ProdOper,
      libGAP_FuncPROD, "src/ariths.c:PROD" },

    { "QUO", 2, "opL, opR", &libGAP_QuoOper,
      libGAP_FuncQUO, "src/ariths.c:QUO" },

    { "LQUO", 2, "opL, opR", &libGAP_LQuoOper,
      libGAP_FuncLQUO, "src/ariths.c:LQUO" },

    { "POW", 2, "opL, opR", &libGAP_PowOper,
      libGAP_FuncPOW, "src/ariths.c:POW" },

    { "COMM", 2, "opL, opR", &libGAP_CommOper,
      libGAP_FuncCOMM, "src/ariths.c:COMM" },

    { "MOD", 2, "opL, opR", &libGAP_ModOper,
      libGAP_FuncMOD, "src/ariths.c:MOD" },

    { "ZERO", 1, "op", &libGAP_ZEROOp,
      libGAP_FuncZERO, "src/ariths.c:ZERO" },

    { "ZERO_MUT", 1, "op", &libGAP_ZeroOp,
      libGAP_FuncZERO_MUT, "src/ariths.c:ZERO_MUT" },

    { "AINV", 1, "op", &libGAP_AInvOp,
      libGAP_FuncAINV, "src/ariths.c:AINV" },

    { "AINV_MUT", 1, "op", &libGAP_AdditiveInverseOp,
      libGAP_FuncAINV_MUT, "src/ariths.c:AINV_MUT" },

    { "ONE", 1, "op", &libGAP_OneOp,
      libGAP_FuncONE, "src/ariths.c:ONE" },

    { "ONE_MUT", 1, "op", &libGAP_OneMutOp,
      libGAP_FuncONE_MUT, "src/ariths.c:ONE_MUT" },

    { "INV", 1, "op", &libGAP_InvOp,
      libGAP_FuncINV, "src/ariths.c:INV" },

    { "INV_MUT", 1, "op", &libGAP_InvMutOp,
      libGAP_FuncINV_MUT, "src/ariths.c:INV_MUT" },

    { 0 }

};


/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "COMM_DEFAULT", 2, "opL, opR",
      libGAP_FuncCOMM_DEFAULT, "src/ariths.c:COMM_DEFAULT" },

    { "POW_DEFAULT", 2, "opL, opR",
      libGAP_FuncPOW_DEFAULT, "src/ariths.c:POW_DEFAULT" },

    { "LQUO_DEFAULT", 2, "opL, opR",
      libGAP_FuncLQUO_DEFAULT, "src/ariths.c:LQUO_DEFAULT" },

    { "QUO_DEFAULT", 2, "opL, opR",
      libGAP_FuncQUO_DEFAULT, "src/ariths.c:QUO_DEFAULT" },

    { "DIFF_DEFAULT", 2, "opL, opR",
      libGAP_FuncDIFF_DEFAULT, "src/ariths.c:DIFF_DEFAULT" },


    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    libGAP_UInt                t1;             /* type of left  operand           */
    libGAP_UInt                t2;             /* type of right operand           */

    /* init filters and functions                                          */
    libGAP_InitHdlrAttrsFromTable( libGAP_GVarAttrs );
    libGAP_InitHdlrOpersFromTable( libGAP_GVarOpers );
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* make and install the 'ZERO' arithmetic operation                    */
    for ( t1 = libGAP_FIRST_REAL_TNUM;  t1 <= libGAP_LAST_VIRTUAL_TNUM;  t1++ ) {
        libGAP_ZeroFuncs[t1] = libGAP_ZeroObject;
    }
    libGAP_InstallZeroObject(0);

    /* make and install the 'ZERO_MUT' arithmetic operation                */
    for ( t1 = libGAP_FIRST_REAL_TNUM;  t1 <= libGAP_LAST_VIRTUAL_TNUM;  t1++ ) {
        libGAP_ZeroMutFuncs[t1] = libGAP_ZeroMutObject;
    }
    libGAP_InstallZeroObject(0);

    /* make and install the 'AINV' arithmetic operation                    */
    for ( t1 = libGAP_FIRST_REAL_TNUM;  t1 <= libGAP_LAST_VIRTUAL_TNUM;  t1++ ) {
        libGAP_AInvFuncs[t1] = libGAP_AInvObject;
    }

    libGAP_InstallAinvMutObject(0);
    /* make and install the 'AINV_MUT' arithmetic operation                */
    for ( t1 = libGAP_FIRST_REAL_TNUM;  t1 <= libGAP_LAST_VIRTUAL_TNUM;  t1++ ) {
        libGAP_AInvMutFuncs[t1] = libGAP_AInvMutObject;
    }
    libGAP_InstallAinvObject(0);

    /* make and install the 'ONE' arithmetic operation                     */
    for ( t1 = libGAP_FIRST_REAL_TNUM;  t1 <= libGAP_LAST_VIRTUAL_TNUM;  t1++ ) {
        libGAP_OneFuncs[t1] = libGAP_OneObject;
    }
    libGAP_InstallOneObject(0);

    /* make and install the 'ONE' arithmetic operation                     */
    for ( t1 = libGAP_FIRST_REAL_TNUM;  t1 <= libGAP_LAST_VIRTUAL_TNUM;  t1++ ) {
        libGAP_OneMutFuncs[t1] = libGAP_OneMutObject;
    }
    libGAP_InstallOneMutObject(0);

    /* make and install the 'INV' arithmetic operation                     */
    for ( t1 = libGAP_FIRST_REAL_TNUM;  t1 <= libGAP_LAST_VIRTUAL_TNUM;  t1++ ) {
        libGAP_InvFuncs[t1] = libGAP_InvObject;
    }
    libGAP_InstallInvObject(0);

    /* make and install the 'INV' arithmetic operation                     */
    for ( t1 = libGAP_FIRST_REAL_TNUM;  t1 <= libGAP_LAST_VIRTUAL_TNUM;  t1++ ) {
        libGAP_InvMutFuncs[t1] = libGAP_InvMutObject;
    }
    libGAP_InstallInvMutObject(0);

    /* make and install the 'EQ' comparison operation                      */
    for ( t1 = libGAP_FIRST_REAL_TNUM;  t1 <= libGAP_LAST_VIRTUAL_TNUM;  t1++ ) {
        for ( t2 = libGAP_FIRST_REAL_TNUM;  t2 <= libGAP_LAST_VIRTUAL_TNUM;  t2++ ) {
            libGAP_EqFuncs[t1][t2] = libGAP_EqNot;
        }
    }
    libGAP_InstallEqObject(0);

    /* make and install the 'LT' comparison operation                      */
    for ( t1 = libGAP_FIRST_REAL_TNUM;  t1 <= libGAP_LAST_VIRTUAL_TNUM;  t1++ ) {
        for ( t2 = libGAP_FIRST_REAL_TNUM;  t2 <= libGAP_LAST_VIRTUAL_TNUM;  t2++ ) {
            libGAP_LtFuncs[t1][t2] = libGAP_LtObject;
        }
    }
    libGAP_InstallLtObject(0);

    /* make and install the 'IN' comparison operation                      */
    for ( t1 = libGAP_FIRST_REAL_TNUM; t1 <= libGAP_LAST_VIRTUAL_TNUM; t1++ ) {
        for ( t2 = libGAP_FIRST_REAL_TNUM; t2 <= libGAP_LAST_VIRTUAL_TNUM; t2++ ) {
            libGAP_InFuncs[t1][t2] = libGAP_InUndefined;
        }
    }
    libGAP_InstallInObject(0);

    /* make and install the 'SUM' arithmetic operation                     */
    for ( t1 = libGAP_FIRST_REAL_TNUM;  t1 <= libGAP_LAST_VIRTUAL_TNUM;  t1++ ) {
        for ( t2 = libGAP_FIRST_REAL_TNUM;  t2 <= libGAP_LAST_VIRTUAL_TNUM;  t2++ ) {
            libGAP_SumFuncs[t1][t2] = libGAP_SumObject;
        }
    }
    libGAP_InstallSumObject(0);

    /* make and install the 'DIFF' arithmetic operation                    */
    for ( t1 = libGAP_FIRST_REAL_TNUM;  t1 <= libGAP_LAST_VIRTUAL_TNUM;  t1++ ) {
        for ( t2 = libGAP_FIRST_REAL_TNUM;  t2 <= libGAP_LAST_VIRTUAL_TNUM;  t2++ ) {
            libGAP_DiffFuncs[t1][t2] = libGAP_DiffDefault;
        }
    }
    libGAP_InstallDiffObject(0);

    /* make and install the 'PROD' arithmetic operation                    */
    for ( t1 = libGAP_FIRST_REAL_TNUM;  t1 <= libGAP_LAST_VIRTUAL_TNUM;  t1++ ) {
        for ( t2 = libGAP_FIRST_REAL_TNUM;  t2 <= libGAP_LAST_VIRTUAL_TNUM;  t2++ ) {
            libGAP_ProdFuncs[t1][t2] = libGAP_ProdObject;
        }
    }
    libGAP_InstallProdObject(0);

    /* make and install the 'QUO' arithmetic operation                     */
    for ( t1 = libGAP_FIRST_REAL_TNUM;  t1 <= libGAP_LAST_VIRTUAL_TNUM;  t1++ ) {
        for ( t2 = libGAP_FIRST_REAL_TNUM;  t2 <= libGAP_LAST_VIRTUAL_TNUM;  t2++ ) {
            libGAP_QuoFuncs[t1][t2] = libGAP_QuoDefault;
        }
    }
    libGAP_InstallQuoObject(0);

    /* make and install the 'LQUO' arithmetic operation                    */
    for ( t1 = libGAP_FIRST_REAL_TNUM;  t1 <= libGAP_LAST_VIRTUAL_TNUM;  t1++ ) {
        for ( t2 = libGAP_FIRST_REAL_TNUM;  t2 <= libGAP_LAST_VIRTUAL_TNUM;  t2++ ) {
            libGAP_LQuoFuncs[t1][t2] = libGAP_LQuoDefault;
        }
    }
    libGAP_InstallLQuoObject(0);

    /* make and install the 'POW' arithmetic operation                     */
    for ( t1 = libGAP_FIRST_REAL_TNUM;  t1 <= libGAP_LAST_VIRTUAL_TNUM;  t1++ ) {
        for ( t2 = libGAP_FIRST_REAL_TNUM;  t2 <= libGAP_LAST_VIRTUAL_TNUM;  t2++ ) {
            libGAP_PowFuncs[t1][t2] = libGAP_PowObject;
        }
    }
    libGAP_InstallPowObject(0);

    /* make and install the 'COMM' arithmetic operation                    */
    for ( t1 = libGAP_FIRST_REAL_TNUM;  t1 <= libGAP_LAST_VIRTUAL_TNUM;  t1++ ) {
        for ( t2 = libGAP_FIRST_REAL_TNUM;  t2 <= libGAP_LAST_VIRTUAL_TNUM;  t2++ ) {
            libGAP_CommFuncs[t1][t2] = libGAP_CommDefault;
        }
    }
    libGAP_InstallCommObject(0);

    /* make and install the 'MOD' arithmetic operation                     */
    for ( t1 = libGAP_FIRST_REAL_TNUM;  t1 <= libGAP_LAST_VIRTUAL_TNUM;  t1++ ) {
        for ( t2 = libGAP_FIRST_REAL_TNUM;  t2 <= libGAP_LAST_VIRTUAL_TNUM;  t2++ ) {
            libGAP_ModFuncs[t1][t2] = libGAP_ModObject;
        }
    }
    libGAP_InstallModObject(0);

    
    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitGVarAttrsFromTable( libGAP_GVarAttrs );
    libGAP_InitGVarOpersFromTable( libGAP_GVarOpers );
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoAriths()  . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "ariths",                           /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoAriths ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}


/****************************************************************************
**

*E  ariths.c  . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/










