/****************************************************************************
**
*W  tietze.c                    GAP source                       Frank Celler
*W                                                           & Volkmar Felsch
**
**
*Y  Copyright 1990-1992,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains the functions for computing with finite presentations.
*/
#include        "system.h"              /* system dependent part           */
#include        "code.h"
#include        "stats.h"               /* for TakeInterrupt */


#include        "gasman.h"              /* garbage collector               */
#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */
#include        "calls.h"               /* generic call mechanism          */
#include        "gvars.h"               /* global variables                */

#include        "bool.h"                /* booleans                        */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "plist.h"               /* plain lists                     */
#include        "string.h"              /* strings                         */

#include        "tietze.h"              /* tietze helper functions         */


/****************************************************************************
**

*V  TZ_SOMETHING  . . . . . . defining some constants for the Tietze routines
*/
#define TZ_NUMGENS               1
#define TZ_NUMRELS               2
#define TZ_TOTAL                 3
#define TZ_GENERATORS            4
#define TZ_INVERSES              5
#define TZ_RELATORS              6
#define TZ_LENGTHS               7
#define TZ_FLAGS                 8
#define TZ_FREEGENS              9
#define TZ_LENGTHTIETZE         21


/****************************************************************************
**

*F  CheckTietzeStack( <tietze>, <ptTietze> )
*/
void libGAP_CheckTietzeStack (
    libGAP_Obj                 tietze,
    libGAP_Obj * *             ptTietze )
{
    /*  check the Tietze stack                                             */
    if ( ! libGAP_IS_PLIST(tietze) ) {
        libGAP_ErrorQuit( "<tietze> must be a plain list (not a %s)",
                   (libGAP_Int)libGAP_TNAM_OBJ(tietze), 0L );
        return;
    }
    if ( libGAP_LEN_PLIST(tietze) != TZ_LENGTHTIETZE ) {
        libGAP_ErrorQuit( "<tietze> must have length %d (not %d)",
                   (libGAP_Int)TZ_LENGTHTIETZE, (libGAP_Int)libGAP_LEN_PLIST(tietze) );
        return;
    }
    *ptTietze = libGAP_ADDR_OBJ(tietze);
}


/****************************************************************************
**
*F  CheckTietzeRelators( <ptTietze>, <rels>, <ptRels>, <numrels> )
*/
void libGAP_CheckTietzeRelators (
    libGAP_Obj *               ptTietze,
    libGAP_Obj *               rels,
    libGAP_Obj * *             ptRels,
    libGAP_Int *              numrels )
{
    *rels    = ptTietze[TZ_RELATORS];
    *numrels = libGAP_INT_INTOBJ(ptTietze[TZ_NUMRELS]);
    if ( *rels == 0 || ! libGAP_IS_PLIST(*rels) || libGAP_LEN_PLIST(*rels) != *numrels ) {
        libGAP_ErrorQuit( "invalid Tietze relators list", 0L, 0L );
        return;
    }
    *ptRels = libGAP_ADDR_OBJ(*rels);
}


/****************************************************************************
**
*F  CheckTietzeInverses( <ptTietze>, <invs>, <ptInvs>, <numgens> )
*/
void libGAP_CheckTietzeInverses (
    libGAP_Obj *               ptTietze,
    libGAP_Obj *               invs,
    libGAP_Obj * *             ptInvs,
    libGAP_Int *               numgens )
{
    /* get and check the Tietze inverses list                              */
    *invs    = ptTietze[TZ_INVERSES];
    *numgens = libGAP_INT_INTOBJ(ptTietze[TZ_NUMGENS]);
    if ( *invs==0 || !libGAP_IS_PLIST(*invs) || libGAP_LEN_PLIST(*invs)!=2*(*numgens)+1 ) {
        libGAP_ErrorQuit( "invalid Tietze inverses list", 0L, 0L );
        return;
    }
    *ptInvs = libGAP_ADDR_OBJ(*invs) + (*numgens+1);
}


/****************************************************************************
**
*F  CheckTietzeLengths( <ptTietze>, <numrels>, <lens>, <ptLens> )
*/
void libGAP_CheckTietzeLengths (
    libGAP_Obj *               ptTietze,
    libGAP_Int                 numrels,
    libGAP_Obj *               lens,
    libGAP_Obj * *             ptLens )
{
    /*  Get and check the Tietze lengths list                              */
    *lens = ptTietze[TZ_LENGTHS];
    if ( *lens == 0 || ! libGAP_IS_PLIST(*lens) || libGAP_LEN_PLIST(*lens) != numrels ) {
        libGAP_ErrorQuit( "invalid Tietze lengths list", 0L, 0L );
        return;
    }
    *ptLens = libGAP_ADDR_OBJ(*lens);
}


/****************************************************************************
**
*F  CheckTietzeFlags( <ptTietze>, <numrels>, <flags>, <ptFlags> )
*/
void libGAP_CheckTietzeFlags (
    libGAP_Obj *               ptTietze,
    libGAP_Int                 numrels,
    libGAP_Obj *               flags,
    libGAP_Obj * *             ptFlags )
{
    /* get and check the Tietze flags list                                 */
    *flags = ptTietze[TZ_FLAGS];
    if ( *flags==0 || ! libGAP_IS_PLIST(*flags) || libGAP_LEN_PLIST(*flags)!=numrels ) {
        libGAP_ErrorQuit( "invalid Tietze flags list", 0L, 0L );
        return;
    }
    *ptFlags = libGAP_ADDR_OBJ(*flags);
}


/****************************************************************************
**
*F  CheckTietzeRelLengths( <ptTietze>, <ptRels>, <ptLens>, <nrels>, <total> )
*/
void libGAP_CheckTietzeRelLengths (
    libGAP_Obj *               ptTietze,
    libGAP_Obj *               ptRels,
    libGAP_Obj *               ptLens,
    libGAP_Int                 numrels,
    libGAP_Int  *              total )
{
    libGAP_Int                i;

    /* Check list <lens> to contain the relator lengths                 */
    *total = 0;
    for ( i = 1;  i <= numrels;  i++ ) {
        if ( ptRels[i] == 0
          || ! libGAP_IS_PLIST(ptRels[i])
          || libGAP_INT_INTOBJ(ptLens[i]) != libGAP_LEN_PLIST(ptRels[i]) )
        {
            libGAP_ErrorQuit( "inconsistent Tietze lengths list", 0L, 0L );
            return;
        }
        *total += libGAP_INT_INTOBJ(ptLens[i]);
    }
    if ( *total != libGAP_INT_INTOBJ(ptTietze[TZ_TOTAL]) ) {
        libGAP_ErrorQuit( "inconsistent total length", 0L, 0L );
        return;
    }
}


/****************************************************************************
**

*F  FuncTzSortC( <self>, <stack> )  . . . . . . . sort the relators by length
*/
libGAP_Obj libGAP_FuncTzSortC (
    libGAP_Obj                 self,
    libGAP_Obj                 tietze )
{
    libGAP_Obj *               ptTietze;       /* pointer to the Tietze stack     */
    libGAP_Obj                 rels;           /* relators list                   */
    libGAP_Obj *               ptRels;         /* pointer to this list            */
    libGAP_Obj                 lens;           /* lengths list                    */
    libGAP_Obj *               ptLens;         /* pointer to this list            */
    libGAP_Obj                 flags;          /* handle of the flags list        */
    libGAP_Obj *               ptFlags;        /* pointer to this list            */
    libGAP_Int                 numrels;        /* number of Tietze relators       */
    libGAP_Int                 i, h, k;        /* loop variables                  */
    libGAP_Obj                 rel, len, flag; /* list entries                    */
    libGAP_Int                 total;

    /* check the Tietze stack                                              */
    libGAP_CheckTietzeStack( tietze, &ptTietze );

    /* get and check the Tietze relators list                              */
    libGAP_CheckTietzeRelators( ptTietze, &rels, &ptRels, &numrels );

    /* get and check the Tietze lengths list                               */
    libGAP_CheckTietzeLengths( ptTietze, numrels, &lens, &ptLens );

    /* get and check the Tietze flags list                                 */
    libGAP_CheckTietzeFlags( ptTietze, numrels, &flags, &ptFlags );

    /* check list <lens> to contain the relator lengths                    */
    libGAP_CheckTietzeRelLengths( ptTietze, ptRels, ptLens, numrels, &total );

    /* sort the list                                                       */
    h = 1;
    while ( 9 * h + 4 < numrels )
        h = 3 * h + 1;
    while ( 0 < h ) {
        for ( i = h + 1; i <= numrels; i++ ) {
            rel = ptRels[i];  len = ptLens[i];  flag = ptFlags[i];
            k = i;
            if ( libGAP_INT_INTOBJ(len) ) {
                while ( h < k
                  && ( !libGAP_INT_INTOBJ(ptLens[k-h])
                     || len < ptLens[k-h]
                     || (len == ptLens[k-h] && flag > ptFlags[k-h])))
                {
                    ptRels[k] = ptRels[k-h];
                    ptLens[k] = ptLens[k-h];
                    ptFlags[k] = ptFlags[k-h];
                    k = k - h;
                }
            }
            ptRels[k] = rel;  ptLens[k] = len;  ptFlags[k] = flag;
        }
        h = h / 3;
    }
    for ( i = numrels;  i > 0;  i-- ) {
        if ( libGAP_INT_INTOBJ(ptLens[i]) )
            break;
    }
    if ( i < numrels ) {
        libGAP_SET_LEN_PLIST( rels,  i );  libGAP_SHRINK_PLIST( rels,  i );
        libGAP_SET_LEN_PLIST( lens,  i );  libGAP_SHRINK_PLIST( lens,  i );
        libGAP_SET_LEN_PLIST( flags, i );  libGAP_SHRINK_PLIST( flags, i );
        ptTietze[TZ_NUMRELS] = libGAP_INTOBJ_INT(i);
        libGAP_CHANGED_BAG(tietze);
    }

    return 0;
}


/****************************************************************************
**
*F  FuncTzRenumberGens( <self>, <stack> ) . .  renumber the Tietze generators
*/
libGAP_Obj libGAP_FuncTzRenumberGens (
    libGAP_Obj                 self,
    libGAP_Obj                 tietze )
{
    libGAP_Obj *               ptTietze;       /* pointer to this stack           */
    libGAP_Obj                 rels;           /* handle of the relators list     */
    libGAP_Obj *               ptRels;         /* pointer to this list            */
    libGAP_Obj                 invs;           /* handle of the inverses list     */
    libGAP_Obj *               ptInvs;         /* pointer to this list            */
    libGAP_Obj *               ptRel;          /* pointer to the ith relator      */
    libGAP_Int                 numgens;        /* number of Tietze generators     */
    libGAP_Int                 numrels;        /* number of Tietze relators       */
    libGAP_Int                 old;            /* generator or inverse            */
    libGAP_Int                 leng;           /* relator length                  */
    libGAP_Int                 i, j;           /* loop variables                  */

    /* check the Tietze stack                                              */
    libGAP_CheckTietzeStack( tietze, &ptTietze );

    /* get and check the Tietze relators list                              */
    libGAP_CheckTietzeRelators( ptTietze, &rels, &ptRels, &numrels );

    /* get and check the Tietze inverses list                              */
    libGAP_CheckTietzeInverses( ptTietze, &invs, &ptInvs, &numgens );

    /*  Loop over all relators and replace the occurring generators        */
    for ( i = 1;  i <= numrels;  i++ ) {
        ptRel = libGAP_ADDR_OBJ( ptRels[i] );
        leng  = libGAP_LEN_PLIST( ptRels[i] );

        /* run through the relator and replace the occurring generators    */
        for ( j = 1;  j <= leng;  j++ ) {
            old = libGAP_INT_INTOBJ( ptRel[j] );
            if ( old < -numgens || numgens < old || old == 0 ) {
                libGAP_ErrorQuit( "gen no. %d in rel no. %d out of range", j,i );
                return 0;
            }
            ptRel[j] = ptInvs[-old];
        }
    }

    return 0;
}


/****************************************************************************
**
*F  FuncTzReplaceGens( <self>, <stack> )  replace Tietze generators by others
*/
libGAP_Obj libGAP_FuncTzReplaceGens (
    libGAP_Obj                 self,
    libGAP_Obj                 tietze )
{
    libGAP_Obj *               ptTietze;       /* pointer to this stack           */
    libGAP_Obj                 rels;           /* handle of the relators list     */
    libGAP_Obj *               ptRels;         /* pointer to this list            */
    libGAP_Obj                 lens;           /* handle of the lengths list      */
    libGAP_Obj *               ptLens;         /* pointer to this list            */
    libGAP_Obj                 flags;          /* handle of the flags list        */
    libGAP_Obj *               ptFlags;        /* pointer to this list            */
    libGAP_Obj                 invs;           /* handle of the inverses list     */
    libGAP_Obj *               ptInvs;         /* pointer to this list            */
    libGAP_Obj                 rel;            /* handle of a relator             */
    libGAP_Obj *               ptRel;          /* pointer to this relator         */
    libGAP_Obj *               pt1;            /* pointer to a relator            */
    libGAP_Obj *               pt2;            /* pointer to a relator            */
    libGAP_Int                 numgens;        /* number of Tietze generators     */
    libGAP_Int                 numrels;        /* number of Tietze relators       */
    libGAP_Int                 total;          /* total length of relators        */
    libGAP_Int                 old, new;       /* generators or inverses          */
    libGAP_Int                 leng, reduced;  /* relator lengths                 */
    libGAP_Int                 altered;        /* flag                            */
    libGAP_Int                 i, j;           /* loop variables                  */

    /* check the Tietze stack                                              */
    libGAP_CheckTietzeStack( tietze, &ptTietze );

    /* get and check the Tietze relators list                              */
    libGAP_CheckTietzeRelators( ptTietze, &rels, &ptRels, &numrels );

    /* get and check the Tietze lengths list                               */
    libGAP_CheckTietzeLengths( ptTietze, numrels, &lens, &ptLens );

    /* check list <lens> to contain the relator lengths                    */
    libGAP_CheckTietzeRelLengths( ptTietze, ptRels, ptLens, numrels, &total );

    /* get and check the Tietze flags list                                 */
    libGAP_CheckTietzeFlags( ptTietze, numrels, &flags, &ptFlags );

    /* get and check the Tietze inverses list                              */
    libGAP_CheckTietzeInverses( ptTietze, &invs, &ptInvs, &numgens );

    /* loop over all relators                                              */
    for ( i = 1; i <= numrels; i++ ) {
        rel = ptRels[i];
        pt2 = ptRel = libGAP_ADDR_OBJ( rel );
        leng = libGAP_INT_INTOBJ( ptLens[i] );
        altered = 0;

        /* don't change a square relator defining a valid involution       */
        if ( libGAP_INT_INTOBJ( ptFlags[i] ) == 3 && leng == 2 &&
            ptRel[1] == ptInvs[-libGAP_INT_INTOBJ(ptRel[1])] ) {
            continue;  /*  loop over i  */
        }

        /* run through the relator and replace the occurring generators    */
        for ( j = 1; j <= leng; j++ ) {

            old = libGAP_INT_INTOBJ( ptRel[j] );
            if ( old < -numgens || numgens < old || old == 0 ) {
                libGAP_ErrorQuit( "gen no. %d in rel no. %d out of range",
                           (libGAP_Int)j, (libGAP_Int)i );
                return 0;
            }

            new = libGAP_INT_INTOBJ( ptInvs[-old] );
            if ( ! new ) {
                altered = 1;
                continue;  /*  loop over j  */
            }

            if ( pt2 > ptRel && *pt2 == ptInvs[new] ) {
                altered = 1;
                --pt2;
            }
            else {
                if ( new != old )  { altered = 1; }
                *++pt2 = libGAP_INTOBJ_INT( new );
            }
        }

        if ( ! altered )  {
            continue; /*  loop over i  */
        }

        /* now cyclically reduce the relator                               */
        pt1 = ++ptRel;
        while ( pt1 < pt2 && *pt1 == ptInvs[libGAP_INT_INTOBJ(*pt2)] ) {
            ++pt1;  --pt2;
        }
        if ( ptRel < pt1 ) {
            while ( pt1 <= pt2 )  { *ptRel++ = *pt1++; }
            pt2 = --ptRel;
        }

        /* resize the resulting relator, if necessary                      */
        ptRel = libGAP_ADDR_OBJ( rel );
        reduced = pt2 - ptRel;
        if ( reduced < leng ) {
            libGAP_SET_LEN_PLIST( rel, reduced );
            ptLens[i] = libGAP_INTOBJ_INT( reduced );
            total = total - leng + reduced;
            libGAP_SHRINK_PLIST( rel, reduced );
            libGAP_CHANGED_BAG(rels);
            ptRels  = libGAP_ADDR_OBJ( rels );
            ptLens  = libGAP_ADDR_OBJ( lens );
            ptFlags = libGAP_ADDR_OBJ( flags );
            ptInvs  = libGAP_ADDR_OBJ( invs ) + (numgens + 1);
        }

        /*  Redefine the corresponding search flag                         */
        libGAP_ADDR_OBJ( flags )[i] = libGAP_INTOBJ_INT( 1 );
    }
    ptTietze = libGAP_ADDR_OBJ( tietze );
    ptTietze[TZ_TOTAL] = libGAP_INTOBJ_INT( total );

    return 0;
}


/****************************************************************************
**
*F  FuncTzSubstituteGen( <self>, <stack>, <gennum>, <word> )
*/
libGAP_Obj libGAP_FuncTzSubstituteGen (
    libGAP_Obj                 self,
    libGAP_Obj                 tietze,
    libGAP_Obj                 gennum,
    libGAP_Obj                 word )
{
    libGAP_Obj *               ptTietze;       /* pointer to this stack           */
    libGAP_Obj                 rels;           /* handle of the relators list     */
    libGAP_Obj *               ptRels;         /* pointer to this list            */
    libGAP_Obj                 lens;           /* handle of the lengths list      */
    libGAP_Obj *               ptLens;         /* pointer to this list            */
    libGAP_Obj                 flags;          /* handle of the flags list        */
    libGAP_Obj *               ptFlags;        /* pointer to this list            */
    libGAP_Obj                 invs;           /* handle of the inverses list     */
    libGAP_Obj *               ptInvs;         /* pointer to this list            */
    libGAP_Obj *               ptWrd;          /* pointer to this word            */
    libGAP_Obj                 iwrd;           /* handle of the inverse word      */
    libGAP_Obj *               ptIwrd;         /* pointer to this word            */
    libGAP_Obj                 new;            /* handle of a modified relator    */
    libGAP_Obj *               ptNew;          /* pointer to this relator         */
    libGAP_Obj                 rel;            /* handle of a relator             */
    libGAP_Obj *               ptRel;          /* pointer to this relator         */
    libGAP_Obj *               pt1;            /* pointer to a relator            */
    libGAP_Obj *               pt2;            /* pointer to a relator            */
    libGAP_Obj *               pt3;            /* pointer to a relator            */
    libGAP_Int                 numgens;        /* number of Tietze generators     */
    libGAP_Int                 numrels;        /* number of Tietze relators       */
    libGAP_Int                 total;          /* total length of relators        */
    libGAP_Int                 given;          /* given generator and inverse     */
    libGAP_Int                 gen, ginv;      /* given generator and inverse     */
    libGAP_Int                 next;           /* generator or inverse            */
    libGAP_Int                 leng, newleng;  /* relator lengths                 */
    libGAP_Int                 wleng;          /* length of the replacing word    */
    libGAP_Int                 occ;            /* number of occurrences           */
    libGAP_Int                 i, j;           /* loop variables                  */
    libGAP_Int                 alen,len;       /* number of changed relators */
    libGAP_Obj                 Idx;
    libGAP_Obj *               ptIdx;          /* List of changed relators */

    /* check the Tietze stack                                              */
    libGAP_CheckTietzeStack( tietze, &ptTietze );

    /* get and check the Tietze relators list                              */
    libGAP_CheckTietzeRelators( ptTietze, &rels, &ptRels, &numrels );

    /* get and check the Tietze lengths list                               */
    libGAP_CheckTietzeLengths( ptTietze, numrels, &lens, &ptLens );

    /* get and check the Tietze flags list                                 */
    libGAP_CheckTietzeFlags( ptTietze, numrels, &flags, &ptFlags );

    /* get and check the Tietze inverses list                              */
    libGAP_CheckTietzeInverses( ptTietze, &invs, &ptInvs, &numgens );

    /* check the second argument (generator number)                        */
    if ( ! libGAP_IS_INTOBJ(gennum) ) {
        libGAP_ErrorQuit( "<gennum> must be an integer", 0L, 0L );
        return 0;
    }
    given = libGAP_INT_INTOBJ(gennum);
    gen   = ( given > 0 ) ? given : -given;
    if ( gen <= 0 || numgens < gen ) {
        libGAP_ErrorQuit( "generator number %d out of range", (libGAP_Int)gen, 0L );
        return 0;
    }
    ginv = libGAP_INT_INTOBJ(ptInvs[gen]);

    /* check the third argument (replacing word)                           */
    if ( ! libGAP_IS_PLIST(word) ) {
        libGAP_ErrorQuit( "invalid replacing word", 0L, 0L );
        return 0;
    }
    ptWrd = libGAP_ADDR_OBJ(word);
    wleng = libGAP_LEN_PLIST(word);
    for ( i = 1;  i <= wleng;  i++ ) {
        next = libGAP_INT_INTOBJ( ptWrd[i] );
        if ( next < -numgens || next == 0 || next > numgens ) {
            libGAP_ErrorQuit( "entry [%d] of <Tietze word> out of range",
                       (libGAP_Int)i, 0L );
            return 0;
        }
    }

    /* check list <lens> to contain the relator lengths                    */
    libGAP_CheckTietzeRelLengths( ptTietze, ptRels, ptLens, numrels, &total );

    /* list of changed relator indices */
    len=0;
    alen=20;
    Idx=libGAP_NEW_PLIST( libGAP_T_PLIST, alen );
    libGAP_SET_LEN_PLIST(Idx,alen);
    ptIdx=libGAP_ADDR_OBJ(Idx);

    /* allocate a bag for the inverse of the replacing word                */
    iwrd   = libGAP_NEW_PLIST( libGAP_T_PLIST, wleng );
    ptRels = libGAP_ADDR_OBJ( rels );
    ptLens = libGAP_ADDR_OBJ( lens );
    ptInvs = libGAP_ADDR_OBJ( invs ) + (numgens + 1);
    ptWrd  = libGAP_ADDR_OBJ( word );
    ptIwrd = libGAP_ADDR_OBJ( iwrd );

    /* invert the replacing word                                           */
    libGAP_SET_LEN_PLIST( iwrd, wleng );
    pt1 = ptWrd;
    pt2 = ptIwrd + wleng;
    while ( pt2 > ptIwrd ) {
        *pt2-- = ptInvs[libGAP_INT_INTOBJ(*++pt1)];
    }
    if ( given < 0 ) {
        new = word;  word = iwrd;  iwrd = new;
        ptWrd = libGAP_ADDR_OBJ(word);  ptIwrd = libGAP_ADDR_OBJ(iwrd);
    }

    /* loop over all relators                                              */
    for ( i = 1;  i <= numrels;  i++ ) {
        /* We assume that ptRels, ptLens and ptIdx are valid at the 
           beginning of this loop (and not rendered invalid by a 
           garbage collection)! */
        rel = ptRels[i];
        ptRel = libGAP_ADDR_OBJ(rel);
        leng = libGAP_INT_INTOBJ(ptLens[i]);
        if ( leng == 0 )  { 
            continue;
        }

        /* run through the relator and count the occurrences of gen        */
        occ = 0;
        for ( j = 1;  j <= leng;  j++ ) {
            next = libGAP_INT_INTOBJ( ptRel[j] );
            if ( next < -numgens || numgens < next ) {
                libGAP_ErrorQuit( "gen no. %d in rel no. %d out of range",
                           (libGAP_Int)j, (libGAP_Int)i );
                return 0;
            }
            if (next == gen || next == ginv )
                ++occ;
        }
        if ( occ == 0 )  {
            continue;
        }

        /* mark that the relator changed */
        if (len>=alen) {
          alen+=100; /* more relators changed */
          libGAP_GROW_PLIST(Idx,alen);
          libGAP_SET_LEN_PLIST(Idx,alen);
          ptIdx=libGAP_ADDR_OBJ(Idx);
        }
        len+=1;
        ptIdx[len]=libGAP_INTOBJ_INT(i);
        libGAP_CHANGED_BAG(Idx);

        /* allocate a bag for the modified Tietze relator                  */
        new = libGAP_NEW_PLIST( libGAP_T_PLIST, leng + occ * (wleng - 1) );
        /* Now renew saved pointers into bags: */
        pt2 = ptNew = libGAP_ADDR_OBJ( new );
        ptIdx  = libGAP_ADDR_OBJ( Idx );
        ptLens = libGAP_ADDR_OBJ( lens );
        ptInvs = libGAP_ADDR_OBJ( invs ) + (numgens + 1);
        ptWrd  = libGAP_ADDR_OBJ( word );
        ptIwrd = libGAP_ADDR_OBJ( iwrd );
        ptRel  = libGAP_ADDR_OBJ( rel );

        /* now run again through the relator and modify it                 */
        for ( j = 1;  j <= leng;  j++ ) {
            next = libGAP_INT_INTOBJ( ptRel[j] );
            if ( next == gen || next == -gen ) {
                pt1 = ( next > 0 ) ? ptWrd : ptIwrd;
                pt3 = pt1 + wleng;
                while ( pt1 < pt3 ) {
                    ++pt1;
                    if ( pt2 > ptNew && *pt2 == ptInvs[libGAP_INT_INTOBJ(*pt1)] )
                        --pt2;
                    else
                        *++pt2 = *pt1;
                }
            }
            else {
                if ( pt2 > ptNew && *pt2 == ptInvs[next] )
                    --pt2;
                else
                    *++pt2 = libGAP_INTOBJ_INT( next );
            }
        }

        /*  now cyclically reduce the relator                              */
        pt1 = ++ptNew;
        while ( pt1 < pt2 && *pt1 == ptInvs[libGAP_INT_INTOBJ(*pt2)] ) {
            ++pt1;  --pt2;
        }
        if ( ptNew < pt1 ) {
            while ( pt1 <= pt2 )   *ptNew++ = *pt1++;
            pt2 = --ptNew;
        }

        /*  resize and save the resulting relator                          */
        ptNew = libGAP_ADDR_OBJ( new );
        newleng = pt2 - ptNew;
        libGAP_SET_LEN_PLIST( new, newleng );
        ptLens[i] = libGAP_INTOBJ_INT( newleng );
        total = total - leng + newleng;
        libGAP_SHRINK_PLIST( new, newleng );
        ptRels = libGAP_ADDR_OBJ( rels );
        ptLens = libGAP_ADDR_OBJ( lens );
        ptIdx  = libGAP_ADDR_OBJ( Idx );
        ptRels[i] = new;
        libGAP_ADDR_OBJ( flags )[i] = libGAP_INTOBJ_INT( 1 );
        libGAP_CHANGED_BAG(rels);
    }

    libGAP_SHRINK_PLIST(Idx,len);
    libGAP_SET_LEN_PLIST(Idx,len);
    libGAP_CHANGED_BAG(Idx);

    ptTietze = libGAP_ADDR_OBJ( tietze );
    ptTietze[TZ_TOTAL] = libGAP_INTOBJ_INT( total );

    return Idx;
}


/****************************************************************************
**
*F  FuncTzOccurrences( <self>, <args> ) . .  occurrences of Tietze generators
*/
libGAP_Obj libGAP_FuncTzOccurrences ( 
    libGAP_Obj                 self,
    libGAP_Obj                 args )
{
    libGAP_Obj                 tietze;         /* handle of the Tietze stack      */
    libGAP_Obj *               ptTietze;       /* pointer to the Tietze stack     */
    libGAP_Obj                 rels;           /* handle of the relators list     */
    libGAP_Obj *               ptRels;         /* pointer to this list            */
    libGAP_Obj                 res;            /* handle of the result            */
    libGAP_Obj                 cnts;           /* list of the counts              */
    libGAP_Obj *               ptCnts;         /* pointer to the counts list      */
    libGAP_Obj                 mins;           /* list of minimal occurence list  */
    libGAP_Obj *               ptMins;         /* pointer to the minimals list    */
    libGAP_Obj                 lens;           /* list of lengths of those        */
    libGAP_Obj *               ptLens;         /* pointer to the lengths list     */
    libGAP_Obj                 rel;            /* handle of a relator             */
    libGAP_Obj *               ptRel;          /* pointer to this relator         */
    libGAP_Obj                 aux;            /* auxiliary list                  */
    libGAP_Int  *              ptAux;          /* pointer to the lengths list     */
    libGAP_Int                 numgens;        /* number of Tietze generators     */
    libGAP_Int                 numrels;        /* number of Tietze relators       */
    libGAP_Int                 leng;           /* length of a relator             */
    libGAP_Int                 num, next;      /* generators or inverses          */
    libGAP_Int                 i, k;           /* loop variables                  */
    libGAP_Int                 c;              /* count of one generator          */
    libGAP_Int                 nr;             /* number of occurrences           */
    libGAP_Int                 nr1;            /* nr of occurrences in one word   */
    libGAP_Int                 nrm;            /* minimal value of 'nr1'          */
    libGAP_Int                 min;            /* word that has this minimum      */

    /* get and check arguments                                             */
    if ( ! libGAP_IS_SMALL_LIST(args) || 2 < libGAP_LEN_LIST(args) || libGAP_LEN_LIST(args) < 1 ) {
        libGAP_ErrorQuit( "usage: TzOccurrences( <Tietze stack>[, <gen no.> ] )",
                   0L, 0L );
        return 0;
    }

    /* check the first argument (Tietze stack)                             */
    tietze = libGAP_ELM_LIST( args, 1 );
    libGAP_CheckTietzeStack( tietze, &ptTietze );

    /* get and check the Tietze relators list                              */
    libGAP_CheckTietzeRelators( ptTietze, &rels, &ptRels, &numrels );
    numgens = libGAP_INT_INTOBJ(ptTietze[TZ_NUMGENS]);

    /* get and check the given generator number                            */
    if ( libGAP_LEN_LIST(args) == 2 ) {
        num = libGAP_INT_INTOBJ( libGAP_ELM_LIST(args,2) );
        if ( num <= 0 || numgens < num ) {
            libGAP_ErrorQuit( "given generator number out of range", 0L, 0L );
            return 0;
        }
        numgens = 1;
    }
    else {
        num = numgens;
    }

    /* allocate the result lists                                           */
    cnts = libGAP_NEW_PLIST( libGAP_T_PLIST, numgens );
    libGAP_SET_LEN_PLIST( cnts, numgens );
    for ( k = 1;  k <= numgens;  k++ )
        libGAP_ADDR_OBJ(cnts)[k] = libGAP_INTOBJ_INT(0);

    mins = libGAP_NEW_PLIST( libGAP_T_PLIST, numgens );
    libGAP_SET_LEN_PLIST( mins, 0 );

    lens = libGAP_NEW_PLIST( libGAP_T_PLIST, numgens );
    libGAP_SET_LEN_PLIST( lens, 0 );
    
    res = libGAP_NEW_PLIST( libGAP_T_PLIST, 3 );
    libGAP_SET_LEN_PLIST( res, 3 );
    libGAP_ADDR_OBJ(res)[1] = cnts;
    libGAP_ADDR_OBJ(res)[2] = mins;
    libGAP_ADDR_OBJ(res)[3] = lens;
    libGAP_CHANGED_BAG(res);

    /* allocate an auxiliary list                                          */
    ptAux = 0;
    if ( numgens > 1 ) {
        aux   = libGAP_NEW_STRING( (numgens+1)*sizeof(libGAP_Int) );
        ptAux = (libGAP_Int*)libGAP_ADDR_OBJ(aux);
        ptAux[0] = numgens;
        for ( k = 1;  k <= numgens;  k++ )
            ptAux[k] = 0;

    }

    /* now we can safely grab pointers                                     */
    ptRels = libGAP_ADDR_OBJ(rels);
    ptCnts = libGAP_ADDR_OBJ(cnts);
    ptLens = libGAP_ADDR_OBJ(lens);
    ptMins = libGAP_ADDR_OBJ(mins);

    /* handle special case of single generator in generator list           */
    if ( numgens == 1 ) {

        /* initialize the counters                                         */
        nr = 0;  nrm = 0;  min = 0;

        /* loop over all relators                                          */
        for ( i = 1;  i <= numrels;  i++ ) {
            rel = ptRels[i];
            if ( rel == 0 || ! libGAP_IS_PLIST(rel) ) {
                libGAP_ErrorQuit( "invalid entry [%d] in Tietze relators list",
                           (libGAP_Int)i, 0L );
                return 0;
            }
            ptRel = libGAP_ADDR_OBJ(rel);
            leng  = libGAP_LEN_PLIST(rel);

            /* loop over the letters of the relator                        */
            nr1 = 0;
            for ( k = 1;  k <= leng;  k++ ) {
                next = libGAP_INT_INTOBJ( ptRel[k] );
                if ( next == num || next == -num )  {
                    nr1++;
                }
            }

            /* check whether the number of occurrences of num is less than */
            /* in the preceding relators                                   */
            nr += nr1;
            if ( nrm == 0
              || (0 < nr1 && nr1 < nrm)
              || (nr1 == nrm && libGAP_LEN_PLIST(rel) < libGAP_LEN_PLIST(ptRels[min])) )
            {
                nrm = nr1;  min = i;
            }
        }

        /* put the information into the result bags                        */
        ptCnts[1] = libGAP_INTOBJ_INT( nr );
        if ( nr != 0 ) {
            ptCnts[1] = libGAP_INTOBJ_INT( nr );
            libGAP_SET_LEN_PLIST( lens, 1 );
            libGAP_SET_ELM_PLIST( lens, 1, libGAP_INTOBJ_INT(nrm) );
            libGAP_SET_LEN_PLIST( mins, 1 );
            libGAP_SET_ELM_PLIST( mins, 1, libGAP_INTOBJ_INT(min) );
        }
    }

    /* handle general case of all Tietze generators                        */
    else {

        /* loop over all relators                                          */
        for ( i = 1;  i <= numrels;  i++ ) {
            rel = ptRels[i];
            if ( rel == 0 || ! libGAP_IS_PLIST(rel) ) {
                libGAP_ErrorQuit( "invalid entry [%d] in Tietze relators list",
                           (libGAP_Int)i, 0L );
                return 0;
            }
            ptRel = libGAP_ADDR_OBJ(rel);
            leng  = libGAP_LEN_PLIST(rel);

            /* loop over the letters of the relator                        */
            for ( k = 1;  k <= leng;  k++ ) {
                next = libGAP_INT_INTOBJ( ptRel[k] );
                if ( next < 0 ) next = -next;
                if ( next == 0 || numgens < next ) {
                    libGAP_ErrorQuit( "invalid entry [%d][%d] in Tietze rels list",
                               (libGAP_Int)i, (libGAP_Int)k );
                    return 0;
                }
                (ptAux[next])++;
            }

            /* loop over the generators, collecting the counts             */
            for ( k = 1;  k <= numgens;  k++ ) {
                c = ptAux[k];
                if ( !c )
                    continue;
                ptAux[k] = 0;
                if ( ! libGAP_SUM_INTOBJS( ptCnts[k], ptCnts[k], libGAP_INTOBJ_INT(c) ) ) {
                    libGAP_ErrorQuit( "integer overflow", 0L, 0L );
                    return 0;
                }
                if ( 0 < c ) {
                    if ( ptLens[k] == 0 || c < libGAP_INT_INTOBJ(ptLens[k])
                      || (c == libGAP_INT_INTOBJ(ptLens[k]) 
                       && libGAP_LEN_PLIST(rel) 
                          < libGAP_LEN_PLIST(ptRels[libGAP_INT_INTOBJ(ptMins[k])])) )
                    {
                        ptLens[k] = libGAP_INTOBJ_INT(c);
                        ptMins[k] = libGAP_INTOBJ_INT(i);
                    }
                }
            }
        }

        /* find the correct length of the minimals and lengths lists       */
        k = numgens;
        while ( ptMins[k] == 0 )
            k--;
        libGAP_SET_LEN_PLIST( mins, k );
        libGAP_SET_LEN_PLIST( lens, k );
    }

    /* return the result                                                   */
    return res;
}


/****************************************************************************
**
*F  FuncTzOccurrencesPairs( <self>, <args> )  . . . . .  occurrences of pairs
*/
libGAP_Obj libGAP_FuncTzOccurrencesPairs (
    libGAP_Obj                 self,
    libGAP_Obj                 args )
{
    libGAP_Obj                 tietze;         /* handle of the Tietze stack      */
    libGAP_Obj *               ptTietze;       /* pointer to the Tietze stack     */
    libGAP_Obj                 rels;           /* handle of the relators list     */
    libGAP_Obj *               ptRels;         /* pointer to this list            */
    libGAP_Obj                 invs;           /* handle of the inverses list     */
    libGAP_Obj *               ptInvs;         /* pointer to this list            */
    libGAP_Obj                 res;            /* handle of the resulting list    */
    libGAP_Obj *               ptRes;          /* pointer to this list            */
    libGAP_Obj                 rel;            /* handle of a relator             */
    libGAP_Obj *               ptRel;          /* pointer to this relator         */
    libGAP_Obj                 numObj;         /* handle of generator number      */
    libGAP_Obj                 invObj;         /* handle of inverse gen number    */
    libGAP_Int                 num, i, ii;     /* generator numbers               */
    libGAP_Int                 numgens;        /* number of Tietze generators     */
    libGAP_Int                 numrels;        /* number of Tietze relators       */
    libGAP_Int                 length;         /* length of the current relator   */
    libGAP_Int                 j1, j2, r;      /* loop variables                  */

    /* get and check arguments                                             */
    if ( ! libGAP_IS_SMALL_LIST(args) || 3 < libGAP_LEN_LIST(args) || libGAP_LEN_LIST(args) < 2 ) {
        libGAP_ErrorQuit(
          "usage: TzOccurrencesPairs( <Tietze stack>, <gen>[, <list>] )",
          0L, 0L );
        return 0;
    }

    /* check the first argument (Tietze stack)                             */
    tietze = libGAP_ELM_LIST( args, 1 );
    libGAP_CheckTietzeStack( tietze, &ptTietze );

    /* get and check the Tietze relators list                              */
    libGAP_CheckTietzeRelators( ptTietze, &rels, &ptRels, &numrels );

    /* get and check the Tietze inverses list                              */
    libGAP_CheckTietzeInverses( ptTietze, &invs, &ptInvs, &numgens );

    /* get and check the Tietze generator number                           */
    numObj = libGAP_ELM_LIST( args, 2 );
    if ( ! libGAP_IS_INTOBJ(numObj) ) {
        libGAP_ErrorQuit( "<gen> must be a Tietze generator number", 0L, 0L );
        return 0;
    }
    num = libGAP_INT_INTOBJ(numObj);
    if ( num <= 0 || num > numgens ) {
        libGAP_ErrorQuit( "given generator number is out of range", 0L, 0L );
        return 0;
    }

    /*  Get and check the list for the results, if specified               */
    if ( libGAP_LEN_PLIST(args) == 2 ) {
        res = libGAP_NEW_PLIST( libGAP_T_PLIST, 4*numgens );
        libGAP_SET_LEN_PLIST( res, 4*numgens );
    }
    else {
        res = libGAP_ELM_LIST( args, 3 );
        if ( res == 0 || ! libGAP_IS_PLIST(res) || libGAP_LEN_PLIST(res) != 4*numgens ) {
            libGAP_ErrorQuit( "<list> must be a list of length %d",
                       (libGAP_Int)4*numgens, 0L );
            return 0;
        }
    }

    /*  return, if num = numgens                                           */
    if ( num == numgens )  {
        return res;
    }

    /*  get pointers to the involved lists                                 */
    ptRels = libGAP_ADDR_OBJ( rels );
    ptInvs = libGAP_ADDR_OBJ( invs ) + (numgens + 1);
    ptRes  = libGAP_ADDR_OBJ( res );

    /* get the handle of the inverse of the given generator                */
    invObj = ptInvs[num];

    /* ptRes[i]           counts the occurrences of gen * gen[i]           */
    /* ptRes[numgens+i]   counts the occurrences of gen * gen[i]^-1        */
    /* ptRes[2*numgens+i] counts the occurrences of gen^-1 * gen[i]        */
    /* ptRes[3*numgens+i] counts the occurrences of gen^-1 * gen[i]^-1     */

    /* initialize the counters                                             */
    for ( i = 1; i <= 4 * numgens; i++ ) {
        ptRes[i] = libGAP_INTOBJ_INT(0);
    }

    /* loop over the relators                                       */
    for ( r = 1; r <= numrels; r++ ) {
        rel = ptRels[r];
        if ( rel == 0 || ! libGAP_IS_PLIST(rel) ) {
            libGAP_ErrorQuit( "invalid Tietze relator [%d]", (libGAP_Int)r, 0L );
            return 0;
        }
        ptRel = libGAP_ADDR_OBJ(rel) + 1;

        /* skip the current relator if its length is less than 2           */
        length = libGAP_LEN_PLIST( rel );
        if ( length < 2 )  {
            continue;
        }

        /* loop over the current relator and investigate the pairs         */
        /* ( ptRel[j1], ptRel[j2] )                                        */
        j1 = length - 1;
        for ( j2 = 0;  j2 < length;  j1 = j2, j2++ ) {

            /* count any "forward" pair  gen * gen[i],  gen * gen[i]^-1,   */
            /* gen^-1 * gen[i],  or  gen^-1 * gen[i]^-1  ( with num < i )  */
            if ( ptRel[j1] == numObj || ptRel[j1] == invObj ) {
                i = libGAP_INT_INTOBJ( ptRel[j2] );
                if ( -num <= i && i <= num )  {
                    continue;
                }
                if ( i < -numgens || numgens < i ) {
                    libGAP_ErrorQuit( "invalid entry %d in Tietze relator [%d]",
                               (libGAP_Int)i, (libGAP_Int)r );
                    return 0;
                }
                if ( i < 0 )
                    i = numgens - i;
                if ( ptRel[j1] != numObj )
                    i = i + 2 * numgens;
                if ( ! libGAP_SUM_INTOBJS( ptRes[i], ptRes[i], libGAP_INTOBJ_INT(1) ) ) {
                    libGAP_ErrorQuit( "integer overflow", 0L, 0L );
                    return 0;
                }
            }

            /* count any "backward" pair  gen[i]^-1 * gen^-1,              */
            /* gen[i] * gen^-1,  gen[i]^-1 * gen,  or  gen[i] * gen        */
            /* ( with num < i )  which is not covered by a forward pair    */
            else if ( ptRel[j2] == numObj || ptRel[j2] == invObj ) {
                i = libGAP_INT_INTOBJ( ptRel[j1] );
                if ( -num <= i && i <= num )  {
                    continue;
                }
                if ( i < - numgens || numgens < i ) {
                    libGAP_ErrorQuit( "invalid entry %d in Tietze relator [%d]",
                               (libGAP_Int)i, (libGAP_Int)r );
                    return 0;
                }
                ii = libGAP_INT_INTOBJ( ptInvs[i] );
                if ( !( (numObj == invObj
                        && ptRel[(j2+1)%length] == libGAP_INTOBJ_INT(ii))
                     || (i == ii
                        && ptInvs[libGAP_INT_INTOBJ(ptRel[(j1+length-1)%length])] 
                           == ptRel[j2]) ) )
                {
                    if ( ii < 0 )
                        ii = numgens - ii;
                    if ( ptRel[j2] != invObj )
                        ii = ii + 2 * numgens;
                    if ( !libGAP_SUM_INTOBJS(ptRes[ii],ptRes[ii],libGAP_INTOBJ_INT(1)) ) {
                        libGAP_ErrorQuit( "integer overflow", 0L, 0L );
                        return 0;
                    }
                }
            }
        }
    }

    return res;
}


/****************************************************************************
**
*F  FuncTzSearchC( <self>, <args> ) . find subword matches in Tietze relators
*/
libGAP_Obj libGAP_FuncTzSearchC (
    libGAP_Obj                 self,
    libGAP_Obj                 args )
{
    libGAP_Obj                 tietze;         /* handle of the Tietze stack      */
    libGAP_Obj *               ptTietze;       /* pointer to this stack           */
    libGAP_Obj                 rels;           /* handle of the relators list     */
    libGAP_Obj *               ptRels;         /* pointer to this list            */
    libGAP_Obj                 lens;           /* handle of the lengths list      */
    libGAP_Obj *               ptLens;         /* pointer to this list            */
    libGAP_Obj                 invs;           /* handle of the inverses list     */
    libGAP_Obj *               ptInvs;         /* pointer to this list            */
    libGAP_Obj                 flags;          /* handle of the flags list        */
    libGAP_Obj *               ptFlags;        /* pointer to this list            */
    libGAP_Obj                 word;           /* handle of the given relator     */
    libGAP_Obj                 lo;             /* handle of current list relator  */
    libGAP_Obj                 wo;             /* handle of a relator             */
    libGAP_Obj                 tmp;            /* handle of the second argument   */
    libGAP_Obj                 equ;            /* handle of the fifth argument    */
    libGAP_UInt1               keys1[8192];    /* hash table of key values        */
    libGAP_UInt1               keys2[8192];    /* hash table of key values        */
    libGAP_UInt1               keys3[8192];    /* hash table of key values        */
    libGAP_UInt                inv;            /* inverse for key computation     */
    libGAP_UInt                key;            /* key value of subword            */
    libGAP_Int                 numgens;        /* number of Tietze generators     */
    libGAP_Int                 numrels;        /* number of Tietze relators       */
    libGAP_Int                 total;          /* total length of relators        */
    libGAP_Obj *               ptr;            /* pointer to a relator            */
    libGAP_Obj *               v;              /* pointers into relators          */
    libGAP_Obj *               w;              /* pointers into relators          */
    libGAP_Obj *               ptx;            /* pointers into relators          */
    libGAP_Obj *               pty;            /* pointers into relators          */
    libGAP_Int                 i1, j1;         /* loop variables                  */
    libGAP_Int                 i2, j2;         /* loop variables                  */
    libGAP_Int                 i3, j3;         /* loop variables                  */
    libGAP_Int                 len1;           /* relator length                  */
    libGAP_Int                 lmin, lmax;     /* bound for relator lengths       */
    libGAP_Int                 pos1, pos2;     /* position of the given relator   */
    libGAP_Int                 xmax;           /* position of the given relator   */
    libGAP_Int                 newflag, flag1; /* Tietze relator flags            */
    libGAP_Int                 xflag, yflag;   /* Tietze relator flags            */
    libGAP_Int                 xlen, xlen1;    /* length of the given relator     */
    libGAP_Int                 mlen;           /* length of the wanted match      */
    libGAP_Int                 ylen, ylen1;    /* length of the current relator   */
    libGAP_Int                 newlen;         /* length of a new relator         */
    libGAP_Int                 n, m;           /* subword lengths                 */
    libGAP_Int                 count;          /* number of altered relators      */
    libGAP_Int                 i, j, jj, x, y; /* loop variables                  */
    libGAP_Int                 lasty;          /* flag                            */
    libGAP_Int                 altered;        /* flag                            */
    libGAP_Int                 equal;          /* flag                            */

    /* get and check arguments                                             */
    if ( ! libGAP_IS_SMALL_LIST(args) || 4 < libGAP_LEN_LIST(args) || libGAP_LEN_LIST(args) < 3 ) {
        libGAP_ErrorQuit(
          "usage: TzSearchC( <Tietze stack>, <pos1>, <pos2>[, <equal>] )",
          0L, 0L );
        return 0;
    }

    /* check the first argument (Tietze stack)                             */
    tietze = libGAP_ELM_LIST( args, 1 );
    libGAP_CheckTietzeStack( tietze, &ptTietze );

    /* get and check the Tietze relators list                              */
    libGAP_CheckTietzeRelators( ptTietze, &rels, &ptRels, &numrels );

    /* get and check the Tietze lengths list                               */
    libGAP_CheckTietzeLengths( ptTietze, numrels, &lens, &ptLens );

    /* get and check the Tietze flags list                                 */
    libGAP_CheckTietzeFlags( ptTietze, numrels, &flags, &ptFlags );

    /* check list <lens> to contain the relator lengths                    */
    libGAP_CheckTietzeRelLengths( ptTietze, ptRels, ptLens, numrels, &total );

    /* get and check the Tietze inverses list                              */
    libGAP_CheckTietzeInverses( ptTietze, &invs, &ptInvs, &numgens );

    /* check the second argument                                           */
    tmp = libGAP_ELM_LIST( args, 2 );
    if ( ! libGAP_IS_INTOBJ(tmp) ) {
        libGAP_ErrorQuit( "<pos1> must be a positive int", 0L ,0L );
        return 0;
    }
    pos1 = libGAP_INT_INTOBJ(tmp);
    if ( pos1 > numrels ) {
        libGAP_ErrorQuit( "<pos1> out of range: %d", (libGAP_Int)pos1, 0L );
        return 0;
    }

    /* check the third argument                                            */
    tmp = libGAP_ELM_LIST( args, 3 );
    if ( ! libGAP_IS_INTOBJ(tmp) ) {
        libGAP_ErrorQuit( "<pos2> must be a positive int", 0L ,0L );
        return 0;
    }
    pos2 = libGAP_INT_INTOBJ(tmp);
    if ( pos2 > numrels ) {
        libGAP_ErrorQuit( "<pos2> out of range: %d", (libGAP_Int)pos2, 0L );
        return 0;
    }

    /* check the fourth argument                                           */
    if ( libGAP_LEN_LIST(args) == 3 ) {
        equ = libGAP_False;
    }
    else {
        equ = libGAP_ELM_LIST( args, 4 );
        if ( equ != libGAP_False && equ != libGAP_True ) {
            libGAP_ErrorQuit( "<equal> must be false or true", 0L, 0L );
            return 0;
        }
    }
    equal = ( equ == libGAP_True );

    /* Skip relators of inconvenient lengths or with inconvenient flags,   */
    /*  and return if the remaining range is empty                         */
    while ( pos1 <= pos2
        && (libGAP_INT_INTOBJ( ptLens[pos1] ) < 2
         || libGAP_INT_INTOBJ( ptFlags[pos1] ) > 1
         || (equal && ( libGAP_INT_INTOBJ( ptLens[pos1] ) < 4
                     || libGAP_INT_INTOBJ( ptLens[pos1] ) % 2 == 1 ) ) ) )
    {
        pos1++;
    }
    if ( pos1 > pos2 || pos1 == numrels ) {
        return libGAP_INTOBJ_INT(0);
    }

    /* get the range of compatible relator lengths                         */
    len1 = libGAP_INT_INTOBJ( ptLens[pos1] );
    lmin = len1 - ( len1 % 2 );
    lmax = ( equal ) ? lmin : lmin + 1;

    /* initialize some variables                                           */
    newflag = ( equal ) ? 1 : 2;
    count = 0;
    lasty = 0;
    xmax = pos1 - 1;
    flag1 = libGAP_INT_INTOBJ( ptFlags[pos1] );

    /* Compute the length of the wanted match and the corresponding        */
    /*  inverse factor                                                     */
    mlen = equal ? ( lmin + 1 ) / 2 : lmin / 2 + 1;
    inv = 1;
    for ( i = 1; i <= mlen; i++ )
       inv = 109109 * inv;

    /* initialize the hash table                                           */
    for ( i = 0; i < 2048; i++ )
       ((libGAP_UInt4 *)keys1)[i] = ((libGAP_UInt4 *)keys2)[i] = ((libGAP_UInt4 *)keys3)[i] = 0;

    /* loop over the Tietze relators, starting at position pos1            */
    for ( y = pos1;  y < numrels;  ) {
       word  = ptRels[y];
       ylen  = libGAP_INT_INTOBJ( ptLens[y] );
       yflag = libGAP_INT_INTOBJ( ptFlags[y] );
       if ( y <= pos2 && lmin <= ylen && ylen <= lmax && yflag <= 1 ) {

          /* add the key values of the current relator to the hash table   */
          ptr = libGAP_ADDR_OBJ(word);

          key = 0;

          for ( i = 0, w = ptr+1;  i < mlen;  i++, w++ )
             key = 109109 * key + ((libGAP_UInt)*w >> 2);

          for ( i = 0, v = ptr+1, w = v+mlen; i < ylen; i++, v++, w++ ) {
             keys1[ key & 8191 ] = 1;
             keys2[ (key >> 11) & 8191 ] |= (1 << ((key >> 8) & 7));
             keys3[ (key >> 19) & 8191 ] |= (1 << ((key >> 16) & 7));
             if ( i == ylen-mlen )
                 w = ptr+1;
             key = 109109 * key - inv * ((libGAP_UInt)*v >> 2) + ((libGAP_UInt)*w >> 2);
          }

          key = 0;

          for ( i = 0, w = ptr+ylen; i < mlen; i++, w-- ) {
             key = 109109 * key + ((libGAP_UInt) ptInvs[libGAP_INT_INTOBJ(*w)] >> 2);
          }

          for ( i = 0, v = ptr+ylen, w = v-mlen; i < ylen; i++, v--, w-- ) {
             keys1[ key & 8191 ] = 1;
             keys2[ (key >> 11) & 8191 ] |= (1 << ((key >> 8) & 7));
             keys3[ (key >> 19) & 8191 ] |= (1 << ((key >> 16) & 7));
             if ( i == ylen-mlen )
                 w = ptr+ylen;
             key = 109109 * key
                 - inv * ((libGAP_UInt) ptInvs[libGAP_INT_INTOBJ(*v)] >> 2)
                 + ( (libGAP_UInt) ptInvs[libGAP_INT_INTOBJ(*w)] >> 2 );
          }
          if ( len1 > ylen )
              len1 = ylen;
          if ( flag1 < yflag )
              flag1 = yflag;
          xmax = y;
       }

       /* move to next relator                                             */
       y++;

       /* initialize some variables                                        */
       lo      = ptRels[y];
       ylen    = libGAP_INT_INTOBJ( ptLens[y] );
       yflag   = libGAP_INT_INTOBJ( ptFlags[y] );
       ylen1   = ylen - 1;
       altered = 0;

       /* Loop to the next relator, if the current relator is too short    */
       if ( y > lasty
         && (ylen < len1 || yflag > 1 || (!equal && !(yflag + flag1)) ) )
       {
          continue;  /*  loop over y */
       }
       lasty = y;

       /* Compute the key values of the current relator                    */
       ptr = libGAP_ADDR_OBJ(lo);

       key = 0;

       for ( j = 0, w = ptr+1; j < mlen; j++, w++ )
          key = 109109 * key + ( (libGAP_UInt)*w >> 2 );

       for ( j = 0; j < ylen; j++ ) {

          /* check for key match in the tables                             */
          if ( keys1[ key & 8191 ]
             && (keys2[ (key >> 11) & 8191 ] & (1 << ((key >> 8) & 7)))
             && (keys3[ (key >> 19) & 8191 ] & (1 << ((key >> 16) & 7))) ){

             /* loop over the (relevant) given relators                    */
             for ( x = pos1; x <= xmax; x++ ) {

                wo    = ptRels[x];
                xlen  = libGAP_INT_INTOBJ( ptLens[x] );
                xflag = libGAP_INT_INTOBJ( ptFlags[x] );
                if ( xlen < len1 || xlen > lmax || xlen > ylen
                  || xflag > 1 || (!equal && !( xflag + yflag )) )
                {
                   continue;  /*  loop over x */
                }

                xlen1 = xlen - 1;
                ptx = libGAP_ADDR_OBJ(wo) + 1;
                pty = libGAP_ADDR_OBJ(lo) + 1;

                /* loop over all possible positions in the given relator   */
                for ( i = 0; i < xlen; i++ ) {

                   /* search forward for a match of length at least mlen   */
                   i2 = i;  j2 = j;
                   for ( n = 0; n < xlen; n++,
                      i2 = (i2 == xlen1) ? 0 : i2 + 1,
                      j2 = (j2 == ylen1) ? 0 : j2 + 1 ) {
                      if ( ptx[i2] != pty[j2] )
                          break;  /* loop over n */
                   }
                   if ( n < mlen )  continue;  /*  loop over i             */

                   /* search backward to find the whole match              */
                   i1 = (i == 0) ? xlen1 : i - 1;
                   j1 = (j == 0) ? ylen1 : j - 1;
                   for ( ; n < xlen; n++,
                      i1 = (i1 == 0) ? xlen1 : i1 - 1,
                      j1 = (j1 == 0) ? ylen1 : j1 - 1 )
                   {
                      if ( ptx[i1] != pty[j1] )
                          break;  /* loop over n */
                   }

                   /* replace a matching substring of equal length         */
                   if ( n == xlen - n ) {
                      j2 = j;
                      for ( m = 0; m < n; m++,
                         i1 = (i1 == 0) ? xlen1 : i1 - 1,
                         j2 = (j2 == ylen1) ? 0 : j2 + 1 )
                      {
                         pty[j2] = ptInvs[libGAP_INT_INTOBJ(ptx[i1])];
                      }

                      /* Now replace all exact occurrences of this string  */
                      /*  in the current word (not relator)                */
                      i3 = (i + n) % xlen;

                      for ( jj = 0; jj <= ylen - n; jj++ ) {
                         i2 = i;  j2 = jj;
                         for ( m = 0; m < n; m++,
                            i2 = (i2 == xlen1) ? 0 : i2 + 1,
                            j2 = (j2 == ylen1) ? 0 : j2 + 1 ) {
                            if ( ptx[i2] != pty[j2] )
                                break;  /* loop over m */
                         }
                         if ( m < n )
                             continue;  /* loop over jj */

                         i1 = (i == 0) ? xlen1 : i - 1;
                         if ( ptx[i1] == pty[(jj + ylen1) % ylen] ||
                            ptx[i3] == pty[(jj + n) % ylen] )
                         {
                            continue;  /* loop over jj */
                         }

                         j2 = jj;
                         for ( m = 0; m < n; m++,
                            i1 = (i1 == 0) ? xlen1 : i1 - 1,
                            j2 = (j2 == ylen1) ? 0 : j2 + 1 ) {
                            pty[j2] = ptInvs[libGAP_INT_INTOBJ(ptx[i1])];
                         }

                         jj = -1;
                      }

                      ptFlags[y] = libGAP_INTOBJ_INT( newflag );
                      altered = 1;
                      ++count;
                      break;  /* loop over i */
                   }

                   m = ylen - n;  n = xlen - n;

                   /* find all canceling factors                           */
                   if ( n == 0 ) {
                      for ( ; 1 < m; m -= 2,
                         j1 = (j1 == 0) ? ylen1 : j1 - 1,
                         j2 = (j2 == ylen1) ? 0 : j2 + 1 )
                      {
                         if ( pty[j1] != ptInvs[libGAP_INT_INTOBJ(pty[j2])] )
                            break;  /*  loop over m                        */
                      }
                   }

                   /* create the modified relator and save it              */
                   newlen = m + n;
                   if ( j2 > 0 ) {
                      if ( j2 <= j1 )  {
                          jj = 0;  j3 = j1;  j1 = m - 1;
                      }
                      else  {
                          jj = j1 + n + 1;  j3 = ylen - 1;
                      }
                      for ( ; j2 <= j3; ) {
                         pty[jj++] = pty[j2++];
                      }
                   }
                   for ( ; n > 0; n--, i1 = (i1 == 0) ? xlen1 : i1 - 1 ) {
                      pty[++j1] = ptInvs[libGAP_INT_INTOBJ(ptx[i1])];
                   }
                   libGAP_SET_LEN_PLIST( lo, newlen );
                   ptLens[y] = libGAP_INTOBJ_INT(newlen);
                   total = total - ylen + newlen;
                   ptFlags[y] = libGAP_INTOBJ_INT(newflag);

                   /* reduce the bag size                                  */
                   libGAP_SHRINK_PLIST( lo, newlen );
                   libGAP_CHANGED_BAG(rels);
                   ptRels  = libGAP_ADDR_OBJ( rels );
                   ptLens  = libGAP_ADDR_OBJ( lens );
                   ptFlags = libGAP_ADDR_OBJ( flags);
                   ptInvs  = libGAP_ADDR_OBJ( invs ) + (numgens + 1);

                   altered = 1;
                   ++count;
                   --y;
                   break;  /* loop over i */
                }

                if ( altered )
                    break;  /* loop over x */

                /* now try the inverse of the given relator                */
                for ( i = 0;  i < xlen;  i++ ) {

                   /* search forward for a match of length at least mlen   */
                   i2 = xlen1 - i;  j2 = j;
                   for ( n = 0; n < xlen; n++,
                      i2 = (i2 == 0) ? xlen1 : i2 - 1,
                      j2 = (j2 == ylen1) ? 0 : j2 + 1 )
                   {
                      if ( ptInvs[libGAP_INT_INTOBJ(ptx[i2])] != pty[j2] )
                         break;  /* loop over n */
                   }
                   if ( n < mlen )
                       continue;  /* loop over i */

                   /* search backward to find the whole match              */
                   i1 = (i == 0) ? 0 : xlen - i;
                   j1 = (j == 0) ? ylen1 : j - 1;
                   for ( ; n < xlen; n++,
                      i1 = (i1 == xlen1) ? 0 : i1 + 1,
                      j1 = (j1 == 0) ? ylen1 : j1 - 1 )
                   {
                      if ( ptInvs[libGAP_INT_INTOBJ(ptx[i1])] != pty[j1] )
                         break;  /* loop over n */
                   }

                   /* replace a matching substring of equal length         */
                   if ( n == xlen - n ) {
                      j2 = j;
                      for ( m = 0; m < n; m++,
                         i1 = (i1 == xlen1) ? 0 : i1 + 1,
                         j2 = (j2 == ylen1) ? 0 : j2 + 1 )
                      {
                         pty[j2] = ptx[i1];
                      }

                      ptFlags[y] = libGAP_INTOBJ_INT( newflag );
                      altered = 1;
                      ++count;
                      break;  /* loop over i */
                   }

                   m = ylen - n;  n = xlen - n;

                   /* Find all canceling factors                           */
                   if ( n == 0 ) {
                      for ( ; 1 < m; m -= 2,
                         j1 = (j1 == 0) ? ylen1 : j1 - 1,
                         j2 = (j2 == ylen1) ? 0 : j2 + 1 )
                      {
                         if ( pty[j1] != ptInvs[libGAP_INT_INTOBJ(pty[j2])] )
                            break;  /* loop over m */
                      }
                   }

                   /* create the modified relator and save it              */
                   newlen = m + n;
                   if ( j2 > 0 )  {
                      if ( j2 <= j1 )  {
                          jj = 0;  j3 = j1;  j1 = m - 1;
                      }
                      else  {
                          jj = j1 + n + 1;  j3 = ylen - 1;
                      }
                      for ( ;  j2 <= j3;  ) {
                         pty[jj++] = pty[j2++];
                      }
                   }
                   for ( ;  n > 0;  n--, i1 = (i1 == xlen1) ? 0 : i1 + 1 ) {
                      pty[++j1] = ptx[i1];
                   }
                   libGAP_SET_LEN_PLIST( lo, newlen );
                   ptLens[y] = libGAP_INTOBJ_INT(newlen);
                   total = total - ylen + newlen;
                   ptFlags[y] = libGAP_INTOBJ_INT(newflag);

                   /* reduce the bag size                                  */
                   libGAP_SHRINK_PLIST( lo, newlen );
                   libGAP_CHANGED_BAG(rels);
                   ptRels  = libGAP_ADDR_OBJ( rels );
                   ptLens  = libGAP_ADDR_OBJ( lens );
                   ptFlags = libGAP_ADDR_OBJ( flags);
                   ptInvs  = libGAP_ADDR_OBJ( invs ) + numgens + 1;

                   altered = 1;
                   ++count;
                   --y;
                   break;  /* loop over i */
                }

                if ( altered )
                    break;  /* loop over x */
             }
          }

          if ( altered )
              break;  /* loop over j */

          v = ptr + ( 1 + j ); 
          w = ptr + ( 1 + ( j + mlen ) % ylen );
          key = 109109 * key - inv * ( (libGAP_UInt)*v >> 2 )
              + ( (libGAP_UInt)*w >> 2 );
       }
    }

    libGAP_ADDR_OBJ( tietze )[TZ_TOTAL] = libGAP_INTOBJ_INT(total);

    /* return the number of altered relators                               */
    return libGAP_INTOBJ_INT( count );
}


/* rewriting using tz form relators */

libGAP_Obj  libGAP_FuncReduceLetterRepWordsRewSys (
 libGAP_Obj  self,
 libGAP_Obj  tzrules,
 libGAP_Obj  a_w )
{
 libGAP_UInt n,lt,i,k,p,j,lrul,eq,rlen,newlen,a;
 libGAP_Obj w,nw,rul;
 libGAP_Obj * wa;
 libGAP_Obj * nwa;
 
 w=a_w;

 /* n := Length( w ); */
 n=libGAP_LEN_PLIST(w);
 
 /* lt := Length( tzrules ); */
 lt=libGAP_LEN_PLIST(tzrules);
 
 /* i := 1; */
 i=1;
 
 /* while i in [ 1 .. n ] od */
 while (i<=n) {
   libGAP_TakeInterrupt();
  
  /* k := 1; */
  k=1;
  
  /* while k in [ 1 .. lt ] od */
  while (k<=lt) {
    
    /* rul := tzrules[k][1]; */
    rul = libGAP_ELM_PLIST(tzrules,k);
    rul = libGAP_ELM_PLIST(rul,1);
    lrul = libGAP_LEN_PLIST(rul);
   
   /* if Length( tzrules[k][1] ) <= i then */
   if (lrul<=i) {
    
    /* eq := true; */
    eq=1;

    /* p := i; */
    p=i;
    
    /* j := Length( rul ); */
    j=lrul;
    
    /* while eq and j > 0 od */
    while ((eq==1) && (j>0) ) {
     
     /* eq := w[p] = rul[j]; */
     eq=((libGAP_ELM_LIST(w,p)==libGAP_ELM_LIST(rul,j))?1:0);
     
     /* p := p - 1; */
     p--;
     
     /* j := j - 1; */
     j--;
     
    }
    /* od */
    
    /* if eq then */
    if (eq==1) {
     
     /* make the new plist */

     rlen=libGAP_LEN_PLIST(libGAP_ELM_PLIST(libGAP_ELM_PLIST(tzrules,k),2));
     newlen = n-lrul+rlen;

     if (newlen==0) {
       nw=libGAP_NEW_PLIST(libGAP_T_PLIST_EMPTY,0);
     }
     else {
        /* make space for the new word */
        nw = libGAP_NEW_PLIST(libGAP_TNUM_OBJ(w),newlen);

       /* addresses */
       wa=libGAP_ADDR_OBJ(w);
       nwa=libGAP_ADDR_OBJ(nw);
       wa++;
       nwa++;

       /* for a in [ 1 .. p ] do */
       /* Add( nw, w[a] ); */
       for (a=1; a<=p;a++) {
         *nwa++=*wa++;
       }
       /* od */

       /* rul := tzrules[k][2]; */
       rul = libGAP_ELM_PLIST(tzrules,k);
       rul = libGAP_ELM_PLIST(rul,2);
       wa=libGAP_ADDR_OBJ(rul);
       wa++;

       /* for a in [ 1 .. Length( rul ) ] do */
       /* Add( nw, rul[a] ); */
       for (a=1;a<=rlen;a++) {
         *nwa++=*wa++;
       }
       /* od */

       /* for a in [ i + 1 .. n ] do */
       /* there must be a better way for giving this address ... */
       wa=(libGAP_Obj*) &(libGAP_ADDR_OBJ(w)[i+1]);
       /* Add( nw, w[a] ); */
       for (a=i+1;a<=n;a++) {
         *nwa++=*wa++;
       }
       /* od */

     }

     /* w := nw; */
     libGAP_SET_LEN_PLIST(nw,newlen);
     w = nw;
     
     /* i := i - Length( tzrules[k][1] ); */
     i=i-lrul;
     
     /* n := Length( w ); */
     n=newlen;
     
     /* k := lt; */
     k = lt;
     
    }
    /* fi */
    
   }
   /* fi */
   
   /* k := k + 1; */
   k++;
   
  }
  /* od */
  
  /* i := i + 1; */
  i++;
  
 }
 /* od */
 
 /* return w; */
 return w;
 
}



/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/

/****************************************************************************
**

*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "TzSortC", 1, "tietze",
      libGAP_FuncTzSortC, "src/tietze.c:TzSortC" },

    { "TzRenumberGens", 1, "tietze",
      libGAP_FuncTzRenumberGens, "src/tietze.c:TzRenumberGens" },
    
    { "TzReplaceGens", 1, "tietze",
      libGAP_FuncTzReplaceGens, "src/tietze.c:TzReplaceGens" },

    { "TzSubstituteGen", 3, "tietze, gennum, word",
      libGAP_FuncTzSubstituteGen, "src/tietze.c:TzSubstituteGen" },

    { "TzOccurrences", -1, "args",
      libGAP_FuncTzOccurrences, "src/tietze.c:TzOccurrences" },

    { "TzOccurrencesPairs", -1, "args",
      libGAP_FuncTzOccurrencesPairs, "src/tietze.c:TzOccurrencesPairs" },

    { "TzSearchC", -1, "args",
      libGAP_FuncTzSearchC, "src/tietze.c:TzSearchC" },

    { "REDUCE_LETREP_WORDS_REW_SYS", 2, "tzwords, word",
  libGAP_FuncReduceLetterRepWordsRewSys,"src/tietze.c:REDUCE_LETREP_WORDS_REW_SYS" },

    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoTietze()  . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "tietze",                           /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoTietze ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}


/****************************************************************************
**

*E  tietze.c  . . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
