/****************************************************************************
**
*W  rational.c                  GAP source                   Martin Schönert
**
**
*Y  Copyright (C)  1996,  Lehrstuhl D für Mathematik,  RWTH Aachen,  Germany
*Y  (C) 1998 School Math and Comp. Sci., University of St Andrews, Scotland
*Y  Copyright (C) 2002 The GAP Group
**
**  This file contains  the  functions  for  the  artithmetic  of  rationals.
**
**  Rationals  are  the union of  integers  and fractions.   A fraction  is a
**  quotient of two integers where the denominator is relatively prime to the
**  numerator.  If in the description of a function we  use the term rational
**  this  implies  that the  function is also   capable of handling integers,
**  though its  function would usually  be performed   by  a routine  in  the
**  integer package.  We will use the  term fraction to  stress the fact that
**  something must not be an integer.
**
**  A  fraction is represented  as a pair of  two integers.  The first is the
**  numerator and the second  is  the  denominator.  This representation   is
**  always reduced, i.e., numerator and denominator  are relative prime.  The
**  denominator is always  positive and  greater than 1.    If it were 1  the
**  fraction would be an integer and would be represented as  integer.  Since
**  the denominator is always positive the numerator carries the sign of  the
**  fraction.
**
**  It  is very easy to  see  that for every  fraction   there is one  unique
**  reduced representation.  Because of   this comparisons of  fractions  are
**  quite easy,  we just compare  numerator  and denominator.  Also numerator
**  and denominator are as small as possible,  reducing the effort to compute
**  with them.   Of course  computing  the reduced  representation comes at a
**  cost.   After every arithmetic operation we  have to compute the greatest
**  common divisor of numerator and denominator, and divide them by the gcd.
**
**  Effort  has been made to improve  efficiency by avoiding unneccessary gcd
**  computations.  Also if  possible this  package will compute  two gcds  of
**  smaller integers instead of one gcd of larger integers.
**
**  However no effort has  been made to write special  code for the case that
**  some of the  integers are small integers   (i.e., less than  2^28).  This
**  would reduce the overhead  introduced by the  calls to the functions like
**  'SumInt', 'ProdInt' or 'GcdInt'.
*/
#include        "system.h"              /* system dependent part           */


#include        "gasman.h"              /* garbage collector               */

#include        "objects.h"             /* objects                         */
#include        "scanner.h"             /* scanner                         */

#include        "gap.h"                 /* error handling, initialisation  */
#include        "gvars.h"               /* global variables                */
#include        "calls.h"               /* generic call mechanism          */
#include        "opers.h"               /* generic operations              */

#include        "ariths.h"              /* basic arithmetic                */

#include        "bool.h"                /* booleans                        */

#include        "integer.h"             /* integers                        */

#include        "rational.h"            /* rationals                       */

#include        "records.h"             /* generic records                 */
#include        "precord.h"             /* plain records                   */

#include        "lists.h"               /* generic lists                   */
#include        "string.h"              /* strings                         */

#include        "saveload.h"            /* saving and loading              */


/****************************************************************************
**

*F  NUM_RAT(<rat>)  . . . . . . . . . . . . . . . . . numerator of a rational
*F  DEN_RAT(<rat>)  . . . . . . . . . . . . . . . . denominator of a rational
*/
#define libGAP_NUM_RAT(rat)    libGAP_ADDR_OBJ(rat)[0]
#define libGAP_DEN_RAT(rat)    libGAP_ADDR_OBJ(rat)[1]

#if 0
#define libGAP_CHECK_RAT(rat) if (libGAP_TNUM_OBJ(rat) == libGAP_T_RAT && \
                           libGAP_GcdInt(libGAP_NUM_RAT(rat),libGAP_DEN_RAT(rat)) != libGAP_INTOBJ_INT(1)) \
                             libGAP_ErrorQuit("bad rational",0L,0L)
#else
#define libGAP_CHECK_RAT(rat)
#endif


/****************************************************************************
**
*F  TypeRat( <rat> )  . . . . . . . . . . . . . . . . . .  kind of a rational
**
**  'TypeRat' returns the kind of the rational <rat>.
**
**  'TypeRat' is the function in 'TypeObjFuncs' for rationals.
*/
libGAP_Obj             libGAP_TYPE_RAT_POS;
libGAP_Obj             libGAP_TYPE_RAT_NEG;

libGAP_Obj             libGAP_TypeRat (
    libGAP_Obj                 rat )
{
    libGAP_Obj                 num;
    libGAP_CHECK_RAT(rat);
    num = libGAP_NUM_RAT(rat);
    if ( libGAP_IS_INTOBJ(num) ) {
        if ( 0 < libGAP_INT_INTOBJ(num) ) {
            return libGAP_TYPE_RAT_POS;
        }
        else {
            return libGAP_TYPE_RAT_NEG;
        }
    }
    else {
        if ( libGAP_TNUM_OBJ(num) == libGAP_T_INTPOS ) {
            return libGAP_TYPE_RAT_POS;
        }
        else {
            return libGAP_TYPE_RAT_NEG;
        }
    }
}


/****************************************************************************
**
*F  PrintRat( <rat> ) . . . . . . . . . . . . . . . . . . .  print a rational
**
**  'PrintRat' prints a rational <rat> in the form
**
**      <numerator> / <denominator>
*/
void            libGAP_PrintRat (
    libGAP_Obj                 rat )
{
    libGAP_Pr( "%>", 0L, 0L );
    libGAP_PrintObj( libGAP_NUM_RAT(rat) );
    libGAP_Pr( "%</%>", 0L, 0L );
    libGAP_PrintObj( libGAP_DEN_RAT(rat) );
    libGAP_Pr( "%<", 0L, 0L );
}


/****************************************************************************
**
*F  EqRat( <opL>, <opR> ) . . . . . . . . . . . . . . test if <ratL> = <ratR>
**
**  'EqRat' returns 'true' if the two rationals <ratL> and <ratR>  are  equal
**  and 'false' otherwise.
*/
libGAP_Int             libGAP_EqRat (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 numL, denL;     /* numerator and denominator left  */
    libGAP_Obj                 numR, denR;     /* numerator and denominator right */

    libGAP_CHECK_RAT(opL);
    libGAP_CHECK_RAT(opR);
    /* get numerator and denominator of the operands                       */
    if ( libGAP_TNUM_OBJ(opL) == libGAP_T_RAT ) {
        numL = libGAP_NUM_RAT(opL);
        denL = libGAP_DEN_RAT(opL);
    }
    else {
        numL = opL;
        denL = libGAP_INTOBJ_INT( 1L );
    }
    if ( libGAP_TNUM_OBJ(opR) == libGAP_T_RAT ) {
        numR = libGAP_NUM_RAT(opR);
        denR = libGAP_DEN_RAT(opR);
    }
    else {
        numR = opR;
        denR = libGAP_INTOBJ_INT( 1L );
    }

    /* compare the numerators                                              */
    if ( ! libGAP_EQ( numL, numR ) ) {
        return 0L;
    }

    /* compare the denominators                                            */
    if ( ! libGAP_EQ( denL, denR ) ) {
        return 0L;
    }

    /* no differences found, they must be equal                            */
    return 1L;
}


/****************************************************************************
**
*F  LtRat( <opL>, <opR> ) . . . . . . . . . . . . . . test if <ratL> < <ratR>
**
**  'LtRat' returns 'true'  if  the  rational  <ratL>  is  smaller  than  the
**  rational <ratR> and 'false' otherwise.  Either operand may be an integer.
*/
libGAP_Int             libGAP_LtRat (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 numL, denL;     /* numerator and denominator left  */
    libGAP_Obj                 numR, denR;     /* numerator and denominator right */

    libGAP_CHECK_RAT(opL);
    libGAP_CHECK_RAT(opR);
    /* get numerator and denominator of the operands                       */
    if ( libGAP_TNUM_OBJ(opL) == libGAP_T_RAT ) {
        numL = libGAP_NUM_RAT(opL);
        denL = libGAP_DEN_RAT(opL);
    }
    else {
        numL = opL;
        denL = libGAP_INTOBJ_INT( 1L );
    }
    if ( libGAP_TNUM_OBJ(opR) == libGAP_T_RAT ) {
        numR = libGAP_NUM_RAT(opR);
        denR = libGAP_DEN_RAT(opR);
    }
    else {
        numR = opR;
        denR = libGAP_INTOBJ_INT( 1L );
    }

    /* a / b < c / d <=> a d < c b                                         */
    return libGAP_LtInt( libGAP_ProdInt( numL, denR ), libGAP_ProdInt( numR, denL ) );
}


/****************************************************************************
**
*F  SumRat( <opL>, <opR> )  . . . . . . . . . . . . . .  sum of two rationals
**
**  'SumRat'  returns the   sum of two  rationals  <opL>  and <opR>.   Either
**  operand may also be an integer.  The sum is reduced.
*/
libGAP_Obj             libGAP_SumRat (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 numL, denL;     /* numerator and denominator left  */
    libGAP_Obj                 numR, denR;     /* numerator and denominator right */
    libGAP_Obj                 gcd1, gcd2;     /* gcd of denominators             */
    libGAP_Obj                 numS, denS;     /* numerator and denominator sum   */
    libGAP_Obj                 sum;            /* sum                             */

    libGAP_CHECK_RAT(opL);
    libGAP_CHECK_RAT(opR);
    /* get numerator and denominator of the operands                       */
    if ( libGAP_TNUM_OBJ(opL) == libGAP_T_RAT ) {
        numL = libGAP_NUM_RAT(opL);
        denL = libGAP_DEN_RAT(opL);
    }
    else {
        numL = opL;
        denL = libGAP_INTOBJ_INT( 1L );
    }
    if ( libGAP_TNUM_OBJ(opR) == libGAP_T_RAT ) {
        numR = libGAP_NUM_RAT(opR);
        denR = libGAP_DEN_RAT(opR);
    }
    else {
        numR = opR;
        denR = libGAP_INTOBJ_INT( 1L );
    }

    /* find the gcd of the denominators                                    */
    gcd1 = libGAP_GcdInt( denL, denR );

    /* nothing can cancel if the gcd is 1                                  */
    if ( gcd1 == libGAP_INTOBJ_INT( 1L ) ) {
        numS = libGAP_SumInt( libGAP_ProdInt( numL, denR ), libGAP_ProdInt( numR, denL ) );
        denS = libGAP_ProdInt( denL, denR );
    }

    /* a little bit more difficult otherwise                               */
    else {
        numS = libGAP_SumInt( libGAP_ProdInt( numL, libGAP_QuoInt( denR, gcd1 ) ),
                       libGAP_ProdInt( numR, libGAP_QuoInt( denL, gcd1 ) ) );
        gcd2 = libGAP_GcdInt( numS, gcd1 );
        numS = libGAP_QuoInt( numS, gcd2 );
        denS = libGAP_ProdInt( libGAP_QuoInt( denL, gcd1 ), libGAP_QuoInt( denR, gcd2 ) );
    }

    /* make the fraction or, if possible, the integer                      */
    if ( denS != libGAP_INTOBJ_INT( 1L ) ) {
        sum  = libGAP_NewBag( libGAP_T_RAT, 2 * sizeof(libGAP_Obj) );
        libGAP_NUM_RAT(sum) = numS;
        libGAP_DEN_RAT(sum) = denS;
        /* 'CHANGED_BAG' not needed, 'sum' is the youngest bag             */
    }
    else {
        sum = numS;
    }

    /* return the result                                                   */
    libGAP_CHECK_RAT(sum);
    return sum;
}


/****************************************************************************
**
*F  ZeroRat(<op>) . . . . . . . . . . . . . . . . . . . .  zero of a rational
*/
libGAP_Obj             libGAP_ZeroRat (
    libGAP_Obj                 op )
{
    return libGAP_INTOBJ_INT( 0L );
}


/****************************************************************************
**
*F  AInvRat(<op>) . . . . . . . . . . . . . .  additive inverse of a rational
*/
libGAP_Obj             libGAP_AInvRat (
    libGAP_Obj                 op )
{
    libGAP_Obj                 res;
    libGAP_Obj                 tmp;
    libGAP_CHECK_RAT(op);
    res = libGAP_NewBag( libGAP_T_RAT, 2 * sizeof(libGAP_Obj) );
    tmp = libGAP_AINV( libGAP_NUM_RAT(op) );
    libGAP_NUM_RAT(res) = tmp;
    libGAP_DEN_RAT(res) = libGAP_DEN_RAT(op);
    libGAP_CHANGED_BAG(res);
    libGAP_CHECK_RAT(res);
    return res;
}


/****************************************************************************
**
*F  DiffRat( <opL>, <opR> ) . . . . . . . . . . . difference of two rationals
**
**  'DiffRat' returns the  difference  of  two  rationals  <opL>  and  <opR>.
**  Either operand may also be an integer.  The difference is reduced.
*/
libGAP_Obj             libGAP_DiffRat (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 numL, denL;     /* numerator and denominator left  */
    libGAP_Obj                 numR, denR;     /* numerator and denominator right */
    libGAP_Obj                 gcd1, gcd2;     /* gcd of denominators             */
    libGAP_Obj                 numD, denD;     /* numerator and denominator diff  */
    libGAP_Obj                 dif;            /* diff                            */

    libGAP_CHECK_RAT(opL);
    libGAP_CHECK_RAT(opR);
    /* get numerator and denominator of the operands                       */
    if ( libGAP_TNUM_OBJ(opL) == libGAP_T_RAT ) {
        numL = libGAP_NUM_RAT(opL);
        denL = libGAP_DEN_RAT(opL);
    }
    else {
        numL = opL;
        denL = libGAP_INTOBJ_INT( 1L );
    }
    if ( libGAP_TNUM_OBJ(opR) == libGAP_T_RAT ) {
        numR = libGAP_NUM_RAT(opR);
        denR = libGAP_DEN_RAT(opR);
    }
    else {
        numR = opR;
        denR = libGAP_INTOBJ_INT( 1L );
    }

    /* find the gcd of the denominators                                    */
    gcd1 = libGAP_GcdInt( denL, denR );

    /* nothing can cancel if the gcd is 1                                  */
    if ( gcd1 == libGAP_INTOBJ_INT( 1L ) ) {
        numD = libGAP_DiffInt( libGAP_ProdInt( numL, denR ), libGAP_ProdInt( numR, denL ) );
        denD = libGAP_ProdInt( denL, denR );
    }

    /* a little bit more difficult otherwise                               */
    else {
        numD = libGAP_DiffInt( libGAP_ProdInt( numL, libGAP_QuoInt( denR, gcd1 ) ),
                        libGAP_ProdInt( numR, libGAP_QuoInt( denL, gcd1 ) ) );
        gcd2 = libGAP_GcdInt( numD, gcd1 );
        numD = libGAP_QuoInt( numD, gcd2 );
        denD = libGAP_ProdInt( libGAP_QuoInt( denL, gcd1 ), libGAP_QuoInt( denR, gcd2 ) );
    }

    /* make the fraction or, if possible, the integer                      */
    if ( denD != libGAP_INTOBJ_INT( 1L ) ) {
        dif  = libGAP_NewBag( libGAP_T_RAT, 2 * sizeof(libGAP_Obj) );
        libGAP_NUM_RAT(dif) = numD;
        libGAP_DEN_RAT(dif) = denD;
        /* 'CHANGED_BAG' not needed, 'dif' is the youngest bag             */
    }
    else {
        dif = numD;
    }

    /* return the result                                                   */
    libGAP_CHECK_RAT(dif);
    return dif;
}


/****************************************************************************
**
*F  ProdRat( <opL>, <opR> ) . . . . . . . . . . . .  product of two rationals
**
**  'ProdRat' returns the  product of two rationals <opL> and  <opR>.  Either
**  operand may also be an integer.  The product is reduced.
*/
libGAP_Obj             libGAP_ProdRat (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 numL, denL;     /* numerator and denominator left  */
    libGAP_Obj                 numR, denR;     /* numerator and denominator right */
    libGAP_Obj                 gcd1, gcd2;     /* gcd of denominators             */
    libGAP_Obj                 numP, denP;     /* numerator and denominator prod  */
    libGAP_Obj                 prd;            /* prod                            */

    libGAP_CHECK_RAT(opL);
    libGAP_CHECK_RAT(opR);
    /* get numerator and denominator of the operands                       */
    if ( libGAP_TNUM_OBJ(opL) == libGAP_T_RAT ) {
        numL = libGAP_NUM_RAT(opL);
        denL = libGAP_DEN_RAT(opL);
    }
    else {
        numL = opL;
        denL = libGAP_INTOBJ_INT( 1L );
    }
    if ( libGAP_TNUM_OBJ(opR) == libGAP_T_RAT ) {
        numR = libGAP_NUM_RAT(opR);
        denR = libGAP_DEN_RAT(opR);
    }
    else {
        numR = opR;
        denR = libGAP_INTOBJ_INT( 1L );
    }

    /* find the gcds                                                       */
    gcd1 = libGAP_GcdInt( numL, denR );
    gcd2 = libGAP_GcdInt( numR, denL );

    /* nothing can cancel if the gcds are 1                                */
    if ( gcd1 == libGAP_INTOBJ_INT( 1L ) && gcd2 == libGAP_INTOBJ_INT( 1L ) ) {
        numP = libGAP_ProdInt( numL, numR );
        denP = libGAP_ProdInt( denL, denR );
    }

    /* a little bit more difficult otherwise                               */
    else {
        numP = libGAP_ProdInt( libGAP_QuoInt( numL, gcd1 ), libGAP_QuoInt( numR, gcd2 ) );
        denP = libGAP_ProdInt( libGAP_QuoInt( denL, gcd2 ), libGAP_QuoInt( denR, gcd1 ) );
    }

    /* make the fraction or, if possible, the integer                      */
    if ( denP != libGAP_INTOBJ_INT( 1L ) ) {
        prd = libGAP_NewBag( libGAP_T_RAT, 2 * sizeof(libGAP_Obj) );
        libGAP_NUM_RAT(prd) = numP;
        libGAP_DEN_RAT(prd) = denP;
        /* 'CHANGED_BAG' not needed, 'prd' is the youngest bag             */
    }
    else {
        prd = numP;
    }

    /* return the result                                                   */
    libGAP_CHECK_RAT(prd);
    return prd;
}


/****************************************************************************
**
*F  OneRat(<op>)  . . . . . . . . . . . . . . . . . . . . . one of a rational
*/
libGAP_Obj             libGAP_OneRat (
    libGAP_Obj                 op )
{
    return libGAP_INTOBJ_INT( 1L );
}


/****************************************************************************
**
*F  InvRat(<op>)  . . . . . . . . . . . . . . . . . . . inverse of a rational
*/
extern  libGAP_Obj             libGAP_QuoRat (
            libGAP_Obj                 opL,
            libGAP_Obj                 opR );

libGAP_Obj             libGAP_InvRat (
    libGAP_Obj                 op )
{
  libGAP_Obj res;
    libGAP_CHECK_RAT(op);
    if (op == libGAP_INTOBJ_INT(0))
      return libGAP_Fail;
    res = libGAP_QuoRat( libGAP_INTOBJ_INT( 1L ), op );
    libGAP_CHECK_RAT(res);
    return res;
}


/****************************************************************************
**
*F  QuoRat( <opL>, <opR> )  . . . . . . . . . . . . quotient of two rationals
**
**  'QuoRat'  returns the quotient of two rationals <opL> and  <opR>.  Either
**  operand may also be an integer.  The quotient is reduced.
*/
libGAP_Obj             libGAP_QuoRat (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 numL, denL;     /* numerator and denominator left  */
    libGAP_Obj                 numR, denR;     /* numerator and denominator right */
    libGAP_Obj                 gcd1, gcd2;     /* gcd of denominators             */
    libGAP_Obj                 numQ, denQ;     /* numerator and denominator Qrod  */
    libGAP_Obj                 quo;            /* Qrod                            */

    libGAP_CHECK_RAT(opL);
    libGAP_CHECK_RAT(opR);
    /* get numerator and denominator of the operands                       */
    if ( libGAP_TNUM_OBJ(opL) == libGAP_T_RAT ) {
        numL = libGAP_NUM_RAT(opL);
        denL = libGAP_DEN_RAT(opL);
    }
    else {
        numL = opL;
        denL = libGAP_INTOBJ_INT( 1L );
    }
    if ( libGAP_TNUM_OBJ(opR) == libGAP_T_RAT ) {
        numR = libGAP_NUM_RAT(opR);
        denR = libGAP_DEN_RAT(opR);
    }
    else {
        numR = opR;
        denR = libGAP_INTOBJ_INT( 1L );
    }

    /* division by zero is an error                                        */
    if ( numR == libGAP_INTOBJ_INT( 0L ) ) {
        opR = libGAP_ErrorReturnObj(
            "Rational operations: <divisor> must not be zero",
            0L, 0L,
            "you can replace <divisor> via 'return <divisor>;'" );
        return libGAP_QUO( opL, opR );
    }

    /* we multiply the left numerator with the right denominator           */
    /* so the right denominator should carry the sign of the right operand */
    if ( (libGAP_TNUM_OBJ(numR) == libGAP_T_INT && libGAP_INT_INTOBJ(numR) < 0)
      || libGAP_TNUM_OBJ(numR) == libGAP_T_INTNEG ) {
        numR = libGAP_ProdInt( libGAP_INTOBJ_INT( -1L ), numR );
        denR = libGAP_ProdInt( libGAP_INTOBJ_INT( -1L ), denR );
    }

    /* find the gcds                                                       */
    gcd1 = libGAP_GcdInt( numL, numR );
    gcd2 = libGAP_GcdInt( denR, denL );

    /* nothing can cancel if the gcds are 1                                */
    if ( gcd1 == libGAP_INTOBJ_INT( 1L ) && gcd2 == libGAP_INTOBJ_INT( 1L ) ) {
        numQ = libGAP_ProdInt( numL, denR );
        denQ = libGAP_ProdInt( denL, numR );
    }

    /* a little bit more difficult otherwise                               */
    else {
        numQ = libGAP_ProdInt( libGAP_QuoInt( numL, gcd1 ), libGAP_QuoInt( denR, gcd2 ) );
        denQ = libGAP_ProdInt( libGAP_QuoInt( denL, gcd2 ), libGAP_QuoInt( numR, gcd1 ) );
    }

    /* make the fraction or, if possible, the integer                      */
    if ( denQ != libGAP_INTOBJ_INT( 1L ) ) {
        quo = libGAP_NewBag( libGAP_T_RAT, 2 * sizeof(libGAP_Obj) );
        libGAP_NUM_RAT(quo) = numQ;
        libGAP_DEN_RAT(quo) = denQ;
        /* 'CHANGED_BAG' not needed, 'quo' is the youngest bag             */
    }
    else {
        quo = numQ;
    }

    /* return the result */
    libGAP_CHECK_RAT(quo);
    return quo;
}


/****************************************************************************
**
*F  ModRat( <opL>, <opR> )  . . . . . . . . remainder of fraction mod integer
**
**  'ModRat' returns the remainder  of the fraction  <opL> modulo the integer
**  <opR>.  The remainder is always an integer.
**
**  '<r>  / <s> mod  <n>' yields  the remainder of   the fraction '<p> / <q>'
**  modulo  the  integer '<n>',  where '<p> / <q>' is  the  reduced  form  of
**  '<r>  / <s>'.
**
**  The modular remainder of $r / s$ mod $n$ is defined to be the integer $k$
**  in $0 .. n-1$ such that $p = k q$ mod $n$, where $p = r / gcd(r, s)$  and
**  $q = s / gcd(r, s)$. In particular, $1  /  s$  mod  $n$  is  the  modular
**  inverse of $s$ modulo $n$, whenever $s$ and $n$ are relatively prime.
**
**  Note that  the  remainder  will  not  exist  if  $s / gcd(r, s)$  is  not
**  relatively prime to $n$. Note that $4 / 6$ mod $32$ does  exist  (and  is
**  $22$), even though $6$ is not invertible modulo  $32$,  because  the  $2$
**  cancels.
**
**  Another possible  definition of $r/s$ mod $n$  would be  a rational $t/s$
**  such that $0 \<= t/s \< n$ and $r/s - t/s$ is a multiple of $n$.  This is
**  rarely needed while computing modular inverses is very useful.
*/
libGAP_Obj             libGAP_ModRat (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 a, aL, b, bL, c, cL, hdQ;

    /* make the integer positive                                           */
    if ( (libGAP_TNUM_OBJ(opR) == libGAP_T_INT && libGAP_INT_INTOBJ(opR) < 0)
      || libGAP_TNUM_OBJ(opR) == libGAP_T_INTNEG ) {
        opR = libGAP_ProdInt( libGAP_INTOBJ_INT( -1L ), opR );
    }

    /* let <p>/<q> represent <r>/<s> in reduced form                       */
    /* invert the denominator <q> modulo <n> with Euclids algorithm        */
    a = opR;           aL = libGAP_INTOBJ_INT( 0L );   /* a = <n>                 */
    b = libGAP_DEN_RAT(opL);  bL = libGAP_INTOBJ_INT( 1L );   /* b = <q>                 */
    while ( a != libGAP_INTOBJ_INT( 1L ) ) {
        while ( b != libGAP_INTOBJ_INT( 0L ) ) {
            hdQ  = libGAP_QuoInt( a, b );
            c  = b;  cL = bL;
            b  = libGAP_DiffInt( a,  libGAP_ProdInt( hdQ, b  ) );
            bL = libGAP_DiffInt( aL, libGAP_ProdInt( hdQ, bL ) );
            a  = c;  aL = cL;
        }

        /* check whether the denominator <q> really was invertible mod <n> */
        if ( a != libGAP_INTOBJ_INT( 1L ) ) {
            opR = libGAP_ErrorReturnObj(
                      "ModRat: for <r>/<s> mod <n>, <s>/gcd(<r>,<s>) and <n> must be coprime",
                      0L, 0L,
                      "you can replace the integer <n> via 'return <n>;'" );
            a = opR;           aL = libGAP_INTOBJ_INT( 0L );   /* a = <n>         */
            b = libGAP_DEN_RAT(opL);  bL = libGAP_INTOBJ_INT( 1L );   /* b = <q>         */
        }
    }

    /* return the remainder                                                */
    return libGAP_ModInt( libGAP_ProdInt( libGAP_NUM_RAT(opL), aL ), opR );
}


/****************************************************************************
**
*F  PowRat( <opL>, <opR> )  . . . . . .  raise a rational to an integer power
**
**  'PowRat' raises the rational <opL> to the  power  given  by  the  integer
**  <opR>.  The power is reduced.
*/
libGAP_Obj             libGAP_PowRat (
    libGAP_Obj                 opL,
    libGAP_Obj                 opR )
{
    libGAP_Obj                 numP, denP;     /* numerator and denominator power */
    libGAP_Obj                 pow;            /* power                           */

    libGAP_CHECK_RAT(opL);
    /* raise numerator and denominator seperately                          */
    numP = libGAP_PowInt( libGAP_NUM_RAT(opL), opR );
    denP = libGAP_PowInt( libGAP_DEN_RAT(opL), opR );

    /* if <opR> == 0 return 1                                              */
    if ( opR == libGAP_INTOBJ_INT( 0L ) ) {
        pow = libGAP_INTOBJ_INT( 1L );
    }

    /* if <opR> == 1 return <opL>                                          */
    else if ( opR == libGAP_INTOBJ_INT( 1L ) ) {
        pow = opL;
    }

    /* if <opR> is positive raise numberator and denominator seperately    */
    else if ( (libGAP_TNUM_OBJ(opR) == libGAP_T_INT && 0 < libGAP_INT_INTOBJ(opR))
           || libGAP_TNUM_OBJ(opR) == libGAP_T_INTPOS ) {
        numP = libGAP_PowInt( libGAP_NUM_RAT(opL), opR );
        denP = libGAP_PowInt( libGAP_DEN_RAT(opL), opR );
        pow = libGAP_NewBag( libGAP_T_RAT, 2 * sizeof(libGAP_Obj) );
        libGAP_NUM_RAT(pow) = numP;
        libGAP_DEN_RAT(pow) = denP;
        /* 'CHANGED_BAG' not needed, 'pow' is the youngest bag             */
    }

    /* if <opR> is negative and numerator is 1 just power the denominator  */
    else if ( libGAP_NUM_RAT(opL) == libGAP_INTOBJ_INT( 1L ) ) {
        pow = libGAP_PowInt( libGAP_DEN_RAT(opL), libGAP_ProdInt( libGAP_INTOBJ_INT(-1L), opR ) );
    }

    /* if <opR> is negative and numerator is -1 return (-1)^r * num(l)     */
    else if ( libGAP_NUM_RAT(opL) == libGAP_INTOBJ_INT( -1L ) ) {
        pow = libGAP_ProdInt(libGAP_PowInt(libGAP_NUM_RAT(opL),libGAP_ProdInt(libGAP_INTOBJ_INT(-1L),opR)),
                      libGAP_PowInt(libGAP_DEN_RAT(opL),libGAP_ProdInt(libGAP_INTOBJ_INT(-1L),opR)));
    }

    /* if <opR> is negative do both powers, take care of the sign          */
    else {
        numP = libGAP_PowInt( libGAP_DEN_RAT(opL), libGAP_ProdInt( libGAP_INTOBJ_INT( -1L ), opR ) );
        denP = libGAP_PowInt( libGAP_NUM_RAT(opL), libGAP_ProdInt( libGAP_INTOBJ_INT( -1L ), opR ) );
        pow  = libGAP_NewBag( libGAP_T_RAT, 2 * sizeof(libGAP_Obj) );
        if ( (libGAP_TNUM_OBJ(denP) == libGAP_T_INT && 0 < libGAP_INT_INTOBJ(denP))
          || libGAP_TNUM_OBJ(denP) == libGAP_T_INTPOS ) {
            libGAP_NUM_RAT(pow) = numP;
            libGAP_DEN_RAT(pow) = denP;
        }
        else {
            libGAP_NUM_RAT(pow) = libGAP_ProdInt( libGAP_INTOBJ_INT( -1L ), numP );
            libGAP_DEN_RAT(pow) = libGAP_ProdInt( libGAP_INTOBJ_INT( -1L ), denP );
        }
        /* 'CHANGED_BAG' not needed, 'pow' is the youngest bag             */
    }

    /* return the result                                                   */
    libGAP_CHECK_RAT(pow);
    return pow;
}


/****************************************************************************
**
*F  IsRatHandler(<self>,<val>)  . . . . . . . . . . . . is a value a rational
**
**  'IsRatHandler' implements the internal function 'IsRat'.
**
**  'IsRat( <val> )'
**
**  'IsRat' returns  'true' if  the  value <val> is  a  rational and  'false'
**  otherwise.
*/
libGAP_Obj             libGAP_IsRatFilt;

libGAP_Obj             libGAP_IsRatHandler (
    libGAP_Obj                 self,
    libGAP_Obj                 val )
{
    /* return 'true' if <val> is a rational and 'false' otherwise          */
    if ( libGAP_TNUM_OBJ(val) == libGAP_T_RAT    || libGAP_TNUM_OBJ(val) == libGAP_T_INT
      || libGAP_TNUM_OBJ(val) == libGAP_T_INTPOS || libGAP_TNUM_OBJ(val) == libGAP_T_INTNEG ) {
        return libGAP_True;
    }
    else if ( libGAP_TNUM_OBJ(val) < libGAP_FIRST_EXTERNAL_TNUM ) {
        return libGAP_False;
    }
    else {
        return libGAP_DoFilter( self, val );
    }
}


/****************************************************************************
**
*F  FuncNumeratorRat(<self>,<rat>)  . . . . . . . . . numerator of a rational
**
**  'FuncNumeratorRat' implements the internal function 'NumeratorRat'.
**
**  'NumeratorRat( <rat> )'
**
**  'NumeratorRat' returns the numerator of the rational <rat>.
*/
libGAP_Obj             libGAP_FuncNumeratorRat (
    libGAP_Obj                 self,
    libGAP_Obj                 rat )
{
    /* check the argument                                                   */
    while ( libGAP_TNUM_OBJ(rat) != libGAP_T_RAT    && libGAP_TNUM_OBJ(rat) != libGAP_T_INT
         && libGAP_TNUM_OBJ(rat) != libGAP_T_INTPOS && libGAP_TNUM_OBJ(rat) != libGAP_T_INTNEG ) {
        rat = libGAP_ErrorReturnObj(
            "Numerator: <rat> must be a rational (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(rat), 0L,
            "you can replace <rat> via 'return <rat>;'" );
    }

    /* return the numerator                                                */
    if ( libGAP_TNUM_OBJ(rat) == libGAP_T_RAT ) {
        return libGAP_NUM_RAT(rat);
    }
    else {
        return rat;
    }
}


/****************************************************************************
**
*F  FuncDenominatorRat(<self>,<rat>)  . . . . . . . denominator of a rational
**
**  'FuncDenominatorRat' implements the internal function 'DenominatorRat'.
**
**  'DenominatorRat( <rat> )'
**
**  'DenominatorRat' returns the denominator of the rational <rat>.
*/
libGAP_Obj             libGAP_FuncDenominatorRat (
    libGAP_Obj                 self,
    libGAP_Obj                 rat )
{
    /* check the argument                                                  */
    while ( libGAP_TNUM_OBJ(rat) != libGAP_T_RAT    && libGAP_TNUM_OBJ(rat) != libGAP_T_INT
         && libGAP_TNUM_OBJ(rat) != libGAP_T_INTPOS && libGAP_TNUM_OBJ(rat) != libGAP_T_INTNEG ) {
        rat = libGAP_ErrorReturnObj(
            "DenominatorRat: <rat> must be a rational (not a %s)",
            (libGAP_Int)libGAP_TNAM_OBJ(rat), 0L,
            "you can replace <rat> via 'return <rat>;'" );
    }

    /* return the denominator                                              */
    if ( libGAP_TNUM_OBJ(rat) == libGAP_T_RAT ) {
        return libGAP_DEN_RAT(rat);
    }
    else {
        return libGAP_INTOBJ_INT( 1L );
    }
}

/****************************************************************************
**
*F  SaveRat( <rat> )
**
*/

void libGAP_SaveRat(libGAP_Obj rat)
{
  libGAP_SaveSubObj(libGAP_NUM_RAT(rat));
  libGAP_SaveSubObj(libGAP_DEN_RAT(rat));
}

/****************************************************************************
**
*F  LoadRat( <rat> )
**
*/

void libGAP_LoadRat(libGAP_Obj rat)
{
  libGAP_NUM_RAT(rat) = libGAP_LoadSubObj();
  libGAP_DEN_RAT(rat) = libGAP_LoadSubObj();
}

/****************************************************************************
**

*F * * * * * * * * * * * * * initialize package * * * * * * * * * * * * * * *
*/

/****************************************************************************
**

*V  GVarFilts . . . . . . . . . . . . . . . . . . . list of filters to export
*/
static libGAP_StructGVarFilt libGAP_GVarFilts [] = {

    { "IS_RAT", "obj", &libGAP_IsRatFilt,
      libGAP_IsRatHandler, "src/rational.c:IS_RAT" },

    { 0 }

};


/****************************************************************************
**
*V  GVarFuncs . . . . . . . . . . . . . . . . . . list of functions to export
*/
static libGAP_StructGVarFunc libGAP_GVarFuncs [] = {

    { "NUMERATOR_RAT", 1, "rat",
      libGAP_FuncNumeratorRat, "src/rational.c:NUMERATOR_RAT" },

    { "DENOMINATOR_RAT", 1, "rat",
      libGAP_FuncDenominatorRat, "src/rational.c:DENOMINATOR_RAT" },

    { 0 }

};


/****************************************************************************
**

*F  InitKernel( <module> )  . . . . . . . . initialise kernel data structures
*/
static libGAP_Int libGAP_InitKernel (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* install the marking function                                        */
    libGAP_InfoBags[         libGAP_T_RAT ].name = "rational";
    libGAP_InitMarkFuncBags( libGAP_T_RAT, libGAP_MarkTwoSubBags );

    /* install the kind function                                           */
    libGAP_ImportGVarFromLibrary( "TYPE_RAT_POS", &libGAP_TYPE_RAT_POS );
    libGAP_ImportGVarFromLibrary( "TYPE_RAT_NEG", &libGAP_TYPE_RAT_NEG );

    libGAP_TypeObjFuncs[ libGAP_T_RAT ] = libGAP_TypeRat;

    /* init filters and functions                                          */
    libGAP_InitHdlrFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitHdlrFuncsFromTable( libGAP_GVarFuncs );

    /* install a saving function */
    libGAP_SaveObjFuncs[ libGAP_T_RAT ] = libGAP_SaveRat;
    libGAP_LoadObjFuncs[ libGAP_T_RAT ] = libGAP_LoadRat;

    /* install the printer                                                 */
    libGAP_PrintObjFuncs[ libGAP_T_RAT ] = libGAP_PrintRat;

    /* install the comparisons                                             */
    libGAP_EqFuncs  [ libGAP_T_RAT    ][ libGAP_T_RAT    ] = libGAP_EqRat;

    libGAP_LtFuncs  [ libGAP_T_RAT    ][ libGAP_T_RAT    ] = libGAP_LtRat;
    libGAP_LtFuncs  [ libGAP_T_INT    ][ libGAP_T_RAT    ] = libGAP_LtRat;
    libGAP_LtFuncs  [ libGAP_T_INTPOS ][ libGAP_T_RAT    ] = libGAP_LtRat;
    libGAP_LtFuncs  [ libGAP_T_INTNEG ][ libGAP_T_RAT    ] = libGAP_LtRat;
    libGAP_LtFuncs  [ libGAP_T_RAT    ][ libGAP_T_INT    ] = libGAP_LtRat;
    libGAP_LtFuncs  [ libGAP_T_RAT    ][ libGAP_T_INTPOS ] = libGAP_LtRat;
    libGAP_LtFuncs  [ libGAP_T_RAT    ][ libGAP_T_INTNEG ] = libGAP_LtRat;

    /* install the arithmetic operations                                   */
    libGAP_ZeroFuncs[ libGAP_T_RAT    ] = libGAP_ZeroRat;
    libGAP_AInvFuncs[ libGAP_T_RAT    ] = libGAP_AInvRat;
    libGAP_AInvMutFuncs[ libGAP_T_RAT    ] = libGAP_AInvRat;
    libGAP_OneFuncs [ libGAP_T_RAT    ] = libGAP_OneRat;
    libGAP_OneMutFuncs [ libGAP_T_RAT    ] = libGAP_OneRat;
    libGAP_InvFuncs [ libGAP_T_INT    ] = libGAP_InvRat;
    libGAP_InvFuncs [ libGAP_T_INTPOS ] = libGAP_InvRat;
    libGAP_InvFuncs [ libGAP_T_INTNEG ] = libGAP_InvRat;
    libGAP_InvFuncs [ libGAP_T_RAT    ] = libGAP_InvRat;
    libGAP_InvMutFuncs [ libGAP_T_INT    ] = libGAP_InvRat;
    libGAP_InvMutFuncs [ libGAP_T_INTPOS ] = libGAP_InvRat;
    libGAP_InvMutFuncs [ libGAP_T_INTNEG ] = libGAP_InvRat;
    libGAP_InvMutFuncs [ libGAP_T_RAT    ] = libGAP_InvRat;
    
    libGAP_SumFuncs [ libGAP_T_RAT    ][ libGAP_T_RAT    ] = libGAP_SumRat;
    libGAP_SumFuncs [ libGAP_T_INT    ][ libGAP_T_RAT    ] = libGAP_SumRat;
    libGAP_SumFuncs [ libGAP_T_INTPOS ][ libGAP_T_RAT    ] = libGAP_SumRat;
    libGAP_SumFuncs [ libGAP_T_INTNEG ][ libGAP_T_RAT    ] = libGAP_SumRat;
    libGAP_SumFuncs [ libGAP_T_RAT    ][ libGAP_T_INT    ] = libGAP_SumRat;
    libGAP_SumFuncs [ libGAP_T_RAT    ][ libGAP_T_INTPOS ] = libGAP_SumRat;
    libGAP_SumFuncs [ libGAP_T_RAT    ][ libGAP_T_INTNEG ] = libGAP_SumRat;

    libGAP_DiffFuncs[ libGAP_T_RAT    ][ libGAP_T_RAT    ] = libGAP_DiffRat;
    libGAP_DiffFuncs[ libGAP_T_INT    ][ libGAP_T_RAT    ] = libGAP_DiffRat;
    libGAP_DiffFuncs[ libGAP_T_INTPOS ][ libGAP_T_RAT    ] = libGAP_DiffRat;
    libGAP_DiffFuncs[ libGAP_T_INTNEG ][ libGAP_T_RAT    ] = libGAP_DiffRat;
    libGAP_DiffFuncs[ libGAP_T_RAT    ][ libGAP_T_INT    ] = libGAP_DiffRat;
    libGAP_DiffFuncs[ libGAP_T_RAT    ][ libGAP_T_INTPOS ] = libGAP_DiffRat;
    libGAP_DiffFuncs[ libGAP_T_RAT    ][ libGAP_T_INTNEG ] = libGAP_DiffRat;

    libGAP_ProdFuncs[ libGAP_T_RAT    ][ libGAP_T_RAT    ] = libGAP_ProdRat;
    libGAP_ProdFuncs[ libGAP_T_INT    ][ libGAP_T_RAT    ] = libGAP_ProdRat;
    libGAP_ProdFuncs[ libGAP_T_INTPOS ][ libGAP_T_RAT    ] = libGAP_ProdRat;
    libGAP_ProdFuncs[ libGAP_T_INTNEG ][ libGAP_T_RAT    ] = libGAP_ProdRat;
    libGAP_ProdFuncs[ libGAP_T_RAT    ][ libGAP_T_INT    ] = libGAP_ProdRat;
    libGAP_ProdFuncs[ libGAP_T_RAT    ][ libGAP_T_INTPOS ] = libGAP_ProdRat;
    libGAP_ProdFuncs[ libGAP_T_RAT    ][ libGAP_T_INTNEG ] = libGAP_ProdRat;

    libGAP_QuoFuncs [ libGAP_T_INT    ][ libGAP_T_INT    ] = libGAP_QuoRat;
    libGAP_QuoFuncs [ libGAP_T_INT    ][ libGAP_T_INTPOS ] = libGAP_QuoRat;
    libGAP_QuoFuncs [ libGAP_T_INT    ][ libGAP_T_INTNEG ] = libGAP_QuoRat;
    libGAP_QuoFuncs [ libGAP_T_INTPOS ][ libGAP_T_INT    ] = libGAP_QuoRat;
    libGAP_QuoFuncs [ libGAP_T_INTPOS ][ libGAP_T_INTPOS ] = libGAP_QuoRat;
    libGAP_QuoFuncs [ libGAP_T_INTPOS ][ libGAP_T_INTNEG ] = libGAP_QuoRat;
    libGAP_QuoFuncs [ libGAP_T_INTNEG ][ libGAP_T_INT    ] = libGAP_QuoRat;
    libGAP_QuoFuncs [ libGAP_T_INTNEG ][ libGAP_T_INTPOS ] = libGAP_QuoRat;
    libGAP_QuoFuncs [ libGAP_T_INTNEG ][ libGAP_T_INTNEG ] = libGAP_QuoRat;

    libGAP_QuoFuncs [ libGAP_T_RAT    ][ libGAP_T_RAT    ] = libGAP_QuoRat;
    libGAP_QuoFuncs [ libGAP_T_INT    ][ libGAP_T_RAT    ] = libGAP_QuoRat;
    libGAP_QuoFuncs [ libGAP_T_INTPOS ][ libGAP_T_RAT    ] = libGAP_QuoRat;
    libGAP_QuoFuncs [ libGAP_T_INTNEG ][ libGAP_T_RAT    ] = libGAP_QuoRat;
    libGAP_QuoFuncs [ libGAP_T_RAT    ][ libGAP_T_INT    ] = libGAP_QuoRat;
    libGAP_QuoFuncs [ libGAP_T_RAT    ][ libGAP_T_INTPOS ] = libGAP_QuoRat;
    libGAP_QuoFuncs [ libGAP_T_RAT    ][ libGAP_T_INTNEG ] = libGAP_QuoRat;

    libGAP_ModFuncs [ libGAP_T_RAT    ][ libGAP_T_INT    ] = libGAP_ModRat;
    libGAP_ModFuncs [ libGAP_T_RAT    ][ libGAP_T_INTPOS ] = libGAP_ModRat;
    libGAP_ModFuncs [ libGAP_T_RAT    ][ libGAP_T_INTNEG ] = libGAP_ModRat;

    libGAP_PowFuncs [ libGAP_T_RAT    ][ libGAP_T_INT    ] = libGAP_PowRat;
    libGAP_PowFuncs [ libGAP_T_RAT    ][ libGAP_T_INTPOS ] = libGAP_PowRat;
    libGAP_PowFuncs [ libGAP_T_RAT    ][ libGAP_T_INTNEG ] = libGAP_PowRat;

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitLibrary( <module> ) . . . . . . .  initialise library data structures
*/
static libGAP_Int libGAP_InitLibrary (
    libGAP_StructInitInfo *    libGAP_module )
{
    /* init filters and functions                                          */
    libGAP_InitGVarFiltsFromTable( libGAP_GVarFilts );
    libGAP_InitGVarFuncsFromTable( libGAP_GVarFuncs );

    /* return success                                                      */
    return 0;
}


/****************************************************************************
**
*F  InitInfoRat() . . . . . . . . . . . . . . . . . . table of init functions
*/
static libGAP_StructInitInfo libGAP_module = {
    libGAP_MODULE_BUILTIN,                     /* type                           */
    "rational",                         /* name                           */
    0,                                  /* revision entry of c file       */
    0,                                  /* revision entry of h file       */
    0,                                  /* version                        */
    0,                                  /* crc                            */
    libGAP_InitKernel,                         /* initKernel                     */
    libGAP_InitLibrary,                        /* initLibrary                    */
    0,                                  /* checkInit                      */
    0,                                  /* preSave                        */
    0,                                  /* postSave                       */
    0                                   /* postRestore                    */
};

libGAP_StructInitInfo * libGAP_InitInfoRat ( void )
{
    libGAP_FillInVersion( &libGAP_module );
    return &libGAP_module;
}


/****************************************************************************
**

*E  rational.c  . . . . . . . . . . . . . . . . . . . . . . . . . . ends here
*/
