/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import {
  HttpStatusCode,
  TalerError,
  TokenInfo,
  assertUnreachable,
} from "@gnu-taler/taler-util";
import {
  LocalNotificationBannerBulma,
  useLocalNotificationBetter,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { Fragment, VNode, h } from "preact";
import { useState } from "preact/hooks";
import { ErrorLoadingMerchant } from "../../../../components/ErrorLoadingMerchant.js";
import { Loading } from "../../../../components/exception/loading.js";
import { ConfirmModal } from "../../../../components/modal/index.js";
import { useSessionContext } from "../../../../context/session.js";
import { useInstanceAccessTokens } from "../../../../hooks/access-tokens.js";
import { LoginPage } from "../../../login/index.js";
import { NotFoundPageOrAdminCreate } from "../../../notfound/index.js";
import { CardTable } from "./Table.js";

const TALER_SCREEN_ID = 31;

interface Props {
  onCreate: () => void;
}

export default function AccessTokenListPage({ onCreate }: Props): VNode {
  const result = useInstanceAccessTokens();
  const { state: session, lib } = useSessionContext();
  const [deleting, setDeleting] = useState<TokenInfo | null>(null);
  const { i18n } = useTranslationContext();

  if (!result) return <Loading />;
  if (result instanceof TalerError) {
    return <ErrorLoadingMerchant error={result} />;
  }
  if (result.type === "fail") {
    switch (result.case) {
      case HttpStatusCode.NotFound: {
        return <NotFoundPageOrAdminCreate />;
      }
      case HttpStatusCode.Unauthorized: {
        return <LoginPage />;
      }
      case HttpStatusCode.Forbidden: {
        return <LoginPage />;
      }
      default: {
        assertUnreachable(result);
      }
    }
  }

  const [notification, safeFunctionHandler] = useLocalNotificationBetter();
  const deleteToken = safeFunctionHandler(
    lib.instance.deleteAccessToken.bind(lib.instance),
    !session.token || !deleting ? undefined : [session.token, deleting.serial],
  );

  deleteToken.onSuccess = () => {
    setDeleting(null);
  };
  deleteToken.onFail = (fail) => {
    switch (fail.case) {
      case HttpStatusCode.Unauthorized:
        return i18n.str`Unauthorized.`;
      case HttpStatusCode.Forbidden:
        return i18n.str`Forbidden.`;
      case HttpStatusCode.NotFound:
        return i18n.str`Not found.`;
    }
  };

  return (
    <Fragment>
      <section class="section is-main-section">
        <LocalNotificationBannerBulma notification={notification} />

        <CardTable
          tokens={result.body.map((o) => ({
            ...o,
            id: String(o.serial),
          }))}
          onLoadMoreBefore={result.loadFirst}
          onLoadMoreAfter={result.loadNext}
          onCreate={onCreate}
          onDelete={(d) => {
            setDeleting(d);
          }}
        />
        {deleting && (
          <ConfirmModal
            label={`Delete access token`}
            description={deleting.description}
            danger
            active
            onCancel={() => setDeleting(null)}
            confirm={deleteToken}
          >
            <p class="warning">
              <i18n.Translate>
                Deleting an access token cannot be undone.
              </i18n.Translate>
            </p>
          </ConfirmModal>
        )}
      </section>
    </Fragment>
  );
}
