/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Imports.
 */
import {
  GlobalTestState,
  setupDb,
  ExchangeService,
  WalletService,
  WalletClient,
  BankService,
} from "../harness/harness.js";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { CoinConfig, defaultCoinConfig } from "../harness/denomStructures.js";
import {
  AmountString,
  NotificationType,
  TalerCorebankApiClient,
  TransactionMajorState,
  URL,
} from "@gnu-taler/taler-util";

/**
 * Withdraw a high amount.  Mostly intended as a perf test.
 *
 * It is useful to see whether the wallet stays responsive while doing a huge withdrawal.
 * (This is not automatic yet. Use taler-wallet-cli to connect to the daemon and make requests to check.)
 */
export async function runWithdrawalHugeTest(t: GlobalTestState) {
  // Set up test environment

  const db = await setupDb(t);

  const bank = await BankService.create(t, {
    currency: "TESTKUDOS",
    httpPort: 8082,
    allowRegistrations: true,
    // Not used by fakebank
    database: db.connStr,
  });

  const exchange = ExchangeService.create(t, {
    name: "testexchange-1",
    currency: "TESTKUDOS",
    httpPort: 8081,
    database: db.connStr,
  });

  let paytoUri = "payto://x-taler-bank/localhost/exchange";

  await exchange.addBankAccount("1", {
    accountName: "exchange",
    accountPassword: "x",
    wireGatewayApiBaseUrl: new URL("accounts/exchange/taler-wire-gateway/", bank.baseUrl).href,
    accountPaytoUri: paytoUri,
  });

  bank.setSuggestedExchange(exchange, paytoUri);

  await bank.start();

  await bank.pingUntilAvailable();

  const bankClient = new TalerCorebankApiClient(bank.corebankApiBaseUrl, {
    auth: {
      username: "admin",
      password: "adminpw",
    }
  });

  await bankClient.registerAccountExtended({
    name: "Exchange",
    password: "x",
    username: "exchange",
    is_taler_exchange: true,
    payto_uri: paytoUri,
  });

  const coinConfig: CoinConfig[] = defaultCoinConfig.map((x) => x("TESTKUDOS"));
  exchange.addCoinConfigList(coinConfig);

  await exchange.start();
  await exchange.pingUntilAvailable();

  console.log("setup done!");

  const walletService = new WalletService(t, { name: "w1" });
  await walletService.start();
  await walletService.pingUntilAvailable();

  const wallet = new WalletClient({
    name: "wallet",
    unixPath: walletService.socketPath,
  });
  await wallet.connect();

  const withdrawalFinishedCond = wallet.waitForNotificationCond(
    (wn) =>
      wn.type === NotificationType.TransactionStateTransition &&
      wn.transactionId.startsWith("txn:withdrawal:") &&
      wn.newTxState.major === TransactionMajorState.Done,
  );

  await wallet.client.call(WalletApiOperation.AddExchange, {
    exchangeBaseUrl: exchange.baseUrl,
  });

  // Results in about 1K coins withdrawn
  await wallet.client.call(WalletApiOperation.WithdrawTestBalance, {
    exchangeBaseUrl: exchange.baseUrl,
    amount: "TESTKUDOS:10000" as AmountString,
    corebankApiBaseUrl: bank.baseUrl,
  });

  await withdrawalFinishedCond;

  // Check balance

  const balResp = await wallet.client.call(WalletApiOperation.GetBalances, {});
  console.log(balResp);
}

runWithdrawalHugeTest.suites = ["wallet-perf"];
// FIXME: Should not be "experimental" but "slow" or something similar.
runWithdrawalHugeTest.experimental = true;
