/*
 This file is part of GNU Taler
 (C) 2020 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 * Integration test for the wallet testing functionality used by the exchange
 * test cases.
 */

/**
 * Imports.
 */
import { AmountString, Amounts, CoinStatus } from "@gnu-taler/taler-util";
import { WalletApiOperation } from "@gnu-taler/taler-wallet-core";
import { CoinConfig, defaultCoinConfig } from "../harness/denomStructures.js";
import { GlobalTestState, setupDb } from "../harness/harness.js";
import {
  SimpleTestEnvironmentNg3,
  createSimpleTestkudosEnvironmentV3,
  createWalletDaemonWithClient,
} from "../harness/helpers.js";

const merchantAuthToken = "secret-token:sandbox";

/**
 * Run a test case with a simple TESTKUDOS Taler environment, consisting
 * of one exchange, one bank and one merchant.
 */
export async function createMyEnvironment(
  t: GlobalTestState,
  coinConfig: CoinConfig[] = defaultCoinConfig.map((x) => x("TESTKUDOS")),
): Promise<SimpleTestEnvironmentNg3> {
  const db = await setupDb(t);

  const { bankClient, exchange, merchant, exchangeBankAccount } =
    await createSimpleTestkudosEnvironmentV3(t, coinConfig, {});

  console.log("setup done!");

  const { walletClient, walletService } = await createWalletDaemonWithClient(
    t,
    {
      name: "w1",
    },
  );

  return {
    commonDb: db,
    exchange,
    merchant,
    walletClient,
    walletService,
    bankClient,
    exchangeBankAccount,
  };
}

/**
 * Run test for basic, bank-integrated withdrawal.
 */
export async function runWallettestingTest(t: GlobalTestState) {
  const { walletClient, bankClient, exchange, merchant } =
    await createMyEnvironment(t);

  await walletClient.call(WalletApiOperation.RunIntegrationTest, {
    amountToSpend: "TESTKUDOS:5" as AmountString,
    amountToWithdraw: "TESTKUDOS:10" as AmountString,
    corebankApiBaseUrl: bankClient.baseUrl,
    exchangeBaseUrl: exchange.baseUrl,
    merchantAuthToken: merchantAuthToken,
    merchantBaseUrl: merchant.makeInstanceBaseUrl(),
  });

  let txns = await walletClient.call(WalletApiOperation.GetTransactions, {});
  console.log(JSON.stringify(txns, undefined, 2));
  let txTypes = txns.transactions.map((x) => x.type);

  t.assertDeepEqual(txTypes, [
    "withdrawal",
    "payment",
    "withdrawal",
    "payment",
    "refund",
    "payment",
  ]);

  await walletClient.call(WalletApiOperation.ClearDb, {});

  await walletClient.call(WalletApiOperation.WithdrawTestBalance, {
    amount: "TESTKUDOS:10" as AmountString,
    corebankApiBaseUrl: bankClient.baseUrl,
    exchangeBaseUrl: exchange.baseUrl,
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  await walletClient.call(WalletApiOperation.TestPay, {
    amount: "TESTKUDOS:5" as AmountString,
    merchantAuthToken: merchantAuthToken,
    merchantBaseUrl: merchant.makeInstanceBaseUrl(),
    summary: "foo",
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  txns = await walletClient.call(WalletApiOperation.GetTransactions, {});
  console.log(JSON.stringify(txns, undefined, 2));
  txTypes = txns.transactions.map((x) => x.type);

  t.assertDeepEqual(txTypes, ["withdrawal", "payment"]);

  await walletClient.call(WalletApiOperation.ClearDb, {});

  await walletClient.call(WalletApiOperation.WithdrawTestBalance, {
    amount: "TESTKUDOS:10" as AmountString,
    corebankApiBaseUrl: bankClient.baseUrl,
    exchangeBaseUrl: exchange.baseUrl,
  });

  await walletClient.call(WalletApiOperation.TestingWaitTransactionsFinal, {});

  const coinDump = await walletClient.call(WalletApiOperation.DumpCoins, {});

  console.log("coin dump:", JSON.stringify(coinDump, undefined, 2));

  let susp: string | undefined;
  {
    for (const c of coinDump.coins) {
      if (
        c.coinStatus === CoinStatus.Fresh &&
        0 === Amounts.cmp(c.denomValue, "TESTKUDOS:8")
      ) {
        susp = c.coinPub;
      }
    }
  }

  t.assertTrue(susp !== undefined);

  console.log("suspending coin");

  await walletClient.call(WalletApiOperation.SetCoinSuspended, {
    coinPub: susp,
    suspended: true,
  });

  // This should fail, as we've suspended a coin that we need
  // to pay.
  await t.assertThrowsAsync(async () => {
    await walletClient.call(WalletApiOperation.TestPay, {
      amount: "TESTKUDOS:5" as AmountString,
      merchantAuthToken: merchantAuthToken,
      merchantBaseUrl: merchant.makeInstanceBaseUrl(),
      summary: "foo",
    });
  });

  console.log("unsuspending coin");

  await walletClient.call(WalletApiOperation.SetCoinSuspended, {
    coinPub: susp,
    suspended: false,
  });

  await walletClient.call(WalletApiOperation.TestPay, {
    amount: "TESTKUDOS:5" as AmountString,
    merchantAuthToken: merchantAuthToken,
    merchantBaseUrl: merchant.makeInstanceBaseUrl(),
    summary: "foo",
  });

  await walletClient.call(WalletApiOperation.ClearDb, {});
  await walletClient.call(WalletApiOperation.RunIntegrationTestV2, {
    corebankApiBaseUrl: bankClient.baseUrl,
    exchangeBaseUrl: exchange.baseUrl,
    merchantAuthToken: merchantAuthToken,
    merchantBaseUrl: merchant.makeInstanceBaseUrl(),
  });
}

runWallettestingTest.suites = ["wallet"];
