/*
 This file is part of GNU Taler
 (C) 2022-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */
import {
  Button,
  InputLine,
  LocalNotificationBanner,
  UIHandlerId,
  useLocalNotificationHandler,
  useTranslationContext
} from "@gnu-taler/web-util/browser";
import { VNode, h } from "preact";
import { FormErrors, useFormState } from "../hooks/form.js";
import { useOfficer } from "../hooks/officer.js";
import { undefinedIfEmpty } from "./CreateAccount.js";

type FormType = {
  password: string;
};

export function UnlockAccount(): VNode {
  const { i18n } = useTranslationContext();

  const officer = useOfficer();
  const [notification, withErrorHandler] = useLocalNotificationHandler();

  const [form, status] = useFormState<FormType>(
    [".password"] as Array<UIHandlerId>,
    {
      password: undefined,
    },
    (state) => {
      const errors = undefinedIfEmpty<FormErrors<FormType>>({
        password: !state.password ? i18n.str`required` : undefined,
      });
      if (errors === undefined) {
        return {
          status: "ok",
          result: state as FormType,
          errors,
        };
      }
      return {
        status: "fail",
        result: state,
        errors,
      };
    },
  );

  const unlockHandler =
    status.status === "fail" || officer.state !== "locked"
      ? undefined
      : withErrorHandler(
          async () => officer.tryUnlock(form.password!.value!),
          () => {},
        );

  const forgetHandler =
    status.status === "fail" || officer.state !== "locked"
      ? undefined
      : withErrorHandler(
          async () => officer.forget(),
          () => {},
        );

  return (
    <div class="flex min-h-full flex-col ">
      <LocalNotificationBanner notification={notification} />

      <div class="sm:mx-auto sm:w-full sm:max-w-md">
        <h1 class="mt-6 text-center text-2xl font-bold leading-9 tracking-tight text-gray-900">
          <i18n.Translate>Account locked</i18n.Translate>
        </h1>
        <p class="mt-6 text-lg leading-8 text-gray-600">
          <i18n.Translate>
            Your account is normally locked anytime you reload. To unlock type
            your password again.
          </i18n.Translate>
        </p>
      </div>

      <div class="mt-10 sm:mx-auto sm:w-full sm:max-w-[480px] ">
        <div class="bg-gray-100 px-6 py-6 shadow sm:rounded-lg sm:px-12">

          <div class="mb-4">
            <InputLine<FormType, "password">
              label={i18n.str`Password`}
              name="password"
              type="password"
              required
              handler={form.password}
            />
          </div>

          <div class="mt-8">
            <Button
              type="submit"
              handler={unlockHandler}
              disabled={!unlockHandler}
              class="disabled:opacity-50 disabled:cursor-default  flex w-full justify-center rounded-md bg-indigo-600 px-3 py-1.5 text-sm font-semibold leading-6 text-white shadow-sm hover:bg-indigo-500 focus-visible:outline focus-visible:outline-2 focus-visible:outline-offset-2 focus-visible:outline-indigo-600"
            >
              <i18n.Translate>Unlock</i18n.Translate>
            </Button>
          </div>

        </div>
        <Button
          type="button"
          handler={forgetHandler}
          disabled={!forgetHandler}
          class="disabled:opacity-50 disabled:cursor-default m-4 block rounded-md bg-red-600 px-3 py-2 text-center text-sm  text-white shadow-sm hover:bg-red-500 "
        >
          <i18n.Translate>Forget account</i18n.Translate>
        </Button>
      </div>
    </div>
  );
}
