package ntlmssp

import (
	"testing"

	"github.com/stretchr/testify/assert"
)

func TestAuthenticateMessageUnmarshal(t *testing.T) {
	tables := []struct {
		got  []byte
		want authenticateMessage
		err  error
	}{
		{
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x03, 0x00, 0x00, 0x00, 0x18, 0x00, 0x18, 0x00,
				0x6c, 0x00, 0x00, 0x00, 0x18, 0x00, 0x18, 0x00,
				0x84, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x0c, 0x00,
				0x48, 0x00, 0x00, 0x00, 0x08, 0x00, 0x08, 0x00,
				0x54, 0x00, 0x00, 0x00, 0x10, 0x00, 0x10, 0x00,
				0x5c, 0x00, 0x00, 0x00, 0x10, 0x00, 0x10, 0x00,
				0x9c, 0x00, 0x00, 0x00, 0x35, 0x82, 0x80, 0xe2,
				0x05, 0x01, 0x28, 0x0a, 0x00, 0x00, 0x00, 0x0f,
				0x44, 0x00, 0x6f, 0x00, 0x6d, 0x00, 0x61, 0x00,
				0x69, 0x00, 0x6e, 0x00, 0x55, 0x00, 0x73, 0x00,
				0x65, 0x00, 0x72, 0x00, 0x43, 0x00, 0x4f, 0x00,
				0x4d, 0x00, 0x50, 0x00, 0x55, 0x00, 0x54, 0x00,
				0x45, 0x00, 0x52, 0x00, 0x98, 0xde, 0xf7, 0xb8,
				0x7f, 0x88, 0xaa, 0x5d, 0xaf, 0xe2, 0xdf, 0x77,
				0x96, 0x88, 0xa1, 0x72, 0xde, 0xf1, 0x1c, 0x7d,
				0x5c, 0xcd, 0xef, 0x13, 0x67, 0xc4, 0x30, 0x11,
				0xf3, 0x02, 0x98, 0xa2, 0xad, 0x35, 0xec, 0xe6,
				0x4f, 0x16, 0x33, 0x1c, 0x44, 0xbd, 0xbe, 0xd9,
				0x27, 0x84, 0x1f, 0x94, 0x51, 0x88, 0x22, 0xb1,
				0xb3, 0xf3, 0x50, 0xc8, 0x95, 0x86, 0x82, 0xec,
				0xbb, 0x3e, 0x3c, 0xb7,
			},
			authenticateMessage{
				authenticateMessageFields: authenticateMessageFields{
					messageHeader: newMessageHeader(ntLmAuthenticate),
					LmChallengeResponseFields: payload{
						Len:    24,
						MaxLen: 24,
						Offset: 108,
					},
					NtChallengeResponseFields: payload{
						Len:    24,
						MaxLen: 24,
						Offset: 132,
					},
					DomainNameFields: payload{
						Len:    12,
						MaxLen: 12,
						Offset: 72,
					},
					UserNameFields: payload{
						Len:    8,
						MaxLen: 8,
						Offset: 84,
					},
					WorkstationFields: payload{
						Len:    16,
						MaxLen: 16,
						Offset: 92,
					},
					EncryptedRandomSessionKeyFields: payload{
						Len:    16,
						MaxLen: 16,
						Offset: 156,
					},
					NegotiateFlags: 0xe2808235,
				},
				LmChallengeResponse: []byte{
					0x98, 0xde, 0xf7, 0xb8, 0x7f, 0x88, 0xaa, 0x5d,
					0xaf, 0xe2, 0xdf, 0x77, 0x96, 0x88, 0xa1, 0x72,
					0xde, 0xf1, 0x1c, 0x7d, 0x5c, 0xcd, 0xef, 0x13,
				},
				NtChallengeResponse: []byte{
					0x67, 0xc4, 0x30, 0x11, 0xf3, 0x02, 0x98, 0xa2,
					0xad, 0x35, 0xec, 0xe6, 0x4f, 0x16, 0x33, 0x1c,
					0x44, 0xbd, 0xbe, 0xd9, 0x27, 0x84, 0x1f, 0x94,
				},
				DomainName:  "Domain",
				UserName:    "User",
				Workstation: "COMPUTER",
				EncryptedRandomSessionKey: []byte{
					0x51, 0x88, 0x22, 0xb1, 0xb3, 0xf3, 0x50, 0xc8,
					0x95, 0x86, 0x82, 0xec, 0xbb, 0x3e, 0x3c, 0xb7,
				},
				Version: &Version{
					ProductMajorVersion: 5,
					ProductMinorVersion: 1,
					ProductBuild:        2600,
					NTLMRevisionCurrent: NTLMSSPRevisionW2K3,
				},
				MIC: nil,
			},
			nil,
		},
	}

	for _, table := range tables {
		m := authenticateMessage{}
		err := m.Unmarshal(table.got)
		assert.Equal(t, table.err, err)
		if err == nil {
			assert.Equal(t, table.want, m)
		}
	}
}

func TestAuthenticateMessageMarshal(t *testing.T) {
	tables := []struct {
		got  authenticateMessage
		want []byte
		err  error
	}{
		{
			authenticateMessage{
				authenticateMessageFields: authenticateMessageFields{
					messageHeader:  newMessageHeader(ntLmAuthenticate),
					NegotiateFlags: 0xe2808235,
				},
				LmChallengeResponse: []byte{
					0x98, 0xde, 0xf7, 0xb8, 0x7f, 0x88, 0xaa, 0x5d,
					0xaf, 0xe2, 0xdf, 0x77, 0x96, 0x88, 0xa1, 0x72,
					0xde, 0xf1, 0x1c, 0x7d, 0x5c, 0xcd, 0xef, 0x13,
				},
				NtChallengeResponse: []byte{
					0x67, 0xc4, 0x30, 0x11, 0xf3, 0x02, 0x98, 0xa2,
					0xad, 0x35, 0xec, 0xe6, 0x4f, 0x16, 0x33, 0x1c,
					0x44, 0xbd, 0xbe, 0xd9, 0x27, 0x84, 0x1f, 0x94,
				},
				DomainName:  "Domain",
				UserName:    "User",
				Workstation: "COMPUTER",
				EncryptedRandomSessionKey: []byte{
					0x51, 0x88, 0x22, 0xb1, 0xb3, 0xf3, 0x50, 0xc8,
					0x95, 0x86, 0x82, 0xec, 0xbb, 0x3e, 0x3c, 0xb7,
				},
				Version: &Version{
					ProductMajorVersion: 5,
					ProductMinorVersion: 1,
					ProductBuild:        2600,
					NTLMRevisionCurrent: NTLMSSPRevisionW2K3,
				},
				MIC: nil,
			},
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x03, 0x00, 0x00, 0x00, 0x18, 0x00, 0x18, 0x00,
				0x6c, 0x00, 0x00, 0x00, 0x18, 0x00, 0x18, 0x00,
				0x84, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x0c, 0x00,
				0x48, 0x00, 0x00, 0x00, 0x08, 0x00, 0x08, 0x00,
				0x54, 0x00, 0x00, 0x00, 0x10, 0x00, 0x10, 0x00,
				0x5c, 0x00, 0x00, 0x00, 0x10, 0x00, 0x10, 0x00,
				0x9c, 0x00, 0x00, 0x00, 0x35, 0x82, 0x80, 0xe2,
				0x05, 0x01, 0x28, 0x0a, 0x00, 0x00, 0x00, 0x0f,
				0x44, 0x00, 0x6f, 0x00, 0x6d, 0x00, 0x61, 0x00,
				0x69, 0x00, 0x6e, 0x00, 0x55, 0x00, 0x73, 0x00,
				0x65, 0x00, 0x72, 0x00, 0x43, 0x00, 0x4f, 0x00,
				0x4d, 0x00, 0x50, 0x00, 0x55, 0x00, 0x54, 0x00,
				0x45, 0x00, 0x52, 0x00, 0x98, 0xde, 0xf7, 0xb8,
				0x7f, 0x88, 0xaa, 0x5d, 0xaf, 0xe2, 0xdf, 0x77,
				0x96, 0x88, 0xa1, 0x72, 0xde, 0xf1, 0x1c, 0x7d,
				0x5c, 0xcd, 0xef, 0x13, 0x67, 0xc4, 0x30, 0x11,
				0xf3, 0x02, 0x98, 0xa2, 0xad, 0x35, 0xec, 0xe6,
				0x4f, 0x16, 0x33, 0x1c, 0x44, 0xbd, 0xbe, 0xd9,
				0x27, 0x84, 0x1f, 0x94, 0x51, 0x88, 0x22, 0xb1,
				0xb3, 0xf3, 0x50, 0xc8, 0x95, 0x86, 0x82, 0xec,
				0xbb, 0x3e, 0x3c, 0xb7,
			},
			nil,
		},
		{
			authenticateMessage{
				authenticateMessageFields: authenticateMessageFields{
					messageHeader:  newMessageHeader(ntLmAuthenticate),
					NegotiateFlags: 0x82088235,
				},
				LmChallengeResponse: []byte{
					0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				},
				NtChallengeResponse: []byte{
					0x75, 0x37, 0xf8, 0x03, 0xae, 0x36, 0x71, 0x28,
					0xca, 0x45, 0x82, 0x04, 0xbd, 0xe7, 0xca, 0xf8,
					0x1e, 0x97, 0xed, 0x26, 0x83, 0x26, 0x72, 0x32,
				},
				DomainName:                "Domain",
				UserName:                  "User",
				Workstation:               "COMPUTER",
				EncryptedRandomSessionKey: nil,
				Version: &Version{
					ProductMajorVersion: 5,
					ProductMinorVersion: 1,
					ProductBuild:        2600,
					NTLMRevisionCurrent: NTLMSSPRevisionW2K3,
				},
				MIC: nil,
			},
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x03, 0x00, 0x00, 0x00, 0x18, 0x00, 0x18, 0x00,
				0x6c, 0x00, 0x00, 0x00, 0x18, 0x00, 0x18, 0x00,
				0x84, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x0c, 0x00,
				0x48, 0x00, 0x00, 0x00, 0x08, 0x00, 0x08, 0x00,
				0x54, 0x00, 0x00, 0x00, 0x10, 0x00, 0x10, 0x00,
				0x5c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x9c, 0x00, 0x00, 0x00, 0x35, 0x82, 0x08, 0x82,
				0x05, 0x01, 0x28, 0x0a, 0x00, 0x00, 0x00, 0x0f,
				0x44, 0x00, 0x6f, 0x00, 0x6d, 0x00, 0x61, 0x00,
				0x69, 0x00, 0x6e, 0x00, 0x55, 0x00, 0x73, 0x00,
				0x65, 0x00, 0x72, 0x00, 0x43, 0x00, 0x4f, 0x00,
				0x4d, 0x00, 0x50, 0x00, 0x55, 0x00, 0x54, 0x00,
				0x45, 0x00, 0x52, 0x00, 0xaa, 0xaa, 0xaa, 0xaa,
				0xaa, 0xaa, 0xaa, 0xaa, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x75, 0x37, 0xf8, 0x03,
				0xae, 0x36, 0x71, 0x28, 0xca, 0x45, 0x82, 0x04,
				0xbd, 0xe7, 0xca, 0xf8, 0x1e, 0x97, 0xed, 0x26,
				0x83, 0x26, 0x72, 0x32,
			},
			nil,
		},
		{
			authenticateMessage{
				authenticateMessageFields: authenticateMessageFields{
					messageHeader:  newMessageHeader(ntLmAuthenticate),
					NegotiateFlags: 0xe2888235,
				},
				LmChallengeResponse: []byte{
					0x86, 0xc3, 0x50, 0x97, 0xac, 0x9c, 0xec, 0x10,
					0x25, 0x54, 0x76, 0x4a, 0x57, 0xcc, 0xcc, 0x19,
					0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
				},
				NtChallengeResponse: []byte{
					0x68, 0xcd, 0x0a, 0xb8, 0x51, 0xe5, 0x1c, 0x96,
					0xaa, 0xbc, 0x92, 0x7b, 0xeb, 0xef, 0x6a, 0x1c,
					0x01, 0x01, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
					0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
					0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x0c, 0x00,
					0x44, 0x00, 0x6f, 0x00, 0x6d, 0x00, 0x61, 0x00,
					0x69, 0x00, 0x6e, 0x00, 0x01, 0x00, 0x0c, 0x00,
					0x53, 0x00, 0x65, 0x00, 0x72, 0x00, 0x76, 0x00,
					0x65, 0x00, 0x72, 0x00, 0x00, 0x00, 0x00, 0x00,
					0x00, 0x00, 0x00, 0x00,
				},
				DomainName:  "Domain",
				UserName:    "User",
				Workstation: "COMPUTER",
				EncryptedRandomSessionKey: []byte{
					0xc5, 0xda, 0xd2, 0x54, 0x4f, 0xc9, 0x79, 0x90,
					0x94, 0xce, 0x1c, 0xe9, 0x0b, 0xc9, 0xd0, 0x3e,
				},
				Version: &Version{
					ProductMajorVersion: 5,
					ProductMinorVersion: 1,
					ProductBuild:        2600,
					NTLMRevisionCurrent: NTLMSSPRevisionW2K3,
				},
				MIC: nil,
			},
			[]byte{
				0x4e, 0x54, 0x4c, 0x4d, 0x53, 0x53, 0x50, 0x00,
				0x03, 0x00, 0x00, 0x00, 0x18, 0x00, 0x18, 0x00,
				0x6c, 0x00, 0x00, 0x00, 0x54, 0x00, 0x54, 0x00,
				0x84, 0x00, 0x00, 0x00, 0x0c, 0x00, 0x0c, 0x00,
				0x48, 0x00, 0x00, 0x00, 0x08, 0x00, 0x08, 0x00,
				0x54, 0x00, 0x00, 0x00, 0x10, 0x00, 0x10, 0x00,
				0x5c, 0x00, 0x00, 0x00, 0x10, 0x00, 0x10, 0x00,
				0xd8, 0x00, 0x00, 0x00, 0x35, 0x82, 0x88, 0xe2,
				0x05, 0x01, 0x28, 0x0a, 0x00, 0x00, 0x00, 0x0f,
				0x44, 0x00, 0x6f, 0x00, 0x6d, 0x00, 0x61, 0x00,
				0x69, 0x00, 0x6e, 0x00, 0x55, 0x00, 0x73, 0x00,
				0x65, 0x00, 0x72, 0x00, 0x43, 0x00, 0x4f, 0x00,
				0x4d, 0x00, 0x50, 0x00, 0x55, 0x00, 0x54, 0x00,
				0x45, 0x00, 0x52, 0x00, 0x86, 0xc3, 0x50, 0x97,
				0xac, 0x9c, 0xec, 0x10, 0x25, 0x54, 0x76, 0x4a,
				0x57, 0xcc, 0xcc, 0x19, 0xaa, 0xaa, 0xaa, 0xaa,
				0xaa, 0xaa, 0xaa, 0xaa, 0x68, 0xcd, 0x0a, 0xb8,
				0x51, 0xe5, 0x1c, 0x96, 0xaa, 0xbc, 0x92, 0x7b,
				0xeb, 0xef, 0x6a, 0x1c, 0x01, 0x01, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0x00, 0x00, 0x00, 0x00, 0xaa, 0xaa, 0xaa, 0xaa,
				0xaa, 0xaa, 0xaa, 0xaa, 0x00, 0x00, 0x00, 0x00,
				0x02, 0x00, 0x0c, 0x00, 0x44, 0x00, 0x6f, 0x00,
				0x6d, 0x00, 0x61, 0x00, 0x69, 0x00, 0x6e, 0x00,
				0x01, 0x00, 0x0c, 0x00, 0x53, 0x00, 0x65, 0x00,
				0x72, 0x00, 0x76, 0x00, 0x65, 0x00, 0x72, 0x00,
				0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
				0xc5, 0xda, 0xd2, 0x54, 0x4f, 0xc9, 0x79, 0x90,
				0x94, 0xce, 0x1c, 0xe9, 0x0b, 0xc9, 0xd0, 0x3e,
			},
			nil,
		},
	}

	for _, table := range tables {
		b, err := table.got.Marshal()
		assert.Equal(t, table.err, err)
		if err == nil {
			assert.Equal(t, table.want, b)
		}
	}
}
