/*
 * Copyright (c) 2003, 2021, Oracle and/or its affiliates. All rights reserved.
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS FILE HEADER.
 *
 * This code is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 only, as
 * published by the Free Software Foundation.  Oracle designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Oracle in the LICENSE file that accompanied this code.
 *
 * This code is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License
 * version 2 for more details (a copy is included in the LICENSE file that
 * accompanied this code).
 *
 * You should have received a copy of the GNU General Public License version
 * 2 along with this work; if not, write to the Free Software Foundation,
 * Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 *
 * Please contact Oracle, 500 Oracle Parkway, Redwood Shores, CA 94065 USA
 * or visit www.oracle.com if you need additional information or have any
 * questions.
 */

package sun.nio.cs;

import java.nio.charset.Charset;
import java.nio.charset.CharsetDecoder;
import java.nio.charset.CharsetEncoder;
import java.util.Arrays;
import sun.nio.cs.DoubleByte;
import sun.nio.cs.HistoricallyNamedCharset;

public class IBM943C extends Charset implements HistoricallyNamedCharset
{

    public IBM943C() {
        super("x-IBM943C", StandardCharsets.aliases_IBM943C());
    }

    public String historicalName() {
        return "Cp943C";
    }

    public boolean contains(Charset cs) {
        return ((cs.name().equals("US-ASCII"))
                || (cs instanceof IBM943C));
    }

    public CharsetDecoder newDecoder() {
        return new DoubleByte.Decoder(this,
                                      IBM943.DecodeHolder.b2c,
                                      Holder.b2cSB,
                                      0x40,
                                      0xfc);
    }

    public CharsetEncoder newEncoder() {
        return new DoubleByte.Encoder(this, Holder.c2b, Holder.c2bIndex);
    }

    private static class Holder {
        static final char[] b2cSB;
        static final char[] c2b;
        static final char[] c2bIndex;

        static {
            b2cSB = new char[0x100];
            for (int i = 0; i < 0x80; i++) {
                b2cSB[i] = (char)i;
            }
            for (int i = 0x80; i < 0x100; i++) {
                b2cSB[i] = IBM943.DecodeHolder.b2cSB[i];
            }
            c2b = Arrays.copyOf(IBM943.EncodeHolder.c2b, IBM943.EncodeHolder.c2b.length);
            c2bIndex = Arrays.copyOf(IBM943.EncodeHolder.c2bIndex, IBM943.EncodeHolder.c2bIndex.length);
            for (char c = '\0'; c < '\u0080'; ++c) {
                int index = c2bIndex[c >> 8];
                c2b[index + (c & 0xff)] = c;
            }
        }
    }
}
