use core:io;
use core:geometry;
use graphics;
use ui;

/**
 * This is the root object that describes general properties of the file itself.
 *
 * In this implementation, we merge the 'catalog' and the 'pages' objects of the PDF file.
 */
class Catalog extends StrObject {
	// All pages in the document.
	Page[] pages;

	// Resources.
	Resources resources;

	// Create.
	init() {
		Pages p;
		init() {
			pages = p.pages;
			pagesData = p;
		}
	}

	// Called when added to somewhere.
	void addedTo(File f) {
		f.add(pagesData);
		f.add(resources);
		if (identityFn)
			f.add(identityFn);
		file = f;
	}

	// Output.
	void output(StrBuf to) {
		to << "<</Type /Catalog /Pages " << pagesData.id << " 0 R >>";
	}

	// Get the ID of the 'pages' object.
	Nat pagesId() {
		pagesData.id;
	}

	// Get the identity function.
	IdentityFunction identity() {
		if (identityFn) {
			identityFn;
		} else {
			IdentityFunction fn;
			identityFn = fn;
			if (file)
				file.add(fn);
			fn;
		}
	}

	// Reference to the 'pages' object.
	private Pages pagesData;

	// File we've been added to.
	private File? file;

	// We need to split this object into two parts, and this object represents the 'Pages' part.
	private class Pages extends StrObject {
		// All pages. Shared with the parent Catalog object.
		Page[] pages;

		// Output.
		void output(StrBuf to) {
			to << "<</Type /Pages /Kids [";
			for (p in pages)
				to << p.id << " 0 R ";
			to << "] /Count " << pages.count << " >>";
		}
	}

	// Identity function. Created when needed.
	IdentityFunction? identityFn;
}

/**
 * Describes a single page of a PDF document.
 */
class Page extends StrObject {
	// Remember the owning file.
	File file;

	// Physical size of this page (in points, 1/74 inches).
	Rect mediaBox;

	// Content of this page.
	Content content;

	// Create a page.
	init(File f, Rect box) {
		init() {
			file = f;
			mediaBox = box;
		}

		f.add(this);
		f.add(content);
	}


	// Output.
	void output(StrBuf to) {
		to << "<</Type /Page /MediaBox [";
		to << mediaBox.p0.x << " " << mediaBox.p0.y << " " << mediaBox.p1.x << " " << mediaBox.p1.y;
		to << "] /Contents " << content.id << " 0 R ";
		to << "/Parent " << file.root.pagesId << " 0 R ";
		to << "/Resources " << file.root.resources.id << " 0 R >>";
	}
}


/**
 * Describes the content of a page.
 */
class Content extends StreamObject {
	// Commands to execute.
	StrBuf data;

	// Output.
	void output(OStream to) {
		var t = textOutput(to);
		t.write(data.toS);
		t.flush();
	}
}


/**
 * Describes a Form XObject (distinct from PDF forms).
 *
 * These are used to group objects, most notably in order to describe transparency properly, and to
 * embed images.
 */
class XObject extends StreamObject {
	// Owning file.
	File file;

	// The name of this XObject in the resource dictionary.
	Str name;

	// Create, associated with a file.
	init(File file) {
		init() {
			file = file;
		}

		// Add ourselves to the file.
		file.add(this);

		// Register ourselves with the global 'resources' object.
		file.root.resources.xObj(this);
	}
}

/**
 * Class describing a Form XObject.
 *
 * We use these objects to implement transparency groups, but they may also be used to group
 * operations together to improve the efficiency of the PDF file.
 *
 * Note: It is possible to specify a transform to apply to XObjects. However, it seems like some PDF
 * readers do not respect that property in all cases (for example, when using the form as a soft map
 * for transparency). However, in those cases, the "bound" property seems to be either ignored or
 * relative to the original coordinate system, which makes it possible to ignore the transform
 * property. For this reason, we don't implement it here.
 */
class FormXObject extends XObject {
	// Bounding box.
	Rect bound;

	// Commands to execute.
	StrBuf data;

	// Colorspace (if any)
	Str? colorspace;

	// Create.
	init(File file, Rect bound) {
		init(file) {
			bound = bound;
		}
	}

	// Output the header.
	void outputHeader(TextOutput to) {
		StrBuf o;
		o << "/Type /XObject ";
		o << "/Subtype /Form ";
		o << "/BBox [" << bound.p0.x << " " << bound.p0.y << " " << bound.p1.x << " " << bound.p1.y << "] ";
		o << "/Resources " << file.root.resources.id << " 0 R ";
		o << "/Group << /S /Transparency ";
		if (colorspace)
			o << "/CS " << colorspace;
		o << " >> ";

		to.write(o.toS);
	}

	// Output.
	void output(OStream to) {
		var t = textOutput(to);
		t.write(data.toS);
		t.flush();
	}
}

// Convert a color component to a byte.
private Byte toByte(Float v) {
	(v * 255.0).int.byte;
}

/**
 * Class describing an image XObject.
 */
class ImageXObject extends XObject {
	// Source bitmap.
	Bitmap src;

	// The alpha channel for this image, if we have one!
	AlphaXObject? alpha;

	// Create.
	init(File file, Bitmap bitmap) {
		init(file) { src = bitmap; }

		if (src.image.hasAlpha) {
			alpha = AlphaXObject(file, bitmap);
		}
	}

	// Output the header.
	void outputHeader(TextOutput to) {
		to.write("/Type /XObject /Subtype /Image ");
		to.write("/Width ${src.size.w.int} /Height ${src.size.h.int} ");
		to.write("/ColorSpace /DeviceRGB /BitsPerComponent 8 /Interpolate true ");
		if (alpha) {
			to.write("/SMask ${alpha.id} 0 R ");
		}
	}

	// Output.
	void output(OStream to) {
		Image img = src.image;
		for (Nat y = 0; y < img.height; y++) {
			outputRow(to, img, y);
		}
	}

	// Output a single scanline.
	void outputRow(OStream to, Image img, Nat y) {
		Buffer b = buffer(img.width * 3);

		Nat at = 0;
		for (Nat x = 0; x < img.width; x++) {
			Color c = img.get(x, y);
			b[at++] = toByte(c.r);
			b[at++] = toByte(c.g);
			b[at++] = toByte(c.b);
		}

		b.filled(img.width * 3);
		to.write(b);
	}
}


/**
 * Class describing an image's alpha channel.
 */
class AlphaXObject extends XObject {
	// Source bitmap.
	Bitmap src;

	// Create.
	init(File file, Bitmap bitmap) {
		init(file) { src = bitmap; }
	}

	// Output the header.
	void outputHeader(TextOutput to) {
		to.write("/Type /XObject /Subtype /Image ");
		to.write("/Width ${src.size.w.int} /Height ${src.size.h.int} ");
		to.write("/ColorSpace /DeviceGray /BitsPerComponent 8 /Interpolate true ");
	}

	// Output.
	void output(OStream to) {
		Image img = src.image;
		for (Nat y = 0; y < img.height; y++) {
			outputRow(to, img, y);
		}
	}

	// Output a single scanline.
	void outputRow(OStream to, Image img, Nat y) {
		Buffer b = buffer(img.width);

		for (Nat x = 0; x < img.width; x++) {
			Color c = img.get(x, y);
			b[x] = toByte(c.a);
		}

		b.filled(img.width);
		to.write(b);
	}

	// Convert a float value to a byte.
	Byte toByte(Float v) : static {
		(v * 255.0).int.byte;
	}
}


/**
 * Describes resources for one or more pages.
 *
 * This object keeps track of objects in Basic Storm and their corresponding representation in a PDF
 * file.
 *
 * The Graphic object creates one global table of all resources so that it can be used for all pages
 * easily.
 */
class Resources extends StrObject {

	// Find (or create) a font specification matching the provided font.
	Str font(ui:Font f) {
		if (fonts.has(f.name)) {
			fonts.get(f.name).name;
		} else {
			Str name = newName("F");
			res.put(name);
			fonts.put(f.name, FontResource(name, f));

			name;
		}
	}

	// Add an XObject.
	void xObj(XObject obj) {
		obj.name = newName("X");
		res.put(obj.name);
		xObjs << obj;
	}

	// Get the name of a bitmap.
	Str bitmap(File file, Bitmap b) {
		if (bitmaps.has(b)) {
			bitmaps.get(b).name;
		} else {
			ImageXObject obj(file, b);
			bitmaps.put(b, obj);
			obj.name;
		}
	}

	// Get the ID of a bitmap object.
	Nat bitmapId(File file, Bitmap b) {
		if (bitmaps.has(b)) {
			bitmaps.get(b).id;
		} else {
			ImageXObject obj(file, b);
			bitmaps.put(b, obj);
			obj.id;
		}
	}

	// Add a graphics state that describes a desired alpha channel.
	Str alpha(Float alpha) {
		Int key = (alpha * 100).int;
		if (alphas.has(key)) {
			alphas.get(key).name;
		} else {
			Str name = newName("GS");
			res.put(name);
			GSResource r(name, alpha);
			alphas.put(key, r);
			gs << r;

			name;
		}
	}

	// Add a graphics state describing a form XObject as a soft mask.
	Str softMask(FormXObject mask) {
		Str name = newName("GS");
		res.put(name);
		gs << GSResource(name, mask);

		name;
	}

	// Add a pattern resource, return its new title. Generally, these are not reusable and as such
	// we don't bother to de-duplicate them.
	Str pattern(PatternResource p) {
		var name = newName("P");
		res.put(name);
		p.name = name;
		patterns << p;
		name;
	}

	// Output.
	void output(StrBuf to) {
		to << "<</Font <<\n";
		for (v in fonts) {
			to << v.name << " ";
			v.output(to);
			to << "\n";
		}
		to << ">>\n";

		to << "/XObject <<\n";
		for (x in xObjs) {
			to << x.name << " " << x.id << " 0 R\n";
		}
		to << ">>\n";

		to << "/ExtGState <<\n";
		for (v in gs) {
			to << v.name << " ";
			v.output(to);
			to << "\n";
		}
		to << ">>\n";

		to << "/Pattern <<\n";
		for (p in patterns) {
			to << p.name << " " << p.id << " 0 R\n";
		}
		to << ">> ";

		to << ">>";
	}

private:
	// The names of all resources, so that we make sure to keep identifiers unique.
	Set<Str> res;

	// All fonts Font name -> resource.
	Str->FontResource fonts;

	// All bitmaps.
	Bitmap->ImageXObject bitmaps;

	// All graphics state resources.
	Int->GSResource alphas;

	// GS resources that are not for alpha.
	GSResource[] gs;

	// All patterns.
	PatternResource[] patterns;

	// All XObjects.
	XObject[] xObjs;

	// Generate a identifier for a resource (with the given prefix). The prefix should not start with "/"
	Str newName(Str prefix) {
		Nat id = 1;
		Str candidate;
		do {
			candidate = "/${prefix}${id}";
		} while (res.has(candidate)) {
			id++;
		}

		candidate;
	}
}


/**
 * A font resource.
 */
class FontResource {
	// The name of this resource
	Str name;

	// Font.
	ui:Font font;

 	// Create.
	init(Str name, ui:Font font) {
		init() {
			name = name;
			font = font;
		}
	}

	// Output.
	void output(StrBuf to) {
		to << "<</Type /Font /Subtype /TrueType /BaseFont /";

		// Remove spaces in the font's name.
		Char space = " ".begin.v;
		for (c in font.name) {
			if (c != space)
				to << c;
		}

		// TODO: Allow other encodings? We should eventually support UTF through custom encodings.
		to << " /Encoding /WinAnsiEncoding >>";
	}
}


/**
 * A graphics state resource.
 */
class GSResource {
	// The name of this resource
	Str name;

	// Alpha value.
	Float alpha;

	// A form object used as the soft mask for this object. If present, we don't use 'alpha'.
	FormXObject? mask;

	// Create.
	init(Str name, Float alpha) {
		init() {
			name = name;
			alpha = alpha;
		}
	}

	// Create.
	init(Str name, FormXObject mask) {
		init() {
			name = name;
			mask = mask;
		}

		mask.colorspace = "/DeviceGray";
	}

	// Output.
	void output(StrBuf to) {
		if (mask) {
			to << "<</Type /ExtGState /SMask << /S /Luminosity /G " << mask.id << " 0 R >> >>";
		} else {
			to << "<</Type /ExtGState /SMask /None /CA " << alpha << " /ca " << alpha << ">>";
		}
	}
}

// Output a transform (inside a stream).
package void putTransform(StrBuf to, Transform tfm) {
	to << tfm.at(0, 0) << " " << tfm.at(0, 1) << " "
	   << tfm.at(1, 0) << " " << tfm.at(1, 1) << " "
	   << tfm.at(3, 0) << " " << tfm.at(3, 1);
}

// Output a matrix.
private void putMatrix(StrBuf to, Transform tfm) {
	to << "/Matrix [ ";
	putTransform(to, tfm);
	to << " ] ";
}

/**
 * A Pattern resource.
 */
class PatternResource extends MaybeStreamObject {
	// The name of this resource. Set by the Resources class when added.
	Str name;

	// Pattern type.
	Nat type;

	// File we're added to.
	File file;

	// Create.
	init(File file, Nat type) {
		init() {
			type = type;
			file = file;
		}

		file.add(this);
	}

	// Output
	void output(StrBuf to) {
		to << "/Type /Pattern /PatternType " << type << " ";
		outputData(to);
	}

	// Output pattern-specific data.
	void outputData(StrBuf to) : abstract;
}

/**
 * A tiling bitmap pattern.
 *
 * This could be used to use any drawing as a tiling pattern, but we're only using it for bitmaps at
 * the moment.
 */
class BitmapPattern extends PatternResource {
	// ID of the bitmap we're repeating.
	Nat repeat;

	// Size of the bitmap.
	Size repeatSize;

	// Transform of the bitmap.
	Transform bitmapTfm;

	// Transform from the current coordinate system to the page.
	Transform pageTfm;

	// Create.
	init(File file, BitmapBrush brush, Transform currentToPage) {
		Bitmap b = brush.bitmap;

		// 1 = tiling pattern
		init(file, 1) {
			repeat = file.root.resources.bitmapId(file, b);
			repeatSize = b.size;
			bitmapTfm = brush.transform;
			pageTfm = currentToPage;
		}
	}

	void outputData(StrBuf to) {
		// Size of the bitmap, used for the bounding rectangle.
		Size size = repeatSize;

		// Transform of the pattern. Remember: Images are 1 by 1 unit large by default!
		var transform = scale(Size(size.w, -size.h)) * bitmapTfm * pageTfm;

		to << "/PaintType 1 "; // colored
		to << "/TilingType 1 "; // constant spacing, may distort the image by at most 1 device pixel
		to << "/BBox [ 0 0 1 1 ] ";
		to << "/XStep 1 ";
		to << "/YStep 1 ";
		to << "/Resources << /XObject << /XO1 " << repeat << " 0 R >> >> ";
		// to << "/Resources " << res.id << " 0 R ";
		putMatrix(to, transform);
	}

	Bool outputStream(OStream to) {
		var text = textOutput(to);
		text.write("/XO1 Do");
		text.flush();

		true;
	}
}

/**
 * A shading pattern.
 */
class ShadingPattern extends PatternResource {
	// Shading type.
	Nat shadingType;

	// Functions for shading according to the stops.
	StitchFunction redFn;
	StitchFunction greenFn;
	StitchFunction blueFn;

	// Alpha shading (if any).
	FormXObject? alpha;

	// Transform from the virtual coordinate system to the physical coordinate system.
	Transform pageTfm;

	// Create.
	init(File file, Nat shadingType, Gradient gradient, Transform currentToPage, Rect pageBounds) {
		Float[] p;
		Float[] r;
		Float[] g;
		Float[] b;
		Bool hasAlpha = false;
		for (s in gradient.stops) {
			p << s.pos;
			r << s.color.r;
			g << s.color.g;
			b << s.color.b;

			hasAlpha |= s.color.a < 0.99;
		}

		// 2 = shading pattern
		init(file, 2) {
			redFn(file, p, r);
			greenFn(file, p, g);
			blueFn(file, p, b);
			shadingType = shadingType;
			pageTfm = currentToPage;
		}

		if (hasAlpha) {
			setupAlpha(file, gradient, currentToPage, pageBounds);
		}
	}

	void setupAlpha(File file, Gradient gradient, Transform currentToPage, Rect bound) {
		Transform inv = currentToPage.inverted;
		Point a = bound.p0 * inv;
		Point b = Point(bound.p0.x, bound.p1.y) * inv;
		Point c = bound.p1 * inv;
		Point d = Point(bound.p1.x, bound.p0.y) * inv;

		Rect bbox = Rect(a, b).normalized.include(c).include(d);

		AlphaShadingPattern alphaPattern(file, this, gradient);
		FormXObject g(file, bbox);

		g.data << "/Pattern cs " << file.root.resources.pattern(alphaPattern) << " scn ";
		g.data << a.x << " " << a.y << " m\n";
		g.data << b.x << " " << b.y << " l\n";
		g.data << c.x << " " << c.y << " l\n";
		g.data << d.x << " " << d.y << " l h f\n";

		alpha = g;
	}

	// Output pattern-specific data.
	void outputData(StrBuf to) {
		outputTransform(to, pageTfm);
		to << "/Shading <<";
		to << "/ShadingType " << shadingType;
		to << " /ColorSpace /DeviceRGB";
		to << " /Function [ " << redFn.id << " 0 R " << greenFn.id << " 0 R " << blueFn.id << " 0 R ] ";
		outputShading(to, pageTfm);
		to << ">> ";
	}

	// Output any transformation matrix we desire.
	void outputTransform(StrBuf to, Transform tfm) {}

	// Output shading specific data.
	void outputShading(StrBuf to, Transform tfm) : abstract;
}


/**
 * A shading pattern suitable for the alpha channel. Re-uses the subclass implementation of other shadings.
 */
class AlphaShadingPattern extends PatternResource {
	// The pattern we're embedded inside. We will ask that pattern to output the specifics so we don't have to.
	ShadingPattern owner;

	// Our function.
	StitchFunction fn;

	// Create.
	init(File file, ShadingPattern owner, Gradient gradient) {
		Float[] p;
		Float[] a;
		for (s in gradient.stops) {
			p << s.pos;
			a << s.color.a;
		}

		init(file, 2) {
			owner = owner;
			fn(file, p, a);
		}
	}

	// Output pattern-specific data.
	void outputData(StrBuf to) {
		owner.outputTransform(to, Transform());
		to << "/Shading <<";
		to << "/ShadingType " << owner.shadingType;
		to << " /ColorSpace /DeviceGray ";
		to << " /Function [ " << fn.id << " 0 R ] ";
		owner.outputShading(to, Transform());
		to << ">> ";
	}
}


/**
 * A linear gradient pattern resource.
 */
class AxialShading extends ShadingPattern {
	// Coordinates defining the line along which the pattern shall be applied.
	Point p0;
	Point p1;

	// Create.
	// 'currentToPage' is the transform from the currently active coordinate space to the physical coordinates
	// of whatever we're currently drawing to.
	init(File file, LinearGradient gradient, Transform currentToPage, Rect pageBounds) {
		// We're an axial shading.
		init(file, 2, gradient, currentToPage, pageBounds) {
			p0 = gradient.start;
			p1 = gradient.end;
		}
	}

	// Output our data.
	void outputShading(StrBuf to, Transform tfm) {
		Point a = p0 * tfm;
		Point b = p1 * tfm;
		to << "/Coords [" << a.x << " " << a.y << " " << b.x << " " << b.y << "] ";
		to << "/Extend [true true] ";
	}
}

/**
 * A radial gradient pattern resource.
 */
class RadialShading extends ShadingPattern {
	// Center point, radius and transform.
	Point center;
	Float radius;
	Transform tfm;

	// Create. See 'AxialShading' for details on parameters.
	init(File file, RadialGradient gradient, Transform currentToPage, Rect pageBounds) {
		// We're a radial shading.
		init(file, 3, gradient, currentToPage, pageBounds) {
			center = gradient.center;
			radius = gradient.radius;
			tfm = gradient.transform;
		}
	}

	// Set a custom transform.
	void outputTransform(StrBuf to, Transform t) {
		putMatrix(to, tfm * t);
	}

	// Output our data.
	void outputShading(StrBuf to, Transform tfm) {
		to << "/Coords [" << center.x << " " << center.y << " 0 " << center.x << " " << center.y << " " << radius << "] ";
		to << "/Extend [true true] ";
	}
}


/**
 * An identity function in the PDF. Stitched together to make shading functions.
 */
class IdentityFunction extends StreamObject {
	void outputHeader(TextOutput to) {
		to.write("/FunctionType 0 ");
		to.write("/Domain [ 0.0 1.0 ] ");
		to.write("/Range [ 0.0 1.0 ] ");
		to.write("/Size [ 2 ] ");
		to.write("/BitsPerSample 16");
	}

	void output(OStream to) {
		Buffer b = buffer(4);
		b[0] = 0;
		b[1] = 0;
		b[2] = 0xFF;
		b[3] = 0xFF;
		b.filled(4);
		to.write(b);
	}
}


/**
 * A unary stitched function usable to approximate shading functions.
 */
class StitchFunction extends StrObject {
	// Identity function ID.
	Nat identityId;

	// Points where we have samples.
	Float[] points;

	// Sample values at those points.
	Float[] values;

	// Create.
	init(File file, Float[] points, Float[] values) {
		points = points.clone;
		values = values.clone;

		if (points.first > 0.0) {
			points.insert(0, 0.0);
			values.insert(0, values.first);
		}

		if (points.last < 1.0) {
			points.push(1.0);
			values.push(values.last);
		}

		init() {
			identityId = file.root.identity.id;
			points = points;
			values = values;
		}

		file.add(this);
	}

	// Output.
	void output(StrBuf to) {
		to << "<</FunctionType 3 /Domain [ 0.0 1.0 ] /Range [ 0.0 1.0 ] /Functions [";
		to << (" ${identityId} 0 R" * (points.count - 1));
		to << " ] /Bounds [";
		for (Nat i = 1; i < points.count - 1; i++)
			to << " " << points[i];

		to << " ] /Encode [";
		for (Nat i = 1; i < points.count; i++)
			to << " " << values[i - 1] << " " << values[i];
		to << "] >>";
	}
}
