%define OT_NearestNeighbourAlgorithm_doc
"Nearest neighbour lookup.

Base class to define an algorithm to search for nearest neighbours of
a list of points.

Parameters
----------
sample : :class:`~openturns.Sample`
    Input points.

Notes
-----

Two algorithms can be selected in any dimension:

* :class:`~openturns.NaiveNearestNeighbour` loops over all points
  of the sample to find the closest one.
* :class:`~openturns.KDTree` builds a binary tree.

Two algorithms are specific to 1D input dimension, and much more efficient:

* :class:`~openturns.RegularGridNearestNeighbour` is the most efficient
  algorithm when points corresponds to a :class:`~openturns.RegularGrid`,
  :meth:`query` works in constant time.
* :class:`~openturns.NearestNeighbour1D` looks for nearest neighbour by
  dichotomy in 1D.

It is recommended to use derived classes in order to select the best algorithm
according to your data.  If you create a generic :class:`NearestNeighbourAlgorithm`,
here is how the derived class is selected:

* If points correspond to a :class:`~openturns.RegularGrid`,
  :class:`~openturns.RegularGridNearestNeighbour` algorithm is selected.
* If input dimension is 1, :class:`~openturns.NearestNeighbour1D` is selected.
* Otherwise, :class:`~openturns.KDTree` is selected.

By default, the algorithm is :class:`~openturns.KDTree`.

See also
--------
KDTree, NearestNeighbour1D, RegularGridNearestNeighbour, NaiveNearestNeighbour

Examples
--------
>>> import openturns as ot
>>> sample = ot.Normal(2).getSample(10)
>>> finder = ot.NearestNeighbourAlgorithm(sample)
>>> neighbour = sample[finder.query([0.1, 0.2])]"
%enddef

%feature("docstring") OT::NearestNeighbourAlgorithmImplementation
OT_NearestNeighbourAlgorithm_doc

// ---------------------------------------------------------------------

%define OT_NearestNeighbourAlgorithm_setSample_doc
"Build a NearestNeighbourAlgorithm from these points.

Parameters
----------
sample : :class:`~openturns.Sample`
    Input points."
%enddef

%feature("docstring") OT::NearestNeighbourAlgorithmImplementation::setSample
OT_NearestNeighbourAlgorithm_setSample_doc

// ---------------------------------------------------------------------

%define OT_NearestNeighbourAlgorithm_getSample_doc
"Get the points which have been used to build this nearest neighbour algorithm.

Returns
-------
sample : :class:`~openturns.Sample`
    Input points."
%enddef

%feature("docstring") OT::NearestNeighbourAlgorithmImplementation::getSample
OT_NearestNeighbourAlgorithm_getSample_doc

// ---------------------------------------------------------------------

%define OT_NearestNeighbourAlgorithm_query_doc
"Get the index of the nearest neighbour of the given point.

Available usages:
    query(*point*)

    query(*sample*)

Parameters
----------
point : sequence of float
    Given point.
sample : 2-d sequence of float
    Given points.

Returns
-------
index : int
    Index of the nearest neighbour of the given point.
indices : :class:`~openturns.Indices`
    Index of the nearest neighbour of the given points."
%enddef

%feature("docstring") OT::NearestNeighbourAlgorithmImplementation::query
OT_NearestNeighbourAlgorithm_query_doc

// ---------------------------------------------------------------------

%define OT_NearestNeighbourAlgorithm_queryK_doc
"Get the indices of nearest neighbours of the given point.

Parameters
----------
x : sequence of float
    Given point.
k : int
    Number of indices to return.
sorted : bool, optional
    Boolean to tell whether returned indices are sorted according to
    the distance to the given point.

Returns
-------
indices : sequence of int
    Indices of the `k` nearest neighbours of the given point."
%enddef

%feature("docstring") OT::NearestNeighbourAlgorithmImplementation::queryK
OT_NearestNeighbourAlgorithm_queryK_doc

