/*
 * Copyright 2021-2022 the original author or authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      https://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.opentest4j.reporting.tooling.validator;

import java.util.Iterator;
import java.util.List;
import java.util.Objects;

import static java.util.stream.Collectors.joining;

/**
 * Result of validating an XML report file.
 */
public class ValidationResult implements Iterable<ValidationMessage> {

	private final List<ValidationMessage> messages;

	ValidationResult(List<ValidationMessage> messages) {
		this.messages = List.copyOf(messages);
	}

	/**
	 * Count the number of validation messages with the supplied severity.
	 *
	 * @param severity the severity to count
	 * @return number of validation messages with the supplied severity
	 */
	public long count(Severity severity) {
		return messages.stream().filter(m -> severity.equals(m.getSeverity())).count();
	}

	@Override
	public Iterator<ValidationMessage> iterator() {
		return messages.iterator();
	}

	@Override
	public String toString() {
		return messages.stream() //
				.map(Objects::toString) //
				.collect(joining("\n"));
	}
}
