{-# LANGUAGE ImplicitParams, RankNTypes, TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- The base object type.
-- 
-- @GObject@ is the fundamental type providing the common attributes and
-- methods for all object types in GTK, Pango and other libraries
-- based on GObject. The @GObject@ class provides methods for object
-- construction and destruction, property access methods, and signal
-- support. Signals are described in detail [here][gobject-Signals].
-- 
-- For a tutorial on implementing a new @GObject@ class, see <http://developer.gnome.org/gobject/stable/tutorial.html#how-to-define-and-implement-a-new-gobject How to define and
-- implement a new GObject>.
-- For a list of naming conventions for GObjects and their methods, see the
-- <http://developer.gnome.org/gobject/stable/concepts.html#conventions GType conventions>. For the high-level concepts
-- behind GObject, read
-- <http://developer.gnome.org/gobject/stable/concepts.html#instantiatable-classed-types-objects Instantiatable classed types: Objects>.
-- 
-- Since GLib 2.72, all @GObject@s are guaranteed to be aligned to at least the
-- alignment of the largest basic GLib type (typically this is @guint64@ or
-- @gdouble@). If you need larger alignment for an element in a @GObject@, you
-- should allocate it on the heap (aligned), or arrange for your @GObject@ to be
-- appropriately padded. This guarantee applies to the @GObject@ (or derived)
-- struct, the @GObjectClass@ (or derived) struct, and any private data allocated
-- by @G_ADD_PRIVATE()@.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.GObject.Objects.Object
    ( 
#if defined(ENABLE_OVERLOADING)
    ObjectInterfaceListPropertiesMethodInfo ,
#endif

-- * Exported types
    Object(..)                              ,
    IsObject                                ,
    toObject                                ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getData]("GI.GObject.Objects.Object#g:method:getData"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata").
-- 
-- ==== Setters
-- [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty").

#if defined(ENABLE_OVERLOADING)
    ResolveObjectMethod                     ,
#endif

-- ** bindProperty #method:bindProperty#

#if defined(ENABLE_OVERLOADING)
    ObjectBindPropertyMethodInfo            ,
#endif
    objectBindProperty                      ,


-- ** bindPropertyFull #method:bindPropertyFull#

#if defined(ENABLE_OVERLOADING)
    ObjectBindPropertyFullMethodInfo        ,
#endif
    objectBindPropertyFull                  ,


-- ** compatControl #method:compatControl#

    objectCompatControl                     ,


-- ** forceFloating #method:forceFloating#

#if defined(ENABLE_OVERLOADING)
    ObjectForceFloatingMethodInfo           ,
#endif
    objectForceFloating                     ,


-- ** freezeNotify #method:freezeNotify#

#if defined(ENABLE_OVERLOADING)
    ObjectFreezeNotifyMethodInfo            ,
#endif
    objectFreezeNotify                      ,


-- ** getData #method:getData#

#if defined(ENABLE_OVERLOADING)
    ObjectGetDataMethodInfo                 ,
#endif
    objectGetData                           ,


-- ** getProperty #method:getProperty#

#if defined(ENABLE_OVERLOADING)
    ObjectGetPropertyMethodInfo             ,
#endif
    objectGetProperty                       ,


-- ** getQdata #method:getQdata#

#if defined(ENABLE_OVERLOADING)
    ObjectGetQdataMethodInfo                ,
#endif
    objectGetQdata                          ,


-- ** getv #method:getv#

#if defined(ENABLE_OVERLOADING)
    ObjectGetvMethodInfo                    ,
#endif
    objectGetv                              ,


-- ** interfaceFindProperty #method:interfaceFindProperty#

    objectInterfaceFindProperty             ,


-- ** interfaceInstallProperty #method:interfaceInstallProperty#

    objectInterfaceInstallProperty          ,


-- ** isFloating #method:isFloating#

#if defined(ENABLE_OVERLOADING)
    ObjectIsFloatingMethodInfo              ,
#endif
    objectIsFloating                        ,


-- ** newv #method:newv#

    objectNewv                              ,


-- ** notify #method:notify#

#if defined(ENABLE_OVERLOADING)
    ObjectNotifyMethodInfo                  ,
#endif
    objectNotify                            ,


-- ** notifyByPspec #method:notifyByPspec#

#if defined(ENABLE_OVERLOADING)
    ObjectNotifyByPspecMethodInfo           ,
#endif
    objectNotifyByPspec                     ,


-- ** ref #method:ref#

#if defined(ENABLE_OVERLOADING)
    ObjectRefMethodInfo                     ,
#endif
    objectRef                               ,


-- ** refSink #method:refSink#

#if defined(ENABLE_OVERLOADING)
    ObjectRefSinkMethodInfo                 ,
#endif
    objectRefSink                           ,


-- ** runDispose #method:runDispose#

#if defined(ENABLE_OVERLOADING)
    ObjectRunDisposeMethodInfo              ,
#endif
    objectRunDispose                        ,


-- ** setData #method:setData#

#if defined(ENABLE_OVERLOADING)
    ObjectSetDataMethodInfo                 ,
#endif
    objectSetData                           ,


-- ** setDataFull #method:setDataFull#

#if defined(ENABLE_OVERLOADING)
    ObjectSetDataFullMethodInfo             ,
#endif
    objectSetDataFull                       ,


-- ** setProperty #method:setProperty#

#if defined(ENABLE_OVERLOADING)
    ObjectSetPropertyMethodInfo             ,
#endif
    objectSetProperty                       ,


-- ** stealData #method:stealData#

#if defined(ENABLE_OVERLOADING)
    ObjectStealDataMethodInfo               ,
#endif
    objectStealData                         ,


-- ** stealQdata #method:stealQdata#

#if defined(ENABLE_OVERLOADING)
    ObjectStealQdataMethodInfo              ,
#endif
    objectStealQdata                        ,


-- ** thawNotify #method:thawNotify#

#if defined(ENABLE_OVERLOADING)
    ObjectThawNotifyMethodInfo              ,
#endif
    objectThawNotify                        ,


-- ** unref #method:unref#

#if defined(ENABLE_OVERLOADING)
    ObjectUnrefMethodInfo                   ,
#endif
    objectUnref                             ,


-- ** watchClosure #method:watchClosure#

#if defined(ENABLE_OVERLOADING)
    ObjectWatchClosureMethodInfo            ,
#endif
    objectWatchClosure                      ,




 -- * Signals


-- ** notify #signal:notify#

    ObjectNotifyCallback                    ,
#if defined(ENABLE_OVERLOADING)
    ObjectNotifySignalInfo                  ,
#endif
    afterObjectNotify                       ,
    onObjectNotify                          ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GObject.Callbacks as GObject.Callbacks
import {-# SOURCE #-} qualified GI.GObject.Flags as GObject.Flags
import {-# SOURCE #-} qualified GI.GObject.Interfaces.TypePlugin as GObject.TypePlugin
import {-# SOURCE #-} qualified GI.GObject.Objects.Binding as GObject.Binding
import {-# SOURCE #-} qualified GI.GObject.Structs.InterfaceInfo as GObject.InterfaceInfo
import {-# SOURCE #-} qualified GI.GObject.Structs.Parameter as GObject.Parameter
import {-# SOURCE #-} qualified GI.GObject.Structs.TypeClass as GObject.TypeClass
import {-# SOURCE #-} qualified GI.GObject.Structs.TypeInfo as GObject.TypeInfo
import {-# SOURCE #-} qualified GI.GObject.Structs.TypeInterface as GObject.TypeInterface
import {-# SOURCE #-} qualified GI.GObject.Structs.TypeValueTable as GObject.TypeValueTable

#else
import qualified GI.GLib.Callbacks as GLib.Callbacks
import {-# SOURCE #-} qualified GI.GObject.Flags as GObject.Flags
import {-# SOURCE #-} qualified GI.GObject.Objects.Binding as GObject.Binding
import {-# SOURCE #-} qualified GI.GObject.Structs.Parameter as GObject.Parameter
import {-# SOURCE #-} qualified GI.GObject.Structs.TypeInterface as GObject.TypeInterface

#endif

-- | Memory-managed wrapper type.
newtype Object = Object (SP.ManagedPtr Object)
    deriving (Eq)

instance SP.ManagedPtrNewtype Object where
    toManagedPtr (Object p) = p

foreign import ccall "g_object_get_type"
    c_g_object_get_type :: IO B.Types.GType

instance B.Types.TypedObject Object where
    glibType = c_g_object_get_type

instance B.Types.GObject Object

-- | Type class for types which can be safely cast to t'Object', for instance with `toObject`.
class (SP.GObject o, O.IsDescendantOf Object o) => IsObject o
instance (SP.GObject o, O.IsDescendantOf Object o) => IsObject o

instance O.HasParentTypes Object
type instance O.ParentTypes Object = '[]

-- | Cast to t'Object', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toObject :: (MIO.MonadIO m, IsObject o) => o -> m Object
toObject = MIO.liftIO . B.ManagedPtr.unsafeCastTo Object

-- | Convert t'Object' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Object) where
    gvalueGType_ = c_g_object_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr Object)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr Object)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject Object ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveObjectMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveObjectMethod "bindProperty" o = ObjectBindPropertyMethodInfo
    ResolveObjectMethod "bindPropertyFull" o = ObjectBindPropertyFullMethodInfo
    ResolveObjectMethod "forceFloating" o = ObjectForceFloatingMethodInfo
    ResolveObjectMethod "freezeNotify" o = ObjectFreezeNotifyMethodInfo
    ResolveObjectMethod "getv" o = ObjectGetvMethodInfo
    ResolveObjectMethod "isFloating" o = ObjectIsFloatingMethodInfo
    ResolveObjectMethod "notify" o = ObjectNotifyMethodInfo
    ResolveObjectMethod "notifyByPspec" o = ObjectNotifyByPspecMethodInfo
    ResolveObjectMethod "ref" o = ObjectRefMethodInfo
    ResolveObjectMethod "refSink" o = ObjectRefSinkMethodInfo
    ResolveObjectMethod "runDispose" o = ObjectRunDisposeMethodInfo
    ResolveObjectMethod "stealData" o = ObjectStealDataMethodInfo
    ResolveObjectMethod "stealQdata" o = ObjectStealQdataMethodInfo
    ResolveObjectMethod "thawNotify" o = ObjectThawNotifyMethodInfo
    ResolveObjectMethod "unref" o = ObjectUnrefMethodInfo
    ResolveObjectMethod "watchClosure" o = ObjectWatchClosureMethodInfo
    ResolveObjectMethod "getData" o = ObjectGetDataMethodInfo
    ResolveObjectMethod "getProperty" o = ObjectGetPropertyMethodInfo
    ResolveObjectMethod "getQdata" o = ObjectGetQdataMethodInfo
    ResolveObjectMethod "setData" o = ObjectSetDataMethodInfo
    ResolveObjectMethod "setDataFull" o = ObjectSetDataFullMethodInfo
    ResolveObjectMethod "setProperty" o = ObjectSetPropertyMethodInfo
    ResolveObjectMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveObjectMethod t Object, O.OverloadedMethod info Object p) => OL.IsLabel t (Object -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveObjectMethod t Object, O.OverloadedMethod info Object p, R.HasField t Object p) => R.HasField t Object p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveObjectMethod t Object, O.OverloadedMethodInfo info Object) => OL.IsLabel t (O.MethodProxy info Object) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- signal Object::notify
-- | The notify signal is emitted on an object when one of its properties has
-- its value set through 'GI.GObject.Objects.Object.objectSetProperty', @/g_object_set()/@, et al.
-- 
-- Note that getting this signal doesn’t itself guarantee that the value of
-- the property has actually changed. When it is emitted is determined by the
-- derived GObject class. If the implementor did not create the property with
-- 'GI.GObject.Flags.ParamFlagsExplicitNotify', then any call to 'GI.GObject.Objects.Object.objectSetProperty' results
-- in [notify](#g:signal:notify) being emitted, even if the new value is the same as the old.
-- If they did pass 'GI.GObject.Flags.ParamFlagsExplicitNotify', then this signal is emitted only
-- when they explicitly call 'GI.GObject.Objects.Object.objectNotify' or 'GI.GObject.Objects.Object.objectNotifyByPspec',
-- and common practice is to do that only when the value has actually changed.
-- 
-- This signal is typically used to obtain change notification for a
-- single property, by specifying the property name as a detail in the
-- @/g_signal_connect()/@ call, like this:
-- 
-- 
-- === /C code/
-- >
-- >g_signal_connect (text_view->buffer, "notify::paste-target-list",
-- >                  G_CALLBACK (gtk_text_view_target_list_notify),
-- >                  text_view)
-- 
-- 
-- It is important to note that you must use
-- [canonical parameter names][class/@gObject@/.ParamSpec@/parameter/@-names] as
-- detail strings for the notify signal.
type ObjectNotifyCallback =
    GParamSpec
    -- ^ /@pspec@/: the t'GI.GObject.Objects.ParamSpec.ParamSpec' of the property which changed.
    -> IO ()

type C_ObjectNotifyCallback =
    Ptr Object ->                           -- object
    Ptr GParamSpec ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_ObjectNotifyCallback`.
foreign import ccall "wrapper"
    mk_ObjectNotifyCallback :: C_ObjectNotifyCallback -> IO (FunPtr C_ObjectNotifyCallback)

wrap_ObjectNotifyCallback :: 
    GObject a => (a -> ObjectNotifyCallback) ->
    C_ObjectNotifyCallback
wrap_ObjectNotifyCallback gi'cb gi'selfPtr pspec _ = do
    pspec' <- B.GParamSpec.newGParamSpecFromPtr pspec
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  pspec'


-- | Connect a signal handler for the [notify](#signal:notify) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' object #notify callback
-- @
-- 
-- This signal admits a optional parameter @detail@.
-- If it's not @Nothing@, we will connect to “@notify::detail@” instead.
-- 
onObjectNotify :: (IsObject a, MonadIO m) => a -> P.Maybe T.Text -> ((?self :: a) => ObjectNotifyCallback) -> m SignalHandlerId
onObjectNotify obj detail cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ObjectNotifyCallback wrapped
    wrapped'' <- mk_ObjectNotifyCallback wrapped'
    connectSignalFunPtr obj "notify" wrapped'' SignalConnectBefore detail

-- | Connect a signal handler for the [notify](#signal:notify) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' object #notify callback
-- @
-- 
-- This signal admits a optional parameter @detail@.
-- If it's not @Nothing@, we will connect to “@notify::detail@” instead.
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterObjectNotify :: (IsObject a, MonadIO m) => a -> P.Maybe T.Text -> ((?self :: a) => ObjectNotifyCallback) -> m SignalHandlerId
afterObjectNotify obj detail cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_ObjectNotifyCallback wrapped
    wrapped'' <- mk_ObjectNotifyCallback wrapped'
    connectSignalFunPtr obj "notify" wrapped'' SignalConnectAfter detail


#if defined(ENABLE_OVERLOADING)
data ObjectNotifySignalInfo
instance SignalInfo ObjectNotifySignalInfo where
    type HaskellCallbackType ObjectNotifySignalInfo = ObjectNotifyCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_ObjectNotifyCallback cb
        cb'' <- mk_ObjectNotifyCallback cb'
        connectSignalFunPtr obj "notify" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GObject.Objects.Object::notify"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gobject-2.0.31/docs/GI-GObject-Objects-Object.html#g:signal:notify"})

#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Object
type instance O.AttributeList Object = ObjectAttributeList
type ObjectAttributeList = ('[ ] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList Object = ObjectSignalList
type ObjectSignalList = ('[ '("notify", ObjectNotifySignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method Object::newv
-- method type : Constructor
-- Args: [ Arg
--           { argCName = "object_type"
--           , argType = TBasicType TGType
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the type id of the #GObject subtype to instantiate"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "n_parameters"
--           , argType = TBasicType TUInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the length of the @parameters array"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "parameters"
--           , argType =
--               TCArray
--                 False
--                 (-1)
--                 1
--                 (TInterface Name { namespace = "GObject" , name = "Parameter" })
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an array of #GParameter"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "n_parameters"
--              , argType = TBasicType TUInt
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "the length of the @parameters array"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Just (TInterface Name { namespace = "GObject" , name = "Object" })
-- throws : False
-- Skip return : False

foreign import ccall "g_object_newv" g_object_newv :: 
    CGType ->                               -- object_type : TBasicType TGType
    Word32 ->                               -- n_parameters : TBasicType TUInt
    Ptr GObject.Parameter.Parameter ->      -- parameters : TCArray False (-1) 1 (TInterface (Name {namespace = "GObject", name = "Parameter"}))
    IO (Ptr Object)

{-# DEPRECATED objectNewv ["(Since version 2.54)","Use @/g_object_new_with_properties()/@ instead.","deprecated. See t'GI.GObject.Structs.Parameter.Parameter' for more information."] #-}
-- | Creates a new instance of a t'GI.GObject.Objects.Object.Object' subtype and sets its properties.
-- 
-- Construction parameters (see 'GI.GObject.Flags.ParamFlagsConstruct', 'GI.GObject.Flags.ParamFlagsConstructOnly')
-- which are not explicitly specified are set to their default values.
objectNewv ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    GType
    -- ^ /@objectType@/: the type id of the t'GI.GObject.Objects.Object.Object' subtype to instantiate
    -> [GObject.Parameter.Parameter]
    -- ^ /@parameters@/: an array of t'GI.GObject.Structs.Parameter.Parameter'
    -> m Object
    -- ^ __Returns:__ a new instance of
    -- /@objectType@/
objectNewv objectType parameters = liftIO $ do
    let nParameters = fromIntegral $ P.length parameters
    let objectType' = gtypeToCGType objectType
    parameters' <- mapM unsafeManagedPtrGetPtr parameters
    parameters'' <- packBlockArray 32 parameters'
    result <- g_object_newv objectType' nParameters parameters''
    checkUnexpectedReturnNULL "objectNewv" result
    result' <- (wrapObject Object) result
    mapM_ touchManagedPtr parameters
    freeMem parameters''
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Object::bind_property
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "source"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the source #GObject"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "source_property"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the property on @source to bind"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "target"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the target #GObject"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "target_property"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the property on @target to bind"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "BindingFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags to pass to #GBinding"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GObject" , name = "Binding" })
-- throws : False
-- Skip return : False

foreign import ccall "g_object_bind_property" g_object_bind_property :: 
    Ptr Object ->                           -- source : TInterface (Name {namespace = "GObject", name = "Object"})
    CString ->                              -- source_property : TBasicType TUTF8
    Ptr Object ->                           -- target : TInterface (Name {namespace = "GObject", name = "Object"})
    CString ->                              -- target_property : TBasicType TUTF8
    CUInt ->                                -- flags : TInterface (Name {namespace = "GObject", name = "BindingFlags"})
    IO (Ptr GObject.Binding.Binding)

-- | Creates a binding between /@sourceProperty@/ on /@source@/ and /@targetProperty@/
-- on /@target@/.
-- 
-- Whenever the /@sourceProperty@/ is changed the /@targetProperty@/ is
-- updated using the same value. For instance:
-- 
-- 
-- === /C code/
-- >
-- >  g_object_bind_property (action, "active", widget, "sensitive", 0);
-- 
-- 
-- Will result in the \"sensitive\" property of the widget t'GI.GObject.Objects.Object.Object' instance to be
-- updated with the same value of the \"active\" property of the action t'GI.GObject.Objects.Object.Object'
-- instance.
-- 
-- If /@flags@/ contains 'GI.GObject.Flags.BindingFlagsBidirectional' then the binding will be mutual:
-- if /@targetProperty@/ on /@target@/ changes then the /@sourceProperty@/ on /@source@/
-- will be updated as well.
-- 
-- The binding will automatically be removed when either the /@source@/ or the
-- /@target@/ instances are finalized. To remove the binding without affecting the
-- /@source@/ and the /@target@/ you can just call 'GI.GObject.Objects.Object.objectUnref' on the returned
-- t'GI.GObject.Objects.Binding.Binding' instance.
-- 
-- Removing the binding by calling 'GI.GObject.Objects.Object.objectUnref' on it must only be done if
-- the binding, /@source@/ and /@target@/ are only used from a single thread and it
-- is clear that both /@source@/ and /@target@/ outlive the binding. Especially it
-- is not safe to rely on this if the binding, /@source@/ or /@target@/ can be
-- finalized from different threads. Keep another reference to the binding and
-- use 'GI.GObject.Objects.Binding.bindingUnbind' instead to be on the safe side.
-- 
-- A t'GI.GObject.Objects.Object.Object' can have multiple bindings.
-- 
-- /Since: 2.26/
objectBindProperty ::
    (B.CallStack.HasCallStack, MonadIO m, IsObject a, IsObject b) =>
    a
    -- ^ /@source@/: the source t'GI.GObject.Objects.Object.Object'
    -> T.Text
    -- ^ /@sourceProperty@/: the property on /@source@/ to bind
    -> b
    -- ^ /@target@/: the target t'GI.GObject.Objects.Object.Object'
    -> T.Text
    -- ^ /@targetProperty@/: the property on /@target@/ to bind
    -> [GObject.Flags.BindingFlags]
    -- ^ /@flags@/: flags to pass to t'GI.GObject.Objects.Binding.Binding'
    -> m GObject.Binding.Binding
    -- ^ __Returns:__ the t'GI.GObject.Objects.Binding.Binding' instance representing the
    --     binding between the two t'GI.GObject.Objects.Object.Object' instances. The binding is released
    --     whenever the t'GI.GObject.Objects.Binding.Binding' reference count reaches zero.
objectBindProperty source sourceProperty target targetProperty flags = liftIO $ do
    source' <- unsafeManagedPtrCastPtr source
    sourceProperty' <- textToCString sourceProperty
    target' <- unsafeManagedPtrCastPtr target
    targetProperty' <- textToCString targetProperty
    let flags' = gflagsToWord flags
    result <- g_object_bind_property source' sourceProperty' target' targetProperty' flags'
    checkUnexpectedReturnNULL "objectBindProperty" result
    result' <- (newObject GObject.Binding.Binding) result
    touchManagedPtr source
    touchManagedPtr target
    freeMem sourceProperty'
    freeMem targetProperty'
    return result'

#if defined(ENABLE_OVERLOADING)
data ObjectBindPropertyMethodInfo
instance (signature ~ (T.Text -> b -> T.Text -> [GObject.Flags.BindingFlags] -> m GObject.Binding.Binding), MonadIO m, IsObject a, IsObject b) => O.OverloadedMethod ObjectBindPropertyMethodInfo a signature where
    overloadedMethod = objectBindProperty

instance O.OverloadedMethodInfo ObjectBindPropertyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GObject.Objects.Object.objectBindProperty",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gobject-2.0.31/docs/GI-GObject-Objects-Object.html#v:objectBindProperty"
        })


#endif

-- method Object::bind_property_full
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "source"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the source #GObject"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "source_property"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the property on @source to bind"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "target"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the target #GObject"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "target_property"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the property on @target to bind"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "flags"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "BindingFlags" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "flags to pass to #GBinding"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "transform_to"
--           , argType = TGClosure Nothing
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GClosure wrapping the transformation function\n    from the @source to the @target, or %NULL to use the default"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "transform_from"
--           , argType = TGClosure Nothing
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "a #GClosure wrapping the transformation function\n    from the @target to the @source, or %NULL to use the default"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GObject" , name = "Binding" })
-- throws : False
-- Skip return : False

foreign import ccall "g_object_bind_property_with_closures" g_object_bind_property_with_closures :: 
    Ptr Object ->                           -- source : TInterface (Name {namespace = "GObject", name = "Object"})
    CString ->                              -- source_property : TBasicType TUTF8
    Ptr Object ->                           -- target : TInterface (Name {namespace = "GObject", name = "Object"})
    CString ->                              -- target_property : TBasicType TUTF8
    CUInt ->                                -- flags : TInterface (Name {namespace = "GObject", name = "BindingFlags"})
    Ptr (GClosure ()) ->                    -- transform_to : TGClosure Nothing
    Ptr (GClosure ()) ->                    -- transform_from : TGClosure Nothing
    IO (Ptr GObject.Binding.Binding)

-- | Creates a binding between /@sourceProperty@/ on /@source@/ and /@targetProperty@/
-- on /@target@/, allowing you to set the transformation functions to be used by
-- the binding.
-- 
-- This function is the language bindings friendly version of
-- @/g_object_bind_property_full()/@, using @/GClosures/@ instead of
-- function pointers.
-- 
-- /Since: 2.26/
objectBindPropertyFull ::
    (B.CallStack.HasCallStack, MonadIO m, IsObject a, IsObject b) =>
    a
    -- ^ /@source@/: the source t'GI.GObject.Objects.Object.Object'
    -> T.Text
    -- ^ /@sourceProperty@/: the property on /@source@/ to bind
    -> b
    -- ^ /@target@/: the target t'GI.GObject.Objects.Object.Object'
    -> T.Text
    -- ^ /@targetProperty@/: the property on /@target@/ to bind
    -> [GObject.Flags.BindingFlags]
    -- ^ /@flags@/: flags to pass to t'GI.GObject.Objects.Binding.Binding'
    -> GClosure c
    -- ^ /@transformTo@/: a t'GI.GObject.Structs.Closure.Closure' wrapping the transformation function
    --     from the /@source@/ to the /@target@/, or 'P.Nothing' to use the default
    -> GClosure d
    -- ^ /@transformFrom@/: a t'GI.GObject.Structs.Closure.Closure' wrapping the transformation function
    --     from the /@target@/ to the /@source@/, or 'P.Nothing' to use the default
    -> m GObject.Binding.Binding
    -- ^ __Returns:__ the t'GI.GObject.Objects.Binding.Binding' instance representing the
    --     binding between the two t'GI.GObject.Objects.Object.Object' instances. The binding is released
    --     whenever the t'GI.GObject.Objects.Binding.Binding' reference count reaches zero.
objectBindPropertyFull source sourceProperty target targetProperty flags transformTo transformFrom = liftIO $ do
    source' <- unsafeManagedPtrCastPtr source
    sourceProperty' <- textToCString sourceProperty
    target' <- unsafeManagedPtrCastPtr target
    targetProperty' <- textToCString targetProperty
    let flags' = gflagsToWord flags
    transformTo' <- unsafeManagedPtrCastPtr transformTo
    transformFrom' <- unsafeManagedPtrCastPtr transformFrom
    result <- g_object_bind_property_with_closures source' sourceProperty' target' targetProperty' flags' transformTo' transformFrom'
    checkUnexpectedReturnNULL "objectBindPropertyFull" result
    result' <- (newObject GObject.Binding.Binding) result
    touchManagedPtr source
    touchManagedPtr target
    touchManagedPtr transformTo
    touchManagedPtr transformFrom
    freeMem sourceProperty'
    freeMem targetProperty'
    return result'

#if defined(ENABLE_OVERLOADING)
data ObjectBindPropertyFullMethodInfo
instance (signature ~ (T.Text -> b -> T.Text -> [GObject.Flags.BindingFlags] -> GClosure c -> GClosure d -> m GObject.Binding.Binding), MonadIO m, IsObject a, IsObject b) => O.OverloadedMethod ObjectBindPropertyFullMethodInfo a signature where
    overloadedMethod = objectBindPropertyFull

instance O.OverloadedMethodInfo ObjectBindPropertyFullMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GObject.Objects.Object.objectBindPropertyFull",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gobject-2.0.31/docs/GI-GObject-Objects-Object.html#v:objectBindPropertyFull"
        })


#endif

-- method Object::force_floating
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GObject" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_object_force_floating" g_object_force_floating :: 
    Ptr Object ->                           -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    IO ()

-- | This function is intended for t'GI.GObject.Objects.Object.Object' implementations to re-enforce
-- a [floating][floating-ref] object reference. Doing this is seldom
-- required: all @/GInitiallyUnowneds/@ are created with a floating reference
-- which usually just needs to be sunken by calling 'GI.GObject.Objects.Object.objectRefSink'.
-- 
-- /Since: 2.10/
objectForceFloating ::
    (B.CallStack.HasCallStack, MonadIO m, IsObject a) =>
    a
    -- ^ /@object@/: a t'GI.GObject.Objects.Object.Object'
    -> m ()
objectForceFloating object = liftIO $ do
    object' <- unsafeManagedPtrCastPtr object
    g_object_force_floating object'
    touchManagedPtr object
    return ()

#if defined(ENABLE_OVERLOADING)
data ObjectForceFloatingMethodInfo
instance (signature ~ (m ()), MonadIO m, IsObject a) => O.OverloadedMethod ObjectForceFloatingMethodInfo a signature where
    overloadedMethod = objectForceFloating

instance O.OverloadedMethodInfo ObjectForceFloatingMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GObject.Objects.Object.objectForceFloating",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gobject-2.0.31/docs/GI-GObject-Objects-Object.html#v:objectForceFloating"
        })


#endif

-- method Object::freeze_notify
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GObject" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_object_freeze_notify" g_object_freeze_notify :: 
    Ptr Object ->                           -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    IO ()

-- | Increases the freeze count on /@object@/. If the freeze count is
-- non-zero, the emission of \"notify\" signals on /@object@/ is
-- stopped. The signals are queued until the freeze count is decreased
-- to zero. Duplicate notifications are squashed so that at most one
-- [Object::notify]("GI.GObject.Objects.Object#g:signal:notify") signal is emitted for each property modified while the
-- object is frozen.
-- 
-- This is necessary for accessors that modify multiple properties to prevent
-- premature notification while the object is still being modified.
objectFreezeNotify ::
    (B.CallStack.HasCallStack, MonadIO m, IsObject a) =>
    a
    -- ^ /@object@/: a t'GI.GObject.Objects.Object.Object'
    -> m ()
objectFreezeNotify object = liftIO $ do
    object' <- unsafeManagedPtrCastPtr object
    g_object_freeze_notify object'
    touchManagedPtr object
    return ()

#if defined(ENABLE_OVERLOADING)
data ObjectFreezeNotifyMethodInfo
instance (signature ~ (m ()), MonadIO m, IsObject a) => O.OverloadedMethod ObjectFreezeNotifyMethodInfo a signature where
    overloadedMethod = objectFreezeNotify

instance O.OverloadedMethodInfo ObjectFreezeNotifyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GObject.Objects.Object.objectFreezeNotify",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gobject-2.0.31/docs/GI-GObject-Objects-Object.html#v:objectFreezeNotify"
        })


#endif

-- method Object::get_data
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "#GObject containing the associations"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "name of the key for that association"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TPtr)
-- throws : False
-- Skip return : False

foreign import ccall "g_object_get_data" g_object_get_data :: 
    Ptr Object ->                           -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    CString ->                              -- key : TBasicType TUTF8
    IO (Ptr ())

-- | Gets a named field from the objects table of associations (see 'GI.GObject.Objects.Object.objectSetData').
objectGetData ::
    (B.CallStack.HasCallStack, MonadIO m, IsObject a) =>
    a
    -- ^ /@object@/: t'GI.GObject.Objects.Object.Object' containing the associations
    -> T.Text
    -- ^ /@key@/: name of the key for that association
    -> m (Ptr ())
    -- ^ __Returns:__ the data if found,
    --          or 'P.Nothing' if no such data exists.
objectGetData object key = liftIO $ do
    object' <- unsafeManagedPtrCastPtr object
    key' <- textToCString key
    result <- g_object_get_data object' key'
    touchManagedPtr object
    freeMem key'
    return result

#if defined(ENABLE_OVERLOADING)
data ObjectGetDataMethodInfo
instance (signature ~ (T.Text -> m (Ptr ())), MonadIO m, IsObject a) => O.OverloadedMethod ObjectGetDataMethodInfo a signature where
    overloadedMethod = objectGetData

instance O.OverloadedMethodInfo ObjectGetDataMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GObject.Objects.Object.objectGetData",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gobject-2.0.31/docs/GI-GObject-Objects-Object.html#v:objectGetData"
        })


#endif

-- method Object::get_property
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GObject" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "property_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of the property to get"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TGValue
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "return location for the property value"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_object_get_property" g_object_get_property :: 
    Ptr Object ->                           -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    CString ->                              -- property_name : TBasicType TUTF8
    Ptr GValue ->                           -- value : TGValue
    IO ()

-- | Gets a property of an object.
-- 
-- The /@value@/ can be:
-- 
-- * an empty t'GI.GObject.Structs.Value.Value' initialized by @/G_VALUE_INIT/@, which will be
--   automatically initialized with the expected type of the property
--   (since GLib 2.60)
-- * a t'GI.GObject.Structs.Value.Value' initialized with the expected type of the property
-- * a t'GI.GObject.Structs.Value.Value' initialized with a type to which the expected type
--   of the property can be transformed
-- 
-- 
-- In general, a copy is made of the property contents and the caller is
-- responsible for freeing the memory by calling 'GI.GObject.Structs.Value.valueUnset'.
-- 
-- Note that 'GI.GObject.Objects.Object.objectGetProperty' is really intended for language
-- bindings, @/g_object_get()/@ is much more convenient for C programming.
objectGetProperty ::
    (B.CallStack.HasCallStack, MonadIO m, IsObject a) =>
    a
    -- ^ /@object@/: a t'GI.GObject.Objects.Object.Object'
    -> T.Text
    -- ^ /@propertyName@/: the name of the property to get
    -> GValue
    -- ^ /@value@/: return location for the property value
    -> m ()
objectGetProperty object propertyName value = liftIO $ do
    object' <- unsafeManagedPtrCastPtr object
    propertyName' <- textToCString propertyName
    value' <- unsafeManagedPtrGetPtr value
    g_object_get_property object' propertyName' value'
    touchManagedPtr object
    touchManagedPtr value
    freeMem propertyName'
    return ()

#if defined(ENABLE_OVERLOADING)
data ObjectGetPropertyMethodInfo
instance (signature ~ (T.Text -> GValue -> m ()), MonadIO m, IsObject a) => O.OverloadedMethod ObjectGetPropertyMethodInfo a signature where
    overloadedMethod = objectGetProperty

instance O.OverloadedMethodInfo ObjectGetPropertyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GObject.Objects.Object.objectGetProperty",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gobject-2.0.31/docs/GI-GObject-Objects-Object.html#v:objectGetProperty"
        })


#endif

-- method Object::get_qdata
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "The GObject to get a stored user data pointer from"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "quark"
--           , argType = TBasicType TUInt32
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GQuark, naming the user data pointer"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TPtr)
-- throws : False
-- Skip return : False

foreign import ccall "g_object_get_qdata" g_object_get_qdata :: 
    Ptr Object ->                           -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    Word32 ->                               -- quark : TBasicType TUInt32
    IO (Ptr ())

-- | This function gets back user data pointers stored via
-- @/g_object_set_qdata()/@.
objectGetQdata ::
    (B.CallStack.HasCallStack, MonadIO m, IsObject a) =>
    a
    -- ^ /@object@/: The GObject to get a stored user data pointer from
    -> Word32
    -- ^ /@quark@/: A @/GQuark/@, naming the user data pointer
    -> m (Ptr ())
    -- ^ __Returns:__ The user data pointer set, or 'P.Nothing'
objectGetQdata object quark = liftIO $ do
    object' <- unsafeManagedPtrCastPtr object
    result <- g_object_get_qdata object' quark
    touchManagedPtr object
    return result

#if defined(ENABLE_OVERLOADING)
data ObjectGetQdataMethodInfo
instance (signature ~ (Word32 -> m (Ptr ())), MonadIO m, IsObject a) => O.OverloadedMethod ObjectGetQdataMethodInfo a signature where
    overloadedMethod = objectGetQdata

instance O.OverloadedMethodInfo ObjectGetQdataMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GObject.Objects.Object.objectGetQdata",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gobject-2.0.31/docs/GI-GObject-Objects-Object.html#v:objectGetQdata"
        })


#endif

-- method Object::getv
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GObject" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "n_properties"
--           , argType = TBasicType TUInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the number of properties"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "names"
--           , argType = TCArray False (-1) 1 (TBasicType TUTF8)
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the names of each property to get"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "values"
--           , argType = TCArray False (-1) 1 TGValue
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the values of each property to get"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: [ Arg
--              { argCName = "n_properties"
--              , argType = TBasicType TUInt
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "the number of properties"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          , Arg
--              { argCName = "n_properties"
--              , argType = TBasicType TUInt
--              , direction = DirectionIn
--              , mayBeNull = False
--              , argDoc =
--                  Documentation
--                    { rawDocText = Just "the number of properties"
--                    , sinceVersion = Nothing
--                    }
--              , argScope = ScopeTypeInvalid
--              , argClosure = -1
--              , argDestroy = -1
--              , argCallerAllocates = False
--              , argCallbackUserData = False
--              , transfer = TransferNothing
--              }
--          ]
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_object_getv" g_object_getv :: 
    Ptr Object ->                           -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    Word32 ->                               -- n_properties : TBasicType TUInt
    Ptr CString ->                          -- names : TCArray False (-1) 1 (TBasicType TUTF8)
    Ptr B.GValue.GValue ->                  -- values : TCArray False (-1) 1 TGValue
    IO ()

-- | Gets /@nProperties@/ properties for an /@object@/.
-- Obtained properties will be set to /@values@/. All properties must be valid.
-- Warnings will be emitted and undefined behaviour may result if invalid
-- properties are passed in.
-- 
-- /Since: 2.54/
objectGetv ::
    (B.CallStack.HasCallStack, MonadIO m, IsObject a) =>
    a
    -- ^ /@object@/: a t'GI.GObject.Objects.Object.Object'
    -> [T.Text]
    -- ^ /@names@/: the names of each property to get
    -> [GValue]
    -- ^ /@values@/: the values of each property to get
    -> m ()
objectGetv object names values = liftIO $ do
    let nProperties = fromIntegral $ P.length values
    let names_expected_length_ = fromIntegral $ P.length names
    when (names_expected_length_ /= nProperties) $
        error "GObject.objectGetv : length of 'names' does not agree with that of 'values'."
    object' <- unsafeManagedPtrCastPtr object
    names' <- packUTF8CArray names
    values' <- B.GValue.packGValueArray values
    g_object_getv object' nProperties names' values'
    touchManagedPtr object
    mapM_ touchManagedPtr values
    (mapCArrayWithLength nProperties) freeMem names'
    freeMem names'
    freeMem values'
    return ()

#if defined(ENABLE_OVERLOADING)
data ObjectGetvMethodInfo
instance (signature ~ ([T.Text] -> [GValue] -> m ()), MonadIO m, IsObject a) => O.OverloadedMethod ObjectGetvMethodInfo a signature where
    overloadedMethod = objectGetv

instance O.OverloadedMethodInfo ObjectGetvMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GObject.Objects.Object.objectGetv",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gobject-2.0.31/docs/GI-GObject-Objects-Object.html#v:objectGetv"
        })


#endif

-- method Object::is_floating
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GObject" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TBoolean)
-- throws : False
-- Skip return : False

foreign import ccall "g_object_is_floating" g_object_is_floating :: 
    Ptr Object ->                           -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    IO CInt

-- | Checks whether /@object@/ has a [floating][floating-ref] reference.
-- 
-- /Since: 2.10/
objectIsFloating ::
    (B.CallStack.HasCallStack, MonadIO m, IsObject a) =>
    a
    -- ^ /@object@/: a t'GI.GObject.Objects.Object.Object'
    -> m Bool
    -- ^ __Returns:__ 'P.True' if /@object@/ has a floating reference
objectIsFloating object = liftIO $ do
    object' <- unsafeManagedPtrCastPtr object
    result <- g_object_is_floating object'
    let result' = (/= 0) result
    touchManagedPtr object
    return result'

#if defined(ENABLE_OVERLOADING)
data ObjectIsFloatingMethodInfo
instance (signature ~ (m Bool), MonadIO m, IsObject a) => O.OverloadedMethod ObjectIsFloatingMethodInfo a signature where
    overloadedMethod = objectIsFloating

instance O.OverloadedMethodInfo ObjectIsFloatingMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GObject.Objects.Object.objectIsFloating",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gobject-2.0.31/docs/GI-GObject-Objects-Object.html#v:objectIsFloating"
        })


#endif

-- method Object::notify
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GObject" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "property_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "the name of a property installed on the class of @object."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_object_notify" g_object_notify :: 
    Ptr Object ->                           -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    CString ->                              -- property_name : TBasicType TUTF8
    IO ()

-- | Emits a \"notify\" signal for the property /@propertyName@/ on /@object@/.
-- 
-- When possible, eg. when signaling a property change from within the class
-- that registered the property, you should use 'GI.GObject.Objects.Object.objectNotifyByPspec'
-- instead.
-- 
-- Note that emission of the notify signal may be blocked with
-- 'GI.GObject.Objects.Object.objectFreezeNotify'. In this case, the signal emissions are queued
-- and will be emitted (in reverse order) when 'GI.GObject.Objects.Object.objectThawNotify' is
-- called.
objectNotify ::
    (B.CallStack.HasCallStack, MonadIO m, IsObject a) =>
    a
    -- ^ /@object@/: a t'GI.GObject.Objects.Object.Object'
    -> T.Text
    -- ^ /@propertyName@/: the name of a property installed on the class of /@object@/.
    -> m ()
objectNotify object propertyName = liftIO $ do
    object' <- unsafeManagedPtrCastPtr object
    propertyName' <- textToCString propertyName
    g_object_notify object' propertyName'
    touchManagedPtr object
    freeMem propertyName'
    return ()

#if defined(ENABLE_OVERLOADING)
data ObjectNotifyMethodInfo
instance (signature ~ (T.Text -> m ()), MonadIO m, IsObject a) => O.OverloadedMethod ObjectNotifyMethodInfo a signature where
    overloadedMethod = objectNotify

instance O.OverloadedMethodInfo ObjectNotifyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GObject.Objects.Object.objectNotify",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gobject-2.0.31/docs/GI-GObject-Objects-Object.html#v:objectNotify"
        })


#endif

-- method Object::notify_by_pspec
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GObject" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "pspec"
--           , argType = TParamSpec
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "the #GParamSpec of a property installed on the class of @object."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_object_notify_by_pspec" g_object_notify_by_pspec :: 
    Ptr Object ->                           -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    Ptr GParamSpec ->                       -- pspec : TParamSpec
    IO ()

-- | Emits a \"notify\" signal for the property specified by /@pspec@/ on /@object@/.
-- 
-- This function omits the property name lookup, hence it is faster than
-- 'GI.GObject.Objects.Object.objectNotify'.
-- 
-- One way to avoid using 'GI.GObject.Objects.Object.objectNotify' from within the
-- class that registered the properties, and using 'GI.GObject.Objects.Object.objectNotifyByPspec'
-- instead, is to store the GParamSpec used with
-- 'GI.GObject.Structs.ObjectClass.objectClassInstallProperty' inside a static array, e.g.:
-- 
-- 
-- === /C code/
-- >
-- >  typedef enum
-- >  {
-- >    PROP_FOO = 1,
-- >    PROP_LAST
-- >  } MyObjectProperty;
-- >
-- >  static GParamSpec *properties[PROP_LAST];
-- >
-- >  static void
-- >  my_object_class_init (MyObjectClass *klass)
-- >  {
-- >    properties[PROP_FOO] = g_param_spec_int ("foo", NULL, NULL,
-- >                                             0, 100,
-- >                                             50,
-- >                                             G_PARAM_READWRITE | G_PARAM_STATIC_STRINGS);
-- >    g_object_class_install_property (gobject_class,
-- >                                     PROP_FOO,
-- >                                     properties[PROP_FOO]);
-- >  }
-- 
-- 
-- and then notify a change on the \"foo\" property with:
-- 
-- 
-- === /C code/
-- >
-- >  g_object_notify_by_pspec (self, properties[PROP_FOO]);
-- 
-- 
-- /Since: 2.26/
objectNotifyByPspec ::
    (B.CallStack.HasCallStack, MonadIO m, IsObject a) =>
    a
    -- ^ /@object@/: a t'GI.GObject.Objects.Object.Object'
    -> GParamSpec
    -- ^ /@pspec@/: the t'GI.GObject.Objects.ParamSpec.ParamSpec' of a property installed on the class of /@object@/.
    -> m ()
objectNotifyByPspec object pspec = liftIO $ do
    object' <- unsafeManagedPtrCastPtr object
    pspec' <- unsafeManagedPtrGetPtr pspec
    g_object_notify_by_pspec object' pspec'
    touchManagedPtr object
    touchManagedPtr pspec
    return ()

#if defined(ENABLE_OVERLOADING)
data ObjectNotifyByPspecMethodInfo
instance (signature ~ (GParamSpec -> m ()), MonadIO m, IsObject a) => O.OverloadedMethod ObjectNotifyByPspecMethodInfo a signature where
    overloadedMethod = objectNotifyByPspec

instance O.OverloadedMethodInfo ObjectNotifyByPspecMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GObject.Objects.Object.objectNotifyByPspec",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gobject-2.0.31/docs/GI-GObject-Objects-Object.html#v:objectNotifyByPspec"
        })


#endif

-- method Object::ref
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GObject" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GObject" , name = "Object" })
-- throws : False
-- Skip return : False

foreign import ccall "g_object_ref" g_object_ref :: 
    Ptr Object ->                           -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    IO (Ptr Object)

-- | Increases the reference count of /@object@/.
-- 
-- Since GLib 2.56, if @GLIB_VERSION_MAX_ALLOWED@ is 2.56 or greater, the type
-- of /@object@/ will be propagated to the return type (using the GCC @/typeof()/@
-- extension), so any casting the caller needs to do on the return type must be
-- explicit.
objectRef ::
    (B.CallStack.HasCallStack, MonadIO m, IsObject a) =>
    a
    -- ^ /@object@/: a t'GI.GObject.Objects.Object.Object'
    -> m Object
    -- ^ __Returns:__ the same /@object@/
objectRef object = liftIO $ do
    object' <- unsafeManagedPtrCastPtr object
    result <- g_object_ref object'
    checkUnexpectedReturnNULL "objectRef" result
    result' <- (wrapObject Object) result
    touchManagedPtr object
    return result'

#if defined(ENABLE_OVERLOADING)
data ObjectRefMethodInfo
instance (signature ~ (m Object), MonadIO m, IsObject a) => O.OverloadedMethod ObjectRefMethodInfo a signature where
    overloadedMethod = objectRef

instance O.OverloadedMethodInfo ObjectRefMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GObject.Objects.Object.objectRef",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gobject-2.0.31/docs/GI-GObject-Objects-Object.html#v:objectRef"
        })


#endif

-- method Object::ref_sink
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GObject" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "GObject" , name = "Object" })
-- throws : False
-- Skip return : False

foreign import ccall "g_object_ref_sink" g_object_ref_sink :: 
    Ptr Object ->                           -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    IO (Ptr Object)

-- | Increase the reference count of /@object@/, and possibly remove the
-- [floating][floating-ref] reference, if /@object@/ has a floating reference.
-- 
-- In other words, if the object is floating, then this call \"assumes
-- ownership\" of the floating reference, converting it to a normal
-- reference by clearing the floating flag while leaving the reference
-- count unchanged.  If the object is not floating, then this call
-- adds a new normal reference increasing the reference count by one.
-- 
-- Since GLib 2.56, the type of /@object@/ will be propagated to the return type
-- under the same conditions as for 'GI.GObject.Objects.Object.objectRef'.
-- 
-- /Since: 2.10/
objectRefSink ::
    (B.CallStack.HasCallStack, MonadIO m, IsObject a) =>
    a
    -- ^ /@object@/: a t'GI.GObject.Objects.Object.Object'
    -> m Object
    -- ^ __Returns:__ /@object@/
objectRefSink object = liftIO $ do
    object' <- unsafeManagedPtrCastPtr object
    result <- g_object_ref_sink object'
    checkUnexpectedReturnNULL "objectRefSink" result
    result' <- (newObject Object) result
    touchManagedPtr object
    return result'

#if defined(ENABLE_OVERLOADING)
data ObjectRefSinkMethodInfo
instance (signature ~ (m Object), MonadIO m, IsObject a) => O.OverloadedMethod ObjectRefSinkMethodInfo a signature where
    overloadedMethod = objectRefSink

instance O.OverloadedMethodInfo ObjectRefSinkMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GObject.Objects.Object.objectRefSink",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gobject-2.0.31/docs/GI-GObject-Objects-Object.html#v:objectRefSink"
        })


#endif

-- method Object::run_dispose
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GObject" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_object_run_dispose" g_object_run_dispose :: 
    Ptr Object ->                           -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    IO ()

-- | Releases all references to other objects. This can be used to break
-- reference cycles.
-- 
-- This function should only be called from object system implementations.
objectRunDispose ::
    (B.CallStack.HasCallStack, MonadIO m, IsObject a) =>
    a
    -- ^ /@object@/: a t'GI.GObject.Objects.Object.Object'
    -> m ()
objectRunDispose object = liftIO $ do
    object' <- unsafeManagedPtrCastPtr object
    g_object_run_dispose object'
    touchManagedPtr object
    return ()

#if defined(ENABLE_OVERLOADING)
data ObjectRunDisposeMethodInfo
instance (signature ~ (m ()), MonadIO m, IsObject a) => O.OverloadedMethod ObjectRunDisposeMethodInfo a signature where
    overloadedMethod = objectRunDispose

instance O.OverloadedMethodInfo ObjectRunDisposeMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GObject.Objects.Object.objectRunDispose",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gobject-2.0.31/docs/GI-GObject-Objects-Object.html#v:objectRunDispose"
        })


#endif

-- method Object::set_data
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "#GObject containing the associations."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "name of the key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "data to associate with that key"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_object_set_data" g_object_set_data :: 
    Ptr Object ->                           -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    CString ->                              -- key : TBasicType TUTF8
    Ptr () ->                               -- data : TBasicType TPtr
    IO ()

-- | Each object carries around a table of associations from
-- strings to pointers.  This function lets you set an association.
-- 
-- If the object already had an association with that name,
-- the old association will be destroyed.
-- 
-- Internally, the /@key@/ is converted to a @/GQuark/@ using 'GI.GLib.Functions.quarkFromString'.
-- This means a copy of /@key@/ is kept permanently (even after /@object@/ has been
-- finalized) — so it is recommended to only use a small, bounded set of values
-- for /@key@/ in your program, to avoid the @/GQuark/@ storage growing unbounded.
objectSetData ::
    (B.CallStack.HasCallStack, MonadIO m, IsObject a) =>
    a
    -- ^ /@object@/: t'GI.GObject.Objects.Object.Object' containing the associations.
    -> T.Text
    -- ^ /@key@/: name of the key
    -> Ptr ()
    -- ^ /@data@/: data to associate with that key
    -> m ()
objectSetData object key data_ = liftIO $ do
    object' <- unsafeManagedPtrCastPtr object
    key' <- textToCString key
    g_object_set_data object' key' data_
    touchManagedPtr object
    freeMem key'
    return ()

#if defined(ENABLE_OVERLOADING)
data ObjectSetDataMethodInfo
instance (signature ~ (T.Text -> Ptr () -> m ()), MonadIO m, IsObject a) => O.OverloadedMethod ObjectSetDataMethodInfo a signature where
    overloadedMethod = objectSetData

instance O.OverloadedMethodInfo ObjectSetDataMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GObject.Objects.Object.objectSetData",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gobject-2.0.31/docs/GI-GObject-Objects-Object.html#v:objectSetData"
        })


#endif

-- method Object::set_data_full
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "#GObject containing the associations"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "name of the key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "data to associate with that key"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "destroy"
--           , argType =
--               TInterface Name { namespace = "GLib" , name = "DestroyNotify" }
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "function to call when the association is destroyed"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeAsync
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_object_set_data_full" g_object_set_data_full :: 
    Ptr Object ->                           -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    CString ->                              -- key : TBasicType TUTF8
    Ptr () ->                               -- data : TBasicType TPtr
    FunPtr GLib.Callbacks.C_DestroyNotify -> -- destroy : TInterface (Name {namespace = "GLib", name = "DestroyNotify"})
    IO ()

-- | Like 'GI.GObject.Objects.Object.objectSetData' except it adds notification
-- for when the association is destroyed, either by setting it
-- to a different value or when the object is destroyed.
-- 
-- Note that the /@destroy@/ callback is not called if /@data@/ is 'P.Nothing'.
objectSetDataFull ::
    (B.CallStack.HasCallStack, MonadIO m, IsObject a) =>
    a
    -- ^ /@object@/: t'GI.GObject.Objects.Object.Object' containing the associations
    -> T.Text
    -- ^ /@key@/: name of the key
    -> Ptr ()
    -- ^ /@data@/: data to associate with that key
    -> Maybe (GLib.Callbacks.DestroyNotify)
    -- ^ /@destroy@/: function to call when the association is destroyed
    -> m ()
objectSetDataFull object key data_ destroy = liftIO $ do
    object' <- unsafeManagedPtrCastPtr object
    key' <- textToCString key
    maybeDestroy <- case destroy of
        Nothing -> return FP.nullFunPtr
        Just jDestroy -> do
            ptrdestroy <- callocMem :: IO (Ptr (FunPtr GLib.Callbacks.C_DestroyNotify))
            jDestroy' <- GLib.Callbacks.mk_DestroyNotify (GLib.Callbacks.wrap_DestroyNotify (Just ptrdestroy) jDestroy)
            poke ptrdestroy jDestroy'
            return jDestroy'
    g_object_set_data_full object' key' data_ maybeDestroy
    touchManagedPtr object
    freeMem key'
    return ()

#if defined(ENABLE_OVERLOADING)
data ObjectSetDataFullMethodInfo
instance (signature ~ (T.Text -> Ptr () -> Maybe (GLib.Callbacks.DestroyNotify) -> m ()), MonadIO m, IsObject a) => O.OverloadedMethod ObjectSetDataFullMethodInfo a signature where
    overloadedMethod = objectSetDataFull

instance O.OverloadedMethodInfo ObjectSetDataFullMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GObject.Objects.Object.objectSetDataFull",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gobject-2.0.31/docs/GI-GObject-Objects-Object.html#v:objectSetDataFull"
        })


#endif

-- method Object::set_property
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GObject" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "property_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the name of the property to set"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "value"
--           , argType = TGValue
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the value" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_object_set_property" g_object_set_property :: 
    Ptr Object ->                           -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    CString ->                              -- property_name : TBasicType TUTF8
    Ptr GValue ->                           -- value : TGValue
    IO ()

-- | Sets a property on an object.
objectSetProperty ::
    (B.CallStack.HasCallStack, MonadIO m, IsObject a) =>
    a
    -- ^ /@object@/: a t'GI.GObject.Objects.Object.Object'
    -> T.Text
    -- ^ /@propertyName@/: the name of the property to set
    -> GValue
    -- ^ /@value@/: the value
    -> m ()
objectSetProperty object propertyName value = liftIO $ do
    object' <- unsafeManagedPtrCastPtr object
    propertyName' <- textToCString propertyName
    value' <- unsafeManagedPtrGetPtr value
    g_object_set_property object' propertyName' value'
    touchManagedPtr object
    touchManagedPtr value
    freeMem propertyName'
    return ()

#if defined(ENABLE_OVERLOADING)
data ObjectSetPropertyMethodInfo
instance (signature ~ (T.Text -> GValue -> m ()), MonadIO m, IsObject a) => O.OverloadedMethod ObjectSetPropertyMethodInfo a signature where
    overloadedMethod = objectSetProperty

instance O.OverloadedMethodInfo ObjectSetPropertyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GObject.Objects.Object.objectSetProperty",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gobject-2.0.31/docs/GI-GObject-Objects-Object.html#v:objectSetProperty"
        })


#endif

-- method Object::steal_data
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "#GObject containing the associations"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "key"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "name of the key" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TPtr)
-- throws : False
-- Skip return : False

foreign import ccall "g_object_steal_data" g_object_steal_data :: 
    Ptr Object ->                           -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    CString ->                              -- key : TBasicType TUTF8
    IO (Ptr ())

-- | Remove a specified datum from the object\'s data associations,
-- without invoking the association\'s destroy handler.
objectStealData ::
    (B.CallStack.HasCallStack, MonadIO m, IsObject a) =>
    a
    -- ^ /@object@/: t'GI.GObject.Objects.Object.Object' containing the associations
    -> T.Text
    -- ^ /@key@/: name of the key
    -> m (Ptr ())
    -- ^ __Returns:__ the data if found, or 'P.Nothing'
    --          if no such data exists.
objectStealData object key = liftIO $ do
    object' <- unsafeManagedPtrCastPtr object
    key' <- textToCString key
    result <- g_object_steal_data object' key'
    touchManagedPtr object
    freeMem key'
    return result

#if defined(ENABLE_OVERLOADING)
data ObjectStealDataMethodInfo
instance (signature ~ (T.Text -> m (Ptr ())), MonadIO m, IsObject a) => O.OverloadedMethod ObjectStealDataMethodInfo a signature where
    overloadedMethod = objectStealData

instance O.OverloadedMethodInfo ObjectStealDataMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GObject.Objects.Object.objectStealData",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gobject-2.0.31/docs/GI-GObject-Objects-Object.html#v:objectStealData"
        })


#endif

-- method Object::steal_qdata
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just "The GObject to get a stored user data pointer from"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "quark"
--           , argType = TBasicType TUInt32
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "A #GQuark, naming the user data pointer"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TPtr)
-- throws : False
-- Skip return : False

foreign import ccall "g_object_steal_qdata" g_object_steal_qdata :: 
    Ptr Object ->                           -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    Word32 ->                               -- quark : TBasicType TUInt32
    IO (Ptr ())

-- | This function gets back user data pointers stored via
-- @/g_object_set_qdata()/@ and removes the /@data@/ from object
-- without invoking its @/destroy()/@ function (if any was
-- set).
-- Usually, calling this function is only required to update
-- user data pointers with a destroy notifier, for example:
-- 
-- === /C code/
-- >
-- >void
-- >object_add_to_user_list (GObject     *object,
-- >                         const gchar *new_string)
-- >{
-- >  // the quark, naming the object data
-- >  GQuark quark_string_list = g_quark_from_static_string ("my-string-list");
-- >  // retrieve the old string list
-- >  GList *list = g_object_steal_qdata (object, quark_string_list);
-- >
-- >  // prepend new string
-- >  list = g_list_prepend (list, g_strdup (new_string));
-- >  // this changed 'list', so we need to set it again
-- >  g_object_set_qdata_full (object, quark_string_list, list, free_string_list);
-- >}
-- >static void
-- >free_string_list (gpointer data)
-- >{
-- >  GList *node, *list = data;
-- >
-- >  for (node = list; node; node = node->next)
-- >    g_free (node->data);
-- >  g_list_free (list);
-- >}
-- 
-- Using 'GI.GObject.Objects.Object.objectGetQdata' in the above example, instead of
-- 'GI.GObject.Objects.Object.objectStealQdata' would have left the destroy function set,
-- and thus the partial string list would have been freed upon
-- @/g_object_set_qdata_full()/@.
objectStealQdata ::
    (B.CallStack.HasCallStack, MonadIO m, IsObject a) =>
    a
    -- ^ /@object@/: The GObject to get a stored user data pointer from
    -> Word32
    -- ^ /@quark@/: A @/GQuark/@, naming the user data pointer
    -> m (Ptr ())
    -- ^ __Returns:__ The user data pointer set, or 'P.Nothing'
objectStealQdata object quark = liftIO $ do
    object' <- unsafeManagedPtrCastPtr object
    result <- g_object_steal_qdata object' quark
    touchManagedPtr object
    return result

#if defined(ENABLE_OVERLOADING)
data ObjectStealQdataMethodInfo
instance (signature ~ (Word32 -> m (Ptr ())), MonadIO m, IsObject a) => O.OverloadedMethod ObjectStealQdataMethodInfo a signature where
    overloadedMethod = objectStealQdata

instance O.OverloadedMethodInfo ObjectStealQdataMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GObject.Objects.Object.objectStealQdata",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gobject-2.0.31/docs/GI-GObject-Objects-Object.html#v:objectStealQdata"
        })


#endif

-- method Object::thaw_notify
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a #GObject" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_object_thaw_notify" g_object_thaw_notify :: 
    Ptr Object ->                           -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    IO ()

-- | Reverts the effect of a previous call to
-- 'GI.GObject.Objects.Object.objectFreezeNotify'. The freeze count is decreased on /@object@/
-- and when it reaches zero, queued \"notify\" signals are emitted.
-- 
-- Duplicate notifications for each property are squashed so that at most one
-- [Object::notify]("GI.GObject.Objects.Object#g:signal:notify") signal is emitted for each property, in the reverse order
-- in which they have been queued.
-- 
-- It is an error to call this function when the freeze count is zero.
objectThawNotify ::
    (B.CallStack.HasCallStack, MonadIO m, IsObject a) =>
    a
    -- ^ /@object@/: a t'GI.GObject.Objects.Object.Object'
    -> m ()
objectThawNotify object = liftIO $ do
    object' <- unsafeManagedPtrCastPtr object
    g_object_thaw_notify object'
    touchManagedPtr object
    return ()

#if defined(ENABLE_OVERLOADING)
data ObjectThawNotifyMethodInfo
instance (signature ~ (m ()), MonadIO m, IsObject a) => O.OverloadedMethod ObjectThawNotifyMethodInfo a signature where
    overloadedMethod = objectThawNotify

instance O.OverloadedMethodInfo ObjectThawNotifyMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GObject.Objects.Object.objectThawNotify",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gobject-2.0.31/docs/GI-GObject-Objects-Object.html#v:objectThawNotify"
        })


#endif

-- method Object::unref
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just ": a #GObject" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferEverything
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_object_unref" g_object_unref :: 
    Ptr Object ->                           -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    IO ()

-- | Decreases the reference count of /@object@/. When its reference count
-- drops to 0, the object is finalized (i.e. its memory is freed).
-- 
-- If the pointer to the t'GI.GObject.Objects.Object.Object' may be reused in future (for example, if it is
-- an instance variable of another object), it is recommended to clear the
-- pointer to 'P.Nothing' rather than retain a dangling pointer to a potentially
-- invalid t'GI.GObject.Objects.Object.Object' instance. Use @/g_clear_object()/@ for this.
objectUnref ::
    (B.CallStack.HasCallStack, MonadIO m, IsObject a) =>
    a
    -- ^ /@object@/: : a t'GI.GObject.Objects.Object.Object'
    -> m ()
objectUnref object = liftIO $ do
    object' <- B.ManagedPtr.disownObject object
    g_object_unref object'
    touchManagedPtr object
    return ()

#if defined(ENABLE_OVERLOADING)
data ObjectUnrefMethodInfo
instance (signature ~ (m ()), MonadIO m, IsObject a) => O.OverloadedMethod ObjectUnrefMethodInfo a signature where
    overloadedMethod = objectUnref

instance O.OverloadedMethodInfo ObjectUnrefMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GObject.Objects.Object.objectUnref",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gobject-2.0.31/docs/GI-GObject-Objects-Object.html#v:objectUnref"
        })


#endif

-- method Object::watch_closure
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "object"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "Object" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "#GObject restricting lifetime of @closure"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "closure"
--           , argType = TGClosure Nothing
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "#GClosure to watch" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_object_watch_closure" g_object_watch_closure :: 
    Ptr Object ->                           -- object : TInterface (Name {namespace = "GObject", name = "Object"})
    Ptr (GClosure ()) ->                    -- closure : TGClosure Nothing
    IO ()

-- | This function essentially limits the life time of the /@closure@/ to
-- the life time of the object. That is, when the object is finalized,
-- the /@closure@/ is invalidated by calling 'GI.GObject.Structs.Closure.closureInvalidate' on
-- it, in order to prevent invocations of the closure with a finalized
-- (nonexisting) object. Also, 'GI.GObject.Objects.Object.objectRef' and 'GI.GObject.Objects.Object.objectUnref' are
-- added as marshal guards to the /@closure@/, to ensure that an extra
-- reference count is held on /@object@/ during invocation of the
-- /@closure@/.  Usually, this function will be called on closures that
-- use this /@object@/ as closure data.
objectWatchClosure ::
    (B.CallStack.HasCallStack, MonadIO m, IsObject a) =>
    a
    -- ^ /@object@/: t'GI.GObject.Objects.Object.Object' restricting lifetime of /@closure@/
    -> GClosure b
    -- ^ /@closure@/: t'GI.GObject.Structs.Closure.Closure' to watch
    -> m ()
objectWatchClosure object closure = liftIO $ do
    object' <- unsafeManagedPtrCastPtr object
    closure' <- unsafeManagedPtrCastPtr closure
    g_object_watch_closure object' closure'
    touchManagedPtr object
    touchManagedPtr closure
    return ()

#if defined(ENABLE_OVERLOADING)
data ObjectWatchClosureMethodInfo
instance (signature ~ (GClosure b -> m ()), MonadIO m, IsObject a) => O.OverloadedMethod ObjectWatchClosureMethodInfo a signature where
    overloadedMethod = objectWatchClosure

instance O.OverloadedMethodInfo ObjectWatchClosureMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.GObject.Objects.Object.objectWatchClosure",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gobject-2.0.31/docs/GI-GObject-Objects-Object.html#v:objectWatchClosure"
        })


#endif

-- method Object::compat_control
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "what"
--           , argType = TBasicType TSize
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation { rawDocText = Nothing , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "data"
--           , argType = TBasicType TPtr
--           , direction = DirectionIn
--           , mayBeNull = True
--           , argDoc =
--               Documentation { rawDocText = Nothing , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TSize)
-- throws : False
-- Skip return : False

foreign import ccall "g_object_compat_control" g_object_compat_control :: 
    FCT.CSize ->                            -- what : TBasicType TSize
    Ptr () ->                               -- data : TBasicType TPtr
    IO FCT.CSize

-- | /No description available in the introspection data./
objectCompatControl ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    FCT.CSize
    -> Ptr ()
    -> m FCT.CSize
objectCompatControl what data_ = liftIO $ do
    result <- g_object_compat_control what data_
    return result

#if defined(ENABLE_OVERLOADING)
#endif

-- method Object::interface_find_property
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "g_iface"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "TypeInterface" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "any interface vtable for the\n interface, or the default vtable for the interface"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "property_name"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "name of a property to look up."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just TParamSpec
-- throws : False
-- Skip return : False

foreign import ccall "g_object_interface_find_property" g_object_interface_find_property :: 
    Ptr GObject.TypeInterface.TypeInterface -> -- g_iface : TInterface (Name {namespace = "GObject", name = "TypeInterface"})
    CString ->                              -- property_name : TBasicType TUTF8
    IO (Ptr GParamSpec)

-- | Find the t'GI.GObject.Objects.ParamSpec.ParamSpec' with the given name for an
-- interface. Generally, the interface vtable passed in as /@gIface@/
-- will be the default vtable from @/g_type_default_interface_ref()/@, or,
-- if you know the interface has already been loaded,
-- 'GI.GObject.Functions.typeDefaultInterfacePeek'.
-- 
-- /Since: 2.4/
objectInterfaceFindProperty ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    GObject.TypeInterface.TypeInterface
    -- ^ /@gIface@/: any interface vtable for the
    --  interface, or the default vtable for the interface
    -> T.Text
    -- ^ /@propertyName@/: name of a property to look up.
    -> m GParamSpec
    -- ^ __Returns:__ the t'GI.GObject.Objects.ParamSpec.ParamSpec' for the property of the
    --          interface with the name /@propertyName@/, or 'P.Nothing' if no
    --          such property exists.
objectInterfaceFindProperty gIface propertyName = liftIO $ do
    gIface' <- unsafeManagedPtrGetPtr gIface
    propertyName' <- textToCString propertyName
    result <- g_object_interface_find_property gIface' propertyName'
    checkUnexpectedReturnNULL "objectInterfaceFindProperty" result
    result' <- B.GParamSpec.newGParamSpecFromPtr result
    touchManagedPtr gIface
    freeMem propertyName'
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Object::interface_install_property
-- method type : MemberFunction
-- Args: [ Arg
--           { argCName = "g_iface"
--           , argType =
--               TInterface Name { namespace = "GObject" , name = "TypeInterface" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText =
--                     Just
--                       "any interface vtable for the\n   interface, or the default\n vtable for the interface."
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "pspec"
--           , argType = TParamSpec
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the #GParamSpec for the new property"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "g_object_interface_install_property" g_object_interface_install_property :: 
    Ptr GObject.TypeInterface.TypeInterface -> -- g_iface : TInterface (Name {namespace = "GObject", name = "TypeInterface"})
    Ptr GParamSpec ->                       -- pspec : TParamSpec
    IO ()

-- | Add a property to an interface; this is only useful for interfaces
-- that are added to GObject-derived types. Adding a property to an
-- interface forces all objects classes with that interface to have a
-- compatible property. The compatible property could be a newly
-- created t'GI.GObject.Objects.ParamSpec.ParamSpec', but normally
-- 'GI.GObject.Structs.ObjectClass.objectClassOverrideProperty' will be used so that the object
-- class only needs to provide an implementation and inherits the
-- property description, default value, bounds, and so forth from the
-- interface property.
-- 
-- This function is meant to be called from the interface\'s default
-- vtable initialization function (the /@classInit@/ member of
-- t'GI.GObject.Structs.TypeInfo.TypeInfo'.) It must not be called after after /@classInit@/ has
-- been called for any object types implementing this interface.
-- 
-- If /@pspec@/ is a floating reference, it will be consumed.
-- 
-- /Since: 2.4/
objectInterfaceInstallProperty ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    GObject.TypeInterface.TypeInterface
    -- ^ /@gIface@/: any interface vtable for the
    --    interface, or the default
    --  vtable for the interface.
    -> GParamSpec
    -- ^ /@pspec@/: the t'GI.GObject.Objects.ParamSpec.ParamSpec' for the new property
    -> m ()
objectInterfaceInstallProperty gIface pspec = liftIO $ do
    gIface' <- unsafeManagedPtrGetPtr gIface
    pspec' <- unsafeManagedPtrGetPtr pspec
    g_object_interface_install_property gIface' pspec'
    touchManagedPtr gIface
    touchManagedPtr pspec
    return ()

#if defined(ENABLE_OVERLOADING)
#endif

-- XXX Could not generate method Object::interface_list_properties
-- Not implemented: unpackCArray : Don't know how to unpack C Array of type TParamSpec
#if defined(ENABLE_OVERLOADING)
-- XXX: Dummy instance, since code generation failed.
-- Please file a bug at http://github.com/haskell-gi/haskell-gi.
data ObjectInterfaceListPropertiesMethodInfo
instance (p ~ (), o ~ O.UnsupportedMethodError "interfaceListProperties" Object) => O.OverloadedMethod ObjectInterfaceListPropertiesMethodInfo o p where
    overloadedMethod = undefined

instance (o ~ O.UnsupportedMethodError "interfaceListProperties" Object) => O.OverloadedMethodInfo ObjectInterfaceListPropertiesMethodInfo o where
    overloadedMethodInfo = undefined

#endif


