/*
 * Decompiled with CFR 0.152.
 */
package net.snowflake.client.core.crl;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.nio.file.attribute.FileAttribute;
import java.nio.file.attribute.FileTime;
import java.nio.file.attribute.PosixFilePermissions;
import java.security.cert.CRLException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509CRL;
import java.time.Duration;
import java.time.Instant;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import net.snowflake.client.core.Constants;
import net.snowflake.client.core.crl.CRLCache;
import net.snowflake.client.core.crl.CRLCacheEntry;
import net.snowflake.client.jdbc.SnowflakeSQLLoggedException;
import net.snowflake.client.log.SFLogger;
import net.snowflake.client.log.SFLoggerFactory;

class CRLFileCache
implements CRLCache {
    private static final SFLogger logger = SFLoggerFactory.getLogger(CRLFileCache.class);
    private final Path cacheDir;
    private final Duration removalDelay;
    private final Lock cacheLock = new ReentrantLock();

    CRLFileCache(Path cacheDir, Duration removalDelay) throws SnowflakeSQLLoggedException {
        this.cacheDir = cacheDir;
        this.removalDelay = removalDelay;
        CRLFileCache.ensureCacheDirectoryExists(cacheDir);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public CRLCacheEntry get(String crlUrl) {
        block11: {
            try {
                CRLCacheEntry cRLCacheEntry;
                block12: {
                    this.cacheLock.lock();
                    Path crlFilePath = this.getCrlFilePath(crlUrl);
                    if (!Files.exists(crlFilePath, new LinkOption[0])) break block11;
                    logger.debug("Found CRL on disk for {}", crlFilePath);
                    BasicFileAttributes attrs = Files.readAttributes(crlFilePath, BasicFileAttributes.class, new LinkOption[0]);
                    Instant downloadTime = attrs.lastModifiedTime().toInstant();
                    CertificateFactory certFactory = CertificateFactory.getInstance("X.509");
                    InputStream crlBytes = Files.newInputStream(crlFilePath, new OpenOption[0]);
                    try {
                        X509CRL crl = (X509CRL)certFactory.generateCRL(crlBytes);
                        cRLCacheEntry = new CRLCacheEntry(crl, downloadTime);
                        if (crlBytes == null) break block12;
                    }
                    catch (Throwable throwable) {
                        try {
                            if (crlBytes != null) {
                                try {
                                    crlBytes.close();
                                }
                                catch (Throwable throwable2) {
                                    throwable.addSuppressed(throwable2);
                                }
                            }
                            throw throwable;
                        }
                        catch (Exception e) {
                            logger.warn("Failed to read CRL from disk cache for {}: {}", crlUrl, e.getMessage());
                            break block11;
                        }
                        catch (Throwable throwable3) {
                            throw throwable3;
                        }
                    }
                    crlBytes.close();
                }
                return cRLCacheEntry;
            }
            finally {
                this.cacheLock.unlock();
            }
        }
        return null;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void put(String crlUrl, CRLCacheEntry entry) {
        try {
            this.cacheLock.lock();
            Path crlFilePath = this.getCrlFilePath(crlUrl);
            Files.write(crlFilePath, entry.getCrl().getEncoded(), StandardOpenOption.CREATE, StandardOpenOption.WRITE, StandardOpenOption.TRUNCATE_EXISTING);
            Files.setLastModifiedTime(crlFilePath, FileTime.from(entry.getDownloadTime()));
            if (Constants.getOS() != Constants.OS.WINDOWS) {
                Files.setPosixFilePermissions(crlFilePath, PosixFilePermissions.fromString("rw-------"));
            }
            logger.debug("Updated disk cache for {}", crlUrl);
        }
        catch (Exception e) {
            logger.warn("Failed to write CRL to disk cache for {}: {}", crlUrl, e.getMessage());
        }
        finally {
            this.cacheLock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void cleanup() {
        Instant now = Instant.now();
        logger.debug("Cleaning up on-disk CRL cache at {}", now);
        try {
            if (!Files.exists(this.cacheDir, new LinkOption[0])) {
                return;
            }
            int removedCount = 0;
            try (Stream<Path> files = Files.list(this.cacheDir);){
                this.cacheLock.lock();
                for (Path filePath : files.filter(x$0 -> Files.isRegularFile(x$0, new LinkOption[0])).collect(Collectors.toList())) {
                    try {
                        InputStream crlBytes = Files.newInputStream(filePath, new OpenOption[0]);
                        try {
                            CertificateFactory certFactory = CertificateFactory.getInstance("X.509");
                            X509CRL crl = (X509CRL)certFactory.generateCRL(crlBytes);
                            CRLCacheEntry entry = new CRLCacheEntry(crl, Files.getLastModifiedTime(filePath, new LinkOption[0]).toInstant());
                            boolean expired = entry.isCrlExpired(now);
                            boolean evicted = entry.isEvicted(now, this.removalDelay);
                            if (!expired && !evicted) continue;
                            Files.delete(filePath);
                            ++removedCount;
                            logger.debug("Removing file based CRL cache entry for {}: expired={}, evicted={}", filePath, expired, evicted);
                        }
                        finally {
                            if (crlBytes == null) continue;
                            crlBytes.close();
                        }
                    }
                    catch (IOException | CRLException | CertificateException e) {
                        try {
                            Files.delete(filePath);
                            ++removedCount;
                        }
                        catch (IOException deleteError) {
                            logger.warn("Failed to delete corrupted CRL file {}: {}", filePath, deleteError.getMessage());
                        }
                    }
                }
            }
            finally {
                this.cacheLock.unlock();
            }
            if (removedCount > 0) {
                logger.debug("Removed {} expired/corrupted files from disk CRL cache", removedCount);
            }
        }
        catch (Exception e) {
            logger.warn("Failed to cleanup disk CRL cache: {}", e.getMessage());
        }
    }

    private Path getCrlFilePath(String crlUrl) throws UnsupportedEncodingException {
        String encodedUrl = URLEncoder.encode(crlUrl, StandardCharsets.UTF_8.toString());
        return this.cacheDir.resolve(encodedUrl);
    }

    private static boolean ownerOnlyPermissions(Path cacheDir) throws IOException {
        return Files.getPosixFilePermissions(cacheDir, new LinkOption[0]).equals(PosixFilePermissions.fromString("rwx------"));
    }

    private static void ensureCacheDirectoryExists(Path cacheDir) throws SnowflakeSQLLoggedException {
        try {
            boolean exists = Files.exists(cacheDir, new LinkOption[0]);
            if (!exists) {
                Files.createDirectories(cacheDir, new FileAttribute[0]);
                logger.debug("Initialized CRL cache directory: {}", cacheDir);
            }
            if (Constants.getOS() != Constants.OS.WINDOWS && !CRLFileCache.ownerOnlyPermissions(cacheDir)) {
                Files.setPosixFilePermissions(cacheDir, PosixFilePermissions.fromString("rwx------"));
                logger.debug("Set CRL cache directory permissions to 'rwx------", new Object[0]);
            }
        }
        catch (Exception e) {
            throw new SnowflakeSQLLoggedException(null, null, "XX000", "Failed to create CRL cache directory", e);
        }
    }
}

