/*
 * Decompiled with CFR 0.152.
 */
package org.mariadb.jdbc.pool;

import java.lang.management.ManagementFactory;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.text.NumberFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.LinkedBlockingDeque;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.ScheduledThreadPoolExecutor;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import javax.management.MBeanServer;
import javax.management.ObjectName;
import javax.sql.ConnectionEvent;
import javax.sql.ConnectionEventListener;
import org.mariadb.jdbc.Configuration;
import org.mariadb.jdbc.Connection;
import org.mariadb.jdbc.Driver;
import org.mariadb.jdbc.Statement;
import org.mariadb.jdbc.pool.MariaDbInnerPoolConnection;
import org.mariadb.jdbc.pool.PoolMBean;
import org.mariadb.jdbc.pool.PoolThreadFactory;
import org.mariadb.jdbc.pool.Pools;
import org.mariadb.jdbc.util.log.Logger;
import org.mariadb.jdbc.util.log.Loggers;

public class Pool
implements AutoCloseable,
PoolMBean {
    private static final Logger logger = Loggers.getLogger(Pool.class);
    private static final int POOL_STATE_OK = 0;
    private static final int POOL_STATE_CLOSING = 1;
    private final AtomicInteger poolState;
    private final Configuration conf;
    private final AtomicInteger pendingRequestNumber;
    private final AtomicInteger totalConnection;
    private final LinkedBlockingDeque<MariaDbInnerPoolConnection> idleConnections;
    private final ThreadPoolExecutor connectionAppender;
    private final BlockingQueue<Runnable> connectionAppenderQueue;
    private final String poolTag;
    private final ScheduledThreadPoolExecutor poolExecutor;
    private final ScheduledFuture<?> scheduledFuture;
    private int waitTimeout;

    public Pool(Configuration conf, int poolIndex, ScheduledThreadPoolExecutor poolExecutor) {
        block13: {
            this.poolState = new AtomicInteger();
            this.pendingRequestNumber = new AtomicInteger();
            this.totalConnection = new AtomicInteger();
            this.conf = conf;
            this.poolTag = this.generatePoolTag(poolIndex);
            this.connectionAppenderQueue = new ArrayBlockingQueue<Runnable>(conf.maxPoolSize());
            this.connectionAppender = new ThreadPoolExecutor(1, 1, 10L, TimeUnit.SECONDS, this.connectionAppenderQueue, new PoolThreadFactory(this.poolTag + "-appender"));
            this.connectionAppender.allowCoreThreadTimeOut(true);
            this.connectionAppender.prestartCoreThread();
            this.idleConnections = new LinkedBlockingDeque();
            int minDelay = Integer.parseInt(conf.nonMappedOptions().getProperty("testMinRemovalDelay", "30"));
            int scheduleDelay = Math.min(minDelay, conf.maxIdleTime() / 2);
            this.poolExecutor = poolExecutor;
            this.scheduledFuture = poolExecutor.scheduleAtFixedRate(this::removeIdleTimeoutConnection, scheduleDelay, scheduleDelay, TimeUnit.SECONDS);
            if (conf.registerJmxPool()) {
                try {
                    this.registerJmx();
                }
                catch (Exception ex) {
                    logger.error("pool " + this.poolTag + " not registered due to exception : " + ex.getMessage());
                }
            }
            try {
                for (int i = 0; i < Math.max(1, conf.minPoolSize()); ++i) {
                    this.addConnection();
                }
                this.waitTimeout = 28800;
                if (this.idleConnections.isEmpty()) break block13;
                try (Statement stmt = this.idleConnections.getFirst().getConnection().createStatement();){
                    ResultSet rs = stmt.executeQuery("SELECT @@wait_timeout");
                    if (rs.next()) {
                        this.waitTimeout = rs.getInt(1);
                    }
                }
            }
            catch (SQLException sqle) {
                logger.error("error initializing pool connection", sqle);
            }
        }
    }

    private void addConnectionRequest() {
        if (this.totalConnection.get() < this.conf.maxPoolSize() && this.poolState.get() == 0) {
            this.connectionAppender.prestartCoreThread();
            this.connectionAppenderQueue.offer(() -> {
                if ((this.totalConnection.get() < this.conf.minPoolSize() || this.pendingRequestNumber.get() > 0) && this.totalConnection.get() < this.conf.maxPoolSize()) {
                    try {
                        this.addConnection();
                    }
                    catch (SQLException sqle) {
                        logger.error("error adding connection to pool", sqle);
                    }
                }
            });
        }
    }

    private void removeIdleTimeoutConnection() {
        Iterator<MariaDbInnerPoolConnection> iterator = this.idleConnections.descendingIterator();
        while (iterator.hasNext()) {
            MariaDbInnerPoolConnection item = iterator.next();
            long idleTime = System.nanoTime() - item.getLastUsed().get();
            boolean timedOut = idleTime > TimeUnit.SECONDS.toNanos(this.conf.maxIdleTime());
            boolean shouldBeReleased = false;
            Connection con = item.getConnection();
            if (this.waitTimeout > 0) {
                if (idleTime > TimeUnit.SECONDS.toNanos(this.waitTimeout - 45)) {
                    shouldBeReleased = true;
                }
                if (timedOut && this.totalConnection.get() > this.conf.minPoolSize()) {
                    shouldBeReleased = true;
                }
            } else if (timedOut) {
                shouldBeReleased = true;
            }
            if (!shouldBeReleased || !this.idleConnections.remove(item)) continue;
            this.totalConnection.decrementAndGet();
            this.silentCloseConnection(con);
            this.addConnectionRequest();
            if (!logger.isDebugEnabled()) continue;
            logger.debug("pool {} connection {} removed due to inactivity (total:{}, active:{}, pending:{})", this.poolTag, con.getThreadId(), this.totalConnection.get(), this.getActiveConnections(), this.pendingRequestNumber.get());
        }
    }

    private void addConnection() throws SQLException {
        Connection connection = Driver.connect(this.conf);
        MariaDbInnerPoolConnection item = new MariaDbInnerPoolConnection(connection);
        item.addConnectionEventListener(new ConnectionEventListener(){

            @Override
            public void connectionClosed(ConnectionEvent event) {
                MariaDbInnerPoolConnection item = (MariaDbInnerPoolConnection)event.getSource();
                if (Pool.this.poolState.get() == 0) {
                    try {
                        if (!Pool.this.idleConnections.contains(item)) {
                            item.getConnection().setPoolConnection(null);
                            item.getConnection().reset();
                            Pool.this.idleConnections.addFirst(item);
                            item.getConnection().setPoolConnection(item);
                        }
                    }
                    catch (SQLException sqle) {
                        Pool.this.totalConnection.decrementAndGet();
                        Pool.this.silentCloseConnection(item.getConnection());
                        logger.debug("connection {} removed from pool {} due to error during reset (total:{}, active:{}, pending:{})", item.getConnection().getThreadId(), Pool.this.poolTag, Pool.this.totalConnection.get(), Pool.this.getActiveConnections(), Pool.this.pendingRequestNumber.get());
                    }
                } else {
                    try {
                        item.getConnection().close();
                    }
                    catch (SQLException sQLException) {
                        // empty catch block
                    }
                    Pool.this.totalConnection.decrementAndGet();
                }
            }

            @Override
            public void connectionErrorOccurred(ConnectionEvent event) {
                MariaDbInnerPoolConnection item = (MariaDbInnerPoolConnection)event.getSource();
                Pool.this.totalConnection.decrementAndGet();
                Pool.this.idleConnections.remove(item);
                Pool.this.idleConnections.forEach(MariaDbInnerPoolConnection::ensureValidation);
                Pool.this.silentCloseConnection(item.getConnection());
                Pool.this.addConnectionRequest();
                logger.debug("connection {} removed from pool {} due to having throw a Connection exception (total:{}, active:{}, pending:{})", item.getConnection().getThreadId(), Pool.this.poolTag, Pool.this.totalConnection.get(), Pool.this.getActiveConnections(), Pool.this.pendingRequestNumber.get());
            }
        });
        if (this.poolState.get() == 0 && this.totalConnection.incrementAndGet() <= this.conf.maxPoolSize()) {
            this.idleConnections.addFirst(item);
            if (logger.isDebugEnabled()) {
                logger.debug("pool {} new physical connection {} created (total:{}, active:{}, pending:{})", this.poolTag, connection.getThreadId(), this.totalConnection.get(), this.getActiveConnections(), this.pendingRequestNumber.get());
            }
            return;
        }
        this.silentCloseConnection(connection);
    }

    private MariaDbInnerPoolConnection getIdleConnection(long timeout, TimeUnit timeUnit) throws InterruptedException {
        while (true) {
            MariaDbInnerPoolConnection item;
            block6: {
                MariaDbInnerPoolConnection mariaDbInnerPoolConnection = item = timeout == 0L ? this.idleConnections.pollFirst() : this.idleConnections.pollFirst(timeout, timeUnit);
                if (item == null) {
                    return null;
                }
                try {
                    if (TimeUnit.NANOSECONDS.toMillis(System.nanoTime() - item.getLastUsed().get()) > (long)this.conf.poolValidMinDelay()) {
                        if (item.getConnection().isValid(10)) {
                            item.lastUsedToNow();
                            return item;
                        }
                        break block6;
                    }
                    item.lastUsedToNow();
                    return item;
                }
                catch (SQLException sQLException) {
                    // empty catch block
                }
            }
            this.silentAbortConnection(item.getConnection());
            this.addConnectionRequest();
            if (!logger.isDebugEnabled()) continue;
            logger.debug("pool {} connection {} removed from pool due to failed validation (total:{}, active:{}, pending:{})", this.poolTag, item.getConnection().getThreadId(), this.totalConnection.get(), this.getActiveConnections(), this.pendingRequestNumber.get());
        }
    }

    private void silentCloseConnection(Connection con) {
        con.setPoolConnection(null);
        try {
            con.close();
        }
        catch (SQLException sQLException) {
            // empty catch block
        }
    }

    private void silentAbortConnection(Connection con) {
        con.setPoolConnection(null);
        try {
            con.abort(this.poolExecutor);
        }
        catch (SQLException sQLException) {
            // empty catch block
        }
    }

    public MariaDbInnerPoolConnection getPoolConnection() throws SQLException {
        this.pendingRequestNumber.incrementAndGet();
        try {
            MariaDbInnerPoolConnection poolConnection = this.getIdleConnection(this.totalConnection.get() > 4 ? 0L : 50L, TimeUnit.MICROSECONDS);
            if (poolConnection != null) {
                MariaDbInnerPoolConnection mariaDbInnerPoolConnection = poolConnection;
                return mariaDbInnerPoolConnection;
            }
            this.addConnectionRequest();
            poolConnection = this.getIdleConnection(TimeUnit.MILLISECONDS.toNanos(this.conf.connectTimeout()), TimeUnit.NANOSECONDS);
            if (poolConnection != null) {
                MariaDbInnerPoolConnection mariaDbInnerPoolConnection = poolConnection;
                return mariaDbInnerPoolConnection;
            }
            try {
                throw new SQLException(String.format("No connection available within the specified time (option 'connectTimeout': %s ms)", NumberFormat.getInstance().format(this.conf.connectTimeout())));
            }
            catch (InterruptedException interrupted) {
                throw new SQLException("Thread was interrupted", "70100", interrupted);
            }
        }
        finally {
            this.pendingRequestNumber.decrementAndGet();
        }
    }

    public MariaDbInnerPoolConnection getPoolConnection(String username, String password) throws SQLException {
        if (username == null ? this.conf.user() == null : (username.equals(this.conf.user()) && (password == null || password.isEmpty()) ? this.conf.password() == null : password.equals(this.conf.password()))) {
            return this.getPoolConnection();
        }
        Configuration tmpConf = this.conf.clone(username, password);
        return new MariaDbInnerPoolConnection(Driver.connect(tmpConf));
    }

    private String generatePoolTag(int poolIndex) {
        if (this.conf.poolName() == null) {
            return "MariaDB-pool";
        }
        return this.conf.poolName() + "-" + poolIndex;
    }

    public Configuration getConf() {
        return this.conf;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void close() {
        try {
            Pool pool = this;
            synchronized (pool) {
                Pools.remove(this);
                this.poolState.set(1);
                this.pendingRequestNumber.set(0);
                this.scheduledFuture.cancel(false);
                this.connectionAppender.shutdown();
                try {
                    this.connectionAppender.awaitTermination(10L, TimeUnit.SECONDS);
                }
                catch (InterruptedException interruptedException) {
                    // empty catch block
                }
                if (logger.isInfoEnabled()) {
                    logger.debug("closing pool {} (total:{}, active:{}, pending:{})", this.poolTag, this.totalConnection.get(), this.getActiveConnections(), this.pendingRequestNumber.get());
                }
                ThreadPoolExecutor connectionRemover = new ThreadPoolExecutor(this.totalConnection.get(), this.conf.maxPoolSize(), 10L, TimeUnit.SECONDS, new LinkedBlockingQueue<Runnable>(this.conf.maxPoolSize()), new PoolThreadFactory(this.poolTag + "-destroyer"));
                long start = System.nanoTime();
                do {
                    this.closeAll(this.idleConnections);
                    if (this.totalConnection.get() <= 0) continue;
                    Thread.sleep(0L, 1000);
                } while (this.totalConnection.get() > 0 && TimeUnit.NANOSECONDS.toSeconds(System.nanoTime() - start) < 10L);
                if (this.totalConnection.get() > 0 || this.idleConnections.isEmpty()) {
                    this.closeAll(this.idleConnections);
                }
                connectionRemover.shutdown();
                try {
                    this.unRegisterJmx();
                }
                catch (Exception exception) {
                    // empty catch block
                }
                connectionRemover.awaitTermination(10L, TimeUnit.SECONDS);
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void closeAll(Collection<MariaDbInnerPoolConnection> collection) {
        Collection<MariaDbInnerPoolConnection> collection2 = collection;
        synchronized (collection2) {
            for (MariaDbInnerPoolConnection item : collection) {
                collection.remove(item);
                this.totalConnection.decrementAndGet();
                this.silentAbortConnection(item.getConnection());
            }
        }
    }

    public String getPoolTag() {
        return this.poolTag;
    }

    @Override
    public long getActiveConnections() {
        return this.totalConnection.get() - this.idleConnections.size();
    }

    @Override
    public long getTotalConnections() {
        return this.totalConnection.get();
    }

    @Override
    public long getIdleConnections() {
        return this.idleConnections.size();
    }

    @Override
    public long getConnectionRequests() {
        return this.pendingRequestNumber.get();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void registerJmx() throws Exception {
        MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
        String jmxName = this.poolTag.replace(":", "_");
        ObjectName name = new ObjectName("org.mariadb.jdbc.pool:type=" + jmxName);
        MBeanServer mBeanServer = mbs;
        synchronized (mBeanServer) {
            if (!mbs.isRegistered(name)) {
                mbs.registerMBean(this, name);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void unRegisterJmx() throws Exception {
        MBeanServer mbs = ManagementFactory.getPlatformMBeanServer();
        String jmxName = this.poolTag.replace(":", "_");
        ObjectName name = new ObjectName("org.mariadb.jdbc.pool:type=" + jmxName);
        MBeanServer mBeanServer = mbs;
        synchronized (mBeanServer) {
            if (mbs.isRegistered(name)) {
                mbs.unregisterMBean(name);
            }
        }
    }

    public List<Long> testGetConnectionIdleThreadIds() {
        ArrayList<Long> threadIds = new ArrayList<Long>();
        for (MariaDbInnerPoolConnection pooledConnection : this.idleConnections) {
            threadIds.add(pooledConnection.getConnection().getThreadId());
        }
        return threadIds;
    }
}

