"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
Object.defineProperty(exports, "SecurityPluginSetup", {
  enumerable: true,
  get: function () {
    return _types.SecurityPluginSetup;
  }
});
Object.defineProperty(exports, "SecurityPluginStart", {
  enumerable: true,
  get: function () {
    return _types.SecurityPluginStart;
  }
});
exports.configSchema = exports.config = void 0;
exports.plugin = plugin;
var _configSchema = require("@osd/config-schema");
var _plugin = require("./plugin");
var _common = require("../common");
var _jwt_auth = require("./auth/types/jwt/jwt_auth");
var _types = require("./types");
/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */

const validateAuthType = value => {
  const supportedAuthTypes = ['', _common.AuthType.BASIC, 'jwt', 'openid', _common.AuthType.SAML, 'proxy', 'kerberos', 'proxycache'];
  value.forEach(authVal => {
    if (!supportedAuthTypes.includes(authVal.toLowerCase())) {
      throw new Error(`Unsupported authentication type: ${authVal}. Allowed auth.type are ${supportedAuthTypes}.`);
    }
  });
};
const configSchema = exports.configSchema = _configSchema.schema.object({
  enabled: _configSchema.schema.boolean({
    defaultValue: true
  }),
  allow_client_certificates: _configSchema.schema.boolean({
    defaultValue: false
  }),
  readonly_mode: _configSchema.schema.object({
    roles: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    })
  }),
  clusterPermissions: _configSchema.schema.object({
    include: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    })
  }),
  indexPermissions: _configSchema.schema.object({
    include: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    })
  }),
  disabledTransportCategories: _configSchema.schema.object({
    exclude: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    })
  }),
  disabledRestCategories: _configSchema.schema.object({
    exclude: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    })
  }),
  cookie: _configSchema.schema.object({
    secure: _configSchema.schema.boolean({
      defaultValue: false
    }),
    name: _configSchema.schema.string({
      defaultValue: 'security_authentication'
    }),
    password: _configSchema.schema.string({
      defaultValue: 'security_cookie_default_password',
      minLength: 32
    }),
    ttl: _configSchema.schema.number({
      defaultValue: 60 * 60 * 1000
    }),
    domain: _configSchema.schema.nullable(_configSchema.schema.string()),
    isSameSite: _configSchema.schema.oneOf([_configSchema.schema.literal('Strict'), _configSchema.schema.literal('Lax'), _configSchema.schema.literal('None'), _configSchema.schema.literal(false)], {
      defaultValue: false
    })
  }),
  session: _configSchema.schema.object({
    ttl: _configSchema.schema.number({
      defaultValue: 60 * 60 * 1000
    }),
    keepalive: _configSchema.schema.boolean({
      defaultValue: true
    })
  }),
  auth: _configSchema.schema.object({
    type: _configSchema.schema.oneOf([_configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: [''],
      validate(value) {
        if (!value || value.length === 0) {
          return `Authentication type is not configured properly. At least one authentication type must be selected.`;
        }
        if (value.length > 1) {
          const includeBasicAuth = value.find(element => {
            return element.toLowerCase() === _common.AuthType.BASIC;
          });
          if (!includeBasicAuth) {
            return `Authentication type is not configured properly. basicauth is mandatory.`;
          }
        }
        validateAuthType(value);
      }
    }), _configSchema.schema.string({
      defaultValue: '',
      validate(value) {
        const valArray = [];
        valArray.push(value);
        validateAuthType(valArray);
      }
    })], {
      defaultValue: ''
    }),
    anonymous_auth_enabled: _configSchema.schema.boolean({
      defaultValue: false
    }),
    unauthenticated_routes: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: ['/api/reporting/stats']
    }),
    forbidden_usernames: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    }),
    logout_url: _configSchema.schema.string({
      defaultValue: ''
    }),
    multiple_auth_enabled: _configSchema.schema.boolean({
      defaultValue: false
    })
  }),
  basicauth: _configSchema.schema.object({
    enabled: _configSchema.schema.boolean({
      defaultValue: true
    }),
    unauthenticated_routes: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    }),
    forbidden_usernames: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
      defaultValue: []
    }),
    header_trumps_session: _configSchema.schema.boolean({
      defaultValue: false
    }),
    alternative_login: _configSchema.schema.object({
      headers: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
        defaultValue: []
      }),
      show_for_parameter: _configSchema.schema.string({
        defaultValue: ''
      }),
      valid_redirects: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
        defaultValue: []
      }),
      button_text: _configSchema.schema.string({
        defaultValue: 'Log in with provider'
      }),
      buttonstyle: _configSchema.schema.string({
        defaultValue: ''
      })
    }),
    loadbalancer_url: _configSchema.schema.maybe(_configSchema.schema.string()),
    login: _configSchema.schema.object({
      title: _configSchema.schema.string({
        defaultValue: 'Log in to OpenSearch Dashboards'
      }),
      subtitle: _configSchema.schema.string({
        defaultValue: 'If you have forgotten your username or password, contact your system administrator.'
      }),
      showbrandimage: _configSchema.schema.boolean({
        defaultValue: true
      }),
      brandimage: _configSchema.schema.string({
        defaultValue: ''
      }),
      // TODO: update brand image
      buttonstyle: _configSchema.schema.string({
        defaultValue: ''
      })
    })
  }),
  multitenancy: _configSchema.schema.object({
    enabled: _configSchema.schema.boolean({
      defaultValue: false
    }),
    show_roles: _configSchema.schema.boolean({
      defaultValue: false
    }),
    enable_filter: _configSchema.schema.boolean({
      defaultValue: false
    }),
    debug: _configSchema.schema.boolean({
      defaultValue: false
    }),
    enable_aggregation_view: _configSchema.schema.boolean({
      defaultValue: false
    }),
    tenants: _configSchema.schema.object({
      enable_private: _configSchema.schema.boolean({
        defaultValue: true
      }),
      enable_global: _configSchema.schema.boolean({
        defaultValue: true
      }),
      preferred: _configSchema.schema.arrayOf(_configSchema.schema.string(), {
        defaultValue: []
      })
    })
  }),
  configuration: _configSchema.schema.object({
    enabled: _configSchema.schema.boolean({
      defaultValue: true
    })
  }),
  accountinfo: _configSchema.schema.object({
    enabled: _configSchema.schema.boolean({
      defaultValue: false
    })
  }),
  openid: _configSchema.schema.maybe(_configSchema.schema.object({
    connect_url: _configSchema.schema.maybe(_configSchema.schema.string()),
    header: _configSchema.schema.string({
      defaultValue: 'Authorization'
    }),
    // TODO: test if siblingRef() works here
    // client_id is required when auth.type is openid
    client_id: _configSchema.schema.conditional(_configSchema.schema.siblingRef('auth.type'), 'openid', _configSchema.schema.string(), _configSchema.schema.maybe(_configSchema.schema.string())),
    client_secret: _configSchema.schema.string({
      defaultValue: ''
    }),
    scope: _configSchema.schema.string({
      defaultValue: 'openid profile email address phone'
    }),
    base_redirect_url: _configSchema.schema.string({
      defaultValue: ''
    }),
    logout_url: _configSchema.schema.string({
      defaultValue: ''
    }),
    root_ca: _configSchema.schema.string({
      defaultValue: ''
    }),
    certificate: _configSchema.schema.string({
      defaultValue: ''
    }),
    private_key: _configSchema.schema.string({
      defaultValue: ''
    }),
    passphrase: _configSchema.schema.string({
      defaultValue: ''
    }),
    pfx: _configSchema.schema.string({
      defaultValue: ''
    }),
    verify_hostnames: _configSchema.schema.boolean({
      defaultValue: true
    }),
    refresh_tokens: _configSchema.schema.boolean({
      defaultValue: true
    }),
    trust_dynamic_headers: _configSchema.schema.boolean({
      defaultValue: false
    }),
    additional_parameters: _configSchema.schema.maybe(_configSchema.schema.any({
      defaultValue: null
    })),
    extra_storage: _configSchema.schema.object({
      cookie_prefix: _configSchema.schema.string({
        defaultValue: 'security_authentication_oidc',
        minLength: 2
      }),
      additional_cookies: _configSchema.schema.number({
        min: 1,
        defaultValue: 5
      })
    })
  })),
  saml: _configSchema.schema.object({
    extra_storage: _configSchema.schema.object({
      cookie_prefix: _configSchema.schema.string({
        defaultValue: 'security_authentication_saml',
        minLength: 2
      }),
      additional_cookies: _configSchema.schema.number({
        min: 0,
        defaultValue: 3
      })
    })
  }),
  proxycache: _configSchema.schema.maybe(_configSchema.schema.object({
    // when auth.type is proxycache, user_header, roles_header and proxy_header_ip are required
    user_header: _configSchema.schema.conditional(_configSchema.schema.siblingRef('auth.type'), 'proxycache', _configSchema.schema.string(), _configSchema.schema.maybe(_configSchema.schema.string())),
    roles_header: _configSchema.schema.conditional(_configSchema.schema.siblingRef('auth.type'), 'proxycache', _configSchema.schema.string(), _configSchema.schema.maybe(_configSchema.schema.string())),
    proxy_header: _configSchema.schema.maybe(_configSchema.schema.string({
      defaultValue: 'x-forwarded-for'
    })),
    proxy_header_ip: _configSchema.schema.conditional(_configSchema.schema.siblingRef('auth.type'), 'proxycache', _configSchema.schema.string(), _configSchema.schema.maybe(_configSchema.schema.string())),
    login_endpoint: _configSchema.schema.maybe(_configSchema.schema.string({
      defaultValue: ''
    }))
  })),
  jwt: _configSchema.schema.maybe(_configSchema.schema.object({
    enabled: _configSchema.schema.boolean({
      defaultValue: false
    }),
    login_endpoint: _configSchema.schema.maybe(_configSchema.schema.string()),
    url_param: _configSchema.schema.string({
      defaultValue: 'authorization'
    }),
    header: _configSchema.schema.string({
      defaultValue: 'Authorization'
    }),
    extra_storage: _configSchema.schema.object({
      cookie_prefix: _configSchema.schema.string({
        defaultValue: _jwt_auth.JWT_DEFAULT_EXTRA_STORAGE_OPTIONS.cookiePrefix,
        minLength: 2
      }),
      additional_cookies: _configSchema.schema.number({
        min: 1,
        defaultValue: _jwt_auth.JWT_DEFAULT_EXTRA_STORAGE_OPTIONS.additionalCookies
      })
    })
  })),
  ui: _configSchema.schema.object({
    basicauth: _configSchema.schema.object({
      // the login config here is the same as old config `_security.basicauth.login`
      // Since we are now rendering login page to browser app, so move these config to browser side.
      login: _configSchema.schema.object({
        title: _configSchema.schema.string({
          defaultValue: 'Log in to OpenSearch Dashboards'
        }),
        subtitle: _configSchema.schema.string({
          defaultValue: 'If you have forgotten your username or password, contact your system administrator.'
        }),
        showbrandimage: _configSchema.schema.boolean({
          defaultValue: true
        }),
        brandimage: _configSchema.schema.string({
          defaultValue: ''
        }),
        buttonstyle: _configSchema.schema.string({
          defaultValue: ''
        })
      })
    }),
    anonymous: _configSchema.schema.object({
      login: _configSchema.schema.object({
        buttonname: _configSchema.schema.string({
          defaultValue: 'Log in as anonymous'
        }),
        showbrandimage: _configSchema.schema.boolean({
          defaultValue: false
        }),
        brandimage: _configSchema.schema.string({
          defaultValue: ''
        }),
        buttonstyle: _configSchema.schema.string({
          defaultValue: ''
        })
      })
    }),
    openid: _configSchema.schema.object({
      login: _configSchema.schema.object({
        buttonname: _configSchema.schema.string({
          defaultValue: 'Log in with single sign-on'
        }),
        showbrandimage: _configSchema.schema.boolean({
          defaultValue: false
        }),
        brandimage: _configSchema.schema.string({
          defaultValue: ''
        }),
        buttonstyle: _configSchema.schema.string({
          defaultValue: ''
        })
      })
    }),
    saml: _configSchema.schema.object({
      login: _configSchema.schema.object({
        buttonname: _configSchema.schema.string({
          defaultValue: 'Log in with single sign-on'
        }),
        showbrandimage: _configSchema.schema.boolean({
          defaultValue: false
        }),
        brandimage: _configSchema.schema.string({
          defaultValue: ''
        }),
        buttonstyle: _configSchema.schema.string({
          defaultValue: ''
        })
      })
    }),
    autologout: _configSchema.schema.boolean({
      defaultValue: true
    }),
    backend_configurable: _configSchema.schema.boolean({
      defaultValue: true
    })
  })
});
const config = exports.config = {
  exposeToBrowser: {
    enabled: true,
    auth: true,
    ui: true,
    multitenancy: true,
    readonly_mode: true,
    clusterPermissions: true,
    indexPermissions: true,
    disabledTransportCategories: true,
    disabledRestCategories: true
  },
  schema: configSchema,
  deprecations: ({
    rename,
    unused
  }) => [rename('basicauth.login.title', 'ui.basicauth.login.title'), rename('basicauth.login.subtitle', 'ui.basicauth.login.subtitle'), rename('basicauth.login.showbrandimage', 'ui.basicauth.login.showbrandimage'), rename('basicauth.login.brandimage', 'ui.basicauth.login.brandimage'), rename('basicauth.login.buttonstyle', 'ui.basicauth.login.buttonstyle')]
};

//  This exports static code and TypeScript types,
//  as well as, OpenSearchDashboards Platform `plugin()` initializer.

function plugin(initializerContext) {
  return new _plugin.SecurityPlugin(initializerContext);
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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