"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = _default;
var _configSchema = require("@osd/config-schema");
var _common = require("../../common");
var _helpers = require("./utils/helpers");
var _createReportDefinition = require("./lib/createReportDefinition");
var _backendToUi = require("./utils/converters/backendToUi");
var _updateReportDefinition = require("./lib/updateReportDefinition");
var _constants = require("./utils/constants");
var _metricHelper = require("./utils/metricHelper");
var _validationHelper = require("../../server/utils/validationHelper");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function _default(router, config) {
  const protocol = config.get('osd_server', 'protocol');
  const hostname = config.get('osd_server', 'hostname');
  const port = config.get('osd_server', 'port');
  const basePath = config.osdConfig.get('server', 'basePath');

  // Create report Definition
  router.post({
    path: `${_common.API_PREFIX}/reportDefinition`,
    validate: {
      body: _configSchema.schema.any()
    }
  }, async (context, request, response) => {
    (0, _metricHelper.addToMetric)('report_definition', 'create', 'count');
    let reportDefinition = request.body;
    //@ts-ignore
    const logger = context.reporting_plugin.logger;
    // input validation
    try {
      reportDefinition.report_params.core_params.origin = `${protocol}://${hostname}:${port}${basePath}`;
      reportDefinition = await (0, _validationHelper.validateReportDefinition)(context.core.opensearch.legacy.client, reportDefinition, basePath);
    } catch (error) {
      logger.error(`Failed input validation for create report definition ${error}`);
      (0, _metricHelper.addToMetric)('report_definition', 'create', 'user_error');
      return response.badRequest({
        body: error
      });
    }

    // save metadata
    try {
      const res = await (0, _createReportDefinition.createReportDefinition)(request, context, reportDefinition);
      return response.ok({
        body: {
          state: 'Report definition created',
          scheduler_response: res
        }
      });
    } catch (error) {
      logger.error(`Failed to create report definition: ${error}`);
      (0, _metricHelper.addToMetric)('report_definition', 'create', (0, _helpers.checkErrorType)(error));
      return (0, _helpers.errorResponse)(response, error);
    }
  });

  // Update report definition by id
  router.put({
    path: `${_common.API_PREFIX}/reportDefinitions/{reportDefinitionId}`,
    validate: {
      body: _configSchema.schema.any(),
      params: _configSchema.schema.object({
        reportDefinitionId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    (0, _metricHelper.addToMetric)('report_definition', 'update', 'count');
    let reportDefinition = request.body;
    //@ts-ignore
    const logger = context.reporting_plugin.logger;
    // input validation
    try {
      reportDefinition.report_params.core_params.origin = request.headers.origin;
      reportDefinition = await (0, _validationHelper.validateReportDefinition)(context.core.opensearch.legacy.client, reportDefinition, basePath);
    } catch (error) {
      logger.error(`Failed input validation for update report definition ${error}`);
      (0, _metricHelper.addToMetric)('report_definition', 'update', 'user_error');
      return response.badRequest({
        body: error
      });
    }
    // Update report definition metadata
    try {
      const opensearchResp = await (0, _updateReportDefinition.updateReportDefinition)(request, context, reportDefinition);
      return response.ok({
        body: {
          state: 'Report definition updated',
          scheduler_response: opensearchResp
        }
      });
    } catch (error) {
      logger.error(`Failed to update report definition: ${error}`);
      (0, _metricHelper.addToMetric)('report_definition', 'update', (0, _helpers.checkErrorType)(error));
      return (0, _helpers.errorResponse)(response, error);
    }
  });

  // get all report definitions details
  router.get({
    path: `${_common.API_PREFIX}/reportDefinitions`,
    validate: {
      query: _configSchema.schema.object({
        fromIndex: _configSchema.schema.maybe(_configSchema.schema.number()),
        maxItems: _configSchema.schema.maybe(_configSchema.schema.number())
      })
    }
  }, async (context, request, response) => {
    (0, _metricHelper.addToMetric)('report_definition', 'list', 'count');
    const {
      fromIndex,
      maxItems
    } = request.query;
    try {
      // @ts-ignore
      const opensearchReportsClient = context.reporting_plugin.opensearchReportsClient.asScoped(request);
      const opensearchResp = await opensearchReportsClient.callAsCurrentUser('opensearch_reports.getReportDefinitions', {
        fromIndex: fromIndex,
        maxItems: maxItems || _constants.DEFAULT_MAX_SIZE
      });
      const reportDefinitionsList = (0, _backendToUi.backendToUiReportDefinitionsList)(opensearchResp.reportDefinitionDetailsList, basePath);
      return response.ok({
        body: {
          data: reportDefinitionsList
        }
      });
    } catch (error) {
      //@ts-ignore
      context.reporting_plugin.logger.error(`Failed to get report definition details: ${error}`);
      (0, _metricHelper.addToMetric)('report_definition', 'list', (0, _helpers.checkErrorType)(error));
      return (0, _helpers.errorResponse)(response, error);
    }
  });

  // get report definition detail by id
  router.get({
    path: `${_common.API_PREFIX}/reportDefinitions/{reportDefinitionId}`,
    validate: {
      params: _configSchema.schema.object({
        reportDefinitionId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    (0, _metricHelper.addToMetric)('report_definition', 'info', 'count');
    try {
      // @ts-ignore
      const opensearchReportsClient = context.reporting_plugin.opensearchReportsClient.asScoped(request);
      const opensearchResp = await opensearchReportsClient.callAsCurrentUser('opensearch_reports.getReportDefinitionById', {
        reportDefinitionId: request.params.reportDefinitionId
      });
      const reportDefinition = (0, _backendToUi.backendToUiReportDefinition)(opensearchResp.reportDefinitionDetails, basePath);
      return response.ok({
        body: {
          report_definition: reportDefinition
        }
      });
    } catch (error) {
      //@ts-ignore
      context.reporting_plugin.logger.error(`Failed to get single report details: ${error}`);
      (0, _metricHelper.addToMetric)('report_definition', 'info', (0, _helpers.checkErrorType)(error));
      return (0, _helpers.errorResponse)(response, error);
    }
  });

  // Delete report definition by id
  router.delete({
    path: `${_common.API_PREFIX}/reportDefinitions/{reportDefinitionId}`,
    validate: {
      params: _configSchema.schema.object({
        reportDefinitionId: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    (0, _metricHelper.addToMetric)('report_definition', 'delete', 'count');
    try {
      // @ts-ignore
      const opensearchReportsClient = context.reporting_plugin.opensearchReportsClient.asScoped(request);
      const opensearchResp = await opensearchReportsClient.callAsCurrentUser('opensearch_reports.deleteReportDefinitionById', {
        reportDefinitionId: request.params.reportDefinitionId
      });
      return response.ok({
        body: {
          state: 'Report definition deleted',
          opensearch_response: opensearchResp
        }
      });
    } catch (error) {
      //@ts-ignore
      context.reporting_plugin.logger.error(`Failed to delete report definition: ${error}`);
      (0, _metricHelper.addToMetric)('report_definition', 'delete', (0, _helpers.checkErrorType)(error));
      return (0, _helpers.errorResponse)(response, error);
    }
  });
}
module.exports = exports.default;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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