"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CSVProcessor = void 0;
var _fastCsv = require("fast-csv");
var _util = require("../utils/util");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

class CSVProcessor {
  async validateText(text, options) {
    if (!!!options.delimiter) {
      return false;
    }
    return await new Promise((resolve, reject) => {
      (0, _fastCsv.parseString)(text, {
        headers: true,
        delimiter: options.delimiter
      }).validate(row => (0, _util.isValidObject)(row)).on('error', error => reject(error)).on('data-invalid', (_, rowNumber) => reject(new Error(`Row ${rowNumber} is invalid`))).on('data', () => {}).on('end', () => resolve(true));
    });
  }
  async ingestText(text, options) {
    const {
      client,
      indexName,
      delimiter
    } = options;
    const failedRows = [];
    const numDocuments = await new Promise((resolve, reject) => {
      const tasks = [];
      let totalRows = 0;
      (0, _fastCsv.parseString)(text, {
        headers: true,
        delimiter
      }).on('data', async row => {
        const task = (async () => {
          const curRow = ++totalRows;
          try {
            await client.index({
              index: indexName,
              body: row
            });
          } catch (_) {
            failedRows.push(curRow);
          }
        })();
        tasks.push(task);
      }).on('error', e => reject(new Error(`Stopped processing after ${totalRows} rows due to: ${e}`))).on('end', async rowCount => {
        await Promise.all(tasks);
        resolve(rowCount);
      });
    });
    return {
      total: numDocuments,
      message: `Indexed ${numDocuments - failedRows.length} documents`,
      failedRows: failedRows.sort((n1, n2) => n1 - n2)
    };
  }
  async ingestFile(file, options) {
    const {
      client,
      indexName,
      delimiter
    } = options;
    if (!!!options.delimiter) {
      throw new Error('Delimiter is required');
    }
    const failedRows = [];
    const numDocuments = await new Promise((resolve, reject) => {
      const tasks = [];
      let totalRows = 0;
      (0, _fastCsv.parseStream)(file, {
        headers: true,
        delimiter
      }).validate(row => (0, _util.isValidObject)(row)).on('data', row => {
        const task = (async () => {
          const curRow = ++totalRows;
          try {
            await client.index({
              index: indexName,
              body: row
            });
          } catch (_) {
            failedRows.push(curRow);
          }
        })();
        tasks.push(task);
      }).on('data-invalid', (_, rowCount) => {
        totalRows++;
        failedRows.push(rowCount);
      }).on('error', e => reject(new Error(`Stopped processing after ${totalRows} rows due to: ${e}`))).on('end', async rowCount => {
        await Promise.all(tasks);
        resolve(rowCount);
      });
    });
    return {
      total: numDocuments,
      message: `Indexed ${numDocuments - failedRows.length} documents`,
      failedRows: failedRows.sort((n1, n2) => n1 - n2)
    };
  }
  async parseFile(file, limit, options) {
    const {
      delimiter
    } = options;
    if (!!!options.delimiter) {
      throw new Error('Delimiter is required');
    }
    const documents = [];
    await new Promise((resolve, reject) => {
      (0, _fastCsv.parseStream)(file, {
        headers: true,
        delimiter
      }).validate(row => (0, _util.isValidObject)(row)).on('data', row => {
        if (documents.length >= limit) {
          resolve();
          file.destroy();
          return;
        }
        documents.push(row);
      }).on('data-invalid', (_, rowNumber) => reject(new Error(`Row ${rowNumber} is invalid`))).on('error', error => reject(error)).on('end', () => resolve());
    });
    return documents;
  }
}
exports.CSVProcessor = CSVProcessor;