// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
import { needs, immutableClass, readOnlyProperty, } from '@aws-crypto/material-management';
import { BRANCH_KEY_TYPE_PREFIX, BRANCH_KEY_IDENTIFIER_FIELD, TABLE_FIELD, TYPE_FIELD, KEY_CREATE_TIME_FIELD, HIERARCHY_VERSION_FIELD, KMS_FIELD, BRANCH_KEY_ACTIVE_VERSION_FIELD, BRANCH_KEY_ACTIVE_TYPE, } from './constants';
//= aws-encryption-sdk-specification/framework/key-store/key-storage.md#activehierarchicalsymmetric
//= type=implication
//# A structure that MUST have one member,
//# the UTF8 Encoded value of the version of the branch key.
export class ActiveHierarchicalSymmetricVersion {
    constructor(activeVersion) {
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#branch-key-materials-from-authenticated-encryption-context
        //# If the `type` attribute is equal to `"branch:ACTIVE"`
        //# then the authenticated encryption context MUST have a `version` attribute
        //# and the version string is this value.
        needs(activeVersion.startsWith(BRANCH_KEY_TYPE_PREFIX), 'Unexpected branch key type.');
        readOnlyProperty(this, 'version', activeVersion.substring(BRANCH_KEY_TYPE_PREFIX.length));
        Object.freeze(this);
    }
}
immutableClass(ActiveHierarchicalSymmetricVersion);
//= aws-encryption-sdk-specification/framework/key-store/key-storage.md#hierarchicalsymmetric
//= type=implication
//# A structure that MUST have one member,
//# the UTF8 Encoded value of the version of the branch key.
export class HierarchicalSymmetricVersion {
    constructor(type_field) {
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#branch-key-materials-from-authenticated-encryption-context
        //# If the `type` attribute start with `"branch:version:"` then the version string MUST be equal to this value.
        needs(type_field.startsWith(BRANCH_KEY_TYPE_PREFIX), 'Type does not start with `branch:version:`');
        readOnlyProperty(this, 'version', type_field.substring(BRANCH_KEY_TYPE_PREFIX.length));
        Object.freeze(this);
    }
}
immutableClass(HierarchicalSymmetricVersion);
export class EncryptedHierarchicalKey {
    constructor(encryptionContext, ciphertextBlob) {
        readOnlyProperty(this, 'branchKeyId', encryptionContext[BRANCH_KEY_IDENTIFIER_FIELD]);
        //= aws-encryption-sdk-specification/framework/branch-key-store.md#branch-key-materials-from-authenticated-encryption-context
        //# The `type` attribute MUST either be equal to `"branch:ACTIVE"` or start with `"branch:version:"`.
        needs(encryptionContext[TYPE_FIELD] == BRANCH_KEY_ACTIVE_TYPE ||
            encryptionContext[TYPE_FIELD].startsWith(BRANCH_KEY_TYPE_PREFIX), 'Unexpected branch key type.');
        readOnlyProperty(this, 'type', encryptionContext[TYPE_FIELD] == BRANCH_KEY_ACTIVE_TYPE
            ? new ActiveHierarchicalSymmetricVersion(encryptionContext[BRANCH_KEY_ACTIVE_VERSION_FIELD])
            : new HierarchicalSymmetricVersion(encryptionContext[TYPE_FIELD]));
        readOnlyProperty(this, 'createTime', encryptionContext[KEY_CREATE_TIME_FIELD]);
        readOnlyProperty(this, 'kmsArn', encryptionContext[KMS_FIELD]);
        readOnlyProperty(this, 'encryptionContext', Object.freeze({ ...encryptionContext }));
        readOnlyProperty(this, 'ciphertextBlob', ciphertextBlob);
        Object.freeze(this);
    }
}
immutableClass(EncryptedHierarchicalKey);
// This is a limited release for JS only.
// The full Key Store operations are available
// in the AWS Cryptographic Material Providers library
// in various languages (Java, .Net, Python, Rust...)
//= aws-encryption-sdk-specification/framework/key-store/key-storage.md#interface
//= type=exception
//# - [WriteNewEncryptedBranchKey](#writenewencryptedbranchkey)
//# - [WriteNewEncryptedBranchKeyVersion](#writenewencryptedbranchkeyversion)
//= aws-encryption-sdk-specification/framework/key-store/key-storage.md#interface
//= type=exception
//# - [GetEncryptedBeaconKey](#getencryptedbeaconkey)
//= aws-encryption-sdk-specification/framework/key-store/key-storage.md#writenewencryptedbranchkey
//= type=exception
//# The WriteNewEncryptedBranchKey caller MUST provide:
//= aws-encryption-sdk-specification/framework/key-store/key-storage.md#writenewencryptedbranchkeyversion
//= type=exception
//# The WriteNewEncryptedBranchKeyVersion caller MUST provide:
//= aws-encryption-sdk-specification/framework/key-store/key-storage.md#getencryptedbeaconkey
//= type=exception
//# The GetEncryptedBeaconKey caller MUST provide the same inputs as the [GetBeaconKey](../branch-key-store.md#getbeaconkey) operation.
//# It MUST return an [EncryptedHierarchicalKey](#encryptedhierarchicalkey).
//# sourceMappingURL=data:application/json;base64,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