#!/usr/bin/env python3
# _*_ coding: utf-8 _*_


def rgba_to_hex(color):
    """
    Return hexadecimal string for :class:`Gdk.RGBA` `color`
    http://wrhansen.blogspot.com/2012/09/how-to-convert-gdkrgba-to-hex-string-in.html
    """
    return "#{0:02x}{1:02x}{2:02x}".format(int(color.red * 255),
                                           int(color.green * 255),
                                           int(color.blue * 255))


def rgb_to_hex(rgb_color):
    return '#%02x%02x%02x' % (rgb_color[0], rgb_color[1], rgb_color[2])


def hex_to_rgb(string):
    string = string.lstrip('#')
    return tuple(int(string[i:i+2], 16) for i in (0, 2, 4))


def rgb_to_rgba(rgb):
    """
    :param rgb: tuple (rrr, ggg, bbb)
    :return: tuple (r.r, g.g, b.b, 1.0)
    """
    rgba = (rgb[0] / 255, rgb[1] / 255, rgb [2] / 255, 1.0)
    return rgba


class WikiColours(object):
    def __init__(self):
        super().__init__()
        # Definitions come from https://en.wikipedia.org/wiki/List_of_colors:_A%E2%80%93F.
        # I removed redundant names arbitrarily.
        self.colours = {"#0048ba": "Absolute Zero",
                        "#b0bf1a": "Acid green",
                        "#7cb9e8": "Aero",
                        "#c9ffe5": "Aero blue",
                        "#b284be": "African violet",
                        "#72a0c1": "Air superiority blue",
                        "#edeae0": "Alabaster",
                        "#f0f8ff": "Alice blue",
                        "#c46210": "Alloy orange",
                        "#efdecd": "Almond",
                        "#e52b50": "Amaranth",
                        "#9f2b68": "Amaranth (M&P)",
                        "#f19cbb": "Amaranth pink",
                        "#ab274f": "Amaranth purple",
                        "#d3212d": "Amaranth red",
                        "#3b7a57": "Amazon",
                        "#ffbf00": "Amber",
                        "#ff7e00": "Amber (SAE/ECE)",
                        "#9966cc": "Amethyst",
                        "#a4c639": "Android green",
                        "#cd9575": "Antique brass",
                        "#665d1e": "Antique bronze",
                        "#915c83": "Antique fuchsia",
                        "#841b2d": "Antique ruby",
                        "#faebd7": "Antique white",
                        "#8db600": "Apple green",
                        "#fbceb1": "Apricot",
                        "#7fffd4": "Aquamarine",
                        "#d0ff14": "Arctic lime",
                        "#4b5320": "Army green",
                        "#8f9779": "Artichoke",
                        "#e9d66b": "Arylide yellow",
                        "#b2beb5": "Ash gray",
                        "#87a96b": "Asparagus",
                        "#ff9966": "Atomic tangerine",
                        "#a52a2a": "Auburn",
                        "#fdee00": "Aureolin",
                        "#568203": "Avocado",
                        "#007fff": "Azure",
                        "#f0ffff": "Azure (X11/web color)",
                        "#89cff0": "Baby blue",
                        "#a1caf1": "Baby blue eyes",
                        "#f4c2c2": "Baby pink",
                        "#fefefa": "Baby powder",
                        "#ff91af": "Baker-Miller pink",
                        "#fae7b5": "Banana Mania",
                        "#da1884": "Barbie Pink",
                        "#7c0a02": "Barn red",
                        "#bcd4e6": "Beau blue",
                        "#9f8170": "Beaver",
                        "#f5f5dc": "Beige",
                        "#2e5894": "B'dazzled blue",
                        "#9c2542": "Big dip o’ruby",
                        "#ffe4c4": "Bisque",
                        "#3d2b1f": "Bistre",
                        "#967117": "Bistre brown",
                        "#cae00d": "Bitter lemon",
                        "#bfff00": "Bitter lime",
                        "#fe6f5e": "Bittersweet",
                        "#bf4f51": "Bittersweet shimmer",
                        "#000000": "Black",
                        "#3d0c02": "Black bean",
                        "#1b1811": "Black chocolate",
                        "#3b2f2f": "Black coffee",
                        "#54626f": "Black coral",
                        "#3b3c36": "Black olive",
                        "#bfafb2": "Black Shadows",
                        "#ffebcd": "Blanched almond",
                        "#a57164": "Blast-off bronze",
                        "#318ce7": "Bleu de France",
                        "#ace5ee": "Blizzard blue",
                        "#faf0be": "Blond",
                        "#660000": "Blood red",
                        "#0000ff": "Blue",
                        "#1f75fe": "Blue (Crayola)",
                        "#0093af": "Blue (Munsell)",
                        "#0087bd": "Blue (NCS)",
                        "#0018a8": "Blue (Pantone)",
                        "#333399": "Blue (pigment)",
                        "#0247fe": "Blue (RYB)",
                        "#a2a2d0": "Blue bell",
                        "#6699cc": "Blue-gray",
                        "#0d98ba": "Blue-green",
                        "#064e40": "Blue-green (color wheel)",
                        "#5dadec": "Blue jeans",
                        "#126180": "Blue sapphire",
                        "#8a2be2": "Blue-violet",
                        "#7366bd": "Blue-violet (Crayola)",
                        "#4d1a7f": "Blue-violet (color wheel)",
                        "#5072a7": "Blue yonder",
                        "#3c69e7": "Bluetiful",
                        "#de5d83": "Blush",
                        "#79443b": "Bole",
                        "#e3dac9": "Bone",
                        "#006a4e": "Bottle green",
                        "#87413f": "Brandy",
                        "#cb4154": "Brick red",
                        "#66ff00": "Bright green",
                        "#d891ef": "Bright lilac",
                        "#c32148": "Bright maroon",
                        "#1974d2": "Bright navy blue",
                        "#ffaa1d": "Bright yellow (Crayola)",
                        "#ff55a3": "Brilliant rose",
                        "#fb607f": "Brink pink",
                        "#004225": "British racing green",
                        "#cd7f32": "Bronze",
                        "#88540b": "Brown",
                        "#af6e4d": "Brown sugar",
                        "#1b4d3e": "Brunswick green",
                        "#7bb661": "Bud green",
                        "#f0dc82": "Buff",
                        "#800020": "Burgundy",
                        "#deb887": "Burlywood",
                        "#a17a74": "Burnished brown",
                        "#cc5500": "Burnt orange",
                        "#e97451": "Burnt sienna",
                        "#8a3324": "Burnt umber",
                        "#bd33a4": "Byzantine",
                        "#702963": "Byzantium",
                        "#536872": "Cadet",
                        "#5f9ea0": "Cadet blue",
                        "#a9b2c3": "Cadet blue (Crayola)",
                        "#91a3b0": "Cadet grey",
                        "#006b3c": "Cadmium green",
                        "#ed872d": "Cadmium orange",
                        "#e30022": "Cadmium red",
                        "#fff600": "Cadmium yellow",
                        "#a67b5b": "Café au lait",
                        "#4b3621": "Café noir",
                        "#a3c1ad": "Cambridge blue",
                        "#efbbcc": "Cameo pink",
                        "#ffff99": "Canary",
                        "#ffef00": "Canary yellow",
                        "#ff0800": "Candy apple red",
                        "#e4717a": "Candy pink",
                        "#592720": "Caput mortuum",
                        "#c41e3a": "Cardinal",
                        "#00cc99": "Caribbean green",
                        "#960018": "Carmine",
                        "#d70040": "Carmine (M&P)",
                        "#ffa6c9": "Carnation pink",
                        "#b31b1b": "Carnelian",
                        "#56a0d3": "Carolina blue",
                        "#ed9121": "Carrot orange",
                        "#00563f": "Castleton green",
                        "#703642": "Catawba",
                        "#c95a49": "Cedar Chest",
                        "#ace1af": "Celadon",
                        "#007ba7": "Celadon blue",
                        "#2f847c": "Celadon green",
                        "#246bce": "Celtic blue",
                        "#de3163": "Cerise",
                        "#2a52be": "Cerulean blue",
                        "#6d9bc3": "Cerulean frost",
                        "#1dacd6": "Cerulean (Crayola)",
                        "#007aa5": "CG blue",
                        "#e03c31": "CG red",
                        "#f7e7ce": "Champagne",
                        "#f1ddcf": "Champagne pink",
                        "#36454f": "Charcoal",
                        "#232b2b": "Charleston green",
                        "#e68fac": "Charm pink",
                        "#dfff00": "Chartreuse (traditional)",
                        "#7fff00": "Chartreuse (web)",
                        "#ffb7c5": "Cherry blossom pink",
                        "#954535": "Chestnut",
                        "#a8516e": "China rose",
                        "#aa381e": "Chinese red",
                        "#856088": "Chinese violet",
                        "#ffb200": "Chinese yellow",
                        "#7b3f00": "Chocolate (traditional)",
                        "#d2691e": "Chocolate (web)",
                        "#ffa700": "Chrome yellow",
                        "#98817b": "Cinereous",
                        "#e34234": "Cinnabar",
                        "#cd607e": "Cinnamon Satin",
                        "#e4d00a": "Citrine",
                        "#9fa91f": "Citron",
                        "#7f1734": "Claret",
                        "#0047ab": "Cobalt blue",
                        "#6f4e37": "Coffee",
                        "#b9d9eb": "Columbia Blue",
                        "#8c92ac": "Cool grey",
                        "#b87333": "Copper",
                        "#da8a67": "Copper (Crayola)",
                        "#ad6f69": "Copper penny",
                        "#cb6d51": "Copper red",
                        "#996666": "Copper rose",
                        "#ff3800": "Coquelicot",
                        "#ff7f50": "Coral",
                        "#893f45": "Cordovan",
                        "#6495ed": "Cornflower blue",
                        "#fff8dc": "Cornsilk",
                        "#2e2d88": "Cosmic cobalt",
                        "#fff8e7": "Cosmic latte",
                        "#81613c": "Coyote brown",
                        "#ffbcd9": "Cotton candy",
                        "#fffdd0": "Cream",
                        "#dc143c": "Crimson",
                        "#9e1b32": "Crimson (UA)",
                        "#f5f5f5": "Cultured",
                        "#00ffff": "Cyan",
                        "#00b7eb": "Cyan (process)",
                        "#58427c": "Cyber grape",
                        "#f56fa1": "Cyclamen",
                        "#666699": "Dark blue-gray",
                        "#654321": "Dark brown",
                        "#5d3954": "Dark byzantium",
                        "#26428b": "Dark cornflower blue",
                        "#008b8b": "Dark cyan",
                        "#536878": "Dark electric blue",
                        "#b8860b": "Dark goldenrod",
                        "#013220": "Dark green",
                        "#006400": "Dark green (X11)",
                        "#1a2421": "Dark jungle green",
                        "#bdb76b": "Dark khaki",
                        "#534b4f": "Dark liver",
                        "#543d37": "Dark liver (horses)",
                        "#8b008b": "Dark magenta",
                        "#4a5d23": "Dark moss green",
                        "#556b2f": "Dark olive green",
                        "#ff8c00": "Dark orange",
                        "#9932cc": "Dark orchid",
                        "#03c03c": "Dark pastel green",
                        "#301934": "Dark purple",
                        "#8b0000": "Dark red",
                        "#e9967a": "Dark salmon",
                        "#8fbc8f": "Dark sea green",
                        "#3c1414": "Dark sienna",
                        "#8cbed6": "Dark sky blue",
                        "#483d8b": "Dark slate blue",
                        "#2f4f4f": "Dark slate gray",
                        "#177245": "Dark spring green",
                        "#00ced1": "Dark turquoise",
                        "#9400d3": "Dark violet",
                        "#00703c": "Dartmouth green",
                        "#555555": "Davy's grey",
                        "#da3287": "Deep cerise",
                        "#b94e48": "Deep chestnut",
                        "#004b49": "Deep jungle green",
                        "#ff1493": "Deep pink",
                        "#ff9933": "Deep saffron",
                        "#00bfff": "Deep sky blue",
                        "#4a646c": "Deep Space Sparkle",
                        "#7e5e60": "Deep taupe",
                        "#1560bd": "Denim",
                        "#2243b6": "Denim blue",
                        "#edc9af": "Desert sand",
                        "#696969": "Dim gray",
                        "#1e90ff": "Dodger blue",
                        "#d71868": "Dogwood rose",
                        "#00009c": "Duke blue",
                        "#efdfbb": "Dutch white",
                        "#e1a95f": "Earth yellow",
                        "#555d50": "Ebony",
                        "#c2b280": "Ecru",
                        "#1b1b1b": "Eerie black",
                        "#614051": "Eggplant",
                        "#f0ead6": "Eggshell",
                        "#1034a6": "Egyptian blue",
                        "#7df9ff": "Electric blue",
                        "#6f00ff": "Electric indigo",
                        "#ccff00": "Electric lime",
                        "#bf00ff": "Electric purple",
                        "#6c3082": "Eminence",
                        "#b48395": "English lavender",
                        "#ab4b52": "English red",
                        "#cc474b": "English vermillion",
                        "#563c5c": "English violet",
                        "#00ff40": "Erin",
                        "#96c8a2": "Eton blue",
                        "#801818": "Falu red",
                        "#b53389": "Fandango",
                        "#de5285": "Fandango pink",
                        "#f400a1": "Fashion fuchsia",
                        "#e5aa70": "Fawn",
                        "#4d5d53": "Feldgrau",
                        "#4f7942": "Fern green",
                        "#6c541e": "Field drab",
                        "#ff5470": "Fiery rose",
                        "#b22222": "Firebrick",
                        "#ce2029": "Fire engine red",
                        "#e95c4b": "Fire opal",
                        "#e25822": "Flame",
                        "#eedc82": "Flax",
                        "#0063dc": "Flickr Blue",
                        "#fb0081": "Flickr Pink",
                        "#a2006d": "Flirt",
                        "#fffaf0": "Floral white",
                        "#15f4ee": "Fluorescent blue",
                        "#5fa777": "Forest green (Crayola)",
                        "#014421": "Forest green (traditional)",
                        "#228b22": "Forest green (web)",
                        "#856d4d": "French bistre",
                        "#0072bb": "French blue",
                        "#fd3f92": "French fuchsia",
                        "#86608e": "French lilac",
                        "#9efd38": "French lime",
                        "#d473d4": "French mauve",
                        "#fd6c9e": "French pink",
                        "#c72c48": "French raspberry",
                        "#f64a8a": "French rose",
                        "#77b5fe": "French sky blue",
                        "#8806ce": "French violet",
                        "#e936a7": "Frostbite",
                        "#c154c1": "Fuchsia (Crayola)",
                        "#cc397b": "Fuchsia purple",
                        "#c74375": "Fuchsia rose",
                        "#e48400": "Fulvous",
                        "#87421f": "Fuzzy Wuzzy",
                        "#dcdcdc": "Gainsboro",
                        "#e49b0f": "Gamboge",
                        "#007f66": "Generic viridian",
                        "#f8f8ff": "Ghost white",
                        "#6082b6": "Glaucous",
                        "#ab92b3": "Glossy grape",
                        "#00ab66": "GO green",
                        "#a57c00": "Gold",
                        "#d4af37": "Gold (metallic)",
                        "#ffd700": "Gold (web) (Golden)",
                        "#e6be8a": "Gold (Crayola)",
                        "#85754e": "Gold Fusion",
                        "#996515": "Golden brown",
                        "#fcc200": "Golden poppy",
                        "#ffdf00": "Golden yellow",
                        "#daa520": "Goldenrod",
                        "#676767": "Granite gray",
                        "#a8e4a0": "Granny Smith apple",
                        "#808080": "Gray (web)",
                        "#bebebe": "Gray (X11 gray)",
                        "#00ff00": "Green",
                        "#1cac78": "Green (Crayola)",
                        "#008000": "Green (web)",
                        "#00a877": "Green (Munsell)",
                        "#009f6b": "Green (NCS)",
                        "#00ad43": "Green (Pantone)",
                        "#00a550": "Green (pigment)",
                        "#66b032": "Green (RYB)",
                        "#1164b4": "Green-blue",
                        "#2887c8": "Green-blue (Crayola)",
                        "#009966": "Green-cyan",
                        "#a7f432": "Green Lizard",
                        "#6eaea1": "Green Sheen",
                        "#adff2f": "Green-yellow",
                        "#f0e891": "Green-yellow (Crayola)",
                        "#a99a86": "Grullo",
                        "#2a3439": "Gunmetal",
                        "#446ccf": "Han blue",
                        "#5218fa": "Han purple",
                        "#3fff00": "Harlequin",
                        "#da9100": "Harvest gold",
                        "#ff7a00": "Heat Wave",
                        "#df73ff": "Heliotrope",
                        "#aa98a9": "Heliotrope gray",
                        "#f0fff0": "Honeydew",
                        "#006db0": "Honolulu blue",
                        "#49796b": "Hooker's green",
                        "#ff1dce": "Hot magenta",
                        "#ff69b4": "Hot pink",
                        "#355e3b": "Hunter green",
                        "#71a6d2": "Iceberg",
                        "#fcf75e": "Icterine",
                        "#319177": "Illuminating emerald",
                        "#b2ec5d": "Inchworm",
                        "#4c516d": "Independence",
                        "#138808": "India green",
                        "#cd5c5c": "Indian red",
                        "#e3a857": "Indian yellow",
                        "#4b0082": "Indigo",
                        "#00416a": "Indigo dye",
                        "#002fa7": "International Klein Blue",
                        "#ff4f00": "International orange (aerospace)",
                        "#ba160c": "International orange (engineering)",
                        "#c0362c": "International orange (Golden Gate Bridge)",
                        "#5a4fcf": "Iris",
                        "#f4f0ec": "Isabelline",
                        "#b2ffff": "Italian sky blue",
                        "#fffff0": "Ivory",
                        "#00a86b": "Jade",
                        "#a50b5e": "Jazzberry jam",
                        "#343434": "Jet",
                        "#f4ca16": "Jonquil",
                        "#bdda57": "June bud",
                        "#29ab87": "Jungle green",
                        "#4cbb17": "Kelly green",
                        "#3ab09e": "Keppel",
                        "#e8f48c": "Key lime",
                        "#c3b091": "Khaki (web)",
                        "#f0e68c": "Khaki (X11) (Light khaki)",
                        "#e79fc4": "Kobi",
                        "#6b4423": "Kobicha",
                        "#354230": "Kombu green",
                        "#512888": "KSU purple",
                        "#d6cadd": "Languid lavender",
                        "#26619c": "Lapis lazuli",
                        "#ffff66": "Laser lemon",
                        "#a9ba9d": "Laurel green",
                        "#cf1020": "Lava",
                        "#b57edc": "Lavender (floral)",
                        "#e6e6fa": "Lavender (web)",
                        "#ccccff": "Lavender blue",
                        "#fff0f5": "Lavender blush",
                        "#c4c3d0": "Lavender gray",
                        "#7cfc00": "Lawn green",
                        "#fff700": "Lemon",
                        "#fffacd": "Lemon chiffon",
                        "#cca01d": "Lemon curry",
                        "#fdff00": "Lemon glacier",
                        "#f6eabe": "Lemon meringue",
                        "#fff44f": "Lemon yellow",
                        "#ffff9f": "Lemon yellow (Crayola)",
                        "#545aa7": "Liberty",
                        "#add8e6": "Light blue",
                        "#f08080": "Light coral",
                        "#93ccea": "Light cornflower blue",
                        "#e0ffff": "Light cyan",
                        "#c8ad7f": "Light French beige",
                        "#fafad2": "Light goldenrod yellow",
                        "#d3d3d3": "Light gray",
                        "#90ee90": "Light green",
                        "#fed8b1": "Light orange",
                        "#c5cbe1": "Light periwinkle",
                        "#ffb6c1": "Light pink",
                        "#ffa07a": "Light salmon",
                        "#20b2aa": "Light sea green",
                        "#87cefa": "Light sky blue",
                        "#778899": "Light slate gray",
                        "#b0c4de": "Light steel blue",
                        "#ffffe0": "Light yellow",
                        "#c8a2c8": "Lilac",
                        "#ae98aa": "Lilac Luster",
                        "#32cd32": "Lime green",
                        "#195905": "Lincoln green",
                        "#faf0e6": "Linen",
                        "#de6fa1": "Liseran purple",
                        "#6ca0dc": "Little boy blue",
                        "#674c47": "Liver",
                        "#b86d29": "Liver (dogs)",
                        "#6c2e1f": "Liver (organ)",
                        "#987456": "Liver chestnut",
                        "#ffbd88": "Macaroni and Cheese",
                        "#cc3336": "Madder Lake",
                        "#ff00ff": "Magenta",
                        "#f653a6": "Magenta (Crayola)",
                        "#ca1f7b": "Magenta (dye)",
                        "#d0417e": "Magenta (Pantone)",
                        "#ff0090": "Magenta (process)",
                        "#9f4576": "Magenta haze",
                        "#aaf0d1": "Magic mint",
                        "#f8f4ff": "Magnolia",
                        "#c04000": "Mahogany",
                        "#fbec5d": "Maize",
                        "#f2c649": "Maize (Crayola)",
                        "#6050dc": "Majorelle blue",
                        "#0bda51": "Malachite",
                        "#979aaa": "Manatee",
                        "#f37a48": "Mandarin",
                        "#fdbe02": "Mango",
                        "#ff8243": "Mango Tango",
                        "#74c365": "Mantis",
                        "#880085": "Mardi Gras",
                        "#eaa221": "Marigold",
                        "#800000": "Maroon (web)",
                        "#b03060": "Maroon (X11)",
                        "#e0b0ff": "Mauve",
                        "#915f6d": "Mauve taupe",
                        "#ef98aa": "Mauvelous",
                        "#47abcc": "Maximum blue",
                        "#30bfbf": "Maximum blue green",
                        "#acace6": "Maximum blue purple",
                        "#5e8c31": "Maximum green",
                        "#d9e650": "Maximum green yellow",
                        "#733380": "Maximum purple",
                        "#d92121": "Maximum red",
                        "#a63a79": "Maximum red purple",
                        "#fafa37": "Maximum yellow",
                        "#f2ba49": "Maximum yellow red",
                        "#4c9141": "May green",
                        "#73c2fb": "Maya blue",
                        "#66ddaa": "Medium aquamarine",
                        "#0000cd": "Medium blue",
                        "#e2062c": "Medium candy apple red",
                        "#af4035": "Medium carmine",
                        "#ba55d3": "Medium orchid",
                        "#9370db": "Medium purple",
                        "#3cb371": "Medium sea green",
                        "#7b68ee": "Medium slate blue",
                        "#00fa9a": "Medium spring green",
                        "#48d1cc": "Medium turquoise",
                        "#f8b878": "Mellow apricot",
                        "#f8de7e": "Mellow yellow",
                        "#febaad": "Melon",
                        "#d3af37": "Metallic gold",
                        "#0a7e8c": "Metallic Seaweed",
                        "#9c7c38": "Metallic Sunburst",
                        "#e4007c": "Mexican pink",
                        "#7ed4e6": "Middle blue",
                        "#8dd9cc": "Middle blue green",
                        "#8b72be": "Middle blue purple",
                        "#8b8680": "Middle grey",
                        "#4d8c57": "Middle green",
                        "#acbf60": "Middle green yellow",
                        "#d982b5": "Middle purple",
                        "#e58e73": "Middle red",
                        "#a55353": "Middle red purple",
                        "#ffeb00": "Middle yellow",
                        "#ecb176": "Middle yellow red",
                        "#702670": "Midnight",
                        "#191970": "Midnight blue",
                        "#004953": "Midnight green (eagle green)",
                        "#ffc40c": "Mikado yellow",
                        "#ffdae9": "Mimi pink",
                        "#e3f988": "Mindaro",
                        "#36747d": "Ming",
                        "#f5e050": "Minion yellow",
                        "#3eb489": "Mint",
                        "#f5fffa": "Mint cream",
                        "#98ff98": "Mint green",
                        "#bbb477": "Misty moss",
                        "#ffe4e1": "Misty rose",
                        "#8da399": "Morning blue",
                        "#8a9a5b": "Moss green",
                        "#30ba8f": "Mountain Meadow",
                        "#997a8d": "Mountbatten pink",
                        "#18453b": "MSU green",
                        "#c54b8c": "Mulberry",
                        "#c8509b": "Mulberry (Crayola)",
                        "#ffdb58": "Mustard",
                        "#317873": "Myrtle green",
                        "#d65282": "Mystic",
                        "#ad4379": "Mystic maroon",
                        "#f6adc6": "Nadeshiko pink",
                        "#ffdead": "Navajo white",
                        "#000080": "Navy blue",
                        "#4666ff": "Neon blue",
                        "#39ff14": "Neon green",
                        "#d7837f": "New York pink",
                        "#727472": "Nickel",
                        "#a4dded": "Non-photo blue",
                        "#e9ffdb": "Nyanza",
                        "#4f42b5": "Ocean Blue",
                        "#48bf91": "Ocean green",
                        "#cc7722": "Ochre",
                        "#43302e": "Old burgundy",
                        "#cfb53b": "Old gold",
                        "#fdf5e6": "Old lace",
                        "#796878": "Old lavender",
                        "#c08081": "Old rose",
                        "#848482": "Old silver",
                        "#808000": "Olive",
                        "#6b8e23": "Olive Drab (Nb3)",
                        "#3c341f": "Olive Drab (Nb7)",
                        "#b5b35c": "Olive green",
                        "#9ab973": "Olivine",
                        "#353839": "Onyx",
                        "#a8c3bc": "Opal",
                        "#b784a7": "Opera mauve",
                        "#ff7f00": "Orange",
                        "#ff7538": "Orange (Crayola)",
                        "#ff5800": "Orange (Pantone)",
                        "#ffa500": "Orange (web)",
                        "#ff9f00": "Orange peel",
                        "#ff681f": "Orange-red",
                        "#fa5b3d": "Orange soda",
                        "#f5bd1f": "Orange-yellow",
                        "#f8d568": "Orange-yellow (Crayola)",
                        "#da70d6": "Orchid",
                        "#f2bdcd": "Orchid pink",
                        "#e29cd2": "Orchid (Crayola)",
                        "#2d383a": "Outer space (Crayola)",
                        "#ff6e4a": "Outrageous Orange",
                        "#002147": "Oxford blue",
                        "#841617": "OU Crimson red",
                        "#1ca9c9": "Pacific blue",
                        "#006600": "Pakistan green",
                        "#682860": "Palatinate purple",
                        "#9bc4e2": "Pale cerulean",
                        "#fadadd": "Pale pink",
                        "#fae6fa": "Pale purple (Pantone)",
                        "#c9c0bb": "Pale silver",
                        "#78184a": "Pansy purple",
                        "#009b7d": "Paolo Veronese green",
                        "#ffefd5": "Papaya whip",
                        "#e63e62": "Paradise pink",
                        "#50c878": "Paris Green",
                        "#dea5a4": "Pastel pink",
                        "#ffe5b4": "Peach",
                        "#ffcba4": "Peach (Crayola)",
                        "#ffdab9": "Peach puff",
                        "#d1e231": "Pear",
                        "#b768a2": "Pearly purple",
                        "#c3cde6": "Periwinkle (Crayola)",
                        "#e12c2c": "Permanent Geranium Lake",
                        "#1c39bb": "Persian blue",
                        "#00a693": "Persian green",
                        "#32127a": "Persian indigo",
                        "#d99058": "Persian orange",
                        "#f77fbe": "Persian pink",
                        "#cc3333": "Persian red",
                        "#fe28a2": "Persian rose",
                        "#ec5800": "Persimmon",
                        "#8ba8b7": "Pewter Blue",
                        "#000f89": "Phthalo blue",
                        "#123524": "Phthalo green",
                        "#2e2787": "Picotee blue",
                        "#c30b4e": "Pictorial carmine",
                        "#fddde6": "Piggy pink",
                        "#01796f": "Pine green",
                        "#2a2f23": "Pine tree",
                        "#ffc0cb": "Pink",
                        "#d74894": "Pink (Pantone)",
                        "#fc74fd": "Pink flamingo",
                        "#ffddf4": "Pink lace",
                        "#d8b2d1": "Pink lavender",
                        "#f78fa7": "Pink Sherbet",
                        "#93c572": "Pistachio",
                        "#e5e4e2": "Platinum",
                        "#8e4585": "Plum",
                        "#dda0dd": "Plum (web)",
                        "#5946b2": "Plump Purple",
                        "#5da493": "Polished Pine",
                        "#be4f62": "Popstar",
                        "#ff5a36": "Portland Orange",
                        "#b0e0e6": "Powder blue",
                        "#f58025": "Princeton orange",
                        "#701c1c": "Prune",
                        "#003153": "Prussian blue",
                        "#df00ff": "Psychedelic purple",
                        "#cc8899": "Puce",
                        "#644117": "Pullman Brown (UPS Brown)",
                        "#ff7518": "Pumpkin",
                        "#6a0dad": "Purple",
                        "#800080": "Purple (web)",
                        "#9f00c5": "Purple (Munsell)",
                        "#a020f0": "Purple (X11)",
                        "#9678b6": "Purple mountain majesty",
                        "#4e5180": "Purple navy",
                        "#fe4eda": "Purple pizzazz",
                        "#9c51b6": "Purple Plum",
                        "#9a4eae": "Purpureus",
                        "#436b95": "Queen blue",
                        "#e8ccd7": "Queen pink",
                        "#a6a6a6": "Quick Silver",
                        "#8e3a59": "Quinacridone magenta",
                        "#ff355e": "Radical Red",
                        "#242124": "Raisin black",
                        "#fbab60": "Rajah",
                        "#e30b5d": "Raspberry",
                        "#b3446c": "Raspberry rose",
                        "#d68a59": "Raw Sienna",
                        "#826644": "Raw umber",
                        "#ff33cc": "Razzle dazzle rose",
                        "#e3256b": "Razzmatazz",
                        "#8d4e85": "Razzmic Berry",
                        "#663399": "Rebecca Purple",
                        "#ff0000": "Red",
                        "#ee204d": "Red (Crayola)",
                        "#f2003c": "Red (Munsell)",
                        "#c40233": "Red (NCS)",
                        "#ed2939": "Red (Pantone)",
                        "#ed1c24": "Red (pigment)",
                        "#fe2712": "Red (RYB)",
                        "#ff5349": "Red-orange",
                        "#ff4500": "Red-orange (Color wheel)",
                        "#e40078": "Red-purple",
                        "#fd3a4a": "Red Salsa",
                        "#c71585": "Red-violet",
                        "#c0448f": "Red-violet (Crayola)",
                        "#922b3e": "Red-violet (Color wheel)",
                        "#a45a52": "Redwood",
                        "#002387": "Resolution blue",
                        "#777696": "Rhythm",
                        "#004040": "Rich black",
                        "#010b13": "Rich black (FOGRA29)",
                        "#010203": "Rich black (FOGRA39)",
                        "#444c38": "Rifle green",
                        "#00cccc": "Robin egg blue",
                        "#8a7f80": "Rocket metallic",
                        "#838996": "Roman silver",
                        "#ff007f": "Rose",
                        "#f9429e": "Rose bonbon",
                        "#9e5e6f": "Rose Dust",
                        "#674846": "Rose ebony",
                        "#e32636": "Rose madder",
                        "#ff66cc": "Rose pink",
                        "#c21e56": "Rose red",
                        "#905d5d": "Rose taupe",
                        "#ab4e52": "Rose vale",
                        "#65000b": "Rosewood",
                        "#d40000": "Rosso corsa",
                        "#bc8f8f": "Rosy brown",
                        "#002366": "Royal blue (dark)",
                        "#4169e1": "Royal blue (light)",
                        "#7851a9": "Royal purple",
                        "#fada5e": "Royal yellow",
                        "#ce4676": "Ruber",
                        "#d10056": "Rubine red",
                        "#e0115f": "Ruby",
                        "#9b111e": "Ruby red",
                        "#a81c07": "Rufous",
                        "#80461b": "Russet",
                        "#679267": "Russian green",
                        "#32174d": "Russian violet",
                        "#b7410e": "Rust",
                        "#da2c43": "Rusty red",
                        "#043927": "Sacramento State green",
                        "#8b4513": "Saddle brown",
                        "#ff7800": "Safety orange",
                        "#ff6700": "Safety orange (blaze orange)",
                        "#eed202": "Safety yellow",
                        "#f4c430": "Saffron",
                        "#bcb88a": "Sage",
                        "#23297a": "St. Patrick's blue",
                        "#fa8072": "Salmon",
                        "#ff91a4": "Salmon pink",
                        "#f4a460": "Sandy brown",
                        "#507d2a": "Sap green",
                        "#0f52ba": "Sapphire",
                        "#0067a5": "Sapphire blue",
                        "#cba135": "Satin sheen gold",
                        "#ff2400": "Scarlet",
                        "#ffd800": "School bus yellow",
                        "#66ff66": "Screamin' Green",
                        "#2e8b57": "Sea green",
                        "#00ffcd": "Sea green (Crayola)",
                        "#59260b": "Seal brown",
                        "#fff5ee": "Seashell",
                        "#ffba00": "Selective yellow",
                        "#704214": "Sepia",
                        "#8a795d": "Shadow",
                        "#778ba5": "Shadow blue",
                        "#009e60": "Shamrock green",
                        "#8fd400": "Sheen green",
                        "#d98695": "Shimmering Blush",
                        "#5fa778": "Shiny Shamrock",
                        "#fc0fc0": "Shocking pink",
                        "#882d17": "Sienna",
                        "#c0c0c0": "Silver",
                        "#aaa9ad": "Silver (Metallic)",
                        "#acacac": "Silver chalice",
                        "#c4aead": "Silver pink",
                        "#bfc1c2": "Silver sand",
                        "#cb410b": "Sinopia",
                        "#ff3855": "Sizzling Red",
                        "#ffdb00": "Sizzling Sunrise",
                        "#007474": "Skobeloff",
                        "#87ceeb": "Sky blue",
                        "#76d7ea": "Sky blue (Crayola)",
                        "#cf71af": "Sky magenta",
                        "#6a5acd": "Slate blue",
                        "#708090": "Slate gray",
                        "#299617": "Slimy green",
                        "#c84186": "Smitten",
                        "#100c08": "Smoky black",
                        "#fffafa": "Snow",
                        "#893843": "Solid pink",
                        "#757575": "Sonic silver",
                        "#1d2951": "Space cadet",
                        "#807532": "Spanish bistre",
                        "#0070b8": "Spanish blue",
                        "#d10047": "Spanish carmine",
                        "#989898": "Spanish gray",
                        "#009150": "Spanish green",
                        "#e86100": "Spanish orange",
                        "#f7bfbe": "Spanish pink",
                        "#e60026": "Spanish red",
                        "#4c2882": "Spanish violet",
                        "#007f5c": "Spanish viridian",
                        "#a7fc00": "Spring bud",
                        "#87ff2a": "Spring Frost",
                        "#00ff7f": "Spring green",
                        "#ecebbd": "Spring green (Crayola)",
                        "#007bb8": "Star command blue",
                        "#4682b4": "Steel blue",
                        "#cc33cc": "Steel pink",
                        "#5f8a8b": "Steel Teal",
                        "#e4d96f": "Straw",
                        "#914e75": "Sugar Plum",
                        "#ffcc33": "Sunglow",
                        "#e3ab57": "Sunray",
                        "#fad6a5": "Sunset",
                        "#cf6ba9": "Super pink",
                        "#a83731": "Sweet Brown",
                        "#d2b48c": "Tan",
                        "#d99a6c": "Tan (Crayola)",
                        "#f28500": "Tangerine",
                        "#fb4d46": "Tart Orange",
                        "#483c32": "Taupe",
                        "#8b8589": "Taupe gray",
                        "#d0f0c0": "Tea green",
                        "#f88379": "Tea rose",
                        "#008080": "Teal",
                        "#367588": "Teal blue",
                        "#cf3476": "Telemagenta",
                        "#cd5700": "Tenné (tawny)",
                        "#e2725b": "Terra cotta",
                        "#d8bfd8": "Thistle",
                        "#fc89ac": "Tickle Me Pink",
                        "#0abab5": "Tiffany Blue",
                        "#dbd7d2": "Timberwolf",
                        "#eee600": "Titanium yellow",
                        "#ff6347": "Tomato",
                        "#00755e": "Tropical rain forest",
                        "#2d68c4": "True Blue",
                        "#1c05b3": "Trypan Blue",
                        "#3e8ede": "Tufts blue",
                        "#deaa88": "Tumbleweed",
                        "#40e0d0": "Turquoise",
                        "#00ffef": "Turquoise blue",
                        "#a0d6b4": "Turquoise green",
                        "#7c4848": "Tuscan red",
                        "#c09999": "Tuscany",
                        "#8a496b": "Twilight lavender",
                        "#66023c": "Tyrian purple",
                        "#0033aa": "UA blue",
                        "#d9004c": "UA red",
                        "#3f00ff": "Ultramarine",
                        "#4166f5": "Ultramarine blue",
                        "#ff6fff": "Ultra pink",
                        "#fc6c85": "Ultra red",
                        "#635147": "Umber",
                        "#ffddca": "Unbleached silk",
                        "#5b92e5": "United Nations blue",
                        "#7b1113": "UP maroon",
                        "#ae2029": "Upsdell red",
                        "#afdbf5": "Uranian blue",
                        "#004f98": "USAFA blue",
                        "#664228": "Van Dyke brown",
                        "#f3e5ab": "Vanilla",
                        "#f38fa9": "Vanilla ice",
                        "#c5b358": "Vegas gold",
                        "#c80815": "Venetian red",
                        "#43b3ae": "Verdigris",
                        "#d9381e": "Vermilion1",
                        "#8f00ff": "Violet",
                        "#7f00ff": "Violet (color wheel)",
                        "#963d7f": "Violet (crayola)",
                        "#8601af": "Violet (RYB)",
                        "#ee82ee": "Violet (web)",
                        "#324ab2": "Violet-blue",
                        "#766ec8": "Violet-blue (Crayola)",
                        "#f75394": "Violet-red",
                        "#40826d": "Viridian",
                        "#009698": "Viridian green",
                        "#9f1d35": "Vivid burgundy",
                        "#00ccff": "Vivid sky blue",
                        "#ffa089": "Vivid tangerine",
                        "#9f00ff": "Vivid violet",
                        "#ceff00": "Volt",
                        "#004242": "Warm black",
                        "#f5deb3": "Wheat",
                        "#ffffff": "White",
                        "#a2add0": "Wild blue yonder",
                        "#d470a2": "Wild orchid",
                        "#ff43a4": "Wild Strawberry",
                        "#a75502": "Windsor tan",
                        "#722f37": "Wine",
                        "#673147": "Wine dregs",
                        "#ff007c": "Winter Sky",
                        "#56887d": "Wintergreen Dream",
                        "#c9a0dc": "Wisteria",
                        "#c19a6b": "Wood brown",
                        "#eeed09": "Xanthic",
                        "#738678": "Xanadu",
                        "#0c020f": "Xiketic",
                        "#0f4d92": "Yale Blue",
                        "#ffff00": "Yellow",
                        "#fce883": "Yellow (Crayola)",
                        "#efcc00": "Yellow (Munsell)",
                        "#ffd300": "Yellow (NCS)",
                        "#fedf00": "Yellow (Pantone)",
                        "#fefe33": "Yellow (RYB)",
                        "#9acd32": "Yellow-green",
                        "#c5e384": "Yellow-green (Crayola)",
                        "#30b21a": "Yellow-green (Color Wheel)",
                        "#ffae42": "Yellow Orange",
                        "#ff9505": "Yellow Orange (Color Wheel)",
                        "#2e5090": "YInMn Blue",
                        "#0014a8": "Zaffre",
                        "#39a78e": "Zomp"}
    
    def closest_colour(self, requested_colour):
        """
        Credits go to fraxel: https://stackoverflow.com/a/9694246/4040598
        :param requested_colour: #rrggbb string
        :return: The closest colour name as a string
        """
        requested_colour = hex_to_rgb(requested_colour)
        min_colours = {}
        for key, name in self.colours.items():
            r_c, g_c, b_c = hex_to_rgb(key)
            rd = (r_c - requested_colour[0]) ** 2
            gd = (g_c - requested_colour[1]) ** 2
            bd = (b_c - requested_colour[2]) ** 2
            min_colours[(rd + gd + bd)] = name

        c_name = min_colours[min(min_colours.keys())]
        hex_value = None
        for key, name in self.colours.items():
            if name == c_name:
                hex_value = ' {}'.format(key)
                break

        return c_name + hex_value

    def get_colour_name(self, requested_colour):
        try:
            closest_name = actual_name = self.colours[requested_colour]
        except KeyError:
            closest_name = self.closest_colour(requested_colour)
            actual_name = None

        return actual_name, closest_name

