﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/geo-places/GeoPlacesRequest.h>
#include <aws/geo-places/GeoPlaces_EXPORTS.h>
#include <aws/geo-places/model/SearchTextAdditionalFeature.h>
#include <aws/geo-places/model/SearchTextFilter.h>
#include <aws/geo-places/model/SearchTextIntendedUse.h>

#include <utility>

namespace Aws {
namespace Http {
class URI;
}  // namespace Http
namespace GeoPlaces {
namespace Model {

/**
 */
class SearchTextRequest : public GeoPlacesRequest {
 public:
  AWS_GEOPLACES_API SearchTextRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "SearchText"; }

  AWS_GEOPLACES_API Aws::String SerializePayload() const override;

  AWS_GEOPLACES_API void AddQueryStringParameters(Aws::Http::URI& uri) const override;

  ///@{
  /**
   * <p>The free-form text query to match addresses against. This is usually a
   * partially typed address from an end user in an address box or form.</p>
   * <p>Exactly one of the following fields must be set: <code>QueryText</code> or
   * <code>QueryId</code>.</p>
   */
  inline const Aws::String& GetQueryText() const { return m_queryText; }
  inline bool QueryTextHasBeenSet() const { return m_queryTextHasBeenSet; }
  template <typename QueryTextT = Aws::String>
  void SetQueryText(QueryTextT&& value) {
    m_queryTextHasBeenSet = true;
    m_queryText = std::forward<QueryTextT>(value);
  }
  template <typename QueryTextT = Aws::String>
  SearchTextRequest& WithQueryText(QueryTextT&& value) {
    SetQueryText(std::forward<QueryTextT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The query Id returned by the suggest API. If passed in the request, the
   * SearchText API will preform a SearchText query with the improved query terms for
   * the original query made to the suggest API.</p>  <p>Exactly one of the
   * following fields must be set: <code>QueryText</code> or
   * <code>QueryId</code>.</p>
   */
  inline const Aws::String& GetQueryId() const { return m_queryId; }
  inline bool QueryIdHasBeenSet() const { return m_queryIdHasBeenSet; }
  template <typename QueryIdT = Aws::String>
  void SetQueryId(QueryIdT&& value) {
    m_queryIdHasBeenSet = true;
    m_queryId = std::forward<QueryIdT>(value);
  }
  template <typename QueryIdT = Aws::String>
  SearchTextRequest& WithQueryId(QueryIdT&& value) {
    SetQueryId(std::forward<QueryIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An optional limit for the number of results returned in a single call.</p>
   * <p>Default value: 20</p>
   */
  inline int GetMaxResults() const { return m_maxResults; }
  inline bool MaxResultsHasBeenSet() const { return m_maxResultsHasBeenSet; }
  inline void SetMaxResults(int value) {
    m_maxResultsHasBeenSet = true;
    m_maxResults = value;
  }
  inline SearchTextRequest& WithMaxResults(int value) {
    SetMaxResults(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The position, in longitude and latitude, that the results should be close to.
   * Typically, place results returned are ranked higher the closer they are to this
   * position. Stored in <code>[lng, lat]</code> and in the WGS 84 format.</p>
   * <p>Exactly one of the following fields must be set: <code>BiasPosition</code>,
   * <code>Filter.BoundingBox</code>, or <code>Filter.Circle</code>.</p>
   */
  inline const Aws::Vector<double>& GetBiasPosition() const { return m_biasPosition; }
  inline bool BiasPositionHasBeenSet() const { return m_biasPositionHasBeenSet; }
  template <typename BiasPositionT = Aws::Vector<double>>
  void SetBiasPosition(BiasPositionT&& value) {
    m_biasPositionHasBeenSet = true;
    m_biasPosition = std::forward<BiasPositionT>(value);
  }
  template <typename BiasPositionT = Aws::Vector<double>>
  SearchTextRequest& WithBiasPosition(BiasPositionT&& value) {
    SetBiasPosition(std::forward<BiasPositionT>(value));
    return *this;
  }
  inline SearchTextRequest& AddBiasPosition(double value) {
    m_biasPositionHasBeenSet = true;
    m_biasPosition.push_back(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A structure which contains a set of inclusion/exclusion properties that
   * results must possess in order to be returned as a result.</p>
   */
  inline const SearchTextFilter& GetFilter() const { return m_filter; }
  inline bool FilterHasBeenSet() const { return m_filterHasBeenSet; }
  template <typename FilterT = SearchTextFilter>
  void SetFilter(FilterT&& value) {
    m_filterHasBeenSet = true;
    m_filter = std::forward<FilterT>(value);
  }
  template <typename FilterT = SearchTextFilter>
  SearchTextRequest& WithFilter(FilterT&& value) {
    SetFilter(std::forward<FilterT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of optional additional parameters, such as time zone, that can be
   * requested for each result.</p>
   */
  inline const Aws::Vector<SearchTextAdditionalFeature>& GetAdditionalFeatures() const { return m_additionalFeatures; }
  inline bool AdditionalFeaturesHasBeenSet() const { return m_additionalFeaturesHasBeenSet; }
  template <typename AdditionalFeaturesT = Aws::Vector<SearchTextAdditionalFeature>>
  void SetAdditionalFeatures(AdditionalFeaturesT&& value) {
    m_additionalFeaturesHasBeenSet = true;
    m_additionalFeatures = std::forward<AdditionalFeaturesT>(value);
  }
  template <typename AdditionalFeaturesT = Aws::Vector<SearchTextAdditionalFeature>>
  SearchTextRequest& WithAdditionalFeatures(AdditionalFeaturesT&& value) {
    SetAdditionalFeatures(std::forward<AdditionalFeaturesT>(value));
    return *this;
  }
  inline SearchTextRequest& AddAdditionalFeatures(SearchTextAdditionalFeature value) {
    m_additionalFeaturesHasBeenSet = true;
    m_additionalFeatures.push_back(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of <a href="https://en.wikipedia.org/wiki/IETF_language_tag">BCP
   * 47</a> compliant language codes for the results to be rendered in. If there is
   * no data for the result in the requested language, data will be returned in the
   * default language for the entry.</p>
   */
  inline const Aws::String& GetLanguage() const { return m_language; }
  inline bool LanguageHasBeenSet() const { return m_languageHasBeenSet; }
  template <typename LanguageT = Aws::String>
  void SetLanguage(LanguageT&& value) {
    m_languageHasBeenSet = true;
    m_language = std::forward<LanguageT>(value);
  }
  template <typename LanguageT = Aws::String>
  SearchTextRequest& WithLanguage(LanguageT&& value) {
    SetLanguage(std::forward<LanguageT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The alpha-2 or alpha-3 character code for the political view of a country.
   * The political view applies to the results of the request to represent unresolved
   * territorial claims through the point of view of the specified country.</p>
   */
  inline const Aws::String& GetPoliticalView() const { return m_politicalView; }
  inline bool PoliticalViewHasBeenSet() const { return m_politicalViewHasBeenSet; }
  template <typename PoliticalViewT = Aws::String>
  void SetPoliticalView(PoliticalViewT&& value) {
    m_politicalViewHasBeenSet = true;
    m_politicalView = std::forward<PoliticalViewT>(value);
  }
  template <typename PoliticalViewT = Aws::String>
  SearchTextRequest& WithPoliticalView(PoliticalViewT&& value) {
    SetPoliticalView(std::forward<PoliticalViewT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Indicates if the results will be stored. Defaults to <code>SingleUse</code>,
   * if left empty.</p>  <p>Storing the response of an SearchText query is
   * required to comply with service terms, but charged at a higher cost per request.
   * Please review the <a href="https://aws.amazon.com/location/sla/">user
   * agreement</a> and <a href="https://aws.amazon.com/location/pricing/">service
   * pricing structure</a> to determine the correct setting for your use case.</p>
   *
   */
  inline SearchTextIntendedUse GetIntendedUse() const { return m_intendedUse; }
  inline bool IntendedUseHasBeenSet() const { return m_intendedUseHasBeenSet; }
  inline void SetIntendedUse(SearchTextIntendedUse value) {
    m_intendedUseHasBeenSet = true;
    m_intendedUse = value;
  }
  inline SearchTextRequest& WithIntendedUse(SearchTextIntendedUse value) {
    SetIntendedUse(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>If <code>nextToken</code> is returned, there are more results available. The
   * value of <code>nextToken</code> is a unique pagination token for each page. </p>
   */
  inline const Aws::String& GetNextToken() const { return m_nextToken; }
  inline bool NextTokenHasBeenSet() const { return m_nextTokenHasBeenSet; }
  template <typename NextTokenT = Aws::String>
  void SetNextToken(NextTokenT&& value) {
    m_nextTokenHasBeenSet = true;
    m_nextToken = std::forward<NextTokenT>(value);
  }
  template <typename NextTokenT = Aws::String>
  SearchTextRequest& WithNextToken(NextTokenT&& value) {
    SetNextToken(std::forward<NextTokenT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Optional: The API key to be used for authorization. Either an API key or
   * valid SigV4 signature must be provided when making a request.</p>
   */
  inline const Aws::String& GetKey() const { return m_key; }
  inline bool KeyHasBeenSet() const { return m_keyHasBeenSet; }
  template <typename KeyT = Aws::String>
  void SetKey(KeyT&& value) {
    m_keyHasBeenSet = true;
    m_key = std::forward<KeyT>(value);
  }
  template <typename KeyT = Aws::String>
  SearchTextRequest& WithKey(KeyT&& value) {
    SetKey(std::forward<KeyT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_queryText;

  Aws::String m_queryId;

  int m_maxResults{0};

  Aws::Vector<double> m_biasPosition;

  SearchTextFilter m_filter;

  Aws::Vector<SearchTextAdditionalFeature> m_additionalFeatures;

  Aws::String m_language;

  Aws::String m_politicalView;

  SearchTextIntendedUse m_intendedUse{SearchTextIntendedUse::NOT_SET};

  Aws::String m_nextToken;

  Aws::String m_key;
  bool m_queryTextHasBeenSet = false;
  bool m_queryIdHasBeenSet = false;
  bool m_maxResultsHasBeenSet = false;
  bool m_biasPositionHasBeenSet = false;
  bool m_filterHasBeenSet = false;
  bool m_additionalFeaturesHasBeenSet = false;
  bool m_languageHasBeenSet = false;
  bool m_politicalViewHasBeenSet = false;
  bool m_intendedUseHasBeenSet = false;
  bool m_nextTokenHasBeenSet = false;
  bool m_keyHasBeenSet = false;
};

}  // namespace Model
}  // namespace GeoPlaces
}  // namespace Aws
