use std::collections::HashSet;

use crate::article_list::ArticleListModel;
use crate::content_page::ArticleListMode;
use crate::gobject_models::{GSidebarSelection, SidebarSelectionType};
use crate::sidebar::FeedListItemID;
use crate::undo_action::UndoAction;
use crate::util::DateUtil;
use crate::util::constants::ARTICLE_LIST_PAGE_SIZE;
use chrono::{DateTime, TimeDelta, Utc};
use news_flash::NewsFlash;
use news_flash::error::NewsFlashError;
use news_flash::models::{
    Article, ArticleFilter, ArticleID, ArticleOrder, CategoryID, Feed, FeedID, Marked, OrderBy, Read, Tag, TagID,
    Tagging,
};

#[derive(Debug, Default, Clone, Copy, PartialEq, Eq)]
pub enum LoadType {
    New,
    #[default]
    Update,
    Extend,
}

#[derive(Debug, Default, Clone)]
pub struct ArticleListLoader {
    load_type: LoadType,
    hide_future_articles: bool,
    search_term: Option<String>,
    order: ArticleOrder,
    order_by: OrderBy,
    mode: ArticleListMode,
    page_size: Option<i64>,
    list_height: i64,
    selected_feed: Option<FeedID>,
    selected_category: Option<CategoryID>,
    selected_tag: Option<TagID>,
    is_today: bool,
    undo_action: Option<UndoAction>,
    last_article_date: Option<DateTime<Utc>>,
    loaded_article_ids: Vec<ArticleID>,
    restore_selected_id: Option<ArticleID>,
}

impl ArticleListLoader {
    pub fn is_type(&self, _type: LoadType) -> bool {
        self.load_type == _type
    }

    pub fn get_type(&self) -> LoadType {
        self.load_type
    }

    pub fn load_type(mut self, _type: LoadType) -> Self {
        self.load_type = _type;
        self
    }

    pub fn hide_furure_articles(mut self, hide_future_articles: bool) -> Self {
        self.hide_future_articles = hide_future_articles;
        self
    }

    pub fn search_term(mut self, search_term: Option<String>) -> Self {
        self.search_term = search_term;
        self
    }

    pub fn order(mut self, order: ArticleOrder) -> Self {
        self.order = order;
        self
    }

    pub fn order_by(mut self, order_by: OrderBy) -> Self {
        self.order_by = order_by;
        self
    }

    pub fn mode(mut self, mode: ArticleListMode) -> Self {
        self.mode = mode;
        self
    }

    pub fn page_size(mut self, page_size: Option<i64>) -> Self {
        self.page_size = page_size;
        self
    }

    pub fn list_height(mut self, list_height: i64) -> Self {
        tracing::debug!(%list_height);
        self.list_height = list_height;
        self
    }

    pub fn sidebar_selection(mut self, sidebar_selection: GSidebarSelection) -> Self {
        let (selected_feed, selected_category, selected_tag) = match &sidebar_selection.selection_type() {
            SidebarSelectionType::All | SidebarSelectionType::Today | SidebarSelectionType::None => (None, None, None),
            SidebarSelectionType::TagList => (None, None, Some(sidebar_selection.tag_id().into())),
            SidebarSelectionType::FeedList => match sidebar_selection.feedlist_id().as_ref() {
                FeedListItemID::Feed(mapping) => (Some(mapping.feed_id.clone()), None, None),
                FeedListItemID::Category(id) => (None, Some(id.clone()), None),
            },
        };

        self.selected_feed = selected_feed;
        self.selected_category = selected_category;
        self.selected_tag = selected_tag;
        self.is_today = sidebar_selection.selection_type() == SidebarSelectionType::Today;
        self
    }

    pub fn undo_action(mut self, undo_action: Option<UndoAction>) -> Self {
        self.undo_action = undo_action;
        self
    }

    pub fn last_article_date(mut self, last_article_date: Option<DateTime<Utc>>) -> Self {
        self.last_article_date = last_article_date;
        self
    }

    pub fn loaded_article_ids(mut self, loaded_article_ids: Vec<ArticleID>) -> Self {
        self.loaded_article_ids = loaded_article_ids;
        self
    }

    pub fn restore_selected_id(mut self, selected_id: Option<ArticleID>) -> Self {
        self.restore_selected_id = selected_id;
        self
    }

    pub fn load(self, news_flash: &NewsFlash) -> Result<ArticleListLoadResult, NewsFlashError> {
        match self.load_type {
            LoadType::New => self.load_new(news_flash),
            LoadType::Update => self.load_update(news_flash),
            LoadType::Extend => self.load_extend(news_flash),
        }
    }

    fn load_new(self, news_flash: &NewsFlash) -> Result<ArticleListLoadResult, NewsFlashError> {
        let page_size = self.calculate_page_size();
        let article_filter = self.build_article_filter(Some(page_size), None, None);
        let mut articles = news_flash.get_articles(article_filter)?;

        // if this is a new unread list load restore_selected_id
        if self.mode == ArticleListMode::Unread
            && let Some(selected_id) = self.restore_selected_id.as_ref()
            && let Ok(article) = news_flash.get_article(selected_id)
        {
            articles.push(article);
        }

        let load_result = self.load_result(news_flash, articles)?;
        Ok(load_result)
    }

    fn load_update(self, news_flash: &NewsFlash) -> Result<ArticleListLoadResult, NewsFlashError> {
        let page_size = self.calculate_page_size();

        let (limit, last_article_date) = if self.last_article_date.is_none() {
            // article list is empty: load default amount of articles to fill it
            (Some(page_size), None)
        } else {
            (None, self.last_article_date)
        };

        let (older_than, newer_than) = if let Some(last_article_date) = last_article_date {
            match self.order {
                // +/- 1s to not exclude last article in list
                ArticleOrder::NewestFirst => (None, Some(last_article_date - TimeDelta::try_seconds(1).unwrap())),
                ArticleOrder::OldestFirst => (Some(last_article_date + TimeDelta::try_seconds(1).unwrap()), None),
            }
        } else {
            (None, None)
        };

        let article_filter = self.build_article_filter(limit, older_than, newer_than);
        let mut articles = news_flash.get_articles(article_filter)?;

        let loaded_article_count = articles.len() as i64;
        if loaded_article_count < page_size
            && let Some(last_article_date) = self.last_article_date
        {
            // article list is not empty but also not filled all the way up to "page size"
            // -> load a few more
            let (older_than, newer_than) = match self.order {
                // newest first => we want articles older than the last in list
                ArticleOrder::NewestFirst => (Some(last_article_date), None),

                // newsest first => we want article newer than the last in list
                ArticleOrder::OldestFirst => (None, Some(last_article_date)),
            };
            let more_articles_filter = self.build_article_filter(Some(page_size), older_than, newer_than);

            let mut more_articles = news_flash.get_articles(more_articles_filter)?;
            articles.append(&mut more_articles);
        }

        // if this is an update of the same unread list also load the already loaded articles
        // so they wont be removed by the update & things like the thumbnail update
        if self.mode == ArticleListMode::Unread
            && let Ok(mut a) = news_flash.get_articles(ArticleFilter::ids(self.loaded_article_ids.clone()))
        {
            articles.append(&mut a);
        }

        let load_result = self.load_result(news_flash, articles)?;
        Ok(load_result)
    }

    fn load_extend(self, news_flash: &NewsFlash) -> Result<ArticleListLoadResult, NewsFlashError> {
        let (older_than, newer_than) = if let Some(last_article_date) = self.last_article_date {
            match self.order {
                // newest first => we want articles older than the last in list
                ArticleOrder::NewestFirst => (Some(last_article_date), None),

                // newsest first => we want article newer than the last in list
                ArticleOrder::OldestFirst => (None, Some(last_article_date)),
            }
        } else {
            (None, None)
        };

        let page_size = self.calculate_page_size();
        let article_filter = self.build_article_filter(Some(page_size), older_than, newer_than);

        let articles = news_flash.get_articles(article_filter)?;
        let load_result = self.load_result(news_flash, articles)?;
        Ok(load_result)
    }

    fn load_result(
        &self,
        news_flash: &NewsFlash,
        mut articles: Vec<Article>,
    ) -> Result<ArticleListLoadResult, NewsFlashError> {
        let (feeds, _feed_mappings) = news_flash.get_feeds()?;
        let (mut tags, mut taggings) = news_flash.get_tags()?;

        if let Some(UndoAction::DeleteTag(deleted_tag_id, _)) = &self.undo_action {
            tags.retain(|tag| &tag.tag_id != deleted_tag_id);
            taggings.retain(|tagging| &tagging.tag_id != deleted_tag_id);
        }

        let order = self.order;
        let order_by = self.order_by;

        articles.sort_by(|a, b| {
            let ordering = if order_by == OrderBy::Published {
                a.date.cmp(&b.date)
            } else {
                let a_date = a.updated.as_ref().unwrap_or(&a.date);
                let b_date = b.updated.as_ref().unwrap_or(&b.date);

                a_date.cmp(b_date)
            };
            if order == ArticleOrder::NewestFirst {
                ordering.reverse()
            } else {
                ordering
            }
        });

        Ok(ArticleListLoadResult::new(
            self.load_type,
            articles,
            feeds,
            tags,
            taggings,
        ))
    }

    fn build_article_filter(
        &self,
        limit: Option<i64>,
        older_than: Option<DateTime<Utc>>,
        newer_than: Option<DateTime<Utc>>,
    ) -> ArticleFilter {
        // mutate older_than to hide articles in the future
        let older_than = self.should_hide_future_articles(older_than);

        let unread = match self.mode {
            ArticleListMode::All | ArticleListMode::Marked => None,
            ArticleListMode::Unread => Some(Read::Unread),
        };
        let marked = match self.mode {
            ArticleListMode::All | ArticleListMode::Unread => None,
            ArticleListMode::Marked => Some(Marked::Marked),
        };

        let (older_than, newer_than) = if self.is_today {
            let (start, end) = DateUtil::today_start_end();

            let newer_than = newer_than
                .map(|date| if date < start { start } else { date })
                .unwrap_or(start);
            let older_than = older_than
                .map(|date| if date > end { end } else { date })
                .unwrap_or(end);
            (Some(older_than), Some(newer_than))
        } else {
            (older_than, newer_than)
        };

        let (feed_blacklist, category_blacklist) = self.load_articles_blacklist();

        ArticleFilter {
            limit,
            offset: None,
            order: Some(self.order),
            order_by: Some(self.order_by),
            unread,
            marked,
            feeds: self.selected_feed.clone().map(|f| vec![f]),
            feed_blacklist,
            categories: self.selected_category.clone().map(|c| vec![c]),
            category_blacklist,
            tags: self.selected_tag.clone().map(|t| vec![t]),
            ids: None,
            newer_than,
            older_than,
            synced_after: None,
            synced_before: None,
            search_term: self.search_term.clone(),
        }
    }

    fn load_articles_blacklist(&self) -> (Option<Vec<FeedID>>, Option<Vec<CategoryID>>) {
        let mut feed_blacklist = Vec::new();
        let mut category_blacklist = Vec::new();

        if let Some(undo_action) = &self.undo_action {
            match undo_action {
                UndoAction::DeleteFeed(feed_id, _label) => feed_blacklist.push(feed_id.clone()),
                UndoAction::DeleteCategory(category_id, _label) => category_blacklist.push(category_id.clone()),
                _ => {}
            }
        }

        let feed_blacklist = if feed_blacklist.is_empty() {
            None
        } else {
            Some(feed_blacklist)
        };
        let category_blacklist = if category_blacklist.is_empty() {
            None
        } else {
            Some(category_blacklist)
        };

        (feed_blacklist, category_blacklist)
    }

    fn should_hide_future_articles(&self, older_than: Option<DateTime<Utc>>) -> Option<DateTime<Utc>> {
        if self.hide_future_articles {
            if let Some(older_than) = older_than {
                if older_than < Utc::now() {
                    Some(older_than)
                } else {
                    Some(Utc::now())
                }
            } else {
                Some(Utc::now())
            }
        } else {
            older_than
        }
    }

    fn calculate_page_size(&self) -> i64 {
        let min_page_size = if self.list_height > 0 {
            let num = self.list_height as f64 / 108.0;
            num.ceil() as i64 + 3
        } else {
            ARTICLE_LIST_PAGE_SIZE
        };

        let page_size = if let Some(page_size) = self.page_size {
            page_size.max(min_page_size)
        } else {
            min_page_size
        };

        tracing::debug!(%page_size);
        page_size
    }
}

#[derive(Debug, Default, Clone)]
pub struct ArticleListLoadResult {
    load_type: LoadType,
    articles: Vec<Article>,
    feeds: Vec<Feed>,
    tags: Vec<Tag>,
    taggings: Vec<Tagging>,
}

impl ArticleListLoadResult {
    pub(self) fn new(
        load_type: LoadType,
        articles: Vec<Article>,
        feeds: Vec<Feed>,
        tags: Vec<Tag>,
        taggings: Vec<Tagging>,
    ) -> Self {
        Self {
            load_type,
            articles,
            feeds,
            tags,
            taggings,
        }
    }

    pub fn load_type(&self) -> LoadType {
        self.load_type
    }

    pub fn build_list_model(self) -> ArticleListModel {
        let articles = self
            .articles
            .into_iter()
            .map(|article| {
                let feed = self.feeds.iter().find(|f| f.feed_id == article.feed_id);
                let taggings: HashSet<&TagID> = self
                    .taggings
                    .iter()
                    .filter(|t| t.article_id == article.article_id)
                    .map(|t| &t.tag_id)
                    .collect();
                let tags = self
                    .tags
                    .iter()
                    .filter(|t| taggings.contains(&t.tag_id))
                    .cloned()
                    .collect::<Vec<_>>();

                (article, feed, tags)
            })
            .collect();

        let list_model = ArticleListModel::new();
        list_model.add(articles);

        list_model
    }
}
