/* plugin-jhbuild-documentation-provider.c
 *
 * Copyright 2025 Christian Hergert <chergert@redhat.com>
 *
 * This library is free software; you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License as
 * published by the Free Software Foundation; either version 2.1 of the
 * License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * SPDX-License-Identifier: LGPL-2.1-or-later
 */

#include "config.h"

#include "plugin-jhbuild-documentation-provider.h"
#include "plugin-jhbuild-sdk.h"

struct _PluginJhbuildDocumentationProvider
{
  FoundryDocumentationProvider parent_instance;
  GListStore *roots;
};

G_DEFINE_FINAL_TYPE (PluginJhbuildDocumentationProvider, plugin_jhbuild_documentation_provider, FOUNDRY_TYPE_DOCUMENTATION_PROVIDER)

static DexFuture *
plugin_jhbuild_documentation_provider_load_fiber (gpointer user_data)
{
  PluginJhbuildDocumentationProvider *self = user_data;
  g_autoptr(FoundrySdkManager) sdk_manager = NULL;
  g_autoptr(FoundryContext) context = NULL;
  g_autoptr(FoundrySdk) sdk = NULL;

  g_assert (PLUGIN_IS_JHBUILD_DOCUMENTATION_PROVIDER (self));

  if ((context = foundry_contextual_dup_context (FOUNDRY_CONTEXTUAL (self))) &&
      (sdk_manager = foundry_context_dup_sdk_manager (context)) &&
      dex_await (foundry_service_when_ready (FOUNDRY_SERVICE (sdk_manager)), NULL) &&
      (sdk = dex_await_object (foundry_sdk_manager_find_by_id (sdk_manager, "jhbuild"), NULL)) &&
      PLUGIN_IS_JHBUILD_SDK (sdk))
    {
      g_autoptr(FoundryDocumentationRoot) root = NULL;
      g_autofree char *install_prefix = plugin_jhbuild_sdk_dup_install_prefix (PLUGIN_JHBUILD_SDK (sdk));
      g_autoptr(GListStore) directories = g_list_store_new (G_TYPE_FILE);
      g_autoptr(GIcon) icon = g_themed_icon_new ("utilities-terminal-symbolic");
      g_autoptr(GFile) doc = g_file_new_build_filename (install_prefix, "share", "doc", NULL);
      g_autoptr(GFile) gtk_doc = g_file_new_build_filename (install_prefix, "share", "gtk-doc", "html", NULL);

      g_debug ("Discovered documentation directory at \"%s\"", g_file_peek_path (doc));
      g_debug ("Discovered documentation directory at \"%s\"", g_file_peek_path (gtk_doc));

      g_list_store_append (directories, doc);
      g_list_store_append (directories, gtk_doc);

      root = foundry_documentation_root_new ("jhbuild", "JHBuild", NULL, icon, G_LIST_MODEL (directories));

      g_list_store_append (self->roots, root);
    }

  return dex_future_new_true ();
}

static DexFuture *
plugin_jhbuild_documentation_provider_load (FoundryDocumentationProvider *provider)
{
  PluginJhbuildDocumentationProvider *self = (PluginJhbuildDocumentationProvider *)provider;

  g_assert (PLUGIN_IS_JHBUILD_DOCUMENTATION_PROVIDER (self));

  self->roots = g_list_store_new (FOUNDRY_TYPE_DOCUMENTATION_ROOT);

  return dex_scheduler_spawn (NULL, 0,
                              plugin_jhbuild_documentation_provider_load_fiber,
                              g_object_ref (self),
                              g_object_unref);
}

static DexFuture *
plugin_jhbuild_documentation_provider_unload (FoundryDocumentationProvider *provider)
{
  PluginJhbuildDocumentationProvider *self = (PluginJhbuildDocumentationProvider *)provider;

  g_assert (PLUGIN_IS_JHBUILD_DOCUMENTATION_PROVIDER (self));

  g_clear_object (&self->roots);

  return dex_future_new_true ();
}

static GListModel *
plugin_jhbuild_documentation_provider_list_roots (FoundryDocumentationProvider *provider)
{
  PluginJhbuildDocumentationProvider *self = (PluginJhbuildDocumentationProvider *)provider;

  g_assert (FOUNDRY_IS_DOCUMENTATION_PROVIDER (self));

  return g_object_ref (G_LIST_MODEL (self->roots));
}

static void
plugin_jhbuild_documentation_provider_class_init (PluginJhbuildDocumentationProviderClass *klass)
{
  FoundryDocumentationProviderClass *provider_class = FOUNDRY_DOCUMENTATION_PROVIDER_CLASS (klass);

  provider_class->load = plugin_jhbuild_documentation_provider_load;
  provider_class->unload = plugin_jhbuild_documentation_provider_unload;
  provider_class->list_roots = plugin_jhbuild_documentation_provider_list_roots;
}

static void
plugin_jhbuild_documentation_provider_init (PluginJhbuildDocumentationProvider *self)
{
}
