// Copyright 2019 Drone IO, Inc.
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//      http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

package web

import (
	"bytes"
	"crypto/md5"
	"fmt"
	"net/http"
	"time"

	"github.com/drone/drone-ui/dist"
	"github.com/drone/drone/core"
)

func HandleIndex(host string, session core.Session, license core.LicenseService) http.HandlerFunc {
	return func(rw http.ResponseWriter, r *http.Request) {
		user, _ := session.Get(r)
		if user == nil && r.URL.Path == "/" {
			http.Redirect(rw, r, "/welcome", 303)
			return
		}

		out := dist.MustLookup("/index.html")
		ctx := r.Context()

		if ok, _ := license.Exceeded(ctx); ok {
			out = bytes.Replace(out, head, exceeded, -1)
		} else if license.Expired(ctx) {
			out = bytes.Replace(out, head, expired, -1)
		}
		rw.Header().Set("Content-Type", "text/html; charset=UTF-8")
		rw.Write(out)
	}
}

var (
	head     = []byte(`<head>`)
	expired  = []byte(`<head><script>window.LICENSE_EXPIRED=true</script>`)
	exceeded = []byte(`<head><script>window.LICENSE_LIMIT_EXCEEDED=true</script>`)
)

func setupCache(h http.Handler) http.Handler {
	data := []byte(time.Now().String())
	etag := fmt.Sprintf("%x", md5.Sum(data))

	return http.HandlerFunc(
		func(w http.ResponseWriter, r *http.Request) {
			w.Header().Set("Cache-Control", "public, max-age=31536000")
			w.Header().Del("Expires")
			w.Header().Del("Pragma")
			w.Header().Set("ETag", etag)
			h.ServeHTTP(w, r)
		},
	)
}

// func userFromSession(r *http.Request, users core.UserStore, secret string) *core.User {
// 	cookie, err := r.Cookie("_session_")
// 	if err != nil {
// 		return nil
// 	}
// 	login := authcookie.Login(cookie.Value, []byte(secret))
// 	if login == "" {
// 		return nil
// 	}
// 	user, err := users.FindLogin(r.Context(), login)
// 	if err != nil {
// 		return nil
// 	}
// 	return user
// }

// var tmpl = mustCreateTemplate(
// 	string(dist.MustLookup("/index.html")),
// )

// // default func map with json parser.
// var funcMap = template.FuncMap{
// 	"json": func(v interface{}) template.JS {
// 		a, _ := json.Marshal(v)
// 		return template.JS(a)
// 	},
// }

// // helper function creates a new template from the text string.
// func mustCreateTemplate(text string) *template.Template {
// 	templ, err := createTemplate(text)
// 	if err != nil {
// 		panic(err)
// 	}
// 	return templ
// }

// // helper function creates a new template from the text string.
// func createTemplate(text string) (*template.Template, error) {
// 	templ, err := template.New("_").Funcs(funcMap).Parse(partials)
// 	if err != nil {
// 		return nil, err
// 	}
// 	return templ.Parse(
// 		injectPartials(text),
// 	)
// }

// // helper function that parses the html file and injects
// // named partial templates.
// func injectPartials(s string) string {
// 	w := new(bytes.Buffer)
// 	r := bytes.NewBufferString(s)
// 	t := html.NewTokenizer(r)
// 	for {
// 		tt := t.Next()
// 		if tt == html.ErrorToken {
// 			break
// 		}
// 		if tt == html.CommentToken {
// 			txt := string(t.Text())
// 			txt = strings.TrimSpace(txt)
// 			seg := strings.Split(txt, ":")
// 			if len(seg) == 2 && seg[0] == "drone" {
// 				fmt.Fprintf(w, "{{ template %q . }}", seg[1])
// 				continue
// 			}
// 		}
// 		w.Write(t.Raw())
// 	}
// 	return w.String()
// }

// const partials = `
// {{define "user"}}
// {{ if .user }}
// <script>
// 	window.DRONE_USER = {{ json .user }};
// 	window.DRONE_SYNC = {{ .syncing }};
// </script>
// {{ end }}
// {{end}}
// {{define "csrf"}}
// {{ if .csrf -}}
// <script>
// 	window.DRONE_CSRF = "{{ .csrf }}"
// </script>
// {{- end }}
// {{end}}
// {{define "version"}}
// 	<meta name="version" content="{{ .version }}">
// {{end}}
// {{define "docs"}}
// {{ if .docs -}}
// <script>
// 	window.DRONE_DOCS = "{{ .docs }}"
// </script>
// {{- end }}
// {{end}}
// `

var landingPage = `
`
