// ===========================================================================
//
//                            PUBLIC DOMAIN NOTICE
//            National Center for Biotechnology Information (NCBI)
//
//  This software/database is a "United States Government Work" under the
//  terms of the United States Copyright Act. It was written as part of
//  the author's official duties as a United States Government employee and
//  thus cannot be copyrighted. This software/database is freely available
//  to the public for use. The National Library of Medicine and the U.S.
//  Government do not place any restriction on its use or reproduction.
//  We would, however, appreciate having the NCBI and the author cited in
//  any work or product based on this material.
//
//  Although all reasonable efforts have been taken to ensure the accuracy
//  and reliability of the software and data, the NLM and the U.S.
//  Government do not and cannot warrant the performance or results that
//  may be obtained by using this software or data. The NLM and the U.S.
//  Government disclaim all warranties, express or implied, including
//  warranties of performance, merchantability or fitness for any particular
//  purpose.
//
// ===========================================================================
//
// File Name:  clean.go
//
// Author:  Jonathan Kans
//
// ==========================================================================

package eutils

import (
	"bufio"
	"os"
	"path/filepath"
	"strconv"
	"strings"
	"sync"
	"unicode/utf8"
)

// dynamically-loaded Unicode-to-ASCII rune maps
// asciiRunes value ranges include 0000-2092, 20A0-303A, 3041-33FF, and FB00-FFFD
// extraRunes include CJK (Chinese, Japanese, and Korean) Unified Ideograph Extensions, and Private Use Surrogates
var (
	asciiRunes map[rune]string
	extraRunes map[rune]string
)

// reencodes < and > to &lt and &gt, and & to &amp
var rfix *strings.Replacer

// Descriptions in comments taken from https://www.ssec.wisc.edu/~tomw/java/unicode.html

// runes are shown in Unicode Hexadecimal format

var germanCapitals = map[rune]string{
	0x0028: "(",  // LEFT PARENTHESIS
	0x0029: ")",  // RIGHT PARENTHESIS
	0x002D: "-",  // HYPHEN-MINUS
	0x0041: "A",  // LATIN CAPITAL LETTER A
	0x0042: "B",  // LATIN CAPITAL LETTER B
	0x0043: "C",  // LATIN CAPITAL LETTER C
	0x0044: "D",  // LATIN CAPITAL LETTER D
	0x0045: "E",  // LATIN CAPITAL LETTER E
	0x0046: "F",  // LATIN CAPITAL LETTER F
	0x0047: "G",  // LATIN CAPITAL LETTER G
	0x0048: "H",  // LATIN CAPITAL LETTER H
	0x0049: "I",  // LATIN CAPITAL LETTER I
	0x004A: "J",  // LATIN CAPITAL LETTER J
	0x004B: "K",  // LATIN CAPITAL LETTER K
	0x004C: "L",  // LATIN CAPITAL LETTER L
	0x004D: "M",  // LATIN CAPITAL LETTER M
	0x004E: "N",  // LATIN CAPITAL LETTER N
	0x004F: "O",  // LATIN CAPITAL LETTER O
	0x0050: "P",  // LATIN CAPITAL LETTER P
	0x0051: "Q",  // LATIN CAPITAL LETTER Q
	0x0052: "R",  // LATIN CAPITAL LETTER R
	0x0053: "S",  // LATIN CAPITAL LETTER S
	0x0054: "T",  // LATIN CAPITAL LETTER T
	0x0055: "U",  // LATIN CAPITAL LETTER U
	0x0056: "V",  // LATIN CAPITAL LETTER V
	0x0057: "W",  // LATIN CAPITAL LETTER W
	0x0058: "X",  // LATIN CAPITAL LETTER X
	0x0059: "Y",  // LATIN CAPITAL LETTER Y
	0x005A: "Z",  // LATIN CAPITAL LETTER Z
	0x00C4: "A",  // LATIN CAPITAL LETTER A WITH DIAERESIS
	0x00D6: "O",  // LATIN CAPITAL LETTER O WITH DIAERESIS
	0x00DC: "U",  // LATIN CAPITAL LETTER U WITH DIAERESIS
	0x1E9E: "SS", // LATIN CAPITAL LETTER SHARP S
}

var germanRunes = map[rune]string{
	0x0028: "(",  // LEFT PARENTHESIS
	0x0029: ")",  // RIGHT PARENTHESIS
	0x002D: "-",  // HYPHEN-MINUS
	0x0041: "A",  // LATIN CAPITAL LETTER A
	0x0042: "B",  // LATIN CAPITAL LETTER B
	0x0043: "C",  // LATIN CAPITAL LETTER C
	0x0044: "D",  // LATIN CAPITAL LETTER D
	0x0045: "E",  // LATIN CAPITAL LETTER E
	0x0046: "F",  // LATIN CAPITAL LETTER F
	0x0047: "G",  // LATIN CAPITAL LETTER G
	0x0048: "H",  // LATIN CAPITAL LETTER H
	0x0049: "I",  // LATIN CAPITAL LETTER I
	0x004A: "J",  // LATIN CAPITAL LETTER J
	0x004B: "K",  // LATIN CAPITAL LETTER K
	0x004C: "L",  // LATIN CAPITAL LETTER L
	0x004D: "M",  // LATIN CAPITAL LETTER M
	0x004E: "N",  // LATIN CAPITAL LETTER N
	0x004F: "O",  // LATIN CAPITAL LETTER O
	0x0050: "P",  // LATIN CAPITAL LETTER P
	0x0051: "Q",  // LATIN CAPITAL LETTER Q
	0x0052: "R",  // LATIN CAPITAL LETTER R
	0x0053: "S",  // LATIN CAPITAL LETTER S
	0x0054: "T",  // LATIN CAPITAL LETTER T
	0x0055: "U",  // LATIN CAPITAL LETTER U
	0x0056: "V",  // LATIN CAPITAL LETTER V
	0x0057: "W",  // LATIN CAPITAL LETTER W
	0x0058: "X",  // LATIN CAPITAL LETTER X
	0x0059: "Y",  // LATIN CAPITAL LETTER Y
	0x005A: "Z",  // LATIN CAPITAL LETTER Z
	0x0061: "a",  // LATIN SMALL LETTER A
	0x0062: "b",  // LATIN SMALL LETTER B
	0x0063: "c",  // LATIN SMALL LETTER C
	0x0064: "d",  // LATIN SMALL LETTER D
	0x0065: "e",  // LATIN SMALL LETTER E
	0x0066: "f",  // LATIN SMALL LETTER F
	0x0067: "g",  // LATIN SMALL LETTER G
	0x0068: "h",  // LATIN SMALL LETTER H
	0x0069: "i",  // LATIN SMALL LETTER I
	0x006A: "j",  // LATIN SMALL LETTER J
	0x006B: "k",  // LATIN SMALL LETTER K
	0x006C: "l",  // LATIN SMALL LETTER L
	0x006D: "m",  // LATIN SMALL LETTER M
	0x006E: "n",  // LATIN SMALL LETTER N
	0x006F: "o",  // LATIN SMALL LETTER O
	0x0070: "p",  // LATIN SMALL LETTER P
	0x0071: "q",  // LATIN SMALL LETTER Q
	0x0072: "r",  // LATIN SMALL LETTER R
	0x0073: "s",  // LATIN SMALL LETTER S
	0x0074: "t",  // LATIN SMALL LETTER T
	0x0075: "u",  // LATIN SMALL LETTER U
	0x0076: "v",  // LATIN SMALL LETTER V
	0x0077: "w",  // LATIN SMALL LETTER W
	0x0078: "x",  // LATIN SMALL LETTER X
	0x0079: "y",  // LATIN SMALL LETTER Y
	0x007A: "z",  // LATIN SMALL LETTER Z
	0x00DF: "ss", // LATIN SMALL LETTER SHARP S
	0x00E4: "a",  // LATIN SMALL LETTER A WITH DIAERESIS
	0x00F6: "o",  // LATIN SMALL LETTER O WITH DIAERESIS
	0x00FC: "u",  // LATIN SMALL LETTER U WITH DIAERESIS
	0x00C4: "A",  // LATIN CAPITAL LETTER A WITH DIAERESIS
	0x00D6: "O",  // LATIN CAPITAL LETTER O WITH DIAERESIS
	0x00DC: "U",  // LATIN CAPITAL LETTER U WITH DIAERESIS
	0x1E9E: "SS", // LATIN CAPITAL LETTER SHARP S
}

var germanVowels = map[rune]string{
	0x0041: "A", // LATIN CAPITAL LETTER A
	0x0045: "E", // LATIN CAPITAL LETTER E
	0x0049: "I", // LATIN CAPITAL LETTER I
	0x004F: "O", // LATIN CAPITAL LETTER O
	0x0055: "U", // LATIN CAPITAL LETTER U
	0x0061: "a", // LATIN SMALL LETTER A
	0x0065: "e", // LATIN SMALL LETTER E
	0x0069: "i", // LATIN SMALL LETTER I
	0x006F: "o", // LATIN SMALL LETTER O
	0x0075: "u", // LATIN SMALL LETTER U
	0x00E4: "a", // LATIN SMALL LETTER A WITH DIAERESIS
	0x00F6: "o", // LATIN SMALL LETTER O WITH DIAERESIS
	0x00FC: "u", // LATIN SMALL LETTER U WITH DIAERESIS
	0x00C4: "A", // LATIN CAPITAL LETTER A WITH DIAERESIS
	0x00D6: "O", // LATIN CAPITAL LETTER O WITH DIAERESIS
	0x00DC: "U", // LATIN CAPITAL LETTER U WITH DIAERESIS
}

var greekRunes = map[rune]string{
	0x0190: "Epsilon", // LATIN CAPITAL LETTER OPEN E
	0x025B: "epsilon", // LATIN SMALL LETTER OPEN E
	0x0391: "Alpha",   // GREEK CAPITAL LETTER ALPHA
	0x0392: "Beta",    // GREEK CAPITAL LETTER BETA
	0x0393: "Gamma",   // GREEK CAPITAL LETTER GAMMA
	0x0394: "Delta",   // GREEK CAPITAL LETTER DELTA
	0x0395: "Epsilon", // GREEK CAPITAL LETTER EPSILON
	0x0396: "Zeta",    // GREEK CAPITAL LETTER ZETA
	0x0397: "Eta",     // GREEK CAPITAL LETTER ETA
	0x0398: "Theta",   // GREEK CAPITAL LETTER THETA
	0x0399: "Iota",    // GREEK CAPITAL LETTER IOTA
	0x039A: "Kappa",   // GREEK CAPITAL LETTER KAPPA
	0x039B: "Lambda",  // GREEK CAPITAL LETTER LAMDA
	0x039C: "Mu",      // GREEK CAPITAL LETTER MU
	0x039D: "Nu",      // GREEK CAPITAL LETTER NU
	0x039E: "Xi",      // GREEK CAPITAL LETTER XI
	0x039F: "Omicron", // GREEK CAPITAL LETTER OMICRON
	0x03A0: "Pi",      // GREEK CAPITAL LETTER PI
	0x03A1: "Rho",     // GREEK CAPITAL LETTER RHO
	0x03A3: "Sigma",   // GREEK CAPITAL LETTER SIGMA
	0x03A4: "Tau",     // GREEK CAPITAL LETTER TAU
	0x03A5: "Upsilon", // GREEK CAPITAL LETTER UPSILON
	0x03A6: "Phi",     // GREEK CAPITAL LETTER PHI
	0x03A7: "Chi",     // GREEK CAPITAL LETTER CHI
	0x03A8: "Psi",     // GREEK CAPITAL LETTER PSI
	0x03A9: "Omega",   // GREEK CAPITAL LETTER OMEGA
	0x03B1: "alpha",   // GREEK SMALL LETTER ALPHA
	0x03B2: "beta",    // GREEK SMALL LETTER BETA
	0x03B3: "gamma",   // GREEK SMALL LETTER GAMMA
	0x03B4: "delta",   // GREEK SMALL LETTER DELTA
	0x03B5: "epsilon", // GREEK SMALL LETTER EPSILON
	0x03B6: "zeta",    // GREEK SMALL LETTER ZETA
	0x03B7: "eta",     // GREEK SMALL LETTER ETA
	0x03B8: "theta",   // GREEK SMALL LETTER THETA
	0x03B9: "iota",    // GREEK SMALL LETTER IOTA
	0x03BA: "kappa",   // GREEK SMALL LETTER KAPPA
	0x03BB: "lambda",  // GREEK SMALL LETTER LAMDA
	0x03BC: "mu",      // GREEK SMALL LETTER MU
	0x03BD: "nu",      // GREEK SMALL LETTER NU
	0x03BE: "xi",      // GREEK SMALL LETTER XI
	0x03BF: "omicron", // GREEK SMALL LETTER OMICRON
	0x03C0: "pi",      // GREEK SMALL LETTER PI
	0x03C1: "rho",     // GREEK SMALL LETTER RHO
	0x03C2: "sigma",   // GREEK SMALL LETTER FINAL SIGMA
	0x03C3: "sigma",   // GREEK SMALL LETTER SIGMA
	0x03C4: "tau",     // GREEK SMALL LETTER TAU
	0x03C5: "upsilon", // GREEK SMALL LETTER UPSILON
	0x03C6: "phi",     // GREEK SMALL LETTER PHI
	0x03C7: "chi",     // GREEK SMALL LETTER CHI
	0x03C8: "psi",     // GREEK SMALL LETTER PSI
	0x03C9: "omega",   // GREEK SMALL LETTER OMEGA
	0x03D0: "beta",    // GREEK BETA SYMBOL
	0x03D1: "theta",   // GREEK THETA SYMBOL
	0x03D5: "phi",     // GREEK PHI SYMBOL
	0x03D6: "pi",      // GREEK PI SYMBOL
	0x03F0: "kappa",   // GREEK KAPPA SYMBOL
	0x03F1: "rho",     // GREEK RHO SYMBOL
	0x03F5: "epsilon", // GREEK LUNATE EPSILON SYMBOL
	0x1D5D: "beta",    // GREEK MODIFIED SMALL LETTER BETA
	0x1D66: "beta",    // GREEK SUBSCRIPT SMALL LETTER BETA
}

var commonRunes = map[rune]string{
	0x007F: "",           // DELETE
	0x00A0: " ",          // NO-BREAK SPACE
	0x00A1: "!",          // INVERTED EXCLAMATION MARK
	0x00A2: " (cent) ",   // CENT SIGN
	0x00A3: " (pound) ",  // POUND SIGN
	0x00A4: " (curren) ", // CURRENCY SIGN
	0x00A5: " (yen) ",    // YEN SIGN
	0x00A6: "|",          // BROKEN BAR
	0x00A7: " SS ",       // SECTION SIGN
	0x00A8: "",           // DIAERESIS
	0x00A9: " (copy) ",   // COPYRIGHT SIGN
	0x00AA: "",           // FEMININE ORDINAL INDICATOR
	0x00AB: "<<",         // LEFT-POINTING DOUBLE ANGLE QUOTATION MAR
	0x00AC: "!",          // NOT SIGN
	0x00AD: "-",          // SOFT HYPHEN
	0x00AE: " (reg) ",    // REGISTERED SIGN
	0x00AF: "-",          // MACRON
	0x00B0: " degrees ",  // DEGREE SIGN
	0x00B1: "+/-",        // PLUS-MINUS SIGN
	0x00B2: "(2)",        // SUPERSCRIPT TWO
	0x00B3: "(3)",        // SUPERSCRIPT THREE
	0x00B4: "'",          // ACUTE ACCENT
	0x00B5: " micro ",    // MICRO SIGN
	0x00B6: " (p) ",      // PILCROW SIGN
	0x00B7: ".",          // MIDDLE DOT
	0x00B8: "",           // CEDILLA
	0x00B9: "(1)",        // SUPERSCRIPT ONE
	0x00BA: "",           // MASCULINE ORDINAL INDICATOR
	0x00BB: ">>",         // RIGHT-POINTING DOUBLE ANGLE QUOTATION MA
	0x00BC: "(1/4)",      // VULGAR FRACTION ONE QUARTER
	0x00BD: "(1/2)",      // VULGAR FRACTION ONE HALF
	0x00BE: "(3/4)",      // VULGAR FRACTION THREE QUARTERS
	0x00BF: "?",          // INVERTED QUESTION MARK
	0x00C0: "A",          // LATIN CAPITAL LETTER A WITH GRAVE
	0x00C1: "A",          // LATIN CAPITAL LETTER A WITH ACUTE
	0x00C2: "A",          // LATIN CAPITAL LETTER A WITH CIRCUMFLEX
	0x00C3: "A",          // LATIN CAPITAL LETTER A WITH TILDE
	0x00C4: "A",          // LATIN CAPITAL LETTER A WITH DIAERESIS
	0x00C5: "A",          // LATIN CAPITAL LETTER A WITH RING ABOVE
	0x00C6: "AE",         // LATIN CAPITAL LETTER AE
	0x00C7: "C",          // LATIN CAPITAL LETTER C WITH CEDILLA
	0x00C8: "E",          // LATIN CAPITAL LETTER E WITH GRAVE
	0x00C9: "E",          // LATIN CAPITAL LETTER E WITH ACUTE
	0x00CA: "E",          // LATIN CAPITAL LETTER E WITH CIRCUMFLEX
	0x00CB: "E",          // LATIN CAPITAL LETTER E WITH DIAERESIS
	0x00CC: "I",          // LATIN CAPITAL LETTER I WITH GRAVE
	0x00CD: "I",          // LATIN CAPITAL LETTER I WITH ACUTE
	0x00CE: "I",          // LATIN CAPITAL LETTER I WITH CIRCUMFLEX
	0x00CF: "I",          // LATIN CAPITAL LETTER I WITH DIAERESIS
	0x00D0: "D",          // LATIN CAPITAL LETTER ETH
	0x00D1: "N",          // LATIN CAPITAL LETTER N WITH TILDE
	0x00D2: "O",          // LATIN CAPITAL LETTER O WITH GRAVE
	0x00D3: "O",          // LATIN CAPITAL LETTER O WITH ACUTE
	0x00D4: "O",          // LATIN CAPITAL LETTER O WITH CIRCUMFLEX
	0x00D5: "O",          // LATIN CAPITAL LETTER O WITH TILDE
	0x00D6: "O",          // LATIN CAPITAL LETTER O WITH DIAERESIS
	0x00D7: "x",          // MULTIPLICATION SIGN
	0x00D8: "O",          // LATIN CAPITAL LETTER O WITH STROKE
	0x00D9: "U",          // LATIN CAPITAL LETTER U WITH GRAVE
	0x00DA: "U",          // LATIN CAPITAL LETTER U WITH ACUTE
	0x00DB: "U",          // LATIN CAPITAL LETTER U WITH CIRCUMFLEX
	0x00DC: "U",          // LATIN CAPITAL LETTER U WITH DIAERESIS
	0x00DD: "Y",          // LATIN CAPITAL LETTER Y WITH ACUTE
	0x00DE: "Th",         // LATIN CAPITAL LETTER THORN
	0x00DF: "ss",         // LATIN SMALL LETTER SHARP S
	0x00E0: "a",          // LATIN SMALL LETTER A WITH GRAVE
	0x00E1: "a",          // LATIN SMALL LETTER A WITH ACUTE
	0x00E2: "a",          // LATIN SMALL LETTER A WITH CIRCUMFLEX
	0x00E3: "a",          // LATIN SMALL LETTER A WITH TILDE
	0x00E4: "a",          // LATIN SMALL LETTER A WITH DIAERESIS
	0x00E5: "a",          // LATIN SMALL LETTER A WITH RING ABOVE
	0x00E6: "ae",         // LATIN SMALL LETTER AE
	0x00E7: "c",          // LATIN SMALL LETTER C WITH CEDILLA
	0x00E8: "e",          // LATIN SMALL LETTER E WITH GRAVE
	0x00E9: "e",          // LATIN SMALL LETTER E WITH ACUTE
	0x00EA: "e",          // LATIN SMALL LETTER E WITH CIRCUMFLEX
	0x00EB: "e",          // LATIN SMALL LETTER E WITH DIAERESIS
	0x00EC: "i",          // LATIN SMALL LETTER I WITH GRAVE
	0x00ED: "i",          // LATIN SMALL LETTER I WITH ACUTE
	0x00EE: "i",          // LATIN SMALL LETTER I WITH CIRCUMFLEX
	0x00EF: "i",          // LATIN SMALL LETTER I WITH DIAERESIS
	0x00F0: "d",          // LATIN SMALL LETTER ETH
	0x00F1: "n",          // LATIN SMALL LETTER N WITH TILDE
	0x00F2: "o",          // LATIN SMALL LETTER O WITH GRAVE
	0x00F3: "o",          // LATIN SMALL LETTER O WITH ACUTE
	0x00F4: "o",          // LATIN SMALL LETTER O WITH CIRCUMFLEX
	0x00F5: "o",          // LATIN SMALL LETTER O WITH TILDE
	0x00F6: "o",          // LATIN SMALL LETTER O WITH DIAERESIS
	0x00F7: "/",          // DIVISION SIGN
	0x00F8: "o",          // LATIN SMALL LETTER O WITH STROKE
	0x00F9: "u",          // LATIN SMALL LETTER U WITH GRAVE
	0x00FA: "u",          // LATIN SMALL LETTER U WITH ACUTE
	0x00FB: "u",          // LATIN SMALL LETTER U WITH CIRCUMFLEX
	0x00FC: "u",          // LATIN SMALL LETTER U WITH DIAERESIS
	0x00FD: "y",          // LATIN SMALL LETTER Y WITH ACUTE
	0x00FE: "th",         // LATIN SMALL LETTER THORN
	0x00FF: "y",          // LATIN SMALL LETTER Y WITH DIAERESIS
	0x0100: "A",          // LATIN CAPITAL LETTER A WITH MACRON
	0x0101: "a",          // LATIN SMALL LETTER A WITH MACRON
	0x0102: "A",          // LATIN CAPITAL LETTER A WITH BREVE
	0x0103: "a",          // LATIN SMALL LETTER A WITH BREVE
	0x0104: "A",          // LATIN CAPITAL LETTER A WITH OGONEK
	0x0105: "a",          // LATIN SMALL LETTER A WITH OGONEK
	0x0106: "C",          // LATIN CAPITAL LETTER C WITH ACUTE
	0x0107: "c",          // LATIN SMALL LETTER C WITH ACUTE
	0x0108: "C",          // LATIN CAPITAL LETTER C WITH CIRCUMFLEX
	0x0109: "c",          // LATIN SMALL LETTER C WITH CIRCUMFLEX
	0x010A: "C",          // LATIN CAPITAL LETTER C WITH DOT ABOVE
	0x010B: "c",          // LATIN SMALL LETTER C WITH DOT ABOVE
	0x010C: "C",          // LATIN CAPITAL LETTER C WITH CARON
	0x010D: "c",          // LATIN SMALL LETTER C WITH CARON
	0x010E: "D",          // LATIN CAPITAL LETTER D WITH CARON
	0x010F: "d",          // LATIN SMALL LETTER D WITH CARON
	0x0110: "D",          // LATIN CAPITAL LETTER D WITH STROKE
	0x0111: "d",          // LATIN SMALL LETTER D WITH STROKE
	0x0112: "E",          // LATIN CAPITAL LETTER E WITH MACRON
	0x0113: "e",          // LATIN SMALL LETTER E WITH MACRON
	0x0114: "E",          // LATIN CAPITAL LETTER E WITH BREVE
	0x0115: "e",          // LATIN SMALL LETTER E WITH BREVE
	0x0116: "E",          // LATIN CAPITAL LETTER E WITH DOT ABOVE
	0x0117: "e",          // LATIN SMALL LETTER E WITH DOT ABOVE
	0x0118: "E",          // LATIN CAPITAL LETTER E WITH OGONEK
	0x0119: "e",          // LATIN SMALL LETTER E WITH OGONEK
	0x011A: "E",          // LATIN CAPITAL LETTER E WITH CARON
	0x011B: "e",          // LATIN SMALL LETTER E WITH CARON
	0x011C: "G",          // LATIN CAPITAL LETTER G WITH CIRCUMFLEX
	0x011D: "g",          // LATIN SMALL LETTER G WITH CIRCUMFLEX
	0x011E: "G",          // LATIN CAPITAL LETTER G WITH BREVE
	0x011F: "g",          // LATIN SMALL LETTER G WITH BREVE
	0x0120: "G",          // LATIN CAPITAL LETTER G WITH DOT ABOVE
	0x0121: "g",          // LATIN SMALL LETTER G WITH DOT ABOVE
	0x0122: "G",          // LATIN CAPITAL LETTER G WITH CEDILLA
	0x0123: "g",          // LATIN SMALL LETTER G WITH CEDILLA
	0x0124: "H",          // LATIN CAPITAL LETTER H WITH CIRCUMFLEX
	0x0125: "h",          // LATIN SMALL LETTER H WITH CIRCUMFLEX
	0x0126: "H",          // LATIN CAPITAL LETTER H WITH STROKE
	0x0127: "h",          // LATIN SMALL LETTER H WITH STROKE
	0x0128: "I",          // LATIN CAPITAL LETTER I WITH TILDE
	0x0129: "i",          // LATIN SMALL LETTER I WITH TILDE
	0x012A: "I",          // LATIN CAPITAL LETTER I WITH MACRON
	0x012B: "i",          // LATIN SMALL LETTER I WITH MACRON
	0x012C: "I",          // LATIN CAPITAL LETTER I WITH BREVE
	0x012D: "i",          // LATIN SMALL LETTER I WITH BREVE
	0x012E: "I",          // LATIN CAPITAL LETTER I WITH OGONEK
	0x012F: "i",          // LATIN SMALL LETTER I WITH OGONEK
	0x0130: "I",          // LATIN CAPITAL LETTER I WITH DOT ABOVE
	0x0131: "i",          // LATIN SMALL LETTER DOTLESS I
	0x0132: "IJ",         // LATIN CAPITAL LIGATURE IJ
	0x0133: "ij",         // LATIN SMALL LIGATURE IJ
	0x0134: "J",          // LATIN CAPITAL LETTER J WITH CIRCUMFLEX
	0x0135: "j",          // LATIN SMALL LETTER J WITH CIRCUMFLEX
	0x0136: "K",          // LATIN CAPITAL LETTER K WITH CEDILLA
	0x0137: "k",          // LATIN SMALL LETTER K WITH CEDILLA
	0x0138: "k",          // LATIN SMALL LETTER KRA
	0x0139: "L",          // LATIN CAPITAL LETTER L WITH ACUTE
	0x013A: "l",          // LATIN SMALL LETTER L WITH ACUTE
	0x013B: "L",          // LATIN CAPITAL LETTER L WITH CEDILLA
	0x013C: "l",          // LATIN SMALL LETTER L WITH CEDILLA
	0x013D: "L",          // LATIN CAPITAL LETTER L WITH CARON
	0x013E: "l",          // LATIN SMALL LETTER L WITH CARON
	0x013F: "L",          // LATIN CAPITAL LETTER L WITH MIDDLE DOT
	0x0140: "l",          // LATIN SMALL LETTER L WITH MIDDLE DOT
	0x0141: "L",          // LATIN CAPITAL LETTER L WITH STROKE
	0x0142: "l",          // LATIN SMALL LETTER L WITH STROKE
	0x0143: "N",          // LATIN CAPITAL LETTER N WITH ACUTE
	0x0144: "n",          // LATIN SMALL LETTER N WITH ACUTE
	0x0145: "N",          // LATIN CAPITAL LETTER N WITH CEDILLA
	0x0146: "n",          // LATIN SMALL LETTER N WITH CEDILLA
	0x0147: "N",          // LATIN CAPITAL LETTER N WITH CARON
	0x0148: "n",          // LATIN SMALL LETTER N WITH CARON
	0x0149: "'n",         // LATIN SMALL LETTER N PRECEDED BY APOSTROPHE
	0x014A: "NG",         // LATIN CAPITAL LETTER ENG
	0x014B: "ng",         // LATIN SMALL LETTER ENG
	0x014C: "O",          // LATIN CAPITAL LETTER O WITH MACRON
	0x014D: "o",          // LATIN SMALL LETTER O WITH MACRON
	0x014E: "O",          // LATIN CAPITAL LETTER O WITH BREVE
	0x014F: "o",          // LATIN SMALL LETTER O WITH BREVE
	0x0150: "O",          // LATIN CAPITAL LETTER O WITH DOUBLE ACUTE
	0x0151: "o",          // LATIN SMALL LETTER O WITH DOUBLE ACUTE
	0x0152: "OE",         // LATIN CAPITAL LIGATURE OE
	0x0153: "oe",         // LATIN SMALL LIGATURE OE
	0x0154: "R",          // LATIN CAPITAL LETTER R WITH ACUTE
	0x0155: "r",          // LATIN SMALL LETTER R WITH ACUTE
	0x0156: "R",          // LATIN CAPITAL LETTER R WITH CEDILLA
	0x0157: "r",          // LATIN SMALL LETTER R WITH CEDILLA
	0x0158: "R",          // LATIN CAPITAL LETTER R WITH CARON
	0x0159: "r",          // LATIN SMALL LETTER R WITH CARON
	0x015A: "S",          // LATIN CAPITAL LETTER S WITH ACUTE
	0x015B: "s",          // LATIN SMALL LETTER S WITH ACUTE
	0x015C: "S",          // LATIN CAPITAL LETTER S WITH CIRCUMFLEX
	0x015D: "s",          // LATIN SMALL LETTER S WITH CIRCUMFLEX
	0x015E: "S",          // LATIN CAPITAL LETTER S WITH CEDILLA
	0x015F: "s",          // LATIN SMALL LETTER S WITH CEDILLA
	0x0160: "S",          // LATIN CAPITAL LETTER S WITH CARON
	0x0161: "s",          // LATIN SMALL LETTER S WITH CARON
	0x0162: "T",          // LATIN CAPITAL LETTER T WITH CEDILLA
	0x0163: "t",          // LATIN SMALL LETTER T WITH CEDILLA
	0x0164: "T",          // LATIN CAPITAL LETTER T WITH CARON
	0x0165: "t",          // LATIN SMALL LETTER T WITH CARON
	0x0166: "T",          // LATIN CAPITAL LETTER T WITH STROKE
	0x0167: "t",          // LATIN SMALL LETTER T WITH STROKE
	0x0168: "U",          // LATIN CAPITAL LETTER U WITH TILDE
	0x0169: "u",          // LATIN SMALL LETTER U WITH TILDE
	0x016A: "U",          // LATIN CAPITAL LETTER U WITH MACRON
	0x016B: "u",          // LATIN SMALL LETTER U WITH MACRON
	0x016C: "U",          // LATIN CAPITAL LETTER U WITH BREVE
	0x016D: "u",          // LATIN SMALL LETTER U WITH BREVE
	0x016E: "U",          // LATIN CAPITAL LETTER U WITH RING ABOVE
	0x016F: "u",          // LATIN SMALL LETTER U WITH RING ABOVE
	0x0170: "U",          // LATIN CAPITAL LETTER U WITH DOUBLE ACUTE
	0x0171: "u",          // LATIN SMALL LETTER U WITH DOUBLE ACUTE
	0x0172: "U",          // LATIN CAPITAL LETTER U WITH OGONEK
	0x0173: "u",          // LATIN SMALL LETTER U WITH OGONEK
	0x0174: "W",          // LATIN CAPITAL LETTER W WITH CIRCUMFLEX
	0x0175: "w",          // LATIN SMALL LETTER W WITH CIRCUMFLEX
	0x0176: "Y",          // LATIN CAPITAL LETTER Y WITH CIRCUMFLEX
	0x0177: "y",          // LATIN SMALL LETTER Y WITH CIRCUMFLEX
	0x0178: "Y",          // LATIN CAPITAL LETTER Y WITH DIAERESIS
	0x0179: "Z",          // LATIN CAPITAL LETTER Z WITH ACUTE
	0x017A: "z",          // LATIN SMALL LETTER Z WITH ACUTE
	0x017B: "Z",          // LATIN CAPITAL LETTER Z WITH DOT ABOVE
	0x017C: "z",          // LATIN SMALL LETTER Z WITH DOT ABOVE
	0x017D: "Z",          // LATIN CAPITAL LETTER Z WITH CARON
	0x017E: "z",          // LATIN SMALL LETTER Z WITH CARON
	0x017F: "s",          // LATIN SMALL LETTER LONG S
	0x0180: "b",          // LATIN SMALL LETTER B WITH STROKE
	0x0181: "B",          // LATIN CAPITAL LETTER B WITH HOOK
	0x0182: "B",          // LATIN CAPITAL LETTER B WITH TOPBAR
	0x0183: "b",          // LATIN SMALL LETTER B WITH TOPBAR
	0x0184: "6",          // LATIN CAPITAL LETTER TONE SIX
	0x0185: "6",          // LATIN SMALL LETTER TONE SIX
	0x0186: "O",          // LATIN CAPITAL LETTER OPEN O
	0x0187: "C",          // LATIN CAPITAL LETTER C WITH HOOK
	0x0188: "c",          // LATIN SMALL LETTER C WITH HOOK
	0x0189: "D",          // LATIN CAPITAL LETTER AFRICAN D
	0x018A: "D",          // LATIN CAPITAL LETTER D WITH HOOK
	0x018B: "D",          // LATIN CAPITAL LETTER D WITH TOPBAR
	0x018C: "d",          // LATIN SMALL LETTER D WITH TOPBAR
	0x018D: "d",          // LATIN SMALL LETTER TURNED DELTA
	0x018E: "3",          // LATIN CAPITAL LETTER REVERSED E
	0x018F: "@",          // LATIN CAPITAL LETTER SCHWA
	0x0190: "E",          // LATIN CAPITAL LETTER OPEN E
	0x0191: "F",          // LATIN CAPITAL LETTER F WITH HOOK
	0x0192: "f",          // LATIN SMALL LETTER F WITH HOOK
	0x0193: "G",          // LATIN CAPITAL LETTER G WITH HOOK
	0x0194: "G",          // LATIN CAPITAL LETTER GAMMA
	0x0195: "hv",         // LATIN SMALL LETTER HV
	0x0196: "I",          // LATIN CAPITAL LETTER IOTA
	0x0197: "I",          // LATIN CAPITAL LETTER I WITH STROKE
	0x0198: "K",          // LATIN CAPITAL LETTER K WITH HOOK
	0x0199: "k",          // LATIN SMALL LETTER K WITH HOOK
	0x019A: "l",          // LATIN SMALL LETTER L WITH BAR
	0x019B: "l",          // LATIN SMALL LETTER LAMBDA WITH STROKE
	0x019C: "W",          // LATIN CAPITAL LETTER TURNED M
	0x019D: "N",          // LATIN CAPITAL LETTER N WITH LEFT HOOK
	0x019E: "n",          // LATIN SMALL LETTER N WITH LONG RIGHT LEG
	0x019F: "O",          // LATIN CAPITAL LETTER O WITH MIDDLE TILDE
	0x01A0: "O",          // LATIN CAPITAL LETTER O WITH HORN
	0x01A1: "o",          // LATIN SMALL LETTER O WITH HORN
	0x01A2: "OI",         // LATIN CAPITAL LETTER OI
	0x01A3: "oi",         // LATIN SMALL LETTER OI
	0x01A4: "P",          // LATIN CAPITAL LETTER P WITH HOOK
	0x01A5: "p",          // LATIN SMALL LETTER P WITH HOOK
	0x01A6: "YR",         // LATIN LETTER YR
	0x01A7: "2",          // LATIN CAPITAL LETTER TONE TWO
	0x01A8: "2",          // LATIN SMALL LETTER TONE TWO
	0x01A9: "SH",         // LATIN CAPITAL LETTER ESH
	0x01AA: "sh",         // LATIN LETTER REVERSED ESH LOOP
	0x01AB: "t",          // LATIN SMALL LETTER T WITH PALATAL HOOK
	0x01AC: "T",          // LATIN CAPITAL LETTER T WITH HOOK
	0x01AD: "t",          // LATIN SMALL LETTER T WITH HOOK
	0x01AE: "T",          // LATIN CAPITAL LETTER T WITH RETROFLEX HOOK
	0x01AF: "U",          // LATIN CAPITAL LETTER U WITH HORN
	0x01B0: "u",          // LATIN SMALL LETTER U WITH HORN
	0x01B1: "Y",          // LATIN CAPITAL LETTER UPSILON
	0x01B2: "V",          // LATIN CAPITAL LETTER V WITH HOOK
	0x01B3: "Y",          // LATIN CAPITAL LETTER Y WITH HOOK
	0x01B4: "y",          // LATIN SMALL LETTER Y WITH HOOK
	0x01B5: "Z",          // LATIN CAPITAL LETTER Z WITH STROKE
	0x01B6: "z",          // LATIN SMALL LETTER Z WITH STROKE
	0x01B7: "ZH",         // LATIN CAPITAL LETTER EZH
	0x01B8: "ZH",         // LATIN CAPITAL LETTER EZH REVERSED
	0x01B9: "zh",         // LATIN SMALL LETTER EZH REVERSED
	0x01BA: "zh",         // LATIN SMALL LETTER EZH WITH TAIL
	0x01BB: "2",          // LATIN LETTER TWO WITH STROKE
	0x01BC: "5",          // LATIN CAPITAL LETTER TONE FIVE
	0x01BD: "5",          // LATIN SMALL LETTER TONE FIVE
	0x01BE: "ts",         // LATIN LETTER INVERTED GLOTTAL STOP WITH STROKE
	0x01BF: "w",          // LATIN LETTER WYNN
	0x01C0: "|",          // LATIN LETTER DENTAL CLICK
	0x01C1: "||",         // LATIN LETTER LATERAL CLICK
	0x01C2: "|=",         // LATIN LETTER ALVEOLAR CLICK
	0x01C3: "!",          // LATIN LETTER RETROFLEX CLICK
	0x01C4: "DZ",         // LATIN CAPITAL LETTER DZ WITH CARON
	0x01C5: "Dz",         // LATIN CAPITAL LETTER D WITH SMALL LETTER Z WITH CA
	0x01C6: "dz",         // LATIN SMALL LETTER DZ WITH CARON
	0x01C7: "LJ",         // LATIN CAPITAL LETTER LJ
	0x01C8: "Lj",         // LATIN CAPITAL LETTER L WITH SMALL LETTER J
	0x01C9: "lj",         // LATIN SMALL LETTER LJ
	0x01CA: "NJ",         // LATIN CAPITAL LETTER NJ
	0x01CB: "Nj",         // LATIN CAPITAL LETTER N WITH SMALL LETTER J
	0x01CC: "nj",         // LATIN SMALL LETTER NJ
	0x01CD: "A",          // LATIN CAPITAL LETTER A WITH CARON
	0x01CE: "a",          // LATIN SMALL LETTER A WITH CARON
	0x01CF: "I",          // LATIN CAPITAL LETTER I WITH CARON
	0x01D0: "i",          // LATIN SMALL LETTER I WITH CARON
	0x01D1: "O",          // LATIN CAPITAL LETTER O WITH CARON
	0x01D2: "o",          // LATIN SMALL LETTER O WITH CARON
	0x01D3: "U",          // LATIN CAPITAL LETTER U WITH CARON
	0x01D4: "u",          // LATIN SMALL LETTER U WITH CARON
	0x01D5: "U",          // LATIN CAPITAL LETTER U WITH DIAERESIS AND MACRON
	0x01D6: "u",          // LATIN SMALL LETTER U WITH DIAERESIS AND MACRON
	0x01D7: "U",          // LATIN CAPITAL LETTER U WITH DIAERESIS AND ACUTE
	0x01D8: "u",          // LATIN SMALL LETTER U WITH DIAERESIS AND ACUTE
	0x01D9: "U",          // LATIN CAPITAL LETTER U WITH DIAERESIS AND CARON
	0x01DA: "u",          // LATIN SMALL LETTER U WITH DIAERESIS AND CARON
	0x01DB: "U",          // LATIN CAPITAL LETTER U WITH DIAERESIS AND GRAVE
	0x01DC: "u",          // LATIN SMALL LETTER U WITH DIAERESIS AND GRAVE
	0x01DD: "@",          // LATIN SMALL LETTER TURNED E
	0x01DE: "A",          // LATIN CAPITAL LETTER A WITH DIAERESIS AND MACRON
	0x01DF: "a",          // LATIN SMALL LETTER A WITH DIAERESIS AND MACRON
	0x01E0: "A",          // LATIN CAPITAL LETTER A WITH DOT ABOVE AND MACRON
	0x01E1: "a",          // LATIN SMALL LETTER A WITH DOT ABOVE AND MACRON
	0x01E2: "AE",         // LATIN CAPITAL LETTER AE WITH MACRON
	0x01E3: "ae",         // LATIN SMALL LETTER AE WITH MACRON
	0x01E4: "G",          // LATIN CAPITAL LETTER G WITH STROKE
	0x01E5: "g",          // LATIN SMALL LETTER G WITH STROKE
	0x01E6: "G",          // LATIN CAPITAL LETTER G WITH CARON
	0x01E7: "g",          // LATIN SMALL LETTER G WITH CARON
	0x01E8: "K",          // LATIN CAPITAL LETTER K WITH CARON
	0x01E9: "k",          // LATIN SMALL LETTER K WITH CARON
	0x01EA: "O",          // LATIN CAPITAL LETTER O WITH OGONEK
	0x01EB: "o",          // LATIN SMALL LETTER O WITH OGONEK
	0x01EC: "O",          // LATIN CAPITAL LETTER O WITH OGONEK AND MACRON
	0x01ED: "o",          // LATIN SMALL LETTER O WITH OGONEK AND MACRON
	0x01EE: "ZH",         // LATIN CAPITAL LETTER EZH WITH CARON
	0x01EF: "zh",         // LATIN SMALL LETTER EZH WITH CARON
	0x01F0: "j",          // LATIN SMALL LETTER J WITH CARON
	0x01F1: "DZ",         // LATIN CAPITAL LETTER DZ
	0x01F2: "Dz",         // LATIN CAPITAL LETTER D WITH SMALL LETTER Z
	0x01F3: "dz",         // LATIN SMALL LETTER DZ
	0x01F4: "G",          // LATIN CAPITAL LETTER G WITH ACUTE
	0x01F5: "g",          // LATIN SMALL LETTER G WITH ACUTE
	0x01F6: "HV",         // LATIN CAPITAL LETTER HWAIR
	0x01F7: "W",          // LATIN CAPITAL LETTER WYNN
	0x01F8: "N",          // LATIN CAPITAL LETTER N WITH GRAVE
	0x01F9: "n",          // LATIN SMALL LETTER N WITH GRAVE
	0x01FA: "A",          // LATIN CAPITAL LETTER A WITH RING ABOVE AND ACUTE
	0x01FB: "a",          // LATIN SMALL LETTER A WITH RING ABOVE AND ACUTE
	0x01FC: "AE",         // LATIN CAPITAL LETTER AE WITH ACUTE
	0x01FD: "ae",         // LATIN SMALL LETTER AE WITH ACUTE
	0x01FE: "O",          // LATIN CAPITAL LETTER O WITH STROKE AND ACUTE
	0x01FF: "o",          // LATIN SMALL LETTER O WITH STROKE AND ACUTE
	0x0200: "A",          // LATIN CAPITAL LETTER A WITH DOUBLE GRAVE
	0x0201: "a",          // LATIN SMALL LETTER A WITH DOUBLE GRAVE
	0x0202: "A",          // LATIN CAPITAL LETTER A WITH INVERTED BREVE
	0x0203: "a",          // LATIN SMALL LETTER A WITH INVERTED BREVE
	0x0204: "E",          // LATIN CAPITAL LETTER E WITH DOUBLE GRAVE
	0x0205: "e",          // LATIN SMALL LETTER E WITH DOUBLE GRAVE
	0x0206: "E",          // LATIN CAPITAL LETTER E WITH INVERTED BREVE
	0x0207: "e",          // LATIN SMALL LETTER E WITH INVERTED BREVE
	0x0208: "I",          // LATIN CAPITAL LETTER I WITH DOUBLE GRAVE
	0x0209: "i",          // LATIN SMALL LETTER I WITH DOUBLE GRAVE
	0x020A: "I",          // LATIN CAPITAL LETTER I WITH INVERTED BREVE
	0x020B: "i",          // LATIN SMALL LETTER I WITH INVERTED BREVE
	0x020C: "O",          // LATIN CAPITAL LETTER O WITH DOUBLE GRAVE
	0x020D: "o",          // LATIN SMALL LETTER O WITH DOUBLE GRAVE
	0x020E: "O",          // LATIN CAPITAL LETTER O WITH INVERTED BREVE
	0x020F: "o",          // LATIN SMALL LETTER O WITH INVERTED BREVE
	0x0210: "R",          // LATIN CAPITAL LETTER R WITH DOUBLE GRAVE
	0x0211: "r",          // LATIN SMALL LETTER R WITH DOUBLE GRAVE
	0x0212: "R",          // LATIN CAPITAL LETTER R WITH INVERTED BREVE
	0x0213: "r",          // LATIN SMALL LETTER R WITH INVERTED BREVE
	0x0214: "U",          // LATIN CAPITAL LETTER U WITH DOUBLE GRAVE
	0x0215: "u",          // LATIN SMALL LETTER U WITH DOUBLE GRAVE
	0x0216: "U",          // LATIN CAPITAL LETTER U WITH INVERTED BREVE
	0x0217: "u",          // LATIN SMALL LETTER U WITH INVERTED BREVE
	0x0218: "S",          // LATIN CAPITAL LETTER S WITH COMMA BELOW
	0x0219: "s",          // LATIN SMALL LETTER S WITH COMMA BELOW
	0x021A: "T",          // LATIN CAPITAL LETTER T WITH COMMA BELOW
	0x021B: "t",          // LATIN SMALL LETTER T WITH COMMA BELOW
	0x021C: "Y",          // LATIN CAPITAL LETTER YOGH
	0x021D: "y",          // LATIN SMALL LETTER YOGH
	0x021E: "H",          // LATIN CAPITAL LETTER H WITH CARON
	0x021F: "h",          // LATIN SMALL LETTER H WITH CARON
	0x0220: "N",          // LATIN CAPITAL LETTER N WITH LONG RIGHT LET
	0x0221: "d",          // LATIN SMALL LETTER D WITH CURL
	0x0222: "OU",         // LATIN CAPITAL LETTER OU
	0x0223: "ou",         // LATIN SMALL LETTER OU
	0x0224: "Z",          // LATIN CAPITAL LETTER Z WITH HOOK
	0x0225: "z",          // LATIN SMALL LETTER Z WITH HOOK
	0x0226: "A",          // LATIN CAPITAL LETTER A WITH DOT ABOVE
	0x0227: "a",          // LATIN SMALL LETTER A WITH DOT ABOVE
	0x0228: "E",          // LATIN CAPITAL LETTER E WITH CEDILLA
	0x0229: "e",          // LATIN SMALL LETTER E WITH CEDILLA
	0x022A: "O",          // LATIN CAPITAL LETTER O WITH DIAERESIS AND MACRON
	0x022B: "o",          // LATIN SMALL LETTER O WITH DIAERESIS AND MACRON
	0x022C: "O",          // LATIN CAPITAL LETTER O WITH TILDE AND MACRON
	0x022D: "o",          // LATIN SMALL LETTER O WITH TILDE AND MACRON
	0x022E: "O",          // LATIN CAPITAL LETTER O WITH DOT ABOVE
	0x022F: "o",          // LATIN SMALL LETTER O WITH DOT ABOVE
	0x0230: "O",          // LATIN CAPITAL LETTER O WITH DOT ABOVE AND MACRON
	0x0231: "o",          // LATIN SMALL LETTER O WITH DOT ABOVE AND MACRON
	0x0232: "Y",          // LATIN CAPITAL LETTER Y WITH MACRON
	0x0233: "y",          // LATIN SMALL LETTER Y WITH MACRON
	0x0234: "l",          // LATIN SMALL LETTER L WITH CURL
	0x0235: "n",          // LATIN SMALL LETTER N WITH CURL
	0x0236: "t",          // LATIN SMALL LETTER T WITH CURL
	0x0237: "j",          // LATIN SMALL LETTER DOTLESS J
	0x0238: "db",         // LATIN SMALL LETTER DB DIGRAPH
	0x0239: "qp",         // LATIN SMALL LETTER QP DIGRAPH
	0x023A: "A",          // LATIN CAPITAL LETTER A WITH STROKE
	0x023B: "C",          // LATIN CAPITAL LETTER C WITH STROKE
	0x023C: "c",          // LATIN SMALL LETTER C WITH STROKE
	0x023D: "L",          // LATIN CAPITAL LETTER L WITH BAR
	0x023E: "T",          // LATIN CAPITAL LETTER T WITH DIAGONAL STROKE
	0x023F: "s",          // LATIN SMALL LETTER S WITH SWASH TAIL
	0x0240: "z",          // LATIN SMALL LETTER Z WITH SWASH TAIL
	0x0243: "B",          // LATIN CAPITAL LETTER B WITH STROKE
	0x0244: "U",          // LATIN CAPITAL LETTER U BAR
	0x0245: "^",          // LATIN CAPITAL LETTER TURNED V
	0x0246: "E",          // LATIN CAPITAL LETTER E WITH STROKE
	0x0247: "e",          // LATIN SMALL LETTER E WITH STROKE
	0x0248: "J",          // LATIN CAPITAL LETTER J WITH STROKE
	0x0249: "j",          // LATIN SMALL LETTER J WITH STROKE
	0x024A: "q",          // LATIN CAPITAL LETTER SMALL Q WITH HOOK TAIL
	0x024B: "q",          // LATIN SMALL LETTER Q WITH HOOK TAIL
	0x024C: "R",          // LATIN CAPITAL LETTER R WITH STROKE
	0x024D: "r",          // LATIN SMALL LETTER R WITH STROKE
	0x024E: "Y",          // LATIN CAPITAL LETTER Y WITH STROKE
	0x024F: "y",          // LATIN SMALL LETTER Y WITH STROKE
	0x0250: "a",          // LATIN SMALL LETTER TURNED A
	0x0251: "a",          // LATIN SMALL LETTER ALPHA
	0x0252: "a",          // LATIN SMALL LETTER TURNED ALPHA
	0x0253: "b",          // LATIN SMALL LETTER B WITH HOOK
	0x0254: "o",          // LATIN SMALL LETTER OPEN O
	0x0255: "c",          // LATIN SMALL LETTER C WITH CURL
	0x0256: "d",          // LATIN SMALL LETTER D WITH TAIL
	0x0257: "d",          // LATIN SMALL LETTER D WITH HOOK
	0x0258: "e",          // LATIN SMALL LETTER REVERSED E
	0x0259: "@",          // LATIN SMALL LETTER SCHWA
	0x025A: "@",          // LATIN SMALL LETTER SCHWA WITH HOOK
	0x025B: "e",          // LATIN SMALL LETTER OPEN E
	0x025C: "e",          // LATIN SMALL LETTER REVERSED OPEN E
	0x025D: "e",          // LATIN SMALL LETTER REVERSED OPEN E WITH HOOK
	0x025E: "e",          // LATIN SMALL LETTER CLOSED REVERSED OPEN E
	0x025F: "j",          // LATIN SMALL LETTER DOTLESS J WITH STROKE
	0x0260: "g",          // LATIN SMALL LETTER G WITH HOOK
	0x0261: "g",          // LATIN SMALL LETTER SCRIPT G
	0x0262: "g",          // LATIN LETTER SMALL CAPITAL G
	0x0263: "g",          // LATIN SMALL LETTER GAMMA
	0x0264: "u",          // LATIN SMALL LETTER RAMS HORN
	0x0265: "Y",          // LATIN SMALL LETTER TURNED H
	0x0266: "h",          // LATIN SMALL LETTER H WITH HOOK
	0x0267: "h",          // LATIN SMALL LETTER HENG WITH HOOK
	0x0268: "i",          // LATIN SMALL LETTER I WITH STROKE
	0x0269: "i",          // LATIN SMALL LETTER IOTA
	0x026A: "I",          // LATIN LETTER SMALL CAPITAL I
	0x026B: "l",          // LATIN SMALL LETTER L WITH MIDDLE TILDE
	0x026C: "l",          // LATIN SMALL LETTER L WITH BELT
	0x026D: "l",          // LATIN SMALL LETTER L WITH RETROFLEX HOOK
	0x026E: "lZ",         // LATIN SMALL LETTER LEZH
	0x026F: "W",          // LATIN SMALL LETTER TURNED M
	0x0270: "W",          // LATIN SMALL LETTER TURNED M WITH LONG LEG
	0x0271: "m",          // LATIN SMALL LETTER M WITH HOOK
	0x0272: "n",          // LATIN SMALL LETTER N WITH LEFT HOOK
	0x0273: "n",          // LATIN SMALL LETTER N WITH RETROFLEX HOOK
	0x0274: "n",          // LATIN LETTER SMALL CAPITAL N
	0x0275: "o",          // LATIN SMALL LETTER BARRED O
	0x0276: "OE",         // LATIN LETTER SMALL CAPITAL OE
	0x0277: "O",          // LATIN SMALL LETTER CLOSED OMEGA
	0x0278: "F",          // LATIN SMALL LETTER PHI
	0x0279: "r",          // LATIN SMALL LETTER TURNED R
	0x027A: "r",          // LATIN SMALL LETTER TURNED R WITH LONG LEG
	0x027B: "r",          // LATIN SMALL LETTER TURNED R WITH HOOK
	0x027C: "r",          // LATIN SMALL LETTER R WITH LONG LEG
	0x027D: "r",          // LATIN SMALL LETTER R WITH TAIL
	0x027E: "r",          // LATIN SMALL LETTER R WITH FISHHOOK
	0x027F: "r",          // LATIN SMALL LETTER REVERSED R WITH FISHHOOK
	0x0280: "R",          // LATIN LETTER SMALL CAPITAL R
	0x0281: "R",          // LATIN LETTER SMALL CAPITAL INVERTED R
	0x0282: "s",          // LATIN SMALL LETTER S WITH HOOK
	0x0283: "S",          // LATIN SMALL LETTER ESH
	0x0284: "j",          // LATIN SMALL LETTER DOTLESS J WITH STROKE AND HOOK
	0x0285: "S",          // LATIN SMALL LETTER SQUAT REVERSED ESH
	0x0286: "S",          // LATIN SMALL LETTER ESH WITH CURL
	0x0287: "t",          // LATIN SMALL LETTER TURNED T
	0x0288: "t",          // LATIN SMALL LETTER T WITH RETROFLEX HOOK
	0x0289: "u",          // LATIN SMALL LETTER U BAR
	0x028A: "U",          // LATIN SMALL LETTER UPSILON
	0x028B: "v",          // LATIN SMALL LETTER V WITH HOOK
	0x028C: "^",          // LATIN SMALL LETTER TURNED V
	0x028D: "w",          // LATIN SMALL LETTER TURNED W
	0x028E: "y",          // LATIN SMALL LETTER TURNED Y
	0x028F: "Y",          // LATIN LETTER SMALL CAPITAL Y
	0x0290: "z",          // LATIN SMALL LETTER Z WITH RETROFLEX HOOK
	0x0291: "z",          // LATIN SMALL LETTER Z WITH CURL
	0x0292: "Z",          // LATIN SMALL LETTER EZH
	0x0293: "Z",          // LATIN SMALL LETTER EZH WITH CURL
	0x0297: "C",          // LATIN LETTER STRETCHED C
	0x0298: "@",          // LATIN LETTER BILABIAL CLICK
	0x0299: "B",          // LATIN LETTER SMALL CAPITAL B
	0x029A: "E",          // LATIN SMALL LETTER CLOSED OPEN E
	0x029B: "G",          // LATIN LETTER SMALL CAPITAL G WITH HOOK
	0x029C: "H",          // LATIN LETTER SMALL CAPITAL H
	0x029D: "j",          // LATIN SMALL LETTER J WITH CROSSED-TAIL
	0x029E: "k",          // LATIN SMALL LETTER TURNED K
	0x029F: "L",          // LATIN LETTER SMALL CAPITAL L
	0x02A0: "q",          // LATIN SMALL LETTER Q WITH HOOK
	0x02A3: "dz",         // LATIN SMALL LETTER DZ DIGRAPH
	0x02A4: "dZ",         // LATIN SMALL LETTER DEZH DIGRAPH
	0x02A5: "dz",         // LATIN SMALL LETTER DZ DIGRAPH WITH CURL
	0x02A6: "ts",         // LATIN SMALL LETTER TS DIGRAPH
	0x02A7: "tS",         // LATIN SMALL LETTER TESH DIGRAPH
	0x02A8: "tC",         // LATIN SMALL LETTER TC DIGRAPH WITH CURL
	0x02A9: "fN",         // LATIN SMALL LETTER FENG DIGRAPH
	0x02AA: "ls",         // LATIN SMALL LETTER LS DIGRAPH
	0x02AB: "lz",         // LATIN SMALL LETTER LZ DIGRAPH
	0x02AC: "WW",         // LATIN LETTER BILABIAL PERCUSSIVE
	0x02AD: "]]",         // LATIN LETTER BIDENTAL PERCUSSIVE
	0x02AE: "h",          // LATIN SMALL LETTER TURNED H WITH FISHHOOK
	0x02AF: "h",          // LATIN SMALL LETTER TURNED H WITH FISHHOOK AND TAIL
	0x02B0: "k",          // MODIFIER LETTER SMALL H
	0x02B1: "h",          // MODIFIER LETTER SMALL H WITH HOOK
	0x02B2: "j",          // MODIFIER LETTER SMALL J
	0x02B3: "r",          // MODIFIER LETTER SMALL R
	0x02B4: "r",          // MODIFIER LETTER SMALL TURNED R
	0x02B5: "r",          // MODIFIER LETTER SMALL TURNED R WITH HOOK
	0x02B6: "r",          // MODIFIER LETTER SMALL CAPITAL INVERTED R
	0x02B7: "w",          // MODIFIER LETTER SMALL W
	0x02B8: "y",          // MODIFIER LETTER SMALL Y
	0x02B9: "'",          // MODIFIER LETTER PRIME
	0x02BA: "''",         // MODIFIER LETTER DOUBLE PRIME
	0x02BB: "`",          // MODIFIER LETTER TURNED COMMA
	0x02BC: "'",          // MODIFIER LETTER APOSTROPHE
	0x02BD: "`",          // MODIFIER LETTER REVERSED COMMA
	0x02BE: "`",          // MODIFIER LETTER RIGHT HALF RING
	0x02BF: "'",          // MODIFIER LETTER LEFT HALF RING
	0x02C0: "?",          // MODIFIER LETTER GLOTTAL STOP
	0x02C1: "?",          // MODIFIER LETTER REVERSED GLOTTAL STOP
	0x02C2: "<",          // MODIFIER LETTER LEFT ARROWHEAD
	0x02C3: ">",          // MODIFIER LETTER RIGHT ARROWHEAD
	0x02C4: "^",          // MODIFIER LETTER UP ARROWHEAD
	0x02C5: "v",          // MODIFIER LETTER DOWN ARROWHEAD
	0x02C6: "^",          // MODIFIER LETTER CIRCUMFLEX ACCENT
	0x02C7: "V",          // CARON
	0x02C8: "'",          // MODIFIER LETTER VERTICAL LINE
	0x02C9: "-",          // MODIFIER LETTER MACRON
	0x02CA: "'",          // MODIFIER LETTER ACUTE ACCENT
	0x02CB: "`",          // MODIFIER LETTER GRAVE ACCENT
	0x02CC: ",",          // MODIFIER LETTER LOW VERTICAL LINE
	0x02CD: "_",          // MODIFIER LETTER LOW MACRON
	0x02CE: "'",          // MODIFIER LETTER LOW GRAVE ACCENT
	0x02CF: "`",          // MODIFIER LETTER LOW ACUTE ACCENT
	0x02D0: ":",          // MODIFIER LETTER TRIANGULAR COLON
	0x02D1: ".",          // MODIFIER LETTER HALF TRIANGULAR COLON
	0x02D2: "`",          // MODIFIER LETTER CENTRED RIGHT HALF RING
	0x02D3: "'",          // MODIFIER LETTER CENTRED LEFT HALF RING
	0x02D4: "^",          // MODIFIER LETTER UP TACK
	0x02D5: "V",          // MODIFIER LETTER DOWN TACK
	0x02D6: "+",          // MODIFIER LETTER PLUS SIGN
	0x02D7: "-",          // MODIFIER LETTER MINUS SIGN
	0x02D8: "V",          // BREVE
	0x02D9: ".",          // DOT ABOVE
	0x02DA: "@",          // RING ABOVE
	0x02DB: ",",          // OGONEK
	0x02DC: "~",          // SMALL TILDE
	0x02DD: "",           // DOUBLE ACUTE ACCENT
	0x02DE: "R",          // MODIFIER LETTER RHOTIC HOOK
	0x02DF: "X",          // MODIFIER LETTER CROSS ACCENT
	0x02E0: "G",          // MODIFIER LETTER SMALL GAMMA
	0x02E1: "l",          // MODIFIER LETTER SMALL L
	0x02E2: "s",          // MODIFIER LETTER SMALL S
	0x02E3: "x",          // MODIFIER LETTER SMALL X
	0x02EC: "V",          // MODIFIER LETTER VOICING
	0x02ED: "=",          // MODIFIER LETTER UNASPIRATED
	0x02EE: "''",         // MODIFIER LETTER DOUBLE APOSTROPHE
	0x02F1: "l",          // MODIFIER LETTER LOW LEFT ARROWHEAD
	0x02F2: "s",          // MODIFIER LETTER LOW RIGHT ARROWHEAD
	0x02F3: "x",          // MODIFIER LETTER LOW RING
	0x02FC: "v",          // MODIFIER LETTER END LOW TONE
	0x02FD: "=",          // MODIFIER LETTER SHELF
	0x0363: "a",          // COMBINING LATIN SMALL LETTER A
	0x0364: "e",          // COMBINING LATIN SMALL LETTER E
	0x0365: "i",          // COMBINING LATIN SMALL LETTER I
	0x0366: "o",          // COMBINING LATIN SMALL LETTER O
	0x0367: "u",          // COMBINING LATIN SMALL LETTER U
	0x0368: "c",          // COMBINING LATIN SMALL LETTER C
	0x0369: "d",          // COMBINING LATIN SMALL LETTER D
	0x036A: "h",          // COMBINING LATIN SMALL LETTER H
	0x036B: "m",          // COMBINING LATIN SMALL LETTER M
	0x036C: "r",          // COMBINING LATIN SMALL LETTER R
	0x036D: "t",          // COMBINING LATIN SMALL LETTER T
	0x036E: "v",          // COMBINING LATIN SMALL LETTER V
	0x036F: "x",          // COMBINING LATIN SMALL LETTER X
	0x0374: "'",          // GREEK NUMERAL SIGN
	0x0375: ",",          // GREEK LOWER NUMERAL SIGN
	0x037E: "?",          // GREEK QUESTION MARK
	0x037F: "J",          // GREEK CAPITAL LETTER YOT
	0x0386: "A",          // GREEK CAPITAL LETTER ALPHA WITH TONOS
	0x0387: "",           // GREEK ANO TELEIA
	0x0388: "E",          // GREEK CAPITAL LETTER EPSILON WITH TONOS
	0x0389: "E",          // GREEK CAPITAL LETTER ETA WITH TONOS
	0x038A: "I",          // GREEK CAPITAL LETTER IOTA WITH TONOS
	0x038C: "O",          // GREEK CAPITAL LETTER OMICRON WITH TONOS
	0x038E: "U",          // GREEK CAPITAL LETTER UPSILON WITH TONOS
	0x038F: " Omega ",    // GREEK CAPITAL LETTER OMEGA WITH TONOS
	0x0390: " iota ",     // GREEK SMALL LETTER IOTA WITH DIALYTIKA AND TONOS
	0x0391: " Alpha ",    // GREEK CAPITAL LETTER ALPHA
	0x0392: " Beta ",     // GREEK CAPITAL LETTER BETA
	0x0393: " Gamma ",    // GREEK CAPITAL LETTER GAMMA
	0x0394: " Delta ",    // GREEK CAPITAL LETTER DELTA
	0x0395: " Epsilon",   // GREEK CAPITAL LETTER EPSILON
	0x0396: " Zeta ",     // GREEK CAPITAL LETTER ZETA
	0x0397: " Eta ",      // GREEK CAPITAL LETTER ETA
	0x0398: " Theta ",    // GREEK CAPITAL LETTER THETA
	0x0399: " Iota ",     // GREEK CAPITAL LETTER IOTA
	0x039A: " Kappa ",    // GREEK CAPITAL LETTER KAPPA
	0x039B: " Lambda ",   // GREEK CAPITAL LETTER LAMDA
	0x039C: " Mu ",       // GREEK CAPITAL LETTER MU
	0x039D: " Nu ",       // GREEK CAPITAL LETTER NU
	0x039E: " Xi ",       // GREEK CAPITAL LETTER XI
	0x039F: " Omicron ",  // GREEK CAPITAL LETTER OMICRON
	0x03A0: " Pi ",       // GREEK CAPITAL LETTER PI
	0x03A1: " Rho ",      // GREEK CAPITAL LETTER RHO
	0x03A3: " Sigma ",    // GREEK CAPITAL LETTER SIGMA
	0x03A4: " Tau ",      // GREEK CAPITAL LETTER TAU
	0x03A5: " Upsilon ",  // GREEK CAPITAL LETTER UPSILON
	0x03A6: " Phi ",      // GREEK CAPITAL LETTER PHI
	0x03A7: " Chi ",      // GREEK CAPITAL LETTER CHI
	0x03A8: " Psi ",      // GREEK CAPITAL LETTER PSI
	0x03A9: " Omega ",    // GREEK CAPITAL LETTER OMEGA
	0x03AA: " Iota ",     // GREEK CAPITAL LETTER IOTA WITH DIALYTIKA
	0x03AB: " Upsilon ",  // GREEK CAPITAL LETTER UPSILON WITH DIALYTIKA
	0x03AC: " alpha ",    // GREEK SMALL LETTER ALPHA WITH TONOS
	0x03AD: " epsilon ",  // GREEK SMALL LETTER EPSILON WITH TONOS
	0x03AE: " eta ",      // GREEK SMALL LETTER ETA WITH TONOS
	0x03AF: " iota ",     // GREEK SMALL LETTER IOTA WITH TONOS
	0x03B0: " upsilon ",  // GREEK SMALL LETTER UPSILON WITH DIALYTIKA AND TONOS
	0x03B1: " alpha ",    // GREEK SMALL LETTER ALPHA
	0x03B2: " beta ",     // GREEK SMALL LETTER BETA
	0x03B3: " gamma ",    // GREEK SMALL LETTER GAMMA
	0x03B4: " delta ",    // GREEK SMALL LETTER DELTA
	0x03B5: " epsilon ",  // GREEK SMALL LETTER EPSILON
	0x03B6: " zeta ",     // GREEK SMALL LETTER ZETA
	0x03B7: " eta ",      // GREEK SMALL LETTER ETA
	0x03B8: " theta ",    // GREEK SMALL LETTER THETA
	0x03B9: " iota ",     // GREEK SMALL LETTER IOTA
	0x03BA: " kappa ",    // GREEK SMALL LETTER KAPPA
	0x03BB: " lambda ",   // GREEK SMALL LETTER LAMDA
	0x03BC: " mu ",       // GREEK SMALL LETTER MU
	0x03BD: " nu ",       // GREEK SMALL LETTER NU
	0x03BE: " xi ",       // GREEK SMALL LETTER XI
	0x03BF: " omicron ",  // GREEK SMALL LETTER OMICRON
	0x03C0: " pi ",       // GREEK SMALL LETTER PI
	0x03C1: " rho ",      // GREEK SMALL LETTER RHO
	0x03C2: " sigma ",    // GREEK SMALL LETTER FINAL SIGMA
	0x03C3: " sigma ",    // GREEK SMALL LETTER SIGMA
	0x03C4: " tau ",      // GREEK SMALL LETTER TAU
	0x03C5: " upsilon ",  // GREEK SMALL LETTER UPSILON
	0x03C6: " phi ",      // GREEK SMALL LETTER PHI
	0x03C7: " chi ",      // GREEK SMALL LETTER CHI
	0x03C8: " psi ",      // GREEK SMALL LETTER PSI
	0x03C9: " omega ",    // GREEK SMALL LETTER OMEGA
	0x03CA: " iota ",     // GREEK SMALL LETTER IOTA WITH DIALYTIKA
	0x03CB: " upsilon ",  // GREEK SMALL LETTER UPSILON WITH DIALYTIKA
	0x03CC: " omicron ",  // GREEK SMALL LETTER OMICRON WITH TONOS
	0x03CD: " upsilon ",  // GREEK SMALL LETTER UPSILON WITH TONOS
	0x03CE: " omega ",    // GREEK SMALL LETTER OMEGA WITH TONOS
	0x03CF: " Kai ",      // GREEK CAPITAL KAI SYMBOL
	0x03D0: " beta ",     // GREEK BETA SYMBOL
	0x03D1: " theta ",    // GREEK THETA SYMBOL
	0x03D2: " Upsilon ",  // GREEK UPSILON WITH HOOK SYMBOL
	0x03D3: "",           // GREEK UPSILON WITH ACUTE AND HOOK SYMBOL
	0x03D4: " Upsilon ",  // GREEK UPSILON WITH DIAERESIS AND HOOK SYMBOL
	0x03D5: " Phi ",      // GREEK PHI SYMBOL
	0x03D6: " Pi ",       // GREEK PI SYMBOL
	0x03D7: " Kai ",      // GREEK KAI SYMBOL
	0x03DA: " Stigma ",   // GREEK LETTER STIGMA
	0x03DB: " stigma ",   // GREEK SMALL LETTER STIGMA
	0x03DC: " Digamma ",  // GREEK LETTER DIGAMMA
	0x03DD: " digamma ",  // GREEK SMALL LETTER DIGAMMA
	0x03DE: " Koppa ",    // GREEK LETTER KOPPA
	0x03DF: " koppa ",    // GREEK SMALL LETTER KOPPA
	0x03E0: " Sampi ",    // GREEK LETTER SAMPI
	0x03E1: " sampi ",    // GREEK SMALL LETTER SAMPI
	0x03E2: "Sh",         // COPTIC CAPITAL LETTER SHEI
	0x03E3: "sh",         // COPTIC SMALL LETTER SHEI
	0x03E4: "F",          // COPTIC CAPITAL LETTER FEI
	0x03E5: "f",          // COPTIC SMALL LETTER FEI
	0x03E6: "Kh",         // COPTIC CAPITAL LETTER KHEI
	0x03E7: "kh",         // COPTIC SMALL LETTER KHEI
	0x03E8: "H",          // COPTIC CAPITAL LETTER HORI
	0x03E9: "h",          // COPTIC SMALL LETTER HORI
	0x03EA: "G",          // COPTIC CAPITAL LETTER GANGIA
	0x03EB: "g",          // COPTIC SMALL LETTER GANGIA
	0x03EC: "CH",         // COPTIC CAPITAL LETTER SHIMA
	0x03ED: "ch",         // COPTIC SMALL LETTER SHIMA
	0x03EE: "Ti",         // COPTIC CAPITAL LETTER DEI
	0x03EF: "ti",         // COPTIC SMALL LETTER DEI
	0x03F0: "k",          // GREEK KAPPA SYMBOL
	0x03F1: "r",          // GREEK RHO SYMBOL
	0x03F2: "c",          // GREEK LUNATE SIGMA SYMBOL
	0x03F3: "j",          // GREEK LETTER YOT
	0x03F4: "Th",         // GREEK CAPITAL THETA SYMBOL
	0x03F5: "e",          // GREEK LUNATE EPSILON SYMBOL
	0x03F6: "e",          // GREEK REVERSED LUNATE EPSILON SYMBOL
	0x03F7: "",           // GREEK CAPITAL LETTER SHO
	0x03F8: "",           // GREEK SMALL LETTER SHO
	0x03F9: "S",          // GREEK CAPITAL LUNATE SIGMA SYMBOL
	0x03FA: "",           // GREEK CAPITAL LETTER SAN
	0x03FB: "",           // GREEK SMALL LETTER SAN
	0x03FC: "r",          // GREEK RHO WITH STROKE SYMBOL
	0x03FD: "S",          // GREEK CAPITAL REVERSED LUNATE SIGMA SYMBOL
	0x03FE: "S",          // GREEK CAPITAL DOTTED LUNATE SIGMA SYMBOL
	0x03FF: "S",          // GREEK CAPITAL REVERSED DOTTED LUNATE SIGMA SYMBOL
	0x0400: "Ie",         // CYRILLIC CAPITAL LETTER IE WITH GRAVE
	0x0401: "Io",         // CYRILLIC CAPITAL LETTER IO
	0x0402: "Dj",         // CYRILLIC CAPITAL LETTER DJE
	0x0403: "Gj",         // CYRILLIC CAPITAL LETTER GJE
	0x0404: "Ie",         // CYRILLIC CAPITAL LETTER UKRAINIAN IE
	0x0405: "Dz",         // CYRILLIC CAPITAL LETTER DZE
	0x0406: "I",          // CYRILLIC CAPITAL LETTER BYELORUSSIAN-UKRAINIAN I
	0x0407: "Yi",         // CYRILLIC CAPITAL LETTER YI
	0x0408: "J",          // CYRILLIC CAPITAL LETTER JE
	0x0409: "Lj",         // CYRILLIC CAPITAL LETTER LJE
	0x040A: "Nj",         // CYRILLIC CAPITAL LETTER NJE
	0x040B: "Tsh",        // CYRILLIC CAPITAL LETTER TSHE
	0x040C: "Kj",         // CYRILLIC CAPITAL LETTER KJE
	0x040D: "I",          // CYRILLIC CAPITAL LETTER I WITH GRAVE
	0x040E: "U",          // CYRILLIC CAPITAL LETTER SHORT U
	0x040F: "Dzh",        // CYRILLIC CAPITAL LETTER DZHE
	0x0410: "A",          // CYRILLIC CAPITAL LETTER A
	0x0411: "B",          // CYRILLIC CAPITAL LETTER BE
	0x0412: "V",          // CYRILLIC CAPITAL LETTER VE
	0x0413: "G",          // CYRILLIC CAPITAL LETTER GHE
	0x0414: "D",          // CYRILLIC CAPITAL LETTER DE
	0x0415: "E",          // CYRILLIC CAPITAL LETTER IE
	0x0416: "Zh",         // CYRILLIC CAPITAL LETTER ZHE
	0x0417: "Z",          // CYRILLIC CAPITAL LETTER ZE
	0x0418: "I",          // CYRILLIC CAPITAL LETTER I
	0x0419: "I",          // CYRILLIC CAPITAL LETTER SHORT I
	0x041A: "K",          // CYRILLIC CAPITAL LETTER KA
	0x041B: "L",          // CYRILLIC CAPITAL LETTER EL
	0x041C: "M",          // CYRILLIC CAPITAL LETTER EM
	0x041D: "N",          // CYRILLIC CAPITAL LETTER EN
	0x041E: "O",          // CYRILLIC CAPITAL LETTER O
	0x041F: "P",          // CYRILLIC CAPITAL LETTER PE
	0x0420: "R",          // CYRILLIC CAPITAL LETTER ER
	0x0421: "S",          // CYRILLIC CAPITAL LETTER ES
	0x0422: "T",          // CYRILLIC CAPITAL LETTER TE
	0x0423: "U",          // CYRILLIC CAPITAL LETTER U
	0x0424: "F",          // CYRILLIC CAPITAL LETTER EF
	0x0425: "Kh",         // CYRILLIC CAPITAL LETTER HA
	0x0426: "Ts",         // CYRILLIC CAPITAL LETTER TSE
	0x0427: "Ch",         // CYRILLIC CAPITAL LETTER CHE
	0x0428: "Sh",         // CYRILLIC CAPITAL LETTER SHA
	0x0429: "Shch",       // CYRILLIC CAPITAL LETTER SHCHA
	0x042A: "'",          // CYRILLIC CAPITAL LETTER HARD SIGN
	0x042B: "Y",          // CYRILLIC CAPITAL LETTER YERU
	0x042C: "'",          // CYRILLIC CAPITAL LETTER SOFT SIGN
	0x042D: "E",          // CYRILLIC CAPITAL LETTER E
	0x042E: "Iu",         // CYRILLIC CAPITAL LETTER YU
	0x042F: "Ia",         // CYRILLIC CAPITAL LETTER YA
	0x0430: "a",          // CYRILLIC SMALL LETTER A
	0x0431: "b",          // CYRILLIC SMALL LETTER BE
	0x0432: "v",          // CYRILLIC SMALL LETTER VE
	0x0433: "g",          // CYRILLIC SMALL LETTER GHE
	0x0434: "d",          // CYRILLIC SMALL LETTER DE
	0x0435: "e",          // CYRILLIC SMALL LETTER IE
	0x0436: "zh",         // CYRILLIC SMALL LETTER ZHE
	0x0437: "z",          // CYRILLIC SMALL LETTER ZE
	0x0438: "i",          // CYRILLIC SMALL LETTER I
	0x0439: "i",          // CYRILLIC SMALL LETTER SHORT I
	0x043A: "k",          // CYRILLIC SMALL LETTER KA
	0x043B: "l",          // CYRILLIC SMALL LETTER EL
	0x043C: "m",          // CYRILLIC SMALL LETTER EM
	0x043D: "n",          // CYRILLIC SMALL LETTER EN
	0x043E: "o",          // CYRILLIC SMALL LETTER O
	0x043F: "p",          // CYRILLIC SMALL LETTER PE
	0x0440: "r",          // CYRILLIC SMALL LETTER ER
	0x0441: "s",          // CYRILLIC SMALL LETTER ES
	0x0442: "t",          // CYRILLIC SMALL LETTER TE
	0x0443: "u",          // CYRILLIC SMALL LETTER U
	0x0444: "f",          // CYRILLIC SMALL LETTER EF
	0x0445: "kh",         // CYRILLIC SMALL LETTER HA
	0x0446: "ts",         // CYRILLIC SMALL LETTER TSE
	0x0447: "ch",         // CYRILLIC SMALL LETTER CHE
	0x0448: "sh",         // CYRILLIC SMALL LETTER SHA
	0x0449: "shch",       // CYRILLIC SMALL LETTER SHCHA
	0x044A: "'",          // CYRILLIC SMALL LETTER HARD SIGN
	0x044B: "y",          // CYRILLIC SMALL LETTER YERU
	0x044C: "'",          // CYRILLIC SMALL LETTER SOFT SIGN
	0x044D: "e",          // CYRILLIC SMALL LETTER E
	0x044E: "iu",         // CYRILLIC SMALL LETTER YU
	0x044F: "ia",         // CYRILLIC SMALL LETTER YA
	0x0450: "ie",         // CYRILLIC SMALL LETTER IE WITH GRAVE
	0x0451: "io",         // CYRILLIC SMALL LETTER IO
	0x0452: "dj",         // CYRILLIC SMALL LETTER DJE
	0x0453: "gj",         // CYRILLIC SMALL LETTER GJE
	0x0454: "ie",         // CYRILLIC SMALL LETTER UKRAINIAN IE
	0x0455: "dz",         // CYRILLIC SMALL LETTER DZE
	0x0456: "i",          // CYRILLIC SMALL LETTER BYELORUSSIAN-UKRAINIAN I
	0x0457: "yi",         // CYRILLIC SMALL LETTER YI
	0x0458: "j",          // CYRILLIC SMALL LETTER JE
	0x0459: "lj",         // CYRILLIC SMALL LETTER LJE
	0x045A: "nj",         // CYRILLIC SMALL LETTER NJE
	0x045B: "tsh",        // CYRILLIC SMALL LETTER TSHE
	0x045C: "kj",         // CYRILLIC SMALL LETTER KJE
	0x045D: "i",          // CYRILLIC SMALL LETTER I WITH GRAVE
	0x045E: "u",          // CYRILLIC SMALL LETTER SHORT U
	0x045F: "dzh",        // CYRILLIC SMALL LETTER DZHE
	0x0460: "O",          // CYRILLIC CAPITAL LETTER OMEGA
	0x0461: "o",          // CYRILLIC SMALL LETTER OMEGA
	0x0462: "E",          // CYRILLIC CAPITAL LETTER YAT
	0x0463: "e",          // CYRILLIC SMALL LETTER YAT
	0x0464: "Ie",         // CYRILLIC CAPITAL LETTER IOTIFIED E
	0x0465: "ie",         // CYRILLIC SMALL LETTER IOTIFIED E
	0x0466: "E",          // CYRILLIC CAPITAL LETTER LITTLE YUS
	0x0467: "e",          // CYRILLIC SMALL LETTER LITTLE YUS
	0x0468: "Ie",         // CYRILLIC CAPITAL LETTER IOTIFIED LITTLE YUS
	0x0469: "ie",         // CYRILLIC SMALL LETTER IOTIFIED LITTLE YUS
	0x046A: "O",          // CYRILLIC CAPITAL LETTER BIG YUS
	0x046B: "o",          // CYRILLIC SMALL LETTER BIG YUS
	0x046C: "Io",         // CYRILLIC CAPITAL LETTER IOTIFIED BIG YUS
	0x046D: "io",         // CYRILLIC SMALL LETTER IOTIFIED BIG YUS
	0x046E: "Ks",         // CYRILLIC CAPITAL LETTER KSI
	0x046F: "ks",         // CYRILLIC SMALL LETTER KSI
	0x0470: "Ps",         // CYRILLIC CAPITAL LETTER PSI
	0x0471: "ps",         // CYRILLIC SMALL LETTER PSI
	0x0472: "F",          // CYRILLIC CAPITAL LETTER FITA
	0x0473: "f",          // CYRILLIC SMALL LETTER FITA
	0x0474: "Y",          // CYRILLIC CAPITAL LETTER IZHITSA
	0x0475: "y",          // CYRILLIC SMALL LETTER IZHITSA
	0x0476: "Y",          // CYRILLIC CAPITAL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT
	0x0477: "y",          // CYRILLIC SMALL LETTER IZHITSA WITH DOUBLE GRAVE ACCENT
	0x0478: "u",          // CYRILLIC CAPITAL LETTER UK
	0x0479: "u",          // CYRILLIC SMALL LETTER UK
	0x047A: "O",          // CYRILLIC CAPITAL LETTER ROUND OMEGA
	0x047B: "o",          // CYRILLIC SMALL LETTER ROUND OMEGA
	0x047C: "O",          // CYRILLIC CAPITAL LETTER OMEGA WITH TITLO
	0x047D: "o",          // CYRILLIC SMALL LETTER OMEGA WITH TITLO
	0x047E: "Ot",         // CYRILLIC CAPITAL LETTER OT
	0x047F: "ot",         // CYRILLIC SMALL LETTER OT
	0x0480: "Q",          // CYRILLIC CAPITAL LETTER KOPPA
	0x0481: "q",          // CYRILLIC SMALL LETTER KOPPA
	0x048C: "",           // CYRILLIC CAPITAL LETTER SEMISOFT SIGN
	0x048D: "",           // CYRILLIC SMALL LETTER SEMISOFT SIGN
	0x048E: "R'",         // CYRILLIC CAPITAL LETTER ER WITH TICK
	0x048F: "r'",         // CYRILLIC SMALL LETTER ER WITH TICK
	0x0490: "G'",         // CYRILLIC CAPITAL LETTER GHE WITH UPTURN
	0x0491: "g'",         // CYRILLIC SMALL LETTER GHE WITH UPTURN
	0x0492: "G'",         // CYRILLIC CAPITAL LETTER GHE WITH STROKE
	0x0493: "g'",         // CYRILLIC SMALL LETTER GHE WITH STROKE
	0x0494: "G'",         // CYRILLIC CAPITAL LETTER GHE WITH MIDDLE HOOK
	0x0495: "g'",         // CYRILLIC SMALL LETTER GHE WITH MIDDLE HOOK
	0x0496: "Zh'",        // CYRILLIC CAPITAL LETTER ZHE WITH DESCENDER
	0x0497: "zh'",        // CYRILLIC SMALL LETTER ZHE WITH DESCENDER
	0x0498: "Z'",         // CYRILLIC CAPITAL LETTER ZE WITH DESCENDER
	0x0499: "z'",         // CYRILLIC SMALL LETTER ZE WITH DESCENDER
	0x049A: "K'",         // CYRILLIC CAPITAL LETTER KA WITH DESCENDER
	0x049B: "k'",         // CYRILLIC SMALL LETTER KA WITH DESCENDER
	0x049C: "K'",         // CYRILLIC CAPITAL LETTER KA WITH VERTICAL STROKE
	0x049D: "k'",         // CYRILLIC SMALL LETTER KA WITH VERTICAL STROKE
	0x049E: "K'",         // CYRILLIC CAPITAL LETTER KA WITH STROKE
	0x049F: "k'",         // CYRILLIC SMALL LETTER KA WITH STROKE
	0x04A0: "K'",         // CYRILLIC CAPITAL LETTER BASHKIR KA
	0x04A1: "k'",         // CYRILLIC SMALL LETTER BASHKIR KA
	0x04A2: "N'",         // CYRILLIC CAPITAL LETTER EN WITH DESCENDER
	0x04A3: "n'",         // CYRILLIC SMALL LETTER EN WITH DESCENDER
	0x04A4: "Ng",         // CYRILLIC CAPITAL LIGATURE EN GHE
	0x04A5: "ng",         // CYRILLIC SMALL LIGATURE EN GHE
	0x04A6: "P'",         // CYRILLIC CAPITAL LETTER PE WITH MIDDLE HOOK
	0x04A7: "p'",         // CYRILLIC SMALL LETTER PE WITH MIDDLE HOOK
	0x04A8: "Kh",         // CYRILLIC CAPITAL LETTER ABKHASIAN HA
	0x04A9: "kh",         // CYRILLIC SMALL LETTER ABKHASIAN HA
	0x04AA: "S'",         // CYRILLIC CAPITAL LETTER ES WITH DESCENDER
	0x04AB: "s'",         // CYRILLIC SMALL LETTER ES WITH DESCENDER
	0x04AC: "T'",         // CYRILLIC CAPITAL LETTER TE WITH DESCENDER
	0x04AD: "t'",         // CYRILLIC SMALL LETTER TE WITH DESCENDER
	0x04AE: "U",          // CYRILLIC CAPITAL LETTER STRAIGHT U
	0x04AF: "u",          // CYRILLIC SMALL LETTER STRAIGHT U
	0x04B0: "U'",         // CYRILLIC CAPITAL LETTER STRAIGHT U WITH STROKE
	0x04B1: "u'",         // CYRILLIC SMALL LETTER STRAIGHT U WITH STROKE
	0x04B2: "Kh'",        // CYRILLIC CAPITAL LETTER HA WITH DESCENDER
	0x04B3: "kh'",        // CYRILLIC SMALL LETTER HA WITH DESCENDER
	0x04B4: "Tts",        // CYRILLIC CAPITAL LIGATURE TE TSE
	0x04B5: "tts",        // CYRILLIC SMALL LIGATURE TE TSE
	0x04B6: "Ch'",        // CYRILLIC CAPITAL LETTER CHE WITH DESCENDER
	0x04B7: "ch'",        // CYRILLIC SMALL LETTER CHE WITH DESCENDER
	0x04B8: "Ch'",        // CYRILLIC CAPITAL LETTER CHE WITH VERTICAL STROKE
	0x04B9: "ch'",        // CYRILLIC SMALL LETTER CHE WITH VERTICAL STROKE
	0x04BA: "H",          // CYRILLIC CAPITAL LETTER SHHA
	0x04BB: "h",          // CYRILLIC SMALL LETTER SHHA
	0x04BC: "Ch",         // CYRILLIC CAPITAL LETTER ABKHASIAN CHE
	0x04BD: "ch",         // CYRILLIC SMALL LETTER ABKHASIAN CHE
	0x04BE: "Ch'",        // CYRILLIC CAPITAL LETTER ABKHASIAN CHE WITH DESCENDER
	0x04BF: "ch'",        // CYRILLIC SMALL LETTER ABKHASIAN CHE WITH DESCENDER
	0x04C0: "`",          // CYRILLIC LETTER PALOCHKA
	0x04C1: "Zh",         // CYRILLIC CAPITAL LETTER ZHE WITH BREVE
	0x04C2: "zh",         // CYRILLIC SMALL LETTER ZHE WITH BREVE
	0x04C3: "K'",         // CYRILLIC CAPITAL LETTER KA WITH HOOK
	0x04C4: "k'",         // CYRILLIC SMALL LETTER KA WITH HOOK
	0x04C7: "N'",         // CYRILLIC CAPITAL LETTER EN WITH HOOK
	0x04C8: "n'",         // CYRILLIC SMALL LETTER EN WITH HOOK
	0x04CB: "Ch",         // CYRILLIC CAPITAL LETTER KHAKASSIAN CHE
	0x04CC: "ch",         // CYRILLIC SMALL LETTER KHAKASSIAN CHE
	0x04D0: "a",          // CYRILLIC CAPITAL LETTER A WITH BREVE
	0x04D1: "a",          // CYRILLIC SMALL LETTER A WITH BREVE
	0x04D2: "A",          // CYRILLIC CAPITAL LETTER A WITH DIAERESIS
	0x04D3: "a",          // CYRILLIC SMALL LETTER A WITH DIAERESIS
	0x04D4: "Ae",         // CYRILLIC CAPITAL LIGATURE A IE
	0x04D5: "ae",         // CYRILLIC SMALL LIGATURE A IE
	0x04D6: "Ie",         // CYRILLIC CAPITAL LETTER IE WITH BREVE
	0x04D7: "ie",         // CYRILLIC SMALL LETTER IE WITH BREVE
	0x04D8: "@",          // CYRILLIC CAPITAL LETTER SCHWA
	0x04D9: "@",          // CYRILLIC SMALL LETTER SCHWA
	0x04DA: "@",          // CYRILLIC CAPITAL LETTER SCHWA WITH DIAERESIS
	0x04DB: "@",          // CYRILLIC SMALL LETTER SCHWA WITH DIAERESIS
	0x04DC: "Zh",         // CYRILLIC CAPITAL LETTER ZHE WITH DIAERESIS
	0x04DD: "zh",         // CYRILLIC SMALL LETTER ZHE WITH DIAERESIS
	0x04DE: "Z",          // CYRILLIC CAPITAL LETTER ZE WITH DIAERESIS
	0x04DF: "z",          // CYRILLIC SMALL LETTER ZE WITH DIAERESIS
	0x04E0: "Dz",         // CYRILLIC CAPITAL LETTER ABKHASIAN DZE
	0x04E1: "dz",         // CYRILLIC SMALL LETTER ABKHASIAN DZE
	0x04E2: "I",          // CYRILLIC CAPITAL LETTER I WITH MACRON
	0x04E3: "i",          // CYRILLIC SMALL LETTER I WITH MACRON
	0x04E4: "I",          // CYRILLIC CAPITAL LETTER I WITH DIAERESIS
	0x04E5: "i",          // CYRILLIC SMALL LETTER I WITH DIAERESIS
	0x04E6: "O",          // CYRILLIC CAPITAL LETTER O WITH DIAERESIS
	0x04E7: "o",          // CYRILLIC SMALL LETTER O WITH DIAERESIS
	0x04E8: "O",          // CYRILLIC CAPITAL LETTER BARRED O
	0x04E9: "o",          // CYRILLIC SMALL LETTER BARRED O
	0x04EA: "O",          // CYRILLIC CAPITAL LETTER BARRED O WITH DIAERESIS
	0x04EB: "o",          // CYRILLIC SMALL LETTER BARRED O WITH DIAERESIS
	0x04EC: "E",          // CYRILLIC CAPITAL LETTER E WITH DIAERESIS
	0x04ED: "e",          // CYRILLIC SMALL LETTER E WITH DIAERESIS
	0x04EE: "U",          // CYRILLIC CAPITAL LETTER U WITH MACRON
	0x04EF: "u",          // CYRILLIC SMALL LETTER U WITH MACRON
	0x04F0: "U",          // CYRILLIC CAPITAL LETTER U WITH DIAERESIS
	0x04F1: "u",          // CYRILLIC SMALL LETTER U WITH DIAERESIS
	0x04F2: "U",          // CYRILLIC CAPITAL LETTER U WITH DOUBLE ACUTE
	0x04F3: "u",          // CYRILLIC SMALL LETTER U WITH DOUBLE ACUTE
	0x04F4: "Ch",         // CYRILLIC CAPITAL LETTER CHE WITH DIAERESIS
	0x04F5: "ch",         // CYRILLIC SMALL LETTER CHE WITH DIAERESIS
	0x04F8: "Y",          // CYRILLIC CAPITAL LETTER YERU WITH DIAERESIS
	0x04F9: "y",          // CYRILLIC SMALL LETTER YERU WITH DIAERESIS
	0x2010: "-",          // HYPHEN
	0x2011: "-",          // NON-BREAKING HYPHEN
	0x2012: "-",          // FIGURE DASH
	0x2013: "-",          // EN DASH
	0x2014: "--",         // EM DASH
	0x2015: "--",         // HORIZONTAL BAR
	0x2016: "||",         // DOUBLE VERTICAL LINE
	0x2017: "_",          // DOUBLE LOW LINE
	0x2018: "'",          // LEFT SINGLE QUOTATION MARK
	0x2019: "'",          // RIGHT SINGLE QUOTATION MARK
	0x201A: ",",          // SINGLE LOW-9 QUOTATION MARK
	0x201B: "'",          // SINGLE HIGH-REVERSED-9 QUOTATION MARK
	0x201C: "\"",         // LEFT DOUBLE QUOTATION MARK
	0x201D: "\"",         // RIGHT DOUBLE QUOTATION MARK
	0x201E: "\"",         // DOUBLE LOW-9 QUOTATION MARK
	0x201F: "\"",         // DOUBLE HIGH-REVERSED-9 QUOTATION MARK
	0x2020: "+",          // DAGGER
	0x2021: "++",         // DOUBLE DAGGER
	0x2022: "*",          // BULLET
	0x2023: "*>",         // TRIANGULAR BULLET
	0x2024: ".",          // ONE DOT LEADER
	0x2025: "..",         // TWO DOT LEADER
	0x2026: "...",        // HORIZONTAL ELLIPSIS
	0x2027: ".",          // HYPHENATION POINT
	0x2030: "%0",         // PER MILLE SIGN
	0x2031: "%00",        // PER TEN THOUSAND SIGN
	0x2032: "'",          // PRIME
	0x2033: "''",         // DOUBLE PRIME
	0x2034: "'''",        // TRIPLE PRIME
	0x2035: "`",          // REVERSED PRIME
	0x2036: "``",         // REVERSED DOUBLE PRIME
	0x2037: "```",        // REVERSED TRIPLE PRIME
	0x2038: "^",          // CARET
	0x203B: "*",          // REFERENCE MARK
	0x203C: "!!",         // DOUBLE EXCLAMATION MARK
	0x203D: "!?",         // INTERROBANG
	0x203E: "-",          // OVERLINE
	0x203F: "_",          // UNDERTIE
	0x2040: "-",          // CHARACTER TIE
	0x2041: "^",          // CARET INSERTION POINT
	0x2042: "***",        // ASTERISM
	0x2043: "--",         // HYPHEN BULLET
	0x2045: "-[",         // LEFT SQUARE BRACKET WITH QUILL
	0x2046: "]-",         // RIGHT SQUARE BRACKET WITH QUILL
	0x2047: "??",         // DOUBLE QUESTION MARK
	0x2048: "?!",         // QUESTION EXCLAMATION MARK
	0x2049: "!?",         // EXCLAMATION QUESTION MARK
	0x204A: "7",          // TIRONIAN SIGN ET
	0x204B: "PP",         // REVERSED PILCROW SIGN
	0x204C: "(]",         // BLACK LEFTWARDS BULLET
	0x204D: "[)",         // BLACK RIGHTWARDS BULLET
	0x204E: "*",          // LOW ASTERISK
	0x2052: "%",          // COMMERCIAL MINUS SIGN
	0x2053: "~",          // SWUNG DASH
	0x206F: "0",          // NOMINAL DIGIT SHAPES
	0x2070: "(0)",        // SUPERSCRIPT ZERO
	0x2071: "(i)",        // SUPERSCRIPT LATIN SMALL LETTER I
	0x2074: "(4)",        // SUPERSCRIPT FOUR
	0x2075: "(5)",        // SUPERSCRIPT FIVE
	0x2076: "(6)",        // SUPERSCRIPT SIX
	0x2077: "(7)",        // SUPERSCRIPT SEVEN
	0x2078: "(8)",        // SUPERSCRIPT EIGHT
	0x2079: "(9)",        // SUPERSCRIPT NINE
	0x207A: "(+)",        // SUPERSCRIPT PLUS SIGN
	0x207B: "(-)",        // SUPERSCRIPT MINUS
	0x207C: "(=)",        // SUPERSCRIPT EQUALS SIGN
	0x207D: "(()",        // SUPERSCRIPT LEFT PARENTHESIS
	0x207E: "())",        // SUPERSCRIPT RIGHT PARENTHESIS
	0x207F: "(n)",        // SUPERSCRIPT LATIN SMALL LETTER N
	0x2080: "(0)",        // SUBSCRIPT ZERO
	0x2081: "(1)",        // SUBSCRIPT ONE
	0x2082: "(2)",        // SUBSCRIPT TWO
	0x2083: "(3)",        // SUBSCRIPT THREE
	0x2084: "(4)",        // SUBSCRIPT FOUR
	0x2085: "(5)",        // SUBSCRIPT FIVE
	0x2086: "(6)",        // SUBSCRIPT SIX
	0x2087: "(7)",        // SUBSCRIPT SEVEN
	0x2088: "(8)",        // SUBSCRIPT EIGHT
	0x2089: "(9)",        // SUBSCRIPT NINE
	0x208A: "(+)",        // SUBSCRIPT PLUS SIGN
	0x208B: "(-)",        // SUBSCRIPT MINUS
	0x208C: "(=)",        // SUBSCRIPT EQUALS SIGN
	0x208D: "(()",        // SUBSCRIPT LEFT PARENTHESIS
	0x208E: "())",        // SUBSCRIPT RIGHT PARENTHESIS
	0x2090: "(a)",        // LATIN SUBSCRIPT SMALL LETTER A
	0x2091: "(e)",        // LATIN SUBSCRIPT SMALL LETTER E
	0x2092: "(o)",        // LATIN SUBSCRIPT SMALL LETTER O
	0x2093: "(x)",        // LATIN SUBSCRIPT SMALL LETTER X
	0x2095: "(h)",        // LATIN SUBSCRIPT SMALL LETTER H
	0x2096: "(k)",        // LATIN SUBSCRIPT SMALL LETTER K
	0x2097: "(l)",        // LATIN SUBSCRIPT SMALL LETTER L
	0x2098: "(m)",        // LATIN SUBSCRIPT SMALL LETTER M
	0x2099: "(n)",        // LATIN SUBSCRIPT SMALL LETTER N
	0x209A: "(p)",        // LATIN SUBSCRIPT SMALL LETTER P
	0x209B: "(s)",        // LATIN SUBSCRIPT SMALL LETTER S
	0x209C: "(t)",        // LATIN SUBSCRIPT SMALL LETTER T
	0x2120: " (sm) ",     // SERVICE MARK
	0x2122: " (tm) ",     // TRADE MARK SIGN
	0x2225: " || ",       // PARALLEL TO
	0x2264: " <= ",       // LESS-THAN OR EQUAL TO
	0x2265: " >= ",       // GREATER-THAN OR EQUAL TO
	0x2266: " <= ",       // LESS-THAN OVER EQUAL TO
	0x2267: " >= ",       // GREATER-THAN OVER EQUAL TO
	0x226A: " << ",       // MUCH LESS-THAN
	0x226B: " >> ",       // MUCH GREATER-THAN
	0x2276: " <> ",       // LESS-THAN OR GREATER-THAN
	0x2277: " >< ",       // GREATER-THAN OR LESS-THAN
	0x2329: " < ",        // LEFT-POINTING ANGLE BRACKET
	0x232A: " > ",        // RIGHT-POINTING ANGLE BRACKET
}

// FixMisusedLetters fixes Greek beta and German sharp s misuse
func FixMisusedLetters(str string, doHomoglyphs, isAuthor, isProse bool) string {

	var arry []string

	isReallyEszett := func(before, after string, mustBeAllCaps bool) bool {

		// trim at closest space before sharp s rune
		idx := strings.LastIndex(before, " ")
		if idx >= 0 {
			before = before[idx+1:]
		}

		// trim at closest space after sharp s rune
		idx = strings.Index(after, " ")
		if idx >= 0 {
			after = after[:idx]
		}

		isAllCaps := true

		// look at characters to the left
		if before == "" {
			// must not be first character
			return false
		}
		last := rune(0)
		for _, ch := range before {
			_, ok := germanRunes[ch]
			if !ok {
				return false
			}
			_, ok = germanCapitals[ch]
			if !ok {
				isAllCaps = false
			}
			last = ch
		}
		_, ok := germanVowels[last]
		if !ok {
			// character immediately to the left must be a vowel (actually, a long vowel or diphthong)
			// though after a consonant, if it really is German, it should become ss
			// perhaps that situation can be added to a future table for handling specific exceptions
			return false
		}

		// look at characters to the right
		for _, ch := range after {
			_, ok := germanRunes[ch]
			if !ok {
				return false
			}
			_, ok = germanCapitals[ch]
			if !ok {
				isAllCaps = false
			}
			last = ch
		}

		if mustBeAllCaps {
			// capital sharp S expects all letters to be capitalized
			if !isAllCaps {
				return false
			}
		} else if isAllCaps {
			// otherwise may be a gene name, should be using beta
			return false
		}

		return true
	}

	for i, ch := range str {

		if ch < 32 {
			// skip ASCII control characters
			continue
		}

		if doHomoglyphs {
			if ch == 976 || ch == 7517 || ch == 7526 {
				// convert curled (U+03D0), modifier (U+1D5D), and subscript (U+1D66) lookalikes
				// to lower-case Greek beta U+03B2)
				ch = 946
			} else if ch == 400 {
				// capital Latin open E (U+0190) to capital Greek Epsilon (U+0395)
				ch = 917
			} else if ch == 603 {
				// lower-case Latin open E (U+025B) to lower-case Greek epsilon (U+03B5)
				ch = 949
			}
		}

		if isAuthor {
			if ch == 946 || ch == 976 || ch == 7517 || ch == 7526 {
				// in author, replace lower-case Greek beta (U+03B2) with German sharp s (U+00DF),
				// also handles curled (U+03D0), modifier (U+1D5D), and subscript (U+1D66) lookalikes
				ch = 223
			} else if ch == 914 {
				// replace upper-case Greek Beta (U+0392) with Latin capital B
				ch = 66
			} else if ch == 34 {
				// and replace double quote by apostrophe
				ch = 39
			}
		}

		if isProse {
			if ch == 223 {
				// in text, German sharp s is occasionally correct
				if !isReallyEszett(str[:i], str[i+utf8.RuneLen(ch):], false) {
					// but in scientific papers usually should be Greek beta
					ch = 946
				}
			} else if ch == 946 || ch == 976 || ch == 7517 || ch == 7526 {
				// sometimes Greek beta should actually be German sharp s
				if isReallyEszett(str[:i], str[i+utf8.RuneLen(ch):], false) {
					ch = 223
				}
			} else if ch == 7838 {
				// also check whether to convert capitalized German sharp S (U+1E9E)
				if !isReallyEszett(str[:i], str[i+utf8.RuneLen(ch):], true) {
					ch = 946
				}
			}
		}

		arry = append(arry, string(ch))
	}

	str = strings.Join(arry, "")

	return str
}

// encodeAngleBracketsAndAmpersand reencodes angle brackets and ampersand in XML
func encodeAngleBracketsAndAmpersand(str string) string {

	if rfix != nil {
		for _, ch := range str {
			// check for presence of angle brackets or ampersand
			if ch == '<' || ch == '>' || ch == '&' {
				// replace all occurrences and return result
				str = rfix.Replace(str)
				return str
			}
		}
	}

	return str
}

// loadRuneTable reads and parses Unicode to ASCII conversion files
// e.g., "00B4    '    ACUTE ACCENT"
func loadRuneTable(rt map[rune]string, fromExec bool, dataPath, fileName string) bool {

	loaded := false

	fullPath := ""

	if fromExec {
		ex, eerr := os.Executable()
		if eerr == nil && ex != "" {
			fullPath = filepath.Dir(ex)
		}
	}

	fpath := filepath.Join(fullPath, dataPath, fileName)
	file, ferr := os.Open(fpath)

	if file != nil && ferr == nil {
		scanner := bufio.NewScanner(file)
		for scanner.Scan() {

			str := scanner.Text()
			if str == "" {
				continue
			}
			cols := strings.SplitN(str, "\t", 3)
			if len(cols) < 2 {
				continue
			}
			n, err := strconv.ParseUint(cols[0], 16, 32)
			if err != nil {
				continue
			}

			ch := rune(n)
			st := cols[1]
			rt[ch] = st

			loaded = true
		}
	}
	file.Close()

	if !loaded {
		DisplayWarning("Unable to load %s", fileName)
	}

	return loaded
}

// mutex to protect loading of external Unicode transformation files
var ulock sync.Mutex

// set to true at program start by init function
var needToLoadRuneTables bool

// TransformAccents converts accented letters and symbols to closest ASCII equivalent
func TransformAccents(str string, spellGreek, reEncode bool) string {

	var arry []string

	for _, ch := range str {

		st := ""
		ok := false

		if ch < 128 {
			// add printable 7-bit ASCII character directly
			if ch > 31 {
				arry = append(arry, string(ch))
			}
			continue
		}

		// leading and trailing spaces, if needed, are in the xxxRunes maps

		if spellGreek {
			// spells Greek letters (e.g., alpha, beta) for easier searching,
			// handles glyph variants, treats Latin letter open E as Greek epsilon
			st, ok = greekRunes[ch]
			if ok {
				arry = append(arry, st)
				continue
			}
		}

		// lookup remaining characters first in internal commonRunes table
		st, ok = commonRunes[ch]
		if ok {
			arry = append(arry, st)
			continue
		}

		if needToLoadRuneTables {

			// this section will run the first time TransformAccents needs to load external data tables
			ulock.Lock()

			// only the first thread will see needToLoadRuneTables true within the mutex locked section
			if needToLoadRuneTables {

				// DisplayWarning("First external character 0x%04X = '%c'", int(ch), ch)

				loadRuneTable(asciiRunes, true, "data", "unicode-ascii.txt")
				loadRuneTable(extraRunes, true, "data", "unicode-extra.txt")

				// set to false even if some external tables failed to read
				needToLoadRuneTables = false
			}

			ulock.Unlock()
		}

		// try asciiRunes table next
		st, ok = asciiRunes[ch]

		if !ok && ch > 0x33FF && ch < 0xFB00 {
			// then try extraRunes table
			st, ok = extraRunes[ch]
		}

		if ok {
			arry = append(arry, st)
		}
	}

	str = strings.Join(arry, "")

	if reEncode {
		// reencode angle brackets and ampersand in XML
		str = encodeAngleBracketsAndAmpersand(str)
	}

	return str
}

// initialize rfix replacer and empty external Runes maps before non-init functions are called
func init() {

	rfix = strings.NewReplacer(
		"<", "&lt;",
		">", "&gt;",
		"&", "&amp;",
	)

	needToLoadRuneTables = true

	asciiRunes = make(map[rune]string)
	extraRunes = make(map[rune]string)
}
