//==----- SYCLDeviceLibReqMask.cpp - get SYCL devicelib required Info ------==//
//
// Part of the LLVM Project, under the Apache License v2.0 with LLVM Exceptions.
// See https://llvm.org/LICENSE.txt for license information.
// SPDX-License-Identifier: Apache-2.0 WITH LLVM-exception
//
//===----------------------------------------------------------------------===//
//
// This function goes through input module's function list to detect all SYCL
// devicelib functions invoked. Each devicelib function invoked is included in
// one 'fallback' SPIR-V library loaded by SYCL runtime. After scanning all
// functions in input module, a mask telling which SPIR-V libraries are needed
// by input module indeed will be returned. This mask will be saved and used by
// SYCL runtime later.
//===----------------------------------------------------------------------===//

#include "llvm/SYCLLowerIR/SYCLDeviceLibReqMask.h"
#include "llvm/ADT/SmallVector.h"
#include "llvm/IR/Function.h"
#include "llvm/IR/Module.h"
#include "llvm/TargetParser/Triple.h"

#include <string>
#include <unordered_map>

static constexpr char DEVICELIB_FUNC_PREFIX[] = "__devicelib_";

using namespace llvm;
// We will gradually remove devicelib spv online linking path but keep
// bfloat16 devicelib spv as an exception for a short-term solution.
// For bfloat16 devicelib spv link, we won't rely on ReqMask but to embed
// the bits into executable if necessary
namespace {

using SYCLDeviceLibFuncMap = std::unordered_map<std::string, DeviceLibExt>;

// Please update SDLMap if any item is added to or removed from
// fallback device libraries in libdevice.
SYCLDeviceLibFuncMap SDLMap = {
    {"__devicelib_abs", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_acosf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_acoshf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_asinf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_asinhf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_atan2f", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_atanf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_atanhf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_cbrtf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_cosf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_coshf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_div", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_erfcf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_erff", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_exp2f", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_expf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_expm1f", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_fdimf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_fmaf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_fmodf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_frexpf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_hypotf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_ilogbf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_labs", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_ldiv", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_ldexpf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_lgammaf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_llabs", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_lldiv", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_log10f", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_log1pf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_log2f", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_logbf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_logf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_modff", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_nextafterf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_powf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_remainderf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_remquof", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_scalbnf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_sinf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_sinhf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_sqrtf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_tanf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_tanhf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_tgammaf", DeviceLibExt::cl_intel_devicelib_math},
    {"__devicelib_acos", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_acosh", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_asin", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_asinh", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_atan", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_atan2", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_atanh", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_cbrt", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_cos", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_cosh", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_erf", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_erfc", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_exp", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_exp2", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_expm1", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_fdim", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_fma", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_fmod", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_frexp", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_hypot", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_ilogb", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_ldexp", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_lgamma", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_log", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_log10", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_log1p", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_log2", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_logb", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_modf", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_nextafter", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_pow", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_remainder", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_remquo", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_scalbn", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_sin", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_sinh", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_sqrt", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_tan", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_tanh", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib_tgamma", DeviceLibExt::cl_intel_devicelib_math_fp64},
    {"__devicelib___divsc3", DeviceLibExt::cl_intel_devicelib_complex},
    {"__devicelib___mulsc3", DeviceLibExt::cl_intel_devicelib_complex},
    {"__devicelib_cabsf", DeviceLibExt::cl_intel_devicelib_complex},
    {"__devicelib_cacosf", DeviceLibExt::cl_intel_devicelib_complex},
    {"__devicelib_cacoshf", DeviceLibExt::cl_intel_devicelib_complex},
    {"__devicelib_cargf", DeviceLibExt::cl_intel_devicelib_complex},
    {"__devicelib_casinf", DeviceLibExt::cl_intel_devicelib_complex},
    {"__devicelib_casinhf", DeviceLibExt::cl_intel_devicelib_complex},
    {"__devicelib_catanf", DeviceLibExt::cl_intel_devicelib_complex},
    {"__devicelib_catanhf", DeviceLibExt::cl_intel_devicelib_complex},
    {"__devicelib_ccosf", DeviceLibExt::cl_intel_devicelib_complex},
    {"__devicelib_ccoshf", DeviceLibExt::cl_intel_devicelib_complex},
    {"__devicelib_cexpf", DeviceLibExt::cl_intel_devicelib_complex},
    {"__devicelib_cimagf", DeviceLibExt::cl_intel_devicelib_complex},
    {"__devicelib_clogf", DeviceLibExt::cl_intel_devicelib_complex},
    {"__devicelib_cpolarf", DeviceLibExt::cl_intel_devicelib_complex},
    {"__devicelib_cpowf", DeviceLibExt::cl_intel_devicelib_complex},
    {"__devicelib_cprojf", DeviceLibExt::cl_intel_devicelib_complex},
    {"__devicelib_crealf", DeviceLibExt::cl_intel_devicelib_complex},
    {"__devicelib_csinf", DeviceLibExt::cl_intel_devicelib_complex},
    {"__devicelib_csinhf", DeviceLibExt::cl_intel_devicelib_complex},
    {"__devicelib_csqrtf", DeviceLibExt::cl_intel_devicelib_complex},
    {"__devicelib_ctanf", DeviceLibExt::cl_intel_devicelib_complex},
    {"__devicelib_ctanhf", DeviceLibExt::cl_intel_devicelib_complex},
    {"__devicelib___divdc3", DeviceLibExt::cl_intel_devicelib_complex_fp64},
    {"__devicelib___muldc3", DeviceLibExt::cl_intel_devicelib_complex_fp64},
    {"__devicelib_cabs", DeviceLibExt::cl_intel_devicelib_complex_fp64},
    {"__devicelib_cacos", DeviceLibExt::cl_intel_devicelib_complex_fp64},
    {"__devicelib_cacosh", DeviceLibExt::cl_intel_devicelib_complex_fp64},
    {"__devicelib_carg", DeviceLibExt::cl_intel_devicelib_complex_fp64},
    {"__devicelib_casin", DeviceLibExt::cl_intel_devicelib_complex_fp64},
    {"__devicelib_casinh", DeviceLibExt::cl_intel_devicelib_complex_fp64},
    {"__devicelib_catan", DeviceLibExt::cl_intel_devicelib_complex_fp64},
    {"__devicelib_catanh", DeviceLibExt::cl_intel_devicelib_complex_fp64},
    {"__devicelib_ccos", DeviceLibExt::cl_intel_devicelib_complex_fp64},
    {"__devicelib_ccosh", DeviceLibExt::cl_intel_devicelib_complex_fp64},
    {"__devicelib_cexp", DeviceLibExt::cl_intel_devicelib_complex_fp64},
    {"__devicelib_cimag", DeviceLibExt::cl_intel_devicelib_complex_fp64},
    {"__devicelib_clog", DeviceLibExt::cl_intel_devicelib_complex_fp64},
    {"__devicelib_cpolar", DeviceLibExt::cl_intel_devicelib_complex_fp64},
    {"__devicelib_cpow", DeviceLibExt::cl_intel_devicelib_complex_fp64},
    {"__devicelib_cproj", DeviceLibExt::cl_intel_devicelib_complex_fp64},
    {"__devicelib_creal", DeviceLibExt::cl_intel_devicelib_complex_fp64},
    {"__devicelib_csin", DeviceLibExt::cl_intel_devicelib_complex_fp64},
    {"__devicelib_csinh", DeviceLibExt::cl_intel_devicelib_complex_fp64},
    {"__devicelib_csqrt", DeviceLibExt::cl_intel_devicelib_complex_fp64},
    {"__devicelib_ctan", DeviceLibExt::cl_intel_devicelib_complex_fp64},
    {"__devicelib_ctanh", DeviceLibExt::cl_intel_devicelib_complex_fp64},
    {"__devicelib_memcpy", DeviceLibExt::cl_intel_devicelib_cstring},
    {"__devicelib_memset", DeviceLibExt::cl_intel_devicelib_cstring},
    {"__devicelib_memcmp", DeviceLibExt::cl_intel_devicelib_cstring},
    {"__devicelib_assert_read", DeviceLibExt::cl_intel_devicelib_assert},
    {"__devicelib_assert_fail", DeviceLibExt::cl_intel_devicelib_assert},
    {"__devicelib_imf_llmax", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_llmin", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_max", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_min", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ullmax", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ullmin", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_umax", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_umin", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_brev", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_brevll", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_byte_perm", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ffs", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ffsll", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_clz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_clzll", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_popc", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_popcll", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_sad", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_usad", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_uhadd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_urhadd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_hadd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_rhadd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_mul24", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_umul24", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_mulhi", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_umulhi", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_mul64hi", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_umul64hi", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_saturatef", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fmaf", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_floorf", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ceilf", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_abs", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_llabs", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fabsf", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_truncf", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_rintf", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_nearbyintf", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_invf", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_sqrtf", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_rsqrtf", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fmaxf", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fminf", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_copysignf", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fast_exp10f", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fast_expf", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fast_logf", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fast_log2f", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fast_log10f", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fast_powf", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fast_fdividef", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fadd_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fadd_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fadd_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fadd_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fsub_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fsub_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fsub_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fsub_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fmul_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fmul_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fmul_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fmul_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fdiv_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fdiv_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fdiv_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fdiv_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fmaf_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fmaf_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fmaf_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fmaf_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_sqrtf_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_sqrtf_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_sqrtf_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_sqrtf_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_float2int_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_float2int_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_float2int_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_float2int_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_float2uint_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_float2uint_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_float2uint_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_float2uint_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_float2ll_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_float2ll_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_float2ll_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_float2ll_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_float2ull_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_float2ull_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_float2ull_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_float2ull_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_float_as_int", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_int2float_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_int2float_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_int2float_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_int2float_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_int_as_float", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_float_as_uint", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ll2float_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ll2float_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ll2float_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ll2float_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_uint2float_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_uint2float_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_uint2float_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_uint2float_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_uint_as_float", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ull2float_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ull2float_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ull2float_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ull2float_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_float2half_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_float2half_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_float2half_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_float2half_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half2float", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half2int_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half2int_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half2int_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half2int_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half2ll_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half2ll_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half2ll_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half2ll_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half2short_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half2short_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half2short_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half2short_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half2uint_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half2uint_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half2uint_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half2uint_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half2ull_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half2ull_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half2ull_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half2ull_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half2ushort_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half2ushort_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half2ushort_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half2ushort_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half_as_short", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_half_as_ushort", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_uint2half_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_uint2half_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_uint2half_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_uint2half_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ull2half_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ull2half_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ull2half_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ull2half_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ushort2half_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ushort2half_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ushort2half_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ushort2half_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ushort_as_half", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_int2half_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_int2half_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_int2half_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_int2half_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ll2half_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ll2half_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ll2half_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ll2half_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_short2half_rd", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_short2half_rn", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_short2half_ru", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_short2half_rz", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_short_as_half", DeviceLibExt::cl_intel_devicelib_imf},

    {"__devicelib_imf_fmaf16", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_floorf16", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_ceilf16", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fabsf16", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_truncf16", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_rintf16", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_nearbyintf16", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_invf16", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_sqrtf16", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_rsqrtf16", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fmaxf16", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_fminf16", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_copysignf16", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vabs2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vabs4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vabsss2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vabsss4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vneg2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vneg4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vnegss2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vnegss4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vabsdiffs2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vabsdiffs4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vabsdiffu2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vabsdiffu4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vadd2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vadd4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vaddss2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vaddss4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vaddus2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vaddus4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsub2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsub4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsubss2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsubss4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsubus2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsubus4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vavgs2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vavgs4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vavgu2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vavgu4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vhaddu2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vhaddu4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vcmpeq2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vcmpeq4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vcmpne2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vcmpne4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vcmpges2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vcmpges4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vcmpgeu2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vcmpgeu4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vcmpgts2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vcmpgts4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vcmpgtu2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vcmpgtu4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vcmples2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vcmples4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vcmpleu2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vcmpleu4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vcmplts2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vcmplts4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vcmpltu2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vcmpltu4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vmaxs2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vmaxs4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vmaxu2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vmaxu4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vmins2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vmins4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vminu2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vminu4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vseteq2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vseteq4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsetne2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsetne4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsetges2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsetges4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsetgeu2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsetgeu4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsetgts2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsetgts4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsetgtu2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsetgtu4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsetles2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsetles4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsetleu2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsetleu4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsetlts2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsetlts4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsetltu2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsetltu4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsads2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsads4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsadu2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vsadu4", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_viaddmax_s16x2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_viaddmax_s16x2_relu",
     DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_viaddmax_s32", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_viaddmax_s32_relu", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_viaddmax_u16x2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_viaddmax_u32", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_viaddmin_s16x2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_viaddmin_s16x2_relu",
     DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_viaddmin_s32", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_viaddmin_s32_relu", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_viaddmin_u16x2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_viaddmin_u32", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vibmax_s16x2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vibmax_s32", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vibmax_u16x2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vibmax_u32", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vibmin_s16x2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vibmin_s32", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vibmin_u16x2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vibmin_u32", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vimax3_s16x2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vimax3_s16x2_relu", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vimin3_s16x2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vimin3_s16x2_relu", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vimax3_s32", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vimax3_s32_relu", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vimin3_s32", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vimin3_s32_relu", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vimax3_u16x2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vimax3_u32", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vimin3_u16x2", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vimin3_u32", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vimax_s16x2_relu", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vimax_s32_relu", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vimin_s16x2_relu", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_vimin_s32_relu", DeviceLibExt::cl_intel_devicelib_imf},
    {"__devicelib_imf_double2half", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_double2bfloat16",
     DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_fma", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_floor", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_ceil", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_fabs", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_trunc", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_rint", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_rcp64h", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_nearbyint", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_inv", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_sqrt", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_rsqrt", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_fmax", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_fmin", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_copysign", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_dadd_rd", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_dadd_rn", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_dadd_ru", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_dadd_rz", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_dsub_rd", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_dsub_rn", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_dsub_ru", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_dsub_rz", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_dmul_rd", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_dmul_rn", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_dmul_ru", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_dmul_rz", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_ddiv_rd", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_ddiv_rn", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_ddiv_ru", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_ddiv_rz", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_double2float_rd",
     DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_double2float_rn",
     DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_double2float_ru",
     DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_double2float_rz",
     DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_double2int_rd",
     DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_double2int_rn",
     DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_double2int_ru",
     DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_double2int_rz",
     DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_double2uint_rd",
     DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_double2uint_rn",
     DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_double2uint_ru",
     DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_double2uint_rz",
     DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_double2hiint", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_double2loint", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_double2ll_rd", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_double2ll_rn", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_double2ll_ru", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_double2ll_rz", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_double2ull_rd",
     DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_double2ull_rn",
     DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_double2ull_ru",
     DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_double2ull_rz",
     DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_double_as_longlong",
     DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_hiloint2double",
     DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_int2double_rn",
     DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_ll2double_rd", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_ll2double_rn", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_ll2double_ru", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_ll2double_rz", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_ull2double_rd",
     DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_ull2double_rn",
     DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_ull2double_ru",
     DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_ull2double_rz",
     DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_uint2double_rn",
     DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_longlong_as_double",
     DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_fma_rd", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_fma_rn", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_fma_ru", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_fma_rz", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_sqrt_rd", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_sqrt_rn", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_sqrt_ru", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_sqrt_rz", DeviceLibExt::cl_intel_devicelib_imf_fp64},
    {"__devicelib_imf_bfloat162float",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_bfloat162int_rd",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_bfloat162int_rn",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_bfloat162int_ru",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_bfloat162int_rz",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_bfloat162short_rd",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_bfloat162short_rn",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_bfloat162short_ru",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_bfloat162short_rz",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_bfloat162ll_rd",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_bfloat162ll_rn",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_bfloat162ll_ru",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_bfloat162ll_rz",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_bfloat162uint_rd",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_bfloat162uint_rn",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_bfloat162uint_ru",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_bfloat162uint_rz",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_bfloat162ushort_rd",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_bfloat162ushort_rn",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_bfloat162ushort_ru",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_bfloat162ushort_rz",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_bfloat162ull_rd",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_bfloat162ull_rn",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_bfloat162ull_ru",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_bfloat162ull_rz",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_float2bfloat16",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_float2bfloat16_rd",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_float2bfloat16_rn",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_float2bfloat16_ru",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_float2bfloat16_rz",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_bfloat16_as_short",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_bfloat16_as_ushort",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_short_as_bfloat16",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_ushort_as_bfloat16",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_ushort2bfloat16_rd",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_ushort2bfloat16_rn",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_ushort2bfloat16_ru",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_ushort2bfloat16_rz",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_uint2bfloat16_rd",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_uint2bfloat16_rn",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_uint2bfloat16_ru",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_uint2bfloat16_rz",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_ull2bfloat16_rd",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_ull2bfloat16_rn",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_ull2bfloat16_ru",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_ull2bfloat16_rz",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_short2bfloat16_rd",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_short2bfloat16_rn",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_short2bfloat16_ru",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_short2bfloat16_rz",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_int2bfloat16_rd",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_int2bfloat16_rn",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_int2bfloat16_ru",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_int2bfloat16_rz",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_ll2bfloat16_rd",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_ll2bfloat16_rn",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_ll2bfloat16_ru",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_ll2bfloat16_rz",
     DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_fmabf16", DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_fmaxbf16", DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_fminbf16", DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_copysignbf16", DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_sqrtbf16", DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_rsqrtbf16", DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_fabsbf16", DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_rintbf16", DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_floorbf16", DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_ceilbf16", DeviceLibExt::cl_intel_devicelib_imf_bf16},
    {"__devicelib_imf_truncbf16", DeviceLibExt::cl_intel_devicelib_imf_bf16},
};

// Each fallback device library corresponds to one bit in "require mask" which
// is an unsigned int32. getDeviceLibBit checks which fallback device library
// is required for FuncName and returns the corresponding bit. The corresponding
// mask for each fallback device library is:
// cl_intel_devicelib_assert:        0x1
// cl_intel_devicelib_math:          0x2
// cl_intel_devicelib_math_fp64:     0x4
// cl_intel_devicelib_complex:       0x8
// cl_intel_devicelib_complex_fp64:  0x10
// cl_intel_devicelib_cstring :      0x20
// cl_intel_devicelib_imf:           0x40
// cl_intel_devicelib_imf_fp64:      0x80
// cl_intel_devicelib_imf_bf16:      0x100
// cl_intel_devicelib_bfloat16:      0x200
uint32_t getDeviceLibBits(const std::string &FuncName) {
  auto DeviceLibFuncIter = SDLMap.find(FuncName);
  return ((DeviceLibFuncIter == SDLMap.end())
              ? 0
              : 0x1 << (static_cast<uint32_t>(DeviceLibFuncIter->second) -
                        static_cast<uint32_t>(
                            DeviceLibExt::cl_intel_devicelib_assert)));
}

} // namespace

// For each device image module, we go through all functions which meets
// 1. The function name has prefix "__devicelib_"
// 2. The function is declaration which means it doesn't have function body
// And we don't expect non-spirv functions with "__devicelib_" prefix.
uint32_t llvm::getSYCLDeviceLibReqMask(const Module &M) {
  // Device libraries will be enabled only for spir-v module.
  if (!Triple(M.getTargetTriple()).isSPIROrSPIRV())
    return 0;
  uint32_t ReqMask = 0;
  for (const Function &SF : M) {
    if (SF.getName().starts_with(DEVICELIB_FUNC_PREFIX) && SF.isDeclaration()) {
      assert(SF.getCallingConv() == CallingConv::SPIR_FUNC);
      uint32_t DeviceLibBits = getDeviceLibBits(SF.getName().str());
      ReqMask |= DeviceLibBits;
    }
  }
  return ReqMask;
}

static llvm::SmallVector<StringRef, 14> BF16DeviceLibFuncs = {
    "__devicelib_ConvertFToBF16INTEL",
    "__devicelib_ConvertBF16ToFINTEL",
    "__devicelib_ConvertFToBF16INTELVec1",
    "__devicelib_ConvertBF16ToFINTELVec1",
    "__devicelib_ConvertFToBF16INTELVec2",
    "__devicelib_ConvertBF16ToFINTELVec2",
    "__devicelib_ConvertFToBF16INTELVec3",
    "__devicelib_ConvertBF16ToFINTELVec3",
    "__devicelib_ConvertFToBF16INTELVec4",
    "__devicelib_ConvertBF16ToFINTELVec4",
    "__devicelib_ConvertFToBF16INTELVec8",
    "__devicelib_ConvertBF16ToFINTELVec8",
    "__devicelib_ConvertFToBF16INTELVec16",
    "__devicelib_ConvertBF16ToFINTELVec16",
};

bool llvm::isSYCLDeviceLibBF16Used(const Module &M) {
  if (!Triple(M.getTargetTriple()).isSPIROrSPIRV())
    return false;

  for (auto Fn : BF16DeviceLibFuncs) {
    Function *BF16Func = M.getFunction(Fn);
    if (BF16Func && BF16Func->isDeclaration())
      return true;
  }

  return false;
}

bool llvm::isBF16DeviceLibFuncDecl(const Function &F) {
  if (!F.isDeclaration() || !F.getName().starts_with(DEVICELIB_FUNC_PREFIX))
    return false;
  for (auto BFunc : BF16DeviceLibFuncs) {
    if (!F.getName().compare(BFunc))
      return true;
  }

  return false;
}
