#!/usr/bin/env python3

# OpenCL built-in library: type conversion functions
#
# Copyright (c) 2013 Victor Oliveira <victormatheus@gmail.com>
# Copyright (c) 2013 Jesse Towner <jessetowner@lavabit.com>
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.

# This script generates the file convert_type.cl, which contains all of the
# convertion functions in the form:
#
# __clc_convert_<destTypen><_sat><_roundingMode>(<sourceTypen>)

import os
import sys
from os.path import dirname, join, abspath

sys.path.insert(0, abspath(join(dirname(__file__), "..", "..", "..", "generic")))

from gen_convert_common import (
    types,
    int_types,
    signed_types,
    unsigned_types,
    unsigned_type,
    half_sizes,
    float_types,
    sizeof_type,
    limit_max,
    limit_min,
    vector_sizes,
    bool_type,
    saturation,
    conditional_guard,
    close_conditional_guard,
    rounding_modes,
    clc_core_fn_name,
)

print(
    """/* !!!! AUTOGENERATED FILE generated by convert_type.py !!!!!

   DON'T CHANGE THIS FILE. MAKE YOUR CHANGES TO convert_type.py AND RUN:
   $ ./generate-conversion-type-cl.sh

   clc core type conversion functions

   Copyright (c) 2013 Victor Oliveira <victormatheus@gmail.com>
   Copyright (c) 2013 Jesse Towner <jessetowner@lavabit.com>

   Permission is hereby granted, free of charge, to any person obtaining a copy
   of this software and associated documentation files (the "Software"), to deal
   in the Software without restriction, including without limitation the rights
   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
   copies of the Software, and to permit persons to whom the Software is
   furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
   THE SOFTWARE.
*/

#include <core/clc_core.h>
#include <clc/integer/clc_abs.h>
#include <clc/shared/clc_clamp.h>
#include <clc/shared/clc_max.h>
#include <clc/shared/clc_min.h>
#include <libspirv/spirv.h>

#ifdef cl_khr_fp16
#pragma OPENCL EXTENSION cl_khr_fp16 : enable
#endif

#ifdef cl_khr_fp64
#pragma OPENCL EXTENSION cl_khr_fp64 : enable

#if defined(__EMBEDDED_PROFILE__) && !defined(cles_khr_int64)
#error Embedded profile that supports cl_khr_fp64 also has to support cles_khr_int64
#endif

#endif

#ifdef cles_khr_int64
#pragma OPENCL EXTENSION cles_khr_int64 : enable
#endif

"""
)

#
# Default Conversions
#
# All conversions are in accordance with the OpenCL specification,
# which cites the C99 conversion rules.
#
# Casting from floating point to integer results in conversions
# with truncation, so it should be suitable for the default convert
# functions.
#
# Conversions from integer to floating-point, and floating-point to
# floating-point through casting is done with the default rounding
# mode. While C99 allows dynamically changing the rounding mode
# during runtime, it is not a supported feature in OpenCL according
# to Section 7.1 - Rounding Modes in the OpenCL 1.2 specification.
#
# Therefore, we can assume for optimization purposes that the
# rounding mode is fixed to round-to-nearest-even. Platform target
# authors should ensure that the rounding-control registers remain
# in this state, and that this invariant holds.
#
# Also note, even though the OpenCL specification isn't entirely
# clear on this matter, we implement all rounding mode combinations
# even for integer-to-integer conversions. When such a conversion
# is used, the rounding mode is ignored.
#


def generate_default_conversion(src, dst, mode):
    close_conditional = conditional_guard(src, dst)

    # scalar conversions
    print(
        """_CLC_DEF _CLC_OVERLOAD
{DST} {FN}({SRC} x)
{{
  return ({DST})x;
}}
""".format(
            FN=clc_core_fn_name(dst, mode=mode), SRC=src, DST=dst, M=mode
        )
    )

    # vector conversions, done through decomposition to components
    for size, half_size in half_sizes:
        conv_fn = clc_core_fn_name(dst, size=size, mode=mode)
        # default mode, so drop the mode for the called function
        half_size_conv_fn = clc_core_fn_name(dst, size=half_size)
        print(
            """_CLC_DEF _CLC_OVERLOAD
{DST}{N} {FN}({SRC}{N} x)
{{
  return ({DST}{N})({HALF_SIZE_FN}(x.lo), {HALF_SIZE_FN}(x.hi));
}}
""".format(
                FN=conv_fn, HALF_SIZE_FN=half_size_conv_fn, SRC=src, DST=dst, N=size
            )
        )

    # 3-component vector conversions
    print(
        """_CLC_DEF _CLC_OVERLOAD
{DST}3 {FN}({SRC}3 x)
{{
  return ({DST}3)({VEC_2_FN}(x.s01), {SCALAR_FN}(x.s2));
}}""".format(
            FN=clc_core_fn_name(dst, size="3", mode=mode),
            VEC_2_FN=clc_core_fn_name(dst, size="2"),
            SCALAR_FN=clc_core_fn_name(dst),
            SRC=src,
            DST=dst,
        )
    )

    close_conditional_guard(close_conditional)


for src in types:
    for dst in types:
        generate_default_conversion(src, dst, "")

for src in int_types:
    for dst in int_types:
        for mode in rounding_modes:
            generate_default_conversion(src, dst, mode)

#
# Saturated Conversions To Integers
#
# These functions are dependent on the unsaturated conversion functions
# generated above, and use clamp, max, min, and select to eliminate
# branching and vectorize the conversions.
#
# Again, as above, we allow all rounding modes for integer-to-integer
# conversions with saturation.
#


def generate_saturated_conversion(src, dst, size):
    # Header
    close_conditional = conditional_guard(src, dst)
    print(
        """_CLC_DEF _CLC_OVERLOAD
{DST}{N} {FN}({SRC}{N} x)
{{""".format(
            FN=clc_core_fn_name(dst, size=size, sat="_sat"), DST=dst, SRC=src, N=size
        )
    )

    # FIXME: This is a work around for lack of select function with
    # signed third argument when the first two arguments are unsigned types.
    # We cast to the signed type for sign-extension, then do a bitcast to
    # the unsigned type.
    bool_conv_fn = clc_core_fn_name(bool_type[dst], size=size)
    if dst in unsigned_types:
        bool_prefix = "as_{DST}{N}({BOOL_CONV_FN}".format(
            DST=dst, BOOL_CONV_FN=bool_conv_fn, N=size
        )
        bool_suffix = ")"
    else:
        bool_prefix = bool_conv_fn
        bool_suffix = ""

    # Body
    if src == dst:

        # Conversion between same types
        print("  return x;")

    elif src in float_types:

        # Conversion from float to int
        print(
            """  {DST}{N} y = {CONV_DEFAULT}(x);
  y = {BP}(x < ({SRC}{N}){DST_MIN}){BS} ? y : ({DST}{N}){DST_MIN};
  y = {BP}(x > ({SRC}{N}){DST_MAX}){BS} ? y : ({DST}{N}){DST_MAX};
  return y;""".format(
                SRC=src,
                DST=dst,
                N=size,
                CONV_DEFAULT=clc_core_fn_name(dst, size=size),
                DST_MIN=limit_min[dst],
                DST_MAX=limit_max[dst],
                BP=bool_prefix,
                BS=bool_suffix,
            )
        )

    else:

        # Integer to integer convesion with sizeof(src) == sizeof(dst)
        if sizeof_type[src] == sizeof_type[dst]:
            if src in unsigned_types:
                print(
                    "  x = __clc_min(x, ({SRC}){DST_MAX});".format(
                        SRC=src, DST_MAX=limit_max[dst]
                    )
                )
            else:
                print("  x = __clc_max(x, ({SRC})0);".format(SRC=src))

        # Integer to integer conversion where sizeof(src) > sizeof(dst)
        elif sizeof_type[src] > sizeof_type[dst]:
            if src in unsigned_types:
                print(
                    "  x = __clc_min(x, ({SRC}){DST_MAX});".format(
                        SRC=src, DST_MAX=limit_max[dst]
                    )
                )
            else:
                print(
                    "  x = __clc_clamp(x, ({SRC}){DST_MIN}, ({SRC}){DST_MAX});".format(
                        SRC=src, DST_MIN=limit_min[dst], DST_MAX=limit_max[dst]
                    )
                )

        # Integer to integer conversion where sizeof(src) < sizeof(dst)
        elif src not in unsigned_types and dst in unsigned_types:
            print("  x = __clc_max(x, ({SRC})0);".format(SRC=src))

        print("  return {FN}(x);".format(FN=clc_core_fn_name(dst, size=size)))

    # Footer
    print("}")
    close_conditional_guard(close_conditional)


for src in types:
    for dst in int_types:
        for size in vector_sizes:
            generate_saturated_conversion(src, dst, size)


def generate_saturated_conversion_with_rounding(src, dst, size, mode):
    # Header
    close_conditional = conditional_guard(src, dst)

    # Body
    print(
        """_CLC_DEF _CLC_OVERLOAD
{DST}{N} {FN}({SRC}{N} x)
{{
  return {FN_DEFAULT}(x);
}}
""".format(
            FN=clc_core_fn_name(dst, size=size, sat="_sat", mode=mode),
            FN_DEFAULT=clc_core_fn_name(dst, size=size, mode=mode),
            DST=dst,
            SRC=src,
            N=size,
        )
    )

    # Footer
    close_conditional_guard(close_conditional)


for src in int_types:
    for dst in int_types:
        for size in vector_sizes:
            for mode in rounding_modes:
                generate_saturated_conversion_with_rounding(src, dst, size, mode)

#
# Conversions To/From Floating-Point With Rounding
#
# Note that we assume as above that casts from floating-point to
# integer are done with truncation, and that the default rounding
# mode is fixed to round-to-nearest-even, as per C99 and OpenCL
# rounding rules.
#
# These functions rely on the use of abs, ceil, fabs, floor,
# nextafter, sign, rint and the above generated conversion functions.
#
# Only conversions to integers can have saturation.
#


def generate_float_conversion(src, dst, size, mode, sat):
    # Header
    close_conditional = conditional_guard(src, dst)
    print(
        """_CLC_DEF _CLC_OVERLOAD
  {DST}{N} {FN}({SRC}{N} x)
{{""".format(
            FN=clc_core_fn_name(dst, size=size, sat=sat, mode=mode),
            SRC=src,
            DST=dst,
            N=size,
        )
    )

    # Perform conversion
    if dst in int_types:
        if mode == "_rte":
            print("  x = __spirv_ocl_rint(x);")
        elif mode == "_rtp":
            print("  x = __spirv_ocl_ceil(x);")
        elif mode == "_rtn":
            print("  x = __spirv_ocl_floor(x);")
        print("  return {FN}(x);".format(FN=clc_core_fn_name(dst, size=size, sat=sat)))
    elif mode == "_rte":
        print("  return {FN}(x);".format(FN=clc_core_fn_name(dst, size=size)))
    else:
        print(
            "  {DST}{N} r = {FN}(x);".format(
                FN=clc_core_fn_name(dst, size=size), DST=dst, N=size
            )
        )
        print(
            "  {SRC}{N} y = {FN}(r);".format(
                FN=clc_core_fn_name(src, size=size), SRC=src, N=size
            )
        )
        if mode == "_rtz":
            if src in int_types:
                print(
                    "  {USRC}{N} abs_x = __clc_abs(x);".format(
                        USRC=unsigned_type[src], N=size
                    )
                )
                print(
                    "  {USRC}{N} abs_y = __clc_abs(y);".format(
                        USRC=unsigned_type[src], N=size
                    )
                )
            else:
                print("  {SRC}{N} abs_x = __spirv_ocl_fabs(x);".format(SRC=src, N=size))
                print("  {SRC}{N} abs_y = __spirv_ocl_fabs(y);".format(SRC=src, N=size))
            print(
                "  {DST}{N} sel = {BOOL_CONVERT}(abs_y > abs_x) ? r:  __spirv_ocl_nextafter(r, __spirv_ocl_sign(r) * ({DST}{N})-INFINITY);".format(
                    DST=dst,
                    N=size,
                    BOOL_CONVERT=clc_core_fn_name(bool_type[dst], size=size),
                )
            )
            if dst == "half" and src in int_types and sizeof_type[src] >= 2:
                dst_max = limit_max[dst]
                # short is 16 bits signed, so the maximum value rounded to zero
                # is 0x1.ffcp+14 (0x1p+15 == 32768 > 0x7fff == 32767)
                if src == "short":
                    dst_max = "0x1.ffcp+14"
                print(
                    "  return __clc_clamp(sel, ({DST}{N}){DST_MIN}, ({DST}{N}){DST_MAX});".format(
                        DST=dst, N=size, DST_MIN=limit_min[dst], DST_MAX=dst_max
                    )
                )
            else:
                print("  return sel;")

        if mode == "_rtp":
            print(
                "  {DST}{N} sel = {BOOL_CONVERT}(y < x) ? r : __spirv_ocl_nextafter(r, ({DST}{N})INFINITY);".format(
                    DST=dst,
                    N=size,
                    BOOL_CONVERT=clc_core_fn_name(bool_type[dst], size=size),
                )
            )
            if dst == "half" and src in int_types and sizeof_type[src] >= 2:
                print(
                    "  return __clc_max(sel, ({DST}{N}){DST_MIN});".format(
                        DST=dst, N=size, DST_MIN=limit_min[dst]
                    )
                )
            else:
                print("  return sel;")

        if mode == "_rtn":
            print(
                "  {DST}{N} sel = {BOOL_CONVERT}(y > x) ? r : __spirv_ocl_nextafter(r, ({DST}{N})-INFINITY);".format(
                    DST=dst,
                    N=size,
                    BOOL_CONVERT=clc_core_fn_name(bool_type[dst], size=size),
                )
            )
            if dst == "half" and src in int_types and sizeof_type[src] >= 2:
                dst_max = limit_max[dst]
                # short is 16 bits signed, so the maximum value rounded to
                # negative infinity is 0x1.ffcp+14 (0x1p+15 == 32768 > 0x7fff
                # == 32767)
                if src == "short":
                    dst_max = "0x1.ffcp+14"
                print(
                    "  return __clc_min(sel, ({DST}{N}){DST_MAX});".format(
                        DST=dst, N=size, DST_MAX=dst_max
                    )
                )
            else:
                print("  return sel;")


    # Footer
    print("}")
    close_conditional_guard(close_conditional)


for src in float_types:
    for dst in int_types:
        for size in vector_sizes:
            for mode in rounding_modes:
                for sat in saturation:
                    generate_float_conversion(src, dst, size, mode, sat)


for src in types:
    for dst in float_types:
        for size in vector_sizes:
            for mode in rounding_modes:
                generate_float_conversion(src, dst, size, mode, "")
