/* NVTV i810 backend -- Dirk Thierbach <dthierbach@gmx.de>
 *
 * This file is part of nvtv, a tool for tv-output on NVidia cards.
 * 
 * nvtv is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * nvtv is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * $Id: back_i810.c,v 1.24 2004/01/27 20:38:39 dthierbach Exp $
 *
 * Contents:
 *
 * Backend for Intel i810, i810dc100, i810e, i815, and i815e cards; 
 * direct access.
 *
 */

#include "local.h" /* before everything else */

#include <stdio.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#include <fcntl.h>

#include "xfree.h" /* via back_direct.h */
#include "mmio.h"
#include "bitmask.h"
#include "backend.h"
#include "card_direct.h"
#include "back_direct.h"
#include "back_null.h"
#include "back_i810.h"
#include "tv_i810.h"
#include "data.h"
#include "xf86PciInfo.h"

/* -------- State of common direct backend driver -------- */

static CardPtr bi810_card;
static int bi810_fd = -1; /* fd/handle for mmap */

static I810Rec bi810_i810 = 
{
  TvChain: NULL,
  TvBusses: {NULL, NULL, NULL},
  TvMaxBus: 0,
};

static DataFunc *bi810_data = NULL;

static TVSettings bi810_set;
static TVRegs bi810_regs;

Bool bi810_set_avail = FALSE;

/* -------- Common backend driver routines -------- */

/*
 *  Update crt regs from hardware
 */

void bi810_updateCrt (void)
{
  /* doesn't manage non-tv registers yet, so no need to get them */
}

/*
 *  Init card: read crt regs, probe chips 
 */

void bi810_initCard (void)
{
  bi810_updateCrt ();
  if (I810TvBusInit (&bi810_i810)) {
    bi810_probeChips ();
  }
}

/*
 *  Map Intel 810 card memory 
 */

void bi810_mapMem (CardPtr card, I810Ptr pI810)
{
  bi810_fd = openDevMem (card);
  pI810->MMIOBase = mapDevMem(card, bi810_fd, card->reg_base, 0x80000);
}

void bi810_unmapMem (CardPtr card, I810Ptr pI810)
{
  unmapDevMem(card, (unsigned long) pI810->MMIOBase, 0x80000);
  closeDevMem (card, bi810_fd);
  bi810_fd = -1;
}

/* 
 *  Open card, map mem, call probe
 */

void bi810_openCard (CardPtr card)
{
  RAISE (MSG_DEBUG, "bi810 open");
  bi810_card = card;
  bi810_i810.Chipset = card->pci_id;
  bi810_i810.arch.major = 0x10;
  bi810_i810.arch.heads = 1;
  bi810_card->arch = "Intel";
  switch (card->pci_id) {
    case PCI_CHIP_I810: 
    case PCI_CHIP_I810_DC100:
    case PCI_CHIP_I810_E:
    case PCI_CHIP_I810_IG:
    case PCI_CHIP_I815: 
      bi810_i810.arch.major = 0x10;
      bi810_card->arch = "Intel 810/815"; /* FIXME: Use sub vendor? */
      break;
    case PCI_CHIP_I830:
      bi810_i810.arch.major = 0x30;
      bi810_i810.arch.heads = 2;
      bi810_card->arch = "Intel 830";
      break;
    case PCI_CHIP_I845: 
      bi810_i810.arch.major = 0x45;
      bi810_card->arch = "Intel 845";
      break;
    case PCI_CHIP_I855: 
      bi810_i810.arch.major = 0x55;
      bi810_card->arch = "Intel 855";
      break;
    case PCI_CHIP_I865: 
      bi810_i810.arch.major = 0x65;
      bi810_card->arch = "Intel 865";
      break;
    default:
      RAISE (MSG_ERROR, "bi810_openCard: Unknown pci id %04x", 
	     card->pci_id);
      break;
  }
  bi810_mapMem (card, &bi810_i810);
  bi810_initCard (); /* does probe */
  /* FIXME Do 830 */
}

/*
 * Close card, unmap memory.
 */

void bi810_closeCard (void)
{
  TVDestroyAll (&bi810_i810.TvChain, &bi810_i810.TvBusses, 
		bi810_i810.TvMaxBus);
  bi810_unmapMem (bi810_card, &bi810_i810);
}

void bi810_setHeads (int main, int tv, int video)
{
  RAISE (MSG_DEBUG, "bi810_setHeads");
}

void bi810_getHeads (int *main, int *tv, int *video, int *max) 
{
  RAISE (MSG_DEBUG, "bi810_getHeads");
  if (tv)    *tv = 1;
  if (video) *video = 1;
  if (max)   *max = 1;
}

void bi810_getHeadDev (int head, int *devFlags) 
{
  RAISE (MSG_DEBUG, "bi810_getHeadDev");
  /* FIXME */
  if (devFlags) *devFlags = IntelGetDevFlags (&bi810_i810, 0);
}

void bi810_probeChips (void)
{
  RAISE (MSG_DEBUG, "bi810_probe");
  bdir_freeChips (bi810_card);
  TVDestroyChainDevices (&bi810_i810.TvChain, &bi810_i810.TvBusses, 
			 bi810_i810.TvMaxBus);
  I810ProbeTvDevices (&bi810_i810);
  bi810_card->chips = bdir_copyChips (bi810_i810.TvChain);
  bi810_setChip (bi810_card->chips, FALSE); /* don't init */
}

void bi810_setChip (ChipPtr chip, Bool init)
{
  RAISE (MSG_DEBUG, "bi810_setChip %s %i", chip ? chip->name : "NULL", init);
  if (!chip) return;
  I810SetTvDevice (&bi810_i810, (I2CChainPtr) chip->private, init);
  bi810_data = data_func (CARD_I810, chip->type);
  bi810_data->defaults (&bi810_set);
  bi810_set_avail = TRUE;
}

/*
 *  Apply any changes: process settings, call lower level part
 */

void bi810_apply (void)
{
  TVRegs temp;

  temp = bi810_regs;
  if ((temp.devFlags & DEV_TELEVISION) && bi810_set_avail) {
    bi810_data->clamp (&bi810_set, &bi810_regs);
    bi810_data->setup (&bi810_set, &temp);
  }
  IntelSetTvMode (&bi810_i810, &temp);
}

/*
 * change settings, update tv chip registers 
 */

void bi810_setSettings (TVSettings *set)
{
  RAISE (MSG_DEBUG, "bi810_setSettings");
  if (set) {
    bi810_set = *set;
    bi810_set_avail = TRUE;
  } else {
    bi810_set_avail = FALSE;
  }
  bi810_apply ();
}

/*
 *  Get current settings
 */

void bi810_getSettings (TVSettings *set)
{
  RAISE (MSG_DEBUG, "bi810_getSettings");
  if (set && bi810_set_avail) *set = bi810_set;
}

/*
 *  Set mode by crt and tv regs
 */

void bi810_setMode (TVRegs *r)
{
  RAISE (MSG_DEBUG, "bi810_setMode");
  if (r) bi810_regs = *r;
  bi810_apply ();
}

/* 
 *  Get current mode. Update crt from hardware.
 */

void bi810_getMode (TVRegs *r)
{
  RAISE (MSG_DEBUG, "bi810_getMode");
  bi810_updateCrt ();
  if (r) *r = bi810_regs;
}

/*
 *  Set both mode and settings
 */

void bi810_setModeSettings (TVRegs *r, TVSettings *set)
{
  RAISE (MSG_DEBUG, "bi810_setModeSettings");
  if (set) {
    bi810_set = *set;
    bi810_set_avail = TRUE;
  } else {
    bi810_set_avail = FALSE;
  }
  if (r) bi810_regs = *r;
  bi810_apply ();
}

void bi810_setTestImage (TVEncoderRegs *tv, TVSettings *set)
{
  RAISE (MSG_DEBUG, "bi810_setTestImage");
  if (set) bi810_set = *set;
  if (tv)  bi810_regs.enc  = *tv;
#if 0 /* FIXME */
  if (set) {
    bi810_data->setup (&bi810_set, ...);
  }
  /* FIXME: Settings?? */
#endif
  I810SetTestImage (&bi810_i810, &bi810_regs.enc);
}

long bi810_getStatus (int index)
{
  RAISE (MSG_DEBUG, "bi810_getStatus");
  return I810GetTvStatus (&bi810_i810, index);
}

TVConnect bi810_getConnection (void)
{
  RAISE (MSG_DEBUG, "bi810_getConnection");
  return I810GetTvConnect (&bi810_i810);
}

/*
 *  List all modes (by system)
 */

int bi810_listModes (TVSystem system, TVMode *(list[]))
{
  return bdir_listModes (CARD_I810, bi810_data, system, list);
}

/*
 *  Find mode by size (and resolution). 
 */

Bool bi810_findBySize (TVSystem system, int xres, int yres, char *size, 
    TVMode *mode)
{
  return bdir_findBySize (CARD_I810, bi810_data, system, xres, yres, 
			  size, mode);
}

/*
 *  Find mode by overscan (and resolution)
 */

Bool bi810_findByOverscan (TVSystem system, int xres, int yres, 
    double hoc, double voc, TVMode *mode)
{
  return bdir_findByOverscan (CARD_I810, bi810_data, system, xres, yres, 
			      hoc, voc, mode);
}

#ifdef DEBUG_PROBE

#define I810_RD32(p,i)    	MMIO_IN32(((p).MMIOBase), (i))
#define I810_P_RD32(p,h,i)	MMIO_IN32(((p).MMIOBase), (i+0x1000*h))

void bi830_probeCard (void)
{
  int i;

  printf ("  DCLK 0D=%08lX 1D=%08lX 0DS=%08lX DPLL A=%08lX B=%08lX\n",
	  I810_RD32(bi810_i810, 0x6000), I810_RD32(bi810_i810, 0x6004), 
	  I810_RD32(bi810_i810, 0x6010),
	  I810_RD32(bi810_i810, 0x6014), I810_RD32(bi810_i810, 0x6018));
  printf ("  ADPA=%08lX  DVO A=%08lX B=%08lX C=%08lX  LVDS=%08lX\n",
	  I810_RD32(bi810_i810, 0x61100), I810_RD32(bi810_i810, 0x61120),
	  I810_RD32(bi810_i810, 0x61140), I810_RD32(bi810_i810, 0x61160),
	  I810_RD32(bi810_i810, 0x61180));
  printf ("  DIM =%08lX  DIM A=%08lX B=%08lX C=%08lX       %08lX\n",
	  I810_RD32(bi810_i810, 0x61104), I810_RD32(bi810_i810, 0x61124),
	  I810_RD32(bi810_i810, 0x61144), I810_RD32(bi810_i810, 0x61164),
	  I810_RD32(bi810_i810, 0x61184));
  printf ("  FPA %08lX %08lX %08lX %08lX\n",
	  I810_RD32(bi810_i810, 0x6040), I810_RD32(bi810_i810, 0x6044), 
	  I810_RD32(bi810_i810, 0x6048), I810_RD32(bi810_i810, 0x604c));
  printf ("  020cc=%08lX 020d8=%08lX 020dc=%08lX\n",
	  I810_RD32(bi810_i810, 0x20cc), I810_RD32(bi810_i810, 0x20d8), 
	  I810_RD32(bi810_i810, 0x20dc));
  printf ("  71280=%08lX 71400=%08lX\n",
	  I810_RD32(bi810_i810, 0x71280), I810_RD32(bi810_i810, 0x71400));
  for (i = 0; i <= 1; i++)
  {
    printf ("  Pipe %c\n", i + 'A');
    printf ("    HTOTAL=%08lX HBLANK=%08lX HSYNC=%08lX\n",
	    I810_P_RD32(bi810_i810, i, 0x60000), 
	    I810_P_RD32(bi810_i810, i, 0x60004), 
	    I810_P_RD32(bi810_i810, i, 0x60008));
    printf ("    VTOTAL=%08lX VBLANK=%08lX VSYNC=%08lX\n",
	    I810_P_RD32(bi810_i810, i, 0x6000c), 
	    I810_P_RD32(bi810_i810, i, 0x60010), 
	    I810_P_RD32(bi810_i810, i, 0x60014));
    printf ("    SOURCE=%08lX CONFIG=%08lX 7z030=%08lX 7z080=%08lX\n",
	    I810_P_RD32(bi810_i810, i, 0x6001c), 
	    I810_P_RD32(bi810_i810, i, 0x70008), 
	    I810_P_RD32(bi810_i810, i, 0x70030),
	    I810_P_RD32(bi810_i810, i, 0x70080));
  }
  for (i = 0; i <= 1; i++)
  {
    printf ("  Display plane %c\n", i + 'A');
    printf ("    CTRL=%08lX BASE=%08lX STRIDE=%08lX\n",
	    I810_P_RD32(bi810_i810, i, 0x70180), 
	    I810_P_RD32(bi810_i810, i, 0x70184), 
	    I810_P_RD32(bi810_i810, i, 0x70188));
  }
  /* dump 71410-71428 BIOS scratch regs? */
}

void bi810_probeCard (void)
{
  /* FIXME */
  bi830_probeCard ();
}

I2CChainPtr bi810_probeBus (void)
{
  /* needs unlocked crt */
  I810UpdateTvState (&bi810_i810);
  TVCheckChain (bi810_i810.TvChain);
  bi810_i810.TvChain = 
    TVProbeCreateAll (bi810_i810.TvBusses, bi810_i810.TvMaxBus, NULL);
  return bi810_i810.TvChain;
}
#endif

BackCardRec bi810_func = {
  openCard:              bi810_openCard,
  closeCard:             bi810_closeCard,
#ifdef DEBUG_PROBE
  probeCard:             bi810_probeCard,
  probeBus:              bi810_probeBus,
#endif
  setHeads:              bi810_setHeads,
  getHeads:              bi810_getHeads,
  getHeadDev:            bi810_getHeadDev,
  probeChips:            bi810_probeChips,
  setChip:               bi810_setChip,
  setSettings:           bi810_setSettings,
  getSettings:           bi810_getSettings,
  setMode:               bi810_setMode,
  getMode:               bi810_getMode,
  setModeSettings:       bi810_setModeSettings,
  setTestImage:          bi810_setTestImage, 
  getStatus:             bi810_getStatus,    
  getConnection:         bi810_getConnection,
  listModes:		 bi810_listModes,
  findBySize:            bi810_findBySize, 
  findByOverscan:        bi810_findByOverscan,
  initSharedView:        bnull_initSharedView,
  getTwinView:           bnull_getTwinView,
  adjustViewport:        bnull_adjustViewport,
  serviceViewportCursor: bnull_serviceViewportCursor,
};
