// Copyright Joyent, Inc. and other Node contributors.
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the
// "Software"), to deal in the Software without restriction, including
// without limitation the rights to use, copy, modify, merge, publish,
// distribute, sublicense, and/or sell copies of the Software, and to permit
// persons to whom the Software is furnished to do so, subject to the
// following conditions:
//
// The above copyright notice and this permission notice shall be included
// in all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS
// OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
// MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN
// NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM,
// DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR
// OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE
// USE OR OTHER DEALINGS IN THE SOFTWARE.

import fs from 'fs';
import path from 'path';

import { marked } from 'marked';

marked.setOptions({ headerIds: true });


import * as common from './common.mjs';
import * as typeParser from './type-parser.mjs';

const docPath = new URL('../../doc/', import.meta.url);

const gtocPath = new URL('./api/index.md', docPath);
const gtocMD = fs.readFileSync(gtocPath, 'utf8')
  .replace(/\(([^#?]+?)\.md\)/ig, (_, filename) => `(${filename}.html)`)
  .replace(/^<!--.*?-->/gms, '');
const gtocHTML = marked.parse(gtocMD, marked.defaults).replace(
  /<a href="(.*?)"/g,
  (all, href) => `<a class="nav-${href.replace('.html', '')
                                      .replace(/\W+/g, '-')}" href="${href}"`
);

const templatePath = new URL('./template.html', docPath);
const template = fs.readFileSync(templatePath, 'utf8');

export async function toHTML({ input, filename, nodeVersion, versions, apilinks }) {
  filename = path.basename(filename, '.md');

  const lexed = marked.lexer(input);

  const firstHeading = lexed.find(({ type }) => type === 'heading');
  const section = firstHeading ? firstHeading.text : 'Index';

  preprocessText(lexed);
  preprocessElements({filename})(lexed);

  // Generate the table of contents. This mutates the lexed contents in-place.
  const toc = buildToc({ filename, apilinks })(lexed);

  const id = filename.replace(/\W+/g, '-');

  let HTML = template.replace('__ID__', id)
                     .replace(/__FILENAME__/g, filename)
                     .replace('__SECTION__', section)
                     .replace(/__VERSION__/g, nodeVersion)
                     .replace('__GTOC__', gtocHTML.replace(
                       `class="nav-${id}`, `class="nav-${id} active`));

  if (toc !== '' || id == "all") HTML = HTML.replace('__TOC__', toc);

  const docCreated = input.match(
    /<!--\s*introduced_in\s*=\s*v([0-9]+)\.([0-9]+)\.[0-9]+\s*-->/);
  if (docCreated) {
    HTML = HTML.replace('__ALTDOCS__', await altDocs(filename, docCreated, versions));
  } else {
    console.error(`Failed to add alternative version links to ${filename}`);
    HTML = HTML.replace('__ALTDOCS__', '');
  }

  HTML = HTML.replace('__EDIT_ON_GITHUB__', editOnGitHub(filename));

  // Content insertion has to be the last thing we do with the lexed tokens,
  // because it's destructive.
  return HTML.replace('__CONTENT__', marked.parser(lexed, marked.defaults));
}

// Handle general body-text replacements.
// For example, link man page references to the actual page.
export function preprocessText(lexed) {
  lexed.forEach((token) => {
    if (token.type === 'table') {
      if (token.header) {
        for (const tok of token.header) {
          tok.text = replaceInText(tok.text);
        }
      }

      if (token.cells) {
        token.cells.forEach((row, i) => {
          for (const tok of token.cells[i]) {
            tok.text = replaceInText(tok.text);
          }
        });
      }
    } else if (token.text && token.type !== 'code') {
      token.text = replaceInText(token.text);
    }
  });
}

// Replace placeholders in text tokens.
function replaceInText(text = '') {
  if (text === '') return text;
  return linkJsTypeDocs(linkManPages(text));
}

// Syscalls which appear in the docs, but which only exist in BSD / macOS.
const BSD_ONLY_SYSCALLS = new Set(['lchmod']);
const MAN_PAGE = /(^|\s)([a-z.]+)\((\d)([a-z]?)\)/gm;

// Handle references to man pages, eg "open(2)" or "lchmod(2)".
// Returns modified text, with such refs replaced with HTML links, for example
// '<a href="http://man7.org/linux/man-pages/man2/open.2.html">open(2)</a>'.
function linkManPages(text) {
  return text.replace(
    MAN_PAGE, (match, beginning, name, number, optionalCharacter) => {
      // Name consists of lowercase letters,
      // number is a single digit with an optional lowercase letter.
      const displayAs = `<code>${name}(${number}${optionalCharacter})</code>`;

      if (BSD_ONLY_SYSCALLS.has(name)) {
        return `${beginning}<a href="https://www.freebsd.org/cgi/man.cgi?query=${name}&sektion=${number}">${displayAs}</a>`;
      }

      return `${beginning}<a href="http://man7.org/linux/man-pages/man${number}/${name}.${number}${optionalCharacter}.html">${displayAs}</a>`;
    });
}

const TYPE_SIGNATURE = /[^_]\{[^}]+\}(?!_)/g;
function linkJsTypeDocs(text) {
  if (text.startsWith('<pre>')) return text;
  const parts = text.split('`');

  // Handle types, for example the source Markdown might say
  // "This argument should be a {number} or {string}".
  try {
    for (let i = 0; i < parts.length; i += 2) {
      const typeMatches = parts[i].match(TYPE_SIGNATURE);
      if (typeMatches) {
        typeMatches.forEach((typeMatch) => {
          parts[i] = parts[i].replace(typeMatch, typeParser.toLink(typeMatch));
        });
      }
    }
  } catch(ex) {
    console.warn(ex, "in", text);
  }
  return parts.join('`');
}

// Preprocess stability blockquotes and YAML blocks.
export function preprocessElements({ filename }) {
  return (lexed) => {
  const STABILITY_RE = /(.*:)\s*(\d)([\s\S]*)/;
  let state = null;
  let headingIndex = -1;
  let heading = null;

  lexed.forEach((token, index) => {
    if (token.type === 'heading') {
      headingIndex = index;
      heading = token;
    }
    if (token.type === 'html' && common.isYAMLBlock(token.text)) {
      token.text = parseYAML(token.text);
    }
    if (token.type === 'blockquote_start') {
      state = 'MAYBE_STABILITY_BQ';
      lexed[index] = { type: 'space' };
    }
    if (token.type === 'blockquote_end' && state === 'MAYBE_STABILITY_BQ') {
      state = null;
      lexed[index] = { type: 'space' };
    }
    if (token.type === 'paragraph' && state === 'MAYBE_STABILITY_BQ') {
      if (token.text.includes('Stability:')) {
        const [, prefix, number, explication] = token.text.match(STABILITY_RE);
        const isStabilityIndex =
          index - 2 === headingIndex || // General.
          index - 3 === headingIndex;   // With api_metadata block.

        if (heading && isStabilityIndex) {
          heading.stability = number;
          headingIndex = -1;
          heading = null;
        }

        // Do not link to the section we are already in.
        const noLinking = filename === 'documentation' &&
          heading !== null && heading.text === 'Stability index';
        token.text = `<div class="api_stability api_stability_${number}">` +
          (noLinking ? '' :
            '<a href="documentation.html#documentation_stability_index">') +
          `${prefix} ${number}${noLinking ? '' : '</a>'}${explication}</div>`
          .replace(/\n/g, ' ');

        lexed[index] = { type: 'html', text: token.text };
      } else if (state === 'MAYBE_STABILITY_BQ') {
        state = null;
        lexed[index - 1] = { type: 'blockquote_start' };
      }
    }
  });
}
}

function parseYAML(text) {
  const meta = common.extractAndParseYAML(text);
  let result = '<div class="api_metadata">\n';

  const added = { description: '' };
  const deprecated = { description: '' };
  const removed = { description: '' };

  if (meta.added) {
    added.version = meta.added.join(', ');
    added.description = `<span>Added in: ${added.version}</span>`;
  }

  if (meta.deprecated) {
    deprecated.version = meta.deprecated.join(', ');
    deprecated.description =
        `<span>Deprecated since: ${deprecated.version}</span>`;
  }

  if (meta.removed) {
    removed.version = meta.removed.join(', ');
    removed.description = `<span>Removed in: ${removed.version}</span>`;
  }

  if (meta.changes.length > 0) {
    if (added.description) meta.changes.push(added);
    if (deprecated.description) meta.changes.push(deprecated);
    if (removed.description) meta.changes.push(removed);

    meta.changes.sort((a, b) => versionSort(a.version, b.version));

    result += '<details class="changelog"><summary>History</summary>\n' +
            '<table>\n<tr><th>Version</th><th>Changes</th></tr>\n';

    meta.changes.forEach((change) => {
      const description = marked.parse(change.description);
      const version = common.arrify(change.version).join(', ');

      result += `<tr><td>${version}</td>\n` +
                  `<td>${description}</td></tr>\n`;
    });

    result += '</table>\n</details>\n';
  } else {
    result += `${added.description}${deprecated.description}${removed.description}\n`;
  }

  if (meta.napiVersion) {
    result += `<span>N-API version: ${meta.napiVersion.join(', ')}</span>\n`;
  }

  result += '</div>';
  return result;
}

function minVersion(a) {
  return common.arrify(a).reduce((min, e) => {
    return !min || versionSort(min, e) < 0 ? e : min;
  });
}

const numberRe = /^\d*/;
function versionSort(a, b) {
  a = minVersion(a).trim();
  b = minVersion(b).trim();
  let i = 0; // Common prefix length.
  while (i < a.length && i < b.length && a[i] === b[i]) i++;
  a = a.substr(i);
  b = b.substr(i);
  return +b.match(numberRe)[0] - +a.match(numberRe)[0];
}

const DEPRECATION_HEADING_PATTERN = /^DEP\d+:/;
export function buildToc({ filename, apilinks }) {
  return (lexed) => {
    const idCounters = Object.create(null);
    const legacyIdCounters = Object.create(null);
    let toc = '';
    let depth = 0;

    lexed.forEach((node) => {
      // Keep track of the current filename along comment wrappers of inclusions.

      if (node.type !== 'heading') return;

      if (node.depth - depth > 1) {
        throw new Error(`Inappropriate heading level:\n${JSON.stringify(node)}`);
      }

      depth = node.depth;
      const realFilename = path.basename(filename, '.md');
      const headingText = node.text.trim();
      const id = getId(headingText, idCounters);
      // Use previous ID generator to create alias
      const legacyId = getLegacyId(`${realFilename}_${headingText}`, legacyIdCounters);

      const isDeprecationHeading =
        DEPRECATION_HEADING_PATTERN.test(headingText);
      if (isDeprecationHeading) {
        // this and optionsId on heading should work
        node.raw = getId(headingText.substring(0, headingText.indexOf(':')), idCounters);
      }

      const hasStability = node.stability !== undefined;
      toc += ' '.repeat((depth - 1) * 2) +
        (hasStability ? `* <span class="stability_${node.stability}">` : '* ') +
          `<a href="#${isDeprecationHeading ? node.raw : id}">${headingText}</a>${hasStability ? '</span>' : ''}\n`;

        let anchor =
          `<span><a class="mark" href="#${id}" id="${id}">#</a></span>`;

        // Add alias anchor to preserve old links
        anchor += `<a aria-hidden="true" class="legacy" id="${legacyId}"></a>`;

      if (realFilename === 'errors' && headingText.startsWith('ERR_')) {
          anchor +=
            `<span><a class="mark" href="#${headingText}" id="${headingText}">#</a></span>`;
      }
      const api = headingText.replace(/^.*:\s+/, '').replace(/\(.*/, '');
      if (apilinks[api]) {
        anchor = `<a class="srclink" href=${apilinks[api]}>[src]</a>${anchor}`;
      }
      node.tokens.push({ type: 'text', text: anchor });
    });

    return marked.parse(toc, marked.defaults);
  };
}

// ID generator that mirrors Github's heading anchor parser
const punctuation = /[^\w\- ]/g;
function getId(text, idCounters) {
  text = text.toLowerCase()
             .replace(punctuation, '')
             .replace(/ /g, '-');
  if (idCounters[text] !== undefined) {
    return `${text}_${++idCounters[text]}`;
  }
  idCounters[text] = 0;
  return text;
}

// This ID generator is purely to generate aliases
// so we can preserve old doc links
const notAlphaNumerics = /[^a-z0-9]+/g;
const edgeUnderscores = /^_+|_+$/g;
const notAlphaStart = /^[^a-z]/;
function getLegacyId(text, idCounters) {
  text = text.toLowerCase()
             .replace(notAlphaNumerics, '_')
             .replace(edgeUnderscores, '')
             .replace(notAlphaStart, '_$&');
  if (idCounters[text] !== undefined) {
    return `${text}_${++idCounters[text]}`;
  }
  idCounters[text] = 0;
  return text;
}

function altDocs(filename, docCreated, versions) {
  const [, docCreatedMajor, docCreatedMinor] = docCreated.map(Number);
  const host = 'https://nodejs.org';

  const getHref = (versionNum) =>
    `${host}/docs/latest-v${versionNum}/api/${filename}.html`;

  const wrapInListItem = (version) =>
    `<li><a href="${getHref(version.num)}">${version.num}${version.lts ? ' <b>LTS</b>' : ''}</a></li>`;

  function isDocInVersion(version) {
    const [versionMajor, versionMinor] = version.num.split('.').map(Number);
    if (docCreatedMajor > versionMajor) return false;
    if (docCreatedMajor < versionMajor) return true;
    if (Number.isNaN(versionMinor)) return true;
    return docCreatedMinor <= versionMinor;
  }

  const list = versions.filter(isDocInVersion).map(wrapInListItem).join('\n');

  return list ? `
    <li class="version-picker">
      <a href="#">View another version <span>&#x25bc;</span></a>
      <ol class="version-picker">${list}</ol>
    </li>
  ` : '';
}

function editOnGitHub(filename) {
  return `<li class="edit_on_github"><a href="https://github.com/nodejs/node/edit/master/doc/api/${filename}.md">Edit on GitHub</a></li>`;
}
