/******************************************************************************
 *                                                                            *
 * File:   trans   .c         Version  1.23            Date: 1997-07-27       *
 *                                                                            *
 * Copyright (c) 1993-1997 by kosta@kostis.net (Kosta Kostis)                 *
 *                                                                            *
 ******************************************************************************
 *                                                                            *
 * Function:                                                                  *
 *        - This program converts one 8-bit character code to another         *
 *                                                                            *
 ******************************************************************************
 *                                                                            *
 * Note:                                                                      *
 *        - This is just the generic program body. Definitions like the       *
 *          actual conversion tables and so on must be supplied.              *
 *                                                                            *
 ******************************************************************************
 *                                                                            *
 * History:                                                                   *
 *     1997-07-27: KK V1.23                                                   *
 *        - minor changes to reflect directory structure changes              *
 *     1997-05-01: KK V1.22                                                   *
 *        - hints from ache@nagual.ru about FreeBSD                           *
 *     1996-06-01: KK V1.21                                                   *
 *        - new release of package, change of email address                   *
 *     1995-07-01: KK V1.20                                                   *
 *        - BadChar is now ushort instead of int                              *
 *     1994-05-05: KK V1.12                                                   *
 *        - some compilers still don't understand "//" comments               *
 *     1994-03-31: KK V1.11                                                   *
 *        - changes to support ISO/IEC 10646-1:1993                           *
 *     1993-11-14: KK V1.10                                                   *
 *        - trying to deal with non-ANSI function utime ()                    *
 *     1993-05-25: KK V1.01                                                   *
 *        - dropped support for K&R C (now only ANSI-C)                       *
 *     1993-05-10: KK V1.00                                                   *
 *        - initial coding                                                    *
 *                                                                            *
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "trans.h"

#ifdef	USE_UTIMBUF
	#include "utimbuf.h"
#endif

/*****************************************************************************/

#include "../src/datatype.h"

static	ushort	BadChar = BAD_CHAR ;    /* default BAD character     */

/******************************************************************************

	Function:
		display copyright, author and program information

	Parameters:
		none

	Returns:
		nothing

 *****************************************************************************/

static  void    Hello (void)
{
	fprintf (stderr, MSG_COPYRIGHT) ;
	fprintf (stderr, MSG_VERSION, PRGNAM, VERSION) ;
	fprintf (stderr, MSG_FROM_TO,
		SRC_NAME, SRC_COMMENT1, SRC_COMMENT2, SRC_COMMENT3,
		DST_NAME, DST_COMMENT1, DST_COMMENT2, DST_COMMENT3) ;
}

/******************************************************************************

	Function:
		copy input to output doing conversion

	Parameters:
		FILE    *Src            source file
		FILE    *Dst            destination file

	Returns:
		ERR_NONE                if all goes well
		ERR_BAD                 if characters have been untranslatable

 *****************************************************************************/

static  int     Trans
(
	FILE    *Src,
	FILE    *Dst
)
{
	int     ch ;			/* character read               */
	int	newch ;			/* translated character         */
	int     codes ;			/* bad code flag                */

	codes = ERR_NONE ;		/* assume no bad code           */

	while ((ch = fgetc (Src)) != EOF)	/* read character       */
	{
		newch = (int) TRANS_C (ch) ;
		if (newch != (int) BAD)
		{
			fputc (newch, Dst) ;	/* output character     */
		}
		else
		{
			codes = ERR_BAD ;
			fputc (BadChar, Dst) ;
		}
	}

	return (codes) ;
}

/*****************************************************************************/

static	char	*InpName ;	/* input file name                           */
static	char	*OutName ;	/* output file name                          */

/******************************************************************************

	ScanFlags () return codes

 *****************************************************************************/

#define	SCN_OK		0	/* parameters are all valid                  */
#define	SCN_UNKNOWN	1	/* unknown parameter used                    */
#define	SCN_BADARGS	2	/* illegal number of params                  */

/******************************************************************************

	Function:
		parse parameters and set flags and variables accordingly

	Parameters:
		int     argc    number of parameters
		char    *argv[] argument strings

	Returns:
		SCN_OK          if all parameters are valid
		SCN_UNKNOWN     if any of the parameters is unknown
		SCN_BADARGS     if the number of parameters is wrong

 *****************************************************************************/

static  int     ScanFlags
(
	int	argc,
	char	*argv	[]
)
{
	int	i ;		/* index through the parameters              */
	int	parms = 0 ;	/* yet no parameter scanned                  */

	/**********************************************************************

		set defaults / initialize

	 *********************************************************************/

	InpName = (char *) NULL ;
	OutName = (char *) NULL ;

	for (i = 1 ; i < argc ; ++i)    /* scan command line                 */
	{
		/**************************************************************

			is it a flag/switch? (*NIX style)

		 *************************************************************/

		if (*argv [i] == '-')
		{
			/******************************************************

				check out which flag it is

			 *****************************************************/

			switch (argv [i][1])
			{
				case	'B':	/* change BAD character      */
				case	'b':

					BadChar = (ushort) argv [i][2] ;
					break ;

				default:        /* Uh! This flag's unknown!  */

					return (SCN_UNKNOWN) ;
			}
		}
		else
		{
			/******************************************************

				if it's no flag it must be a filename

			 *****************************************************/

			++parms ;       /* check for multiple filenames      */
			switch (parms)
			{
				case	1:

					InpName = argv [i] ;
					break ;

				case	2:

					OutName = argv [i] ;
					break ;

				default :

					return (SCN_BADARGS) ;
			}
		}
	}

	/**********************************************************************

		everything's been fine - let's quit!

	 *********************************************************************/

	return (SCN_OK) ;
}

/*****************************************************************************/

int     main
(
	int     argc,
	char    *argv   []
)
{
	FILE    *SRCfile = stdin ;      /* Source default is stdin           */
	FILE    *DSTfile = stdout ;     /* Dest. default is stdout           */
	char    *TmpName = (char *) NULL ;
	char    *Help ;
	int     codes ;                 /* bad code flag                     */
	int     syntax ;                /* return code for ScanFlags ()      */

#ifdef	USE_UTIMBUF
	struct  stat    f_stat ;
	struct  utimbuf f_date ;
#endif

	/**********************************************************************

		parse parameters/arguments

	 *********************************************************************/

	syntax = ScanFlags (argc, argv) ;
	if (syntax != SCN_OK)
	{
		/**************************************************************

			display syntax error message and quit

		 *************************************************************/

		Hello () ;
		fprintf (stderr, ERR_MSG_SYNTAX, PRGNAM) ;

		return (ERR_SYNTAX) ;
	}

	/**********************************************************************

		check for output file - create if necessary

	 *********************************************************************/

	if (OutName != (char *) NULL)
	{
		/**************************************************************

			file type must be binary

		 *************************************************************/

		DSTfile = fopen (OutName, "wb") ;
		if (DSTfile == (FILE *) NULL)
		{
			Hello () ;
			fprintf (stderr, ERR_MSG_DST, OutName) ;

			return (ERR_DST) ;
		}
	}

	/**********************************************************************

		check for input file - open if necessary

	 *********************************************************************/

	if (InpName != (char *) NULL)
	{       
		SRCfile = fopen (InpName, "rb") ;
		if (SRCfile == (FILE *) NULL)
		{
			Hello () ;
			fprintf (stderr, ERR_MSG_SRC, InpName) ;
			if (DSTfile != stdout)
				fclose (DSTfile) ;

			return (ERR_SRC) ;
		}

		/**************************************************************

			if no output file name is supplied, we have to create
			a temporary file which gets renamed afterwards

			preserve original file date - if possible

		 *************************************************************/

		if (OutName == (char *) NULL)
		{
#ifdef	USE_UTIMBUF
			stat (InpName, &f_stat) ;
			f_date.actime  = f_stat.st_atime ;
			f_date.modtime = f_stat.st_mtime ;
#endif

			TmpName = tmpnam ((char *) NULL) ;
			Help = strrchr (TmpName, '/') ;
			if (Help != (char *) NULL)
				TmpName = ++Help ;
			DSTfile = fopen (TmpName, "wb") ;
			if (DSTfile == (FILE *) NULL)
			{
				fclose (SRCfile) ;
				return (ERR_DST) ;
			}
		}
	}

	/**********************************************************************

		now do the actual conversion

	 *********************************************************************/

	codes = Trans (SRCfile, DSTfile) ;

	/**********************************************************************

		close files if opened

	 *********************************************************************/

	if (DSTfile != stdout)
		fclose (DSTfile) ;
	if (SRCfile != stdin)
		fclose (SRCfile) ;

	/**********************************************************************

		if there is a temporary file, rename it to the original name

	 *********************************************************************/

	if (TmpName != (char *) NULL)
	{
		remove (InpName) ;
		rename (TmpName, InpName) ;
#ifdef	USE_UTIMBUF
		utime (InpName, &f_date) ;
#endif
	}

	return (codes) ;
}
