package Color::Library::Dictionary::NBS_ISCC;

use strict;
use warnings;

use base qw/Color::Library::Dictionary/;

__PACKAGE__->_register_dictionary;

package Color::Library::Dictionary::NBS_ISCC;

=pod

=head1 NAME

Color::Library::Dictionary::NBS_ISCC - (NBS/ISCC Centroids) Centroids of the NBS/ISCC catalog

=head1 DESCRIPTION

    Kenneth L. Kelly and Deanne B. Judd.
    "Color: Universal Language and Dictionary of Names",
    National Bureau of Standards,
    Spec. Publ. 440, Dec. 1976, 189 pages.


    Color Name Dictionary: NBS/ISCC Centroids

L<http://swissnet.ai.mit.edu/~jaffer/Color/Dictionaries#NBS-ISCC%20Centroids>

=head1 COLORS

	Black                         black                      #222222

	Blackish Blue                 blackishblue               #202830

	Blackish Green                blackishgreen              #1a2421

	Blackish Purple               blackishpurple             #291e29

	Blackish Red                  blackishred                #2e1d21

	Bluish Black                  bluishblack                #202428

	Bluish Gray                   bluishgray                 #81878b

	Bluish White                  bluishwhite                #e9e9ed

	Brilliant Blue                brilliantblue              #4997d0

	Brilliant Bluish Green        brilliantbluishgreen       #00a693

	Brilliant Green               brilliantgreen             #3eb489

	Brilliant Greenish Blue       brilliantgreenishblue      #239eba

	Brilliant Greenish Yellow     brilliantgreenishyellow    #e9e450

	Brilliant Orange              brilliantorange            #fd943f

	Brilliant Orange Yellow       brilliantorangeyellow      #ffc14f

	Brilliant Purple              brilliantpurple            #d399e6

	Brilliant Purplish Blue       brilliantpurplishblue      #6c79b8

	Brilliant Purplish Pink       brilliantpurplishpink      #ffc8d6

	Brilliant Violet              brilliantviolet            #7e73b8

	Brilliant Yellow              brilliantyellow            #fada5e

	Brilliant Yellow Green        brilliantyellowgreen       #bdda57

	Brilliant Yellowish Green     brilliantyellowishgreen    #83d37d

	Brownish Black                brownishblack              #28201c

	Brownish Gray                 brownishgray               #5b504f

	Brownish Orange               brownishorange             #ae6938

	Brownish Pink                 brownishpink               #c2ac99

	Dark Blue                     darkblue                   #00304e

	Dark Bluish Gray              darkbluishgray             #51585e

	Dark Bluish Green             darkbluishgreen            #004b49

	Dark Brown                    darkbrown                  #422518

	Dark Gray                     darkgray                   #555555

	Dark Grayish Blue             darkgrayishblue            #36454f

	Dark Grayish Brown            darkgrayishbrown           #3e322c

	Dark Grayish Green            darkgrayishgreen           #3a4b47

	Dark Grayish Olive            darkgrayisholive           #363527

	Dark Grayish Olive Green      darkgrayisholivegreen      #31362b

	Dark Grayish Purple           darkgrayishpurple          #50404d

	Dark Grayish Red              darkgrayishred             #543d3f

	Dark Grayish Reddish Brown    darkgrayishreddishbrown    #43302e

	Dark Grayish Yellow           darkgrayishyellow          #a18f60

	Dark Grayish Yellowish Brown  darkgrayishyellowishbrown  #483c32

	Dark Green                    darkgreen                  #1b4d3e

	Dark Greenish Blue            darkgreenishblue           #004958

	Dark Greenish Gray            darkgreenishgray           #4e5755

	Dark Greenish Yellow          darkgreenishyellow         #98943e

	Dark Olive                    darkolive                  #403d21

	Dark Olive Brown              darkolivebrown             #3b3121

	Dark Olive Green              darkolivegreen             #2b3d26

	Dark Orange Yellow            darkorangeyellow           #be8a3d

	Dark Pink                     darkpink                   #c08081

	Dark Purple                   darkpurple                 #563c5c

	Dark Purplish Blue            darkpurplishblue           #252440

	Dark Purplish Gray            darkpurplishgray           #5d555b

	Dark Purplish Pink            darkpurplishpink           #c17e91

	Dark Purplish Red             darkpurplishred            #673147

	Dark Red                      darkred                    #722f37

	Dark Reddish Brown            darkreddishbrown           #3e1d1e

	Dark Reddish Gray             darkreddishgray            #5c504f

	Dark Reddish Orange           darkreddishorange          #9e4732

	Dark Reddish Purple           darkreddishpurple          #5d3954

	Dark Violet                   darkviolet                 #2f2140

	Dark Yellow                   darkyellow                 #ab9144

	Dark Yellowish Brown          darkyellowishbrown         #4b3621

	Dark Yellowish Green          darkyellowishgreen         #355e3b

	Dark Yellowish Pink           darkyellowishpink          #c48379

	Deep Blue                     deepblue                   #00416a

	Deep Bluish Green             deepbluishgreen            #00443f

	Deep Brown                    deepbrown                  #593319

	Deep Green                    deepgreen                  #00543d

	Deep Greenish Blue            deepgreenishblue           #2e8495

	Deep Greenish Yellow          deepgreenishyellow         #9b9400

	Deep Olive Green              deepolivegreen             #232f00

	Deep Orange                   deeporange                 #be6516

	Deep Orange Yellow            deeporangeyellow           #c98500

	Deep Pink                     deeppink                   #e4717a

	Deep Purple                   deeppurple                 #602f6b

	Deep Purplish Blue            deeppurplishblue           #272458

	Deep Purplish Pink            deeppurplishpink           #de6fa1

	Deep Purplish Red             deeppurplishred            #78184a

	Deep Red                      deepred                    #841b2d

	Deep Reddish Brown            deepreddishbrown           #56070c

	Deep Reddish Orange           deepreddishorange          #aa381e

	Deep Reddish Purple           deepreddishpurple          #702963

	Deep Violet                   deepviolet                 #32174d

	Deep Yellow                   deepyellow                 #af8d13

	Deep Yellow Green             deepyellowgreen            #467129

	Deep Yellowish Brown          deepyellowishbrown         #654522

	Deep Yellowish Green          deepyellowishgreen         #00622d

	Deep Yellowish Pink           deepyellowishpink          #e66721

	Grayish Blue                  grayishblue                #536878

	Grayish Brown                 grayishbrown               #635147

	Grayish Green                 grayishgreen               #5e716a

	Grayish Greenish Yellow       grayishgreenishyellow      #b9b57d

	Grayish Olive                 grayisholive               #5b5842

	Grayish Olive Green           grayisholivegreen          #515744

	Grayish Pink                  grayishpink                #c4aead

	Grayish Purple                grayishpurple              #796878

	Grayish Purplish Blue         grayishpurplishblue        #4c516d

	Grayish Purplish Pink         grayishpurplishpink        #c3a6b1

	Grayish Purplish Red          grayishpurplishred         #915f6d

	Grayish Red                   grayishred                 #905d5d

	Grayish Reddish Brown         grayishreddishbrown        #674c47

	Grayish Reddish Orange        grayishreddishorange       #b4745e

	Grayish Reddish Purple        grayishreddishpurple       #836479

	Grayish Violet                grayishviolet              #554c69

	Grayish Yellow                grayishyellow              #c2b280

	Grayish Yellow Green          grayishyellowgreen         #8f9779

	Grayish Yellowish Brown       grayishyellowishbrown      #7e6d5a

	Grayish Yellowish Pink        grayishyellowishpink       #c7ada3

	Greenish Black                greenishblack              #1e2321

	Greenish Gray                 greenishgray               #7d8984

	Greenish White                greenishwhite              #dfede8

	Light Blue                    lightblue                  #70a3cc

	Light Bluish Gray             lightbluishgray            #b4bcc0

	Light Bluish Green            lightbluishgreen           #66ada4

	Light Brown                   lightbrown                 #a67b5b

	Light Brownish Gray           lightbrownishgray          #8e8279

	Light Gray                    lightgray                  #b9b8b5

	Light Grayish Brown           lightgrayishbrown          #958070

	Light Grayish Olive           lightgrayisholive          #8c8767

	Light Grayish Purplish Red    lightgrayishpurplishred    #af868e

	Light Grayish Red             lightgrayishred            #ad8884

	Light Grayish Reddish Brown   lightgrayishreddishbrown   #977f73

	Light Grayish Yellowish Brown lightgrayishyellowishbrown #ae9b82

	Light Green                   lightgreen                 #6aab8e

	Light Greenish Blue           lightgreenishblue          #66aabc

	Light Greenish Gray           lightgreenishgray          #b2beb5

	Light Greenish Yellow         lightgreenishyellow        #eae679

	Light Olive                   lightolive                 #867e36

	Light Olive Brown             lightolivebrown            #967117

	Light Olive Gray              lightolivegray             #8a8776

	Light Orange                  lightorange                #fab57f

	Light Orange Yellow           lightorangeyellow          #fbc97f

	Light Pink                    lightpink                  #f9ccca

	Light Purple                  lightpurple                #b695c0

	Light Purplish Blue           lightpurplishblue          #8791bf

	Light Purplish Gray           lightpurplishgray          #bfb9bd

	Light Purplish Pink           lightpurplishpink          #efbbcc

	Light Reddish Brown           lightreddishbrown          #a87c6d

	Light Reddish Purple          lightreddishpurple         #b784a7

	Light Violet                  lightviolet                #8c82b6

	Light Yellow                  lightyellow                #f8de7e

	Light Yellow Green            lightyellowgreen           #c9dc89

	Light Yellowish Brown         lightyellowishbrown        #c19a6b

	Light Yellowish Green         lightyellowishgreen        #93c592

	Light Yellowish Pink          lightyellowishpink         #f4c2c2

	Medium Gray                   mediumgray                 #848482

	Moderate Blue                 moderateblue               #436b95

	Moderate Bluish Green         moderatebluishgreen        #317873

	Moderate Brown                moderatebrown              #6f4e37

	Moderate Green                moderategreen              #3b7861

	Moderate Greenish Blue        moderategreenishblue       #367588

	Moderate Greenish Yellow      moderategreenishyellow     #b9b459

	Moderate Olive                moderateolive              #665d1e

	Moderate Olive Brown          moderateolivebrown         #6c541e

	Moderate Olive Green          moderateolivegreen         #4a5d23

	Moderate Orange               moderateorange             #d99058

	Moderate Orange Yellow        moderateorangeyellow       #e3a857

	Moderate Pink                 moderatepink               #dea5a4

	Moderate Purple               moderatepurple             #86608e

	Moderate Purplish Blue        moderatepurplishblue       #4e5180

	Moderate Purplish Pink        moderatepurplishpink       #d597ae

	Moderate Purplish Red         moderatepurplishred        #a8516e

	Moderate Red                  moderatered                #ab4e52

	Moderate Reddish Brown        moderatereddishbrown       #79443b

	Moderate Reddish Orange       moderatereddishorange      #cb6d51

	Moderate Reddish Purple       moderatereddishpurple      #915c83

	Moderate Violet               moderateviolet             #604e81

	Moderate Yellow               moderateyellow             #c9ae5d

	Moderate Yellow Green         moderateyellowgreen        #8a9a5b

	Moderate Yellowish Brown      moderateyellowishbrown     #826644

	Moderate Yellowish Green      moderateyellowishgreen     #679267

	Moderate Yellowish Pink       moderateyellowishpink      #d9a6a9

	Olive Black                   oliveblack                 #25241d

	Olive Gray                    olivegray                  #57554c

	Pale Blue                     paleblue                   #91a3b0

	Pale Green                    palegreen                  #8da399

	Pale Greenish Yellow          palegreenishyellow         #ebe8a4

	Pale Orange Yellow            paleorangeyellow           #fad6a5

	Pale Pink                     palepink                   #ead8d7

	Pale Purple                   palepurple                 #aa98a9

	Pale Purplish Blue            palepurplishblue           #8c92ac

	Pale Purplish Pink            palepurplishpink           #e8ccd7

	Pale Reddish Purple           palereddishpurple          #aa8a9e

	Pale Violet                   paleviolet                 #9690ab

	Pale Yellow                   paleyellow                 #f3e5ab

	Pale Yellow Green             paleyellowgreen            #dadfb7

	Pale Yellowish Pink           paleyellowishpink          #ecd5c5

	Pinkish Gray                  pinkishgray                #c1b6b3

	Pinkish White                 pinkishwhite               #eae3e1

	Purplish Black                purplishblack              #242124

	Purplish Gray                 purplishgray               #8b8589

	Purplish White                purplishwhite              #e8e3e5

	Reddish Black                 reddishblack               #282022

	Reddish Gray                  reddishgray                #8f817f

	Strong Blue                   strongblue                 #0067a5

	Strong Bluish Green           strongbluishgreen          #007a74

	Strong Brown                  strongbrown                #80461b

	Strong Green                  stronggreen                #007959

	Strong Greenish Blue          stronggreenishblue         #007791

	Strong Greenish Yellow        stronggreenishyellow       #beb72e

	Strong Olive Green            strongolivegreen           #404f00

	Strong Orange                 strongorange               #ed872d

	Strong Orange Yellow          strongorangeyellow         #eaa221

	Strong Pink                   strongpink                 #ea9399

	Strong Purple                 strongpurple               #875692

	Strong Purplish Blue          strongpurplishblue         #545aa7

	Strong Purplish Pink          strongpurplishpink         #e68fac

	Strong Purplish Red           strongpurplishred          #b3446c

	Strong Red                    strongred                  #bc3f4a

	Strong Reddish Brown          strongreddishbrown         #882d17

	Strong Reddish Orange         strongreddishorange        #d9603b

	Strong Reddish Purple         strongreddishpurple        #9e4f88

	Strong Violet                 strongviolet               #604e97

	Strong Yellow                 strongyellow               #d4af37

	Strong Yellow Green           strongyellowgreen          #7e9f2e

	Strong Yellowish Brown        strongyellowishbrown       #996515

	Strong Yellowish Green        strongyellowishgreen       #44944a

	Strong Yellowish Pink         strongyellowishpink        #f99379

	Very Dark Bluish Green        verydarkbluishgreen        #002a29

	Very Dark Green               verydarkgreen              #1c352d

	Very Dark Greenish Blue       verydarkgreenishblue       #002e3b

	Very Dark Purple              verydarkpurple             #301934

	Very Dark Purplish Red        verydarkpurplishred        #38152c

	Very Dark Red                 verydarkred                #3f1728

	Very Dark Reddish Purple      verydarkreddishpurple      #341731

	Very Dark Yellowish Green     verydarkyellowishgreen     #173620

	Very Deep Purple              verydeeppurple             #401a4c

	Very Deep Purplish Red        verydeeppurplishred        #54133b

	Very Deep Red                 verydeepred                #5c0923

	Very Deep Reddish Purple      verydeepreddishpurple      #54194e

	Very Deep Yellowish Green     verydeepyellowishgreen     #003118

	Very Light Blue               verylightblue              #a1caf1

	Very Light Bluish Green       verylightbluishgreen       #96ded1

	Very Light Green              verylightgreen             #8ed1b2

	Very Light Greenish Blue      verylightgreenishblue      #9cd1dc

	Very Light Purple             verylightpurple            #d5badb

	Very Light Purplish Blue      verylightpurplishblue      #b3bce2

	Very Light Violet             verylightviolet            #dcd0ff

	Very Light Yellowish Green    verylightyellowishgreen    #b6e5af

	Very Pale Blue                verypaleblue               #bcd4e6

	Very Pale Green               verypalegreen              #c7e6d7

	Very Pale Purple              verypalepurple             #d6cadd

	Very Pale Purplish Blue       verypalepurplishblue       #c0c8e1

	Very Pale Violet              verypaleviolet             #c4c3dd

	Vivid Blue                    vividblue                  #00a1c2

	Vivid Bluish Green            vividbluishgreen           #008882

	Vivid Green                   vividgreen                 #008856

	Vivid Greenish Blue           vividgreenishblue          #0085a1

	Vivid Greenish Yellow         vividgreenishyellow        #dcd300

	Vivid Orange                  vividorange                #f38400

	Vivid Orange Yellow           vividorangeyellow          #f6a600

	Vivid Pink                    vividpink                  #ffb5ba

	Vivid Purple                  vividpurple                #9a4eae

	Vivid Purplish Blue           vividpurplishblue          #30267a

	Vivid Purplish Red            vividpurplishred           #ce4676

	Vivid Red                     vividred                   #be0032

	Vivid Reddish Orange          vividreddishorange         #e25822

	Vivid Reddish Purple          vividreddishpurple         #870074

	Vivid Violet                  vividviolet                #9065ca

	Vivid Yellow                  vividyellow                #f3c300

	Vivid Yellow Green            vividyellowgreen           #8db600

	Vivid Yellowish Green         vividyellowishgreen        #27a64c

	Vivid Yellowish Pink          vividyellowishpink         #ffb7a5

	White                         white                      #f2f3f4

	Yellowish Gray                yellowishgray              #bfb8a5

	Yellowish White               yellowishwhite             #f0ead6


=cut

sub _load_color_list() {
    return [
['nbs-iscc:black','black','Black',[34,34,34],'222222',2236962],
['nbs-iscc:blackishblue','blackishblue','Blackish Blue',[32,40,48],'202830',2107440],
['nbs-iscc:blackishgreen','blackishgreen','Blackish Green',[26,36,33],'1a2421',1713185],
['nbs-iscc:blackishpurple','blackishpurple','Blackish Purple',[41,30,41],'291e29',2694697],
['nbs-iscc:blackishred','blackishred','Blackish Red',[46,29,33],'2e1d21',3022113],
['nbs-iscc:bluishblack','bluishblack','Bluish Black',[32,36,40],'202428',2106408],
['nbs-iscc:bluishgray','bluishgray','Bluish Gray',[129,135,139],'81878b',8488843],
['nbs-iscc:bluishwhite','bluishwhite','Bluish White',[233,233,237],'e9e9ed',15329773],
['nbs-iscc:brilliantblue','brilliantblue','Brilliant Blue',[73,151,208],'4997d0',4822992],
['nbs-iscc:brilliantbluishgreen','brilliantbluishgreen','Brilliant Bluish Green',[0,166,147],'00a693',42643],
['nbs-iscc:brilliantgreen','brilliantgreen','Brilliant Green',[62,180,137],'3eb489',4109449],
['nbs-iscc:brilliantgreenishblue','brilliantgreenishblue','Brilliant Greenish Blue',[35,158,186],'239eba',2334394],
['nbs-iscc:brilliantgreenishyellow','brilliantgreenishyellow','Brilliant Greenish Yellow',[233,228,80],'e9e450',15328336],
['nbs-iscc:brilliantorange','brilliantorange','Brilliant Orange',[253,148,63],'fd943f',16618559],
['nbs-iscc:brilliantorangeyellow','brilliantorangeyellow','Brilliant Orange Yellow',[255,193,79],'ffc14f',16761167],
['nbs-iscc:brilliantpurple','brilliantpurple','Brilliant Purple',[211,153,230],'d399e6',13867494],
['nbs-iscc:brilliantpurplishblue','brilliantpurplishblue','Brilliant Purplish Blue',[108,121,184],'6c79b8',7109048],
['nbs-iscc:brilliantpurplishpink','brilliantpurplishpink','Brilliant Purplish Pink',[255,200,214],'ffc8d6',16763094],
['nbs-iscc:brilliantviolet','brilliantviolet','Brilliant Violet',[126,115,184],'7e73b8',8287160],
['nbs-iscc:brilliantyellow','brilliantyellow','Brilliant Yellow',[250,218,94],'fada5e',16439902],
['nbs-iscc:brilliantyellowgreen','brilliantyellowgreen','Brilliant Yellow Green',[189,218,87],'bdda57',12442199],
['nbs-iscc:brilliantyellowishgreen','brilliantyellowishgreen','Brilliant Yellowish Green',[131,211,125],'83d37d',8639357],
['nbs-iscc:brownishblack','brownishblack','Brownish Black',[40,32,28],'28201c',2629660],
['nbs-iscc:brownishgray','brownishgray','Brownish Gray',[91,80,79],'5b504f',5984335],
['nbs-iscc:brownishorange','brownishorange','Brownish Orange',[174,105,56],'ae6938',11430200],
['nbs-iscc:brownishpink','brownishpink','Brownish Pink',[194,172,153],'c2ac99',12758169],
['nbs-iscc:darkblue','darkblue','Dark Blue',[0,48,78],'00304e',12366],
['nbs-iscc:darkbluishgray','darkbluishgray','Dark Bluish Gray',[81,88,94],'51585e',5331038],
['nbs-iscc:darkbluishgreen','darkbluishgreen','Dark Bluish Green',[0,75,73],'004b49',19273],
['nbs-iscc:darkbrown','darkbrown','Dark Brown',[66,37,24],'422518',4334872],
['nbs-iscc:darkgray','darkgray','Dark Gray',[85,85,85],'555555',5592405],
['nbs-iscc:darkgrayishblue','darkgrayishblue','Dark Grayish Blue',[54,69,79],'36454f',3556687],
['nbs-iscc:darkgrayishbrown','darkgrayishbrown','Dark Grayish Brown',[62,50,44],'3e322c',4076076],
['nbs-iscc:darkgrayishgreen','darkgrayishgreen','Dark Grayish Green',[58,75,71],'3a4b47',3820359],
['nbs-iscc:darkgrayisholive','darkgrayisholive','Dark Grayish Olive',[54,53,39],'363527',3552551],
['nbs-iscc:darkgrayisholivegreen','darkgrayisholivegreen','Dark Grayish Olive Green',[49,54,43],'31362b',3225131],
['nbs-iscc:darkgrayishpurple','darkgrayishpurple','Dark Grayish Purple',[80,64,77],'50404d',5259341],
['nbs-iscc:darkgrayishred','darkgrayishred','Dark Grayish Red',[84,61,63],'543d3f',5520703],
['nbs-iscc:darkgrayishreddishbrown','darkgrayishreddishbrown','Dark Grayish Reddish Brown',[67,48,46],'43302e',4403246],
['nbs-iscc:darkgrayishyellow','darkgrayishyellow','Dark Grayish Yellow',[161,143,96],'a18f60',10588000],
['nbs-iscc:darkgrayishyellowishbrown','darkgrayishyellowishbrown','Dark Grayish Yellowish Brown',[72,60,50],'483c32',4734002],
['nbs-iscc:darkgreen','darkgreen','Dark Green',[27,77,62],'1b4d3e',1789246],
['nbs-iscc:darkgreenishblue','darkgreenishblue','Dark Greenish Blue',[0,73,88],'004958',18776],
['nbs-iscc:darkgreenishgray','darkgreenishgray','Dark Greenish Gray',[78,87,85],'4e5755',5134165],
['nbs-iscc:darkgreenishyellow','darkgreenishyellow','Dark Greenish Yellow',[152,148,62],'98943e',9999422],
['nbs-iscc:darkolive','darkolive','Dark Olive',[64,61,33],'403d21',4209953],
['nbs-iscc:darkolivebrown','darkolivebrown','Dark Olive Brown',[59,49,33],'3b3121',3879201],
['nbs-iscc:darkolivegreen','darkolivegreen','Dark Olive Green',[43,61,38],'2b3d26',2833702],
['nbs-iscc:darkorangeyellow','darkorangeyellow','Dark Orange Yellow',[190,138,61],'be8a3d',12487229],
['nbs-iscc:darkpink','darkpink','Dark Pink',[192,128,129],'c08081',12615809],
['nbs-iscc:darkpurple','darkpurple','Dark Purple',[86,60,92],'563c5c',5651548],
['nbs-iscc:darkpurplishblue','darkpurplishblue','Dark Purplish Blue',[37,36,64],'252440',2434112],
['nbs-iscc:darkpurplishgray','darkpurplishgray','Dark Purplish Gray',[93,85,91],'5d555b',6116699],
['nbs-iscc:darkpurplishpink','darkpurplishpink','Dark Purplish Pink',[193,126,145],'c17e91',12680849],
['nbs-iscc:darkpurplishred','darkpurplishred','Dark Purplish Red',[103,49,71],'673147',6762823],
['nbs-iscc:darkred','darkred','Dark Red',[114,47,55],'722f37',7483191],
['nbs-iscc:darkreddishbrown','darkreddishbrown','Dark Reddish Brown',[62,29,30],'3e1d1e',4070686],
['nbs-iscc:darkreddishgray','darkreddishgray','Dark Reddish Gray',[92,80,79],'5c504f',6049871],
['nbs-iscc:darkreddishorange','darkreddishorange','Dark Reddish Orange',[158,71,50],'9e4732',10372914],
['nbs-iscc:darkreddishpurple','darkreddishpurple','Dark Reddish Purple',[93,57,84],'5d3954',6109524],
['nbs-iscc:darkviolet','darkviolet','Dark Violet',[47,33,64],'2f2140',3088704],
['nbs-iscc:darkyellow','darkyellow','Dark Yellow',[171,145,68],'ab9144',11243844],
['nbs-iscc:darkyellowishbrown','darkyellowishbrown','Dark Yellowish Brown',[75,54,33],'4b3621',4929057],
['nbs-iscc:darkyellowishgreen','darkyellowishgreen','Dark Yellowish Green',[53,94,59],'355e3b',3497531],
['nbs-iscc:darkyellowishpink','darkyellowishpink','Dark Yellowish Pink',[196,131,121],'c48379',12878713],
['nbs-iscc:deepblue','deepblue','Deep Blue',[0,65,106],'00416a',16746],
['nbs-iscc:deepbluishgreen','deepbluishgreen','Deep Bluish Green',[0,68,63],'00443f',17471],
['nbs-iscc:deepbrown','deepbrown','Deep Brown',[89,51,25],'593319',5845785],
['nbs-iscc:deepgreen','deepgreen','Deep Green',[0,84,61],'00543d',21565],
['nbs-iscc:deepgreenishblue','deepgreenishblue','Deep Greenish Blue',[46,132,149],'2e8495',3048597],
['nbs-iscc:deepgreenishyellow','deepgreenishyellow','Deep Greenish Yellow',[155,148,0],'9b9400',10195968],
['nbs-iscc:deepolivegreen','deepolivegreen','Deep Olive Green',[35,47,0],'232f00',2305792],
['nbs-iscc:deeporange','deeporange','Deep Orange',[190,101,22],'be6516',12477718],
['nbs-iscc:deeporangeyellow','deeporangeyellow','Deep Orange Yellow',[201,133,0],'c98500',13206784],
['nbs-iscc:deeppink','deeppink','Deep Pink',[228,113,122],'e4717a',14971258],
['nbs-iscc:deeppurple','deeppurple','Deep Purple',[96,47,107],'602f6b',6303595],
['nbs-iscc:deeppurplishblue','deeppurplishblue','Deep Purplish Blue',[39,36,88],'272458',2565208],
['nbs-iscc:deeppurplishpink','deeppurplishpink','Deep Purplish Pink',[222,111,161],'de6fa1',14577569],
['nbs-iscc:deeppurplishred','deeppurplishred','Deep Purplish Red',[120,24,74],'78184a',7870538],
['nbs-iscc:deepred','deepred','Deep Red',[132,27,45],'841b2d',8657709],
['nbs-iscc:deepreddishbrown','deepreddishbrown','Deep Reddish Brown',[86,7,12],'56070c',5637900],
['nbs-iscc:deepreddishorange','deepreddishorange','Deep Reddish Orange',[170,56,30],'aa381e',11155486],
['nbs-iscc:deepreddishpurple','deepreddishpurple','Deep Reddish Purple',[112,41,99],'702963',7350627],
['nbs-iscc:deepviolet','deepviolet','Deep Violet',[50,23,77],'32174d',3282765],
['nbs-iscc:deepyellow','deepyellow','Deep Yellow',[175,141,19],'af8d13',11504915],
['nbs-iscc:deepyellowgreen','deepyellowgreen','Deep Yellow Green',[70,113,41],'467129',4616489],
['nbs-iscc:deepyellowishbrown','deepyellowishbrown','Deep Yellowish Brown',[101,69,34],'654522',6636834],
['nbs-iscc:deepyellowishgreen','deepyellowishgreen','Deep Yellowish Green',[0,98,45],'00622d',25133],
['nbs-iscc:deepyellowishpink','deepyellowishpink','Deep Yellowish Pink',[230,103,33],'e66721',15099681],
['nbs-iscc:grayishblue','grayishblue','Grayish Blue',[83,104,120],'536878',5466232],
['nbs-iscc:grayishbrown','grayishbrown','Grayish Brown',[99,81,71],'635147',6508871],
['nbs-iscc:grayishgreen','grayishgreen','Grayish Green',[94,113,106],'5e716a',6189418],
['nbs-iscc:grayishgreenishyellow','grayishgreenishyellow','Grayish Greenish Yellow',[185,181,125],'b9b57d',12170621],
['nbs-iscc:grayisholive','grayisholive','Grayish Olive',[91,88,66],'5b5842',5986370],
['nbs-iscc:grayisholivegreen','grayisholivegreen','Grayish Olive Green',[81,87,68],'515744',5330756],
['nbs-iscc:grayishpink','grayishpink','Grayish Pink',[196,174,173],'c4aead',12889773],
['nbs-iscc:grayishpurple','grayishpurple','Grayish Purple',[121,104,120],'796878',7956600],
['nbs-iscc:grayishpurplishblue','grayishpurplishblue','Grayish Purplish Blue',[76,81,109],'4c516d',5001581],
['nbs-iscc:grayishpurplishpink','grayishpurplishpink','Grayish Purplish Pink',[195,166,177],'c3a6b1',12822193],
['nbs-iscc:grayishpurplishred','grayishpurplishred','Grayish Purplish Red',[145,95,109],'915f6d',9527149],
['nbs-iscc:grayishred','grayishred','Grayish Red',[144,93,93],'905d5d',9461085],
['nbs-iscc:grayishreddishbrown','grayishreddishbrown','Grayish Reddish Brown',[103,76,71],'674c47',6769735],
['nbs-iscc:grayishreddishorange','grayishreddishorange','Grayish Reddish Orange',[180,116,94],'b4745e',11826270],
['nbs-iscc:grayishreddishpurple','grayishreddishpurple','Grayish Reddish Purple',[131,100,121],'836479',8610937],
['nbs-iscc:grayishviolet','grayishviolet','Grayish Violet',[85,76,105],'554c69',5590121],
['nbs-iscc:grayishyellow','grayishyellow','Grayish Yellow',[194,178,128],'c2b280',12759680],
['nbs-iscc:grayishyellowgreen','grayishyellowgreen','Grayish Yellow Green',[143,151,121],'8f9779',9410425],
['nbs-iscc:grayishyellowishbrown','grayishyellowishbrown','Grayish Yellowish Brown',[126,109,90],'7e6d5a',8285530],
['nbs-iscc:grayishyellowishpink','grayishyellowishpink','Grayish Yellowish Pink',[199,173,163],'c7ada3',13086115],
['nbs-iscc:greenishblack','greenishblack','Greenish Black',[30,35,33],'1e2321',1975073],
['nbs-iscc:greenishgray','greenishgray','Greenish Gray',[125,137,132],'7d8984',8227204],
['nbs-iscc:greenishwhite','greenishwhite','Greenish White',[223,237,232],'dfede8',14675432],
['nbs-iscc:lightblue','lightblue','Light Blue',[112,163,204],'70a3cc',7381964],
['nbs-iscc:lightbluishgray','lightbluishgray','Light Bluish Gray',[180,188,192],'b4bcc0',11844800],
['nbs-iscc:lightbluishgreen','lightbluishgreen','Light Bluish Green',[102,173,164],'66ada4',6729124],
['nbs-iscc:lightbrown','lightbrown','Light Brown',[166,123,91],'a67b5b',10910555],
['nbs-iscc:lightbrownishgray','lightbrownishgray','Light Brownish Gray',[142,130,121],'8e8279',9339513],
['nbs-iscc:lightgray','lightgray','Light Gray',[185,184,181],'b9b8b5',12171445],
['nbs-iscc:lightgrayishbrown','lightgrayishbrown','Light Grayish Brown',[149,128,112],'958070',9797744],
['nbs-iscc:lightgrayisholive','lightgrayisholive','Light Grayish Olive',[140,135,103],'8c8767',9209703],
['nbs-iscc:lightgrayishpurplishred','lightgrayishpurplishred','Light Grayish Purplish Red',[175,134,142],'af868e',11503246],
['nbs-iscc:lightgrayishred','lightgrayishred','Light Grayish Red',[173,136,132],'ad8884',11372676],
['nbs-iscc:lightgrayishreddishbrown','lightgrayishreddishbrown','Light Grayish Reddish Brown',[151,127,115],'977f73',9928563],
['nbs-iscc:lightgrayishyellowishbrown','lightgrayishyellowishbrown','Light Grayish Yellowish Brown',[174,155,130],'ae9b82',11443074],
['nbs-iscc:lightgreen','lightgreen','Light Green',[106,171,142],'6aab8e',6990734],
['nbs-iscc:lightgreenishblue','lightgreenishblue','Light Greenish Blue',[102,170,188],'66aabc',6728380],
['nbs-iscc:lightgreenishgray','lightgreenishgray','Light Greenish Gray',[178,190,181],'b2beb5',11714229],
['nbs-iscc:lightgreenishyellow','lightgreenishyellow','Light Greenish Yellow',[234,230,121],'eae679',15394425],
['nbs-iscc:lightolive','lightolive','Light Olive',[134,126,54],'867e36',8814134],
['nbs-iscc:lightolivebrown','lightolivebrown','Light Olive Brown',[150,113,23],'967117',9859351],
['nbs-iscc:lightolivegray','lightolivegray','Light Olive Gray',[138,135,118],'8a8776',9078646],
['nbs-iscc:lightorange','lightorange','Light Orange',[250,181,127],'fab57f',16430463],
['nbs-iscc:lightorangeyellow','lightorangeyellow','Light Orange Yellow',[251,201,127],'fbc97f',16501119],
['nbs-iscc:lightpink','lightpink','Light Pink',[249,204,202],'f9ccca',16370890],
['nbs-iscc:lightpurple','lightpurple','Light Purple',[182,149,192],'b695c0',11965888],
['nbs-iscc:lightpurplishblue','lightpurplishblue','Light Purplish Blue',[135,145,191],'8791bf',8884671],
['nbs-iscc:lightpurplishgray','lightpurplishgray','Light Purplish Gray',[191,185,189],'bfb9bd',12564925],
['nbs-iscc:lightpurplishpink','lightpurplishpink','Light Purplish Pink',[239,187,204],'efbbcc',15711180],
['nbs-iscc:lightreddishbrown','lightreddishbrown','Light Reddish Brown',[168,124,109],'a87c6d',11041901],
['nbs-iscc:lightreddishpurple','lightreddishpurple','Light Reddish Purple',[183,132,167],'b784a7',12027047],
['nbs-iscc:lightviolet','lightviolet','Light Violet',[140,130,182],'8c82b6',9208502],
['nbs-iscc:lightyellow','lightyellow','Light Yellow',[248,222,126],'f8de7e',16309886],
['nbs-iscc:lightyellowgreen','lightyellowgreen','Light Yellow Green',[201,220,137],'c9dc89',13229193],
['nbs-iscc:lightyellowishbrown','lightyellowishbrown','Light Yellowish Brown',[193,154,107],'c19a6b',12687979],
['nbs-iscc:lightyellowishgreen','lightyellowishgreen','Light Yellowish Green',[147,197,146],'93c592',9684370],
['nbs-iscc:lightyellowishpink','lightyellowishpink','Light Yellowish Pink',[244,194,194],'f4c2c2',16040642],
['nbs-iscc:mediumgray','mediumgray','Medium Gray',[132,132,130],'848482',8684674],
['nbs-iscc:moderateblue','moderateblue','Moderate Blue',[67,107,149],'436b95',4418453],
['nbs-iscc:moderatebluishgreen','moderatebluishgreen','Moderate Bluish Green',[49,120,115],'317873',3242099],
['nbs-iscc:moderatebrown','moderatebrown','Moderate Brown',[111,78,55],'6f4e37',7294519],
['nbs-iscc:moderategreen','moderategreen','Moderate Green',[59,120,97],'3b7861',3897441],
['nbs-iscc:moderategreenishblue','moderategreenishblue','Moderate Greenish Blue',[54,117,136],'367588',3569032],
['nbs-iscc:moderategreenishyellow','moderategreenishyellow','Moderate Greenish Yellow',[185,180,89],'b9b459',12170329],
['nbs-iscc:moderateolive','moderateolive','Moderate Olive',[102,93,30],'665d1e',6708510],
['nbs-iscc:moderateolivebrown','moderateolivebrown','Moderate Olive Brown',[108,84,30],'6c541e',7099422],
['nbs-iscc:moderateolivegreen','moderateolivegreen','Moderate Olive Green',[74,93,35],'4a5d23',4873507],
['nbs-iscc:moderateorange','moderateorange','Moderate Orange',[217,144,88],'d99058',14258264],
['nbs-iscc:moderateorangeyellow','moderateorangeyellow','Moderate Orange Yellow',[227,168,87],'e3a857',14919767],
['nbs-iscc:moderatepink','moderatepink','Moderate Pink',[222,165,164],'dea5a4',14591396],
['nbs-iscc:moderatepurple','moderatepurple','Moderate Purple',[134,96,142],'86608e',8806542],
['nbs-iscc:moderatepurplishblue','moderatepurplishblue','Moderate Purplish Blue',[78,81,128],'4e5180',5132672],
['nbs-iscc:moderatepurplishpink','moderatepurplishpink','Moderate Purplish Pink',[213,151,174],'d597ae',13997998],
['nbs-iscc:moderatepurplishred','moderatepurplishred','Moderate Purplish Red',[168,81,110],'a8516e',11030894],
['nbs-iscc:moderatered','moderatered','Moderate Red',[171,78,82],'ab4e52',11226706],
['nbs-iscc:moderatereddishbrown','moderatereddishbrown','Moderate Reddish Brown',[121,68,59],'79443b',7947323],
['nbs-iscc:moderatereddishorange','moderatereddishorange','Moderate Reddish Orange',[203,109,81],'cb6d51',13331793],
['nbs-iscc:moderatereddishpurple','moderatereddishpurple','Moderate Reddish Purple',[145,92,131],'915c83',9526403],
['nbs-iscc:moderateviolet','moderateviolet','Moderate Violet',[96,78,129],'604e81',6311553],
['nbs-iscc:moderateyellow','moderateyellow','Moderate Yellow',[201,174,93],'c9ae5d',13217373],
['nbs-iscc:moderateyellowgreen','moderateyellowgreen','Moderate Yellow Green',[138,154,91],'8a9a5b',9083483],
['nbs-iscc:moderateyellowishbrown','moderateyellowishbrown','Moderate Yellowish Brown',[130,102,68],'826644',8545860],
['nbs-iscc:moderateyellowishgreen','moderateyellowishgreen','Moderate Yellowish Green',[103,146,103],'679267',6787687],
['nbs-iscc:moderateyellowishpink','moderateyellowishpink','Moderate Yellowish Pink',[217,166,169],'d9a6a9',14263977],
['nbs-iscc:oliveblack','oliveblack','Olive Black',[37,36,29],'25241d',2434077],
['nbs-iscc:olivegray','olivegray','Olive Gray',[87,85,76],'57554c',5723468],
['nbs-iscc:paleblue','paleblue','Pale Blue',[145,163,176],'91a3b0',9544624],
['nbs-iscc:palegreen','palegreen','Pale Green',[141,163,153],'8da399',9282457],
['nbs-iscc:palegreenishyellow','palegreenishyellow','Pale Greenish Yellow',[235,232,164],'ebe8a4',15460516],
['nbs-iscc:paleorangeyellow','paleorangeyellow','Pale Orange Yellow',[250,214,165],'fad6a5',16438949],
['nbs-iscc:palepink','palepink','Pale Pink',[234,216,215],'ead8d7',15390935],
['nbs-iscc:palepurple','palepurple','Pale Purple',[170,152,169],'aa98a9',11180201],
['nbs-iscc:palepurplishblue','palepurplishblue','Pale Purplish Blue',[140,146,172],'8c92ac',9212588],
['nbs-iscc:palepurplishpink','palepurplishpink','Pale Purplish Pink',[232,204,215],'e8ccd7',15256791],
['nbs-iscc:palereddishpurple','palereddishpurple','Pale Reddish Purple',[170,138,158],'aa8a9e',11176606],
['nbs-iscc:paleviolet','paleviolet','Pale Violet',[150,144,171],'9690ab',9867435],
['nbs-iscc:paleyellow','paleyellow','Pale Yellow',[243,229,171],'f3e5ab',15984043],
['nbs-iscc:paleyellowgreen','paleyellowgreen','Pale Yellow Green',[218,223,183],'dadfb7',14344119],
['nbs-iscc:paleyellowishpink','paleyellowishpink','Pale Yellowish Pink',[236,213,197],'ecd5c5',15521221],
['nbs-iscc:pinkishgray','pinkishgray','Pinkish Gray',[193,182,179],'c1b6b3',12695219],
['nbs-iscc:pinkishwhite','pinkishwhite','Pinkish White',[234,227,225],'eae3e1',15393761],
['nbs-iscc:purplishblack','purplishblack','Purplish Black',[36,33,36],'242124',2367780],
['nbs-iscc:purplishgray','purplishgray','Purplish Gray',[139,133,137],'8b8589',9143689],
['nbs-iscc:purplishwhite','purplishwhite','Purplish White',[232,227,229],'e8e3e5',15262693],
['nbs-iscc:reddishblack','reddishblack','Reddish Black',[40,32,34],'282022',2629666],
['nbs-iscc:reddishgray','reddishgray','Reddish Gray',[143,129,127],'8f817f',9404799],
['nbs-iscc:strongblue','strongblue','Strong Blue',[0,103,165],'0067a5',26533],
['nbs-iscc:strongbluishgreen','strongbluishgreen','Strong Bluish Green',[0,122,116],'007a74',31348],
['nbs-iscc:strongbrown','strongbrown','Strong Brown',[128,70,27],'80461b',8406555],
['nbs-iscc:stronggreen','stronggreen','Strong Green',[0,121,89],'007959',31065],
['nbs-iscc:stronggreenishblue','stronggreenishblue','Strong Greenish Blue',[0,119,145],'007791',30609],
['nbs-iscc:stronggreenishyellow','stronggreenishyellow','Strong Greenish Yellow',[190,183,46],'beb72e',12498734],
['nbs-iscc:strongolivegreen','strongolivegreen','Strong Olive Green',[64,79,0],'404f00',4214528],
['nbs-iscc:strongorange','strongorange','Strong Orange',[237,135,45],'ed872d',15566637],
['nbs-iscc:strongorangeyellow','strongorangeyellow','Strong Orange Yellow',[234,162,33],'eaa221',15376929],
['nbs-iscc:strongpink','strongpink','Strong Pink',[234,147,153],'ea9399',15373209],
['nbs-iscc:strongpurple','strongpurple','Strong Purple',[135,86,146],'875692',8869522],
['nbs-iscc:strongpurplishblue','strongpurplishblue','Strong Purplish Blue',[84,90,167],'545aa7',5528231],
['nbs-iscc:strongpurplishpink','strongpurplishpink','Strong Purplish Pink',[230,143,172],'e68fac',15110060],
['nbs-iscc:strongpurplishred','strongpurplishred','Strong Purplish Red',[179,68,108],'b3446c',11748460],
['nbs-iscc:strongred','strongred','Strong Red',[188,63,74],'bc3f4a',12336970],
['nbs-iscc:strongreddishbrown','strongreddishbrown','Strong Reddish Brown',[136,45,23],'882d17',8924439],
['nbs-iscc:strongreddishorange','strongreddishorange','Strong Reddish Orange',[217,96,59],'d9603b',14245947],
['nbs-iscc:strongreddishpurple','strongreddishpurple','Strong Reddish Purple',[158,79,136],'9e4f88',10375048],
['nbs-iscc:strongviolet','strongviolet','Strong Violet',[96,78,151],'604e97',6311575],
['nbs-iscc:strongyellow','strongyellow','Strong Yellow',[212,175,55],'d4af37',13938487],
['nbs-iscc:strongyellowgreen','strongyellowgreen','Strong Yellow Green',[126,159,46],'7e9f2e',8298286],
['nbs-iscc:strongyellowishbrown','strongyellowishbrown','Strong Yellowish Brown',[153,101,21],'996515',10052885],
['nbs-iscc:strongyellowishgreen','strongyellowishgreen','Strong Yellowish Green',[68,148,74],'44944a',4494410],
['nbs-iscc:strongyellowishpink','strongyellowishpink','Strong Yellowish Pink',[249,147,121],'f99379',16356217],
['nbs-iscc:verydarkbluishgreen','verydarkbluishgreen','Very Dark Bluish Green',[0,42,41],'002a29',10793],
['nbs-iscc:verydarkgreen','verydarkgreen','Very Dark Green',[28,53,45],'1c352d',1848621],
['nbs-iscc:verydarkgreenishblue','verydarkgreenishblue','Very Dark Greenish Blue',[0,46,59],'002e3b',11835],
['nbs-iscc:verydarkpurple','verydarkpurple','Very Dark Purple',[48,25,52],'301934',3152180],
['nbs-iscc:verydarkpurplishred','verydarkpurplishred','Very Dark Purplish Red',[56,21,44],'38152c',3675436],
['nbs-iscc:verydarkred','verydarkred','Very Dark Red',[63,23,40],'3f1728',4134696],
['nbs-iscc:verydarkreddishpurple','verydarkreddishpurple','Very Dark Reddish Purple',[52,23,49],'341731',3413809],
['nbs-iscc:verydarkyellowishgreen','verydarkyellowishgreen','Very Dark Yellowish Green',[23,54,32],'173620',1521184],
['nbs-iscc:verydeeppurple','verydeeppurple','Very Deep Purple',[64,26,76],'401a4c',4201036],
['nbs-iscc:verydeeppurplishred','verydeeppurplishred','Very Deep Purplish Red',[84,19,59],'54133b',5509947],
['nbs-iscc:verydeepred','verydeepred','Very Deep Red',[92,9,35],'5c0923',6031651],
['nbs-iscc:verydeepreddishpurple','verydeepreddishpurple','Very Deep Reddish Purple',[84,25,78],'54194e',5511502],
['nbs-iscc:verydeepyellowishgreen','verydeepyellowishgreen','Very Deep Yellowish Green',[0,49,24],'003118',12568],
['nbs-iscc:verylightblue','verylightblue','Very Light Blue',[161,202,241],'a1caf1',10603249],
['nbs-iscc:verylightbluishgreen','verylightbluishgreen','Very Light Bluish Green',[150,222,209],'96ded1',9887441],
['nbs-iscc:verylightgreen','verylightgreen','Very Light Green',[142,209,178],'8ed1b2',9359794],
['nbs-iscc:verylightgreenishblue','verylightgreenishblue','Very Light Greenish Blue',[156,209,220],'9cd1dc',10277340],
['nbs-iscc:verylightpurple','verylightpurple','Very Light Purple',[213,186,219],'d5badb',14007003],
['nbs-iscc:verylightpurplishblue','verylightpurplishblue','Very Light Purplish Blue',[179,188,226],'b3bce2',11779298],
['nbs-iscc:verylightviolet','verylightviolet','Very Light Violet',[220,208,255],'dcd0ff',14471423],
['nbs-iscc:verylightyellowishgreen','verylightyellowishgreen','Very Light Yellowish Green',[182,229,175],'b6e5af',11986351],
['nbs-iscc:verypaleblue','verypaleblue','Very Pale Blue',[188,212,230],'bcd4e6',12375270],
['nbs-iscc:verypalegreen','verypalegreen','Very Pale Green',[199,230,215],'c7e6d7',13100759],
['nbs-iscc:verypalepurple','verypalepurple','Very Pale Purple',[214,202,221],'d6cadd',14076637],
['nbs-iscc:verypalepurplishblue','verypalepurplishblue','Very Pale Purplish Blue',[192,200,225],'c0c8e1',12634337],
['nbs-iscc:verypaleviolet','verypaleviolet','Very Pale Violet',[196,195,221],'c4c3dd',12895197],
['nbs-iscc:vividblue','vividblue','Vivid Blue',[0,161,194],'00a1c2',41410],
['nbs-iscc:vividbluishgreen','vividbluishgreen','Vivid Bluish Green',[0,136,130],'008882',34946],
['nbs-iscc:vividgreen','vividgreen','Vivid Green',[0,136,86],'008856',34902],
['nbs-iscc:vividgreenishblue','vividgreenishblue','Vivid Greenish Blue',[0,133,161],'0085a1',34209],
['nbs-iscc:vividgreenishyellow','vividgreenishyellow','Vivid Greenish Yellow',[220,211,0],'dcd300',14471936],
['nbs-iscc:vividorange','vividorange','Vivid Orange',[243,132,0],'f38400',15959040],
['nbs-iscc:vividorangeyellow','vividorangeyellow','Vivid Orange Yellow',[246,166,0],'f6a600',16164352],
['nbs-iscc:vividpink','vividpink','Vivid Pink',[255,181,186],'ffb5ba',16758202],
['nbs-iscc:vividpurple','vividpurple','Vivid Purple',[154,78,174],'9a4eae',10112686],
['nbs-iscc:vividpurplishblue','vividpurplishblue','Vivid Purplish Blue',[48,38,122],'30267a',3155578],
['nbs-iscc:vividpurplishred','vividpurplishred','Vivid Purplish Red',[206,70,118],'ce4676',13518454],
['nbs-iscc:vividred','vividred','Vivid Red',[190,0,50],'be0032',12451890],
['nbs-iscc:vividreddishorange','vividreddishorange','Vivid Reddish Orange',[226,88,34],'e25822',14833698],
['nbs-iscc:vividreddishpurple','vividreddishpurple','Vivid Reddish Purple',[135,0,116],'870074',8847476],
['nbs-iscc:vividviolet','vividviolet','Vivid Violet',[144,101,202],'9065ca',9463242],
['nbs-iscc:vividyellow','vividyellow','Vivid Yellow',[243,195,0],'f3c300',15975168],
['nbs-iscc:vividyellowgreen','vividyellowgreen','Vivid Yellow Green',[141,182,0],'8db600',9287168],
['nbs-iscc:vividyellowishgreen','vividyellowishgreen','Vivid Yellowish Green',[39,166,76],'27a64c',2598476],
['nbs-iscc:vividyellowishpink','vividyellowishpink','Vivid Yellowish Pink',[255,183,165],'ffb7a5',16758693],
['nbs-iscc:white','white','White',[242,243,244],'f2f3f4',15922164],
['nbs-iscc:yellowishgray','yellowishgray','Yellowish Gray',[191,184,165],'bfb8a5',12564645],
['nbs-iscc:yellowishwhite','yellowishwhite','Yellowish White',[240,234,214],'f0ead6',15788758]
    ];
}

sub _description {
    return {
          'subtitle' => 'Centroids of the NBS/ISCC catalog',
          'title' => 'NBS/ISCC Centroids',
          'description' => '    Kenneth L. Kelly and Deanne B. Judd.
    "Color: Universal Language and Dictionary of Names",
    National Bureau of Standards,
    Spec. Publ. 440, Dec. 1976, 189 pages.


    Color Name Dictionary: NBS/ISCC Centroids

[http://swissnet.ai.mit.edu/~jaffer/Color/Dictionaries#NBS-ISCC%20Centroids]
'
        }

}

1;
