/*
    This file is part of the KDE libraries
    SPDX-FileCopyrightText: 2017 Chinmoy Ranjan Pradhan <chinmoyrp65@gmail.com>

    SPDX-License-Identifier: LGPL-2.1-only OR LGPL-3.0-only OR LicenseRef-KDE-Accepted-LGPL
*/

#ifndef BATCHRENAMEJOB_H
#define BATCHRENAMEJOB_H

#include "job_base.h"
#include "kiocore_export.h"

namespace KIO
{
class BatchRenameJobPrivate;

/*!
 * \class KIO::BatchRenameJob
 * \inheaderfile KIO/BatchRenameJob
 * \inmodule KIOCore
 *
 * \brief A KIO job that renames multiple files in one go.
 *
 * \since 5.42
 */
class KIOCORE_EXPORT BatchRenameJob : public Job
{
    Q_OBJECT

public:
    ~BatchRenameJob() override;

Q_SIGNALS:
    /*!
     * Signals that a file was renamed.
     */
    void fileRenamed(const QUrl &oldUrl, const QUrl &newUrl);

protected Q_SLOTS:
    /*!
     * \reimp
     */
    void slotResult(KJob *job) override;

protected:
    /*!
     * \internal
     */
    KIOCORE_NO_EXPORT explicit BatchRenameJob(BatchRenameJobPrivate &dd);

private:
    Q_DECLARE_PRIVATE(BatchRenameJob)
};

/*!
 * \relates KIO::BatchRenameJob
 *
 * Renames multiple files at once.
 *
 * The new filename is obtained by replacing the characters represented by
 * \a placeHolder by the index \a index.
 *
 * E.g. Calling batchRename({"file:///Test.jpg"}, "Test #" 12, '#') renames
 * the file to "Test 12.jpg". A connected sequence of placeholders results in
 * leading zeros. batchRename({"file:///Test.jpg"}, "Test ####" 12, '#') renames
 * the file to "Test 0012.jpg". And if no placeholder is there then \a index is
 * appended to \a newName. Calling batchRename({"file:///Test.jpg"}, "NewTest" 12, '#')
 * renames the file to "NewTest12.jpg".
 *
 * \a srcList The list of items to rename.
 *
 * \a newName The base name to use in all new filenames.
 *
 * \a startIndex The integer(incremented after renaming a file) to add to the base name.
 *
 * \a placeHolder The character(s) which \a index will replace.
 *
 * Returns a pointer to the job handling the operation.
 * \since 5.42
 */
// TODO KF7 remove, replaces by batchRenameWithFunction
KIOCORE_EXPORT BatchRenameJob *
batchRename(const QList<QUrl> &srcList, const QString &newName, int startIndex, QChar placeHolder, JobFlags flags = DefaultFlags);

using renameFunctionType = std::function<QString(QStringView currentFileNameWithoutExtension)>;
/*!
 * \relates KIO::BatchRenameJob
 *
 * Renames multiple files at once. The new names are generated by the passed-in function \p renameFunction.
 *
 * \a srcList The list of items to rename.
 *
 * \a renameFunction A function used to find the new name of each input files
 *
 * Returns a pointer to the job handling the operation.
 *
 * \since 6.16
 */
KIOCORE_EXPORT BatchRenameJob *batchRenameWithFunction(const QList<QUrl> &srcList, const renameFunctionType renameFunction, KIO::JobFlags flags = DefaultFlags);
}

#endif
