<TeXmacs|2.1>

<style|<tuple|tmdoc|british|old-dots|old-lengths>>

<\body>
  <tmdoc-title|A Tour of Sage>

  Here is a tour of Sage adapted from the official
  <hlink|tutorial|http://doc.sagemath.org/html/en/a_tour_of_sage/> for GNU
  <TeXmacs>.

  <paragraph*|Sage as a Calculator>

  The Sage command line has a <code*|sage:> prompt; you do not have to add
  it. If you use the Sage plugin for <TeXmacs>, then put everything after the
  <code*|sage]> prompt in an input cell, and press enter to compute the
  corresponding output.

  <\session|sage|default>
    <\unfolded-io>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|unfolded-io>
      3 + 5
    <|unfolded-io>
      <math|8>
    </unfolded-io>

    <\input>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|input>
      \;
    </input>
  </session>

  The caret symbol means \Praise to a power\Q.

  <\session|sage|default>
    <\unfolded-io>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|unfolded-io>
      57.1 ^ 100
    <|unfolded-io>
      <math|4.60904368661396\<times\>10<rsup|175>>
    </unfolded-io>

    <\input>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|input>
      \;
    </input>
  </session>

  We compute the inverse of a <math|2\<times\>2> matrix in Sage.

  <\session|sage|default>
    <\unfolded-io>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|unfolded-io>
      matrix([[1,2], [3,4]])^(-1)
    <|unfolded-io>
      <math|<around*|(|<tabular*|<tformat|<cwith|1|-1|1|1|cell-halign|r>|<cwith|1|-1|1|1|cell-lborder|0ln>|<cwith|1|-1|2|2|cell-halign|r>|<cwith|1|-1|2|2|cell-rborder|0ln>|<table|<row|<cell|-2>|<cell|1>>|<row|<cell|<frac|3|2>>|<cell|-<frac|1|2>>>>>>|)>>
    </unfolded-io>

    <\input>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|input>
      \;
    </input>
  </session>

  Here we integrate a simple function.

  <\session|sage|default>
    <\input>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|input>
      x = var('x') \ \ # create a symbolic variable
    </input>

    <\unfolded-io>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|unfolded-io>
      integrate(sqrt(x)*sqrt(1+x), x)
    <|unfolded-io>
      <math|<frac|<frac|<around*|(|x+1|)><rsup|<frac|3|2>>|x<rsup|<frac|3|2>>>+<frac|<sqrt|x+1>|<sqrt|x>>|4*<space|0.17em><around*|(|<frac|<around*|(|x+1|)><rsup|2>|x<rsup|2>>-<frac|2*<space|0.17em><around*|(|x+1|)>|x>+1|)>>-<frac|1|8>*<space|0.17em>log
      <around*|(|<frac|<sqrt|x+1>|<sqrt|x>>+1|)>+<frac|1|8>*<space|0.17em>log
      <around*|(|<frac|<sqrt|x+1>|<sqrt|x>>-1|)>>
    </unfolded-io>

    <\input>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|input>
      \;
    </input>
  </session>

  This asks Sage to solve a quadratic equation. The symbol <math|==>
  represents equality in Sage.

  <\session|sage|default>
    <\input>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|input>
      a = var('a')
    </input>

    <\input>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|input>
      S = solve(x^2 + x == a, x)
    </input>

    <\unfolded-io>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|unfolded-io>
      S
    <|unfolded-io>
      <math|<around*|[|x=-<frac|1|2>*<space|0.17em><sqrt|4*<space|0.17em>a+1>-<frac|1|2>,x=<frac|1|2>*<space|0.17em><sqrt|4*<space|0.17em>a+1>-<frac|1|2>|]>>
    </unfolded-io>

    <\input>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|input>
      \;
    </input>
  </session>

  The result is a list of equalities.

  <\session|sage|default>
    <\unfolded-io>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|unfolded-io>
      S[0].rhs()
    <|unfolded-io>
      <math|-<frac|1|2>*<space|0.17em><sqrt|4*<space|0.17em>a+1>-<frac|1|2>>
    </unfolded-io>

    <\input>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|input>
      \;
    </input>
  </session>

  Naturally, Sage can plot various useful functions.

  <\session|sage|default>
    <\unfolded-io>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|unfolded-io>
      plot(sin(x) + sin(1.6*x), 0, 40)
    <|unfolded-io>
      <image|<tuple|<#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>|ps>|0.7par|||>
    </unfolded-io>

    <\input>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|input>
      \;
    </input>
  </session>

  <paragraph*|Power Computing with Sage>

  First we create a 500\<times\>500 matrix of random numbers.

  <\session|sage|default>
    <\input>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|input>
      m = random_matrix(RDF,500)
    </input>

    <\input>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|input>
      \;
    </input>
  </session>

  It takes Sage a few seconds to compute the eigenvalues of the matrix and
  plot them.

  <\session|sage|default>
    <\input>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|input>
      e = m.eigenvalues() \ #about 2 seconds
    </input>

    <\input>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|input>
      w = [(i, abs(e[i])) for i in range(len(e))]
    </input>

    <\unfolded-io>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|unfolded-io>
      points(w)
    <|unfolded-io>
      <image|<tuple|<#252150532D41646F62652D332E300A2525446F63756D656E74506170657253697A65733A206C65747465720A252550616765733A20310A25255469746C653A202F746D702F746D706A6E3035666266312E70730A252543726561746F723A206D6174706C6F746C69622076657273696F6E20332E332E312C20687474703A2F2F6D6174706C6F746C69622E6F72672F0A25254372656174696F6E446174653A20536174204A756C2031302032323A32333A303120323032310A25254F7269656E746174696F6E3A20706F7274726169740A2525426F756E64696E67426F783A2038302E3435343839323535323631383836203232372E3338333433333331393533383433203533312E35343531303734343733383132203536342E363136353636363830343631360A2525456E64436F6D6D656E74730A2525426567696E50726F6C6F670A2F6D706C6469637420382064696374206465660A6D706C6469637420626567696E0A2F6D207B206D6F7665746F207D2062696E64206465660A2F6C207B206C696E65746F207D2062696E64206465660A2F72207B20726C696E65746F207D2062696E64206465660A2F63207B206375727665746F207D2062696E64206465660A2F636C207B20636C6F736570617468207D2062696E64206465660A2F626F78207B0A2020202020206D0A2020202020203120696E646578203020720A20202020202030206578636820720A2020202020206E6567203020720A202020202020636C0A202020207D2062696E64206465660A2F636C6970626F78207B0A202020202020626F780A202020202020636C69700A2020202020206E6577706174680A202020207D2062696E64206465660A252150532D41646F62652D332E30205265736F757263652D466F6E740A25255469746C653A2044656A6156752053616E730A2525436F707972696768743A20436F707972696768742028632920323030332062792042697473747265616D2C20496E632E20416C6C205269676874732052657365727665642E20436F70797269676874202863292032303036206279205461766D6A6F6E67204261682E20416C6C205269676874732052657365727665642E2044656A615675206368616E6765732061726520696E207075626C696320646F6D61696E200A252543726561746F723A20436F6E7665727465642066726F6D20547275655479706520746F20747970652033206279205050520A3235206469637420626567696E0A2F5F647B62696E64206465667D62696E64206465660A2F5F6D7B6D6F7665746F7D5F640A2F5F6C7B6C696E65746F7D5F640A2F5F636C7B636C6F73657061746820656F66696C6C7D5F640A2F5F637B6375727665746F7D5F640A2F5F73637B37202D3120726F6C6C7B73657463616368656465766963657D7B706F7020706F7020706F7020706F7020706F7020706F707D6966656C73657D5F640A2F5F657B657865637D5F640A2F466F6E744E616D65202F44656A61567553616E73206465660A2F5061696E74547970652030206465660A2F466F6E744D61747269785B2E30303120302030202E303031203020305D6465660A2F466F6E7442426F785B2D31303231202D343633203137393320313233325D6465660A2F466F6E74547970652033206465660A2F456E636F64696E67205B202F7A65726F202F6F6E65202F74776F202F7468726565202F666F7572202F66697665202F736978202F6569676874205D206465660A2F466F6E74496E666F20313020646963742064757020626567696E0A2F46616D696C794E616D65202844656A6156752053616E7329206465660A2F46756C6C4E616D65202844656A6156752053616E7329206465660A2F4E6F746963652028436F707972696768742028632920323030332062792042697473747265616D2C20496E632E20416C6C205269676874732052657365727665642E20436F70797269676874202863292032303036206279205461766D6A6F6E67204261682E20416C6C205269676874732052657365727665642E2044656A615675206368616E6765732061726520696E207075626C696320646F6D61696E2029206465660A2F5765696768742028426F6F6B29206465660A2F56657273696F6E202856657273696F6E20322E333529206465660A2F4974616C6963416E676C6520302E30206465660A2F6973466978656450697463682066616C7365206465660A2F556E6465726C696E65506F736974696F6E202D313330206465660A2F556E6465726C696E65546869636B6E657373203930206465660A656E6420726561646F6E6C79206465660A2F43686172537472696E6773203920646963742064757020626567696E0A2F2E6E6F746465662030206465660A2F7A65726F7B3633362030203636202D31332035373020373432205F73630A33313820363634205F6D0A3236372036363420323239203633392032303320353839205F630A3137372035333920313635203436342031363520333634205F630A3136352032363420313737203138392032303320313339205F630A3232392038392032363720363420333138203634205F630A333639203634203430372038392034333320313339205F630A3435382031383920343731203236342034373120333634205F630A3437312034363420343538203533392034333320353839205F630A3430372036333920333639203636342033313820363634205F630A33313820373432205F6D0A3339392037343220343631203730392035303520363435205F630A3534382035383020353730203438362035373020333634205F630A35373020323431203534382031343720353035203833205F630A34363120313920333939202D313320333138202D3133205F630A323336202D31332031373320313920313330203833205F630A3837203134372036362032343120363620333634205F630A363620343836203837203538302031333020363435205F630A3137332037303920323336203734322033313820373432205F630A5F636C7D5F640A2F6F6E657B36333620302031313020302035343420373239205F73630A313234203833205F6D0A323835203833205F6C0A32383520363339205F6C0A31313020363034205F6C0A31313020363934205F6C0A32383420373239205F6C0A33383320373239205F6C0A333833203833205F6C0A353434203833205F6C0A3534342030205F6C0A3132342030205F6C0A313234203833205F6C0A5F636C7D5F640A2F74776F7B7B363336203020373320302035333620373432205F73630A313932203833205F6D0A353336203833205F6C0A3533362030205F6C0A37332030205F6C0A3733203833205F6C0A3131302031323120313631203137332032323620323339205F630A3239302033303420333331203334362033343820333635205F630A3338302034303020343032203433302034313420343535205F630A3432362034373920343333203530342034333320353238205F630A3433332035363620343139203539382033393220363232205F630A3336352036343620333330203635392032383620363539205F630A3235352036353920323232203635332031383820363433205F630A31353420363332203131372036313620373820353934205F630A373820363934205F6C0A3131382037313020313535203732322031383920373330205F630A3232332037333820323535203734322032383420373432205F630A7D5F657B3335392037343220343139203732332034363420363835205F630A3530392036343720353332203539372035333220353334205F630A3533322035303420353236203437352035313520343439205F630A3530342034323220343834203339302034353420333534205F630A3434362033343420343230203331372033373620323732205F630A33333220323237203237312031363420313932203833205F630A5F636C7D5F657D5F640A2F74687265657B7B3633362030203736202D31332035353620373432205F73630A34303620333933205F6D0A3435332033383320343930203336322035313620333330205F630A3534322032393820353536203235382035353620323132205F630A353536203134302035333120383420343832203435205F630A343332203620333632202D313320323731202D3133205F630A323430202D313320323038202D313020313736202D34205F630A313434203120313130203130203736203232205F630A373620313137205F6C0A313033203130312031333320383920313636203831205F630A3139382037332032333220363920323638203639205F630A333330203639203337372038312034303920313035205F630A3434312031323920343538203136352034353820323132205F630A3435382032353420343433203238382034313320333132205F630A3338332033333620333431203334392032383720333439205F630A7D5F657B32303220333439205F6C0A32303220343330205F6C0A32393120343330205F6C0A3333392034333020333736203433392034303220343539205F630A3432382034373820343431203530362034343120353433205F630A3434312035383020343237203630392034303120363239205F630A3337342036343920333336203635392032383720363539205F630A3236302036353920323331203635362032303020363530205F630A31363920363434203133352036333520393820363233205F630A393820373131205F6C0A3133352037323120313730203732392032303320373334205F630A3233352037333920323636203734322032393620373432205F630A3337302037343220343239203732352034373320363931205F630A3531372036353720353339203631312035333920353533205F630A3533392035313320353237203437392035303420343531205F630A3438312034323320343438203430332034303620333933205F630A5F636C7D5F657D5F640A2F666F75727B363336203020343920302035383020373239205F73630A33373820363433205F6D0A31323920323534205F6C0A33373820323534205F6C0A33373820363433205F6C0A33353220373239205F6D0A34373620373239205F6C0A34373620323534205F6C0A35383020323534205F6C0A35383020313732205F6C0A34373620313732205F6C0A3437362030205F6C0A3337382030205F6C0A33373820313732205F6C0A343920313732205F6C0A343920323637205F6C0A33353220373239205F6C0A5F636C7D5F640A2F666976657B7B3633362030203737202D31332035343920373239205F73630A31303820373239205F6D0A34393520373239205F6C0A34393520363436205F6C0A31393820363436205F6C0A31393820343637205F6C0A3231322034373220323237203437362032343120343738205F630A3235352034383020323730203438322032383420343832205F630A3336352034383220343239203435392034373720343135205F630A3532352033373020353439203331302035343920323334205F630A353439203135352035323420393420343735203531205F630A343236203820333537202D313320323639202D3133205F630A323338202D313320323037202D313020313735202D36205F630A313433202D31203131312036203737203137205F630A373720313136205F6C0A313036203130302031333620383820313638203830205F630A3139392037322032333220363920323637203639205F630A7D5F657B333233203639203336382038332034303120313133205F630A3433332031343320343530203138332034353020323334205F630A3435302032383420343333203332342034303120333534205F630A3336382033383420333233203339392032363720333939205F630A3234312033393920323134203339362031383820333930205F630A3136322033383420313335203337352031303820333633205F630A31303820373239205F6C0A5F636C7D5F657D5F640A2F7369787B7B3633362030203730202D31332035373320373432205F73630A33333020343034205F6D0A3238362034303420323531203338382032323520333538205F630A3139392033323820313836203238362031383620323334205F630A3138362031383120313939203133392032323520313039205F630A3235312037392032383620363420333330203634205F630A333734203634203430392037392034333520313039205F630A3436312031333920343734203138312034373420323334205F630A3437342032383620343631203332382034333520333538205F630A3430392033383820333734203430342033333020343034205F630A35323620373133205F6D0A35323620363233205F6C0A3530312036333520343736203634342034353120363530205F630A3432352036353620343030203635392033373620363539205F630A3331302036353920323630203633372032323620353933205F630A7D5F657B3139322035343920313732203438322031363820333934205F630A3138372034323220323131203434342032343020343539205F630A3236392034373420333031203438322033333620343832205F630A3430392034383220343637203435392035303920343135205F630A3535312033373120353733203331302035373320323334205F630A353733203135392035353020393920353036203534205F630A343632203920343033202D313320333330202D3133205F630A323436202D31332031383120313920313337203833205F630A3932203134372037302032343120373020333634205F630A373020343739203937203537312031353220363339205F630A3230362037303720323830203734322033373220373432205F630A3339362037343220343231203733392034343720373335205F630A3437322037333020343938203732332035323620373133205F630A5F636C7D5F657D5F640A2F65696768747B7B3633362030203638202D31332035363820373432205F73630A33313820333436205F6D0A3237312033343620323334203333332032303720333038205F630A3138302032383320313637203234392031363720323035205F630A3136372031363120313830203132362032303720313031205F630A3233342037362032373120363420333138203634205F630A333634203634203430312037362034323820313032205F630A3435352031323720343639203136312034363920323035205F630A3436392032343920343535203238332034323920333038205F630A3430322033333320333635203334362033313820333436205F630A32313920333838205F6D0A3137372033393820313434203431382031323020343437205F630A3936203437362038352035313120383520353533205F630A38352036313120313035203635372031343720363931205F630A3138382037323520323435203734322033313820373432205F630A7D5F657B3339302037343220343437203732352034383920363931205F630A3533302036353720353531203631312035353120353533205F630A3535312035313120353339203437362035313520343437205F630A3439312034313820343539203339382034313720333838205F630A3436342033373720353031203335352035323820333233205F630A3535342032393120353638203235312035363820323035205F630A353638203133342035343620383020353033203433205F630A343539203520333938202D313320333138202D3133205F630A323337202D313320313735203520313332203433205F630A38392038302036382031333420363820323035205F630A363820323531203831203239312031303820333233205F630A3133342033353520313731203337372032313920333838205F630A31383320353434205F6D0A3138332035303620313934203437362032313820343535205F630A7D5F657B3234322034333420323735203432342033313820343234205F630A3336302034323420333933203433342034313720343535205F630A3434312034373620343533203530362034353320353434205F630A3435332035383220343431203631312034313720363332205F630A3339332036353320333630203636342033313820363634205F630A3237352036363420323432203635332032313820363332205F630A3139342036313120313833203538322031383320353434205F630A5F636C7D5F657D5F640A656E6420726561646F6E6C79206465660A0A2F4275696C64476C7970680A207B6578636820626567696E0A2043686172537472696E677320657863680A203220636F7079206B6E6F776E206E6F747B706F70202F2E6E6F746465667D69660A20747275652033203120726F6C6C2067657420657865630A20656E647D5F640A0A2F4275696C6443686172207B0A203120696E646578202F456E636F64696E67206765742065786368206765740A203120696E646578202F4275696C64476C7970682067657420657865630A7D5F640A0A466F6E744E616D652063757272656E746469637420656E6420646566696E65666F6E7420706F700A656E640A2525456E6450726F6C6F670A2525506167653A203120310A6D706C6469637420626567696E0A38302E343535203232372E333833207472616E736C6174650A3435312E3039203333372E3233332030203020636C6970626F780A67736176650A302030206D0A3435312E3039303231352030206C0A3435312E303930323135203333372E323333313333206C0A30203333372E323333313333206C0A636C0A312E30303020736574677261790A66696C6C0A67726573746F72650A67736176650A31382E3738363339392031382E363532383332206D0A3434312E3636383633382031382E363532383332206C0A3434312E363638363338203333302E303333313333206C0A31382E373836333939203333302E303333313333206C0A636C0A312E30303020736574677261790A66696C6C0A67726573746F72650A2F70305F30207B0A6E6577706174680A7472616E736C6174650A30202D312E353831313339206D0A302E343139333233202D312E35383131333920302E383231353238202D312E343134353420312E313138303334202D312E31313830333420630A312E3431343534202D302E38323135323820312E353831313339202D302E34313933323320312E353831313339203020630A312E35383131333920302E34313933323320312E343134353420302E38323135323820312E31313830333420312E31313830333420630A302E38323135323820312E343134353420302E34313933323320312E353831313339203020312E35383131333920630A2D302E34313933323320312E353831313339202D302E38323135323820312E3431343534202D312E31313830333420312E31313830333420630A2D312E343134353420302E383231353238202D312E35383131333920302E343139333233202D312E353831313339203020630A2D312E353831313339202D302E343139333233202D312E3431343534202D302E383231353238202D312E313138303334202D312E31313830333420630A2D302E383231353238202D312E3431343534202D302E343139333233202D312E3538313133392030202D312E35383131333920630A636C0A0A7D2062696E64206465660A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A32362E39313838203332342E3034352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A32372E37333336203332342E3034352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A32382E35343835203331342E3133362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A32392E33363333203331342E3133362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A33302E31373832203331352E3439342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A33302E39393331203331352E3439342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A33312E38303739203330382E3036322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A33322E36323238203330382E3036322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A33332E34333737203331352E3537332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A33342E32353235203331352E3537332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A33352E30363734203331322E3938332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A33352E38383233203331322E3938332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A33362E36393731203330362E3636352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A33372E353132203330362E3636352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A33382E33323639203330392E3330382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A33392E31343137203330392E3330382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A33392E39353636203331392E3635392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A34302E37373135203331392E3635392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A34312E35383633203331332E3132352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A34322E34303132203331332E3132352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A34332E323136203331352E3030322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A34342E30333039203331352E3030322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A34342E38343538203331312E3632332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A34352E36363036203331312E3632332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A34362E34373535203330332E3432392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A34372E32393034203330332E3432392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A34382E31303532203330342E3233342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A34382E39323031203330342E3233342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A34392E373335203331312E3638312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A35302E35343938203331312E3638312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A35312E33363437203330362E3336352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A35322E31373936203330362E3336352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A35322E39393434203330342E35312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A35332E38303933203330342E35312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A35342E36323432203330342E3330392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A35352E343339203330342E3330392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A35362E32353339203330322E3131382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A35372E30363837203330322E3131382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A35372E38383336203330362E3637322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A35382E36393835203239392E3032322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A35392E35313333203239392E3032322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A36302E33323832203239382E39322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A36312E31343331203239382E39322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A36312E39353739203330312E3432382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A36322E37373238203330312E3432382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A36332E35383737203330322E3634392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A36342E34303235203330322E3634392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A36352E32313734203330332E31342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A36362E30333233203330332E31342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A36362E38343731203330322E3130322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A36372E363632203330322E3130322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A36382E34373639203239372E3136352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A36392E32393137203239372E3136352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A37302E31303636203239382E3835352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A37302E39323134203239382E3835352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A37312E37333633203239332E3232382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A37322E35353132203239332E3232382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A37332E333636203239352E3730352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A37342E31383039203239352E3730352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A37342E39393538203239322E3936392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A37352E38313036203239322E3936392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A37362E36323535203239342E32352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A37372E34343034203239342E32352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A37382E32353532203239332E3535342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A37392E30373031203239332E3535342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A37392E383835203239312E3236352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A38302E36393938203239312E3236352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A38312E35313437203239312E3433332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A38322E33323936203239312E3433332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A38332E31343434203238392E3930312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A38332E39353933203238392E3930312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A38342E37373432203239322E3837312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A38352E353839203239322E3837312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A38362E34303339203239302E3930342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A38372E32313837203239302E3930342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A38382E30333336203239332E3937352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A38382E38343835203239332E3937352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A38392E36363333203239302E3739342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A39302E34373832203239302E3739342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A39312E32393331203239302E3033382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A39322E31303739203239302E3033382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A39322E39323238203238382E3238322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A39332E37333737203238382E3238322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A39342E35353235203238362E3933362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A39352E33363734203238362E3933362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A39362E31383233203238362E3532362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A39362E39393731203238362E3532362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A39372E383132203238372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A39382E36323639203238372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A39392E34343137203238372E3135332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3130302E323537203238372E3135332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3130312E303731203238352E3131362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3130312E383836203238352E3131362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3130322E373031203238332E3332322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3130332E353136203238332E3332322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3130342E333331203238342E3731352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3130352E313436203238342E3731352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3130352E393631203238332E3030332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3130362E373736203238332E3030332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3130372E3539203238332E3533312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3130382E343035203238332E3533312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3130392E3232203238332E3338322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3131302E303335203238332E3338322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3131302E3835203238332E3036362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3131312E363635203238332E3036362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3131322E3438203238302E3536322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3131332E323934203238302E3536322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3131342E313039203238322E3337392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3131342E393234203238322E3337392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3131352E373339203238302E3935342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3131362E353534203238302E3935342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3131372E333639203237392E3236312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3131382E313834203237392E3236312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3131382E393938203237382E3031342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3131392E383133203237382E3031342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3132302E363238203237382E3632392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3132312E343433203237382E3632392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3132322E323538203237332E3834392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3132332E303733203237332E3834392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3132332E383838203237372E3233342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3132342E373033203237372E3233342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3132352E353137203237392E3935392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3132362E333332203237392E3935392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3132372E313437203237382E3430382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3132372E393632203237382E3430382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3132382E373737203237352E3730382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3132392E353932203237352E3730382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3133302E343037203237352E3839322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3133312E323231203237352E3839322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3133322E303336203237342E3338372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3133322E383531203237342E3338372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3133332E363636203236392E33382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3133342E343831203236392E33382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3133352E323936203237302E3132322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3133362E313131203237302E3132322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3133362E393235203237332E3634392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3133372E3734203237332E3634392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3133382E353535203237312E3438332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3133392E3337203237312E3438332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3134302E313835203237342E3133382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A313431203237342E3133382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3134312E383135203237322E3230322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3134322E3633203237322E3230322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3134332E343434203236382E3233372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3134342E323539203236382E3233372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3134352E303734203236352E3235372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3134352E383839203236352E3235372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3134362E373034203236342E3039322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3134372E353139203236342E3039322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3134382E333334203236332E37332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3134392E313438203236332E37332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3134392E393633203236312E3434332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3135302E373738203236312E3434332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3135312E353933203235392E3830322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3135322E343038203235392E3830322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3135332E323233203235392E3738392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3135342E303338203235392E3738392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3135342E383533203235372E3438342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3135352E363637203235372E3438342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3135362E343832203235372E3138342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3135372E323937203235372E3138342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3135382E313132203236322E3831312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3135382E393237203236322E3831312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3135392E373432203236342E3535332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3136302E353537203236342E3535332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3136312E333731203236312E3533392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3136322E313836203236312E3533392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3136332E303031203235392E3036382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3136332E383136203235392E3036382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3136342E363331203235362E37392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3136352E343436203235362E37392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3136362E323631203235372E3933322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3136372E303735203235372E3933322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3136372E3839203235352E3432312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3136382E373035203235352E3432312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3136392E3532203235362E3136322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3137302E333335203235362E3136322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3137312E3135203235342E3037312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3137312E393635203235342E3037312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3137322E3738203235342E3031322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3137332E353934203235342E3031322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3137342E343039203235342E3631372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3137352E323234203235342E3631372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3137362E303339203235332E3233352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3137362E383534203235332E3233352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3137372E363639203235352E3235352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3137382E343834203235352E3235352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3137392E323938203235302E3038372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3138302E313133203235322E3230322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3138302E393238203235322E3230322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3138312E373433203235312E3637382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3138322E353538203235312E3637382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3138332E333733203234392E3935392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3138342E313838203234392E3935392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3138352E303033203235302E3839332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3138352E383137203235302E3839332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3138362E363332203234342E3938392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3138372E343437203234342E3938392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3138382E323632203234332E3233332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3138392E303737203234332E3233332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3138392E383932203234352E3332342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3139302E373037203234352E3332342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3139312E353231203234332E3239332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3139322E333336203234332E3239332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3139332E313531203234352E3735342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3139332E393636203234352E3735342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3139342E373831203234322E3335392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3139352E353936203234322E3335392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3139362E343131203234342E30362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3139372E323235203234342E30362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3139382E3034203234332E36322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3139382E383535203234332E36322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3139392E3637203234332E3430332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3230302E343835203234332E3430332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3230312E33203234322E3233362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3230322E313135203234322E3233362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3230322E3933203234302E3532312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3230332E373434203234302E3532312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3230342E353539203234302E3131352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3230352E333734203234302E3131352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3230362E313839203233382E3435392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3230372E303034203233382E3435392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3230372E383139203233372E362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3230382E363334203233372E362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3230392E343438203233362E3535332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3231302E323633203233362E3535332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3231312E303738203233332E3630392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3231312E383933203233332E3630392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3231322E373038203233362E3137392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3231332E353233203233362E3137392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3231342E333338203233352E3632392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3231352E313533203233352E3632392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3231352E393637203233362E3832342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3231362E373832203233362E3832342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3231372E353937203233342E3136342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3231382E343132203233342E3136342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3231392E323237203233342E3937312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3232302E303432203233312E3431372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3232302E383537203233322E3134322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3232312E363731203233322E3134322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3232322E343836203233302E3037322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3232332E333031203233302E3037322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3232342E313136203232382E3634322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3232342E393331203232382E3634322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3232352E373436203232382E3030332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3232362E353631203232382E3030332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3232372E333735203233312E3134352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3232382E3139203233312E3134352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3232392E303035203233312E3538342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3232392E3832203233312E3538342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3233302E363335203233302E3034392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3233312E3435203233302E3034392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3233322E323635203233302E3334372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3233332E3038203233302E3334372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3233332E383934203232382E3036332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3233342E373039203232382E3036332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3233352E353234203232352E3430342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3233362E333339203232352E3430342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3233372E313534203232342E3737312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3233372E393639203232342E3737312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3233382E373834203232332E3239392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3233392E353938203232332E3239392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3234302E343133203232332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3234312E323238203232332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3234322E303433203231392E3736332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3234322E383538203231392E3736332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3234332E363733203232312E3436352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3234342E343838203232312E3436352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3234352E333033203231392E3030342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3234362E313137203231392E3030342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3234362E393332203231382E3334332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3234372E373437203231382E3334332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3234382E353632203231382E3832342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3234392E333737203231382E3832342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3235302E313932203231352E3532352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3235312E303037203231352E3532352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3235312E383231203231382E36312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3235322E363336203231382E36312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3235332E343531203232312E3338342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3235342E323636203232312E3338342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3235352E303831203231362E38342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3235352E383936203231362E38342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3235362E373131203231372E3034342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3235372E353235203231372E3034342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3235382E3334203231372E3137312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3235392E313535203231372E3137312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3235392E3937203231362E3035392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3236302E373835203231362E3035392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3236312E36203231342E3435352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3236322E343135203231342E3435352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3236332E3233203231332E3334372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3236342E303434203231332E3334372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3236342E383539203231312E3736342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3236352E363734203231312E3736342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3236362E343839203231322E3933352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3236372E333034203231322E3933352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3236382E313139203231312E3037382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3236382E393334203231312E3037382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3236392E373438203230372E3932382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3237302E353633203230372E3932382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3237312E333738203230352E3639322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3237322E313933203230352E3639322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3237332E303038203230382E3736362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3237332E383233203230382E3736362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3237342E363338203230352E3230362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3237352E343533203230352E3230362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3237362E323637203230342E3530312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3237372E303832203230342E3530312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3237372E383937203230332E3135382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3237382E373132203230332E3135382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3237392E353237203230342E3734352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3238302E333432203230322E3039392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3238312E313537203230322E3039392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3238312E393731203230302E3539342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3238322E373836203230302E3539342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3238332E363031203139392E3732332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3238342E343136203139392E3732332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3238352E323331203139382E30342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3238362E303436203139382E30342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3238362E383631203139362E3730332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3238372E363735203139362E3730332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3238382E3439203139372E3435332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3238392E333035203139372E3435332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3239302E3132203139372E3038362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3239302E393335203139342E3535372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3239312E3735203139342E3535372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3239322E353635203139332E392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3239332E3338203139332E392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3239342E313934203139362E3034362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3239352E303039203139362E3034362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3239352E383234203139352E3134372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3239362E363339203139352E3134372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3239372E343534203138392E3934342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3239382E323639203138392E3934342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3239392E303834203139322E3839332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3239392E383938203139322E3839332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3330302E373133203139312E3030362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3330312E353238203139312E3030362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3330322E333433203138392E3234372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3330332E313538203138392E3234372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3330332E393733203138352E3136352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3330342E373838203138352E3136352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3330352E363033203138312E3837322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3330362E343137203138312E3837322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3330372E323332203138322E3634382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3330382E303437203138322E3634382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3330382E383632203137392E3334362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3330392E363737203137392E3334362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3331302E343932203137372E3830332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3331312E333037203137372E3830332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3331322E313231203138302E3439382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3331322E393336203138302E3439382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3331332E373531203137372E3935362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3331342E353636203137372E3935362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3331352E333831203138322E3030372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3331362E313936203138322E3030372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3331372E303131203138302E31352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3331372E383235203138302E31352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3331382E3634203137362E3737322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3331392E343535203137362E3737322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3332302E3237203137352E3632382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3332312E303835203137352E3632382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3332312E39203137342E3734342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3332322E373135203137342E3734342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3332332E3533203137362E3935382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3332342E333434203137362E3935382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3332352E313539203137322E3334312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3332352E393734203137322E3334312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3332362E373839203137312E3737342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3332372E363034203137312E3737342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3332382E343139203136382E3138322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3332392E323334203136382E3138322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3333302E303438203136362E3135392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3333302E383633203136362E3135392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3333312E363738203136352E3935342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3333322E343933203136352E3935342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3333332E333038203136322E3830332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3333342E313233203136322E3830332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3333342E393338203136302E3538352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3333352E373533203136302E3538352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3333362E353637203135362E37322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3333372E333832203135362E37322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3333382E313937203135332E3139392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3333392E303132203135332E3139392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3333392E383237203135362E3737382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3334302E363432203135362E3737382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3334312E343537203136332E3433312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3334322E323731203136332E3433312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3334332E303836203135322E33392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3334332E393031203136312E3535322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3334342E373136203136312E3535322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3334352E353331203135392E3239392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3334362E333436203135392E3239392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3334372E313631203135322E3236332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3334372E393735203135322E3236332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3334382E3739203135332E342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3334392E363035203135332E342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3335302E3432203135332E3935342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3335312E323335203135332E3935342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3335322E3035203135312E3832342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3335322E383635203135312E3832342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3335332E3638203136322E3239312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3335342E343934203136322E3239312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3335352E333039203136312E3931392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3335362E313234203135372E3035312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3335362E393339203135372E3035312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3335372E373534203135332E3338362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3335382E353639203135332E3338362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3335392E333834203134362E3437362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3336302E313938203134362E3437362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3336312E303133203134352E3736312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3336312E383238203134352E3736312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3336322E363433203134362E3430392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3336332E343538203134362E3430392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3336342E323733203134322E3134382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3336352E303838203134322E3134382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3336352E393033203133382E3339322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3336362E373137203133382E3339322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3336372E353332203134312E3538392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3336382E333437203134312E3538392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3336392E313632203133392E3635362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3336392E393737203133392E3635362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3337302E373932203133382E34322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3337312E363037203133382E34322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3337322E343231203133372E3031372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3337332E323336203133372E3031372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3337342E303531203133302E3138372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3337342E383636203133302E3138372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3337352E363831203133312E3231352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3337362E343936203133312E3231352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3337372E333131203133352E30322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3337382E313235203133352E30322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3337382E3934203131392E3231352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3337392E373535203131392E3231352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3338302E3537203132392E3637372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3338312E333835203132392E3637372070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3338322E32203132382E3333392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3338332E303135203132382E3333392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3338332E3833203132332E3830322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3338342E363434203132332E3830322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3338352E343539203132372E3932312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3338362E323734203132302E3734382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3338372E303839203132302E3734382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3338372E393034203131382E3933382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3338382E373139203131382E3933382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3338392E353334203131362E3937332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3339302E333438203131362E3937332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3339312E313633203131392E3336352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3339312E393738203131392E3336352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3339322E373933203131372E3634362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3339332E363038203131372E3634362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3339342E343233203131352E3637312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3339352E323338203131352E3637312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3339362E303533203131312E3531322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3339362E383637203131312E3531322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3339372E363832203130362E3539332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3339382E343937203131312E3536322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3339392E333132203131312E3536322070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3430302E313237203131312E3732312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3430302E393432203131312E3732312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3430312E373537203130382E3932392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3430322E353731203130382E3932392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3430332E333836203130332E3833352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3430342E323031203130332E3833352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3430352E3031362039372E363433342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3430352E3833312039372E363433342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3430362E3634362039362E313036352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3430372E3436312039362E313036352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3430382E3237352039302E373939332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3430392E30392039302E373939332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3430392E3930352038392E313635352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3431302E37322038392E313635352070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3431312E3533352039322E333931312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3431322E33352039322E333931312070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3431332E3136352038382E383933382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3431332E39382038382E383933382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3431342E3739342039322E353133382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3431352E3630392038372E393136392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3431362E3432342038372E393136392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3431372E3233392037392E323535382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3431382E3035342037392E323535382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3431382E3836392037362E353635362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3431392E3638342037362E353635362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3432302E3439382037332E333333382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3432312E3331332037332E333333382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3432322E3132382037322E353434362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3432322E3934332036382E323833392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3432332E3735382036352E343733332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3432342E3537332036352E343733332070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3432352E3338382035362E353334342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3432362E3230332035362E353334342070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3432372E3031372036302E313237362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3432372E3833322036302E313237362070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3432382E3634372036332E373131392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3432392E3436322036332E373131392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3433302E3237372035382E343438392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3433312E3039322035382E343438392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3433312E3930372033382E3736382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3433322E3732312032342E363430392070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A67736176650A3432322E39203331312E342031382E37392031382E363520636C6970626F780A3433332E3533362033382E303236382070305F300A302E30303020302E30303020312E30303020736574726762636F6C6F720A66696C6C0A67726573746F72650A302E383030207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A5B5D203020736574646173680A302E30303020736574677261790A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E38207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A302030206D0A30202D332E35206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E393138382032332E38373139206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E38207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A302030206D0A30202D332E35206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A3130382E3430352032332E38373139206F0A67726573746F72650A2F44656A61567553616E732066696E64666F6E740A31302E303030207363616C65666F6E740A736574666F6E740A67736176650A39382E38363631363820392E323738313235207472616E736C6174650A302E30303030303020726F746174650A302E3030303030302030206D202F6F6E6520676C79706873686F770A362E3336323330352030206D202F7A65726F20676C79706873686F770A31322E3732343630392030206D202F7A65726F20676C79706873686F770A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E38207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A302030206D0A30202D332E35206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A3138392E3839322032332E38373139206F0A67726573746F72650A67736176650A3138302E33353236343820392E323738313235207472616E736C6174650A302E30303030303020726F746174650A302E3030303030302030206D202F74776F20676C79706873686F770A362E3336323330352030206D202F7A65726F20676C79706873686F770A31322E3732343630392030206D202F7A65726F20676C79706873686F770A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E38207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A302030206D0A30202D332E35206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A3237312E3337382032332E38373139206F0A67726573746F72650A67736176650A3236312E38333931323920392E323738313235207472616E736C6174650A302E30303030303020726F746174650A302E3030303030302030206D202F746872656520676C79706873686F770A362E3336323330352030206D202F7A65726F20676C79706873686F770A31322E3732343630392030206D202F7A65726F20676C79706873686F770A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E38207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A302030206D0A30202D332E35206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A3335322E3836352032332E38373139206F0A67726573746F72650A67736176650A3334332E33323536303920392E323738313235207472616E736C6174650A302E30303030303020726F746174650A302E3030303030302030206D202F666F757220676C79706873686F770A362E3336323330352030206D202F7A65726F20676C79706873686F770A31322E3732343630392030206D202F7A65726F20676C79706873686F770A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E38207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A302030206D0A30202D332E35206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A3433342E3335312032332E38373139206F0A67726573746F72650A67736176650A3432342E38313230393020392E323738313235207472616E736C6174650A302E30303030303020726F746174650A302E3030303030302030206D202F6669766520676C79706873686F770A362E3336323330352030206D202F7A65726F20676C79706873686F770A31322E3732343630392030206D202F7A65726F20676C79706873686F770A67726573746F72650A302E363030207365746C696E6577696474680A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A302030206D0A30202D32206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A34332E3231362032332E38373139206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A302030206D0A30202D32206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A35392E353133332032332E38373139206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A302030206D0A30202D32206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A37352E383130362032332E38373139206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A302030206D0A30202D32206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A39322E313037392032332E38373139206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A302030206D0A30202D32206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A3132342E3730332032332E38373139206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A302030206D0A30202D32206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A3134312032332E38373139206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A302030206D0A30202D32206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A3135372E3239372032332E38373139206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A302030206D0A30202D32206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A3137332E3539342032332E38373139206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A302030206D0A30202D32206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A3230362E3138392032332E38373139206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A302030206D0A30202D32206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A3232322E3438362032332E38373139206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A302030206D0A30202D32206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A3233382E3738342032332E38373139206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A302030206D0A30202D32206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A3235352E3038312032332E38373139206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A302030206D0A30202D32206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A3238372E3637352032332E38373139206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A302030206D0A30202D32206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A3330332E3937332032332E38373139206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A302030206D0A30202D32206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A3332302E32372032332E38373139206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A302030206D0A30202D32206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A3333362E3536372032332E38373139206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A302030206D0A30202D32206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A3336392E3136322032332E38373139206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A302030206D0A30202D32206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A3338352E3435392032332E38373139206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A302030206D0A30202D32206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A3430312E3735372032332E38373139206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A302030206D0A30202D32206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A3431382E3035342032332E38373139206F0A67726573746F72650A302E383030207365746C696E6577696474680A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E38207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D332E352030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E393138382032332E38373139206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E38207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D332E352030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E393138382036382E3731206F0A67726573746F72650A67736176650A31332E3535393337352036342E393133313433207472616E736C6174650A302E30303030303020726F746174650A302E3030303030302030206D202F74776F20676C79706873686F770A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E38207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D332E352030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E39313838203131332E353438206F0A67726573746F72650A67736176650A31332E353539333735203130392E373531323835207472616E736C6174650A302E30303030303020726F746174650A302E3030303030302030206D202F666F757220676C79706873686F770A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E38207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D332E352030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E39313838203135382E333836206F0A67726573746F72650A67736176650A31332E353539333735203135342E353839343238207472616E736C6174650A302E30303030303020726F746174650A302E3030303030302030206D202F73697820676C79706873686F770A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E38207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D332E352030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E39313838203230332E323234206F0A67726573746F72650A67736176650A31332E353539333735203139392E343237353730207472616E736C6174650A302E30303030303020726F746174650A302E3030303030302030206D202F656967687420676C79706873686F770A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E38207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D332E352030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E39313838203234382E303633206F0A67726573746F72650A67736176650A372E323030303030203234342E323635373133207472616E736C6174650A302E30303030303020726F746174650A302E3030303030302030206D202F6F6E6520676C79706873686F770A362E3336323330352030206D202F7A65726F20676C79706873686F770A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E38207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D332E352030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E39313838203239322E393031206F0A67726573746F72650A67736176650A372E323030303030203238392E313033383536207472616E736C6174650A302E30303030303020726F746174650A302E3030303030302030206D202F6F6E6520676C79706873686F770A362E3336323330352030206D202F74776F20676C79706873686F770A67726573746F72650A302E363030207365746C696E6577696474680A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D322030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E393138382033352E30383134206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D322030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E393138382034362E32393039206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D322030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E393138382035372E35303035206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D322030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E393138382037392E39313936206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D322030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E393138382039312E31323931206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D322030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E39313838203130322E333339206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D322030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E39313838203132342E373538206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D322030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E39313838203133352E393637206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D322030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E39313838203134372E313737206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D322030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E39313838203136392E353936206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D322030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E39313838203138302E383035206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D322030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E39313838203139322E303135206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D322030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E39313838203231342E343334206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D322030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E39313838203232352E363434206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D322030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E39313838203233362E383533206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D322030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E39313838203235392E323732206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D322030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E39313838203237302E343832206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D322030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E39313838203238312E363931206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D322030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E39313838203330342E3131206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D322030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E39313838203331352E3332206F0A67726573746F72650A67736176650A2F6F207B0A67736176650A6E6577706174680A7472616E736C6174650A302E36207365746C696E6577696474680A31207365746C696E656A6F696E0A30207365746C696E656361700A2D302030206D0A2D322030206C0A0A67736176650A302E30303020736574677261790A66696C6C0A67726573746F72650A7374726F6B650A67726573746F72650A7D2062696E64206465660A32362E39313838203332362E353239206F0A67726573746F72650A302E383030207365746C696E6577696474680A30207365746C696E656A6F696E0A32207365746C696E656361700A67736176650A32362E39313837352031382E363532383332206D0A32362E3931383735203333302E303333313333206C0A7374726F6B650A67726573746F72650A67736176650A31382E3738363339392032332E383731383735206D0A3434312E3636383633382032332E383731383735206C0A7374726F6B650A67726573746F72650A0A656E640A73686F77706167650A2525454F460A>|ps>|0.7par|||>
    </unfolded-io>

    <\input>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|input>
      \;
    </input>
  </session>

  Thanks to the GNU Multiprecision Library (GMP), Sage can handle very large
  numbers, even numbers with millions or billions of digits.

  <\session|sage|default>
    <\unfolded-io>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|unfolded-io>
      factorial(100)
    <|unfolded-io>
      <math|93326215443944152681699238856266700490715968264381621468592963895217599993229915608941463976156518286253697920827223758251185210916864000000000000000000000000>
    </unfolded-io>

    <\input>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|input>
      n = factorial(1000000) \ #about 2.5 seconds
    </input>

    <\input>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|input>
      \;
    </input>
  </session>

  This computes at least 100 digits of \<pi\>.

  <\session|sage|default>
    <\unfolded-io>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|unfolded-io>
      N(pi, digits=100)
    <|unfolded-io>
      <math|3.141592653589793238462643383279502884197169399375105820974944592307816406286208998628034825342117068>
    </unfolded-io>

    <\input>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|input>
      \;
    </input>
  </session>

  This asks Sage to factor a polynomial in two variables.

  <\session|sage|default>
    <\input>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|input>
      R.\<less\>x,y\<gtr\> = QQ[]
    </input>

    <\input>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|input>
      F = factor(x^99 + y^99)
    </input>

    <\unfolded-io>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|unfolded-io>
      F
    <|unfolded-io>
      <math|<around|(|x+y|)>\<cdot\><around|(|x<rsup|2>-x*y+y<rsup|2>|)>\<cdot\><around|(|x<rsup|6>-x<rsup|3>*y<rsup|3>+y<rsup|6>|)>\<cdot\><around|(|x<rsup|10>-x<rsup|9>*y+x<rsup|8>*y<rsup|2>-x<rsup|7>*y<rsup|3>+x<rsup|6>*y<rsup|4>-x<rsup|5>*y<rsup|5>+x<rsup|4>*y<rsup|6>-x<rsup|3>*y<rsup|7>+x<rsup|2>*y<rsup|8>-x*y<rsup|9>+y<rsup|10>|)>\<cdot\><around|(|x<rsup|20>+x<rsup|19>*y-x<rsup|17>*y<rsup|3>-x<rsup|16>*y<rsup|4>+x<rsup|14>*y<rsup|6>+x<rsup|13>*y<rsup|7>-x<rsup|11>*y<rsup|9>-x<rsup|10>*y<rsup|10>-x<rsup|9>*y<rsup|11>+x<rsup|7>*y<rsup|13>+x<rsup|6>*y<rsup|14>-x<rsup|4>*y<rsup|16>-x<rsup|3>*y<rsup|17>+x*y<rsup|19>+y<rsup|20>|)>\<cdot\><around|(|x<rsup|60>+x<rsup|57>*y<rsup|3>-x<rsup|51>*y<rsup|9>-x<rsup|48>*y<rsup|12>+x<rsup|42>*y<rsup|18>+x<rsup|39>*y<rsup|21>-x<rsup|33>*y<rsup|27>-x<rsup|30>*y<rsup|30>-x<rsup|27>*y<rsup|33>+x<rsup|21>*y<rsup|39>+x<rsup|18>*y<rsup|42>-x<rsup|12>*y<rsup|48>-x<rsup|9>*y<rsup|51>+x<rsup|3>*y<rsup|57>+y<rsup|60>|)>>
    </unfolded-io>

    <\unfolded-io>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|unfolded-io>
      F.expand()
    <|unfolded-io>
      <math|x<rsup|99>+y<rsup|99>>
    </unfolded-io>

    <\input>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|input>
      \;
    </input>
  </session>

  Sage takes just under 5 seconds to compute the numbers of ways to partition
  one hundred million as a sum of positive integers.

  <\session|sage|default>
    <\input>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|input>
      z = Partitions(10^8).cardinality() #about 4.5 seconds
    </input>

    <\unfolded-io>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|unfolded-io>
      str(z)[:40]
    <|unfolded-io>
      1760517045946249141360373894679135204009
    </unfolded-io>

    <\input>
      \<gtr\>\<gtr\>\<gtr\>\ 
    <|input>
      \;
    </input>
  </session>

  <tmdoc-copyright|2018-2021|Darcy Shen>

  <tmdoc-license|Permission is granted to copy, distribute and/or modify this
  document under the terms of the GNU Free Documentation License, Version 1.1
  or any later version published by the Free Software Foundation; with no
  Invariant Sections, with no Front-Cover Texts, and with no Back-Cover
  Texts. A copy of the license is included in the section entitled "GNU Free
  Documentation License".>
</body>

<initial|<\collection>
</collection>>