/*
 * Decompiled with CFR 0.152.
 */
package org.openhab.core.scheduler;

import java.time.DateTimeException;
import java.time.DayOfWeek;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoField;
import java.time.temporal.ChronoUnit;
import java.time.temporal.Temporal;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.eclipse.jdt.annotation.NonNullByDefault;
import org.eclipse.jdt.annotation.Nullable;
import org.openhab.core.scheduler.SchedulerTemporalAdjuster;

@NonNullByDefault
public class CronAdjuster
implements SchedulerTemporalAdjuster {
    public static final String ANNUALLY = "@annually";
    public static final String YEARLY = "@yearly";
    public static final String MONTHLY = "@monthly";
    public static final String WEEKLY = "@weekly";
    public static final String DAILY = "@daily";
    public static final String HOURLY = "@hourly";
    public static final String REBOOT = "@reboot";
    private static final Pattern WEEKDAY_PATTERN = Pattern.compile("(?<day>\\d+|MON|TUE|WED|THU|FRI|SAT|SUN)(#(?<nr>\\d+)|(?<l>L))?", 2);
    private static final String[] MONTHS2 = new String[]{"JAN", "FEB", "MAR", "APR", "MAY", "JUN", "JUL", "AUG", "SEP", "OCT", "NOV", "DEC"};
    private static final Map<String, Integer> MONTHS = IntStream.range(0, MONTHS2.length).mapToObj(i -> Map.entry(MONTHS2[i], i)).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
    private static final String[] WEEK_DAYS_STRINGS = new String[]{"MON", "TUE", "WED", "THU", "FRI", "SAT", "SUN"};
    private static final Map<String, Integer> WEEK_DAYS = IntStream.range(0, WEEK_DAYS_STRINGS.length).mapToObj(i -> Map.entry(WEEK_DAYS_STRINGS[i], i + 1)).collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue));
    private final List<Field> fields = new ArrayList<Field>(7);
    private final Map<String, String> environmentMap;
    private final boolean reboot;

    public CronAdjuster(String specification) {
        String[] parts;
        String[] entries = specification.split("[\n\r]+");
        this.environmentMap = this.parseEnvironment(entries);
        String cronExpression = entries[entries.length - 1].trim();
        this.reboot = REBOOT.equals(cronExpression);
        if (cronExpression.startsWith("@")) {
            cronExpression = this.preDeclared(cronExpression);
        }
        if ((parts = cronExpression.trim().toUpperCase().split("\\s+")).length < 6 || parts.length > 7) {
            throw new IllegalArgumentException(String.format("Invalid cron expression, too %s fields. 6 or 7 (with year) expected but was: '%s'", parts.length < 6 ? "little" : "many", cronExpression));
        }
        if (parts.length > 6) {
            this.parseAndAdd(cronExpression, parts[6], ChronoField.YEAR);
        }
        this.parse(cronExpression, parts[5], ChronoField.DAY_OF_WEEK, WEEK_DAYS);
        this.parse(cronExpression, parts[4], ChronoField.MONTH_OF_YEAR, MONTHS);
        this.parseAndAdd(cronExpression, parts[3], ChronoField.DAY_OF_MONTH);
        this.parseAndAdd(cronExpression, parts[2], ChronoField.HOUR_OF_DAY);
        this.parseAndAdd(cronExpression, parts[1], ChronoField.MINUTE_OF_HOUR);
        this.parseAndAdd(cronExpression, parts[0], ChronoField.SECOND_OF_MINUTE);
        try {
            this.adjustInto(ZonedDateTime.now());
        }
        catch (DateTimeException e) {
            throw new IllegalArgumentException(String.format("Invalid cron expression '%s': %s", cronExpression, e.getMessage()));
        }
    }

    public Map<String, String> getEnv() {
        return this.environmentMap;
    }

    public boolean isReboot() {
        return this.reboot;
    }

    @Override
    public boolean isDone(Temporal temporal) {
        return CronAdjuster.checkMaxYear(temporal);
    }

    private Map<String, String> parseEnvironment(String[] entries) {
        HashMap<String, String> map = new HashMap<String, String>();
        if (entries.length > 1) {
            int i = 0;
            while (i < entries.length - 1) {
                String entry = entries[i];
                if (!entry.startsWith("#") && !entry.isEmpty()) {
                    int n = entry.indexOf(61);
                    if (n >= 0) {
                        String key = entry.substring(0, n).trim();
                        String value = entry.substring(n + 1).trim();
                        map.put(key, value);
                    } else {
                        map.put(entry.trim(), Boolean.TRUE.toString());
                    }
                }
                ++i;
            }
            return map;
        }
        return Map.of();
    }

    private String preDeclared(String expression) {
        switch (expression) {
            case "@annually": 
            case "@yearly": {
                return "0 0 0 1 1 *";
            }
            case "@monthly": {
                return "0 0 0 1 * *";
            }
            case "@weekly": {
                return "0 0 0 ? * MON";
            }
            case "@daily": {
                return "0 0 0 * * ?";
            }
            case "@hourly": {
                return "0 0 * * * ?";
            }
            case "@reboot": {
                return "0 0 0 1 1 ? 1900";
            }
        }
        throw new IllegalArgumentException(String.format("Unrecognized @ expression: '%s'", expression));
    }

    private void parseAndAdd(String cronExpression, String part, ChronoField chronoField) {
        this.parse(cronExpression, part, chronoField, Map.of());
    }

    private void parse(String cronExpression, String part, ChronoField chronoField, Map<String, Integer> names) {
        String[] split;
        if ("*".equals(part) || "?".equals(part)) {
            return;
        }
        ArrayList<Checker> checkers = new ArrayList<Checker>();
        String[] stringArray = split = part.split(",");
        int n = split.length;
        int n2 = 0;
        while (n2 < n) {
            String sub = stringArray[n2];
            checkers.add(this.parseSub(cronExpression, chronoField, sub, names));
            ++n2;
        }
        if (chronoField == ChronoField.YEAR) {
            checkers.add(CronAdjuster::checkMaxYear);
        }
        this.fields.add(new Field(chronoField, this.or(checkers)));
    }

    private Checker parseSub(String cronExpression, ChronoField chronoField, String sub, Map<String, Integer> names) {
        int min = (int)chronoField.range().getMinimum();
        int max = (int)chronoField.range().getMaximum();
        if (chronoField == ChronoField.DAY_OF_WEEK) {
            if ("L".equals(sub)) {
                return this.parseSub(cronExpression, chronoField, "SUN", names);
            }
            Matcher m = WEEKDAY_PATTERN.matcher(sub);
            if (m.matches()) {
                int day = this.parseDayOfWeek(cronExpression, m.group("day"), names);
                Checker c = temporal -> temporal.get(ChronoField.DAY_OF_WEEK) == day;
                if (m.group("nr") != null) {
                    int n = this.parseInt(cronExpression, chronoField, m.group("nr"));
                    return this.and(c, temporal -> CronAdjuster.isNthWeekDayInMonth(temporal, n));
                }
                if (m.group("l") != null) {
                    return this.and(c, CronAdjuster::isLastOfThisWeekDayInMonth);
                }
                return c;
            }
        } else if (chronoField == ChronoField.DAY_OF_MONTH) {
            if ("L".equals(sub)) {
                return CronAdjuster::isLastDayInMonth;
            }
            if ("LW".equals(sub) || "WL".equals(sub)) {
                return CronAdjuster::isLastWorkingDayInMonth;
            }
            if (sub.endsWith("W")) {
                int n = this.parseInt(cronExpression, chronoField, sub.substring(0, sub.length() - 1));
                return temporal -> CronAdjuster.isNearestWorkDay(temporal, n);
            }
        }
        String[] increments = sub.split("/");
        int[] range = this.parseRange(cronExpression, chronoField, increments[0], min, max, names);
        if (increments.length == 2) {
            int increment = this.parseInt(cronExpression, chronoField, increments[1]);
            if (range[0] == range[1]) {
                range[1] = max;
            }
            if (range[0] > range[1]) {
                return temporal -> {
                    int n2 = temporal.get(chronoField);
                    return (n2 >= range[0] || n2 <= range[1]) && (n2 - range[0]) % increment == 0;
                };
            }
            return temporal -> {
                int n2 = temporal.get(chronoField);
                return n2 >= range[0] && n2 <= range[1] && (n2 - range[0]) % increment == 0;
            };
        }
        if (range[0] > range[1]) {
            return temporal -> {
                int n = temporal.get(chronoField);
                return n >= range[0] || n <= range[1];
            };
        }
        return temporal -> {
            int n = temporal.get(chronoField);
            return n >= range[0] && n <= range[1];
        };
    }

    private static boolean isNthWeekDayInMonth(Temporal temporal, int nDayInMonth) {
        int day = temporal.get(ChronoField.DAY_OF_MONTH);
        int occurrences = 1 + (day - 1) / 7;
        return nDayInMonth == occurrences;
    }

    private static boolean isLastOfThisWeekDayInMonth(Temporal temporal) {
        int max;
        int day = temporal.get(ChronoField.DAY_OF_MONTH);
        return day + 7 > (max = (int)ChronoField.DAY_OF_MONTH.rangeRefinedBy(temporal).getMaximum());
    }

    private static boolean isLastDayInMonth(Temporal temporal) {
        int max;
        int day = temporal.get(ChronoField.DAY_OF_MONTH);
        return day == (max = (int)ChronoField.DAY_OF_MONTH.rangeRefinedBy(temporal).getMaximum());
    }

    private static boolean isLastWorkingDayInMonth(Temporal temporal) {
        int day = temporal.get(ChronoField.DAY_OF_MONTH);
        DayOfWeek type = DayOfWeek.of(temporal.get(ChronoField.DAY_OF_WEEK));
        int max = (int)ChronoField.DAY_OF_MONTH.rangeRefinedBy(temporal).getMaximum();
        switch (type) {
            case MONDAY: 
            case TUESDAY: 
            case WEDNESDAY: 
            case THURSDAY: {
                return day == max;
            }
            case FRIDAY: {
                return day + 2 >= max;
            }
        }
        return false;
    }

    static boolean isNearestWorkDay(Temporal temporal, int target) {
        int day = temporal.get(ChronoField.DAY_OF_MONTH);
        DayOfWeek type = DayOfWeek.of(temporal.get(ChronoField.DAY_OF_WEEK));
        switch (type) {
            case MONDAY: {
                return day == target || day == target + 1 || day == target + 2 && day == 3;
            }
            case TUESDAY: 
            case WEDNESDAY: 
            case THURSDAY: {
                return day == target;
            }
            case FRIDAY: {
                return day == target || day + 1 == target;
            }
        }
        return false;
    }

    private static boolean checkMaxYear(Temporal temporal) {
        return temporal.get(ChronoField.YEAR) >= 2200;
    }

    private int[] parseRange(String cronExpression, ChronoField chronoField, String range, int min, int max, Map<String, Integer> names) {
        int[] r = new int[]{min, max};
        if ("*".equals(range)) {
            return r;
        }
        String[] parts = range.split("-");
        r[0] = r[1] = this.parseInt(cronExpression, chronoField, parts[0], min, names);
        if (parts.length == 2) {
            r[1] = this.parseInt(cronExpression, chronoField, parts[1], min, names);
        }
        if (r[0] < min) {
            throw new IllegalArgumentException(String.format("Value too small in range in cron expression '%s' in field '%s': value %s, minimum: %s", cronExpression, chronoField, r[0], min));
        }
        if (r[1] > max) {
            throw new IllegalArgumentException(String.format("Value too high in range in cron expression '%s' in field '%s': value %s, minimum: %s", cronExpression, chronoField, r[1], max));
        }
        return r;
    }

    private int parseDayOfWeek(String cronExpression, String value, Map<String, Integer> names) {
        Integer nameIndex = names.get(value);
        if (nameIndex == null) {
            int dayOfWeek = this.parseInt(cronExpression, ChronoField.DAY_OF_WEEK, value) - 1;
            if (dayOfWeek < 0 || dayOfWeek > 6) {
                throw new IllegalArgumentException(String.format("Day of week in cron expression '%s' in field '%s': value %s is outside range", cronExpression, ChronoField.DAY_OF_WEEK, dayOfWeek));
            }
            return dayOfWeek == 0 ? 7 : dayOfWeek;
        }
        return nameIndex;
    }

    private int parseInt(String cronExpression, ChronoField chronoField, String name, int min, Map<String, Integer> names) {
        if (name.isEmpty()) {
            return 0;
        }
        Integer nameIndex = names.get(name);
        if (nameIndex == null) {
            return this.parseInt(cronExpression, chronoField, name);
        }
        return min + nameIndex - (chronoField == ChronoField.DAY_OF_WEEK ? 1 : 0);
    }

    private int parseInt(String cronExpression, ChronoField chronoField, String value) {
        try {
            return Integer.parseInt(value);
        }
        catch (NumberFormatException e) {
            throw new IllegalArgumentException(String.format("Value not a number in cron expression '%s' in field '%s': %s", cronExpression, chronoField, e.getMessage()));
        }
    }

    @Override
    public Temporal adjustInto(@Nullable Temporal temporal) {
        Temporal ret = temporal.plus(1L, ChronoUnit.SECONDS);
        int index = 0;
        int restarts = 0;
        int length = this.fields.size();
        while (index < length) {
            Field field = this.fields.get(index);
            Temporal out = field.isOk(ret);
            if (out == null) {
                ++index;
                continue;
            }
            if (restarts++ > 1000) {
                throw new DateTimeException("Conditions not satisfied.");
            }
            ret = out;
            index = 0;
        }
        return ret;
    }

    private Checker or(List<Checker> checkers) {
        return checkers.size() > 1 ? temporal -> checkers.stream().anyMatch(c -> c.matches(temporal)) : checkers.get(0);
    }

    private Checker and(Checker a, Checker b) {
        return temporal -> a.matches(temporal) && b.matches(temporal);
    }

    @FunctionalInterface
    static interface Checker {
        public boolean matches(Temporal var1);
    }

    private static class Field {
        final ChronoField type;
        final Checker checker;

        public Field(ChronoField type, Checker checker) {
            this.type = type;
            this.checker = checker;
        }

        @Nullable Temporal isOk(Temporal t) {
            if (this.checker.matches(t)) {
                return null;
            }
            Temporal out = t.plus(1L, this.type.getBaseUnit());
            switch (this.type) {
                case YEAR: {
                    out = out.with(ChronoField.MONTH_OF_YEAR, 1L);
                }
                case MONTH_OF_YEAR: {
                    out = out.with(ChronoField.DAY_OF_MONTH, 1L);
                }
                case DAY_OF_WEEK: 
                case DAY_OF_MONTH: {
                    out = out.with(ChronoField.HOUR_OF_DAY, 0L);
                }
                case HOUR_OF_DAY: {
                    out = out.with(ChronoField.MINUTE_OF_HOUR, 0L);
                }
                case MINUTE_OF_HOUR: {
                    out = out.with(ChronoField.SECOND_OF_MINUTE, 0L);
                }
                case SECOND_OF_MINUTE: {
                    return out;
                }
            }
            throw new IllegalArgumentException("Invalid field type " + String.valueOf(this.type));
        }

        public String toString() {
            return "Field [type=" + String.valueOf(this.type) + "]";
        }
    }
}

