/*
 * Decompiled with CFR 0.152.
 */
package io.quarkus.deployment.util;

import io.smallrye.config.SmallRyeConfig;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import org.apache.commons.lang3.SystemUtils;
import org.eclipse.microprofile.config.ConfigProvider;
import org.jboss.logging.Logger;

public final class ContainerRuntimeUtil {
    private static final Logger log = Logger.getLogger(ContainerRuntimeUtil.class);
    private static final String CONTAINER_EXECUTABLE = ((SmallRyeConfig)ConfigProvider.getConfig().unwrap(SmallRyeConfig.class)).getOptionalValue("quarkus.native.container-runtime", String.class).orElse(null);
    private static final Pattern PODMAN_PATTERN = Pattern.compile("^podman(?:\\.exe)? version.*", 32);
    private static final String CONTAINER_RUNTIME_SYS_PROP = "quarkus-local-container-runtime";
    private static final int MAX_ANTICIPATED_CHARACTERS_IN_DOCKER_INFO = 3000;

    private ContainerRuntimeUtil() {
    }

    public static ContainerRuntime detectContainerRuntime() {
        return ContainerRuntimeUtil.detectContainerRuntime(true, new ContainerRuntime[0]);
    }

    public static ContainerRuntime detectContainerRuntime(ContainerRuntime ... orderToCheckRuntimes) {
        return ContainerRuntimeUtil.detectContainerRuntime(true, orderToCheckRuntimes);
    }

    public static ContainerRuntime detectContainerRuntime(boolean required, ContainerRuntime ... orderToCheckRuntimes) {
        return ContainerRuntimeUtil.detectContainerRuntime(required, orderToCheckRuntimes != null && orderToCheckRuntimes.length > 0 ? Arrays.asList(orderToCheckRuntimes) : List.of(ContainerRuntime.DOCKER, ContainerRuntime.PODMAN));
    }

    public static ContainerRuntime detectContainerRuntime(boolean required, List<ContainerRuntime> orderToCheckRuntimes) {
        ContainerRuntime containerRuntime = ContainerRuntimeUtil.loadContainerRuntimeFromSystemProperty();
        if (containerRuntime != null && orderToCheckRuntimes.contains((Object)containerRuntime)) {
            return containerRuntime;
        }
        ContainerRuntime containerRuntimeEnvironment = ContainerRuntimeUtil.getContainerRuntimeEnvironment(orderToCheckRuntimes);
        if (containerRuntimeEnvironment == ContainerRuntime.UNAVAILABLE) {
            ContainerRuntimeUtil.storeContainerRuntimeInSystemProperty(ContainerRuntime.UNAVAILABLE);
            if (required) {
                throw new IllegalStateException("No container runtime was found. Make sure you have either Docker or Podman installed in your environment.");
            }
            return ContainerRuntime.UNAVAILABLE;
        }
        containerRuntime = ContainerRuntimeUtil.fullyResolveContainerRuntime(containerRuntimeEnvironment);
        ContainerRuntimeUtil.storeContainerRuntimeInSystemProperty(containerRuntime);
        return containerRuntime;
    }

    private static ContainerRuntime getContainerRuntimeEnvironment(List<ContainerRuntime> orderToCheckRuntimes) {
        ArrayList runtimesToCheck = new ArrayList(orderToCheckRuntimes.stream().distinct().toList());
        runtimesToCheck.retainAll(List.of(ContainerRuntime.DOCKER, ContainerRuntime.PODMAN));
        if (CONTAINER_EXECUTABLE != null) {
            Optional<ContainerRuntime> runtime = runtimesToCheck.stream().filter(containerRuntime -> CONTAINER_EXECUTABLE.trim().equalsIgnoreCase(containerRuntime.getExecutableName())).findFirst().filter(r -> {
                String versionOutput = ContainerRuntimeUtil.getVersionOutputFor(r);
                return switch (r.ordinal()) {
                    case 0, 1 -> versionOutput.contains("Docker version");
                    case 4, 5 -> PODMAN_PATTERN.matcher(versionOutput).matches();
                    default -> false;
                };
            });
            if (runtime.isPresent()) {
                return runtime.get();
            }
            log.warn((Object)"quarkus.native.container-runtime config property must be set to either podman or docker and the executable must be available. Ignoring it.");
        }
        for (ContainerRuntime runtime : runtimesToCheck) {
            String versionOutput = ContainerRuntimeUtil.getVersionOutputFor(runtime);
            switch (runtime.ordinal()) {
                case 0: 
                case 1: {
                    boolean dockerAvailable = versionOutput.contains("Docker version");
                    if (!dockerAvailable) break;
                    return PODMAN_PATTERN.matcher(versionOutput).matches() ? ContainerRuntime.PODMAN : ContainerRuntime.DOCKER;
                }
                case 4: 
                case 5: {
                    if (!PODMAN_PATTERN.matcher(versionOutput).matches()) break;
                    return ContainerRuntime.PODMAN;
                }
            }
        }
        return ContainerRuntime.UNAVAILABLE;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private static ContainerRuntime fullyResolveContainerRuntime(ContainerRuntime containerRuntimeEnvironment) {
        ContainerRuntime containerRuntime;
        boolean rootless = false;
        boolean isInWindowsWSL = false;
        Process rootlessProcess = null;
        ProcessBuilder pb = null;
        try {
            pb = new ProcessBuilder(containerRuntimeEnvironment.getExecutableName(), "info").redirectErrorStream(true);
            rootlessProcess = pb.start();
            int exitCode = rootlessProcess.waitFor();
            if (exitCode != 0) {
                log.warnf("Command \"%s\" exited with error code %d. Rootless container runtime detection might not be reliable or the container service is not running at all.", (Object)String.join((CharSequence)" ", pb.command()), (Object)exitCode);
            }
            try {
                BufferedReader bufferedReader;
                block33: {
                    InputStream inputStream = rootlessProcess.getInputStream();
                    InputStreamReader inputStreamReader = new InputStreamReader(inputStream);
                    bufferedReader = new BufferedReader(inputStreamReader);
                    bufferedReader.mark(3000);
                    if (exitCode == 0) break block33;
                    log.debugf("Command \"%s\" output: %s", (Object)String.join((CharSequence)" ", pb.command()), (Object)bufferedReader.lines().collect(Collectors.joining(System.lineSeparator())));
                    ContainerRuntime containerRuntime2 = ContainerRuntime.UNAVAILABLE;
                    bufferedReader.close();
                    inputStreamReader.close();
                    if (inputStream == null) return containerRuntime2;
                    inputStream.close();
                    return containerRuntime2;
                }
                try {
                    Predicate<String> stringPredicate = containerRuntimeEnvironment.isDocker() ? line -> line.trim().equals("rootless") || line.contains("Docker Desktop") || line.contains("desktop-linux") : line -> line.trim().equals("rootless: true");
                    rootless = bufferedReader.lines().anyMatch(stringPredicate);
                    if (SystemUtils.IS_OS_LINUX && containerRuntimeEnvironment.isDocker()) {
                        stringPredicate = line -> line.trim().contains("WSL");
                        bufferedReader.reset();
                        isInWindowsWSL = bufferedReader.lines().anyMatch(stringPredicate);
                    }
                }
                finally {
                    bufferedReader.close();
                }
            }
            catch (Exception ex) {
                log.debugf((Throwable)ex, "Failure to read info output from %s", (Object)String.join((CharSequence)" ", pb.command()));
                ContainerRuntime containerRuntime3 = ContainerRuntime.UNAVAILABLE;
                if (rootlessProcess == null) return containerRuntime3;
                rootlessProcess.destroy();
                return containerRuntime3;
            }
        }
        catch (IOException | InterruptedException e) {
            log.debugf((Throwable)e, "Failure to execute %s", (Object)String.join((CharSequence)" ", pb.command()));
            ContainerRuntime containerRuntime4 = ContainerRuntime.UNAVAILABLE;
            return containerRuntime4;
        }
        finally {
            if (rootlessProcess != null) {
                rootlessProcess.destroy();
            }
        }
        if (!rootless) {
            if (!isInWindowsWSL) return containerRuntimeEnvironment;
            return ContainerRuntime.WSL;
        }
        if (isInWindowsWSL) {
            return ContainerRuntime.WSL_ROOTLESS;
        }
        if (containerRuntimeEnvironment == ContainerRuntime.DOCKER) {
            containerRuntime = ContainerRuntime.DOCKER_ROOTLESS;
            return containerRuntime;
        }
        containerRuntime = ContainerRuntime.PODMAN_ROOTLESS;
        return containerRuntime;
    }

    private static ContainerRuntime loadContainerRuntimeFromSystemProperty() {
        String runtime = System.getProperty(CONTAINER_RUNTIME_SYS_PROP);
        if (runtime == null) {
            return null;
        }
        ContainerRuntime containerRuntime = ContainerRuntime.of(runtime);
        if (containerRuntime == null) {
            log.warnf("System property %s contains an unknown value %s. Ignoring it.", (Object)CONTAINER_RUNTIME_SYS_PROP, (Object)runtime);
        }
        return containerRuntime;
    }

    private static void storeContainerRuntimeInSystemProperty(ContainerRuntime containerRuntime) {
        System.setProperty(CONTAINER_RUNTIME_SYS_PROP, containerRuntime.name());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private static String getVersionOutputFor(ContainerRuntime containerRuntime) {
        Process versionProcess = null;
        try {
            ProcessBuilder pb = new ProcessBuilder(containerRuntime.getExecutableName(), "--version").redirectErrorStream(true);
            versionProcess = pb.start();
            int timeoutS = 10;
            boolean processExitedBeforeWaitTimeElapsed = versionProcess.waitFor(10L, TimeUnit.SECONDS);
            int exitStatus = versionProcess.exitValue();
            if (exitStatus == 0 && processExitedBeforeWaitTimeElapsed) {
                String string = new String(versionProcess.getInputStream().readAllBytes(), StandardCharsets.UTF_8);
                return string;
            }
            if (exitStatus != 0) {
                log.debugf("Failure. Exit status (%d) for command %s was not 0.", exitStatus, (Object)containerRuntime.getExecutableName());
            } else {
                log.debugf("Failure. It took command %s more than %d seconds to execute.", (Object)containerRuntime.getExecutableName(), (Object)10);
            }
            String string = "";
            return string;
        }
        catch (IOException | InterruptedException e) {
            log.debugf((Throwable)e, "Failure to read version output from %s", (Object)containerRuntime.getExecutableName());
            String string = "";
            return string;
        }
        finally {
            if (versionProcess != null) {
                versionProcess.destroy();
            }
        }
    }

    public static enum ContainerRuntime {
        DOCKER("docker", false),
        DOCKER_ROOTLESS("docker", true),
        WSL("docker", false),
        WSL_ROOTLESS("docker", false),
        PODMAN("podman", false),
        PODMAN_ROOTLESS("podman", true),
        UNAVAILABLE(null, false);

        private final String executableName;
        private final boolean rootless;

        private ContainerRuntime(String executableName, boolean rootless) {
            this.executableName = executableName;
            this.rootless = rootless;
        }

        public String getExecutableName() {
            if (this == UNAVAILABLE) {
                throw new IllegalStateException("Cannot get an executable name when no container runtime is available");
            }
            return this.executableName;
        }

        public boolean isDocker() {
            return this.executableName.equals("docker");
        }

        public boolean isPodman() {
            return this == PODMAN || this == PODMAN_ROOTLESS;
        }

        public boolean isInWindowsWSL() {
            return this == WSL || this == WSL_ROOTLESS;
        }

        public boolean isRootless() {
            return this.rootless;
        }

        public static ContainerRuntime of(String value) {
            for (ContainerRuntime containerRuntime : ContainerRuntime.values()) {
                if (!containerRuntime.name().equalsIgnoreCase(value)) continue;
                return containerRuntime;
            }
            return null;
        }
    }
}

