/* Copyright (c) 2013-2022 Jeffrey Pfau
 *
 * This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */
#include "util/test/suite.h"

#include <mgba-util/crc32.h>
#include <mgba-util/md5.h>
#include <mgba-util/sha1.h>

M_TEST_DEFINE(emptyCrc32) {
	uint8_t buffer[1] = {0};
	assert_int_equal(doCrc32(buffer, 0), 0);
}

M_TEST_DEFINE(newlineCrc32) {
uint8_t buffer[1] = { '\n' };
	assert_int_equal(doCrc32(buffer, 1), 0x32D70693);
}

M_TEST_DEFINE(helloWorldCrc32) {
	const char* buffer = "Hello, world!";
	assert_int_equal(doCrc32(buffer, strlen(buffer)), 0xEBE6C6E6);
}

#ifndef HAVE_CRC32
M_TEST_DEFINE(stagedCrc32) {
	uint8_t buffer[1] = { '\n\n' };
	assert_int_equal(crc32(0, buffer, 1), 0x32D70693);
	assert_int_equal(crc32(0, buffer, 2), 0x09304EBD);
	assert_int_equal(crc32(0x32D70693, buffer, 1), 0x09304EBD);
}
#endif

M_TEST_DEFINE(emptyMd5) {
	uint8_t buffer[1] = {0};
	uint8_t digest[0x10] = {0};
	md5Buffer(buffer, 0, digest);
	assert_memory_equal(digest, ((uint8_t[]) {
		0xD4, 0x1D, 0x8C, 0xD9, 0x8F, 0x00, 0xB2, 0x04,
		0xE9, 0x80, 0x09, 0x98, 0xEC, 0xF8, 0x42, 0x7E
	}), 16);
}

M_TEST_DEFINE(newlineMd5) {
	uint8_t buffer[1] = { '\n' };
	uint8_t digest[0x10] = {0};
	md5Buffer(buffer, 1, digest);
	assert_memory_equal(digest, ((uint8_t[]) {
		0x68, 0xB3, 0x29, 0xDA, 0x98, 0x93, 0xE3, 0x40,
		0x99, 0xC7, 0xD8, 0xAD, 0x5C, 0xB9, 0xC9, 0x40
	}), 16);
}

M_TEST_DEFINE(fullBlockMd5) {
	uint8_t buffer[56] = {
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	};
	uint8_t digest[0x10] = {0};
	md5Buffer(buffer, 56, digest);
	assert_memory_equal(digest, ((uint8_t[]) {
		0xA3, 0x31, 0x42, 0x53, 0x78, 0x54, 0xFE, 0xE2,
		0xAF, 0xD6, 0xCF, 0xF4, 0xC5, 0xA1, 0xDD, 0x39
	}), 16);
}

M_TEST_DEFINE(overflowBlockMd5) {
	uint8_t buffer[57] = {
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x0a,
	};
	uint8_t digest[0x10] = {0};
	md5Buffer(buffer, 57, digest);
	assert_memory_equal(digest, ((uint8_t[]) {
		0xBA, 0x49, 0x77, 0x29, 0x15, 0x5B, 0x13, 0x5D,
		0xBA, 0x27, 0xF3, 0xD8, 0x53, 0xCF, 0xD2, 0x1A
	}), 16);
}

M_TEST_DEFINE(twoBlockMd5) {
	uint8_t buffer[120] = {
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	};
	uint8_t digest[0x10] = {0};
	md5Buffer(buffer, 120, digest);
	assert_memory_equal(digest, ((uint8_t[]) {
		0xB5, 0x68, 0xA7, 0x7E, 0xD4, 0xC2, 0x39, 0xFB,
		0x4B, 0x74, 0xD7, 0x5B, 0xDB, 0xFD, 0x94, 0x93
	}), 16);
}

M_TEST_DEFINE(emptySha1) {
	uint8_t buffer[1] = {0};
	uint8_t digest[20] = {0};
	sha1Buffer(buffer, 0, digest);
	assert_memory_equal(digest, ((uint8_t[]) {
		0xDA, 0x39, 0xA3, 0xEE, 0x5E, 0x6B, 0x4B, 0x0D, 0x32, 0x55,
		0xBF, 0xEF, 0x95, 0x60, 0x18, 0x90, 0xAF, 0xD8, 0x07, 0x09
	}), 16);
}

M_TEST_DEFINE(newlineSha1) {
	uint8_t buffer[1] = { '\n' };
	uint8_t digest[20] = {0};
	sha1Buffer(buffer, 1, digest);
	assert_memory_equal(digest, ((uint8_t[]) {
		0xAD, 0xC8, 0x3B, 0x19, 0xE7, 0x93, 0x49, 0x1B, 0x1C, 0x6E,
		0xA0, 0xFD, 0x8B, 0x46, 0xCD, 0x9F, 0x32, 0xE5, 0x92, 0xFC
	}), 20);
}

M_TEST_DEFINE(fullBlockSha1) {
	uint8_t buffer[64] = {
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	};
	uint8_t digest[20] = {0};
	sha1Buffer(buffer, 64, digest);
	assert_memory_equal(digest, ((uint8_t[]) {
		0xCB, 0x4D, 0xD3, 0xDA, 0xCA, 0x2D, 0x6F, 0x25, 0x44, 0xBC,
		0x0D, 0xAA, 0x6B, 0xEB, 0xB7, 0x8A, 0xED, 0x0B, 0xD0, 0x34
	}), 20);
}

M_TEST_DEFINE(overflowBlockSha1) {
	uint8_t buffer[65] = {
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x0a,
	};
	uint8_t digest[20] = {0};
	sha1Buffer(buffer, 65, digest);
	assert_memory_equal(digest, ((uint8_t[]) {
		0xA3, 0x96, 0x68, 0x5E, 0xF7, 0x73, 0x87, 0x13, 0x2C, 0x43,
		0x64, 0x42, 0x2D, 0x16, 0x65, 0x39, 0x65, 0x6F, 0xB8, 0x93
	}), 20);
}

M_TEST_DEFINE(twoBlockSha1) {
	uint8_t buffer[128] = {
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
		0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20,
	};
	uint8_t digest[20] = {0};
	sha1Buffer(buffer, 128, digest);
	assert_memory_equal(digest, ((uint8_t[]) {
		0xFF, 0xB5, 0xE5, 0xD9, 0x6E, 0x19, 0x71, 0x4F, 0xFE, 0xF6,
		0x0A, 0xC8, 0x74, 0x9E, 0xCA, 0xEF, 0xBE, 0xC9, 0xD2, 0x95
	}), 20);
}

M_TEST_SUITE_DEFINE(Hashes,
	cmocka_unit_test(emptyCrc32),
	cmocka_unit_test(newlineCrc32),
	cmocka_unit_test(helloWorldCrc32),
#ifndef HAVE_CRC32
	cmocka_unit_test(stagedCrc32),
#endif
	cmocka_unit_test(emptyMd5),
	cmocka_unit_test(newlineMd5),
	cmocka_unit_test(fullBlockMd5),
	cmocka_unit_test(overflowBlockMd5),
	cmocka_unit_test(twoBlockMd5),
	cmocka_unit_test(emptySha1),
	cmocka_unit_test(newlineSha1),
	cmocka_unit_test(fullBlockSha1),
	cmocka_unit_test(overflowBlockSha1),
	cmocka_unit_test(twoBlockSha1),
)
