/**
 * @license
 * Copyright Google LLC All Rights Reserved.
 *
 * Use of this source code is governed by an MIT-style license that can be
 * found in the LICENSE file at https://angular.io/license
 */
// Below are constants for LView indices to help us look up LView members
// without having to remember the specific indices.
// Uglify will inline these when minifying so there shouldn't be a cost.
export const HOST = 0;
export const TVIEW = 1;
export const FLAGS = 2;
export const PARENT = 3;
export const NEXT = 4;
export const TRANSPLANTED_VIEWS_TO_REFRESH = 5;
export const T_HOST = 6;
export const CLEANUP = 7;
export const CONTEXT = 8;
export const INJECTOR = 9;
export const RENDERER_FACTORY = 10;
export const RENDERER = 11;
export const SANITIZER = 12;
export const CHILD_HEAD = 13;
export const CHILD_TAIL = 14;
// FIXME(misko): Investigate if the three declarations aren't all same thing.
export const DECLARATION_VIEW = 15;
export const DECLARATION_COMPONENT_VIEW = 16;
export const DECLARATION_LCONTAINER = 17;
export const PREORDER_HOOK_FLAGS = 18;
export const QUERIES = 19;
export const ID = 20;
export const EMBEDDED_VIEW_INJECTOR = 21;
/**
 * Size of LView's header. Necessary to adjust for it when setting slots.
 *
 * IMPORTANT: `HEADER_OFFSET` should only be referred to the in the `ɵɵ*` instructions to translate
 * instruction index into `LView` index. All other indexes should be in the `LView` index space and
 * there should be no need to refer to `HEADER_OFFSET` anywhere else.
 */
export const HEADER_OFFSET = 22;
/**
 * Converts `TViewType` into human readable text.
 * Make sure this matches with `TViewType`
 */
export const TViewTypeAsString = [
    'Root',
    'Component',
    'Embedded', // 2
];
// Note: This hack is necessary so we don't erroneously get a circular dependency
// failure based on types.
export const unusedValueExportToPlacateAjd = 1;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoidmlldy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uLy4uLy4uLy4uLy4uLy4uL3BhY2thZ2VzL2NvcmUvc3JjL3JlbmRlcjMvaW50ZXJmYWNlcy92aWV3LnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBOzs7Ozs7R0FNRztBQW1CSCx5RUFBeUU7QUFDekUsbURBQW1EO0FBQ25ELHdFQUF3RTtBQUN4RSxNQUFNLENBQUMsTUFBTSxJQUFJLEdBQUcsQ0FBQyxDQUFDO0FBQ3RCLE1BQU0sQ0FBQyxNQUFNLEtBQUssR0FBRyxDQUFDLENBQUM7QUFDdkIsTUFBTSxDQUFDLE1BQU0sS0FBSyxHQUFHLENBQUMsQ0FBQztBQUN2QixNQUFNLENBQUMsTUFBTSxNQUFNLEdBQUcsQ0FBQyxDQUFDO0FBQ3hCLE1BQU0sQ0FBQyxNQUFNLElBQUksR0FBRyxDQUFDLENBQUM7QUFDdEIsTUFBTSxDQUFDLE1BQU0sNkJBQTZCLEdBQUcsQ0FBQyxDQUFDO0FBQy9DLE1BQU0sQ0FBQyxNQUFNLE1BQU0sR0FBRyxDQUFDLENBQUM7QUFDeEIsTUFBTSxDQUFDLE1BQU0sT0FBTyxHQUFHLENBQUMsQ0FBQztBQUN6QixNQUFNLENBQUMsTUFBTSxPQUFPLEdBQUcsQ0FBQyxDQUFDO0FBQ3pCLE1BQU0sQ0FBQyxNQUFNLFFBQVEsR0FBRyxDQUFDLENBQUM7QUFDMUIsTUFBTSxDQUFDLE1BQU0sZ0JBQWdCLEdBQUcsRUFBRSxDQUFDO0FBQ25DLE1BQU0sQ0FBQyxNQUFNLFFBQVEsR0FBRyxFQUFFLENBQUM7QUFDM0IsTUFBTSxDQUFDLE1BQU0sU0FBUyxHQUFHLEVBQUUsQ0FBQztBQUM1QixNQUFNLENBQUMsTUFBTSxVQUFVLEdBQUcsRUFBRSxDQUFDO0FBQzdCLE1BQU0sQ0FBQyxNQUFNLFVBQVUsR0FBRyxFQUFFLENBQUM7QUFDN0IsNkVBQTZFO0FBQzdFLE1BQU0sQ0FBQyxNQUFNLGdCQUFnQixHQUFHLEVBQUUsQ0FBQztBQUNuQyxNQUFNLENBQUMsTUFBTSwwQkFBMEIsR0FBRyxFQUFFLENBQUM7QUFDN0MsTUFBTSxDQUFDLE1BQU0sc0JBQXNCLEdBQUcsRUFBRSxDQUFDO0FBQ3pDLE1BQU0sQ0FBQyxNQUFNLG1CQUFtQixHQUFHLEVBQUUsQ0FBQztBQUN0QyxNQUFNLENBQUMsTUFBTSxPQUFPLEdBQUcsRUFBRSxDQUFDO0FBQzFCLE1BQU0sQ0FBQyxNQUFNLEVBQUUsR0FBRyxFQUFFLENBQUM7QUFDckIsTUFBTSxDQUFDLE1BQU0sc0JBQXNCLEdBQUcsRUFBRSxDQUFDO0FBQ3pDOzs7Ozs7R0FNRztBQUNILE1BQU0sQ0FBQyxNQUFNLGFBQWEsR0FBRyxFQUFFLENBQUM7QUFnZGhDOzs7R0FHRztBQUNILE1BQU0sQ0FBQyxNQUFNLGlCQUFpQixHQUFHO0lBQy9CLE1BQU07SUFDTixXQUFXO0lBQ1gsVUFBVSxFQUFJLElBQUk7Q0FDVixDQUFDO0FBOFVYLGlGQUFpRjtBQUNqRiwwQkFBMEI7QUFDMUIsTUFBTSxDQUFDLE1BQU0sNkJBQTZCLEdBQUcsQ0FBQyxDQUFDIiwic291cmNlc0NvbnRlbnQiOlsiLyoqXG4gKiBAbGljZW5zZVxuICogQ29weXJpZ2h0IEdvb2dsZSBMTEMgQWxsIFJpZ2h0cyBSZXNlcnZlZC5cbiAqXG4gKiBVc2Ugb2YgdGhpcyBzb3VyY2UgY29kZSBpcyBnb3Zlcm5lZCBieSBhbiBNSVQtc3R5bGUgbGljZW5zZSB0aGF0IGNhbiBiZVxuICogZm91bmQgaW4gdGhlIExJQ0VOU0UgZmlsZSBhdCBodHRwczovL2FuZ3VsYXIuaW8vbGljZW5zZVxuICovXG5cbmltcG9ydCB7SW5qZWN0b3J9IGZyb20gJy4uLy4uL2RpL2luamVjdG9yJztcbmltcG9ydCB7UHJvdmlkZXJUb2tlbn0gZnJvbSAnLi4vLi4vZGkvcHJvdmlkZXJfdG9rZW4nO1xuaW1wb3J0IHtUeXBlfSBmcm9tICcuLi8uLi9pbnRlcmZhY2UvdHlwZSc7XG5pbXBvcnQge1NjaGVtYU1ldGFkYXRhfSBmcm9tICcuLi8uLi9tZXRhZGF0YS9zY2hlbWEnO1xuaW1wb3J0IHtTYW5pdGl6ZXJ9IGZyb20gJy4uLy4uL3Nhbml0aXphdGlvbi9zYW5pdGl6ZXInO1xuXG5pbXBvcnQge0xDb250YWluZXJ9IGZyb20gJy4vY29udGFpbmVyJztcbmltcG9ydCB7Q29tcG9uZW50RGVmLCBDb21wb25lbnRUZW1wbGF0ZSwgRGlyZWN0aXZlRGVmLCBEaXJlY3RpdmVEZWZMaXN0LCBIb3N0QmluZGluZ3NGdW5jdGlvbiwgUGlwZURlZiwgUGlwZURlZkxpc3QsIFZpZXdRdWVyaWVzRnVuY3Rpb259IGZyb20gJy4vZGVmaW5pdGlvbic7XG5pbXBvcnQge0kxOG5VcGRhdGVPcENvZGVzLCBUSTE4biwgVEljdX0gZnJvbSAnLi9pMThuJztcbmltcG9ydCB7VENvbnN0YW50cywgVE5vZGV9IGZyb20gJy4vbm9kZSc7XG5pbXBvcnQge0xRdWVyaWVzLCBUUXVlcmllc30gZnJvbSAnLi9xdWVyeSc7XG5pbXBvcnQge1JlbmRlcmVyLCBSZW5kZXJlckZhY3Rvcnl9IGZyb20gJy4vcmVuZGVyZXInO1xuaW1wb3J0IHtSQ29tbWVudCwgUkVsZW1lbnR9IGZyb20gJy4vcmVuZGVyZXJfZG9tJztcbmltcG9ydCB7VFN0eWxpbmdLZXksIFRTdHlsaW5nUmFuZ2V9IGZyb20gJy4vc3R5bGluZyc7XG5cblxuXG4vLyBCZWxvdyBhcmUgY29uc3RhbnRzIGZvciBMVmlldyBpbmRpY2VzIHRvIGhlbHAgdXMgbG9vayB1cCBMVmlldyBtZW1iZXJzXG4vLyB3aXRob3V0IGhhdmluZyB0byByZW1lbWJlciB0aGUgc3BlY2lmaWMgaW5kaWNlcy5cbi8vIFVnbGlmeSB3aWxsIGlubGluZSB0aGVzZSB3aGVuIG1pbmlmeWluZyBzbyB0aGVyZSBzaG91bGRuJ3QgYmUgYSBjb3N0LlxuZXhwb3J0IGNvbnN0IEhPU1QgPSAwO1xuZXhwb3J0IGNvbnN0IFRWSUVXID0gMTtcbmV4cG9ydCBjb25zdCBGTEFHUyA9IDI7XG5leHBvcnQgY29uc3QgUEFSRU5UID0gMztcbmV4cG9ydCBjb25zdCBORVhUID0gNDtcbmV4cG9ydCBjb25zdCBUUkFOU1BMQU5URURfVklFV1NfVE9fUkVGUkVTSCA9IDU7XG5leHBvcnQgY29uc3QgVF9IT1NUID0gNjtcbmV4cG9ydCBjb25zdCBDTEVBTlVQID0gNztcbmV4cG9ydCBjb25zdCBDT05URVhUID0gODtcbmV4cG9ydCBjb25zdCBJTkpFQ1RPUiA9IDk7XG5leHBvcnQgY29uc3QgUkVOREVSRVJfRkFDVE9SWSA9IDEwO1xuZXhwb3J0IGNvbnN0IFJFTkRFUkVSID0gMTE7XG5leHBvcnQgY29uc3QgU0FOSVRJWkVSID0gMTI7XG5leHBvcnQgY29uc3QgQ0hJTERfSEVBRCA9IDEzO1xuZXhwb3J0IGNvbnN0IENISUxEX1RBSUwgPSAxNDtcbi8vIEZJWE1FKG1pc2tvKTogSW52ZXN0aWdhdGUgaWYgdGhlIHRocmVlIGRlY2xhcmF0aW9ucyBhcmVuJ3QgYWxsIHNhbWUgdGhpbmcuXG5leHBvcnQgY29uc3QgREVDTEFSQVRJT05fVklFVyA9IDE1O1xuZXhwb3J0IGNvbnN0IERFQ0xBUkFUSU9OX0NPTVBPTkVOVF9WSUVXID0gMTY7XG5leHBvcnQgY29uc3QgREVDTEFSQVRJT05fTENPTlRBSU5FUiA9IDE3O1xuZXhwb3J0IGNvbnN0IFBSRU9SREVSX0hPT0tfRkxBR1MgPSAxODtcbmV4cG9ydCBjb25zdCBRVUVSSUVTID0gMTk7XG5leHBvcnQgY29uc3QgSUQgPSAyMDtcbmV4cG9ydCBjb25zdCBFTUJFRERFRF9WSUVXX0lOSkVDVE9SID0gMjE7XG4vKipcbiAqIFNpemUgb2YgTFZpZXcncyBoZWFkZXIuIE5lY2Vzc2FyeSB0byBhZGp1c3QgZm9yIGl0IHdoZW4gc2V0dGluZyBzbG90cy5cbiAqXG4gKiBJTVBPUlRBTlQ6IGBIRUFERVJfT0ZGU0VUYCBzaG91bGQgb25seSBiZSByZWZlcnJlZCB0byB0aGUgaW4gdGhlIGDJtcm1KmAgaW5zdHJ1Y3Rpb25zIHRvIHRyYW5zbGF0ZVxuICogaW5zdHJ1Y3Rpb24gaW5kZXggaW50byBgTFZpZXdgIGluZGV4LiBBbGwgb3RoZXIgaW5kZXhlcyBzaG91bGQgYmUgaW4gdGhlIGBMVmlld2AgaW5kZXggc3BhY2UgYW5kXG4gKiB0aGVyZSBzaG91bGQgYmUgbm8gbmVlZCB0byByZWZlciB0byBgSEVBREVSX09GRlNFVGAgYW55d2hlcmUgZWxzZS5cbiAqL1xuZXhwb3J0IGNvbnN0IEhFQURFUl9PRkZTRVQgPSAyMjtcblxuXG4vLyBUaGlzIGludGVyZmFjZSByZXBsYWNlcyB0aGUgcmVhbCBMVmlldyBpbnRlcmZhY2UgaWYgaXQgaXMgYW4gYXJnIG9yIGFcbi8vIHJldHVybiB2YWx1ZSBvZiBhIHB1YmxpYyBpbnN0cnVjdGlvbi4gVGhpcyBlbnN1cmVzIHdlIGRvbid0IG5lZWQgdG8gZXhwb3NlXG4vLyB0aGUgYWN0dWFsIGludGVyZmFjZSwgd2hpY2ggc2hvdWxkIGJlIGtlcHQgcHJpdmF0ZS5cbmV4cG9ydCBpbnRlcmZhY2UgT3BhcXVlVmlld1N0YXRlIHtcbiAgJ19fYnJhbmRfXyc6ICdCcmFuZCBmb3IgT3BhcXVlVmlld1N0YXRlIHRoYXQgbm90aGluZyB3aWxsIG1hdGNoJztcbn1cblxuXG4vKipcbiAqIGBMVmlld2Agc3RvcmVzIGFsbCBvZiB0aGUgaW5mb3JtYXRpb24gbmVlZGVkIHRvIHByb2Nlc3MgdGhlIGluc3RydWN0aW9ucyBhc1xuICogdGhleSBhcmUgaW52b2tlZCBmcm9tIHRoZSB0ZW1wbGF0ZS4gRWFjaCBlbWJlZGRlZCB2aWV3IGFuZCBjb21wb25lbnQgdmlldyBoYXMgaXRzXG4gKiBvd24gYExWaWV3YC4gV2hlbiBwcm9jZXNzaW5nIGEgcGFydGljdWxhciB2aWV3LCB3ZSBzZXQgdGhlIGB2aWV3RGF0YWAgdG8gdGhhdFxuICogYExWaWV3YC4gV2hlbiB0aGF0IHZpZXcgaXMgZG9uZSBwcm9jZXNzaW5nLCB0aGUgYHZpZXdEYXRhYCBpcyBzZXQgYmFjayB0b1xuICogd2hhdGV2ZXIgdGhlIG9yaWdpbmFsIGB2aWV3RGF0YWAgd2FzIGJlZm9yZSAodGhlIHBhcmVudCBgTFZpZXdgKS5cbiAqXG4gKiBLZWVwaW5nIHNlcGFyYXRlIHN0YXRlIGZvciBlYWNoIHZpZXcgZmFjaWxpdGllcyB2aWV3IGluc2VydGlvbiAvIGRlbGV0aW9uLCBzbyB3ZVxuICogZG9uJ3QgaGF2ZSB0byBlZGl0IHRoZSBkYXRhIGFycmF5IGJhc2VkIG9uIHdoaWNoIHZpZXdzIGFyZSBwcmVzZW50LlxuICovXG5leHBvcnQgaW50ZXJmYWNlIExWaWV3PFQgPSB1bmtub3duPiBleHRlbmRzIEFycmF5PGFueT4ge1xuICAvKipcbiAgICogSHVtYW4gcmVhZGFibGUgcmVwcmVzZW50YXRpb24gb2YgdGhlIGBMVmlld2AuXG4gICAqXG4gICAqIE5PVEU6IFRoaXMgcHJvcGVydHkgb25seSBleGlzdHMgaWYgYG5nRGV2TW9kZWAgaXMgc2V0IHRvIGB0cnVlYCBhbmQgaXQgaXMgbm90IHByZXNlbnQgaW5cbiAgICogcHJvZHVjdGlvbi4gSXRzIHByZXNlbmNlIGlzIHB1cmVseSB0byBoZWxwIGRlYnVnIGlzc3VlIGluIGRldmVsb3BtZW50LCBhbmQgc2hvdWxkIG5vdCBiZSByZWxpZWRcbiAgICogb24gaW4gcHJvZHVjdGlvbiBhcHBsaWNhdGlvbi5cbiAgICovXG4gIGRlYnVnPzogTFZpZXdEZWJ1ZztcblxuICAvKipcbiAgICogVGhlIG5vZGUgaW50byB3aGljaCB0aGlzIGBMVmlld2AgaXMgaW5zZXJ0ZWQuXG4gICAqL1xuICBbSE9TVF06IFJFbGVtZW50fG51bGw7XG5cbiAgLyoqXG4gICAqIFRoZSBzdGF0aWMgZGF0YSBmb3IgdGhpcyB2aWV3LiBXZSBuZWVkIGEgcmVmZXJlbmNlIHRvIHRoaXMgc28gd2UgY2FuIGVhc2lseSB3YWxrIHVwIHRoZVxuICAgKiBub2RlIHRyZWUgaW4gREkgYW5kIGdldCB0aGUgVFZpZXcuZGF0YSBhcnJheSBhc3NvY2lhdGVkIHdpdGggYSBub2RlICh3aGVyZSB0aGVcbiAgICogZGlyZWN0aXZlIGRlZnMgYXJlIHN0b3JlZCkuXG4gICAqL1xuICByZWFkb25seVtUVklFV106IFRWaWV3O1xuXG4gIC8qKiBGbGFncyBmb3IgdGhpcyB2aWV3LiBTZWUgTFZpZXdGbGFncyBmb3IgbW9yZSBpbmZvLiAqL1xuICBbRkxBR1NdOiBMVmlld0ZsYWdzO1xuXG4gIC8qKlxuICAgKiBUaGlzIG1heSBzdG9yZSBhbiB7QGxpbmsgTFZpZXd9IG9yIHtAbGluayBMQ29udGFpbmVyfS5cbiAgICpcbiAgICogYExWaWV3YCAtIFRoZSBwYXJlbnQgdmlldy4gVGhpcyBpcyBuZWVkZWQgd2hlbiB3ZSBleGl0IHRoZSB2aWV3IGFuZCBtdXN0IHJlc3RvcmUgdGhlIHByZXZpb3VzXG4gICAqIExWaWV3LiBXaXRob3V0IHRoaXMsIHRoZSByZW5kZXIgbWV0aG9kIHdvdWxkIGhhdmUgdG8ga2VlcCBhIHN0YWNrIG9mXG4gICAqIHZpZXdzIGFzIGl0IGlzIHJlY3Vyc2l2ZWx5IHJlbmRlcmluZyB0ZW1wbGF0ZXMuXG4gICAqXG4gICAqIGBMQ29udGFpbmVyYCAtIFRoZSBjdXJyZW50IHZpZXcgaXMgcGFydCBvZiBhIGNvbnRhaW5lciwgYW5kIGlzIGFuIGVtYmVkZGVkIHZpZXcuXG4gICAqL1xuICBbUEFSRU5UXTogTFZpZXd8TENvbnRhaW5lcnxudWxsO1xuXG4gIC8qKlxuICAgKlxuICAgKiBUaGUgbmV4dCBzaWJsaW5nIExWaWV3IG9yIExDb250YWluZXIuXG4gICAqXG4gICAqIEFsbG93cyB1cyB0byBwcm9wYWdhdGUgYmV0d2VlbiBzaWJsaW5nIHZpZXcgc3RhdGVzIHRoYXQgYXJlbid0IGluIHRoZSBzYW1lXG4gICAqIGNvbnRhaW5lci4gRW1iZWRkZWQgdmlld3MgYWxyZWFkeSBoYXZlIGEgbm9kZS5uZXh0LCBidXQgaXQgaXMgb25seSBzZXQgZm9yXG4gICAqIHZpZXdzIGluIHRoZSBzYW1lIGNvbnRhaW5lci4gV2UgbmVlZCBhIHdheSB0byBsaW5rIGNvbXBvbmVudCB2aWV3cyBhbmQgdmlld3NcbiAgICogYWNyb3NzIGNvbnRhaW5lcnMgYXMgd2VsbC5cbiAgICovXG4gIFtORVhUXTogTFZpZXd8TENvbnRhaW5lcnxudWxsO1xuXG4gIC8qKiBRdWVyaWVzIGFjdGl2ZSBmb3IgdGhpcyB2aWV3IC0gbm9kZXMgZnJvbSBhIHZpZXcgYXJlIHJlcG9ydGVkIHRvIHRob3NlIHF1ZXJpZXMuICovXG4gIFtRVUVSSUVTXTogTFF1ZXJpZXN8bnVsbDtcblxuICAvKipcbiAgICogU3RvcmUgdGhlIGBUTm9kZWAgb2YgdGhlIGxvY2F0aW9uIHdoZXJlIHRoZSBjdXJyZW50IGBMVmlld2AgaXMgaW5zZXJ0ZWQgaW50by5cbiAgICpcbiAgICogR2l2ZW46XG4gICAqIGBgYFxuICAgKiA8ZGl2PlxuICAgKiAgIDxuZy10ZW1wbGF0ZT48c3Bhbj48L3NwYW4+PC9uZy10ZW1wbGF0ZT5cbiAgICogPC9kaXY+XG4gICAqIGBgYFxuICAgKlxuICAgKiBXZSBlbmQgdXAgd2l0aCB0d28gYFRWaWV3YHMuXG4gICAqIC0gYHBhcmVudGAgYFRWaWV3YCB3aGljaCBjb250YWlucyBgPGRpdj48IS0tIGFuY2hvciAtLT48L2Rpdj5gXG4gICAqIC0gYGNoaWxkYCBgVFZpZXdgIHdoaWNoIGNvbnRhaW5zIGA8c3Bhbj48L3NwYW4+YFxuICAgKlxuICAgKiBUeXBpY2FsbHkgdGhlIGBjaGlsZGAgaXMgaW5zZXJ0ZWQgaW50byB0aGUgZGVjbGFyYXRpb24gbG9jYXRpb24gb2YgdGhlIGBwYXJlbnRgLCBidXQgaXQgY2FuIGJlXG4gICAqIGluc2VydGVkIGFueXdoZXJlLiBCZWNhdXNlIGl0IGNhbiBiZSBpbnNlcnRlZCBhbnl3aGVyZSBpdCBpcyBub3QgcG9zc2libGUgdG8gc3RvcmUgdGhlXG4gICAqIGluc2VydGlvbiBpbmZvcm1hdGlvbiBpbiB0aGUgYFRWaWV3YCBhbmQgaW5zdGVhZCB3ZSBtdXN0IHN0b3JlIGl0IGluIHRoZSBgTFZpZXdbVF9IT1NUXWAuXG4gICAqXG4gICAqIFNvIHRvIGRldGVybWluZSB3aGVyZSBpcyBvdXIgaW5zZXJ0aW9uIHBhcmVudCB3ZSB3b3VsZCBleGVjdXRlOlxuICAgKiBgYGBcbiAgICogY29uc3QgcGFyZW50TFZpZXcgPSBsVmlld1tQQVJFTlRdO1xuICAgKiBjb25zdCBwYXJlbnRUTm9kZSA9IGxWaWV3W1RfSE9TVF07XG4gICAqIGNvbnN0IGluc2VydGlvblBhcmVudCA9IHBhcmVudExWaWV3W3BhcmVudFROb2RlLmluZGV4XTtcbiAgICogYGBgXG4gICAqXG4gICAqXG4gICAqIElmIGBudWxsYCwgdGhpcyBpcyB0aGUgcm9vdCB2aWV3IG9mIGFuIGFwcGxpY2F0aW9uIChyb290IGNvbXBvbmVudCBpcyBpbiB0aGlzIHZpZXcpIGFuZCBpdCBoYXNcbiAgICogbm8gcGFyZW50cy5cbiAgICovXG4gIFtUX0hPU1RdOiBUTm9kZXxudWxsO1xuXG4gIC8qKlxuICAgKiBXaGVuIGEgdmlldyBpcyBkZXN0cm95ZWQsIGxpc3RlbmVycyBuZWVkIHRvIGJlIHJlbGVhc2VkIGFuZCBvdXRwdXRzIG5lZWQgdG8gYmVcbiAgICogdW5zdWJzY3JpYmVkLiBUaGlzIGNvbnRleHQgYXJyYXkgc3RvcmVzIGJvdGggbGlzdGVuZXIgZnVuY3Rpb25zIHdyYXBwZWQgd2l0aFxuICAgKiB0aGVpciBjb250ZXh0IGFuZCBvdXRwdXQgc3Vic2NyaXB0aW9uIGluc3RhbmNlcyBmb3IgYSBwYXJ0aWN1bGFyIHZpZXcuXG4gICAqXG4gICAqIFRoZXNlIGNoYW5nZSBwZXIgTFZpZXcgaW5zdGFuY2UsIHNvIHRoZXkgY2Fubm90IGJlIHN0b3JlZCBvbiBUVmlldy4gSW5zdGVhZCxcbiAgICogVFZpZXcuY2xlYW51cCBzYXZlcyBhbiBpbmRleCB0byB0aGUgbmVjZXNzYXJ5IGNvbnRleHQgaW4gdGhpcyBhcnJheS5cbiAgICpcbiAgICogQWZ0ZXIgYExWaWV3YCBpcyBjcmVhdGVkIGl0IGlzIHBvc3NpYmxlIHRvIGF0dGFjaCBhZGRpdGlvbmFsIGluc3RhbmNlIHNwZWNpZmljIGZ1bmN0aW9ucyBhdCB0aGVcbiAgICogZW5kIG9mIHRoZSBgbFZpZXdbQ0xFQU5VUF1gIGJlY2F1c2Ugd2Uga25vdyB0aGF0IG5vIG1vcmUgYFRgIGxldmVsIGNsZWFudXAgZnVuY3Rpb25zIHdpbGwgYmVcbiAgICogYWRkZWQgaGVyZS5cbiAgICovXG4gIFtDTEVBTlVQXTogYW55W118bnVsbDtcblxuICAvKipcbiAgICogLSBGb3IgZHluYW1pYyB2aWV3cywgdGhpcyBpcyB0aGUgY29udGV4dCB3aXRoIHdoaWNoIHRvIHJlbmRlciB0aGUgdGVtcGxhdGUgKGUuZy5cbiAgICogICBgTmdGb3JDb250ZXh0YCksIG9yIGB7fWAgaWYgbm90IGRlZmluZWQgZXhwbGljaXRseS5cbiAgICogLSBGb3Igcm9vdCB2aWV3IG9mIHRoZSByb290IGNvbXBvbmVudCBpdCdzIGEgcmVmZXJlbmNlIHRvIHRoZSBjb21wb25lbnQgaW5zdGFuY2UgaXRzZWxmLlxuICAgKiAtIEZvciBjb21wb25lbnRzLCB0aGUgY29udGV4dCBpcyBhIHJlZmVyZW5jZSB0byB0aGUgY29tcG9uZW50IGluc3RhbmNlIGl0c2VsZi5cbiAgICogLSBGb3IgaW5saW5lIHZpZXdzLCB0aGUgY29udGV4dCBpcyBudWxsLlxuICAgKi9cbiAgW0NPTlRFWFRdOiBUO1xuXG4gIC8qKiBBbiBvcHRpb25hbCBNb2R1bGUgSW5qZWN0b3IgdG8gYmUgdXNlZCBhcyBmYWxsIGJhY2sgYWZ0ZXIgRWxlbWVudCBJbmplY3RvcnMgYXJlIGNvbnN1bHRlZC4gKi9cbiAgcmVhZG9ubHlbSU5KRUNUT1JdOiBJbmplY3RvcnxudWxsO1xuXG4gIC8qKiBGYWN0b3J5IHRvIGJlIHVzZWQgZm9yIGNyZWF0aW5nIFJlbmRlcmVyLiAqL1xuICBbUkVOREVSRVJfRkFDVE9SWV06IFJlbmRlcmVyRmFjdG9yeTtcblxuICAvKiogUmVuZGVyZXIgdG8gYmUgdXNlZCBmb3IgdGhpcyB2aWV3LiAqL1xuICBbUkVOREVSRVJdOiBSZW5kZXJlcjtcblxuICAvKiogQW4gb3B0aW9uYWwgY3VzdG9tIHNhbml0aXplci4gKi9cbiAgW1NBTklUSVpFUl06IFNhbml0aXplcnxudWxsO1xuXG4gIC8qKlxuICAgKiBSZWZlcmVuY2UgdG8gdGhlIGZpcnN0IExWaWV3IG9yIExDb250YWluZXIgYmVuZWF0aCB0aGlzIExWaWV3IGluXG4gICAqIHRoZSBoaWVyYXJjaHkuXG4gICAqXG4gICAqIE5lY2Vzc2FyeSB0byBzdG9yZSB0aGlzIHNvIHZpZXdzIGNhbiB0cmF2ZXJzZSB0aHJvdWdoIHRoZWlyIG5lc3RlZCB2aWV3c1xuICAgKiB0byByZW1vdmUgbGlzdGVuZXJzIGFuZCBjYWxsIG9uRGVzdHJveSBjYWxsYmFja3MuXG4gICAqL1xuICBbQ0hJTERfSEVBRF06IExWaWV3fExDb250YWluZXJ8bnVsbDtcblxuICAvKipcbiAgICogVGhlIGxhc3QgTFZpZXcgb3IgTENvbnRhaW5lciBiZW5lYXRoIHRoaXMgTFZpZXcgaW4gdGhlIGhpZXJhcmNoeS5cbiAgICpcbiAgICogVGhlIHRhaWwgYWxsb3dzIHVzIHRvIHF1aWNrbHkgYWRkIGEgbmV3IHN0YXRlIHRvIHRoZSBlbmQgb2YgdGhlIHZpZXcgbGlzdFxuICAgKiB3aXRob3V0IGhhdmluZyB0byBwcm9wYWdhdGUgc3RhcnRpbmcgZnJvbSB0aGUgZmlyc3QgY2hpbGQuXG4gICAqL1xuICBbQ0hJTERfVEFJTF06IExWaWV3fExDb250YWluZXJ8bnVsbDtcblxuICAvKipcbiAgICogVmlldyB3aGVyZSB0aGlzIHZpZXcncyB0ZW1wbGF0ZSB3YXMgZGVjbGFyZWQuXG4gICAqXG4gICAqIFRoZSB0ZW1wbGF0ZSBmb3IgYSBkeW5hbWljYWxseSBjcmVhdGVkIHZpZXcgbWF5IGJlIGRlY2xhcmVkIGluIGEgZGlmZmVyZW50IHZpZXcgdGhhblxuICAgKiBpdCBpcyBpbnNlcnRlZC4gV2UgYWxyZWFkeSB0cmFjayB0aGUgXCJpbnNlcnRpb24gdmlld1wiICh2aWV3IHdoZXJlIHRoZSB0ZW1wbGF0ZSB3YXNcbiAgICogaW5zZXJ0ZWQpIGluIExWaWV3W1BBUkVOVF0sIGJ1dCB3ZSBhbHNvIG5lZWQgYWNjZXNzIHRvIHRoZSBcImRlY2xhcmF0aW9uIHZpZXdcIlxuICAgKiAodmlldyB3aGVyZSB0aGUgdGVtcGxhdGUgd2FzIGRlY2xhcmVkKS4gT3RoZXJ3aXNlLCB3ZSB3b3VsZG4ndCBiZSBhYmxlIHRvIGNhbGwgdGhlXG4gICAqIHZpZXcncyB0ZW1wbGF0ZSBmdW5jdGlvbiB3aXRoIHRoZSBwcm9wZXIgY29udGV4dHMuIENvbnRleHQgc2hvdWxkIGJlIGluaGVyaXRlZCBmcm9tXG4gICAqIHRoZSBkZWNsYXJhdGlvbiB2aWV3IHRyZWUsIG5vdCB0aGUgaW5zZXJ0aW9uIHZpZXcgdHJlZS5cbiAgICpcbiAgICogRXhhbXBsZSAoQXBwQ29tcG9uZW50IHRlbXBsYXRlKTpcbiAgICpcbiAgICogPG5nLXRlbXBsYXRlICNmb28+PC9uZy10ZW1wbGF0ZT4gICAgICAgPC0tIGRlY2xhcmVkIGhlcmUgLS0+XG4gICAqIDxzb21lLWNvbXAgW3RwbF09XCJmb29cIj48L3NvbWUtY29tcD4gICAgPC0tIGluc2VydGVkIGluc2lkZSB0aGlzIGNvbXBvbmVudCAtLT5cbiAgICpcbiAgICogVGhlIDxuZy10ZW1wbGF0ZT4gYWJvdmUgaXMgZGVjbGFyZWQgaW4gdGhlIEFwcENvbXBvbmVudCB0ZW1wbGF0ZSwgYnV0IGl0IHdpbGwgYmUgcGFzc2VkIGludG9cbiAgICogU29tZUNvbXAgYW5kIGluc2VydGVkIHRoZXJlLiBJbiB0aGlzIGNhc2UsIHRoZSBkZWNsYXJhdGlvbiB2aWV3IHdvdWxkIGJlIHRoZSBBcHBDb21wb25lbnQsXG4gICAqIGJ1dCB0aGUgaW5zZXJ0aW9uIHZpZXcgd291bGQgYmUgU29tZUNvbXAuIFdoZW4gd2UgYXJlIHJlbW92aW5nIHZpZXdzLCB3ZSB3b3VsZCB3YW50IHRvXG4gICAqIHRyYXZlcnNlIHRocm91Z2ggdGhlIGluc2VydGlvbiB2aWV3IHRvIGNsZWFuIHVwIGxpc3RlbmVycy4gV2hlbiB3ZSBhcmUgY2FsbGluZyB0aGVcbiAgICogdGVtcGxhdGUgZnVuY3Rpb24gZHVyaW5nIGNoYW5nZSBkZXRlY3Rpb24sIHdlIG5lZWQgdGhlIGRlY2xhcmF0aW9uIHZpZXcgdG8gZ2V0IGluaGVyaXRlZFxuICAgKiBjb250ZXh0LlxuICAgKi9cbiAgW0RFQ0xBUkFUSU9OX1ZJRVddOiBMVmlld3xudWxsO1xuXG5cbiAgLyoqXG4gICAqIFBvaW50cyB0byB0aGUgZGVjbGFyYXRpb24gY29tcG9uZW50IHZpZXcsIHVzZWQgdG8gdHJhY2sgdHJhbnNwbGFudGVkIGBMVmlld2BzLlxuICAgKlxuICAgKiBTZWU6IGBERUNMQVJBVElPTl9WSUVXYCB3aGljaCBwb2ludHMgdG8gdGhlIGFjdHVhbCBgTFZpZXdgIHdoZXJlIGl0IHdhcyBkZWNsYXJlZCwgd2hlcmVhc1xuICAgKiBgREVDTEFSQVRJT05fQ09NUE9ORU5UX1ZJRVdgIHBvaW50cyB0byB0aGUgY29tcG9uZW50IHdoaWNoIG1heSBub3QgYmUgc2FtZSBhc1xuICAgKiBgREVDTEFSQVRJT05fVklFV2AuXG4gICAqXG4gICAqIEV4YW1wbGU6XG4gICAqIGBgYFxuICAgKiA8I1ZJRVcgI215Q29tcD5cbiAgICogIDxkaXYgKm5nSWY9XCJ0cnVlXCI+XG4gICAqICAgPG5nLXRlbXBsYXRlICNteVRtcGw+Li4uPC9uZy10ZW1wbGF0ZT5cbiAgICogIDwvZGl2PlxuICAgKiA8LyNWSUVXPlxuICAgKiBgYGBcbiAgICogSW4gdGhlIGFib3ZlIGNhc2UgYERFQ0xBUkFUSU9OX1ZJRVdgIGZvciBgbXlUbXBsYCBwb2ludHMgdG8gdGhlIGBMVmlld2Agb2YgYG5nSWZgIHdoZXJlYXNcbiAgICogYERFQ0xBUkFUSU9OX0NPTVBPTkVOVF9WSUVXYCBwb2ludHMgdG8gYExWaWV3YCBvZiB0aGUgYG15Q29tcGAgd2hpY2ggb3ducyB0aGUgdGVtcGxhdGUuXG4gICAqXG4gICAqIFRoZSByZWFzb24gZm9yIHRoaXMgaXMgdGhhdCBhbGwgZW1iZWRkZWQgdmlld3MgYXJlIGFsd2F5cyBjaGVjay1hbHdheXMgd2hlcmVhcyB0aGUgY29tcG9uZW50XG4gICAqIHZpZXcgY2FuIGJlIGNoZWNrLWFsd2F5cyBvciBvbi1wdXNoLiBXaGVuIHdlIGhhdmUgYSB0cmFuc3BsYW50ZWQgdmlldyBpdCBpcyBpbXBvcnRhbnQgdG9cbiAgICogZGV0ZXJtaW5lIGlmIHdlIGhhdmUgdHJhbnNwbGFudGVkIGEgdmlldyBmcm9tIGNoZWNrLWFsd2F5cyBkZWNsYXJhdGlvbiB0byBvbi1wdXNoIGluc2VydGlvblxuICAgKiBwb2ludC4gSW4gc3VjaCBhIGNhc2UgdGhlIHRyYW5zcGxhbnRlZCB2aWV3IG5lZWRzIHRvIGJlIGFkZGVkIHRvIHRoZSBgTENvbnRhaW5lcmAgaW4gdGhlXG4gICAqIGRlY2xhcmVkIGBMVmlld2AgYW5kIENEIGR1cmluZyB0aGUgZGVjbGFyZWQgdmlldyBDRCAoaW4gYWRkaXRpb24gdG8gdGhlIENEIGF0IHRoZSBpbnNlcnRpb25cbiAgICogcG9pbnQuKSAoQW55IHRyYW5zcGxhbnRlZCB2aWV3cyB3aGljaCBhcmUgaW50cmEgQ29tcG9uZW50IGFyZSBvZiBubyBpbnRlcmVzdCBiZWNhdXNlIHRoZSBDRFxuICAgKiBzdHJhdGVneSBvZiBkZWNsYXJhdGlvbiBhbmQgaW5zZXJ0aW9uIHdpbGwgYWx3YXlzIGJlIHRoZSBzYW1lLCBiZWNhdXNlIGl0IGlzIHRoZSBzYW1lXG4gICAqIGNvbXBvbmVudC4pXG4gICAqXG4gICAqIFF1ZXJpZXMgYWxyZWFkeSB0cmFjayBtb3ZlZCB2aWV3cyBpbiBgTFZpZXdbREVDTEFSQVRJT05fTENPTlRBSU5FUl1gIGFuZFxuICAgKiBgTENvbnRhaW5lcltNT1ZFRF9WSUVXU11gLiBIb3dldmVyIHRoZSBxdWVyaWVzIGFsc28gdHJhY2sgYExWaWV3YHMgd2hpY2ggbW92ZWQgd2l0aGluIHRoZSBzYW1lXG4gICAqIGNvbXBvbmVudCBgTFZpZXdgLiBUcmFuc3BsYW50ZWQgdmlld3MgYXJlIGEgc3Vic2V0IG9mIG1vdmVkIHZpZXdzLCBhbmQgd2UgdXNlXG4gICAqIGBERUNMQVJBVElPTl9DT01QT05FTlRfVklFV2AgdG8gZGlmZmVyZW50aWF0ZSB0aGVtLiBBcyBpbiB0aGlzIGV4YW1wbGUuXG4gICAqXG4gICAqIEV4YW1wbGUgc2hvd2luZyBpbnRyYSBjb21wb25lbnQgYExWaWV3YCBtb3ZlbWVudC5cbiAgICogYGBgXG4gICAqIDwjVklFVyAjbXlDb21wPlxuICAgKiAgIDxkaXYgKm5nSWY9XCJjb25kaXRpb247IHRoZW4gdGhlbkJsb2NrIGVsc2UgZWxzZUJsb2NrXCI+PC9kaXY+XG4gICAqICAgPG5nLXRlbXBsYXRlICN0aGVuQmxvY2s+Q29udGVudCB0byByZW5kZXIgd2hlbiBjb25kaXRpb24gaXMgdHJ1ZS48L25nLXRlbXBsYXRlPlxuICAgKiAgIDxuZy10ZW1wbGF0ZSAjZWxzZUJsb2NrPkNvbnRlbnQgdG8gcmVuZGVyIHdoZW4gY29uZGl0aW9uIGlzIGZhbHNlLjwvbmctdGVtcGxhdGU+XG4gICAqIDwvI1ZJRVc+XG4gICAqIGBgYFxuICAgKiBUaGUgYHRoZW5CbG9ja2AgYW5kIGBlbHNlQmxvY2tgIGlzIG1vdmVkIGJ1dCBub3QgdHJhbnNwbGFudGVkLlxuICAgKlxuICAgKiBFeGFtcGxlIHNob3dpbmcgaW50ZXIgY29tcG9uZW50IGBMVmlld2AgbW92ZW1lbnQgKHRyYW5zcGxhbnRlZCB2aWV3KS5cbiAgICogYGBgXG4gICAqIDwjVklFVyAjbXlDb21wPlxuICAgKiAgIDxuZy10ZW1wbGF0ZSAjbXlUbXBsPi4uLjwvbmctdGVtcGxhdGU+XG4gICAqICAgPGluc2VydGlvbi1jb21wb25lbnQgW3RlbXBsYXRlXT1cIm15VG1wbFwiPjwvaW5zZXJ0aW9uLWNvbXBvbmVudD5cbiAgICogPC8jVklFVz5cbiAgICogYGBgXG4gICAqIEluIHRoZSBhYm92ZSBleGFtcGxlIGBteVRtcGxgIGlzIHBhc3NlZCBpbnRvIGEgZGlmZmVyZW50IGNvbXBvbmVudC4gSWYgYGluc2VydGlvbi1jb21wb25lbnRgXG4gICAqIGluc3RhbnRpYXRlcyBgbXlUbXBsYCBhbmQgYGluc2VydGlvbi1jb21wb25lbnRgIGlzIG9uLXB1c2ggdGhlbiB0aGUgYExDb250YWluZXJgIG5lZWRzIHRvIGJlXG4gICAqIG1hcmtlZCBhcyBjb250YWluaW5nIHRyYW5zcGxhbnRlZCB2aWV3cyBhbmQgdGhvc2Ugdmlld3MgbmVlZCB0byBiZSBDRCBhcyBwYXJ0IG9mIHRoZVxuICAgKiBkZWNsYXJhdGlvbiBDRC5cbiAgICpcbiAgICpcbiAgICogV2hlbiBjaGFuZ2UgZGV0ZWN0aW9uIHJ1bnMsIGl0IGl0ZXJhdGVzIG92ZXIgYFtNT1ZFRF9WSUVXU11gIGFuZCBDRHMgYW55IGNoaWxkIGBMVmlld2BzIHdoZXJlXG4gICAqIHRoZSBgREVDTEFSQVRJT05fQ09NUE9ORU5UX1ZJRVdgIG9mIHRoZSBjdXJyZW50IGNvbXBvbmVudCBhbmQgdGhlIGNoaWxkIGBMVmlld2AgZG9lcyBub3QgbWF0Y2hcbiAgICogKGl0IGhhcyBiZWVuIHRyYW5zcGxhbnRlZCBhY3Jvc3MgY29tcG9uZW50cy4pXG4gICAqXG4gICAqIE5vdGU6IGBbREVDTEFSQVRJT05fQ09NUE9ORU5UX1ZJRVddYCBwb2ludHMgdG8gaXRzZWxmIGlmIHRoZSBMVmlldyBpcyBhIGNvbXBvbmVudCB2aWV3ICh0aGVcbiAgICogICAgICAgc2ltcGxlc3QgLyBtb3N0IGNvbW1vbiBjYXNlKS5cbiAgICpcbiAgICogc2VlIGFsc286XG4gICAqICAgLSBodHRwczovL2hhY2ttZC5pby9AbWhldmVyeS9ySlVKc3Z2OUggd3JpdGUgdXAgb2YgdGhlIHByb2JsZW1cbiAgICogICAtIGBMQ29udGFpbmVyW0hBU19UUkFOU1BMQU5URURfVklFV1NdYCB3aGljaCBtYXJrcyB3aGljaCBgTENvbnRhaW5lcmAgaGFzIHRyYW5zcGxhbnRlZCB2aWV3cy5cbiAgICogICAtIGBMQ29udGFpbmVyW1RSQU5TUExBTlRfSEVBRF1gIGFuZCBgTENvbnRhaW5lcltUUkFOU1BMQU5UX1RBSUxdYCBzdG9yYWdlIGZvciB0cmFuc3BsYW50ZWRcbiAgICogICAtIGBMVmlld1tERUNMQVJBVElPTl9MQ09OVEFJTkVSXWAgc2ltaWxhciBwcm9ibGVtIGZvciBxdWVyaWVzXG4gICAqICAgLSBgTENvbnRhaW5lcltNT1ZFRF9WSUVXU11gIHNpbWlsYXIgcHJvYmxlbSBmb3IgcXVlcmllc1xuICAgKi9cbiAgW0RFQ0xBUkFUSU9OX0NPTVBPTkVOVF9WSUVXXTogTFZpZXc7XG5cbiAgLyoqXG4gICAqIEEgZGVjbGFyYXRpb24gcG9pbnQgb2YgZW1iZWRkZWQgdmlld3MgKG9uZXMgaW5zdGFudGlhdGVkIGJhc2VkIG9uIHRoZSBjb250ZW50IG9mIGFcbiAgICogPG5nLXRlbXBsYXRlPiksIG51bGwgZm9yIG90aGVyIHR5cGVzIG9mIHZpZXdzLlxuICAgKlxuICAgKiBXZSBuZWVkIHRvIHRyYWNrIGFsbCBlbWJlZGRlZCB2aWV3cyBjcmVhdGVkIGZyb20gYSBnaXZlbiBkZWNsYXJhdGlvbiBwb2ludCBzbyB3ZSBjYW4gcHJlcGFyZVxuICAgKiBxdWVyeSBtYXRjaGVzIGluIGEgcHJvcGVyIG9yZGVyIChxdWVyeSBtYXRjaGVzIGFyZSBvcmRlcmVkIGJhc2VkIG9uIHRoZWlyIGRlY2xhcmF0aW9uIHBvaW50IGFuZFxuICAgKiBfbm90XyB0aGUgaW5zZXJ0aW9uIHBvaW50KS5cbiAgICovXG4gIFtERUNMQVJBVElPTl9MQ09OVEFJTkVSXTogTENvbnRhaW5lcnxudWxsO1xuXG4gIC8qKlxuICAgKiBNb3JlIGZsYWdzIGZvciB0aGlzIHZpZXcuIFNlZSBQcmVPcmRlckhvb2tGbGFncyBmb3IgbW9yZSBpbmZvLlxuICAgKi9cbiAgW1BSRU9SREVSX0hPT0tfRkxBR1NdOiBQcmVPcmRlckhvb2tGbGFncztcblxuICAvKipcbiAgICogVGhlIG51bWJlciBvZiBkaXJlY3QgdHJhbnNwbGFudGVkIHZpZXdzIHdoaWNoIG5lZWQgYSByZWZyZXNoIG9yIGhhdmUgZGVzY2VuZGFudHMgdGhlbXNlbHZlc1xuICAgKiB0aGF0IG5lZWQgYSByZWZyZXNoIGJ1dCBoYXZlIG5vdCBtYXJrZWQgdGhlaXIgYW5jZXN0b3JzIGFzIERpcnR5LiBUaGlzIHRlbGxzIHVzIHRoYXQgZHVyaW5nXG4gICAqIGNoYW5nZSBkZXRlY3Rpb24gd2Ugc2hvdWxkIHN0aWxsIGRlc2NlbmQgdG8gZmluZCB0aG9zZSBjaGlsZHJlbiB0byByZWZyZXNoLCBldmVuIGlmIHRoZSBwYXJlbnRzXG4gICAqIGFyZSBub3QgYERpcnR5YC9gQ2hlY2tBbHdheXNgLlxuICAgKi9cbiAgW1RSQU5TUExBTlRFRF9WSUVXU19UT19SRUZSRVNIXTogbnVtYmVyO1xuXG4gIC8qKiBVbmlxdWUgSUQgb2YgdGhlIHZpZXcuIFVzZWQgZm9yIGBfX25nQ29udGV4dF9fYCBsb29rdXBzIGluIHRoZSBgTFZpZXdgIHJlZ2lzdHJ5LiAqL1xuICBbSURdOiBudW1iZXI7XG5cbiAgLyoqXG4gICAqIE9wdGlvbmFsIGluamVjdG9yIGFzc2lnbmVkIHRvIGVtYmVkZGVkIHZpZXdzIHRoYXQgdGFrZXNcbiAgICogcHJlY2VkZW5jZSBvdmVyIHRoZSBlbGVtZW50IGFuZCBtb2R1bGUgaW5qZWN0b3JzLlxuICAgKi9cbiAgcmVhZG9ubHlbRU1CRURERURfVklFV19JTkpFQ1RPUl06IEluamVjdG9yfG51bGw7XG59XG5cbi8qKiBGbGFncyBhc3NvY2lhdGVkIHdpdGggYW4gTFZpZXcgKHNhdmVkIGluIExWaWV3W0ZMQUdTXSkgKi9cbmV4cG9ydCBjb25zdCBlbnVtIExWaWV3RmxhZ3Mge1xuICAvKiogVGhlIHN0YXRlIG9mIHRoZSBpbml0IHBoYXNlIG9uIHRoZSBmaXJzdCAyIGJpdHMgKi9cbiAgSW5pdFBoYXNlU3RhdGVJbmNyZW1lbnRlciA9IDBiMDAwMDAwMDAwMDEsXG4gIEluaXRQaGFzZVN0YXRlTWFzayA9IDBiMDAwMDAwMDAwMTEsXG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgb3Igbm90IHRoZSB2aWV3IGlzIGluIGNyZWF0aW9uTW9kZS5cbiAgICpcbiAgICogVGhpcyBtdXN0IGJlIHN0b3JlZCBpbiB0aGUgdmlldyByYXRoZXIgdGhhbiB1c2luZyBgZGF0YWAgYXMgYSBtYXJrZXIgc28gdGhhdFxuICAgKiB3ZSBjYW4gcHJvcGVybHkgc3VwcG9ydCBlbWJlZGRlZCB2aWV3cy4gT3RoZXJ3aXNlLCB3aGVuIGV4aXRpbmcgYSBjaGlsZCB2aWV3XG4gICAqIGJhY2sgaW50byB0aGUgcGFyZW50IHZpZXcsIGBkYXRhYCB3aWxsIGJlIGRlZmluZWQgYW5kIGBjcmVhdGlvbk1vZGVgIHdpbGwgYmVcbiAgICogaW1wcm9wZXJseSByZXBvcnRlZCBhcyBmYWxzZS5cbiAgICovXG4gIENyZWF0aW9uTW9kZSA9IDBiMDAwMDAwMDAxMDAsXG5cbiAgLyoqXG4gICAqIFdoZXRoZXIgb3Igbm90IHRoaXMgTFZpZXcgaW5zdGFuY2UgaXMgb24gaXRzIGZpcnN0IHByb2Nlc3NpbmcgcGFzcy5cbiAgICpcbiAgICogQW4gTFZpZXcgaW5zdGFuY2UgaXMgY29uc2lkZXJlZCB0byBiZSBvbiBpdHMgXCJmaXJzdCBwYXNzXCIgdW50aWwgaXRcbiAgICogaGFzIGNvbXBsZXRlZCBvbmUgY3JlYXRpb24gbW9kZSBydW4gYW5kIG9uZSB1cGRhdGUgbW9kZSBydW4uIEF0IHRoaXNcbiAgICogdGltZSwgdGhlIGZsYWcgaXMgdHVybmVkIG9mZi5cbiAgICovXG4gIEZpcnN0TFZpZXdQYXNzID0gMGIwMDAwMDAwMTAwMCxcblxuICAvKiogV2hldGhlciB0aGlzIHZpZXcgaGFzIGRlZmF1bHQgY2hhbmdlIGRldGVjdGlvbiBzdHJhdGVneSAoY2hlY2tzIGFsd2F5cykgb3Igb25QdXNoICovXG4gIENoZWNrQWx3YXlzID0gMGIwMDAwMDAxMDAwMCxcblxuICAvKiogV2hldGhlciBvciBub3QgdGhpcyB2aWV3IGlzIGN1cnJlbnRseSBkaXJ0eSAobmVlZGluZyBjaGVjaykgKi9cbiAgRGlydHkgPSAwYjAwMDAwMTAwMDAwLFxuXG4gIC8qKiBXaGV0aGVyIG9yIG5vdCB0aGlzIHZpZXcgaXMgY3VycmVudGx5IGF0dGFjaGVkIHRvIGNoYW5nZSBkZXRlY3Rpb24gdHJlZS4gKi9cbiAgQXR0YWNoZWQgPSAwYjAwMDAwMTAwMDAwMCxcblxuICAvKiogV2hldGhlciBvciBub3QgdGhpcyB2aWV3IGlzIGRlc3Ryb3llZC4gKi9cbiAgRGVzdHJveWVkID0gMGIwMDAwMTAwMDAwMDAsXG5cbiAgLyoqIFdoZXRoZXIgb3Igbm90IHRoaXMgdmlldyBpcyB0aGUgcm9vdCB2aWV3ICovXG4gIElzUm9vdCA9IDBiMDAwMTAwMDAwMDAwLFxuXG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoaXMgbW92ZWQgTFZpZXcgd2FzIG5lZWRzIHRvIGJlIHJlZnJlc2hlZCBhdCB0aGUgaW5zZXJ0aW9uIGxvY2F0aW9uIGJlY2F1c2UgdGhlXG4gICAqIGRlY2xhcmF0aW9uIHdhcyBkaXJ0eS5cbiAgICovXG4gIFJlZnJlc2hUcmFuc3BsYW50ZWRWaWV3ID0gMGIwMDEwMDAwMDAwMDAsXG5cbiAgLyoqIEluZGljYXRlcyB0aGF0IHRoZSB2aWV3ICoqb3IgYW55IG9mIGl0cyBhbmNlc3RvcnMqKiBoYXZlIGFuIGVtYmVkZGVkIHZpZXcgaW5qZWN0b3IuICovXG4gIEhhc0VtYmVkZGVkVmlld0luamVjdG9yID0gMGIwMDEwMDAwMDAwMDAwLFxuXG4gIC8qKlxuICAgKiBJbmRleCBvZiB0aGUgY3VycmVudCBpbml0IHBoYXNlIG9uIGxhc3QgMjEgYml0c1xuICAgKi9cbiAgSW5kZXhXaXRoaW5Jbml0UGhhc2VJbmNyZW1lbnRlciA9IDBiMDEwMDAwMDAwMDAwMCxcbiAgSW5kZXhXaXRoaW5Jbml0UGhhc2VTaGlmdCA9IDExLFxuICBJbmRleFdpdGhpbkluaXRQaGFzZVJlc2V0ID0gMGIwMDExMTExMTExMTExLFxufVxuXG4vKipcbiAqIFBvc3NpYmxlIHN0YXRlcyBvZiB0aGUgaW5pdCBwaGFzZTpcbiAqIC0gMDA6IE9uSW5pdCBob29rcyB0byBiZSBydW4uXG4gKiAtIDAxOiBBZnRlckNvbnRlbnRJbml0IGhvb2tzIHRvIGJlIHJ1blxuICogLSAxMDogQWZ0ZXJWaWV3SW5pdCBob29rcyB0byBiZSBydW5cbiAqIC0gMTE6IEFsbCBpbml0IGhvb2tzIGhhdmUgYmVlbiBydW5cbiAqL1xuZXhwb3J0IGNvbnN0IGVudW0gSW5pdFBoYXNlU3RhdGUge1xuICBPbkluaXRIb29rc1RvQmVSdW4gPSAwYjAwLFxuICBBZnRlckNvbnRlbnRJbml0SG9va3NUb0JlUnVuID0gMGIwMSxcbiAgQWZ0ZXJWaWV3SW5pdEhvb2tzVG9CZVJ1biA9IDBiMTAsXG4gIEluaXRQaGFzZUNvbXBsZXRlZCA9IDBiMTEsXG59XG5cbi8qKiBNb3JlIGZsYWdzIGFzc29jaWF0ZWQgd2l0aCBhbiBMVmlldyAoc2F2ZWQgaW4gTFZpZXdbUFJFT1JERVJfSE9PS19GTEFHU10pICovXG5leHBvcnQgY29uc3QgZW51bSBQcmVPcmRlckhvb2tGbGFncyB7XG4gIC8qKlxuICAgICBUaGUgaW5kZXggb2YgdGhlIG5leHQgcHJlLW9yZGVyIGhvb2sgdG8gYmUgY2FsbGVkIGluIHRoZSBob29rcyBhcnJheSwgb24gdGhlIGZpcnN0IDE2XG4gICAgIGJpdHNcbiAgICovXG4gIEluZGV4T2ZUaGVOZXh0UHJlT3JkZXJIb29rTWFza01hc2sgPSAwYjAxMTExMTExMTExMTExMTExLFxuXG4gIC8qKlxuICAgKiBUaGUgbnVtYmVyIG9mIGluaXQgaG9va3MgdGhhdCBoYXZlIGFscmVhZHkgYmVlbiBjYWxsZWQsIG9uIHRoZSBsYXN0IDE2IGJpdHNcbiAgICovXG4gIE51bWJlck9mSW5pdEhvb2tzQ2FsbGVkSW5jcmVtZW50ZXIgPSAwYjAxMDAwMDAwMDAwMDAwMDAwMCxcbiAgTnVtYmVyT2ZJbml0SG9va3NDYWxsZWRTaGlmdCA9IDE2LFxuICBOdW1iZXJPZkluaXRIb29rc0NhbGxlZE1hc2sgPSAwYjExMTExMTExMTExMTExMTEwMDAwMDAwMDAwMDAwMDAwLFxufVxuXG4vKipcbiAqIFN0b3JlcyBhIHNldCBvZiBPcENvZGVzIHRvIHByb2Nlc3MgYEhvc3RCaW5kaW5nc0Z1bmN0aW9uYCBhc3NvY2lhdGVkIHdpdGggYSBjdXJyZW50IHZpZXcuXG4gKlxuICogSW4gb3JkZXIgdG8gaW52b2tlIGBIb3N0QmluZGluZ3NGdW5jdGlvbmAgd2UgbmVlZDpcbiAqIDEuICdlbGVtZW50SWR4YDogSW5kZXggdG8gdGhlIGVsZW1lbnQgYXNzb2NpYXRlZCB3aXRoIHRoZSBgSG9zdEJpbmRpbmdzRnVuY3Rpb25gLlxuICogMi4gJ2RpcmVjdGl2ZUlkeGA6IEluZGV4IHRvIHRoZSBkaXJlY3RpdmUgYXNzb2NpYXRlZCB3aXRoIHRoZSBgSG9zdEJpbmRpbmdzRnVuY3Rpb25gLiAoVGhpcyB3aWxsXG4gKiAgICBiZWNvbWUgdGhlIGNvbnRleHQgZm9yIHRoZSBgSG9zdEJpbmRpbmdzRnVuY3Rpb25gIGludm9jYXRpb24uKVxuICogMy4gYGJpbmRpbmdSb290SWR4YDogTG9jYXRpb24gd2hlcmUgdGhlIGJpbmRpbmdzIGZvciB0aGUgYEhvc3RCaW5kaW5nc0Z1bmN0aW9uYCBzdGFydC4gSW50ZXJuYWxseVxuICogICAgYEhvc3RCaW5kaW5nc0Z1bmN0aW9uYCBiaW5kaW5nIGluZGV4ZXMgc3RhcnQgZnJvbSBgMGAgc28gd2UgbmVlZCB0byBhZGQgYGJpbmRpbmdSb290SWR4YCB0b1xuICogICAgaXQuXG4gKiA0LiBgSG9zdEJpbmRpbmdzRnVuY3Rpb25gOiBBIGhvc3QgYmluZGluZyBmdW5jdGlvbiB0byBleGVjdXRlLlxuICpcbiAqIFRoZSBhYm92ZSBpbmZvcm1hdGlvbiBuZWVkcyB0byBiZSBlbmNvZGVkIGludG8gdGhlIGBIb3N0QmluZGluZ09wQ29kZXNgIGluIGFuIGVmZmljaWVudCBtYW5uZXIuXG4gKlxuICogMS4gYGVsZW1lbnRJZHhgIGlzIGVuY29kZWQgaW50byB0aGUgYEhvc3RCaW5kaW5nT3BDb2Rlc2AgYXMgYH5lbGVtZW50SWR4YCAoc28gYSBuZWdhdGl2ZSBudW1iZXIpO1xuICogMi4gYGRpcmVjdGl2ZUlkeGBcbiAqIDMuIGBiaW5kaW5nUm9vdElkeGBcbiAqIDQuIGBIb3N0QmluZGluZ3NGdW5jdGlvbmAgaXMgcGFzc2VkIGluIGFzIGlzLlxuICpcbiAqIFRoZSBgSG9zdEJpbmRpbmdPcENvZGVzYCBhcnJheSBjb250YWluczpcbiAqIC0gbmVnYXRpdmUgbnVtYmVyIHRvIHNlbGVjdCB0aGUgZWxlbWVudCBpbmRleC5cbiAqIC0gZm9sbG93ZWQgYnkgMSBvciBtb3JlIG9mOlxuICogICAgLSBhIG51bWJlciB0byBzZWxlY3QgdGhlIGRpcmVjdGl2ZSBpbmRleFxuICogICAgLSBhIG51bWJlciB0byBzZWxlY3QgdGhlIGJpbmRpbmdSb290IGluZGV4XG4gKiAgICAtIGFuZCBhIGZ1bmN0aW9uIHRvIGludm9rZS5cbiAqXG4gKiAjIyBFeGFtcGxlXG4gKlxuICogYGBgXG4gKiBjb25zdCBob3N0QmluZGluZ09wQ29kZXMgPSBbXG4gKiAgIH4zMCwgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgLy8gU2VsZWN0IGVsZW1lbnQgMzBcbiAqICAgNDAsIDQ1LCBNeURpci7JtWRpci5ob3N0QmluZGluZ3MgICAgLy8gSW52b2tlIGhvc3QgYmluZGluZ3Mgb24gTXlEaXIgb24gZWxlbWVudCAzMDtcbiAqICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAvLyBkaXJlY3RpdmVJZHggPSA0MDsgYmluZGluZ1Jvb3RJZHggPSA0NTtcbiAqICAgNTAsIDU1LCBPdGhlckRpci7JtWRpci5ob3N0QmluZGluZ3MgLy8gSW52b2tlIGhvc3QgYmluZGluZ3Mgb24gT3RoZXJEaXJlIG9uIGVsZW1lbnQgMzBcbiAqICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAgICAvLyBkaXJlY3RpdmVJZHggPSA1MDsgYmluZGluZ1Jvb3RJZHggPSA1NTtcbiAqIF1cbiAqIGBgYFxuICpcbiAqICMjIFBzZXVkb2NvZGVcbiAqIGBgYFxuICogY29uc3QgaG9zdEJpbmRpbmdPcENvZGVzID0gdFZpZXcuaG9zdEJpbmRpbmdPcENvZGVzO1xuICogaWYgKGhvc3RCaW5kaW5nT3BDb2RlcyA9PT0gbnVsbCkgcmV0dXJuO1xuICogZm9yIChsZXQgaSA9IDA7IGkgPCBob3N0QmluZGluZ09wQ29kZXMubGVuZ3RoOyBpKyspIHtcbiAqICAgY29uc3Qgb3BDb2RlID0gaG9zdEJpbmRpbmdPcENvZGVzW2ldIGFzIG51bWJlcjtcbiAqICAgaWYgKG9wQ29kZSA8IDApIHtcbiAqICAgICAvLyBOZWdhdGl2ZSBudW1iZXJzIGFyZSBlbGVtZW50IGluZGV4ZXMuXG4gKiAgICAgc2V0U2VsZWN0ZWRJbmRleCh+b3BDb2RlKTtcbiAqICAgfSBlbHNlIHtcbiAqICAgICAvLyBQb3NpdGl2ZSBudW1iZXJzIGFyZSBOdW1iZXJUdXBsZSB3aGljaCBzdG9yZSBiaW5kaW5nUm9vdEluZGV4IGFuZCBkaXJlY3RpdmVJbmRleC5cbiAqICAgICBjb25zdCBkaXJlY3RpdmVJZHggPSBvcENvZGU7XG4gKiAgICAgY29uc3QgYmluZGluZ1Jvb3RJbmR4ID0gaG9zdEJpbmRpbmdPcENvZGVzWysraV0gYXMgbnVtYmVyO1xuICogICAgIGNvbnN0IGhvc3RCaW5kaW5nRm4gPSBob3N0QmluZGluZ09wQ29kZXNbKytpXSBhcyBIb3N0QmluZGluZ3NGdW5jdGlvbjxhbnk+O1xuICogICAgIHNldEJpbmRpbmdSb290Rm9ySG9zdEJpbmRpbmdzKGJpbmRpbmdSb290SW5keCwgZGlyZWN0aXZlSWR4KTtcbiAqICAgICBjb25zdCBjb250ZXh0ID0gbFZpZXdbZGlyZWN0aXZlSWR4XTtcbiAqICAgICBob3N0QmluZGluZ0ZuKFJlbmRlckZsYWdzLlVwZGF0ZSwgY29udGV4dCk7XG4gKiAgIH1cbiAqIH1cbiAqIGBgYFxuICpcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBIb3N0QmluZGluZ09wQ29kZXMgZXh0ZW5kcyBBcnJheTxudW1iZXJ8SG9zdEJpbmRpbmdzRnVuY3Rpb248YW55Pj4ge1xuICBfX2JyYW5kX186ICdIb3N0QmluZGluZ09wQ29kZXMnO1xuICBkZWJ1Zz86IHN0cmluZ1tdO1xufVxuXG4vKipcbiAqIEV4cGxpY2l0bHkgbWFya3MgYFRWaWV3YCBhcyBhIHNwZWNpZmljIHR5cGUgaW4gYG5nRGV2TW9kZWBcbiAqXG4gKiBJdCBpcyB1c2VmdWwgdG8ga25vdyBjb25jZXB0dWFsbHkgd2hhdCB0aW1lIG9mIGBUVmlld2Agd2UgYXJlIGRlYWxpbmcgd2l0aCB3aGVuXG4gKiBkZWJ1Z2dpbmcgYW4gYXBwbGljYXRpb24gKGV2ZW4gaWYgdGhlIHJ1bnRpbWUgZG9lcyBub3QgbmVlZCBpdC4pIEZvciB0aGlzIHJlYXNvblxuICogd2Ugc3RvcmUgdGhpcyBpbmZvcm1hdGlvbiBpbiB0aGUgYG5nRGV2TW9kZWAgYFRWaWV3YCBhbmQgdGhhbiB1c2UgaXQgZm9yXG4gKiBiZXR0ZXIgZGVidWdnaW5nIGV4cGVyaWVuY2UuXG4gKi9cbmV4cG9ydCBjb25zdCBlbnVtIFRWaWV3VHlwZSB7XG4gIC8qKlxuICAgKiBSb290IGBUVmlld2AgaXMgdGhlIHVzZWQgdG8gYm9vdHN0cmFwIGNvbXBvbmVudHMgaW50by4gSXQgaXMgdXNlZCBpbiBjb25qdW5jdGlvbiB3aXRoXG4gICAqIGBMVmlld2Agd2hpY2ggdGFrZXMgYW4gZXhpc3RpbmcgRE9NIG5vZGUgbm90IG93bmVkIGJ5IEFuZ3VsYXIgYW5kIHdyYXBzIGl0IGluIGBUVmlld2AvYExWaWV3YFxuICAgKiBzbyB0aGF0IG90aGVyIGNvbXBvbmVudHMgY2FuIGJlIGxvYWRlZCBpbnRvIGl0LlxuICAgKi9cbiAgUm9vdCA9IDAsXG5cbiAgLyoqXG4gICAqIGBUVmlld2AgYXNzb2NpYXRlZCB3aXRoIGEgQ29tcG9uZW50LiBUaGlzIHdvdWxkIGJlIHRoZSBgVFZpZXdgIGRpcmVjdGx5IGFzc29jaWF0ZWQgd2l0aCB0aGVcbiAgICogY29tcG9uZW50IHZpZXcgKGFzIG9wcG9zZWQgYW4gYEVtYmVkZGVkYCBgVFZpZXdgIHdoaWNoIHdvdWxkIGJlIGEgY2hpbGQgb2YgYENvbXBvbmVudGAgYFRWaWV3YClcbiAgICovXG4gIENvbXBvbmVudCA9IDEsXG5cbiAgLyoqXG4gICAqIGBUVmlld2AgYXNzb2NpYXRlZCB3aXRoIGEgdGVtcGxhdGUuIFN1Y2ggYXMgYCpuZ0lmYCwgYDxuZy10ZW1wbGF0ZT5gIGV0Yy4uLiBBIGBDb21wb25lbnRgXG4gICAqIGNhbiBoYXZlIHplcm8gb3IgbW9yZSBgRW1iZWRkZWRgIGBUVmlld2BzLlxuICAgKi9cbiAgRW1iZWRkZWQgPSAyLFxufVxuXG4vKipcbiAqIENvbnZlcnRzIGBUVmlld1R5cGVgIGludG8gaHVtYW4gcmVhZGFibGUgdGV4dC5cbiAqIE1ha2Ugc3VyZSB0aGlzIG1hdGNoZXMgd2l0aCBgVFZpZXdUeXBlYFxuICovXG5leHBvcnQgY29uc3QgVFZpZXdUeXBlQXNTdHJpbmcgPSBbXG4gICdSb290JywgICAgICAgLy8gMFxuICAnQ29tcG9uZW50JywgIC8vIDFcbiAgJ0VtYmVkZGVkJywgICAvLyAyXG5dIGFzIGNvbnN0O1xuXG5cbi8qKlxuICogVGhlIHN0YXRpYyBkYXRhIGZvciBhbiBMVmlldyAoc2hhcmVkIGJldHdlZW4gYWxsIHRlbXBsYXRlcyBvZiBhXG4gKiBnaXZlbiB0eXBlKS5cbiAqXG4gKiBTdG9yZWQgb24gdGhlIGBDb21wb25lbnREZWYudFZpZXdgLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIFRWaWV3IHtcbiAgLyoqXG4gICAqIFR5cGUgb2YgYFRWaWV3YCAoYFJvb3RgfGBDb21wb25lbnRgfGBFbWJlZGRlZGApLlxuICAgKi9cbiAgdHlwZTogVFZpZXdUeXBlO1xuXG4gIC8qKlxuICAgKiBUaGlzIGlzIGEgYmx1ZXByaW50IHVzZWQgdG8gZ2VuZXJhdGUgTFZpZXcgaW5zdGFuY2VzIGZvciB0aGlzIFRWaWV3LiBDb3B5aW5nIHRoaXNcbiAgICogYmx1ZXByaW50IGlzIGZhc3RlciB0aGFuIGNyZWF0aW5nIGEgbmV3IExWaWV3IGZyb20gc2NyYXRjaC5cbiAgICovXG4gIGJsdWVwcmludDogTFZpZXc7XG5cbiAgLyoqXG4gICAqIFRoZSB0ZW1wbGF0ZSBmdW5jdGlvbiB1c2VkIHRvIHJlZnJlc2ggdGhlIHZpZXcgb2YgZHluYW1pY2FsbHkgY3JlYXRlZCB2aWV3c1xuICAgKiBhbmQgY29tcG9uZW50cy4gV2lsbCBiZSBudWxsIGZvciBpbmxpbmUgdmlld3MuXG4gICAqL1xuICB0ZW1wbGF0ZTogQ29tcG9uZW50VGVtcGxhdGU8e30+fG51bGw7XG5cbiAgLyoqXG4gICAqIEEgZnVuY3Rpb24gY29udGFpbmluZyBxdWVyeS1yZWxhdGVkIGluc3RydWN0aW9ucy5cbiAgICovXG4gIHZpZXdRdWVyeTogVmlld1F1ZXJpZXNGdW5jdGlvbjx7fT58bnVsbDtcblxuICAvKipcbiAgICogQSBgVE5vZGVgIHJlcHJlc2VudGluZyB0aGUgZGVjbGFyYXRpb24gbG9jYXRpb24gb2YgdGhpcyBgVFZpZXdgIChub3QgcGFydCBvZiB0aGlzIFRWaWV3KS5cbiAgICovXG4gIGRlY2xUTm9kZTogVE5vZGV8bnVsbDtcblxuICAvLyBGSVhNRShtaXNrbyk6IFdoeSBkb2VzIGBUVmlld2Agbm90IGhhdmUgYGRlY2xhcmF0aW9uVFZpZXdgIHByb3BlcnR5P1xuXG4gIC8qKiBXaGV0aGVyIG9yIG5vdCB0aGlzIHRlbXBsYXRlIGhhcyBiZWVuIHByb2Nlc3NlZCBpbiBjcmVhdGlvbiBtb2RlLiAqL1xuICBmaXJzdENyZWF0ZVBhc3M6IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqICBXaGV0aGVyIG9yIG5vdCB0aGlzIHRlbXBsYXRlIGhhcyBiZWVuIHByb2Nlc3NlZCBpbiB1cGRhdGUgbW9kZSAoZS5nLiBjaGFuZ2UgZGV0ZWN0ZWQpXG4gICAqXG4gICAqIGBmaXJzdFVwZGF0ZVBhc3NgIGlzIHVzZWQgYnkgc3R5bGluZyB0byBzZXQgdXAgYFREYXRhYCB0byBjb250YWluIG1ldGFkYXRhIGFib3V0IHRoZSBzdHlsaW5nXG4gICAqIGluc3RydWN0aW9ucy4gKE1haW5seSB0byBidWlsZCB1cCBhIGxpbmtlZCBsaXN0IG9mIHN0eWxpbmcgcHJpb3JpdHkgb3JkZXIuKVxuICAgKlxuICAgKiBUeXBpY2FsbHkgdGhpcyBmdW5jdGlvbiBnZXRzIGNsZWFyZWQgYWZ0ZXIgZmlyc3QgZXhlY3V0aW9uLiBJZiBleGNlcHRpb24gaXMgdGhyb3duIHRoZW4gdGhpc1xuICAgKiBmbGFnIGNhbiByZW1haW4gdHVybmVkIHVuIHVudGlsIHRoZXJlIGlzIGZpcnN0IHN1Y2Nlc3NmdWwgKG5vIGV4Y2VwdGlvbikgcGFzcy4gVGhpcyBtZWFucyB0aGF0XG4gICAqIGluZGl2aWR1YWwgc3R5bGluZyBpbnN0cnVjdGlvbnMga2VlcCB0cmFjayBvZiBpZiB0aGV5IGhhdmUgYWxyZWFkeSBiZWVuIGFkZGVkIHRvIHRoZSBsaW5rZWRcbiAgICogbGlzdCB0byBwcmV2ZW50IGRvdWJsZSBhZGRpbmcuXG4gICAqL1xuICBmaXJzdFVwZGF0ZVBhc3M6IGJvb2xlYW47XG5cbiAgLyoqIFN0YXRpYyBkYXRhIGVxdWl2YWxlbnQgb2YgTFZpZXcuZGF0YVtdLiBDb250YWlucyBUTm9kZXMsIFBpcGVEZWZJbnRlcm5hbCBvciBUSTE4bi4gKi9cbiAgZGF0YTogVERhdGE7XG5cbiAgLyoqXG4gICAqIFRoZSBiaW5kaW5nIHN0YXJ0IGluZGV4IGlzIHRoZSBpbmRleCBhdCB3aGljaCB0aGUgZGF0YSBhcnJheVxuICAgKiBzdGFydHMgdG8gc3RvcmUgYmluZGluZ3Mgb25seS4gU2F2aW5nIHRoaXMgdmFsdWUgZW5zdXJlcyB0aGF0IHdlXG4gICAqIHdpbGwgYmVnaW4gcmVhZGluZyBiaW5kaW5ncyBhdCB0aGUgY29ycmVjdCBwb2ludCBpbiB0aGUgYXJyYXkgd2hlblxuICAgKiB3ZSBhcmUgaW4gdXBkYXRlIG1vZGUuXG4gICAqXG4gICAqIC0xIG1lYW5zIHRoYXQgaXQgaGFzIG5vdCBiZWVuIGluaXRpYWxpemVkLlxuICAgKi9cbiAgYmluZGluZ1N0YXJ0SW5kZXg6IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIGluZGV4IHdoZXJlIHRoZSBcImV4cGFuZG9cIiBzZWN0aW9uIG9mIGBMVmlld2AgYmVnaW5zLiBUaGUgZXhwYW5kb1xuICAgKiBzZWN0aW9uIGNvbnRhaW5zIGluamVjdG9ycywgZGlyZWN0aXZlIGluc3RhbmNlcywgYW5kIGhvc3QgYmluZGluZyB2YWx1ZXMuXG4gICAqIFVubGlrZSB0aGUgXCJkZWNsc1wiIGFuZCBcInZhcnNcIiBzZWN0aW9ucyBvZiBgTFZpZXdgLCB0aGUgbGVuZ3RoIG9mIHRoaXNcbiAgICogc2VjdGlvbiBjYW5ub3QgYmUgY2FsY3VsYXRlZCBhdCBjb21waWxlLXRpbWUgYmVjYXVzZSBkaXJlY3RpdmVzIGFyZSBtYXRjaGVkXG4gICAqIGF0IHJ1bnRpbWUgdG8gcHJlc2VydmUgbG9jYWxpdHkuXG4gICAqXG4gICAqIFdlIHN0b3JlIHRoaXMgc3RhcnQgaW5kZXggc28gd2Uga25vdyB3aGVyZSB0byBzdGFydCBjaGVja2luZyBob3N0IGJpbmRpbmdzXG4gICAqIGluIGBzZXRIb3N0QmluZGluZ3NgLlxuICAgKi9cbiAgZXhwYW5kb1N0YXJ0SW5kZXg6IG51bWJlcjtcblxuICAvKipcbiAgICogV2hldGhlciBvciBub3QgdGhlcmUgYXJlIGFueSBzdGF0aWMgdmlldyBxdWVyaWVzIHRyYWNrZWQgb24gdGhpcyB2aWV3LlxuICAgKlxuICAgKiBXZSBzdG9yZSB0aGlzIHNvIHdlIGtub3cgd2hldGhlciBvciBub3Qgd2Ugc2hvdWxkIGRvIGEgdmlldyBxdWVyeVxuICAgKiByZWZyZXNoIGFmdGVyIGNyZWF0aW9uIG1vZGUgdG8gY29sbGVjdCBzdGF0aWMgcXVlcnkgcmVzdWx0cy5cbiAgICovXG4gIHN0YXRpY1ZpZXdRdWVyaWVzOiBib29sZWFuO1xuXG4gIC8qKlxuICAgKiBXaGV0aGVyIG9yIG5vdCB0aGVyZSBhcmUgYW55IHN0YXRpYyBjb250ZW50IHF1ZXJpZXMgdHJhY2tlZCBvbiB0aGlzIHZpZXcuXG4gICAqXG4gICAqIFdlIHN0b3JlIHRoaXMgc28gd2Uga25vdyB3aGV0aGVyIG9yIG5vdCB3ZSBzaG91bGQgZG8gYSBjb250ZW50IHF1ZXJ5XG4gICAqIHJlZnJlc2ggYWZ0ZXIgY3JlYXRpb24gbW9kZSB0byBjb2xsZWN0IHN0YXRpYyBxdWVyeSByZXN1bHRzLlxuICAgKi9cbiAgc3RhdGljQ29udGVudFF1ZXJpZXM6IGJvb2xlYW47XG5cbiAgLyoqXG4gICAqIEEgcmVmZXJlbmNlIHRvIHRoZSBmaXJzdCBjaGlsZCBub2RlIGxvY2F0ZWQgaW4gdGhlIHZpZXcuXG4gICAqL1xuICBmaXJzdENoaWxkOiBUTm9kZXxudWxsO1xuXG4gIC8qKlxuICAgKiBTdG9yZXMgdGhlIE9wQ29kZXMgdG8gYmUgcmVwbGF5ZWQgZHVyaW5nIGNoYW5nZS1kZXRlY3Rpb24gdG8gcHJvY2VzcyB0aGUgYEhvc3RCaW5kaW5nc2BcbiAgICpcbiAgICogU2VlIGBIb3N0QmluZGluZ09wQ29kZXNgIGZvciBlbmNvZGluZyBkZXRhaWxzLlxuICAgKi9cbiAgaG9zdEJpbmRpbmdPcENvZGVzOiBIb3N0QmluZGluZ09wQ29kZXN8bnVsbDtcblxuICAvKipcbiAgICogRnVsbCByZWdpc3RyeSBvZiBkaXJlY3RpdmVzIGFuZCBjb21wb25lbnRzIHRoYXQgbWF5IGJlIGZvdW5kIGluIHRoaXMgdmlldy5cbiAgICpcbiAgICogSXQncyBuZWNlc3NhcnkgdG8ga2VlcCBhIGNvcHkgb2YgdGhlIGZ1bGwgZGVmIGxpc3Qgb24gdGhlIFRWaWV3IHNvIGl0J3MgcG9zc2libGVcbiAgICogdG8gcmVuZGVyIHRlbXBsYXRlIGZ1bmN0aW9ucyB3aXRob3V0IGEgaG9zdCBjb21wb25lbnQuXG4gICAqL1xuICBkaXJlY3RpdmVSZWdpc3RyeTogRGlyZWN0aXZlRGVmTGlzdHxudWxsO1xuXG4gIC8qKlxuICAgKiBGdWxsIHJlZ2lzdHJ5IG9mIHBpcGVzIHRoYXQgbWF5IGJlIGZvdW5kIGluIHRoaXMgdmlldy5cbiAgICpcbiAgICogVGhlIHByb3BlcnR5IGlzIGVpdGhlciBhbiBhcnJheSBvZiBgUGlwZURlZnNgcyBvciBhIGZ1bmN0aW9uIHdoaWNoIHJldHVybnMgdGhlIGFycmF5IG9mXG4gICAqIGBQaXBlRGVmc2BzLiBUaGUgZnVuY3Rpb24gaXMgbmVjZXNzYXJ5IHRvIGJlIGFibGUgdG8gc3VwcG9ydCBmb3J3YXJkIGRlY2xhcmF0aW9ucy5cbiAgICpcbiAgICogSXQncyBuZWNlc3NhcnkgdG8ga2VlcCBhIGNvcHkgb2YgdGhlIGZ1bGwgZGVmIGxpc3Qgb24gdGhlIFRWaWV3IHNvIGl0J3MgcG9zc2libGVcbiAgICogdG8gcmVuZGVyIHRlbXBsYXRlIGZ1bmN0aW9ucyB3aXRob3V0IGEgaG9zdCBjb21wb25lbnQuXG4gICAqL1xuICBwaXBlUmVnaXN0cnk6IFBpcGVEZWZMaXN0fG51bGw7XG5cbiAgLyoqXG4gICAqIEFycmF5IG9mIG5nT25Jbml0LCBuZ09uQ2hhbmdlcyBhbmQgbmdEb0NoZWNrIGhvb2tzIHRoYXQgc2hvdWxkIGJlIGV4ZWN1dGVkIGZvciB0aGlzIHZpZXcgaW5cbiAgICogY3JlYXRpb24gbW9kZS5cbiAgICpcbiAgICogVGhpcyBhcnJheSBoYXMgYSBmbGF0IHN0cnVjdHVyZSBhbmQgY29udGFpbnMgVE5vZGUgaW5kaWNlcywgZGlyZWN0aXZlIGluZGljZXMgKHdoZXJlIGFuXG4gICAqIGluc3RhbmNlIGNhbiBiZSBmb3VuZCBpbiBgTFZpZXdgKSBhbmQgaG9vayBmdW5jdGlvbnMuIFROb2RlIGluZGV4IGlzIGZvbGxvd2VkIGJ5IHRoZSBkaXJlY3RpdmVcbiAgICogaW5kZXggYW5kIGEgaG9vayBmdW5jdGlvbi4gSWYgdGhlcmUgYXJlIG11bHRpcGxlIGhvb2tzIGZvciBhIGdpdmVuIFROb2RlLCB0aGUgVE5vZGUgaW5kZXggaXNcbiAgICogbm90IHJlcGVhdGVkIGFuZCB0aGUgbmV4dCBsaWZlY3ljbGUgaG9vayBpbmZvcm1hdGlvbiBpcyBzdG9yZWQgcmlnaHQgYWZ0ZXIgdGhlIHByZXZpb3VzIGhvb2tcbiAgICogZnVuY3Rpb24uIFRoaXMgaXMgZG9uZSBzbyB0aGF0IGF0IHJ1bnRpbWUgdGhlIHN5c3RlbSBjYW4gZWZmaWNpZW50bHkgaXRlcmF0ZSBvdmVyIGFsbCBvZiB0aGVcbiAgICogZnVuY3Rpb25zIHRvIGludm9rZSB3aXRob3V0IGhhdmluZyB0byBtYWtlIGFueSBkZWNpc2lvbnMvbG9va3Vwcy5cbiAgICovXG4gIHByZU9yZGVySG9va3M6IEhvb2tEYXRhfG51bGw7XG5cbiAgLyoqXG4gICAqIEFycmF5IG9mIG5nT25DaGFuZ2VzIGFuZCBuZ0RvQ2hlY2sgaG9va3MgdGhhdCBzaG91bGQgYmUgZXhlY3V0ZWQgZm9yIHRoaXMgdmlldyBpbiB1cGRhdGUgbW9kZS5cbiAgICpcbiAgICogVGhpcyBhcnJheSBoYXMgdGhlIHNhbWUgc3RydWN0dXJlIGFzIHRoZSBgcHJlT3JkZXJIb29rc2Agb25lLlxuICAgKi9cbiAgcHJlT3JkZXJDaGVja0hvb2tzOiBIb29rRGF0YXxudWxsO1xuXG4gIC8qKlxuICAgKiBBcnJheSBvZiBuZ0FmdGVyQ29udGVudEluaXQgYW5kIG5nQWZ0ZXJDb250ZW50Q2hlY2tlZCBob29rcyB0aGF0IHNob3VsZCBiZSBleGVjdXRlZFxuICAgKiBmb3IgdGhpcyB2aWV3IGluIGNyZWF0aW9uIG1vZGUuXG4gICAqXG4gICAqIEV2ZW4gaW5kaWNlczogRGlyZWN0aXZlIGluZGV4XG4gICAqIE9kZCBpbmRpY2VzOiBIb29rIGZ1bmN0aW9uXG4gICAqL1xuICBjb250ZW50SG9va3M6IEhvb2tEYXRhfG51bGw7XG5cbiAgLyoqXG4gICAqIEFycmF5IG9mIG5nQWZ0ZXJDb250ZW50Q2hlY2tlZCBob29rcyB0aGF0IHNob3VsZCBiZSBleGVjdXRlZCBmb3IgdGhpcyB2aWV3IGluIHVwZGF0ZVxuICAgKiBtb2RlLlxuICAgKlxuICAgKiBFdmVuIGluZGljZXM6IERpcmVjdGl2ZSBpbmRleFxuICAgKiBPZGQgaW5kaWNlczogSG9vayBmdW5jdGlvblxuICAgKi9cbiAgY29udGVudENoZWNrSG9va3M6IEhvb2tEYXRhfG51bGw7XG5cbiAgLyoqXG4gICAqIEFycmF5IG9mIG5nQWZ0ZXJWaWV3SW5pdCBhbmQgbmdBZnRlclZpZXdDaGVja2VkIGhvb2tzIHRoYXQgc2hvdWxkIGJlIGV4ZWN1dGVkIGZvclxuICAgKiB0aGlzIHZpZXcgaW4gY3JlYXRpb24gbW9kZS5cbiAgICpcbiAgICogRXZlbiBpbmRpY2VzOiBEaXJlY3RpdmUgaW5kZXhcbiAgICogT2RkIGluZGljZXM6IEhvb2sgZnVuY3Rpb25cbiAgICovXG4gIHZpZXdIb29rczogSG9va0RhdGF8bnVsbDtcblxuICAvKipcbiAgICogQXJyYXkgb2YgbmdBZnRlclZpZXdDaGVja2VkIGhvb2tzIHRoYXQgc2hvdWxkIGJlIGV4ZWN1dGVkIGZvciB0aGlzIHZpZXcgaW5cbiAgICogdXBkYXRlIG1vZGUuXG4gICAqXG4gICAqIEV2ZW4gaW5kaWNlczogRGlyZWN0aXZlIGluZGV4XG4gICAqIE9kZCBpbmRpY2VzOiBIb29rIGZ1bmN0aW9uXG4gICAqL1xuICB2aWV3Q2hlY2tIb29rczogSG9va0RhdGF8bnVsbDtcblxuICAvKipcbiAgICogQXJyYXkgb2YgbmdPbkRlc3Ryb3kgaG9va3MgdGhhdCBzaG91bGQgYmUgZXhlY3V0ZWQgd2hlbiB0aGlzIHZpZXcgaXMgZGVzdHJveWVkLlxuICAgKlxuICAgKiBFdmVuIGluZGljZXM6IERpcmVjdGl2ZSBpbmRleFxuICAgKiBPZGQgaW5kaWNlczogSG9vayBmdW5jdGlvblxuICAgKi9cbiAgZGVzdHJveUhvb2tzOiBEZXN0cm95SG9va0RhdGF8bnVsbDtcblxuICAvKipcbiAgICogV2hlbiBhIHZpZXcgaXMgZGVzdHJveWVkLCBsaXN0ZW5lcnMgbmVlZCB0byBiZSByZWxlYXNlZCBhbmQgb3V0cHV0cyBuZWVkIHRvIGJlXG4gICAqIHVuc3Vic2NyaWJlZC4gVGhpcyBjbGVhbnVwIGFycmF5IHN0b3JlcyBib3RoIGxpc3RlbmVyIGRhdGEgKGluIGNodW5rcyBvZiA0KVxuICAgKiBhbmQgb3V0cHV0IGRhdGEgKGluIGNodW5rcyBvZiAyKSBmb3IgYSBwYXJ0aWN1bGFyIHZpZXcuIENvbWJpbmluZyB0aGUgYXJyYXlzXG4gICAqIHNhdmVzIG9uIG1lbW9yeSAoNzAgYnl0ZXMgcGVyIGFycmF5KSBhbmQgb24gYSBmZXcgYnl0ZXMgb2YgY29kZSBzaXplIChmb3IgdHdvXG4gICAqIHNlcGFyYXRlIGZvciBsb29wcykuXG4gICAqXG4gICAqIElmIGl0J3MgYSBuYXRpdmUgRE9NIGxpc3RlbmVyIG9yIG91dHB1dCBzdWJzY3JpcHRpb24gYmVpbmcgc3RvcmVkOlxuICAgKiAxc3QgaW5kZXggaXM6IGV2ZW50IG5hbWUgIGBuYW1lID0gdFZpZXcuY2xlYW51cFtpKzBdYFxuICAgKiAybmQgaW5kZXggaXM6IGluZGV4IG9mIG5hdGl2ZSBlbGVtZW50IG9yIGEgZnVuY3Rpb24gdGhhdCByZXRyaWV2ZXMgZ2xvYmFsIHRhcmdldCAod2luZG93LFxuICAgKiAgICAgICAgICAgICAgIGRvY3VtZW50IG9yIGJvZHkpIHJlZmVyZW5jZSBiYXNlZCBvbiB0aGUgbmF0aXZlIGVsZW1lbnQ6XG4gICAqICAgIGB0eXBlb2YgaWR4T3JUYXJnZXRHZXR0ZXIgPT09ICdmdW5jdGlvbidgOiBnbG9iYWwgdGFyZ2V0IGdldHRlciBmdW5jdGlvblxuICAgKiAgICBgdHlwZW9mIGlkeE9yVGFyZ2V0R2V0dGVyID09PSAnbnVtYmVyJ2A6IGluZGV4IG9mIG5hdGl2ZSBlbGVtZW50XG4gICAqXG4gICAqIDNyZCBpbmRleCBpczogaW5kZXggb2YgbGlzdGVuZXIgZnVuY3Rpb24gYGxpc3RlbmVyID0gbFZpZXdbQ0xFQU5VUF1bdFZpZXcuY2xlYW51cFtpKzJdXWBcbiAgICogNHRoIGluZGV4IGlzOiBgdXNlQ2FwdHVyZU9ySW5keCA9IHRWaWV3LmNsZWFudXBbaSszXWBcbiAgICogICAgYHR5cGVvZiB1c2VDYXB0dXJlT3JJbmR4ID09ICdib29sZWFuJyA6IHVzZUNhcHR1cmUgYm9vbGVhblxuICAgKiAgICBgdHlwZW9mIHVzZUNhcHR1cmVPckluZHggPT0gJ251bWJlcic6XG4gICAqICAgICAgICAgYHVzZUNhcHR1cmVPckluZHggPj0gMGAgYHJlbW92ZUxpc3RlbmVyID0gTFZpZXdbQ0xFQU5VUF1bdXNlQ2FwdHVyZU9ySW5keF1gXG4gICAqICAgICAgICAgYHVzZUNhcHR1cmVPckluZHggPCAgMGAgYHN1YnNjcmlwdGlvbiA9IExWaWV3W0NMRUFOVVBdWy11c2VDYXB0dXJlT3JJbmR4XWBcbiAgICpcbiAgICogSWYgaXQncyBhbiBvdXRwdXQgc3Vic2NyaXB0aW9uIG9yIHF1ZXJ5IGxpc3QgZGVzdHJveSBob29rOlxuICAgKiAxc3QgaW5kZXggaXM6IG91dHB1dCB1bnN1YnNjcmliZSBmdW5jdGlvbiAvIHF1ZXJ5IGxpc3QgZGVzdHJveSBmdW5jdGlvblxuICAgKiAybmQgaW5kZXggaXM6IGluZGV4IG9mIGZ1bmN0aW9uIGNvbnRleHQgaW4gTFZpZXcuY2xlYW51cEluc3RhbmNlc1tdXG4gICAqICAgICAgICAgICAgICAgYHRWaWV3LmNsZWFudXBbaSswXS5jYWxsKGxWaWV3W0NMRUFOVVBdW3RWaWV3LmNsZWFudXBbaSsxXV0pYFxuICAgKi9cbiAgY2xlYW51cDogYW55W118bnVsbDtcblxuICAvKipcbiAgICogQSBsaXN0IG9mIGVsZW1lbnQgaW5kaWNlcyBmb3IgY2hpbGQgY29tcG9uZW50cyB0aGF0IHdpbGwgbmVlZCB0byBiZVxuICAgKiByZWZyZXNoZWQgd2hlbiB0aGUgY3VycmVudCB2aWV3IGhhcyBmaW5pc2hlZCBpdHMgY2hlY2suIFRoZXNlIGluZGljZXMgaGF2ZVxuICAgKiBhbHJlYWR5IGJlZW4gYWRqdXN0ZWQgZm9yIHRoZSBIRUFERVJfT0ZGU0VULlxuICAgKlxuICAgKi9cbiAgY29tcG9uZW50czogbnVtYmVyW118bnVsbDtcblxuICAvKipcbiAgICogQSBjb2xsZWN0aW9uIG9mIHF1ZXJpZXMgdHJhY2tlZCBpbiBhIGdpdmVuIHZpZXcuXG4gICAqL1xuICBxdWVyaWVzOiBUUXVlcmllc3xudWxsO1xuXG4gIC8qKlxuICAgKiBBbiBhcnJheSBvZiBpbmRpY2VzIHBvaW50aW5nIHRvIGRpcmVjdGl2ZXMgd2l0aCBjb250ZW50IHF1ZXJpZXMgYWxvbmdzaWRlIHdpdGggdGhlXG4gICAqIGNvcnJlc3BvbmRpbmcgcXVlcnkgaW5kZXguIEVhY2ggZW50cnkgaW4gdGhpcyBhcnJheSBpcyBhIHR1cGxlIG9mOlxuICAgKiAtIGluZGV4IG9mIHRoZSBmaXJzdCBjb250ZW50IHF1ZXJ5IGluZGV4IGRlY2xhcmVkIGJ5IGEgZ2l2ZW4gZGlyZWN0aXZlO1xuICAgKiAtIGluZGV4IG9mIGEgZGlyZWN0aXZlLlxuICAgKlxuICAgKiBXZSBhcmUgc3RvcmluZyB0aG9zZSBpbmRleGVzIHNvIHdlIGNhbiByZWZyZXNoIGNvbnRlbnQgcXVlcmllcyBhcyBwYXJ0IG9mIGEgdmlldyByZWZyZXNoXG4gICAqIHByb2Nlc3MuXG4gICAqL1xuICBjb250ZW50UXVlcmllczogbnVtYmVyW118bnVsbDtcblxuICAvKipcbiAgICogU2V0IG9mIHNjaGVtYXMgdGhhdCBkZWNsYXJlIGVsZW1lbnRzIHRvIGJlIGFsbG93ZWQgaW5zaWRlIHRoZSB2aWV3LlxuICAgKi9cbiAgc2NoZW1hczogU2NoZW1hTWV0YWRhdGFbXXxudWxsO1xuXG4gIC8qKlxuICAgKiBBcnJheSBvZiBjb25zdGFudHMgZm9yIHRoZSB2aWV3LiBJbmNsdWRlcyBhdHRyaWJ1dGUgYXJyYXlzLCBsb2NhbCBkZWZpbml0aW9uIGFycmF5cyBldGMuXG4gICAqIFVzZWQgZm9yIGRpcmVjdGl2ZSBtYXRjaGluZywgYXR0cmlidXRlIGJpbmRpbmdzLCBsb2NhbCBkZWZpbml0aW9ucyBhbmQgbW9yZS5cbiAgICovXG4gIGNvbnN0czogVENvbnN0YW50c3xudWxsO1xuXG4gIC8qKlxuICAgKiBJbmRpY2F0ZXMgdGhhdCB0aGVyZSB3YXMgYW4gZXJyb3IgYmVmb3JlIHdlIG1hbmFnZWQgdG8gY29tcGxldGUgdGhlIGZpcnN0IGNyZWF0ZSBwYXNzIG9mIHRoZVxuICAgKiB2aWV3LiBUaGlzIG1lYW5zIHRoYXQgdGhlIHZpZXcgaXMgbGlrZWx5IGNvcnJ1cHRlZCBhbmQgd2Ugc2hvdWxkIHRyeSB0byByZWNvdmVyIGl0LlxuICAgKi9cbiAgaW5jb21wbGV0ZUZpcnN0UGFzczogYm9vbGVhbjtcbn1cblxuLyoqIFNpbmdsZSBob29rIGNhbGxiYWNrIGZ1bmN0aW9uLiAqL1xuZXhwb3J0IHR5cGUgSG9va0ZuID0gKCkgPT4gdm9pZDtcblxuLyoqXG4gKiBJbmZvcm1hdGlvbiBuZWNlc3NhcnkgdG8gY2FsbCBhIGhvb2suIEUuZy4gdGhlIGNhbGxiYWNrIHRoYXRcbiAqIG5lZWRzIHRvIGludm9rZWQgYW5kIHRoZSBpbmRleCBhdCB3aGljaCB0byBmaW5kIGl0cyBjb250ZXh0LlxuICovXG5leHBvcnQgdHlwZSBIb29rRW50cnkgPSBudW1iZXJ8SG9va0ZuO1xuXG4vKipcbiAqIEFycmF5IG9mIGhvb2tzIHRoYXQgc2hvdWxkIGJlIGV4ZWN1dGVkIGZvciBhIHZpZXcgYW5kIHRoZWlyIGRpcmVjdGl2ZSBpbmRpY2VzLlxuICpcbiAqIEZvciBlYWNoIG5vZGUgb2YgdGhlIHZpZXcsIHRoZSBmb2xsb3dpbmcgZGF0YSBpcyBzdG9yZWQ6XG4gKiAxKSBOb2RlIGluZGV4IChvcHRpb25hbClcbiAqIDIpIEEgc2VyaWVzIG9mIG51bWJlci9mdW5jdGlvbiBwYWlycyB3aGVyZTpcbiAqICAtIGV2ZW4gaW5kaWNlcyBhcmUgZGlyZWN0aXZlIGluZGljZXNcbiAqICAtIG9kZCBpbmRpY2VzIGFyZSBob29rIGZ1bmN0aW9uc1xuICpcbiAqIFNwZWNpYWwgY2FzZXM6XG4gKiAgLSBhIG5lZ2F0aXZlIGRpcmVjdGl2ZSBpbmRleCBmbGFncyBhbiBpbml0IGhvb2sgKG5nT25Jbml0LCBuZ0FmdGVyQ29udGVudEluaXQsIG5nQWZ0ZXJWaWV3SW5pdClcbiAqL1xuZXhwb3J0IHR5cGUgSG9va0RhdGEgPSBIb29rRW50cnlbXTtcblxuLyoqXG4gKiBBcnJheSBvZiBkZXN0cm95IGhvb2tzIHRoYXQgc2hvdWxkIGJlIGV4ZWN1dGVkIGZvciBhIHZpZXcgYW5kIHRoZWlyIGRpcmVjdGl2ZSBpbmRpY2VzLlxuICpcbiAqIFRoZSBhcnJheSBpcyBzZXQgdXAgYXMgYSBzZXJpZXMgb2YgbnVtYmVyL2Z1bmN0aW9uIG9yIG51bWJlci8obnVtYmVyfGZ1bmN0aW9uKVtdOlxuICogLSBFdmVuIGluZGljZXMgcmVwcmVzZW50IHRoZSBjb250ZXh0IHdpdGggd2hpY2ggaG9va3Mgc2hvdWxkIGJlIGNhbGxlZC5cbiAqIC0gT2RkIGluZGljZXMgYXJlIHRoZSBob29rIGZ1bmN0aW9ucyB0aGVtc2VsdmVzLiBJZiBhIHZhbHVlIGF0IGFuIG9kZCBpbmRleCBpcyBhbiBhcnJheSxcbiAqICAgaXQgcmVwcmVzZW50cyB0aGUgZGVzdHJveSBob29rcyBvZiBhIGBtdWx0aWAgcHJvdmlkZXIgd2hlcmU6XG4gKiAgICAgLSBFdmVuIGluZGljZXMgcmVwcmVzZW50IHRoZSBpbmRleCBvZiB0aGUgcHJvdmlkZXIgZm9yIHdoaWNoIHdlJ3ZlIHJlZ2lzdGVyZWQgYSBkZXN0cm95IGhvb2ssXG4gKiAgICAgICBpbnNpZGUgb2YgdGhlIGBtdWx0aWAgcHJvdmlkZXIgYXJyYXkuXG4gKiAgICAgLSBPZGQgaW5kaWNlcyBhcmUgdGhlIGRlc3Ryb3kgaG9vayBmdW5jdGlvbnMuXG4gKiBGb3IgZXhhbXBsZTpcbiAqIExWaWV3OiBgWzAsIDEsIDIsIEFTZXJ2aWNlLCA0LCBbQlNlcnZpY2UsIENTZXJ2aWNlLCBEU2VydmljZV1dYFxuICogZGVzdHJveUhvb2tzOiBgWzMsIEFTZXJ2aWNlLm5nT25EZXN0cm95LCA1LCBbMCwgQlNlcnZpY2UubmdPbkRlc3Ryb3ksIDIsIERTZXJ2aWNlLm5nT25EZXN0cm95XV1gXG4gKlxuICogSW4gdGhlIGV4YW1wbGUgYWJvdmUgYEFTZXJ2aWNlYCBpcyBhIHR5cGUgcHJvdmlkZXIgd2l0aCBhbiBgbmdPbkRlc3Ryb3lgLCB3aGVyZWFzIGBCU2VydmljZWAsXG4gKiBgQ1NlcnZpY2VgIGFuZCBgRFNlcnZpY2VgIGFyZSBwYXJ0IG9mIGEgYG11bHRpYCBwcm92aWRlciB3aGVyZSBvbmx5IGBCU2VydmljZWAgYW5kIGBEU2VydmljZWBcbiAqIGhhdmUgYW4gYG5nT25EZXN0cm95YCBob29rLlxuICovXG5leHBvcnQgdHlwZSBEZXN0cm95SG9va0RhdGEgPSAoSG9va0VudHJ5fEhvb2tEYXRhKVtdO1xuXG4vKipcbiAqIFN0YXRpYyBkYXRhIHRoYXQgY29ycmVzcG9uZHMgdG8gdGhlIGluc3RhbmNlLXNwZWNpZmljIGRhdGEgYXJyYXkgb24gYW4gTFZpZXcuXG4gKlxuICogRWFjaCBub2RlJ3Mgc3RhdGljIGRhdGEgaXMgc3RvcmVkIGluIHREYXRhIGF0IHRoZSBzYW1lIGluZGV4IHRoYXQgaXQncyBzdG9yZWRcbiAqIGluIHRoZSBkYXRhIGFycmF5LiAgQW55IG5vZGVzIHRoYXQgZG8gbm90IGhhdmUgc3RhdGljIGRhdGEgc3RvcmUgYSBudWxsIHZhbHVlIGluXG4gKiB0RGF0YSB0byBhdm9pZCBhIHNwYXJzZSBhcnJheS5cbiAqXG4gKiBFYWNoIHBpcGUncyBkZWZpbml0aW9uIGlzIHN0b3JlZCBoZXJlIGF0IHRoZSBzYW1lIGluZGV4IGFzIGl0cyBwaXBlIGluc3RhbmNlIGluXG4gKiB0aGUgZGF0YSBhcnJheS5cbiAqXG4gKiBFYWNoIGhvc3QgcHJvcGVydHkncyBuYW1lIGlzIHN0b3JlZCBoZXJlIGF0IHRoZSBzYW1lIGluZGV4IGFzIGl0cyB2YWx1ZSBpbiB0aGVcbiAqIGRhdGEgYXJyYXkuXG4gKlxuICogRWFjaCBwcm9wZXJ0eSBiaW5kaW5nIG5hbWUgaXMgc3RvcmVkIGhlcmUgYXQgdGhlIHNhbWUgaW5kZXggYXMgaXRzIHZhbHVlIGluXG4gKiB0aGUgZGF0YSBhcnJheS4gSWYgdGhlIGJpbmRpbmcgaXMgYW4gaW50ZXJwb2xhdGlvbiwgdGhlIHN0YXRpYyBzdHJpbmcgdmFsdWVzXG4gKiBhcmUgc3RvcmVkIHBhcmFsbGVsIHRvIHRoZSBkeW5hbWljIHZhbHVlcy4gRXhhbXBsZTpcbiAqXG4gKiBpZD1cInByZWZpeCB7eyB2MCB9fSBhIHt7IHYxIH19IGIge3sgdjIgfX0gc3VmZml4XCJcbiAqXG4gKiBMVmlldyAgICAgICB8ICAgVFZpZXcuZGF0YVxuICotLS0tLS0tLS0tLS0tLS0tLS0tLS0tLS1cbiAqICB2MCB2YWx1ZSAgIHwgICAnYSdcbiAqICB2MSB2YWx1ZSAgIHwgICAnYidcbiAqICB2MiB2YWx1ZSAgIHwgICBpZCDvv70gcHJlZml4IO+/vSBzdWZmaXhcbiAqXG4gKiBJbmplY3RvciBibG9vbSBmaWx0ZXJzIGFyZSBhbHNvIHN0b3JlZCBoZXJlLlxuICovXG5leHBvcnQgdHlwZSBURGF0YSA9IChUTm9kZXxQaXBlRGVmPGFueT58RGlyZWN0aXZlRGVmPGFueT58Q29tcG9uZW50RGVmPGFueT58bnVtYmVyfFRTdHlsaW5nUmFuZ2V8XG4gICAgICAgICAgICAgICAgICAgICBUU3R5bGluZ0tleXxQcm92aWRlclRva2VuPGFueT58VEkxOG58STE4blVwZGF0ZU9wQ29kZXN8VEljdXxudWxsfHN0cmluZylbXTtcblxuLy8gTm90ZTogVGhpcyBoYWNrIGlzIG5lY2Vzc2FyeSBzbyB3ZSBkb24ndCBlcnJvbmVvdXNseSBnZXQgYSBjaXJjdWxhciBkZXBlbmRlbmN5XG4vLyBmYWlsdXJlIGJhc2VkIG9uIHR5cGVzLlxuZXhwb3J0IGNvbnN0IHVudXNlZFZhbHVlRXhwb3J0VG9QbGFjYXRlQWpkID0gMTtcblxuLyoqXG4gKiBIdW1hbiByZWFkYWJsZSB2ZXJzaW9uIG9mIHRoZSBgTFZpZXdgLlxuICpcbiAqIGBMVmlld2AgaXMgYSBkYXRhIHN0cnVjdHVyZSB1c2VkIGludGVybmFsbHkgdG8ga2VlcCB0cmFjayBvZiB2aWV3cy4gVGhlIGBMVmlld2AgaXMgZGVzaWduZWQgZm9yXG4gKiBlZmZpY2llbmN5IGFuZCBzbyBhdCB0aW1lcyBpdCBpcyBkaWZmaWN1bHQgdG8gcmVhZCBvciB3cml0ZSB0ZXN0cyB3aGljaCBhc3NlcnQgb24gaXRzIHZhbHVlcy4gRm9yXG4gKiB0aGlzIHJlYXNvbiB3aGVuIGBuZ0Rldk1vZGVgIGlzIHRydWUgd2UgcGF0Y2ggYSBgTFZpZXcuZGVidWdgIHByb3BlcnR5IHdoaWNoIHBvaW50cyB0b1xuICogYExWaWV3RGVidWdgIGZvciBlYXNpZXIgZGVidWdnaW5nIGFuZCB0ZXN0IHdyaXRpbmcuIEl0IGlzIHRoZSBpbnRlbnQgb2YgYExWaWV3RGVidWdgIHRvIGJlIHVzZWRcbiAqIGluIHRlc3RzLlxuICovXG5leHBvcnQgaW50ZXJmYWNlIExWaWV3RGVidWc8VCA9IHVua25vd24+IHtcbiAgLyoqXG4gICAqIEZsYWdzIGFzc29jaWF0ZWQgd2l0aCB0aGUgYExWaWV3YCB1bnBhY2tlZCBpbnRvIGEgbW9yZSByZWFkYWJsZSBzdGF0ZS5cbiAgICpcbiAgICogU2VlIGBMVmlld0ZsYWdzYCBmb3IgdGhlIGZsYWcgbWVhbmluZ3MuXG4gICAqL1xuICByZWFkb25seSBmbGFnczoge1xuICAgIGluaXRQaGFzZVN0YXRlOiBudW1iZXIsXG4gICAgY3JlYXRpb25Nb2RlOiBib29sZWFuLFxuICAgIGZpcnN0Vmlld1Bhc3M6IGJvb2xlYW4sXG4gICAgY2hlY2tBbHdheXM6IGJvb2xlYW4sXG4gICAgZGlydHk6IGJvb2xlYW4sXG4gICAgYXR0YWNoZWQ6IGJvb2xlYW4sXG4gICAgZGVzdHJveWVkOiBib29sZWFuLFxuICAgIGlzUm9vdDogYm9vbGVhbixcbiAgICBpbmRleFdpdGhpbkluaXRQaGFzZTogbnVtYmVyLFxuICB9O1xuXG4gIC8qKlxuICAgKiBBc3NvY2lhdGVkIFRWaWV3XG4gICAqL1xuICByZWFkb25seSB0VmlldzogVFZpZXc7XG5cbiAgLyoqXG4gICAqIFBhcmVudCB2aWV3IChvciBjb250YWluZXIpXG4gICAqL1xuICByZWFkb25seSBwYXJlbnQ6IExWaWV3RGVidWd8TENvbnRhaW5lckRlYnVnfG51bGw7XG5cbiAgLyoqXG4gICAqIE5leHQgc2libGluZyB0byB0aGUgYExWaWV3YC5cbiAgICovXG4gIHJlYWRvbmx5IG5leHQ6IExWaWV3RGVidWd8TENvbnRhaW5lckRlYnVnfG51bGw7XG5cbiAgLyoqXG4gICAqIFRoZSBjb250ZXh0IHVzZWQgZm9yIGV2YWx1YXRpb24gb2YgdGhlIGBMVmlld2BcbiAgICpcbiAgICogKFVzdWFsbHkgdGhlIGNvbXBvbmVudClcbiAgICovXG4gIHJlYWRvbmx5IGNvbnRleHQ6IFQ7XG5cbiAgLyoqXG4gICAqIEhpZXJhcmNoaWNhbCB0cmVlIG9mIG5vZGVzLlxuICAgKi9cbiAgcmVhZG9ubHkgbm9kZXM6IERlYnVnTm9kZVtdO1xuXG4gIC8qKlxuICAgKiBUZW1wbGF0ZSBzdHJ1Y3R1cmUgKG5vIGluc3RhbmNlIGRhdGEpLlxuICAgKiAoU2hvd3MgaG93IFROb2RlcyBhcmUgY29ubmVjdGVkKVxuICAgKi9cbiAgcmVhZG9ubHkgdGVtcGxhdGU6IHN0cmluZztcblxuICAvKipcbiAgICogSFRNTCByZXByZXNlbnRhdGlvbiBvZiB0aGUgYExWaWV3YC5cbiAgICpcbiAgICogVGhpcyBpcyBvbmx5IGFwcHJveGltYXRlIHRvIGFjdHVhbCBIVE1MIGFzIGNoaWxkIGBMVmlld2BzIGFyZSByZW1vdmVkLlxuICAgKi9cbiAgcmVhZG9ubHkgaHRtbDogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBUaGUgaG9zdCBlbGVtZW50IHRvIHdoaWNoIHRoaXMgYExWaWV3YCBpcyBhdHRhY2hlZC5cbiAgICovXG4gIHJlYWRvbmx5IGhvc3RIVE1MOiBzdHJpbmd8bnVsbDtcblxuICAvKipcbiAgICogQ2hpbGQgYExWaWV3YHNcbiAgICovXG4gIHJlYWRvbmx5IGNoaWxkVmlld3M6IEFycmF5PExWaWV3RGVidWd8TENvbnRhaW5lckRlYnVnPjtcblxuICAvKipcbiAgICogU3ViIHJhbmdlIG9mIGBMVmlld2AgY29udGFpbmluZyBkZWNscyAoRE9NIGVsZW1lbnRzKS5cbiAgICovXG4gIHJlYWRvbmx5IGRlY2xzOiBMVmlld0RlYnVnUmFuZ2U7XG5cbiAgLyoqXG4gICAqIFN1YiByYW5nZSBvZiBgTFZpZXdgIGNvbnRhaW5pbmcgdmFycyAoYmluZGluZ3MpLlxuICAgKi9cbiAgcmVhZG9ubHkgdmFyczogTFZpZXdEZWJ1Z1JhbmdlO1xuXG4gIC8qKlxuICAgKiBTdWIgcmFuZ2Ugb2YgYExWaWV3YCBjb250YWluaW5nIGV4cGFuZG8gKHVzZWQgYnkgREkpLlxuICAgKi9cbiAgcmVhZG9ubHkgZXhwYW5kbzogTFZpZXdEZWJ1Z1JhbmdlO1xufVxuXG4vKipcbiAqIEh1bWFuIHJlYWRhYmxlIHZlcnNpb24gb2YgdGhlIGBMQ29udGFpbmVyYFxuICpcbiAqIGBMQ29udGFpbmVyYCBpcyBhIGRhdGEgc3RydWN0dXJlIHVzZWQgaW50ZXJuYWxseSB0byBrZWVwIHRyYWNrIG9mIGNoaWxkIHZpZXdzLiBUaGUgYExDb250YWluZXJgXG4gKiBpcyBkZXNpZ25lZCBmb3IgZWZmaWNpZW5jeSBhbmQgc28gYXQgdGltZXMgaXQgaXMgZGlmZmljdWx0IHRvIHJlYWQgb3Igd3JpdGUgdGVzdHMgd2hpY2ggYXNzZXJ0IG9uXG4gKiBpdHMgdmFsdWVzLiBGb3IgdGhpcyByZWFzb24gd2hlbiBgbmdEZXZNb2RlYCBpcyB0cnVlIHdlIHBhdGNoIGEgYExDb250YWluZXIuZGVidWdgIHByb3BlcnR5IHdoaWNoXG4gKiBwb2ludHMgdG8gYExDb250YWluZXJEZWJ1Z2AgZm9yIGVhc2llciBkZWJ1Z2dpbmcgYW5kIHRlc3Qgd3JpdGluZy4gSXQgaXMgdGhlIGludGVudCBvZlxuICogYExDb250YWluZXJEZWJ1Z2AgdG8gYmUgdXNlZCBpbiB0ZXN0cy5cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBMQ29udGFpbmVyRGVidWcge1xuICByZWFkb25seSBuYXRpdmU6IFJDb21tZW50O1xuICAvKipcbiAgICogQ2hpbGQgYExWaWV3YHMuXG4gICAqL1xuICByZWFkb25seSB2aWV3czogTFZpZXdEZWJ1Z1tdO1xuICByZWFkb25seSBwYXJlbnQ6IExWaWV3RGVidWd8bnVsbDtcbiAgcmVhZG9ubHkgbW92ZWRWaWV3czogTFZpZXdbXXxudWxsO1xuICByZWFkb25seSBob3N0OiBSRWxlbWVudHxSQ29tbWVudHxMVmlldztcbiAgcmVhZG9ubHkgbmV4dDogTFZpZXdEZWJ1Z3xMQ29udGFpbmVyRGVidWd8bnVsbDtcbiAgcmVhZG9ubHkgaGFzVHJhbnNwbGFudGVkVmlld3M6IGJvb2xlYW47XG59XG5cblxuXG4vKipcbiAqIGBMVmlld2AgaXMgc3ViZGl2aWRlZCB0byByYW5nZXMgd2hlcmUgdGhlIGFjdHVhbCBkYXRhIGlzIHN0b3JlZC4gU29tZSBvZiB0aGVzZSByYW5nZXMgc3VjaCBhc1xuICogYGRlY2xzYCBhbmQgYHZhcnNgIGFyZSBrbm93biBhdCBjb21waWxlIHRpbWUuIE90aGVyIHN1Y2ggYXMgYGkxOG5gIGFuZCBgZXhwYW5kb2AgYXJlIHJ1bnRpbWUgb25seVxuICogY29uY2VwdHMuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTFZpZXdEZWJ1Z1JhbmdlIHtcbiAgLyoqXG4gICAqIFRoZSBzdGFydGluZyBpbmRleCBpbiBgTFZpZXdgIHdoZXJlIHRoZSByYW5nZSBiZWdpbnMuIChJbmNsdXNpdmUpXG4gICAqL1xuICBzdGFydDogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgZW5kaW5nIGluZGV4IGluIGBMVmlld2Agd2hlcmUgdGhlIHJhbmdlIGVuZHMuIChFeGNsdXNpdmUpXG4gICAqL1xuICBlbmQ6IG51bWJlcjtcblxuICAvKipcbiAgICogVGhlIGxlbmd0aCBvZiB0aGUgcmFuZ2VcbiAgICovXG4gIGxlbmd0aDogbnVtYmVyO1xuXG4gIC8qKlxuICAgKiBUaGUgbWVyZ2VkIGNvbnRlbnQgb2YgdGhlIHJhbmdlLiBgdGAgY29udGFpbnMgZGF0YSBmcm9tIGBUVmlldy5kYXRhYCBhbmQgYGxgIGNvbnRhaW5zIGBMVmlld2BcbiAgICogZGF0YSBhdCBhbiBpbmRleC5cbiAgICovXG4gIGNvbnRlbnQ6IExWaWV3RGVidWdSYW5nZUNvbnRlbnRbXTtcbn1cblxuLyoqXG4gKiBGb3IgY29udmVuaWVuY2UgdGhlIHN0YXRpYyBhbmQgaW5zdGFuY2UgcG9ydGlvbnMgb2YgYFRWaWV3YCBhbmQgYExWaWV3YCBhcmUgbWVyZ2VkIGludG8gYSBzaW5nbGVcbiAqIG9iamVjdCBpbiBgTFZpZXdSYW5nZWAuXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgTFZpZXdEZWJ1Z1JhbmdlQ29udGVudCB7XG4gIC8qKlxuICAgKiBJbmRleCBpbnRvIG9yaWdpbmFsIGBMVmlld2Agb3IgYFRWaWV3LmRhdGFgLlxuICAgKi9cbiAgaW5kZXg6IG51bWJlcjtcblxuICAvKipcbiAgICogVmFsdWUgZnJvbSB0aGUgYFRWaWV3LmRhdGFbaW5kZXhdYCBsb2NhdGlvbi5cbiAgICovXG4gIHQ6IGFueTtcblxuICAvKipcbiAgICogVmFsdWUgZnJvbSB0aGUgYExWaWV3W2luZGV4XWAgbG9jYXRpb24uXG4gICAqL1xuICBsOiBhbnk7XG59XG5cblxuLyoqXG4gKiBBIGxvZ2ljYWwgbm9kZSB3aGljaCBjb21wcmlzZSBpbnRvIGBMVmlld2BzLlxuICpcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBEZWJ1Z05vZGUge1xuICAvKipcbiAgICogSFRNTCByZXByZXNlbnRhdGlvbiBvZiB0aGUgbm9kZS5cbiAgICovXG4gIGh0bWw6IHN0cmluZ3xudWxsO1xuXG4gIC8qKlxuICAgKiBBc3NvY2lhdGVkIGBUTm9kZWBcbiAgICovXG4gIHROb2RlOiBUTm9kZTtcblxuICAvKipcbiAgICogSHVtYW4gcmVhZGFibGUgbm9kZSB0eXBlLlxuICAgKi9cbiAgdHlwZTogc3RyaW5nO1xuXG4gIC8qKlxuICAgKiBET00gbmF0aXZlIG5vZGUuXG4gICAqL1xuICBuYXRpdmU6IE5vZGU7XG5cbiAgLyoqXG4gICAqIENoaWxkIG5vZGVzXG4gICAqL1xuICBjaGlsZHJlbjogRGVidWdOb2RlW107XG5cbiAgLyoqXG4gICAqIEEgbGlzdCBvZiBDb21wb25lbnQvRGlyZWN0aXZlIHR5cGVzIHdoaWNoIG5lZWQgdG8gYmUgaW5zdGFudGlhdGVkIGFuIHRoaXMgbG9jYXRpb24uXG4gICAqL1xuICBmYWN0b3JpZXM6IFR5cGU8dW5rbm93bj5bXTtcblxuICAvKipcbiAgICogQSBsaXN0IG9mIENvbXBvbmVudC9EaXJlY3RpdmUgaW5zdGFuY2VzIHdoaWNoIHdlcmUgaW5zdGFudGlhdGVkIGFuIHRoaXMgbG9jYXRpb24uXG4gICAqL1xuICBpbnN0YW5jZXM6IHVua25vd25bXTtcblxuICAvKipcbiAgICogTm9kZUluamVjdG9yIGluZm9ybWF0aW9uLlxuICAgKi9cbiAgaW5qZWN0b3I6IE5vZGVJbmplY3RvckRlYnVnO1xuXG4gIC8qKlxuICAgKiBJbmplY3RvciByZXNvbHV0aW9uIHBhdGguXG4gICAqL1xuICBpbmplY3RvclJlc29sdXRpb25QYXRoOiBhbnk7XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgTm9kZUluamVjdG9yRGVidWcge1xuICAvKipcbiAgICogSW5zdGFuY2UgYmxvb20uIERvZXMgdGhlIGN1cnJlbnQgaW5qZWN0b3IgaGF2ZSBhIHByb3ZpZGVyIHdpdGggYSBnaXZlbiBibG9vbSBtYXNrLlxuICAgKi9cbiAgYmxvb206IHN0cmluZztcblxuXG4gIC8qKlxuICAgKiBDdW11bGF0aXZlIGJsb29tLiBEbyBhbnkgb2YgdGhlIGFib3ZlIGluamVjdG9ycyBoYXZlIGEgcHJvdmlkZXIgd2l0aCBhIGdpdmVuIGJsb29tIG1hc2suXG4gICAqL1xuICBjdW11bGF0aXZlQmxvb206IHN0cmluZztcblxuICAvKipcbiAgICogQSBsaXN0IG9mIHByb3ZpZGVycyBhc3NvY2lhdGVkIHdpdGggdGhpcyBpbmplY3Rvci5cbiAgICovXG4gIHByb3ZpZGVyczogKFR5cGU8dW5rbm93bj58RGlyZWN0aXZlRGVmPHVua25vd24+fENvbXBvbmVudERlZjx1bmtub3duPilbXTtcblxuICAvKipcbiAgICogQSBsaXN0IG9mIHByb3ZpZGVycyBhc3NvY2lhdGVkIHdpdGggdGhpcyBpbmplY3RvciB2aXNpYmxlIHRvIHRoZSB2aWV3IG9mIHRoZSBjb21wb25lbnQgb25seS5cbiAgICovXG4gIHZpZXdQcm92aWRlcnM6IFR5cGU8dW5rbm93bj5bXTtcblxuXG4gIC8qKlxuICAgKiBMb2NhdGlvbiBvZiB0aGUgcGFyZW50IGBUTm9kZWAuXG4gICAqL1xuICBwYXJlbnRJbmplY3RvckluZGV4OiBudW1iZXI7XG59XG4iXX0=