# frozen_string_literal: true

module Lumberjack
  class Formatter
    # Truncate a string object to a specific length. This is useful
    # for formatting messages when there is a limit on the number of
    # characters that can be logged per message. This formatter should
    # only be used when necessary since it is a lossy formatter.
    #
    # When a string is truncated, it will have a unicode ellipsis
    # character (U+2026) appended to the end of the string.
    class TruncateFormatter
      FormatterRegistry.add(:truncate, self)

      # @param length [Integer] The maximum length of the string (defaults to 32K).
      def initialize(length = 32768)
        @length = length
      end

      # Truncate a string if it exceeds the maximum length.
      #
      # @param obj [Object] The object to format. If it's a string longer than the maximum
      #   length, it will be truncated. Other objects are returned unchanged.
      # @return [String, Object] The truncated string or original object.
      def call(obj)
        if obj.is_a?(String) && obj.length > @length
          "#{obj[0, @length - 1]}…"
        else
          obj
        end
      end
    end
  end
end
