package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link SecretFile}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableSecretFile.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableSecretFile implements SecretFile {
  private final String name;
  private final @Nullable String uid;
  private final @Nullable String gid;
  private final @Nullable Long mode;

  private ImmutableSecretFile(
      String name,
      @Nullable String uid,
      @Nullable String gid,
      @Nullable Long mode) {
    this.name = name;
    this.uid = uid;
    this.gid = gid;
    this.mode = mode;
  }

  /**
   * @return The value of the {@code name} attribute
   */
  @JsonProperty("Name")
  @Override
  public String name() {
    return name;
  }

  /**
   * @return The value of the {@code uid} attribute
   */
  @JsonProperty("UID")
  @Override
  public @Nullable String uid() {
    return uid;
  }

  /**
   * @return The value of the {@code gid} attribute
   */
  @JsonProperty("GID")
  @Override
  public @Nullable String gid() {
    return gid;
  }

  /**
   * @return The value of the {@code mode} attribute
   */
  @JsonProperty("Mode")
  @Override
  public @Nullable Long mode() {
    return mode;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link SecretFile#name() name} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for name
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableSecretFile withName(String value) {
    String newValue = Objects.requireNonNull(value, "name");
    if (this.name.equals(newValue)) return this;
    return new ImmutableSecretFile(newValue, this.uid, this.gid, this.mode);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link SecretFile#uid() uid} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for uid (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableSecretFile withUid(@Nullable String value) {
    if (Objects.equals(this.uid, value)) return this;
    return new ImmutableSecretFile(this.name, value, this.gid, this.mode);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link SecretFile#gid() gid} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for gid (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableSecretFile withGid(@Nullable String value) {
    if (Objects.equals(this.gid, value)) return this;
    return new ImmutableSecretFile(this.name, this.uid, value, this.mode);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link SecretFile#mode() mode} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for mode (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableSecretFile withMode(@Nullable Long value) {
    if (Objects.equals(this.mode, value)) return this;
    return new ImmutableSecretFile(this.name, this.uid, this.gid, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableSecretFile} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableSecretFile
        && equalsByValue((ImmutableSecretFile) another);
  }

  private boolean equalsByValue(ImmutableSecretFile another) {
    return name.equals(another.name)
        && Objects.equals(uid, another.uid)
        && Objects.equals(gid, another.gid)
        && Objects.equals(mode, another.mode);
  }

  /**
   * Computes a hash code from attributes: {@code name}, {@code uid}, {@code gid}, {@code mode}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + name.hashCode();
    h += (h << 5) + Objects.hashCode(uid);
    h += (h << 5) + Objects.hashCode(gid);
    h += (h << 5) + Objects.hashCode(mode);
    return h;
  }

  /**
   * Prints the immutable value {@code SecretFile} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "SecretFile{"
        + "name=" + name
        + ", uid=" + uid
        + ", gid=" + gid
        + ", mode=" + mode
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link SecretFile} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable SecretFile instance
   */
  public static ImmutableSecretFile copyOf(SecretFile instance) {
    if (instance instanceof ImmutableSecretFile) {
      return (ImmutableSecretFile) instance;
    }
    return ImmutableSecretFile.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableSecretFile ImmutableSecretFile}.
   * <pre>
   * ImmutableSecretFile.builder()
   *    .name(String) // required {@link SecretFile#name() name}
   *    .uid(String | null) // nullable {@link SecretFile#uid() uid}
   *    .gid(String | null) // nullable {@link SecretFile#gid() gid}
   *    .mode(Long | null) // nullable {@link SecretFile#mode() mode}
   *    .build();
   * </pre>
   * @return A new ImmutableSecretFile builder
   */
  public static ImmutableSecretFile.Builder builder() {
    return new ImmutableSecretFile.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableSecretFile ImmutableSecretFile}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements SecretFile.Builder {
    private static final long INIT_BIT_NAME = 0x1L;
    private long initBits = 0x1L;

    private String name;
    private String uid;
    private String gid;
    private Long mode;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code SecretFile} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(SecretFile instance) {
      Objects.requireNonNull(instance, "instance");
      this.name(instance.name());
      @Nullable String uidValue = instance.uid();
      if (uidValue != null) {
        uid(uidValue);
      }
      @Nullable String gidValue = instance.gid();
      if (gidValue != null) {
        gid(gidValue);
      }
      @Nullable Long modeValue = instance.mode();
      if (modeValue != null) {
        mode(modeValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link SecretFile#name() name} attribute.
     * @param name The value for name 
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Name")
    public final Builder name(String name) {
      this.name = Objects.requireNonNull(name, "name");
      initBits &= ~INIT_BIT_NAME;
      return this;
    }

    /**
     * Initializes the value for the {@link SecretFile#uid() uid} attribute.
     * @param uid The value for uid (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("UID")
    public final Builder uid(@Nullable String uid) {
      this.uid = uid;
      return this;
    }

    /**
     * Initializes the value for the {@link SecretFile#gid() gid} attribute.
     * @param gid The value for gid (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("GID")
    public final Builder gid(@Nullable String gid) {
      this.gid = gid;
      return this;
    }

    /**
     * Initializes the value for the {@link SecretFile#mode() mode} attribute.
     * @param mode The value for mode (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Mode")
    public final Builder mode(@Nullable Long mode) {
      this.mode = mode;
      return this;
    }

    /**
     * Builds a new {@link ImmutableSecretFile ImmutableSecretFile}.
     * @return An immutable instance of SecretFile
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableSecretFile build() {
      if (initBits != 0) {
        throw new IllegalStateException(formatRequiredAttributesMessage());
      }
      return new ImmutableSecretFile(name, uid, gid, mode);
    }

    private String formatRequiredAttributesMessage() {
      List<String> attributes = new ArrayList<>();
      if ((initBits & INIT_BIT_NAME) != 0) attributes.add("name");
      return "Cannot build SecretFile, some of required attributes are not set " + attributes;
    }
  }
}
