package org.mandas.docker.client.messages.swarm;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link Placement}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutablePlacement.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutablePlacement implements Placement {
  private final @Nullable List<String> constraints;
  private final @Nullable List<Preference> preferences;
  private final @Nullable Integer maxReplicas;

  private ImmutablePlacement(
      @Nullable List<String> constraints,
      @Nullable List<Preference> preferences,
      @Nullable Integer maxReplicas) {
    this.constraints = constraints;
    this.preferences = preferences;
    this.maxReplicas = maxReplicas;
  }

  /**
   * @return The value of the {@code constraints} attribute
   */
  @JsonProperty("Constraints")
  @Override
  public @Nullable List<String> constraints() {
    return constraints;
  }

  /**
   * @return The value of the {@code preferences} attribute
   */
  @JsonProperty("Preferences")
  @Override
  public @Nullable List<Preference> preferences() {
    return preferences;
  }

  /**
   * @return The value of the {@code maxReplicas} attribute
   */
  @JsonProperty("MaxReplicas")
  @Override
  public @Nullable Integer maxReplicas() {
    return maxReplicas;
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Placement#constraints() constraints}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutablePlacement withConstraints(@Nullable String... elements) {
    if (elements == null) {
      if (this.constraints == null) return this;
      return new ImmutablePlacement(null, this.preferences, this.maxReplicas);
    }
    List<String> newValue = createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutablePlacement(newValue, this.preferences, this.maxReplicas);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Placement#constraints() constraints}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of constraints elements to set
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutablePlacement withConstraints(@Nullable Iterable<String> elements) {
    if (this.constraints == elements) return this;
    @Nullable List<String> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutablePlacement(newValue, this.preferences, this.maxReplicas);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Placement#preferences() preferences}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutablePlacement withPreferences(@Nullable Preference... elements) {
    if (elements == null) {
      if (this.preferences == null) return this;
      return new ImmutablePlacement(this.constraints, null, this.maxReplicas);
    }
    List<Preference> newValue = createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutablePlacement(this.constraints, newValue, this.maxReplicas);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link Placement#preferences() preferences}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of preferences elements to set
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutablePlacement withPreferences(@Nullable Iterable<? extends Preference> elements) {
    if (this.preferences == elements) return this;
    @Nullable List<Preference> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutablePlacement(this.constraints, newValue, this.maxReplicas);
  }

  /**
   * Copy the current immutable object by setting a value for the {@link Placement#maxReplicas() maxReplicas} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for maxReplicas (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutablePlacement withMaxReplicas(@Nullable Integer value) {
    if (Objects.equals(this.maxReplicas, value)) return this;
    return new ImmutablePlacement(this.constraints, this.preferences, value);
  }

  /**
   * This instance is equal to all instances of {@code ImmutablePlacement} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutablePlacement
        && equalsByValue((ImmutablePlacement) another);
  }

  private boolean equalsByValue(ImmutablePlacement another) {
    return Objects.equals(constraints, another.constraints)
        && Objects.equals(preferences, another.preferences)
        && Objects.equals(maxReplicas, another.maxReplicas);
  }

  /**
   * Computes a hash code from attributes: {@code constraints}, {@code preferences}, {@code maxReplicas}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(constraints);
    h += (h << 5) + Objects.hashCode(preferences);
    h += (h << 5) + Objects.hashCode(maxReplicas);
    return h;
  }

  /**
   * Prints the immutable value {@code Placement} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "Placement{"
        + "constraints=" + constraints
        + ", preferences=" + preferences
        + ", maxReplicas=" + maxReplicas
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link Placement} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable Placement instance
   */
  public static ImmutablePlacement copyOf(Placement instance) {
    if (instance instanceof ImmutablePlacement) {
      return (ImmutablePlacement) instance;
    }
    return ImmutablePlacement.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutablePlacement ImmutablePlacement}.
   * <pre>
   * ImmutablePlacement.builder()
   *    .constraints(List&lt;String&gt; | null) // nullable {@link Placement#constraints() constraints}
   *    .preferences(List&lt;org.mandas.docker.client.messages.swarm.Preference&gt; | null) // nullable {@link Placement#preferences() preferences}
   *    .maxReplicas(Integer | null) // nullable {@link Placement#maxReplicas() maxReplicas}
   *    .build();
   * </pre>
   * @return A new ImmutablePlacement builder
   */
  public static ImmutablePlacement.Builder builder() {
    return new ImmutablePlacement.Builder();
  }

  /**
   * Builds instances of type {@link ImmutablePlacement ImmutablePlacement}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder {
    private List<String> constraints = null;
    private List<Preference> preferences = null;
    private Integer maxReplicas;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code Placement} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(Placement instance) {
      Objects.requireNonNull(instance, "instance");
      @Nullable List<String> constraintsValue = instance.constraints();
      if (constraintsValue != null) {
        addAllConstraints(constraintsValue);
      }
      @Nullable List<Preference> preferencesValue = instance.preferences();
      if (preferencesValue != null) {
        addAllPreferences(preferencesValue);
      }
      @Nullable Integer maxReplicasValue = instance.maxReplicas();
      if (maxReplicasValue != null) {
        maxReplicas(maxReplicasValue);
      }
      return this;
    }

    /**
     * Adds one element to {@link Placement#constraints() constraints} list.
     * @param element A constraints element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder constraint(String element) {
      if (this.constraints == null) {
        this.constraints = new ArrayList<String>();
      }
      this.constraints.add(Objects.requireNonNull(element, "constraints element"));
      return this;
    }

    /**
     * Adds elements to {@link Placement#constraints() constraints} list.
     * @param elements An array of constraints elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder constraints(String... elements) {
      if (this.constraints == null) {
        this.constraints = new ArrayList<String>();
      }
      for (String element : elements) {
        this.constraints.add(Objects.requireNonNull(element, "constraints element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link Placement#constraints() constraints} list.
     * @param elements An iterable of constraints elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Constraints")
    public final Builder constraints(@Nullable Iterable<String> elements) {
      if (elements == null) {
        this.constraints = null;
        return this;
      }
      this.constraints = new ArrayList<String>();
      return addAllConstraints(elements);
    }

    /**
     * Adds elements to {@link Placement#constraints() constraints} list.
     * @param elements An iterable of constraints elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllConstraints(Iterable<String> elements) {
      Objects.requireNonNull(elements, "constraints element");
      if (this.constraints == null) {
        this.constraints = new ArrayList<String>();
      }
      for (String element : elements) {
        this.constraints.add(Objects.requireNonNull(element, "constraints element"));
      }
      return this;
    }

    /**
     * Adds one element to {@link Placement#preferences() preferences} list.
     * @param element A preferences element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder preference(Preference element) {
      if (this.preferences == null) {
        this.preferences = new ArrayList<Preference>();
      }
      this.preferences.add(Objects.requireNonNull(element, "preferences element"));
      return this;
    }

    /**
     * Adds elements to {@link Placement#preferences() preferences} list.
     * @param elements An array of preferences elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder preferences(Preference... elements) {
      if (this.preferences == null) {
        this.preferences = new ArrayList<Preference>();
      }
      for (Preference element : elements) {
        this.preferences.add(Objects.requireNonNull(element, "preferences element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link Placement#preferences() preferences} list.
     * @param elements An iterable of preferences elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Preferences")
    public final Builder preferences(@Nullable Iterable<? extends Preference> elements) {
      if (elements == null) {
        this.preferences = null;
        return this;
      }
      this.preferences = new ArrayList<Preference>();
      return addAllPreferences(elements);
    }

    /**
     * Adds elements to {@link Placement#preferences() preferences} list.
     * @param elements An iterable of preferences elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllPreferences(Iterable<? extends Preference> elements) {
      Objects.requireNonNull(elements, "preferences element");
      if (this.preferences == null) {
        this.preferences = new ArrayList<Preference>();
      }
      for (Preference element : elements) {
        this.preferences.add(Objects.requireNonNull(element, "preferences element"));
      }
      return this;
    }

    /**
     * Initializes the value for the {@link Placement#maxReplicas() maxReplicas} attribute.
     * @param maxReplicas The value for maxReplicas (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("MaxReplicas")
    public final Builder maxReplicas(@Nullable Integer maxReplicas) {
      this.maxReplicas = maxReplicas;
      return this;
    }

    /**
     * Builds a new {@link ImmutablePlacement ImmutablePlacement}.
     * @return An immutable instance of Placement
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutablePlacement build() {
      return new ImmutablePlacement(
          constraints == null ? null : createUnmodifiableList(true, constraints),
          preferences == null ? null : createUnmodifiableList(true, preferences),
          maxReplicas);
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>(size);
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<? extends T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }
}
