package org.mandas.docker.client.messages;

import com.fasterxml.jackson.annotation.JsonProperty;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.List;
import java.util.Objects;
import org.mandas.docker.Nullable;

/**
 * Immutable implementation of {@link ContainerCreation}.
 * <p>
 * Use the builder to create immutable instances:
 * {@code ImmutableContainerCreation.builder()}.
 */
@SuppressWarnings({"all"})
final class ImmutableContainerCreation implements ContainerCreation {
  private final @Nullable String id;
  private final @Nullable List<String> warnings;

  private ImmutableContainerCreation(
      @Nullable String id,
      @Nullable List<String> warnings) {
    this.id = id;
    this.warnings = warnings;
  }

  /**
   * @return The value of the {@code id} attribute
   */
  @JsonProperty("Id")
  @Override
  public @Nullable String id() {
    return id;
  }

  /**
   * @return The value of the {@code warnings} attribute
   */
  @JsonProperty("Warnings")
  @Override
  public @Nullable List<String> warnings() {
    return warnings;
  }

  /**
   * Copy the current immutable object by setting a value for the {@link ContainerCreation#id() id} attribute.
   * An equals check used to prevent copying of the same value by returning {@code this}.
   * @param value A new value for id (can be {@code null})
   * @return A modified copy or the {@code this} object
   */
  public final ImmutableContainerCreation withId(@Nullable String value) {
    if (Objects.equals(this.id, value)) return this;
    return new ImmutableContainerCreation(value, this.warnings);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link ContainerCreation#warnings() warnings}.
   * @param elements The elements to set
   * @return A modified copy of {@code this} object
   */
  public final ImmutableContainerCreation withWarnings(@Nullable String... elements) {
    if (elements == null) {
      if (this.warnings == null) return this;
      return new ImmutableContainerCreation(this.id, null);
    }
    List<String> newValue = createUnmodifiableList(false, createSafeList(Arrays.asList(elements), true, false));
    return new ImmutableContainerCreation(this.id, newValue);
  }

  /**
   * Copy the current immutable object with elements that replace the content of {@link ContainerCreation#warnings() warnings}.
   * A shallow reference equality check is used to prevent copying of the same value by returning {@code this}.
   * @param elements An iterable of warnings elements to set
   * @return A modified copy or {@code this} if not changed
   */
  public final ImmutableContainerCreation withWarnings(@Nullable Iterable<String> elements) {
    if (this.warnings == elements) return this;
    @Nullable List<String> newValue = elements == null ? null : createUnmodifiableList(false, createSafeList(elements, true, false));
    return new ImmutableContainerCreation(this.id, newValue);
  }

  /**
   * This instance is equal to all instances of {@code ImmutableContainerCreation} that have equal attribute values.
   * @return {@code true} if {@code this} is equal to {@code another} instance
   */
  @Override
  public boolean equals(Object another) {
    if (this == another) return true;
    return another instanceof ImmutableContainerCreation
        && equalsByValue((ImmutableContainerCreation) another);
  }

  private boolean equalsByValue(ImmutableContainerCreation another) {
    return Objects.equals(id, another.id)
        && Objects.equals(warnings, another.warnings);
  }

  /**
   * Computes a hash code from attributes: {@code id}, {@code warnings}.
   * @return hashCode value
   */
  @Override
  public int hashCode() {
    int h = 5381;
    h += (h << 5) + Objects.hashCode(id);
    h += (h << 5) + Objects.hashCode(warnings);
    return h;
  }

  /**
   * Prints the immutable value {@code ContainerCreation} with attribute values.
   * @return A string representation of the value
   */
  @Override
  public String toString() {
    return "ContainerCreation{"
        + "id=" + id
        + ", warnings=" + warnings
        + "}";
  }

  /**
   * Creates an immutable copy of a {@link ContainerCreation} value.
   * Uses accessors to get values to initialize the new immutable instance.
   * If an instance is already immutable, it is returned as is.
   * @param instance The instance to copy
   * @return A copied immutable ContainerCreation instance
   */
  public static ImmutableContainerCreation copyOf(ContainerCreation instance) {
    if (instance instanceof ImmutableContainerCreation) {
      return (ImmutableContainerCreation) instance;
    }
    return ImmutableContainerCreation.builder()
        .from(instance)
        .build();
  }

  /**
   * Creates a builder for {@link ImmutableContainerCreation ImmutableContainerCreation}.
   * <pre>
   * ImmutableContainerCreation.builder()
   *    .id(String | null) // nullable {@link ContainerCreation#id() id}
   *    .warnings(List&lt;String&gt; | null) // nullable {@link ContainerCreation#warnings() warnings}
   *    .build();
   * </pre>
   * @return A new ImmutableContainerCreation builder
   */
  public static ImmutableContainerCreation.Builder builder() {
    return new ImmutableContainerCreation.Builder();
  }

  /**
   * Builds instances of type {@link ImmutableContainerCreation ImmutableContainerCreation}.
   * Initialize attributes and then invoke the {@link #build()} method to create an
   * immutable instance.
   * <p><em>{@code Builder} is not thread-safe and generally should not be stored in a field or collection,
   * but instead used immediately to create instances.</em>
   */
  static final class Builder implements ContainerCreation.Builder {
    private String id;
    private List<String> warnings = null;

    private Builder() {
    }

    /**
     * Fill a builder with attribute values from the provided {@code ContainerCreation} instance.
     * Regular attribute values will be replaced with those from the given instance.
     * Absent optional values will not replace present values.
     * Collection elements and entries will be added, not replaced.
     * @param instance The instance from which to copy values
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder from(ContainerCreation instance) {
      Objects.requireNonNull(instance, "instance");
      @Nullable String idValue = instance.id();
      if (idValue != null) {
        id(idValue);
      }
      @Nullable List<String> warningsValue = instance.warnings();
      if (warningsValue != null) {
        addAllWarnings(warningsValue);
      }
      return this;
    }

    /**
     * Initializes the value for the {@link ContainerCreation#id() id} attribute.
     * @param id The value for id (can be {@code null})
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Id")
    public final Builder id(@Nullable String id) {
      this.id = id;
      return this;
    }

    /**
     * Adds one element to {@link ContainerCreation#warnings() warnings} list.
     * @param element A warnings element
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder warning(String element) {
      if (this.warnings == null) {
        this.warnings = new ArrayList<String>();
      }
      this.warnings.add(Objects.requireNonNull(element, "warnings element"));
      return this;
    }

    /**
     * Adds elements to {@link ContainerCreation#warnings() warnings} list.
     * @param elements An array of warnings elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder warnings(String... elements) {
      if (this.warnings == null) {
        this.warnings = new ArrayList<String>();
      }
      for (String element : elements) {
        this.warnings.add(Objects.requireNonNull(element, "warnings element"));
      }
      return this;
    }


    /**
     * Sets or replaces all elements for {@link ContainerCreation#warnings() warnings} list.
     * @param elements An iterable of warnings elements
     * @return {@code this} builder for use in a chained invocation
     */
    @JsonProperty("Warnings")
    public final Builder warnings(@Nullable Iterable<String> elements) {
      if (elements == null) {
        this.warnings = null;
        return this;
      }
      this.warnings = new ArrayList<String>();
      return addAllWarnings(elements);
    }

    /**
     * Adds elements to {@link ContainerCreation#warnings() warnings} list.
     * @param elements An iterable of warnings elements
     * @return {@code this} builder for use in a chained invocation
     */
    public final Builder addAllWarnings(Iterable<String> elements) {
      Objects.requireNonNull(elements, "warnings element");
      if (this.warnings == null) {
        this.warnings = new ArrayList<String>();
      }
      for (String element : elements) {
        this.warnings.add(Objects.requireNonNull(element, "warnings element"));
      }
      return this;
    }

    /**
     * Builds a new {@link ImmutableContainerCreation ImmutableContainerCreation}.
     * @return An immutable instance of ContainerCreation
     * @throws java.lang.IllegalStateException if any required attributes are missing
     */
    public ImmutableContainerCreation build() {
      return new ImmutableContainerCreation(id, warnings == null ? null : createUnmodifiableList(true, warnings));
    }
  }

  private static <T> List<T> createSafeList(Iterable<? extends T> iterable, boolean checkNulls, boolean skipNulls) {
    ArrayList<T> list;
    if (iterable instanceof Collection<?>) {
      int size = ((Collection<?>) iterable).size();
      if (size == 0) return Collections.emptyList();
      list = new ArrayList<>(size);
    } else {
      list = new ArrayList<>();
    }
    for (T element : iterable) {
      if (skipNulls && element == null) continue;
      if (checkNulls) Objects.requireNonNull(element, "element");
      list.add(element);
    }
    return list;
  }

  private static <T> List<T> createUnmodifiableList(boolean clone, List<? extends T> list) {
    switch(list.size()) {
    case 0: return Collections.emptyList();
    case 1: return Collections.singletonList(list.get(0));
    default:
      if (clone) {
        return Collections.unmodifiableList(new ArrayList<>(list));
      } else {
        if (list instanceof ArrayList<?>) {
          ((ArrayList<?>) list).trimToSize();
        }
        return Collections.unmodifiableList(list);
      }
    }
  }
}
