//////////////////////////////////////////////////////////////////////////////
// Copyright (c) 2010, 2026 Contributors to the Eclipse Foundation
//
// See the NOTICE file(s) distributed with this work for additional
// information regarding copyright ownership.
//
// This program and the accompanying materials are made available
// under the terms of the MIT License which is available at
// https://opensource.org/licenses/MIT
//
// SPDX-License-Identifier: MIT
//////////////////////////////////////////////////////////////////////////////

package org.eclipse.escet.cif.common;

import static org.eclipse.escet.common.java.Sets.set;
import static org.eclipse.escet.common.java.Sets.setc;

import java.util.ArrayDeque;
import java.util.Deque;
import java.util.Set;

import org.eclipse.emf.ecore.EObject;
import org.eclipse.escet.cif.metamodel.cif.declarations.DiscVariable;
import org.eclipse.escet.cif.metamodel.cif.expressions.DiscVariableExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.Expression;
import org.eclipse.escet.cif.metamodel.cif.expressions.FunctionExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.ProjectionExpression;
import org.eclipse.escet.cif.metamodel.cif.expressions.TupleExpression;
import org.eclipse.escet.cif.metamodel.cif.functions.AssignmentFuncStatement;
import org.eclipse.escet.cif.metamodel.cif.functions.BreakFuncStatement;
import org.eclipse.escet.cif.metamodel.cif.functions.ContinueFuncStatement;
import org.eclipse.escet.cif.metamodel.cif.functions.ElifFuncStatement;
import org.eclipse.escet.cif.metamodel.cif.functions.FunctionParameter;
import org.eclipse.escet.cif.metamodel.cif.functions.FunctionStatement;
import org.eclipse.escet.cif.metamodel.cif.functions.IfFuncStatement;
import org.eclipse.escet.cif.metamodel.cif.functions.InternalFunction;
import org.eclipse.escet.cif.metamodel.cif.functions.ReturnFuncStatement;
import org.eclipse.escet.cif.metamodel.cif.functions.WhileFuncStatement;
import org.eclipse.escet.cif.metamodel.java.CifWithArgWalker;
import org.eclipse.escet.common.java.DependencyOrderer;

/** CIF internal function utility methods. */
public class CifInternalFuncUtils {
    /** Constructor for the {@link CifInternalFuncUtils} class. */
    private CifInternalFuncUtils() {
        // Static class.
    }

    /**
     * Class to order internal user-defined functions by their usage dependencies.
     *
     * <p>
     * Usage of this class:
     * <ol>
     * <li>Instantiate this orderer class.</li>
     * <li>Call {@link #computeOrder} with the collection of internal user-defined functions that should be
     * ordered.</li>
     * <li>There are two possible return values from the call:
     * <ul>
     * <li>If a list is returned, there are no dependency cycles. Each function in the list may use any of the functions
     * before it in the list.</li>
     * <li>If {@code null} is returned, there is a cycle. One of the cycles can be retrieved with
     * {@link #getCycle}.</li>
     * </ul>
     * </li>
     * <li>Discard the orderer instance.</li>
     * </ol>
     * For more details, please consult the documentation of the base class.
     * </p>
     */
    public static class OrderInternalFunctions extends DependencyOrderer<InternalFunction> {
        /** Collector of direct usage of internal user-defined functions in an internal user-defined function. */
        private CollectInternalFunctionsUsage collector = new CollectInternalFunctionsUsage();

        @Override
        protected Set<InternalFunction> findDirectDependencies(InternalFunction intFunc) {
            return collector.findInternalFunctionUsage(intFunc);
        }
    }

    /**
     * Collect uses of internal user-defined functions in expressions of an internal user-defined function.
     *
     * <p>
     * This class examines all expressions, including variable initialization as well as expressions of statements. It
     * also makes no distinction between calling an internal user-defined function or storing it in a local variable.
     * </p>
     * <p>
     * Note that this check does not consider values supplied through parameters or results from calls to other
     * functions at runtime. To be safe that this function collects all cases in a statically decidable way, also forbid
     * internal user-defined functions as data. Alternatively, do a much more elaborate analysis of the specification
     * with respect to runtime behavior.
     * </p>
     */
    public static class CollectInternalFunctionsUsage extends CifWithArgWalker<Set<InternalFunction>> {
        /**
         * Find static references to internal user-defined functions inside expressions of an internal user-defined
         * function.
         *
         * @param intFunc Function to search.
         * @return The found references in expressions to internal user-defined functions.
         */
        public Set<InternalFunction> findInternalFunctionUsage(InternalFunction intFunc) {
            Set<InternalFunction> collected = set();
            walkInternalFunction(intFunc, collected);
            return collected;
        }

        @Override
        protected void preprocessFunctionExpression(FunctionExpression funcExpr, Set<InternalFunction> collected) {
            if (funcExpr.getFunction() instanceof InternalFunction intFunction) {
                collected.add(intFunction);
            }
        }
    }

    /**
     * Get the function parameters of the given function that may get modified in the function.
     *
     * @param func Function to analyze.
     * @return Function parameters that may get modified in the function.
     */
    public static Set<DiscVariable> getAssignedParameters(InternalFunction func) {
        Set<DiscVariable> parameters = setc(func.getParameters().size());

        Deque<FunctionStatement> notDone = new ArrayDeque<>();
        notDone.addAll(func.getStatements());

        while (true) {
            FunctionStatement stat = notDone.poll();
            if (stat == null) {
                break; // Done.
            }

            if (stat instanceof BreakFuncStatement) {
                continue;
            }
            if (stat instanceof ContinueFuncStatement) {
                continue;
            }
            if (stat instanceof ReturnFuncStatement) {
                continue;
            }

            if (stat instanceof IfFuncStatement) {
                IfFuncStatement s = (IfFuncStatement)stat;
                notDone.addAll(s.getThens());
                notDone.addAll(s.getElses());
                for (ElifFuncStatement es: s.getElifs()) {
                    notDone.addAll(es.getThens());
                }
                continue;
            }
            if (stat instanceof WhileFuncStatement) {
                WhileFuncStatement s = (WhileFuncStatement)stat;
                notDone.addAll(s.getStatements());
                continue;
            }
            if (stat instanceof AssignmentFuncStatement) {
                AssignmentFuncStatement s = (AssignmentFuncStatement)stat;
                getAssignedParameters(s, parameters);
                continue;
            }

            throw new RuntimeException("Unexpected/unsupported function statement: " + stat);
        }
        return parameters;
    }

    /**
     * Analyze the LHS of an assignment statement for updates to function parameters. Add such updates to the
     * {@code parameters} set.
     *
     * @param asg Assignment function statement to analyze.
     * @param parameters Function parameters already found to be assigned (updated in-place).
     */
    private static void getAssignedParameters(AssignmentFuncStatement asg, Set<DiscVariable> parameters) {
        Deque<Expression> notDone = new ArrayDeque<>();
        notDone.add(asg.getAddressable());

        while (true) {
            Expression lhs = notDone.poll();
            if (lhs == null) {
                break; // Done.
            }

            if (lhs instanceof ProjectionExpression) {
                ProjectionExpression pe = (ProjectionExpression)lhs;
                notDone.add(pe.getChild());
                continue;
            }
            if (lhs instanceof TupleExpression) {
                TupleExpression te = (TupleExpression)lhs;
                notDone.addAll(te.getFields());
                continue;
            }
            if (lhs instanceof DiscVariableExpression) {
                DiscVariableExpression ve = (DiscVariableExpression)lhs;
                DiscVariable var = ve.getVariable();
                EObject parent = var.eContainer();
                if (parent instanceof FunctionParameter) {
                    parameters.add(var);
                }
                continue;
            }

            throw new RuntimeException("Unexpected/unsupported LHS expression: " + lhs);
        }
    }

    /**
     * Returns whether the given discrete variable object represents a parameter of an internal user-defined function.
     *
     * @param discVar The discrete variable object.
     * @return {@code true} if the discrete variable object represents a parameter of an internal user-defined function,
     *     {@code false} otherwise.
     */
    public static boolean isFuncParam(DiscVariable discVar) {
        return discVar.eContainer() instanceof FunctionParameter;
    }

    /**
     * Returns whether the given discrete variable object represents a local variable of an internal user-defined
     * function.
     *
     * @param discVar The discrete variable object.
     * @return {@code true} if the discrete variable object represents a local variable of an internal user-defined
     *     function, {@code false} otherwise.
     */
    public static boolean isFuncLocalVar(DiscVariable discVar) {
        return discVar.eContainer() instanceof InternalFunction;
    }

    /**
     * Returns whether the given discrete variable object represents a parameter or local variable of an internal
     * user-defined function.
     *
     * @param discVar The discrete variable object.
     * @return {@code true} if the discrete variable object represents a parameter or local variable of an internal
     *     user-defined function, {@code false} otherwise.
     */
    public static boolean isFuncParamOrLocalVar(DiscVariable discVar) {
        return isFuncParam(discVar) || isFuncLocalVar(discVar);
    }
}
